/*----------------------------------------------------------------------------
--
--  Module:           xtmRemote
--
--  Project:          XDiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Upload or download an XDiary calendar.
--
--  Filename:         xtmRemote.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1993-09-15
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmRemote.c, Version: 1.1, Date: 95/02/18 15:52:42";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>

#include <X11/Intrinsic.h>

#include <Xm/Xm.h>
#include <Xm/RowColumn.h>
#include <Xm/Separator.h>
#include <Xm/Text.h>

#include "System.h"
#include "Message.h"
#include "TimDate.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmCalDb.h"
#include "xtmHelp.h"
#include "xtmTools.h"
#include "xitError.h"
#include "xitTools.h"
#include "xtmRemote.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Local widgets in the remote window. */
#define connLa            dataLocalW[  0 ]
#define connRc            dataLocalW[  1 ]
#define connTx            dataLocalW[  2 ]
#define explainLa         dataLocalW[  3 ]
#define hidden1La         dataLocalW[  4 ]
#define hostLa            dataLocalW[  5 ]
#define hostRc            dataLocalW[  6 ]
#define hostTx            dataLocalW[  7 ]
#define pwdChangePb       dataLocalW[  8 ]
#define pwdLa             dataLocalW[  9 ]
#define pwdRc             dataLocalW[ 10 ]
#define pwdStringLa       dataLocalW[ 11 ]
#define remoteRc          dataLocalW[ 12 ]
#define scriptLa          dataLocalW[ 13 ]
#define scriptRc          dataLocalW[ 14 ]
#define scriptTx          dataLocalW[ 15 ]
#define uidLa             dataLocalW[ 16 ]
#define uidRc             dataLocalW[ 17 ]
#define uidTx             dataLocalW[ 18 ]

/* Local widgets in the password window. */
#define pwdWhatLa         dataLocalW[  0 ]
#define pwdPasswordTx     dataLocalW[  1 ]


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Our own user data. */
typedef struct {

  /* The window is application modal? */
  Boolean  appl_modal;

  /* Name of the calendar. */
  char  calendar[ XTM_GL_MAX_CAL_NAME + 1 ];

  /* External password. */
  char  extern_pwd[ XTM_GL_MAX_PASSWORD + 1 ];

  /* The remote password window. */
  Widget  remotePwdW;

  /* The remote window. */
  Widget  remoteW;

  /* Application wide data. */
  XTM_GL_BASE_DATA_REF  appl_data_ref;

  /* Callback to inform our creator of specific actions. */
  void              *user_data;
  XTM_RT_ACTION_CB  actionCB;

} REMOTE_REC, *REMOTE_REC_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmRemote";

/* IDs for the help windows. */
static char  *remote_window_id = "Remote";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void 
  changePasswordCB( Widget          widget,
                    REMOTE_REC_REF  remote_ref,
                    XtPointer       call_data );

static void
  closeCB( Widget          widget,
           REMOTE_REC_REF  remote_ref,
           XtPointer       call_data );

static Widget
  createRemoteWindow( REMOTE_REC_REF  remote_ref,
                      Widget          parent );

static void 
  destroyCB( Widget          widget,
             REMOTE_REC_REF  remote_ref,
             XtPointer       call_data );

static void
  doAction( REMOTE_REC_REF    remote_ref,
            XTM_RT_DIRECTION  direction );

static void
  downloadCB( Widget          widget,
              REMOTE_REC_REF  remote_ref,
              XtPointer       call_data );

static void
  fetchRemoteData( REMOTE_REC_REF    remote_ref,
                   char              *host,
                   char              *uid,
                   char              *password,
                   char              *connection_type,
                   char              *remote_script );

static void 
  helpCB( Widget          widget,
          REMOTE_REC_REF  remote_ref,
          XtPointer       call_data );

static void
  pwdOkCB( Widget          widget,
           REMOTE_REC_REF  remote_ref,
           XtPointer       call_data );

static void
  setRemoteData( REMOTE_REC_REF    remote_ref,
                 char              *calendar,
                 XTM_RT_DIRECTION  direction,
                 Boolean           can_upload,
                 Boolean           can_download );

static void
  startRemoteAction( REMOTE_REC_REF    remote_ref,
                     char              *host,
                     char              *uid,
                     char              *password,
                     char              *connection_type,
                     char              *remote_script,
                     XTM_RT_DIRECTION  direction );

static void
  uploadCB( Widget          widget,
            REMOTE_REC_REF  remote_ref,
            XtPointer       call_data );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

XTM_RT_HANDLE
  xtmRtInitialize( XTM_GL_BASE_DATA_REF  appl_data_ref,
                   Widget                parent,
                   Boolean               appl_modal,
                   XTM_RT_ACTION_CB      actionCB,
                   void                  *user_data )
{

  /* Variables. */
  REMOTE_REC_REF  remote_ref;


  /* Code. */

  /* Create and initialize our private data. */
  remote_ref = SysNew( REMOTE_REC );
  if( remote_ref == NULL )
    return( NULL );

  remote_ref -> appl_data_ref = appl_data_ref;
  remote_ref -> appl_modal    = appl_modal;
  remote_ref -> actionCB      = actionCB;
  remote_ref -> user_data     = user_data;


  /* Create the remote window. */
  remote_ref -> remoteW = createRemoteWindow( remote_ref, parent );

  if( remote_ref -> remoteW == NULL ) {
    SysFree( remote_ref );

    return( NULL );
  }


  return( (XTM_RT_HANDLE) remote_ref );

} /* xtmRtInitialize */


/*----------------------------------------------------------------------*/

void
  xtmRtDestroy( XTM_RT_HANDLE  remote_handle )
{

  /* Variables. */
  REMOTE_REC_REF  remote_ref;


  /* Code. */

  if( remote_handle == NULL )
    return;

  /* Our private data. */
  remote_ref = (REMOTE_REC_REF) remote_handle;


  /* Destroy the window. */
  XtDestroyWidget( remote_ref -> remoteW );


  return;

} /* xtmRtDestroy */


/*----------------------------------------------------------------------*/

void 
  xtmRtDoRemote( XTM_RT_HANDLE     remote_handle,
                 XTM_RT_DIRECTION  direction,
                 char              *calendar,
                 Boolean           can_upload,
                 Boolean           can_download )
{

  /* Variables. */
  REMOTE_REC_REF  remote_ref;


  /* Code. */

  if( remote_handle == NULL )
    return;

  /* Our private data. */
  remote_ref = (REMOTE_REC_REF) remote_handle;

  /* Fill in the actual data. */
  setRemoteData( remote_ref, calendar, direction, can_upload, can_download );

  /* Make sure the window is visible. */
  XtManageChild( remote_ref -> remoteW );


  return;

} /* xtmRtDoRemote */


/*----------------------------------------------------------------------*/

static Widget
  createRemoteWindow( REMOTE_REC_REF  remote_ref,
                      Widget          parent )
{

  /* Variables. */
  Arg       args[ 10 ];
  Cardinal  n;
  Widget    remoteFd;
  Widget    workFo;
  Widget    dataLocalW[ 19 ];

  static XIT_PUSH_STRUCT button_def[] = {
    { "PwdChangePb", "", "", True, NULL }
  };

  static XIT_TEXT_STRUCT text_buffer[] = {
    { "HostTx",   NULL, 1, True },
    { "UidTx",    NULL, 1, True },
    { "ConnTx",   NULL, 1, True },
    { "ScriptTx", NULL, 1, True },
  };

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "",   uploadCB,   NULL },
    { "",   downloadCB, NULL },
    { "",   closeCB,    NULL },
    { "",   helpCB,     NULL },
  };


  /* Code. */

  /* Set message strings. */
  button_def[ 0 ].title = msgGetText( MXDI_CHANGE_BUTTON );

  action_buttons[ 0 ].label = msgGetText( MXDI_UPLOAD_BUTTON );
  action_buttons[ 0 ].data  = remote_ref;
  action_buttons[ 1 ].label = msgGetText( MXDI_DOWNLOAD_BUTTON );
  action_buttons[ 1 ].data  = remote_ref;
  action_buttons[ 2 ].label = msgGetText( MXDI_CLOSE_BUTTON );
  action_buttons[ 2 ].data  = remote_ref;
  action_buttons[ 3 ].label = msgGetText( MXDI_HELP_BUTTON );
  action_buttons[ 3 ].data  = remote_ref;

  /* Create a form dialog with buttons. */
  remoteFd = xitCreateFormDialog( parent, "RemoteFd",
                                  1, 0,
                                  action_buttons,
                                  XtNumber( action_buttons ) );

  XtAddCallback( remoteFd, XmNdestroyCallback, 
                 (XtCallbackProc) destroyCB, (XtPointer) remote_ref );

  if( remote_ref -> appl_modal ) {
    n = 0;
    XtSetArg( args[ n ], XmNdialogStyle, XmDIALOG_APPLICATION_MODAL ); n++;
    XtSetValues( remoteFd, args, n );
  }


  /* Container for the contents of the window. */
  workFo = XtNameToWidget( remoteFd, "RemoteFdFo" );


  /* What is this all about? */
  explainLa = xitCreateLabel( workFo, "ExplainLa", 
                              msgGetText( MXDI_REMOTE_EXPLAIN_LABEL ), 
                              XmALIGNMENT_CENTER );

  /* Information for the remote calendar. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmVERTICAL ); n++;
  XtSetArg( args[ n ], XmNpacking, XmPACK_COLUMN ); n++;
  XtSetArg( args[ n ], XmNnumColumns, 2 ); n++;
  remoteRc = XmCreateRowColumn( workFo, "RemoteRc", args, n );


  /* Name of host. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  hostRc = XmCreateRowColumn( remoteRc, "HostRc", args, n );

  hostLa = xitCreateLabel( hostRc, "HostLa", 
                           msgGetText( MXDI_HOSTNAME_LABEL ), -1 );

  hostTx = xitCreateTextCols( hostRc, &text_buffer[ 0 ], 20 );


  /* Name of renote user ID. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  uidRc = XmCreateRowColumn( remoteRc, "UidRc", args, n );

  uidLa = xitCreateLabel( uidRc, "UidLa", 
                          msgGetText( MXDI_USERID_LABEL ), -1 );

  uidTx = xitCreateTextCols( uidRc, &text_buffer[ 1 ], 20 );


  /* Name of renote user password. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  pwdRc = XmCreateRowColumn( remoteRc, "PwdRc", args, n );

  pwdLa = xitCreateLabel( pwdRc, "PwdLa", 
                          msgGetText( MXDI_PASSWORD_LABEL ), -1 );

  pwdStringLa = xitCreateLabel( pwdRc, "PwdStringLa", "               ", -1 );
  pwdChangePb = xitCreatePushButton( pwdRc, &button_def[ 0 ] );

  XtAddCallback( pwdChangePb, XmNactivateCallback, 
                 (XtCallbackProc) changePasswordCB, (XtPointer) remote_ref );


  /* Type of connection. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  connRc = XmCreateRowColumn( remoteRc, "ConnRc", args, n );

  connLa = xitCreateLabel( connRc, "ConnLa", 
                           msgGetText( MXDI_CONNECTION_TYPE_LABEL ), -1 );

  connTx = xitCreateTextCols( connRc, &text_buffer[ 2 ], 10 );


  /* Script to run. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  scriptRc = XmCreateRowColumn( remoteRc, "ScriptRc", args, n );

  scriptLa = xitCreateLabel( scriptRc, "ScriptLa", 
                             msgGetText( MXDI_REMOTE_SCRIPT_LABEL ), -1 );

  scriptTx = xitCreateTextCols( scriptRc, &text_buffer[ 3 ], 20 );


  /* 'Shit' widgets. */
  hidden1La = xitCreateLabel( remoteRc, "", " ", -1 );

  n = 0;
  XtSetArg( args[ n ], XmNmappedWhenManaged, False ); n++;
  XtSetValues( hidden1La, args, n );


  /* Put the elements together. */
  xitAttachWidget( explainLa,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_FORM, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( remoteRc,
                   XmATTACH_WIDGET, explainLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,      XmATTACH_NONE, NULL );


  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetValues( remoteRc, args, n );


  /* Manage the widgets. */
  xitManageChildren( dataLocalW,  XtNumber( dataLocalW ) );

  /* Set the size of the window. */
  xitSetSizeFormDialog( remoteFd, True );


  return( remoteFd );

} /* createRemoteWindow */


/*----------------------------------------------------------------------*/

static void
  doAction( REMOTE_REC_REF    remote_ref,
            XTM_RT_DIRECTION  direction )
{

  /* Variables. */
  char  host[ XTM_GL_MAX_HOSTNAME + 1 ];
  char  uid[ XTM_GL_MAX_USERID + 1 ];
  char  password[ XTM_GL_MAX_PASSWORD + 1 ];
  char  connection_type[ XTM_GL_MAX_EXTERN_TYPE + 1 ];
  char  remote_script[ XTM_GL_MAX_DIR + 1 ];


  /* Code. */

  /* Call callback function? */
  if( remote_ref -> actionCB != NULL )
    (* remote_ref -> actionCB )( XTM_RT_REASON_APPLY,
                                 remote_ref -> user_data );

  /* Fetch remote data. */
  fetchRemoteData( remote_ref, host, uid, password,
                   connection_type, remote_script );

  /* Start the remote action. */
  startRemoteAction( remote_ref, host, uid, password,
                     connection_type, remote_script, direction );


  return;

} /* doAction */


/*----------------------------------------------------------------------*/

static void
  fetchRemoteData( REMOTE_REC_REF    remote_ref,
                   char              *host,
                   char              *uid,
                   char              *password,
                   char              *connection_type,
                   char              *remote_script )
{

  /* Variables. */
  char    format[ 50 ];
  char    *char_ref;
  Widget  mainW;
  Widget  tempW;


  /* Code. */

  mainW  = XtNameToWidget( remote_ref -> remoteW, "RemoteFdFo" );


  /* External host. */
  sprintf( format, "%%%ds", XTM_GL_MAX_HOSTNAME );

  tempW = XtNameToWidget( mainW, "RemoteRc.HostRc.HostTx" );
  char_ref = xitStringGetText( tempW );

  sscanf( char_ref, format, host );
  SysFree( char_ref );


  /* External UID. */
  sprintf( format, "%%%ds", XTM_GL_MAX_USERID );

  tempW = XtNameToWidget( mainW, "RemoteRc.UidRc.UidTx" );
  char_ref = xitStringGetText( tempW );

  sscanf( char_ref, format, uid );
  SysFree( char_ref );


  /* Password. */
  strcpy( password, remote_ref -> extern_pwd );


  /* External type. */
  sprintf( format, "%%%ds", XTM_GL_MAX_EXTERN_TYPE );

  tempW = XtNameToWidget( mainW, "RemoteRc.ConnRc.ConnTx" );
  char_ref = xitStringGetText( tempW );

  sscanf( char_ref, format, connection_type );
  SysFree( char_ref );


  /* Script to call. */
  sprintf( format, "%%%ds", XTM_GL_MAX_DIR );

  tempW = XtNameToWidget( mainW, "RemoteRc.ScriptRc.ScriptTx" );
  char_ref = xitStringGetText( tempW );

  sscanf( char_ref, format, remote_script );
  SysFree( char_ref );


  return;

} /* fetchRemoteData */


/*----------------------------------------------------------------------*/

static void
  setRemoteData( REMOTE_REC_REF    remote_ref,
                 char              *calendar,
                 XTM_RT_DIRECTION  direction,
                 Boolean           can_upload,
                 Boolean           can_download )
{

  /* Variables. */
  Boolean                 ok;
  int                     index;
  char                    buffer[ 250 ];
  Arg                     args[ 10 ];
  Cardinal                n;
  Widget                  mainW;
  Widget                  tempW;
  XTM_CD_CAL_INFO         cal_info;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  custom_data_ref = remote_ref -> appl_data_ref -> custom_data;
  mainW           = XtNameToWidget( remote_ref -> remoteW, "RemoteFdFo" );


  /* Information about the calendar. */
  ok = xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle, 
                          calendar,
                          &cal_info, NULL );
  if( ! ok )
    return;

  /* Save the calendar. */
  strcpy( remote_ref -> calendar, calendar );

  /* Keep the password local. */
  strcpy( remote_ref -> extern_pwd, cal_info.extern_pwd );


  /* Calendar name. */
  sprintf( buffer, msgGetText( MXDI_REMOTE_TITLE ), cal_info.short_name );

  n = 0;
  XtSetArg( args[ n ], XmNtitle, buffer ); n++;
  XtSetValues( XtParent( remote_ref -> remoteW ), args, n );


  /* External host. */
  tempW = XtNameToWidget( mainW, "RemoteRc.HostRc.HostTx" );
  XmTextSetString( tempW, cal_info.extern_host );


  /* External user ID. */
  tempW = XtNameToWidget( mainW, "RemoteRc.UidRc.UidTx" );
  XmTextSetString( tempW, cal_info.extern_uid );


  /* External user password. */
  strcpy( buffer, remote_ref -> extern_pwd );
  for( index = 0; index < strlen( buffer ); index++ )
    buffer[ index ] = '-';
    
  tempW = XtNameToWidget( mainW, "RemoteRc.PwdRc.PwdStringLa" );
  xitStringSetLabel( tempW, buffer );


  /* Connection type. */
  tempW = XtNameToWidget( mainW, "RemoteRc.ConnRc.ConnTx" );
  XmTextSetString( tempW, cal_info.extern_type );


  /* Script to call. */
  tempW = XtNameToWidget( mainW, "RemoteRc.ScriptRc.ScriptTx" );
  XmTextSetString( tempW, custom_data_ref -> remote_script );


  /* Upload or download? */
  if( direction == XTM_RT_UPLOAD )
    tempW = XtNameToWidget( remote_ref -> remoteW, "Bu1" );
  else
    tempW = XtNameToWidget( remote_ref -> remoteW, "Bu2" );

  n = 0;
  XtSetArg( args[ n ], XmNshowAsDefault, True ); n++;
  XtSetValues( tempW, args, n );


  /* Can we upload and/or download? */
  tempW = XtNameToWidget( remote_ref -> remoteW, "Bu1" );
  if( can_upload )
    XtSetSensitive( tempW, True );
  else
    XtSetSensitive( tempW, False );

  tempW = XtNameToWidget( remote_ref -> remoteW, "Bu2" );
  if( can_download )
    XtSetSensitive( tempW, True );
  else
    XtSetSensitive( tempW, False );


  return;

} /* setRemoteData */


/*----------------------------------------------------------------------*/

static void
  startRemoteAction( REMOTE_REC_REF    remote_ref,
                     char              *host,
                     char              *uid,
                     char              *password,
                     char              *connection_type,
                     char              *remote_script,
                     XTM_RT_DIRECTION  direction )
{

  /* Variables. */
  Boolean                 ok;
  int                     pid;
  int                     status;
  Display                 *display = NULL;
  FILE                    *file_ref;
  XTM_CD_CAL_INFO         cal_info;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  custom_data_ref = remote_ref -> appl_data_ref -> custom_data;
  display         = XtDisplay( remote_ref -> remoteW );


  /* Does the executable file exist? */
  status = access( remote_script, (R_OK | X_OK) );

  if( status != 0 ) {
    xitErMessage( remote_ref -> remoteW, XIT_ER_ERROR,
                  module_name, "startRemoteAction",
                  msgGetText( MXDI_PROCESS_NO_FILE ) );
    return;
  }


  /* Information about the calendar. */
  ok = xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle, 
                          remote_ref -> calendar,
                          &cal_info, NULL );
  if( ! ok )
    return;


  /* Start the process. */
  pid = fork();

  switch( pid ) {

    /* Child running. */
    case 0:
      if( display != NULL )
        close( ConnectionNumber( display ) );

      /* Start the remote command. */
      file_ref = (FILE *) popen( remote_script, "w" );
      if( file_ref == NULL )
        exit( 1 );

      /* Send the parameters to be read by the script. */
      if( direction == XTM_RT_UPLOAD )
        fprintf( file_ref, "%s\n", "upload" );
      else
        fprintf( file_ref, "%s\n", "download" );

      fprintf( file_ref, "%s\n", cal_info.short_name );
      fprintf( file_ref, "%s\n", cal_info.directory );
      fprintf( file_ref, "%s\n", host );
      fprintf( file_ref, "%s\n", uid );
      fprintf( file_ref, "%s\n", password );
      fprintf( file_ref, "%s\n", connection_type );

      /* That's it. */
#ifdef XD_HAS_NO_WAITPID
      fclose( file_ref );
#else
      pclose( file_ref );
#endif
      exit( 0 );

    /* Error in fork. */
    case -1:
      xitErMessage( remote_ref -> remoteW, XIT_ER_ERROR,
                    module_name, "startRemoteAction",
                    msgGetText( MXDI_PROCESS_CANNOT_FORK ) );
      return;

    /* Parent */
    default:
      break;

  } /* switch */


  /* Tell the user what we are doing since this might take time. */
  (void) xitCreateInformationDialog(
           remote_ref -> remoteW, "InformationDialog", 
           msgGetText( MXDI_INFORMATION_LABEL ),
           msgGetText( MXDI_REMOTE_OPERATION_RUNNING_MESSAGE ),
           NULL, NULL );


  return;

} /* startRemoteAction */


/*----------------------------------------------------------------------*/

static void 
  changePasswordCB( Widget          widget,
                    REMOTE_REC_REF  remote_ref,
                    XtPointer       call_data )
{

  /* Variables. */
  Arg       args[ 10 ];
  Cardinal  n;
  Pixel     bg;
  Widget    pwdFd;
  Widget    workFo;
  Widget    dataLocalW[ 2 ];

  static XIT_TEXT_STRUCT text_buffer[] = {
    { "PwdPasswordTx", NULL, 1, True },
  };

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "",   pwdOkCB, NULL },
    { NULL, NULL,    NULL },
    { "",   NULL,    NULL },
  };


  /* Code. */

  /* Set message strings. */
  action_buttons[ 0 ].label = msgGetText( MXDI_OK_BUTTON );
  action_buttons[ 0 ].data  = remote_ref;
  action_buttons[ 2 ].label = msgGetText( MXDI_CANCEL_BUTTON );
  action_buttons[ 2 ].data  = remote_ref;

  /* Create a form dialog with buttons. */
  pwdFd = xitCreateFormDialog( remote_ref -> remoteW, "PwdFd",
                               1, 0,
                               action_buttons,
                               XtNumber( action_buttons ) );

  n = 0;
  XtSetArg( args[ n ], XmNtitle, " " ); n++;
  XtSetValues( XtParent( pwdFd ), args, n );

  n = 0;
  XtSetArg( args[ n ], XmNdialogStyle, XmDIALOG_APPLICATION_MODAL ); n++;
  XtSetValues( pwdFd, args, n );


  /* Container for the contents of the window. */
  workFo = XtNameToWidget( pwdFd, "PwdFdFo" );


  /* What is this all about? */
  pwdWhatLa = xitCreateLabel( workFo, "PwdWhatLa", 
                              msgGetText( MXDI_REMOTE_PWD_EXPLAIN_LABEL ),
                              -1 );

  pwdPasswordTx = xitCreateTextCols( workFo, &text_buffer[ 0 ], 20 );


  /* Do not display characters. */
  n = 0;
  XtSetArg( args[ n ], XmNbackground, &bg ); n++;
  XtGetValues( pwdPasswordTx, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNforeground, bg ); n++;
  XtSetValues( pwdPasswordTx, args, n );


  /* Put the elements together. */
  xitAttachWidget( pwdWhatLa,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( pwdPasswordTx,
                   XmATTACH_WIDGET, pwdWhatLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,      XmATTACH_NONE, NULL );


  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetValues( pwdWhatLa,     args, n );
  XtSetValues( pwdPasswordTx, args, n );

  /* Manage the widgets. */
  xitManageChildren( dataLocalW, XtNumber( dataLocalW ) );

  /* Set the size of the window. */
  xitSetSizeFormDialog( pwdFd, True );

  remote_ref -> remotePwdW = pwdFd;
  
  /* Make sure the window is visible. */
  XtManageChild( remote_ref -> remotePwdW );

  /* Set focus. */
  xitSetFocus( pwdFd, pwdPasswordTx );


  return;

} /* changePasswordCB */


/*----------------------------------------------------------------------*/

static void
  closeCB( Widget          widget,
           REMOTE_REC_REF  remote_ref,
           XtPointer       call_data )
{

  /* Code. */

  XtUnmanageChild( remote_ref -> remoteW );


  return;

} /* closeCB */


/*----------------------------------------------------------------------*/

static void 
  destroyCB( Widget          widget,
             REMOTE_REC_REF  remote_ref,
             XtPointer       call_data )
{

  /* Code. */

  /* Call callback function? */
  if( remote_ref -> actionCB != NULL )
    (* remote_ref -> actionCB )( XTM_RT_REASON_DESTROY,
                                 remote_ref -> user_data );

  SysFree( remote_ref );


  return;

} /* destroyCB */


/*----------------------------------------------------------------------*/

static void
  downloadCB( Widget          widget,
              REMOTE_REC_REF  remote_ref,
              XtPointer       call_data )
{

  /* Code. */

  doAction( remote_ref, XTM_RT_DOWNLOAD );


  return;

} /* downloadCB */


/*----------------------------------------------------------------------*/

static void 
  helpCB( Widget          widget,
          REMOTE_REC_REF  remote_ref,
          XtPointer       call_data )
{

  /* Code. */

  xtmHlDisplayHelp( remote_ref -> appl_data_ref -> info_handle,
                    XTM_HL_WINDOW_INDEX,
                    remote_window_id, "" );

  return;

} /* helpCB */


/*----------------------------------------------------------------------*/

static void
  pwdOkCB( Widget          widget,
           REMOTE_REC_REF  remote_ref,
           XtPointer       call_data )
{

  /* Variables. */
  Boolean                 ok;
  int                     index;
  char                    buffer[ 50 ];
  char                    format[ 50 ];
  char                    *char_ref;
  Widget                  mainW;
  Widget                  tempW;
  XTM_CD_CAL_INFO         cal_info;
  XTM_CD_INCL_CALS        cal_incl;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  custom_data_ref = remote_ref -> appl_data_ref -> custom_data;
  mainW           = XtNameToWidget( remote_ref -> remotePwdW, "PwdFdFo" );


  /* Password. */
  sprintf( format, "%%%ds", XTM_GL_MAX_PASSWORD );

  tempW = XtNameToWidget( mainW, "PwdPasswordTx" );
  char_ref = xitStringGetText( tempW );

  sscanf( char_ref, format, remote_ref -> extern_pwd );
  SysFree( char_ref );


  /* Save the password in the calendar record. */
  ok = xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle, 
                          remote_ref -> calendar,
                          &cal_info, &cal_incl );
  if( ok ) {
    strcpy( cal_info.extern_pwd, remote_ref -> extern_pwd );

    (void) xtmCdChangeEntry( custom_data_ref -> cal_db_handle,
                             remote_ref -> calendar,
                             &cal_info, &cal_incl );
  }


  /* Display. */
  mainW = XtNameToWidget( remote_ref -> remoteW, "RemoteFdFo" );

  strcpy( buffer, remote_ref -> extern_pwd );
  for( index = 0; index < strlen( buffer ); index++ )
    buffer[ index ] = '-';
    
  tempW = XtNameToWidget( mainW, "RemoteRc.PwdRc.PwdStringLa" );
  xitStringSetLabel( tempW, buffer );


  /* Remove window. */
  XtDestroyWidget( remote_ref -> remotePwdW );


  return;

} /* pwdOkCB */


/*----------------------------------------------------------------------*/

static void
  uploadCB( Widget          widget,
            REMOTE_REC_REF  remote_ref,
            XtPointer       call_data )
{

  /* Code. */

  doAction( remote_ref, XTM_RT_UPLOAD );


  return;

} /* uploadCB */
