/*----------------------------------------------------------------------------
--
--  Module:           xtmPrEntries
--
--  Project:          Xdiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Print a selection of days.
--
--  Filename:         xtmPrEntries.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1991-01-05
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmPrEntries.c, Version: 1.1, Date: 95/02/18 15:52:39";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>

#include <X11/Intrinsic.h>
#include <X11/Shell.h>

#include <Xm/Protocols.h>

#include <Xm/Xm.h>
#include <Xm/CascadeB.h>
#include <Xm/Form.h>
#include <Xm/Frame.h>
#include <Xm/List.h>
#include <Xm/PushB.h>
#include <Xm/RowColumn.h>
#include <Xm/Text.h>
#include <Xm/ToggleB.h>

#include "System.h"
#include "Message.h"
#include "TimDate.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmCalDb.h"
#include "xtmDbFilter.h"
#include "xtmDbInclude.h"
#include "xtmDbMisc.h"
#include "xtmDbTools.h"
#include "xtmFields.h"
#include "xtmFormat.h"
#include "xtmHelp.h"
#include "xtmIcons.h"
#include "xtmTools.h"
#include "xitError.h"
#include "xitFieldSel.h"
#include "xitTools.h"
#include "XmUbTimeB.h"
#include "xtmPrEntries.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Windows in the cache? */
#define  MAX_CACHE_ENTRIES   3


/* Local widgets in the print window. */
#define menuBr             dataLocalW[  0 ]
#define outStyleFs         dataLocalW[  1 ]
#define outStyleLa         dataLocalW[  2 ]
#define outStyleRc         dataLocalW[  3 ]
#define outputFs           dataLocalW[  4 ]
#define outputLa           dataLocalW[  5 ]
#define outputRc           dataLocalW[  6 ]
#define paperFs            dataLocalW[  7 ]
#define paperLa            dataLocalW[  8 ]
#define paperRc            dataLocalW[  9 ]
#define prDateLa           dataLocalW[ 10 ]
#define prDayRc            dataLocalW[ 11 ]
#define prNameLa           dataLocalW[ 12 ]
#define prNameTx           dataLocalW[ 13 ]
#define prParamFr          dataLocalW[ 14 ]
#define prParamRc          dataLocalW[ 15 ]
#define prProfileFr        dataLocalW[ 16 ]
#define prProfileRc        dataLocalW[ 17 ]
#define prSelectRc         dataLocalW[ 18 ]
#define prTypeFs           dataLocalW[ 19 ]
#define prTypeLa           dataLocalW[ 20 ]
#define printerLi          dataLocalW[ 21 ]
#define rangeTb            dataLocalW[ 22 ]
#define sheetFs            dataLocalW[ 23 ]
#define sheetLa            dataLocalW[ 24 ]
#define sheetRc            dataLocalW[ 25 ]
#define shLineLa           dataLocalW[ 26 ]
#define shLineRc           dataLocalW[ 27 ]
#define shLineTx           dataLocalW[ 28 ]
#define whatPrintersLa     dataLocalW[ 29 ]
#define yourOwnFr          dataLocalW[ 30 ]
#define yourOwnLa          dataLocalW[ 31 ]


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Record while printing entries. */
typedef struct {

  /* Lines to display. */
  int  display_lines;

  /* The print view is for this database. */
  char  db_name[ XTM_GL_MAX_CAL_NAME + 1 ];

  /* Name of selected printer. */
  char  pr_name[ 30 ];

  /* Type for selected printer. */
  char  pr_type[ 30 ];

  /* Look for entries with these tags. */
  char  search_tags[ 200 ];

  /* The printer window. */
  Widget  printerW;

  /* Print between dates. */
  TIM_TIME_REF  from_date;
  TIM_TIME_REF  to_date;

  /* Include handle. */
  XTM_DI_HANDLE  include_handle;

  /* Filter record. */
  XTM_DM_FILTER_REC  filter_rec;

  /* Filter handle. */
  XTM_FI_HANDLE  filter_handle;

  /* Application wide data. */
  XTM_GL_BASE_DATA_REF  appl_data_ref;

  /* Callback to inform our creator of specific actions. */
  void              *user_data;
  XTM_PR_ACTION_CB  actionCB;

} PRINTER_REC, *PRINTER_REC_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmPrintEntries";

/* IDs for the help windows. */
static char  *print_window_id = "Print";

/* Cache entries. */
static Boolean          init_cache = True;
static PRINTER_REC_REF  cache_entries[ MAX_CACHE_ENTRIES ];


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void
  closeCB( Widget           widget,
           PRINTER_REC_REF  printer_ref,
           XtPointer        call_data );

static Widget
  createPrintWindow( PRINTER_REC_REF  printer_ref,
                     Widget           parent );

static void 
  ctrlMenuCB( Widget                     widget,
              PRINTER_REC_REF            printer_ref,
              XmRowColumnCallbackStruct  *call_data );

static void 
  destroyCB( Widget           widget,
             PRINTER_REC_REF  printer_ref,
             XtPointer        call_data );

static void 
  filterApplyCB( XTM_FI_REASON      reason,
                 XTM_DM_FILTER_REC  *filter_ref,
                 void               *user_data );

static void 
  filterCB( Widget           widget,
            PRINTER_REC_REF  printer_ref,
            XtPointer        call_data );

static void
  includeDbApplyCB( XTM_DI_REASON         reason,
                    XTM_CD_INCL_CALS_REF  new_db_incl_ref,
                    void                  *user_data );

static void 
  infoCB( Widget                     widget,
          PRINTER_REC_REF            printer_ref,
          XmRowColumnCallbackStruct  *call_data );

static void
  loadPrinterList( PRINTER_REC_REF  printer_ref );

static void
  printCB( Widget           widget,
           PRINTER_REC_REF  printer_ref,
           XtPointer        call_data );

static void 
  printerSelectCB( Widget                widget,
                   PRINTER_REC_REF       printer_ref,
                   XmListCallbackStruct  *call_data );

static void
  saveRecordToFile( XTM_GL_BASE_DATA_REF  appl_data_ref,
                    XTM_DB_ALL_ENTRY_REF  entry_ref,
                    FILE                  *file_ref,
                    int                   display_lines );

static void
  startPrintJob( PRINTER_REC_REF  printer_ref );

static void
  startPrintProcess( PRINTER_REC_REF  printer_ref,
                     char             *filename );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

XTM_PR_HANDLE
  xtmPrInitialize( XTM_GL_BASE_DATA_REF  appl_data_ref,
                   Widget                toplevel,
                   char                  *db_name,
                   XTM_PR_ACTION_CB      actionCB,
                   void                  *user_data )
{

  /* Variables. */
  Boolean                 ok;
  int                     index;
  int                     length;
  PRINTER_REC_REF         printer_ref;
  XTM_CD_CAL_INFO         db_info;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  custom_data_ref = appl_data_ref -> custom_data;


  /* Fetch database information. */
  ok = xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle,
                          db_name,
                          &db_info, NULL );
  if( ! ok )
    return( False );


  /* Minium permissions are read. */
  if( ! flagIsSet( db_info.operations, XTM_DB_FLAG_MODE_READ ) )
    return( NULL );


  /* Initialize the cache? */
  if( init_cache ) {
    for( index = 0; index < MAX_CACHE_ENTRIES; index++ )
      cache_entries[ index ] = NULL;

    init_cache = False;
  }


  /* Anything in the cache? */
  printer_ref = NULL;

  for( index = 0; index < MAX_CACHE_ENTRIES; index++ ) {
    if( cache_entries[ index ] != NULL ) {
      printer_ref = cache_entries[ index ];
      cache_entries[ index ] = NULL;

      break;
    }
  }

  /* Create the showrecord? */
  if( printer_ref == NULL ) {

    /* Create and initialize our private data. */
    printer_ref = SysNew( PRINTER_REC );
    if( printer_ref == NULL )
      return( NULL );

    printer_ref -> printerW = NULL;

  } /* if */

  printer_ref -> appl_data_ref    = appl_data_ref;
  printer_ref -> search_tags[ 0 ] = '\0';
  printer_ref -> filter_handle    = NULL;
  printer_ref -> include_handle   = NULL;
  printer_ref -> actionCB         = actionCB;
  printer_ref -> user_data        = user_data;

  printer_ref -> filter_rec.flags            = XTM_DM_FLAG_ALL;
  printer_ref -> filter_rec.text_string[ 0 ] = '\0';
  printer_ref -> filter_rec.tag_string[  0 ] = '\0';

  length = strlen( custom_data_ref -> entry_tags_filter );

  if( length > 0 && length < XTM_DM_MAX_TAG_SEARCH ) {
    strcpy( printer_ref -> filter_rec.tag_string,
            custom_data_ref -> entry_tags_filter );
    flagSet( printer_ref -> filter_rec.flags, XTM_DM_FLAG_SEARCH_TAG );
  }

  if( strlen( custom_data_ref -> entry_tags_menu ) < 200 )
    strcpy( printer_ref -> search_tags, custom_data_ref -> entry_tags_menu );


  /* Create a shadow calendar database. */
  ok = xtmCdCreateShadowEntry( custom_data_ref -> cal_db_handle,
                               db_name, printer_ref -> db_name );
  if( ! ok ) {
    SysFree( printer_ref );
    return( NULL );
  }


  /* Create the printer window. */
  if( printer_ref -> printerW == NULL ) {
    printer_ref -> printerW = createPrintWindow( printer_ref, toplevel );

    if( printer_ref -> printerW == NULL ) {
      xtmCdFreeShadowEntry( custom_data_ref -> cal_db_handle,
                            printer_ref -> db_name );
      SysFree( printer_ref );

      return( NULL );
    }

    /* Take the printers and write them into the list. */
    loadPrinterList( printer_ref );
  } /* if */


  return( (XTM_PR_HANDLE) printer_ref );

} /* xtmPrInitialize */


/*----------------------------------------------------------------------*/

void
  xtmPrDestroy( XTM_PR_HANDLE  printer_handle )
{

  /* Variables. */
  PRINTER_REC_REF  printer_ref;


  /* Code. */

  if( printer_handle == NULL )
    return;

  /* Our private data. */
  printer_ref = (PRINTER_REC_REF) printer_handle;


  /* Destroy the window. */
  closeCB( NULL, printer_ref, NULL );


  return;

} /* xtmPrDestroy */


/*----------------------------------------------------------------------*/

void
  xtmPrEmptyCache()
{

  /* Variables. */
  int  index;


  /* Code. */

  if( ! init_cache )
    return;

  for( index = 0; index < MAX_CACHE_ENTRIES; index++ ) {
    if( cache_entries[ index ] != NULL )
      XtDestroyWidget( cache_entries[ index ] -> printerW );

    cache_entries[ index ] = NULL;
  }


  return;

} /* xtmPrEmptyCache */


/*----------------------------------------------------------------------*/

void 
  xtmPrDisplayPrinter( XTM_PR_HANDLE      printer_handle,
                       TIM_TIME_REF       from_date,
                       TIM_TIME_REF       to_date,
                       int                display_lines,
                       XTM_DM_FILTER_REC  *filter_ref )
{

  /* Variables. */
  char                    buffer[ 256 ];
  Widget                  mainW;
  Widget                  tempW;
  PRINTER_REC_REF         printer_ref;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  printer_ref = (PRINTER_REC_REF) printer_handle;

  appl_data_ref   = printer_ref  -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;


  mainW = XtNameToWidget( printer_ref -> printerW, "PrintTlBase.PrintTlFo" );

  /* Date range. */
  tempW = XtNameToWidget( mainW, "PrDayRc.RangeTb" );
  XmUbTimeBoxSetStartDate( tempW, from_date );
  XmUbTimeBoxSetEndDate( tempW, to_date );


  /* Lines to display. */
  if( display_lines > 0 ) {
    sprintf( buffer, "%d ", display_lines );

    tempW = XtNameToWidget( mainW, "ShLineRc.ShLineTx" );
    XmTextSetString( tempW, buffer );
  }

  /* Save the filter. */
  if( filter_ref != NULL ) {
    memcpy( (void *) &printer_ref -> filter_rec,
            (void *) filter_ref,
            sizeof( XTM_DM_FILTER_REC ) );
  } else {
    printer_ref -> filter_rec.flags            = XTM_DM_FLAG_ALL;
    printer_ref -> filter_rec.text_string[ 0 ] = '\0';
    printer_ref -> filter_rec.tag_string[  0 ] = '\0';
  }


  /* Display the window. */
  XtPopup( printer_ref -> printerW, XtGrabNone );


  /* Make sure the editor window is visable. */
  XRaiseWindow( XtDisplay( printer_ref -> printerW ), 
                XtWindow(  printer_ref -> printerW ) );

  XtMapWidget( printer_ref -> printerW );


  return;

} /* xtmPrDisplayPrinter */


/*----------------------------------------------------------------------*/

void
  xtmPrSetCalendar( XTM_PR_HANDLE  printer_handle,
                    char           *cal_name )
{

  /* Variables. */
  Boolean                 ok;
  char                    buffer[ 100 ];
  char                    new_cal_name[ 21 ];
  Arg                     args[ 10 ];
  Cardinal                n;
  PRINTER_REC_REF         printer_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;
  XTM_CD_CAL_INFO         db_info;


  /* Code. */

  if( printer_handle == NULL )
    return;

  /* Our private data. */
  printer_ref = (PRINTER_REC_REF) printer_handle;


  custom_data_ref = printer_ref -> appl_data_ref -> custom_data;


  /* Create a shadow calendar database. */
  ok = xtmCdCreateShadowEntry( custom_data_ref -> cal_db_handle,
                               cal_name, new_cal_name );
  if( ! ok )
    return;


  /* Free our old shadow calendar. */
  xtmCdFreeShadowEntry( custom_data_ref -> cal_db_handle,
                        printer_ref -> db_name );

  strcpy( printer_ref -> db_name, new_cal_name );


  /* The new calendar database. */
  (void) xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle,
                            printer_ref -> db_name,
                            &db_info, NULL );


  /* Window title and icon name. */
  sprintf( buffer, "%s %s", 
           db_info.short_name, msgGetText( MXDI_PRINTING_TITLE ) );

  n = 0;
  XtSetArg( args[ n ], XmNtitle, buffer ); n++;
  XtSetValues( printer_ref -> printerW, args, n );

  sprintf( buffer, "%s", db_info.short_name );

  n = 0;
  XtSetArg( args[ n ], XmNiconName, buffer ); n++;
  XtSetValues( printer_ref -> printerW, args, n );


  return;

} /* xtmPrSetIncludedDbInfo */


/*----------------------------------------------------------------------*/

static Widget
  createPrintWindow( PRINTER_REC_REF  printer_ref,
                     Widget           parent )
{

  /* Variables. */
  int                     index;
  char                    buffer[  100 ];
  char                    *char_ref;
  Arg                     args[ 10 ];
  Cardinal                n;
  Widget                  dataLocalW[ 32 ];
  Widget                  menuCasc[ 3 ];
  Widget                  menuFileBu[ 1 ];
  Widget                  menuHelpBu[ 6 ];
  Widget                  menuCtrlBu[ 2 ];
  Widget                  printTl;
  Widget                  pulldownMenu[ 3 ];
  Widget                  paramTb[ 2 ];
  Widget                  tempW;
  Widget                  toplevel;
  Widget                  workFo;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;
  XTM_CD_CAL_INFO         db_info;

  static char  *pull_downs[] = { "pdown1", "pdown2", "pdown3" };

  static XIT_TEXT_STRUCT text_field_def[] = {
    { "ShLineTx",   NULL, 1, True },
    { "PrNameTx",   NULL, 1, True },
  };

  static XIT_CASCADE_STRUCT menupane[] = {
    { "", "", "FilePane" },
    { "", "", "ControlPane" },
    { "", "", "HelpPane" }
  };

  static XIT_MENU_BUTTON_STRUCT file_casc[] = {
    { "", "", NULL, "CloseBu", True, False, False },
  };

  static XIT_MENU_BUTTON_STRUCT ctrl_casc[] = {
    { "", "", NULL, "FilterBu",  True,  False, False },
    { "", "", NULL, "IncludeBu", True,  False, False },
  };

  static XIT_MENU_BUTTON_STRUCT help_casc[] = {
    { "", "", NULL, "ContextBu", True, False, False },
    { "", "", NULL, "WindowsBu", True, False, False },
    { "", "", NULL, "KeysBu",    True, False, False },
    { "", "", NULL, "IndexBu",   True, False, False },
    { "", "", NULL, "HelpBu",    True, False, False },
    { "", "", NULL, "VersionBu", True, False, False },
  };

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "", printCB,  NULL },
    { "", filterCB, NULL },
    { "", closeCB,  NULL },
  };


  /* Code. */

  appl_data_ref   = printer_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;


  /* Get text for menues and buttons. */
  action_buttons[ 0 ].label = msgGetText( MXDI_PRINT_BUTTON );
  action_buttons[ 0 ].data  = printer_ref;
  action_buttons[ 1 ].label = msgGetText( MXDI_FILTER_MENU );
  action_buttons[ 1 ].data  = printer_ref;
  action_buttons[ 2 ].label = msgGetText( MXDI_CLOSE_BUTTON );
  action_buttons[ 2 ].data  = printer_ref;

  menupane[ 0 ].title    = msgGetText( MXDI_FILE_MENU );
  menupane[ 0 ].mnemonic = msgGetText( MXDI_FILE_MENU_ACC );
  menupane[ 1 ].title    = msgGetText( MXDI_CTRL_MENU );
  menupane[ 1 ].mnemonic = msgGetText( MXDI_CTRL_MENU_ACC );
  menupane[ 2 ].title    = msgGetText( MXDI_HELP_MENU );
  menupane[ 2 ].mnemonic = msgGetText( MXDI_HELP_MENU_ACC );

  file_casc[ 0 ].title    = msgGetText( MXDI_CLOSE_MENU );
  file_casc[ 0 ].mnemonic = msgGetText( MXDI_CLOSE_MENU_ACC );

  ctrl_casc[  0 ].title    = msgGetText( MXDI_FILTER_MENU );
  ctrl_casc[  0 ].mnemonic = msgGetText( MXDI_FILTER_MENU_ACC );
  ctrl_casc[  1 ].title    = msgGetText( MXDI_INCLUDE_MENU );
  ctrl_casc[  1 ].mnemonic = msgGetText( MXDI_INCLUDE_MENU_ACC );

  help_casc[ 0 ].title    = msgGetText( MXDI_HELP_CONTEXT );
  help_casc[ 0 ].mnemonic = msgGetText( MXDI_HELP_CONTEXT_ACC );
  help_casc[ 1 ].title    = msgGetText( MXDI_HELP_WINDOWS );
  help_casc[ 1 ].mnemonic = msgGetText( MXDI_HELP_WINDOWS_ACC );
  help_casc[ 2 ].title    = msgGetText( MXDI_HELP_KEYS );
  help_casc[ 2 ].mnemonic = msgGetText( MXDI_HELP_KEYS_ACC );
  help_casc[ 3 ].title    = msgGetText( MXDI_HELP_INDEX );
  help_casc[ 3 ].mnemonic = msgGetText( MXDI_HELP_INDEX_ACC );
  help_casc[ 4 ].title    = msgGetText( MXDI_HELP_HELP );
  help_casc[ 4 ].mnemonic = msgGetText( MXDI_HELP_HELP_ACC );
  help_casc[ 5 ].title    = msgGetText( MXDI_HELP_VERSION );
  help_casc[ 5 ].mnemonic = msgGetText( MXDI_HELP_VERSION_ACC );


  /* Create a separate window. */
  toplevel = xitGetToplevelWidget( parent );

  printTl = xitCreateToplevelDialog( toplevel, "PrintTl", 
                                     1, 0,
                                     action_buttons, 
                                     XtNumber( action_buttons ) );

  (void) xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle,
                            printer_ref -> db_name,
                            &db_info, NULL );

  sprintf( buffer, "%s %s", 
           db_info.short_name, 
           msgGetText( MXDI_PRINTING_TITLE ) );

  n = 0;
  XtSetArg( args[ n ], XmNtitle, buffer ); n++;
  XtSetValues( printTl, args, n );

  sprintf( buffer, "%s", db_info.short_name );

  n = 0;
  XtSetArg( args[ n ], XmNiconName, buffer ); n++;
  XtSetValues( printTl, args, n );

  XtAddCallback( printTl, XmNdestroyCallback, 
                 (XtCallbackProc) destroyCB, (XtPointer) printer_ref );


  /* Close the window if this window is deleted. */
  {
    Atom  wm_delete_window;

    wm_delete_window = XmInternAtom( XtDisplay( printTl ),
                                     "WM_DELETE_WINDOW", False );

    XmAddWMProtocols( printTl, &wm_delete_window, 1 );
    XmAddWMProtocolCallback( printTl, wm_delete_window, 
                             (XtCallbackProc) closeCB,
                             (XtPointer) printer_ref );
  } /* block */


  /* Form to enclose the print window. */
  n = 0;
  workFo = XtNameToWidget( printTl, "PrintTlBase.PrintTlFo" );


  /* Create the menubar and menu cascades. */
  menuBr = XmCreateMenuBar( workFo, "MenuBr", args, 0 );

  n = 0;
  for( index = 0; index < XtNumber( pulldownMenu ); index++ )
    pulldownMenu[ index ] = XmCreatePulldownMenu( menuBr, 
                                                  pull_downs[ index ], 
                                                  NULL, n );

  for( index = 0; index < XtNumber( menuCasc ); index++ )
    menuCasc[ index ] = xitCreateCascadeButton( menuBr, 
                                                pulldownMenu[ index ], 
                                                &menupane[ index ] );

  /* The help button should be placed to the right. */
  index = XtNumber( menuCasc ) - 1;
  n     = 0;
  XtSetArg( args[ n ], XmNmenuHelpWidget, menuCasc[ index ] ); n++;
  XtSetValues( menuBr, args, n );


  /* Create the file menu. */
  for( index = 0; index < XtNumber( menuFileBu ); index++ )
    menuFileBu[ index ] = xitCreateMenuPushButton( pulldownMenu[ 0 ], 
                                                   &file_casc[ index ] );

  XtAddCallback( menuFileBu[ 0 ], XmNactivateCallback, 
                 (XtCallbackProc) closeCB, (XtPointer) printer_ref );

  /* Create the control menu. */
  XtAddCallback( pulldownMenu[ 1 ], XmNentryCallback, 
                 (XtCallbackProc) ctrlMenuCB, (XtPointer) printer_ref );

  for( index = 0; index < XtNumber( menuCtrlBu ); index++ ) {
    menuCtrlBu[ index ] = xitCreateMenuPushButton( pulldownMenu[ 1 ], 
                                                   &ctrl_casc[ index ] );

    if( XmIsPushButton( menuCtrlBu[ index ] ) )
      XtAddCallback( menuCtrlBu[ index ], XmNactivateCallback, 
                     (XtCallbackProc) ctrlMenuCB, (XtPointer) index );
    else if( XmIsToggleButton( menuCtrlBu[ index ] ) )
      XtAddCallback( menuCtrlBu[ index ], XmNvalueChangedCallback, 
                     (XtCallbackProc) ctrlMenuCB, (XtPointer) index );
  }

  /* Create the help menu. */
  XtAddCallback( pulldownMenu[ 2 ], XmNentryCallback, 
                 (XtCallbackProc) infoCB, (XtPointer) printer_ref );

  for( index = 0; index < XtNumber( menuHelpBu ); index++ ) {
    menuHelpBu[ index ] = xitCreateMenuPushButton( pulldownMenu[ 2 ], 
                                                   &help_casc[ index ] );

    XtAddCallback( menuHelpBu[ index ], XmNactivateCallback, 
                   (XtCallbackProc) infoCB, (XtPointer) index );
  }

  /* We can't do context sensitive help. */
  XtSetSensitive( menuHelpBu[ 0 ], False );


  /* Create the list of printers. */
  whatPrintersLa = xitCreateLabel( workFo, "WhatPrintersLa",
                                   msgGetText( MXDI_WHAT_PRINTERS_LABEL ),
                                   -1 );

  n = 0;
  XtSetArg( args[ n ], XmNlistSizePolicy, XmRESIZE_IF_POSSIBLE ); n++;
  XtSetArg( args[ n ], XmNscrollBarDisplayPolicy, XmSTATIC ); n++;
  XtSetArg( args[ n ], XmNselectionPolicy, XmSINGLE_SELECT ); n++;
  XtSetArg( args[ n ], XmNlistMarginHeight, 5 ); n++;
  XtSetArg( args[ n ], XmNlistMarginWidth, 5 ); n++;
  printerLi = XmCreateScrolledList( workFo, "PrinterLi", args, n );

  XtAddCallback( printerLi, XmNsingleSelectionCallback,
                 (XtCallbackProc) printerSelectCB, (XtPointer) printer_ref );


  /* Howto make your own printer selection list. */
  yourOwnFr = XmCreateFrame( workFo,  "YourOwnFr", args, 0 );
  yourOwnLa = xitCreateLabel( yourOwnFr, "YourOwnLa",
                              msgGetText( MXDI_YOUR_OWN_PRINTERS_LABEL ),
                              -1 );

  /* Create a print select form. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  XtSetArg( args[ n ], XmNpacking,     XmPACK_TIGHT ); n++;
  prSelectRc = XmCreateRowColumn( workFo, "PrSelectRc", args, n );

  /* Name of printer. */
  prNameLa = xitCreateLabel( prSelectRc, "PrNameLa",
                             msgGetText( MXDI_PRINTER_NAME ), -1 );

  prNameTx = xitCreateTextCols( prSelectRc, &text_field_def[ 1 ], 10 );

  n = 0;
  XtSetArg( args[ n ], XmNmaxLength, 50 );  n++;
  XtSetValues( prNameTx, args, n );  

  /* Type of printer. */
  prTypeLa = xitCreateLabel( prSelectRc, "PrTypeLa",
                             msgGetText( MXDI_PRINTER_TYPE ), -1 );

  prTypeFs = xtmFlCreateSelField( prSelectRc, "PrTypeFs",
                                  8, custom_data_ref -> printer_types,
                                  ' ', True,
                                  NULL, NULL );

  xitFieldSelectSetEditable( prTypeFs, False );

  tempW = xitFieldSelectGetChild( prTypeFs, xitFIELD_SELECT_TEXT_FIELD );


  /* Create a print date form. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  XtSetArg( args[ n ], XmNpacking,     XmPACK_TIGHT ); n++;
  prDayRc = XmCreateRowColumn( workFo, "PrDayRc", args, n );

  prDateLa = xitCreateLabel( prDayRc, "PrDateLa",
                             msgGetText( MXDI_BETWEEN_DATES ), -1 );

  /* Date range selector. */
  rangeTb = xtmFlCreateRangeField( prDayRc, "RangeTb" );


  /* Lines to display. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  shLineRc = XmCreateRowColumn( workFo, "ShLineRc", args, n );

  shLineLa = xitCreateLabel( shLineRc, "ShLineLa", 
                             msgGetText( MXDI_SUMMARY_LINES ), -1 );

  shLineTx = xitCreateTextCols( shLineRc, &text_field_def[ 0 ], 5 );

  n = 0;
  XtSetArg( args[ n ], XmNmaxLength, 5 );  n++;
  XtSetValues( shLineTx, args, n );  


  /* Create the print profile form. */
  prProfileFr = XmCreateFrame( workFo, "PrProfileFr", args, 0 );
  prProfileRc = XmCreateRowColumn( prProfileFr, "PrProfileRc", args, 0 );


  /* Output style. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  outStyleRc = XmCreateRowColumn( prProfileRc, "OutStyleRc", args, n );

  outStyleLa = xitCreateLabel( outStyleRc, "OutStyleLa",
                               msgGetText( MXDI_STYLE_LABEL ), -1 );

  outStyleFs = xtmFlCreateSelField( outStyleRc, "OutStyleFs", 0,
                                    msgGetText( MXDI_PRINT_FORMATS ),
                                    ',', False, NULL, NULL );

  xitFieldSelectSetEditable( outStyleFs, False );
  xitFieldSelectSetIndex(    outStyleFs, 2, False );


  /* Paper format. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  paperRc = XmCreateRowColumn( prProfileRc, "PaperRc", args, n );

  paperLa = xitCreateLabel( paperRc, "PaperLa",
                            msgGetText( MXDI_FORMAT_LABEL ), -1 );

  paperFs = xtmFlCreateSelField( paperRc, "PaperFs", 0,
                                 msgGetText( MXDI_PRINT_PAPER_FORMAT ),
                                 ',', False, NULL, NULL );

  xitFieldSelectSetEditable( paperFs, False );


  /* Output to. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  outputRc = XmCreateRowColumn( prProfileRc, "OutputRc", args, n );

  outputLa = xitCreateLabel( outputRc, "OutputLa",
                             msgGetText( MXDI_OUTPUT_LABEL ), -1 );

  outputFs = xtmFlCreateSelField( outputRc, "OutputFs", 0,
                                  msgGetText( MXDI_PRINT_ON ),
                                 ',', False, NULL, NULL );

  xitFieldSelectSetEditable( outputFs, False );


  /* Create the print profile form. */
  prParamFr = XmCreateFrame( workFo, "PrParamFr", args, 0 );
  prParamRc = XmCreateRowColumn( prParamFr, "PrParamRc", args, 0 );

  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  sheetRc = XmCreateRowColumn( prParamRc, "SheetRc", args, n );

  sheetLa = xitCreateLabel( sheetRc, "SheetLa",
                            msgGetText( MXDI_SHEETS_PER_PAGE_LABEL ), -1 );

  sheetFs = xtmFlCreateSelField( sheetRc, "SheetFs", 0,
                                 msgGetText( MXDI_SHEETS_PER_PAGE ),
                                 ' ', False, NULL, NULL );

  xitFieldSelectSetEditable( sheetFs, False );

  paramTb[ 0 ] = xitCreateToggleButton( 
                   prParamRc, "Param1Tb", 
                   msgGetText( MXDI_PARAM_PR_ONLY_WORKDAYS_LABEL ), False );

  paramTb[ 1 ] = xitCreateToggleButton( 
                   prParamRc, "Param2Tb", 
                   msgGetText( MXDI_PARAM_PR_3D_LABEL ), True );


  /* Put the Parts together. */
  xitAttachWidget( menuBr, 
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_FORM, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( whatPrintersLa,
                   XmATTACH_WIDGET, menuBr, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,   XmATTACH_NONE, NULL );
  xitAttachWidget( XtParent( printerLi ),
                   XmATTACH_WIDGET, whatPrintersLa, XmATTACH_FORM, NULL,
                   XmATTACH_FORM,   NULL,           XmATTACH_NONE, NULL );
  xitAttachWidget( yourOwnFr,
                   XmATTACH_WIDGET, XtParent( printerLi ),
                   XmATTACH_FORM,   NULL,
                   XmATTACH_FORM,   NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( prSelectRc, 
                   XmATTACH_WIDGET, yourOwnFr, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,      XmATTACH_NONE, NULL );
  xitAttachWidget( prDayRc, 
                   XmATTACH_WIDGET, prSelectRc, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,       XmATTACH_NONE, NULL );
  xitAttachWidget( shLineRc, 
                   XmATTACH_WIDGET, prDayRc, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,    XmATTACH_NONE, NULL );
  xitAttachWidget( prProfileFr, 
                   XmATTACH_WIDGET, shLineRc, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,     XmATTACH_NONE, NULL );
  xitAttachWidget( prParamFr, 
                   XmATTACH_WIDGET, shLineRc, XmATTACH_WIDGET, prProfileFr,
                   XmATTACH_NONE,   NULL,     XmATTACH_NONE,   NULL );


  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( XtParent( printerLi ), args, n );
  XtSetValues( yourOwnLa,             args, n );
  XtSetValues( yourOwnFr,             args, n );
  XtSetValues( whatPrintersLa,        args, n );
  XtSetValues( prSelectRc,            args, n );
  XtSetValues( prDayRc,               args, n );
  XtSetValues( shLineRc,              args, n );
  XtSetValues( prProfileFr,           args, n );
  XtSetValues( prParamFr,             args, n );


  /* Manage all the children. */
  XtManageChildren( menuCasc,     XtNumber( menuCasc ) );
  XtManageChildren( menuFileBu,   XtNumber( menuFileBu ) );
  XtManageChildren( menuHelpBu,   XtNumber( menuHelpBu ) );
  XtManageChildren( menuCtrlBu,   XtNumber( menuCtrlBu ) );
  XtManageChildren( pulldownMenu, XtNumber( pulldownMenu ) );
  XtManageChildren( paramTb,      XtNumber( paramTb ) );

  xitManageChildren( dataLocalW,  XtNumber( dataLocalW ) );


  /* Set icon for this window. */
  xtmIcSetSimpleIcon( printTl, workFo, XTM_IC_ICON_PRINT );

  /* Set the initial sizes. */
  xitSetSizeToplevelDialog( printTl, False );


  /* Make the final attachments. */
  xitAttachWidget( prProfileFr, 
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL );
  xitAttachWidget( shLineRc, 
                   XmATTACH_NONE, NULL, XmATTACH_FORM,   NULL,
                   XmATTACH_NONE, NULL, XmATTACH_WIDGET, prProfileFr );
  xitAttachWidget( prDayRc, 
                   XmATTACH_NONE, NULL, XmATTACH_FORM,   NULL,
                   XmATTACH_NONE, NULL, XmATTACH_WIDGET, shLineRc );
  xitAttachWidget( prSelectRc, 
                   XmATTACH_NONE, NULL, XmATTACH_FORM,   NULL,
                   XmATTACH_NONE, NULL, XmATTACH_WIDGET, prDayRc );
  xitAttachWidget( yourOwnFr, 
                   XmATTACH_NONE, NULL, XmATTACH_FORM,   NULL,
                   XmATTACH_FORM, NULL, XmATTACH_WIDGET, prSelectRc );
  xitAttachWidget( XtParent( printerLi ),
                   XmATTACH_WIDGET, whatPrintersLa, XmATTACH_FORM, NULL,
                   XmATTACH_FORM,   NULL, XmATTACH_WIDGET, yourOwnFr );


  /* Default printer and printer type? */
  char_ref = getenv( "PRINTER" );

  if( strlen( custom_data_ref -> default_printer ) > 0 )
    XmTextSetString( prNameTx, custom_data_ref -> default_printer );
  else if( char_ref != NULL )
    XmTextSetString( prNameTx, char_ref );

  if( strlen( custom_data_ref -> default_printer_type ) > 0 )
    xitFieldSelectSetCurrent( prTypeFs,
                              custom_data_ref -> default_printer_type,
                              False );

  /* Number of lines to display. */
  sprintf( buffer, "%d", custom_data_ref -> summary_def_lines );
  XmTextSetString( shLineTx, buffer );

  /* Print only workdays? */
  if( custom_data_ref -> pr_only_workdays )
    XmToggleButtonSetState( paramTb[ 0 ], True, False );
  else
    XmToggleButtonSetState( paramTb[ 0 ], False, False );

  /* Paper sizes. */
  xitFieldSelectSetIndex( paperFs,
                          custom_data_ref -> default_paper_size, False );

  /* Sheets per page. */
  xitFieldSelectSetIndex( sheetFs,
                          custom_data_ref -> print_def_sheets, False );

  /* Print in 3D? */
  if( custom_data_ref -> pr_in_3d )
    XmToggleButtonSetState( paramTb[ 1 ], True, False );
  else
    XmToggleButtonSetState( paramTb[ 1 ], False, False );


  return( printTl );

} /* createPrintWindow */


/*----------------------------------------------------------------------*/

static void
  loadPrinterList( PRINTER_REC_REF  printer_ref )
{

  /* Variables. */
  int                     index;
  int                     items = 0;
  int                     parsed_items;
  int                     source_index = 0;
  char                    buffer[ 100 ];
  char                    delimiter[ 10 ];
  char                    printer_desc[ 100 ];
  char                    printer_flags[ 50 ];
  char                    printer_name[ 50 ];
  char                    printer_type[ 50 ];
  char                    *char_ref;
  char                    *printer_source[ 10 ];
  Arg                     args[ 10 ];
  Cardinal                n;
  FILE                    *file_ref;
  XmString                list_items[ 100 ];
  Widget                  mainW;
  Widget                  prListW;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  appl_data_ref = printer_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;

  mainW   = XtNameToWidget( printer_ref -> printerW, "PrintTlBase.PrintTlFo" );
  prListW = XtNameToWidget( mainW, "PrinterLiSW.PrinterLi" );

  /* Whre do we find our printer definitions? */
  printer_source[ 0 ] = custom_data_ref -> user_printer_file;
  printer_source[ 1 ] = custom_data_ref -> system_printer_file;
  printer_source[ 2 ] = NULL;


  /* Process all the printer sources. */
  while( printer_source[ source_index ] != NULL ) {

    /* Try to open the file, if we fail, try the next. */
    file_ref = fopen( printer_source[ source_index ], "r" );
    if( file_ref == NULL ) {
      source_index++;
      continue;
    }

    /* Read all the printer definitions and build the printer list. */
    while( fgets( buffer, sizeof( buffer ), file_ref ) != NULL ) {

      /* Skip all leading blanks. */
      char_ref = buffer;
      while( isspace( *char_ref ) && *char_ref != '\0' )
        char_ref++;

      /* Is this a comment? */
      if( *char_ref == '#' || *char_ref == '!' )
        continue;

      /* Parse the printer definition. */
      printer_desc[ 0 ]  = '\0';
      printer_flags[ 0 ] = '\0';
      printer_name[ 0 ]  = '\0';
      printer_type[ 0 ]  = '\0';

      parsed_items = sscanf( char_ref, "%[^$]%c%[^$]%c%[^$]%c%[^\n]",
                             printer_name,  delimiter,
                             printer_type,  delimiter,
                             printer_desc,  delimiter,
                             printer_flags );
      if( parsed_items < 5 )
        continue;

      if( printer_desc[ strlen( printer_desc ) -1 ] == '\n' )
        printer_desc[ strlen( printer_desc ) - 1 ] = '\0';

      /* Create the entry in the printer list. */
      sprintf( buffer, "%-15.15s  %-5.5s  %-40.40s",
               printer_name, printer_type, printer_desc );

      /* Add the printer definition to the list. */
      list_items[ items ] = XmStringCreateLtoR( buffer, CS );
      items++;

    } /* while */

    fclose( file_ref );
    source_index++;

  } /* while */

  if( items < 1 )
    return;


  /* Assign the printers to the list. */
  n = 0;
  XtSetArg( args[ n ], XmNitems, list_items ); n++;
  XtSetArg( args[ n ], XmNitemCount, items ); n++;
  XtSetValues( prListW, args, n );

  /* Free the allocated strings. */
  for( index = 0; index < items; index++ )
    XmStringFree( list_items[ index ] );


  return;

} /* loadPrinterList */


/*----------------------------------------------------------------------*/

static void
  saveRecordToFile( XTM_GL_BASE_DATA_REF  appl_data_ref,
                    XTM_DB_ALL_ENTRY_REF  entry_ref,
                    FILE                  *file_ref,
                    int                   display_lines )
{

  /* Variables. */
  int           line_count;
  int           line_size;
  char          buffer[ 100 ];
  char          flags[ 20 ];
  char          *char_ref;


  /* Code. */

  /* Get the flags. */
  flags[ 0 ] = '\0';

  if( flagIsSet( entry_ref -> entry.flags, XTM_DB_FLAG_ALARM ) )
    strcat( flags, "A" );

  if( flagIsSet( entry_ref -> entry.flags, XTM_DB_FLAG_IMPORTANT ) )
    strcat( flags, "I" );

  if( flagIsSet( entry_ref -> entry.flags, XTM_DB_FLAG_PRIVATE ) )
    strcat( flags, "P" );

  if( entry_ref -> entry.entry_type == XTM_DB_DAY_NOTE )
    strcat( flags, "N" );

  if( entry_ref -> entry.entry_type == XTM_DB_DAY_NOTE )
    if( flagIsSet( entry_ref -> entry.flags, XTM_DB_FLAG_NOTE_DONE ) )
      strcat( flags, "D" );

  if( entry_ref -> entry.entry_category == XTM_DB_REP_ENTRY_LIST )
    strcat( flags, "S" );

  if( entry_ref -> entry.entry_category == XTM_DB_STICKY_LIST )
    strcat( flags, "T" );


  /* Entry start. */
  fprintf( file_ref, "@!XDiaryDump-V1.1\n" );


  /* Date and time for the entry. */
  TimFormatIsoDate( entry_ref -> entry.date_stamp, buffer, sizeof( buffer ) );
  fprintf( file_ref, "Ti %s ", buffer );

  TimFormatIsoTime( entry_ref -> entry.time_stamp, buffer, sizeof( buffer ) );

  fprintf( file_ref, "%s ",  buffer );
  fprintf( file_ref, "%d\n", (int) entry_ref -> entry.duration );


  /* Flags (if any). */
  if( flags[ 0 ] != '\0' )
    fprintf( file_ref, "Fl %s\n", flags );


  /* Date and time information. */
  TimFormatStrTime( entry_ref -> entry.date_stamp, "%Y %m %d %w", 
                    buffer, sizeof( buffer ) );
  fprintf( file_ref, "Dt %s ", buffer );

  TimFormatStrTime( entry_ref -> entry.time_stamp, "%H %M ", 
                    buffer, sizeof( buffer ) );
  fprintf( file_ref, "%s ", buffer );

  TimFormatStrTime( entry_ref -> entry.time_stamp + 
                    entry_ref -> entry.duration * 60, "%H %M ", 
                    buffer, sizeof( buffer ) );
  fprintf( file_ref, "%s ", buffer );

  TimFormatStrTime( entry_ref -> entry.time_stamp, "%I %M %p ", 
                    buffer, sizeof( buffer ) );
  fprintf( file_ref, "%s ", buffer );

  TimFormatStrTime( entry_ref -> entry.time_stamp + 
                    entry_ref -> entry.duration * 60, "%I %M %p ", 
                    buffer, sizeof( buffer ) );
  fprintf( file_ref, "%s\n", buffer );


  /* The alarm offsets. */
  fprintf( file_ref, "Al %d %d %d %d %d %d %d %d %d %d\n",
           (int) entry_ref -> entry.alarm_valid[  0 ], 
           (int) entry_ref -> entry.alarm_offset[ 0 ],
           (int) entry_ref -> entry.alarm_valid[  1 ], 
           (int) entry_ref -> entry.alarm_offset[ 1 ],
           (int) entry_ref -> entry.alarm_valid[  2 ], 
           (int) entry_ref -> entry.alarm_offset[ 2 ],
           (int) entry_ref -> entry.alarm_valid[  3 ], 
           (int) entry_ref -> entry.alarm_offset[ 3 ],
           (int) entry_ref -> entry.alarm_valid[  4 ], 
           (int) entry_ref -> entry.alarm_offset[ 4 ] );


  /* Entry tag. */
  if( strlen( entry_ref -> entry.tag ) > 0 )
    fprintf( file_ref, "Tg %s\n", entry_ref -> entry.tag );


  /* Fetch the text for the entry. */
  if( entry_ref -> all_text == NULL )
     char_ref = entry_ref -> entry.text;
  else
    char_ref = entry_ref -> all_text;


  /* Write each line in the entry. */
  line_count = 0;

  while( *char_ref != '\0' && line_count < display_lines ) {
    line_size = strcspn( char_ref, "\n" );

    fprintf( file_ref, "<<" );
    fwrite(  char_ref, 1, line_size, file_ref );
    fprintf( file_ref, "\n" );

    char_ref = char_ref + line_size;
    if( *char_ref == '\0' )
      break;

    /* Skip the \n character. */
    char_ref ++;

    line_count++;
  } /* while */

  fprintf( file_ref, "$$\n\n" );


  return;

} /* saveRecordToFile */


/*----------------------------------------------------------------------*/

static void
  startPrintJob( PRINTER_REC_REF  printer_ref )
{

  /* Variables. */
  Boolean               pr_only_workdays;
  Boolean               pr_in_3d;
  int                   index;
  int                   start_on;
  int                   stop_on;
  UINT32                flags;
  char                  buffer[ 250 ];
  char                  filename[ 250 ];
  char                  *char_ref;
  Widget                baseW;
  Widget                mainW;
  Widget                tempW;
  FILE                  *file_ref;
  LST_DESC_TYPE         entries;
  LST_STATUS            lst_status;
  TIM_TIME_REF          current_date;
  TIM_TIME_REF          entry_date;
  TIM_TIME_REF          from_date;
  TIM_TIME_REF          to_date;
  XTM_DB_ALL_ENTRY_DEF  entry_record;
  XTM_CD_CAL_INFO       db_info;
  XTM_DB_STATUS         db_status;
  XTM_GL_BASE_DATA_REF  appl_data_ref;


  /* Code. */

  appl_data_ref = printer_ref -> appl_data_ref;

  mainW = XtNameToWidget( printer_ref -> printerW, "PrintTlBase.PrintTlFo" );


  (void) xtmCdFetchNamedDb( appl_data_ref -> custom_data -> cal_db_handle,
                            printer_ref -> db_name,
                            &db_info, NULL );


  /* The file where the data to print is written. */
  char_ref = getenv( "HOME" );
  if( char_ref != NULL )
    sprintf( filename, "%s/XDiary.out", char_ref );
  else
    sprintf( filename, "./XDiary.out" );

  file_ref = fopen( filename, "w" );
  if( file_ref == NULL ) {
    xitErMessage( printer_ref -> printerW, XIT_ER_ERROR, 
                  module_name, "startPrintJob",
                  msgGetText( MXDI_PR_FILE_OPEN_ERR ) );
    return;
  }


  /* Serach between the dates. */
  from_date = printer_ref -> from_date;
  to_date   = printer_ref -> to_date;


  /* Only on workdays? */
  tempW = XtNameToWidget( mainW, "PrParamFr.PrParamRc.Param1Tb" );
  if( XmToggleButtonGetState( tempW ) )
    pr_only_workdays = True;
  else
    pr_only_workdays = False;

  /* Print in 3D? */
  tempW = XtNameToWidget( mainW, "PrParamFr.PrParamRc.Param2Tb" );
  if( XmToggleButtonGetState( tempW ) )
    pr_in_3d = True;
  else
    pr_in_3d = False;


  /* Start on first weekday for week style. Display a week. */
  baseW = XtNameToWidget( mainW, "PrProfileFr.PrProfileRc" );
  tempW = XtNameToWidget( baseW, "OutStyleRc.OutStyleFs" );

  (void) xitFieldSelectGetIndex( tempW, &index );

  if( index == 2 ) {
    if( TimIndexOfFirstDayInWeek() == 0 ) {
      start_on = 7;
      stop_on = 6;
    } else {
      stop_on = 7;
      start_on = 1;
    }

    while( TimIndexOfDayInIsoWeek( from_date ) != start_on )
      TimAddDays( &from_date, -1 );

    while( TimIndexOfDayInIsoWeek( to_date ) != stop_on )
      TimAddDays( &to_date, 1 );
  }


  /* Start on first day in month for month style. */
  baseW = XtNameToWidget( mainW, "PrProfileFr.PrProfileRc" );
  tempW = XtNameToWidget( baseW, "OutStyleRc.OutStyleFs" );

  (void) xitFieldSelectGetIndex( tempW, &index );

  if( index == 1 ) {
    start_on = TimIndexOfMonth( from_date );
    stop_on  = TimIndexOfMonth( to_date );

    while( TimIndexOfMonth( from_date ) == start_on )
      TimAddDays( &from_date, -1 );
    TimAddDays( &from_date, 1 );

    while( TimIndexOfMonth( to_date ) == stop_on )
      TimAddDays( &to_date, 1 );
    TimAddDays( &to_date, -1 );
  }


  /* Create the list of entries to display. */
  flags = XTM_DB_FETCH_ALL_TEXT;

  db_status = xtmDmFetchFilteredEntriesRange( 
                appl_data_ref,
                printer_ref   -> db_name,
                appl_data_ref -> context,
                printer_ref   -> printerW,
                from_date, to_date,
                flags,
                &printer_ref  -> filter_rec,
                &entries );


  /* Header comments. */
  fprintf( file_ref, "#%%XDiaryPrint:\n" );
  fprintf( file_ref, "#%%Calendar: %s\n", db_info.short_name );

  fprintf( file_ref, "#%%DateRange: " );
  TimFormatIsoDate( from_date, buffer, sizeof( buffer ) );
  fprintf( file_ref, "%s ", buffer );

  TimFormatIsoDate( to_date, buffer, sizeof( buffer ) );
  fprintf( file_ref, "%s\n", buffer );

  fprintf( file_ref, "#%%DefaultTimeMin: %d.00\n",
           appl_data_ref -> custom_data -> start_hour );
  fprintf( file_ref, "#%%DefaultTimeMax: %d.00\n",
           appl_data_ref -> custom_data -> stop_hour );

  fprintf( file_ref, "#%%DateFormat: %s\n",
           appl_data_ref -> custom_data -> date_format );
  fprintf( file_ref, "#%%TimeFormat: %s\n",
           appl_data_ref -> custom_data -> time_format );

  if( pr_only_workdays )
    fprintf( file_ref, "#%%PrOnlyWorkdays: true\n" );
  else
    fprintf( file_ref, "#%%PrOnlyWorkdays: false\n" );

  if( pr_in_3d )
    fprintf( file_ref, "#%%PrIn3d: true\n" );
  else
    fprintf( file_ref, "#%%PrIn3d: false\n" );

  fprintf( file_ref, "\n" );


  /* Create the entries. */
  current_date = from_date;


  /* Fetch the first entry. */
  lst_status = LstLinkCurrentFirst( entries );

  if( lst_status == LST_OK ) {
    lst_status = LstLinkGetCurrent( entries, &entry_record );
    entry_date = entry_record.entry.date_stamp;
  } else
    entry_date = 0;


  /* Loop through all days. */
  while( current_date <= to_date ) {

    /* Only workdays? */
    if( pr_only_workdays &&
        TimIndexOfDayInIsoWeek( current_date ) > 5  ) {

      /* Skip entries. */
      while( current_date == entry_date ) {
        lst_status = LstLinkCurrentNext( entries );

        entry_date = 0;
        if( lst_status == LST_OK ) {
          lst_status = LstLinkGetCurrent( entries, &entry_record );
          entry_date = entry_record.entry.date_stamp;
        }
      }

      TimAddDays( &current_date, 1 );
      continue;

    } /* if */


    /* Write entry? */
    while( current_date == entry_date ) {

      /* Write the entry. */
      if( flagIsClear( entry_record.entry.flags, XTM_DB_FLAG_HIDE_IN_PRINT ) &&
          entry_record.entry.day_list_lines >= 1 ) {

        /* Entry information. */
        saveRecordToFile( appl_data_ref, 
                          &entry_record, file_ref,
                          printer_ref -> display_lines );

      } /* if */


      /* Fetch next entry. */
      lst_status = LstLinkCurrentNext( entries );

      if( lst_status == LST_OK ) {
        lst_status = LstLinkGetCurrent( entries, &entry_record );
        entry_date = entry_record.entry.date_stamp;
      } else
        entry_date = 0;

    } /* while */


    /* Next day. */
    TimAddDays( &current_date, 1 );

  } /* while */


  /* Clear the list with entries. */
  xtmDmDeleteEntriesList( entries );

  fclose( file_ref );


  /* Run the print process? */
  if( db_status != XTM_DB_OK )
    return;

  startPrintProcess( printer_ref, filename );


  return;

} /* startPrintJob */


/*----------------------------------------------------------------------*/

static void
  startPrintProcess( PRINTER_REC_REF  printer_ref,
                     char             *filename )
{

  /* Variables. */
  int                   index;
  char                  buffer[ 250 ];
  char                  *output_dest = "";
  char                  *paper_size = "";
  char                  *process_args[ 10 ];
  char                  *sheets_per_page = "";
  char                  *style = "";
  Widget                baseW;
  Widget                mainW;
  Widget                tempW;
  XTM_GL_BASE_DATA_REF  appl_data_ref;


  /* Code. */

  appl_data_ref = printer_ref -> appl_data_ref;

  mainW = XtNameToWidget( printer_ref -> printerW, "PrintTlBase.PrintTlFo" );


  /* Fetch the style. */
  baseW = XtNameToWidget( mainW, "PrProfileFr.PrProfileRc" );
  tempW = XtNameToWidget( baseW, "OutStyleRc.OutStyleFs" );

  (void) xitFieldSelectGetIndex( tempW, &index );

  switch( index ) {
    case 1:
      style = "MonthList";
      break;
    case 2:
      style = "WeekList";
      break;
    case 3:
      style = "CompWeekList";
      break;
    case 4:
      style = "DayList";
      break;
    case 5:
      style = "CompDayList";
      break;
    default:
      style = "EntryList";
      break;
  } /* switch */

  /* Fetch the paper size. */
  baseW = XtNameToWidget( mainW, "PrProfileFr.PrProfileRc" );
  tempW = XtNameToWidget( baseW, "PaperRc.PaperFs" );

  (void) xitFieldSelectGetIndex( tempW, &index );

  switch( index ) {
    case 0:
      paper_size = "A3";
      break;
    case 1:
      paper_size = "A4";
      break;
    case 2:
      paper_size = "A5";
      break;
    case 3:
      paper_size = "Letter";
      break;
    case 4:
      paper_size = "Legal";
      break;
    default:
      paper_size = "A4";
      break;
  } /* switch */

  /* Fetch sheets per page. */
  baseW = XtNameToWidget( mainW, "PrParamFr.PrParamRc" );
  tempW = XtNameToWidget( baseW, "SheetRc.SheetFs" );

  (void) xitFieldSelectGetIndex( tempW, &index );

  switch( index ) {
    case 0:
      sheets_per_page = "1";
      break;
    case 1:
      sheets_per_page = "2";
      break;
    default:
      sheets_per_page = "1";
      break;
  } /* switch */

  /* Fetch the output destination. */
  baseW = XtNameToWidget( mainW, "PrProfileFr.PrProfileRc" );
  tempW = XtNameToWidget( baseW, "OutputRc.OutputFs" );

  (void) xitFieldSelectGetIndex( tempW, &index );

  switch( index ) {
    case 1:
      output_dest = "File";    
      break;
    case 2:
      output_dest = "PrinterFile";    
      break;
    default:
      output_dest = "Printer";
      break;
  } /* switch */


  /* Notify the user about the result. */
  baseW = XtNameToWidget( mainW, "PrProfileFr.PrProfileRc" );
  tempW = XtNameToWidget( baseW, "OutputRc.OutputFs" );

  (void) xitFieldSelectGetIndex( tempW, &index );

  if( index == 1 )
    sprintf( buffer, msgGetText( MXDI_PRINT_FILE_OUTPUT_OK ), filename );
  else
    strcpy( buffer, msgGetText( MXDI_PRINT_OUTPUT_OK ) );


  /* Start the printout process. */
  process_args[ 0 ] = appl_data_ref -> custom_data -> print_script;
  process_args[ 1 ] = filename;
  process_args[ 2 ] = printer_ref -> pr_name;
  process_args[ 3 ] = printer_ref -> pr_type;
  process_args[ 4 ] = style;
  process_args[ 5 ] = output_dest;
  process_args[ 6 ] = paper_size;
  process_args[ 7 ] = sheets_per_page;
  process_args[ 8 ] = NULL;

  xtmToStartProcess( printer_ref -> printerW, False, buffer, process_args );


  return;

} /* startPrintProcess */


/*----------------------------------------------------------------------*/

static void
  closeCB( Widget           widget,
           PRINTER_REC_REF  printer_ref,
           XtPointer        call_data )
{

  /* Variables. */
  int  index;


  /* Code. */

  /* Do we have a user action callback registered? */
  if( printer_ref -> actionCB != NULL )
    (* printer_ref -> actionCB)( XTM_PR_REASON_DESTROY, 
                                 printer_ref -> user_data );


  /* Keep the window in cache? */
  for( index = 0; index < MAX_CACHE_ENTRIES; index++ ) {
    if( cache_entries[ index ] == NULL ) {
      cache_entries[ index ] = printer_ref;

      destroyCB( NULL, printer_ref, NULL );
      XtPopdown( printer_ref -> printerW );

      return;
    }
  }

  /* Window not kept in cache, really remove it. */
  XtDestroyWidget( printer_ref -> printerW );


  return;

} /* closeCB */


/*----------------------------------------------------------------------*/

static void 
  ctrlMenuCB( Widget                     widget,
              PRINTER_REC_REF            printer_ref,
              XmRowColumnCallbackStruct  *call_data )
{

  /* Variables. */
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  custom_data_ref = printer_ref -> appl_data_ref -> custom_data;

  /* Select what to do. */
  switch( (int) call_data -> data ) {

    /* Filter window. */
    case 0:
      filterCB( widget, printer_ref, NULL );
      break;


    /* Include calendars. */
    case 1:
      if( printer_ref -> include_handle == NULL )
        printer_ref -> include_handle = xtmDiInitialize(
                                          printer_ref -> appl_data_ref,
                                          printer_ref -> printerW,
                                          False,
                                          includeDbApplyCB,
                                          (void *) printer_ref );

      /* Display the include window. */
      xtmDiDisplayIncludeWindow( printer_ref -> include_handle,
                                 printer_ref -> db_name );
      break;

  } /* switch */


  return;

} /* ctrlMenuCB */


/*----------------------------------------------------------------------*/

static void 
  destroyCB( Widget           widget,
             PRINTER_REC_REF  printer_ref,
             XtPointer        call_data )
{

  /* Variables. */
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  custom_data_ref = printer_ref -> appl_data_ref -> custom_data;


  /* Do we have a user action callback registered? */
  if( printer_ref -> actionCB != NULL )
    (* printer_ref -> actionCB)( XTM_PR_REASON_DESTROY, 
                                 printer_ref -> user_data );

  /* Destroy child windows. */
  if( printer_ref -> include_handle != NULL ) {
    xtmDiDestroy( printer_ref -> include_handle );
    printer_ref -> include_handle = NULL;
  }

  if( printer_ref -> filter_handle != NULL ) {
    xtmFiDestroy( printer_ref -> filter_handle );
    printer_ref -> filter_handle = NULL;
  }

  /* Release the user data. */
  xtmCdFreeShadowEntry( custom_data_ref -> cal_db_handle,
                        printer_ref -> db_name );


  /* Release the user data (only if not cached). */
  if( widget != NULL )
    SysFree( printer_ref );


  return;

} /* destroyCB */


/*----------------------------------------------------------------------*/

static void 
  filterApplyCB( XTM_FI_REASON      reason,
                 XTM_DM_FILTER_REC  *filter_ref,
                 void               *user_data )
{

  /* Variables. */
  PRINTER_REC_REF  printer_ref;


  /* Code. */

  if( reason != XTM_FI_REASON_APPLY && reason != XTM_FI_REASON_OK )
    return;

  printer_ref = (PRINTER_REC_REF) user_data;


  /* Save the filter. */
  memcpy( (void *) &printer_ref -> filter_rec,
          (void *) filter_ref,
          sizeof( XTM_DM_FILTER_REC ) );


  return;

} /* filterApplyCB */


/*----------------------------------------------------------------------*/

static void 
  filterCB( Widget           widget,
            PRINTER_REC_REF  printer_ref,
            XtPointer        call_data )
{

  /* Code. */

  /* Initialize the filter? */
  if( printer_ref -> filter_handle == NULL )
    printer_ref -> filter_handle = xtmFiInitialize(
                                     printer_ref -> appl_data_ref,
                                     printer_ref -> printerW,
                                     printer_ref -> search_tags,
                                     False,
                                     filterApplyCB, (void *) printer_ref );

  /* Display the filter window. */
  xtmFiDisplayFilterWindow( printer_ref  -> filter_handle,
                            &printer_ref -> filter_rec );


  return;

} /* filterCB */


/*----------------------------------------------------------------------*/

static void
  includeDbApplyCB( XTM_DI_REASON         reason,
                    XTM_CD_INCL_CALS_REF  new_db_incl_ref,
                    void                  *user_data )
{

  /* Variables. */
  PRINTER_REC_REF         printer_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;
  XTM_CD_CAL_INFO         db_info;
  XTM_CD_INCL_CALS        db_incl;


  /* Code. */

  if( reason != XTM_DI_REASON_APPLY && reason != XTM_DI_REASON_OK )
    return;

  printer_ref     = (PRINTER_REC_REF) user_data;
  custom_data_ref = printer_ref -> appl_data_ref -> custom_data;


  /* Fetch the current database. */
  (void) xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle,
                            printer_ref -> db_name,
                            &db_info, &db_incl );

  /* Save the new include data. */
  (void) xtmCdChangeEntry( custom_data_ref -> cal_db_handle,
                           printer_ref -> db_name,
                           &db_info, new_db_incl_ref );


  return;

} /* includeDbApplyCB */


/*----------------------------------------------------------------------*/

static void 
  infoCB( Widget                     widget,
          PRINTER_REC_REF            printer_ref,
          XmRowColumnCallbackStruct  *call_data )
{

  /* Code. */

  xtmHlDisplayHelp( printer_ref -> appl_data_ref -> info_handle,
                    (int) call_data -> data,
                    print_window_id, "" );

  return;

} /* infoCB */


/*----------------------------------------------------------------------*/

static void
  printCB( Widget           widget,
           PRINTER_REC_REF  printer_ref,
           XtPointer        call_data )
{

  /* Variables. */
  Boolean               need_printer = True;
  Boolean               ok;
  int                   index;
  int                   items;
  char                  *char_ref;
  Widget                mainW;
  Widget                tempW;
  TIM_DELTA_TYPE        delta;
  XTM_GL_BASE_DATA_REF  appl_data_ref;


  /* Code. */

  appl_data_ref = printer_ref -> appl_data_ref;

  mainW = XtNameToWidget( printer_ref -> printerW, "PrintTlBase.PrintTlFo" );


  /* Do we need a printer name? */
  tempW = XtNameToWidget( mainW,
                          "PrProfileFr.PrProfileRc.OutputRc.OutputFs" );

  (void) xitFieldSelectGetIndex( tempW, &index );
  if( index == 1 )
    need_printer = False;


  /* Printer name. */
  tempW = XtNameToWidget( mainW, "PrSelectRc.PrNameTx" );

  char_ref = xitStringGetText( tempW );
  if( char_ref == NULL )
    return;

  strcpy( printer_ref -> pr_name, char_ref );
  SysFree( char_ref );

  char_ref = printer_ref -> pr_name;
  while( isspace( *char_ref ) )
    char_ref++;

  if( *char_ref == '\0' && need_printer ) {
    xitErMessage( printer_ref -> printerW, XIT_ER_ERROR, 
                  module_name, "printCB",
                  msgGetText( MXDI_NO_PRINTER_GIVEN ) );
    return;
  } else if( *char_ref == '\0' ) {
    strcpy( printer_ref -> pr_name, "x" );
  }


  /* Printer type. */
  tempW = XtNameToWidget( mainW, "PrSelectRc.PrTypeFs" );

  xitFieldSelectGetCurrent( tempW, &char_ref );
  if( char_ref == NULL )
    return;

  strcpy( printer_ref -> pr_type, char_ref );
  SysFree( char_ref );


  /* Fetch the date range. */
  tempW = XtNameToWidget( mainW, "PrDayRc.RangeTb" );

  ok = xtmFoFetchDate( mainW, tempW, XTM_FO_START_DATE, False,
                       &printer_ref -> from_date );
  if( ! ok )
    return;                                    

  ok = xtmFoFetchDate( mainW, tempW, XTM_FO_END_DATE, False,
                       &printer_ref -> to_date );
  if( ! ok )
    return;                                    

  if( printer_ref -> from_date > printer_ref -> to_date ) {
    xitErMessage( printer_ref -> printerW, XIT_ER_ERROR, 
                  module_name, "printCB",
                  msgGetText( MXDI_INVALID_DATE_RANGE ) );
    return;
  }


  /* How many days can we print? */
  (void) TimDelta( printer_ref -> from_date, printer_ref -> to_date,
                   &delta );

  if( delta.days > 366 ) {
    xitErMessage( printer_ref -> printerW, XIT_ER_ERROR, 
                  module_name, "printCB",
                  msgGetText( MXDI_PRINT_SHOW_TO_MANY_DAYS ) );
    return;
  }


  /* Number of lines to display? */
  tempW    = XtNameToWidget( mainW, "ShLineRc.ShLineTx" );
  char_ref = xitStringGetText( tempW );

  items = sscanf( char_ref, "%d", &printer_ref -> display_lines );
  if( items != 1 || printer_ref -> display_lines < 0 )
    printer_ref -> display_lines = 10000;

  SysFree( char_ref );


  startPrintJob( printer_ref );


  return;

} /* printCB */


/*----------------------------------------------------------------------*/

static void 
  printerSelectCB( Widget                widget,
                   PRINTER_REC_REF       printer_ref,
                   XmListCallbackStruct  *call_data )
{

  /* Variables. */
  char    printer_name[ 20 ];
  char    printer_type[ 20 ];
  char    *char_ref;
  Widget  mainW;
  Widget  tempW;


  /* Code. */

  mainW = XtNameToWidget( printer_ref -> printerW, "PrintTlBase.PrintTlFo" );


  /* Fetch the selected printer. */
  char_ref = xitStringGetString( call_data -> item, CS );

  sscanf( char_ref, "%[^ ] %[^ ]", printer_name, printer_type );
  SysFree( char_ref );


  /* Set the printer name and type. */
  tempW = XtNameToWidget( mainW, "PrSelectRc.PrNameTx" );
  XmTextSetString( tempW, printer_name );

  tempW = XtNameToWidget( mainW, "PrSelectRc.PrTypeFs" );
  xitFieldSelectSetCurrent( tempW, printer_type, False );


  return;

} /* printerSelectCB */
