/*----------------------------------------------------------------------------
--
--  Module:           xtmPlan
--
--  Project:          XDiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Plan a meeting with the help of one or more calendars.
--
--  Filename:         xtmPlan.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1993-06-01
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmPlan.c, Version: 1.1, Date: 95/02/18 15:52:38";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <ctype.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/cursorfont.h>

#include <Xm/Protocols.h>

#include <Xm/Xm.h>
#include <Xm/ArrowB.h>
#include <Xm/BulletinB.h>
#include <Xm/CascadeB.h>
#include <Xm/Form.h>
#include <Xm/PushB.h>
#include <Xm/RowColumn.h>
#include <Xm/ScrolledW.h>
#include <Xm/Separator.h>
#include <Xm/Text.h>
#include <Xm/ToggleB.h>
#include <Xm/ScrollBar.h>

#include "LstLinked.h"
#include "Message.h"
#include "System.h"
#include "TimDate.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmDbMisc.h"
#include "xtmDbTools.h"
#include "xtmFields.h"
#include "xtmFormat.h"
#include "xtmGroupDb.h"
#include "xtmHelp.h"
#include "xtmIcons.h"
#include "xtmMsgSend.h"
#include "xtmNavCal.h"
#include "xtmOpenView.h"
#include "xtmTools.h"
#include "xtmUpdate.h"
#include "xitError.h"
#include "xitFieldSel.h"
#include "xitTools.h"
#include "XmUbArrLab.h"
#include "xtmPlan.h"
#include "xtmHoliday.h"
#include "XmUbTimeB.h"

/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Flags to signal various actions to take. */
#define FLAG_CAL_DISPLAYED    (1<<1)

/* Day views in the cache? */
#define  MAX_CACHE_ENTRIES   3

/* Local widgets in the database select window. */
#define addCalPb          dataLocalW[  0 ]
#define calSelLa          dataLocalW[  1 ]
#define calSelRc          dataLocalW[  2 ]
#define calSelSw          dataLocalW[  3 ]
#define clearPb           dataLocalW[  4 ]
#define dateInfoLa        dataLocalW[  5 ]
#define dateLa            dataLocalW[  6 ]
#define dateRc            dataLocalW[  7 ]
#define dateTb            dataLocalW[  8 ]
#define dayDispBb         dataLocalW[  9 ]
#define dayDispSw         dataLocalW[ 10 ]
#define grpSelLa          dataLocalW[ 11 ]
#define grpSelRc          dataLocalW[ 12 ]
#define grpSelSw          dataLocalW[ 13 ]
#define howToDoLa         dataLocalW[ 14 ]
#define menuBr            dataLocalW[ 15 ]
#define navMonthAl        dataLocalW[ 16 ]
#define navFo             dataLocalW[ 17 ]
#define navWeekAl         dataLocalW[ 18 ]
#define planBb            dataLocalW[ 19 ]
#define planSw            dataLocalW[ 20 ]
#define planWhatFo        dataLocalW[ 21 ]
#define timeDispBb        dataLocalW[ 22 ]
#define timeDispLa        dataLocalW[ 23 ]
#define timeDispSw        dataLocalW[ 24 ]
#define zoomFs            dataLocalW[ 25 ]


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Result when searching, creating new entries. */
typedef enum { EXISTS, IS_NEW, NOT_FOUND } FIND_RESULT;

/* A date range with calendars. */
typedef struct {
  TIM_TIME_REF   from_date;
  TIM_TIME_REF   to_date;
  LST_DESC_TYPE  cals;
} PLAN_DATES, *PLAN_DATES_REF;

/* A calendar with entries. */
typedef struct {
  char           name[ XTM_GL_MAX_CAL_NAME + 1 ];
  UINT32         flags;
  LST_DESC_TYPE  entries;
} PLAN_CAL, *PLAN_CAL_REF;

/* A single entry. */
typedef struct {
  int           day_no;
  int           duration;
  int           min_since_midnight;
  UINT32        flags;
  TIM_TIME_REF  start;
} PLAN_ENTRY, *PLAN_ENTRY_REF;


/* Basic data structure for the planner. */
typedef struct {

  /* Busy-times matrix. */
  Boolean  **busy_days;

  /* First time displayed? */
  Boolean  first_time;

  /* Instant update? */
  Boolean  instant_update;

  /* Number of calendars in the list? */
  int  no_of_cals;

  /* Delta in minutes between time ticks. */
  int  entry_delta;
  int  old_entry_delta;

  /* Number of days to display. */
  int  show_no_days;

  /* ID for updates. */
  int  update_id;

  /* Various flags to use. */
  UINT32  flags;

  /* Pixels per minute in the scale. */
  float  pixel_per_min;

  /* Width of an individual day. */
  Dimension  day_width;

  /* Height of the time label. */
  Dimension  time_label_height;

  /* Height of a single line in the time label. */
  Dimension time_line_height;

  /* Planner window. */
  Widget  plannerW;

  /* Entries for the planner. */
  LST_DESC_TYPE  plan_what_ref;

  /* Display planner from this date. */
  TIM_TIME_REF  planner_start;
  TIM_TIME_REF  old_planner_start;

  /* Display planner to this date. */
  TIM_TIME_REF  planner_stop;
  TIM_TIME_REF  old_planner_stop;

  /* Start display at this time. */
  TIM_TIME_REF  default_entry_start;
  TIM_TIME_REF  entry_start;
  TIM_TIME_REF  old_entry_start;

  /* Stop display at this time. */
  TIM_TIME_REF  default_entry_stop;
  TIM_TIME_REF  entry_stop;
  TIM_TIME_REF  old_entry_stop;

  /* Application wide resources. */
  XTM_GL_BASE_DATA_REF  appl_data_ref;

  /* Additional calendar handle. */
  XTM_OV_HANDLE  open_handle;

  /* Message window. */
  XTM_MS_HANDLE  msg_handle;

  /* Navigation calendar. */
  XTM_NC_HANDLE  nav_cal_handle;

  /* Callback to inform our creator of specific actions. */
  void              *user_data;
  XTM_PL_ACTION_CB  actionCB;

} PLAN_REC, *PLAN_REC_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmPlan";

/* IDs for the help windows. */
static char  *planner_window_id = "Planner";

/* Cache entries. */
static Boolean       init_cache = True;
static PLAN_REC_REF  cache_entries[ MAX_CACHE_ENTRIES ];


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void 
  addCalendarCB( Widget        widget,
                 PLAN_REC_REF  plan_ref,
                 XtPointer     call_data );

static void
  addCalendarApplyCB( XTM_OV_REASON    reason,
                      XTM_CD_CAL_INFO  *db_info_ref,
                      void             *user_data );

static void 
  applyCB( Widget        widget,
           PLAN_REC_REF  plan_ref,
           XtPointer     call_data );

static void 
  calendarSelectedCB( Widget                        widget,
                      PLAN_REC_REF                  plan_ref,
                      XmToggleButtonCallbackStruct  *call_data );

static void 
  clearCB( Widget        widget,
           PLAN_REC_REF  plan_ref,
           XtPointer     call_data );

static void 
  clearCalsAndGroups( PLAN_REC_REF  plan_ref,
                      Boolean       do_update );

static void 
  closeCB( Widget        widget,
           PLAN_REC_REF  plan_ref,
           XtPointer     call_data );

static Widget
  createPlanWindow( Widget        parent,
                    PLAN_REC_REF  plan_ref );

static void 
  ctrlMenuCB( Widget                     widget,
              PLAN_REC_REF               plan_ref,
              XmRowColumnCallbackStruct  *call_data );

static void
  ctrlMenuMapCB( Widget        widget,
                 PLAN_REC_REF  plan_ref,
                 XtPointer     call_data );

static void 
  destroyCB( Widget        widget,
             PLAN_REC_REF  plan_ref,
             XtPointer     call_data );

static void 
  doMsgDateCB( Widget        widget,
               PLAN_REC_REF  plan_ref,
               XtPointer     call_data );

static FIND_RESULT
  fetchCal( LST_DESC_TYPE  cals_ref,
            char           *search_cal,
            PLAN_CAL_REF   *cal_ref );

static Widget
  fetchCalendarWidget( PLAN_REC_REF  plan_ref,
                       char          *search_cal );

static FIND_RESULT
  fetchDates( PLAN_REC_REF    plan_ref,
              TIM_TIME_REF    start_date,
              TIM_TIME_REF    to_date,
              PLAN_DATES_REF  *plan_dates_ref );

static void 
  fileMenuCB( Widget                     widget,
              PLAN_REC_REF               plan_ref,
              XmRowColumnCallbackStruct  *call_data );

static void 
  groupSelectedCB( Widget                        widget,
                   PLAN_REC_REF                  plan_ref,
                   XmToggleButtonCallbackStruct  *call_data );

static void 
  infoCB( Widget                     widget,
          PLAN_REC_REF               plan_ref,
          XmRowColumnCallbackStruct  *call_data );

static void
  loadEntries( PLAN_REC_REF   plan_ref,
               TIM_TIME_REF   start_date,
               TIM_TIME_REF   to_date,
               char           *cal_name,
               LST_DESC_TYPE  entries_ref );

static void
  loadPlan( PLAN_REC_REF  plan_ref,
            TIM_TIME_REF  start_date,
            TIM_TIME_REF  to_date );

static void 
  navigateCB( Widget                        widget,
              PLAN_REC_REF                  plan_ref,
              XmUbArrowLabelCallbackStruct  *call_data );

static void 
  optMenuCB( Widget                     widget,
             PLAN_REC_REF               plan_ref,
             XmRowColumnCallbackStruct  *call_data );

static void 
  presentBusyMarker( PLAN_REC_REF  plan_ref,
                     Widget        parentW,
                     int           day_no,
                     int           start_min,
                     int           stop_min,
                     int           minute_offset,
                     int           *marker_index );

static void 
  refreshCB( Widget        widget,
             PLAN_REC_REF  plan_ref,
             XtPointer     call_data );

void 
  scrollHorizCB( Widget                     widget,
                 PLAN_REC_REF               plan_ref,
                 XmScrollBarCallbackStruct  *call_data );

void 
  scrollVertCB( Widget                     widget,
                PLAN_REC_REF               plan_ref,
                XmScrollBarCallbackStruct  *call_data );

static void 
  scrollWindow( PLAN_REC_REF  plan_ref,
                Widget        scrollBar,
                int           value );

static void 
  sendMessage( PLAN_REC_REF  plan_ref,
               TIM_TIME_REF  to_date );

static void
  setAdditionalCals( PLAN_REC_REF  plan_ref );

static void
  setCalsAndGroups( PLAN_REC_REF  plan_ref );

static void
  startOnDay( PLAN_REC_REF  plan_ref,
              TIM_TIME_REF  *start_time );

static void
  updateCB( UINT32  flags,
            void    *user_data,
            void    *update_user_data );

static void
  updateDayNames( PLAN_REC_REF  plan_ref,
                  TIM_TIME_REF  start_date,
                  TIM_TIME_REF  stop_date );

static void
  updateEntries( PLAN_REC_REF  plan_ref,
                 TIM_TIME_REF  start_date,
                 TIM_TIME_REF  stop_date );

static void
  updateEntriesBg( PLAN_REC_REF  plan_ref,
                   TIM_TIME_REF  start_date,
                   TIM_TIME_REF  stop_date );

static void
  updatePlanner( PLAN_REC_REF  plan_ref );

static void
  updatePlannerDates( PLAN_REC_REF  plan_ref,
                      Boolean       fetch );

static void
  updateTimeScale( PLAN_REC_REF  plan_ref,
                   TIM_TIME_REF  start_time,
                   TIM_TIME_REF  stop_time );

static void 
  zoomCB( Widget        widget,
          PLAN_REC_REF  plan_ref,
          XtPointer     call_data );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

XTM_PL_HANDLE
  xtmPlInitialize( XTM_GL_BASE_DATA_REF  appl_data_ref,
                   Widget                toplevel,
                   XTM_PL_ACTION_CB      actionCB,
                   void                  *user_data )
{

  /* Variables. */
  int                     index;
  PLAN_REC_REF            plan_ref;
  TIM_TIME_REF            now;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  custom_data_ref = appl_data_ref -> custom_data;

  now = TimLocalTime( TimMakeTimeNow() );
  now = TimMakeTime(  TimIndexOfYear(  now ),
                      TimIndexOfMonth( now ),
                      TimIndexOfDay(   now ),
                      0, 0, 0 );

  /* Initialize the cache? */
  if( init_cache ) {
    for( index = 0; index < MAX_CACHE_ENTRIES; index++ )
      cache_entries[ index ] = NULL;

    init_cache = False;
  }


  /* Anything in the cache? */
  plan_ref = NULL;

  for( index = 0; index < MAX_CACHE_ENTRIES; index++ ) {
    if( cache_entries[ index ] != NULL ) {
      plan_ref = cache_entries[ index ];
      cache_entries[ index ] = NULL;

      break;
    }
  }

  /* Create the planner record? */
  if( plan_ref == NULL ) {

    plan_ref = SysNew( PLAN_REC );
    if( plan_ref == NULL )
      return( NULL );

    plan_ref -> plannerW = NULL;

  } /* if */

  plan_ref -> appl_data_ref  = appl_data_ref;
  plan_ref -> day_width      = 50;
  plan_ref -> instant_update = False;
  plan_ref -> show_no_days   = 7;
  plan_ref -> update_id      = 0;
  plan_ref -> no_of_cals     = 0;
  plan_ref -> msg_handle     = NULL;
  plan_ref -> nav_cal_handle = NULL;
  plan_ref -> open_handle    = NULL;
  plan_ref -> actionCB       = actionCB;
  plan_ref -> user_data      = user_data;

  plan_ref -> entry_delta     = custom_data_ref -> default_entry_delta;
  plan_ref -> old_entry_delta = 0;

  plan_ref -> planner_start     = now;
  plan_ref -> old_planner_start = (TIM_TIME_REF) 0;
  plan_ref -> planner_stop      = now;
  plan_ref -> old_planner_stop  = (TIM_TIME_REF) 0;

  plan_ref -> old_entry_start = (TIM_TIME_REF) 0;
  plan_ref -> entry_start     = TimMakeTime( 1970, 1, 1, 
                                             custom_data_ref -> start_hour,
                                             0, 0 );

  plan_ref -> old_entry_stop  = (TIM_TIME_REF) 0;
  plan_ref -> entry_stop      = TimMakeTime( 1970, 1, 1, 
                                             custom_data_ref -> stop_hour,
                                             0, 0 );

  plan_ref -> default_entry_start = plan_ref -> entry_start;
  plan_ref -> default_entry_stop  = plan_ref -> entry_stop;

  /* Create the planner window and entry cache? */
  if( plan_ref -> plannerW == NULL ) {

    /* Plan window. */
    plan_ref -> plannerW = createPlanWindow( toplevel, plan_ref );

    if( plan_ref -> plannerW == NULL ) {
      SysFree( plan_ref );
      return( NULL );
    }

    plan_ref -> first_time    = True;
    plan_ref -> plan_what_ref = LstLinkNew( sizeof( PLAN_DATES ), NULL );

    /* The Busy-times matrix. */
    plan_ref -> busy_days = (Boolean **)
                               SysMalloc( sizeof( Boolean **) *
                                          plan_ref -> show_no_days );

    for( index = 0; index < plan_ref -> show_no_days; index++ ) {
      plan_ref -> busy_days[ index ] = 
        (Boolean *) SysMalloc( 24 * 60 * sizeof( Boolean ) );
    }

  } /* if */


  return( (XTM_PL_HANDLE) plan_ref );

} /* xtmPlInitialize */


/*----------------------------------------------------------------------*/

void
  xtmPlEmptyCache()
{

  /* Variables. */
  int  index;


  /* Code. */

  if( ! init_cache )
    return;

  for( index = 0; index < MAX_CACHE_ENTRIES; index++ ) {
    if( cache_entries[ index ] != NULL )
      XtDestroyWidget( cache_entries[ index ] -> plannerW );

    cache_entries[ index ] = NULL;
  }


  return;

} /* xtmPlEmptyCache */


/*----------------------------------------------------------------------*/

void
  xtmPlDestroy( XTM_PL_HANDLE  plan_handle )
{

  /* Variables. */
  PLAN_REC_REF  plan_ref;


  /* Code. */

  if( plan_handle == NULL )
    return;

  /* Our private data. */
  plan_ref = (PLAN_REC_REF) plan_handle;

  /* Destroy the planner widget. */
  if( plan_ref -> plannerW != NULL )
    closeCB( NULL, plan_ref, NULL );


  return;

} /* xtmPlDestroy */


/*----------------------------------------------------------------------*/

void
  xtmPlViewPlanner( XTM_PL_HANDLE  plan_handle,
                    TIM_TIME_REF   start_date,
                    Boolean        iconic )
{

  /* Variables. */
  Arg                     args[ 10 ];
  Cardinal                n;
  PLAN_REC_REF            plan_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  if( plan_handle == NULL )
    return;

  /* Our private data. */
  plan_ref = (PLAN_REC_REF) plan_handle;

  custom_data_ref = plan_ref -> appl_data_ref -> custom_data;


  /* Start the planner view on selected day. */
  if( start_date != 0 )
    plan_ref -> planner_start = start_date;

  startOnDay( plan_ref, &plan_ref -> planner_start );


  /* Start as icon? */
  n = 0;
  XtSetArg( args[ n ], XmNiconic, iconic ); n++;
  XtSetValues( plan_ref -> plannerW, args, n );


  /* Make sure the planner window is visable. */
  XtPopup( plan_ref -> plannerW, XtGrabNone );


  if( ! iconic ) {
    XRaiseWindow( XtDisplay( plan_ref -> plannerW ), 
                  XtWindow(  plan_ref -> plannerW ) );

    XtMapWidget( plan_ref -> plannerW );
  }

  /* Don't let our children spoil our size. */
  n = 0;
  XtSetArg( args[ n ], XmNallowShellResize, False ); n++;
  XtSetValues( plan_ref -> plannerW, args, n );


  /* Register for updates? */
  if( plan_ref -> update_id == 0 )
    plan_ref -> update_id = xtmUpRegister( XTM_UP_PLANNER,
                                           updateCB, (void *) plan_ref );


  /* Display default calendar if first time. */
  if( plan_ref -> first_time ) {

    Boolean          ok;
    Widget           calW;
    XTM_CD_CAL_INFO  db_info;

    /* Calendars and groups. */
    setCalsAndGroups( plan_ref );

    /* Select the default calendar. */
    ok = xtmCdFetchDefaultDb( custom_data_ref -> cal_db_handle,
                              &db_info, NULL );
    if( ok ) {
      calW = fetchCalendarWidget( plan_ref, db_info.short_name );

      if( calW != NULL )
        XmToggleButtonSetState( calW, True, False );
    }

    plan_ref -> first_time = False;

  /* We have been here before. */
  } else {
    setAdditionalCals( plan_ref );

  } /* if */


  /* Update the planner with current data. */
  updatePlannerDates( plan_ref, False );
  updatePlanner( plan_ref );


  return;

} /* xtmPlViewPlanner */


/*----------------------------------------------------------------------*/

static void 
  clearCalsAndGroups( PLAN_REC_REF  plan_ref,
                      Boolean       do_update )
{

  /* Variables. */
  int       index;
  char      buffer[ 50 ];
  Arg       args[ 10 ];
  Cardinal  n;
  Widget    baseW;
  Widget    mainW;
  Widget    tempW;


  /* Code. */

  mainW = XtNameToWidget( plan_ref -> plannerW, "PlanTlBase.PlanTlFo" );

  /* Clear all calendars. */
  baseW = XtNameToWidget( mainW,
            "PlanWhatFo.CalSelSw.ClipWindow.CalSelRc" );

  index = 1;
  do {
    sprintf( buffer, "WhatCal%d", index );

    tempW = XtNameToWidget( baseW, buffer );
    if( tempW == NULL )
      break;

    XmToggleButtonSetState( tempW, False, False );

    n = 0;
    XtSetArg( args[ n ], XmNuserData, (XtPointer) 0 ); n++;
    XtGetValues( tempW, args, n );

    index++;
  } while( True );


  /* Clear all groups. */
  baseW = XtNameToWidget( mainW,
            "PlanWhatFo.GrpSelSw.ClipWindow.GrpSelRc" );

  index = 1;
  do {
    sprintf( buffer, "WhatGroup%d", index );

    tempW = XtNameToWidget( baseW, buffer );
    if( tempW == NULL )
      break;

    XmToggleButtonSetState( tempW, False, False );

    index++;
  } while( True );


  /* Refresh the planner. */
  if( do_update )
    refreshCB( NULL, plan_ref, NULL );


  return;

} /* clearCalsAndGroups */


/*----------------------------------------------------------------------*/

static Widget
  createPlanWindow( Widget        parent,
                    PLAN_REC_REF  plan_ref )
{

  /* Variables. */
  int                     index;
  char                    buffer[ 128 ];
  Arg                     args[ 10 ];
  Cardinal                n;
  Dimension               height;
  Dimension               width;
  Widget                  dataLocalW[ 26 ];
  Widget                  menuCasc[ 4 ];
  Widget                  menuCtrlBu[ 1 ];
  Widget                  menuFileBu[ 1 ];
  Widget                  menuHelpBu[ 7 ];
  Widget                  menuOptBu[ 3 ];
  Widget                  pulldownMenu[ 4 ];
  Widget                  planTl;
  Widget                  tempW;
  Widget                  toplevel;
  Widget                  workFo;
  XmString                xstr;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;

  static char  *zoom_times = "0:10 0:15 0:30 0:45 1:00 1:30 2:00 4:00 6:00";

  static char  *pull_downs[] = { 
    "pdown1", "pdown2", "pdown3", "pdown4" };

  static XIT_PUSH_STRUCT button_def[] = {
    { "ClearPb",  "", "", True, NULL },
    { "AddCalPb", "", "", True, NULL },
  };

  static XIT_CASCADE_STRUCT menupane[] = {
    { "", "", "FilePane" },
    { "", "", "CtrlPane" },
    { "", "", "OptPane" },
    { "", "", "HelpPane" },
  };

  static XIT_MENU_BUTTON_STRUCT file_casc[] = {
    { "", "",  NULL, "CloseBu", True,  False, False },
  };

  static XIT_MENU_BUTTON_STRUCT ctrl_casc[] = {
    { "", "",  NULL, "InstantUpdateBu", True, True, False },
  };

  static XIT_MENU_BUTTON_STRUCT opt_casc[] = {
    { "",      "",  NULL, "SendMsgBu",   True,  False, False },
    { XIT_SEP, " ", NULL, "",            False, False, False },
    { "",      "",  NULL, "RedisplayBu", True,  False, False },
  };

  static XIT_MENU_BUTTON_STRUCT help_casc[] = {
    { "", "", NULL, "ContextBu", True, False, False },
    { "", "", NULL, "WindowsBu", True, False, False },
    { "", "", NULL, "KeysBu",    True, False, False },
    { "", "", NULL, "IndexBu",   True, False, False },
    { "", "", NULL, "HelpBu",    True, False, False },
    { "", "", NULL, "VersionBu", True, False, False },
    { "", "", NULL, "AboutBu",   True, False, False },
  };

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "",   applyCB,   NULL },
    { NULL, NULL,      NULL },
    { "",   refreshCB, NULL },
    { NULL, NULL,      NULL },
    { "",   closeCB,   NULL },
  };


  /* Code. */

  /* Initialize the menu and label items. */
  action_buttons[ 0 ].label = msgGetText( MXDI_APPLY_BUTTON );
  action_buttons[ 0 ].data  = plan_ref;
  action_buttons[ 2 ].label = msgGetText( MXDI_REFRESH_BUTTON );
  action_buttons[ 2 ].data  = plan_ref;
  action_buttons[ 4 ].label = msgGetText( MXDI_CLOSE_BUTTON );
  action_buttons[ 4 ].data  = plan_ref;

  menupane[ 0 ].title    = msgGetText( MXDI_FILE_MENU );
  menupane[ 0 ].mnemonic = msgGetText( MXDI_FILE_MENU_ACC );
  menupane[ 1 ].title    = msgGetText( MXDI_CTRL_MENU );
  menupane[ 1 ].mnemonic = msgGetText( MXDI_CTRL_MENU_ACC );
  menupane[ 2 ].title    = msgGetText( MXDI_OPT_MENU );
  menupane[ 2 ].mnemonic = msgGetText( MXDI_OPT_MENU_ACC );
  menupane[ 3 ].title    = msgGetText( MXDI_HELP_MENU );
  menupane[ 3 ].mnemonic = msgGetText( MXDI_HELP_MENU_ACC );

  file_casc[ 0 ].title    = msgGetText( MXDI_CLOSE_MENU );
  file_casc[ 0 ].mnemonic = msgGetText( MXDI_CLOSE_MENU_ACC );

  ctrl_casc[ 0 ].title    = msgGetText( MXDI_INSTANT_UPDATE_MENU );
  ctrl_casc[ 0 ].mnemonic = msgGetText( MXDI_INSTANT_UPDATE_MENU_ACC );

  opt_casc[ 0 ].title    = msgGetText( MXDI_SEND_MSG_MENU );
  opt_casc[ 0 ].mnemonic = msgGetText( MXDI_SEND_MSG_MENU_ACC );
  opt_casc[ 2 ].title    = msgGetText( MXDI_REDISPLAY_MENU );
  opt_casc[ 2 ].mnemonic = msgGetText( MXDI_REDISPLAY_MENU_ACC );

  help_casc[ 0 ].title    = msgGetText( MXDI_HELP_CONTEXT );
  help_casc[ 0 ].mnemonic = msgGetText( MXDI_HELP_CONTEXT_ACC );
  help_casc[ 1 ].title    = msgGetText( MXDI_HELP_WINDOWS );
  help_casc[ 1 ].mnemonic = msgGetText( MXDI_HELP_WINDOWS_ACC );
  help_casc[ 2 ].title    = msgGetText( MXDI_HELP_KEYS );
  help_casc[ 2 ].mnemonic = msgGetText( MXDI_HELP_KEYS_ACC );
  help_casc[ 3 ].title    = msgGetText( MXDI_HELP_INDEX );
  help_casc[ 3 ].mnemonic = msgGetText( MXDI_HELP_INDEX_ACC );
  help_casc[ 4 ].title    = msgGetText( MXDI_HELP_HELP );
  help_casc[ 4 ].mnemonic = msgGetText( MXDI_HELP_HELP_ACC );
  help_casc[ 5 ].title    = msgGetText( MXDI_HELP_VERSION );
  help_casc[ 5 ].mnemonic = msgGetText( MXDI_HELP_VERSION_ACC );
  help_casc[ 6 ].title    = msgGetText( MXDI_HELP_ABOUT );
  help_casc[ 6 ].mnemonic = msgGetText( MXDI_HELP_ABOUT_ACC );

  button_def[ 0 ].title   = msgGetText( MXDI_CLEAR_BUTTON );
  button_def[ 1 ].title   = msgGetText( MXDI_ADD_CAL_BUTTON );

  custom_data_ref = plan_ref -> appl_data_ref -> custom_data;


  /* Create a toplevel dialog with buttons. */
  toplevel = xitGetToplevelWidget( parent );

  planTl = xitCreateToplevelDialog( toplevel, "PlanTl",
                                    1, 0,
                                    action_buttons,
                                    XtNumber( action_buttons ) );

  XtAddCallback( planTl, XmNdestroyCallback, 
                 (XtCallbackProc) destroyCB, (XtPointer) plan_ref );


  /* Window title and icon name. */
  n = 0;
  XtSetArg( args[ n ], XmNtitle, msgGetText( MXDI_PLAN_TITLE ) ); n++;
  XtSetArg( args[ n ], XmNiconName, msgGetText( MXDI_PLAN_ICON_TITLE ) ); n++;
  XtSetValues( planTl, args, n );


  /* Exit the application if this window is deleted. */
  {
    Atom  wm_delete_window;

    wm_delete_window = XmInternAtom( XtDisplay( planTl ),
                                     "WM_DELETE_WINDOW", False );

    XmAddWMProtocols( planTl, &wm_delete_window, 1 );
    XmAddWMProtocolCallback( planTl, wm_delete_window, 
                             (XtCallbackProc) closeCB, (XtPointer) plan_ref );
  } /* block */


  /* Container for the contents of the window. */
  workFo = XtNameToWidget( planTl, "PlanTlBase.PlanTlFo" );

  XtAddCallback( planTl, XmNdestroyCallback, 
                 (XtCallbackProc) destroyCB, (XtPointer) plan_ref );


  /* Create the menubar and menu cascades. */
  menuBr = XmCreateMenuBar( workFo, "MenuBr", args, 0 );

  n = 0;
  for( index = 0; index < XtNumber( pulldownMenu ); index++ )
    pulldownMenu[ index ] = XmCreatePulldownMenu( menuBr, 
                                                  pull_downs[ index ], 
                                                  NULL, n );

  for( index = 0; index < XtNumber( menuCasc ); index++ )
    menuCasc[ index ] = xitCreateCascadeButton( menuBr, 
                                                pulldownMenu[ index ], 
                                                &menupane[ index ] );

  /* The help button should be placed to the right. */
  index = XtNumber( menuCasc ) - 1;
  n     = 0;
  XtSetArg( args[ n ], XmNmenuHelpWidget, menuCasc[ index ] ); n++;
  XtSetValues( menuBr, args, n );


  /* Create the file menu. */
  XtAddCallback( pulldownMenu[ 0 ], XmNentryCallback, 
                 (XtCallbackProc) fileMenuCB, (XtPointer) plan_ref );

  for( index = 0; index < XtNumber( menuFileBu ); index++ ) {
    menuFileBu[ index ] = xitCreateMenuPushButton( pulldownMenu[ 0 ], 
                                                   &file_casc[ index ] );

    if( XmIsPushButton( menuFileBu[ index ] ) )
      XtAddCallback( menuFileBu[ index ], XmNactivateCallback, 
                     (XtCallbackProc) fileMenuCB, (XtPointer) index );
  }

  /* Create the control menu. */
  XtAddCallback( pulldownMenu[ 1 ], XmNentryCallback, 
                 (XtCallbackProc) ctrlMenuCB, (XtPointer) plan_ref );
  XtAddCallback( pulldownMenu[ 1 ],  XmNmapCallback, 
                 (XtCallbackProc) ctrlMenuMapCB, (XtPointer) plan_ref );

  for( index = 0; index < XtNumber( menuCtrlBu ); index++ ) {
    menuCtrlBu[ index ] = xitCreateMenuPushButton( pulldownMenu[ 1 ], 
                                                   &ctrl_casc[ index ] );

    if( XmIsPushButton( menuCtrlBu[ index ] ) )
      XtAddCallback( menuCtrlBu[ index ], XmNactivateCallback, 
                     (XtCallbackProc) ctrlMenuCB, (XtPointer) index );
    else if( XmIsToggleButton( menuCtrlBu[ index ] ) )
      XtAddCallback( menuCtrlBu[ index ], XmNvalueChangedCallback, 
                     (XtCallbackProc) ctrlMenuCB, (XtPointer) index );
  }

  /* Create the options menu. */
  XtAddCallback( pulldownMenu[ 2 ], XmNentryCallback, 
                 (XtCallbackProc) optMenuCB, (XtPointer) plan_ref );

  for( index = 0; index < XtNumber( menuOptBu ); index++ ) {
    menuOptBu[ index ] = xitCreateMenuPushButton( pulldownMenu[ 2 ], 
                                                  &opt_casc[ index ] );

    if( XmIsPushButton( menuOptBu[ index ] ) )
      XtAddCallback( menuOptBu[ index ], XmNactivateCallback, 
                     (XtCallbackProc) optMenuCB, (XtPointer) index );
  }

  /* Create the help menu. */
  XtAddCallback( pulldownMenu[ 3 ], XmNentryCallback, 
                 (XtCallbackProc) infoCB, (XtPointer) plan_ref );

  for( index = 0; index < XtNumber( menuHelpBu ); index++ ) {
    menuHelpBu[ index ] = xitCreateMenuPushButton( pulldownMenu[ 3 ], 
                                                   &help_casc[ index ] );

    if( XmIsPushButton( menuHelpBu[ index ] ) )
      XtAddCallback( menuHelpBu[ index ], XmNactivateCallback, 
                     (XtCallbackProc) infoCB, (XtPointer) index );
  }

  /* We can't do context sensitive help. */
  XtSetSensitive( menuHelpBu[ 0 ], False );


  /* Some instructions. */
  howToDoLa = xitCreateLabel( workFo, "HowToDoLa",
                              msgGetText( MXDI_HOW_TO_PLAN_LABEL ), -1 );


  /* Create a select date form. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  dateRc = XmCreateRowColumn( workFo, "DateRc", args, n );

  dateLa = xitCreateLabel( dateRc, "DateLa",
                           msgGetText( MXDI_SCHED_BTW_DATES ), -1 );

  /* From date. */
  dateTb = xtmFlCreateDateField( dateRc, "DateTb" );
  XtAddCallback( dateTb,  XmNactivateCallback, 
                 (XtCallbackProc) applyCB, (XtPointer) plan_ref );

  dateInfoLa = xitCreateLabel( dateRc, "DateInfoLa", " ", -1 );


  /* Time scale zoom. */
  zoomFs = xtmFlCreateSelField( workFo, "ZoomFs",
                                4, zoom_times, ' ', True,
                                (XtCallbackProc) zoomCB,
                                (XtPointer) plan_ref );

  tempW = xitFieldSelectGetChild( zoomFs, xitFIELD_SELECT_TEXT_FIELD );

  n = 0;
  XtSetArg( args[ n ], XmNeditable, False ); n++;
  XtSetArg( args[ n ], XmNcursorPositionVisible, False ); n++;
  XtSetValues( tempW, args, n );

  sprintf( buffer, "%1d:%2.2d",
           plan_ref -> entry_delta / 60,
           plan_ref -> entry_delta % 60 ); 

  xitFieldSelectSetCurrent( zoomFs, buffer, False );


  /* Day display window. */
  n = 0;
  XtSetArg( args[ n ], XmNscrollBarDisplayPolicy, XmSTATIC ); n++;
  XtSetArg( args[ n ], XmNscrollingPolicy, XmAUTOMATIC ); n++;
  XtSetArg( args[ n ], XmNscrollBarPlacement, XmTOP_RIGHT ); n++;
  XtSetArg( args[ n ], XmNshadowThickness, 1 );  n++;
  XtSetArg( args[ n ], XmNspacing, 0 );  n++;
  dayDispSw = XmCreateScrolledWindow( workFo, "DayDispSw", args, n );

  n = 0;
  XtSetArg( args[ n ], XmNhorizontalScrollBar, &tempW ); n++;
  XtGetValues( dayDispSw, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNheight, 1 ); n++;
  XtSetArg( args[ n ], XmNmappedWhenManaged, False ); n++;
  XtSetValues( tempW, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNallowOverlap, True ); n++;
  XtSetArg( args[ n ], XmNmarginHeight, 0 ); n++;
  XtSetArg( args[ n ], XmNmarginWidth, 0 ); n++;
  XtSetArg( args[ n ], XmNresizePolicy, XmRESIZE_NONE ); n++;
  dayDispBb = XmCreateBulletinBoard( dayDispSw, "DayDispBb", args, n );

  tempW = xitCreateLabel( dayDispBb, "", " ", -1 );

  n = 0;
  XtSetArg( args[ n ], XmNheight, &height ); n++;
  XtGetValues( tempW, args, n );

  XtDestroyWidget( tempW );

  n = 0;
  XtSetArg( args[ n ], XmNheight, height + 10 ); n++;
  XtSetValues( dayDispBb, args, n );


  /* Time display window. */
  n = 0;
  XtSetArg( args[ n ], XmNscrollBarDisplayPolicy, XmSTATIC ); n++;
  XtSetArg( args[ n ], XmNscrollingPolicy, XmAUTOMATIC ); n++;
  XtSetArg( args[ n ], XmNscrollBarPlacement, XmBOTTOM_LEFT ); n++;
  XtSetArg( args[ n ], XmNshadowThickness, 1 );  n++;
  timeDispSw = XmCreateScrolledWindow( workFo, "TimeDispSw", args, n );

  n = 0;
  XtSetArg( args[ n ], XmNverticalScrollBar, &tempW ); n++;
  XtGetValues( timeDispSw, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNwidth, 1 ); n++;
  XtSetArg( args[ n ], XmNmappedWhenManaged, False ); n++;
  XtSetValues( tempW, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNallowOverlap, True ); n++;
  XtSetArg( args[ n ], XmNmarginHeight, 0 ); n++;
  XtSetArg( args[ n ], XmNmarginWidth, 0 ); n++;
  timeDispBb = XmCreateBulletinBoard( timeDispSw, "TimeDispBb", args, n );

  /* Label with the times. */
  timeDispLa = xitCreateLabelWidget( timeDispBb, "TimeDispLa", 
                                     "        ", -1 );

  n = 0;
  XtSetArg( args[ n ], XmNx, 0 ); n++;
  XtSetArg( args[ n ], XmNy, 0 ); n++;
  XtSetValues( timeDispLa, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNwidth, &width ); n++;
  XtGetValues( timeDispLa, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNwidth, width + 15 ); n++;
  XtSetValues( timeDispSw, args, n );


  /* Planner window. */
  n = 0;
  XtSetArg( args[ n ], XmNscrollBarDisplayPolicy, XmSTATIC ); n++;
  XtSetArg( args[ n ], XmNscrollingPolicy, XmAUTOMATIC ); n++;
  XtSetArg( args[ n ], XmNshadowThickness, 1 );  n++;
  planSw = XmCreateScrolledWindow( workFo, "PlanSw", args, n );

  n = 0;
  XtSetArg( args[ n ], XmNhorizontalScrollBar, &tempW ); n++;
  XtGetValues( planSw, args, n );

  XtAddCallback( tempW, XmNincrementCallback, 
                 (XtCallbackProc) scrollHorizCB,
                 (XtPointer) plan_ref );
  XtAddCallback( tempW, XmNdecrementCallback, 
                 (XtCallbackProc) scrollHorizCB,
                 (XtPointer) plan_ref );
  XtAddCallback( tempW, XmNpageIncrementCallback, 
                 (XtCallbackProc) scrollHorizCB,
                 (XtPointer) plan_ref );
  XtAddCallback( tempW, XmNpageDecrementCallback, 
                 (XtCallbackProc) scrollHorizCB,
                 (XtPointer) plan_ref );
  XtAddCallback( tempW, XmNvalueChangedCallback, 
                 (XtCallbackProc) scrollHorizCB,
                 (XtPointer) plan_ref );
  XtAddCallback( tempW, XmNdragCallback, 
                 (XtCallbackProc) scrollHorizCB,
                 (XtPointer) plan_ref );

  n = 0;
  XtSetArg( args[ n ], XmNverticalScrollBar, &tempW ); n++;
  XtGetValues( planSw, args, n );

  XtAddCallback( tempW, XmNincrementCallback, 
                 (XtCallbackProc) scrollVertCB,
                 (XtPointer) plan_ref );
  XtAddCallback( tempW, XmNdecrementCallback, 
                 (XtCallbackProc) scrollVertCB,
                 (XtPointer) plan_ref );
  XtAddCallback( tempW, XmNpageIncrementCallback, 
                 (XtCallbackProc) scrollVertCB,
                 (XtPointer) plan_ref );
  XtAddCallback( tempW, XmNpageDecrementCallback, 
                 (XtCallbackProc) scrollVertCB,
                 (XtPointer) plan_ref );
  XtAddCallback( tempW, XmNvalueChangedCallback, 
                 (XtCallbackProc) scrollVertCB,
                 (XtPointer) plan_ref );
  XtAddCallback( tempW, XmNdragCallback, 
                 (XtCallbackProc) scrollVertCB,
                 (XtPointer) plan_ref );

  n = 0;
  XtSetArg( args[ n ], XmNallowOverlap, True ); n++;
  XtSetArg( args[ n ], XmNmarginHeight, 0 ); n++;
  XtSetArg( args[ n ], XmNmarginWidth, 0 ); n++;
  XtSetArg( args[ n ], XmNresizePolicy, XmRESIZE_NONE ); n++;
  planBb = XmCreateBulletinBoard( planSw, "PlanBb", args, n );


  /* Calender and group selector. */
  planWhatFo = XmCreateForm( workFo, "PlanWhatFo", args, n );

  /* The calendar selector. */
  calSelLa = xitCreateLabel( planWhatFo, "CalSelLa",
                             msgGetText( MXDI_MSG_DB_LIST_LABEL ), -1 );

  n = 0;
  XtSetArg( args[ n ], XmNscrollBarDisplayPolicy, XmSTATIC ); n++;
  XtSetArg( args[ n ], XmNscrollingPolicy, XmAUTOMATIC ); n++;
  calSelSw = XmCreateScrolledWindow( planWhatFo, "CalSelSw", args, n );

  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmVERTICAL ); n++;
  calSelRc = XmCreateRowColumn( calSelSw, "CalSelRc", args, n );


  /* The group selector. */
  grpSelLa = xitCreateLabel( planWhatFo, "GrpSelLa",
                             msgGetText( MXDI_MSG_GROUP_LIST_LABEL ), -1 );

  n = 0;
  XtSetArg( args[ n ], XmNscrollBarDisplayPolicy, XmSTATIC ); n++;
  XtSetArg( args[ n ], XmNscrollingPolicy, XmAUTOMATIC ); n++;
  grpSelSw = XmCreateScrolledWindow( planWhatFo, "GrpSelSw", args, n );

  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmVERTICAL ); n++;
  XtSetArg( args[ n ], XmNorientation, XmVERTICAL ); n++;
  grpSelRc = XmCreateRowColumn( grpSelSw, "GrpSelRc", args, n );


  /* Create the navigate buttons. */
  n = 0;
  navFo = XmCreateForm( workFo, "NavFo", args, n );


  /* Create next week action. */
  xstr = XmStringCreateLtoR( msgGetText( MXDI_WEEK_LABEL ), CS );

  n = 0;
  XtSetArg( args[ n ], XmNlabelString, xstr ); n++;
  navWeekAl = XmUbCreateArrowLabel( navFo, "NavWeekAl", args, n );

  XmStringFree( xstr );

  XtAddCallback( navWeekAl, XmNactivateCallback,
                 (XtCallbackProc) navigateCB, (XtPointer) plan_ref );


  /* Create next month action. */
  xstr = XmStringCreateLtoR( msgGetText( MXDI_MONTH_LABEL ), CS );

  n = 0;
  XtSetArg( args[ n ], XmNlabelString, xstr ); n++;
  navMonthAl = XmUbCreateArrowLabel( navFo, "NavMonthAl", args, n );

  XmStringFree( xstr );

  XtAddCallback( navMonthAl, XmNactivateCallback,
                 (XtCallbackProc) navigateCB, (XtPointer) plan_ref );


  /* The 'add another cal ' button. */
  addCalPb = xitCreatePushButton( workFo, &button_def[ 1 ] );

  XtAddCallback( addCalPb, XmNactivateCallback, 
                 (XtCallbackProc) addCalendarCB, (XtPointer) plan_ref );

  /* The 'clear' button. */
  clearPb = xitCreatePushButton( workFo, &button_def[ 0 ] );

  XtAddCallback( clearPb, XmNactivateCallback, 
                 (XtCallbackProc) clearCB, (XtPointer) plan_ref );


  /* Parts in the planWhat form. */
  xitAttachWidget( calSelLa,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( calSelSw,
                   XmATTACH_WIDGET, calSelLa, XmATTACH_FORM, NULL,
                   XmATTACH_FORM,   NULL,     XmATTACH_NONE, NULL );
  xitAttachWidget( grpSelLa,
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( grpSelSw,
                   XmATTACH_WIDGET, grpSelLa, XmATTACH_FORM, NULL,
                   XmATTACH_FORM,   NULL,     XmATTACH_FORM, NULL );

  n = 0;
  XtSetArg( args[ n ], XmNbottomAttachment, XmATTACH_POSITION );  n++;
  XtSetArg( args[ n ], XmNbottomPosition, 60 );  n++;
  XtSetValues( calSelSw, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNtopAttachment, XmATTACH_POSITION );  n++;
  XtSetArg( args[ n ], XmNtopPosition, 60 );  n++;
  XtSetValues( grpSelLa, args, n );


  /* Parts in the navigate window. */
  xitAttachWidget( navWeekAl,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( navMonthAl,
                   XmATTACH_FORM, NULL, XmATTACH_WIDGET, navWeekAl,
                   XmATTACH_NONE, NULL, XmATTACH_NONE,   NULL );

  /* Parts in the main window. */
  xitAttachWidget( menuBr,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_FORM, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( howToDoLa,
                   XmATTACH_WIDGET, menuBr, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,   XmATTACH_NONE, NULL );
  xitAttachWidget( dateRc,
                   XmATTACH_WIDGET, howToDoLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,      XmATTACH_NONE, NULL );
  xitAttachWidget( zoomFs,
                   XmATTACH_WIDGET, dateRc, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,   XmATTACH_NONE, NULL );
  xitAttachWidget( dayDispSw,
                   XmATTACH_WIDGET, dateRc, XmATTACH_WIDGET, timeDispSw,
                   XmATTACH_WIDGET, planWhatFo,
                   XmATTACH_OPPOSITE_WIDGET, zoomFs );
  xitAttachWidget( timeDispSw,
                   XmATTACH_WIDGET, zoomFs, XmATTACH_FORM,   NULL,
                   XmATTACH_NONE,   NULL,   XmATTACH_WIDGET, navFo );
  xitAttachWidget( planSw,
                   XmATTACH_WIDGET, zoomFs,     XmATTACH_WIDGET, timeDispSw,
                   XmATTACH_WIDGET, planWhatFo, XmATTACH_WIDGET, navFo );
  xitAttachWidget( planWhatFo,
                   XmATTACH_WIDGET, dateRc, XmATTACH_NONE,   NULL,
                   XmATTACH_FORM,   NULL,   XmATTACH_WIDGET, navFo );
  xitAttachWidget( navFo,
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL );
  xitAttachWidget( addCalPb,
                   XmATTACH_NONE, NULL,      XmATTACH_NONE, NULL,
                   XmATTACH_WIDGET, clearPb, XmATTACH_FORM, NULL );
  xitAttachWidget( clearPb,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL );


  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( howToDoLa,  args, n );
  XtSetValues( dateRc,     args, n );
  XtSetValues( planWhatFo, args, n );
  XtSetValues( navFo,      args, n );
  XtSetValues( addCalPb,   args, n );
  XtSetValues( clearPb,    args, n );
  XtSetValues( navMonthAl, args, n );
  XtSetValues( navWeekAl,  args, n );

  n = 0;
  XtSetArg( args[ n ], XmNleftOffset,  10 ); n++;
  XtSetArg( args[ n ], XmNrightOffset, 10 ); n++;
  XtSetValues( navMonthAl, args, n );
  XtSetValues( navWeekAl,  args, n );


  /* Manage all the children. */
  XtManageChildren( menuCasc,   XtNumber( menuCasc ) );
  XtManageChildren( menuFileBu, XtNumber( menuFileBu ) );
  XtManageChildren( menuCtrlBu, XtNumber( menuCtrlBu ) );
  XtManageChildren( menuOptBu,  XtNumber( menuOptBu ) );
  XtManageChildren( menuHelpBu, XtNumber( menuHelpBu ) );

  xitManageChildren( dataLocalW, XtNumber( dataLocalW ) );


  /* Fix the layout. */
  {
    Dimension  sb_height;

    n = 0;
    XtSetArg( args[ n ], XmNhorizontalScrollBar, &tempW ); n++;
    XtGetValues( planSw, args, n );

    n = 0;
    XtSetArg( args[ n ], XmNheight, &sb_height ); n++;
    XtGetValues( tempW, args, n );

    /* Day names window. */
    n = 0;
    XtSetArg( args[ n ], XmNtopOffset, 7 ); n++;
    XtSetArg( args[ n ], XmNbottomOffset, 0 ); n++;
    XtSetArg( args[ n ], XmNleftOffset, 4 ); n++;
    XtSetArg( args[ n ], XmNrightOffset, sb_height + 10 ); n++;
    XtSetValues( dayDispSw, args, n );

    /* Zoom button. */
    n = 0;
    XtSetArg( args[ n ], XmNtopOffset, 5 ); n++;
    XtSetArg( args[ n ], XmNleftOffset, 5 ); n++;
    XtSetValues( zoomFs, args, n );

    /* Time display window. */
    n = 0;
    XtSetArg( args[ n ], XmNtopOffset, 5 ); n++;
    XtSetArg( args[ n ], XmNleftOffset, 0 ); n++;
    XtSetArg( args[ n ], XmNbottomOffset, sb_height + 4 ); n++;
    XtSetValues( timeDispSw, args, n );

    /* Entry display window. */
    n = 0;
    XtSetArg( args[ n ], XmNtopOffset, 5 ); n++;
    XtSetArg( args[ n ], XmNleftOffset, 4 ); n++;
    XtSetArg( args[ n ], XmNrightOffset, 10 ); n++;
    XtSetArg( args[ n ], XmNbottomOffset, 4 ); n++;
    XtSetValues( planSw, args, n );

    /* Buttons on the bottom. */
    n = 0;
    XtSetArg( args[ n ], XmNbottomOffset, 7 ); n++;
    XtSetValues( clearPb,  args, n );
    XtSetValues( addCalPb, args, n );

    /* Clear button. */
    n = 0;
    XtSetArg( args[ n ], XmNrightOffset, 25 ); n++;
    XtSetValues( clearPb, args, n );

  } /* block */


  /* We don't display all scroll bars. */
  n = 0;
  XtSetArg( args[ n ], XmNhorizontalScrollBar, &tempW ); n++;
  XtGetValues( timeDispSw, args, n );
  XtUnmanageChild( tempW );

  n = 0;
  XtSetArg( args[ n ], XmNverticalScrollBar, &tempW ); n++;
  XtGetValues( dayDispSw, args, n );
  XtUnmanageChild( tempW );


  /* Set icon for this window. */
  xtmIcSetSimpleIcon( planTl, workFo, XTM_IC_ICON_PLANNER );


  return( planTl );

} /* createPlanWindow */


/*----------------------------------------------------------------------*/

static FIND_RESULT
  fetchCal( LST_DESC_TYPE  cals_ref,
            char           *search_cal,
            PLAN_CAL_REF   *cal_ref )
{

  /* Variables. */
  LST_STATUS     lst_status;
  PLAN_CAL       cal_rec;
  PLAN_CAL_REF   cal_rec_ref;


  /* Code. */

  /* Search the calendars from these we know. */
  lst_status = LstLinkCurrentFirst( cals_ref );

  while( lst_status == LST_OK ) {

    /* Fetch the record. */
    cal_rec_ref = (PLAN_CAL_REF) LstLinkGetCurrentRef( cals_ref );

    /* Is this what we want? */
    if( strcmp( cal_rec_ref -> name, search_cal ) == 0 ) {
      if( cal_ref != NULL )
        *cal_ref = cal_rec_ref;

      return( EXISTS );
    }

    lst_status = LstLinkCurrentNext( cals_ref );

  } /* while */

  /* Not found, let's make a new one. */
  cal_rec.flags   = 0;
  cal_rec.entries = LstLinkNew( sizeof( PLAN_ENTRY ), NULL );
  strcpy( cal_rec.name, search_cal );

  (void) LstLinkInsertLast(  cals_ref, &cal_rec );
  (void) LstLinkCurrentLast( cals_ref );

  cal_rec_ref = (PLAN_CAL_REF) LstLinkGetCurrentRef( cals_ref );

  if( cal_ref != NULL )
    *cal_ref = cal_rec_ref;


  return( IS_NEW );

} /* fetchCal */


/*----------------------------------------------------------------------*/

static Widget
  fetchCalendarWidget( PLAN_REC_REF  plan_ref,
                       char          *search_cal )
{

  /* Variables. */
  int     index;
  char    buffer[ 50 ];
  char    *cal_name;
  Widget  baseW;
  Widget  mainW;
  Widget  tempW;


  /* Code. */

  mainW = XtNameToWidget( plan_ref -> plannerW, "PlanTlBase.PlanTlFo" );

  baseW = XtNameToWidget( mainW,
            "PlanWhatFo.CalSelSw.ClipWindow.CalSelRc" );

  index = 1;

  do {

    sprintf( buffer, "WhatCal%d", index );

    tempW = XtNameToWidget( baseW, buffer );
    if( tempW == NULL )
      break;

    cal_name = xitStringGetLabel( tempW );
    if( strcmp( cal_name, search_cal ) == 0 ) {
      SysFree( cal_name );

      return( tempW );
    }

    index++;

  } while( True );


  return( NULL );

} /* fetchCalendarWidget */


/*----------------------------------------------------------------------*/

static FIND_RESULT
  fetchDates( PLAN_REC_REF    plan_ref,
              TIM_TIME_REF    start_date,
              TIM_TIME_REF    to_date,
              PLAN_DATES_REF  *plan_dates_ref )
{

  /* Variables. */
  LST_DESC_TYPE   plan_what;
  LST_STATUS      lst_status;
  PLAN_DATES      dates_rec;
  PLAN_DATES_REF  dates_rec_ref;


  /* Code. */

  plan_what = plan_ref -> plan_what_ref;

  /* Search the dates from dates we know. */
  lst_status = LstLinkCurrentFirst( plan_what );

  while( lst_status == LST_OK ) {

    /* Fetch the record. */
    dates_rec_ref = (PLAN_DATES_REF) LstLinkGetCurrentRef( plan_what );

    /* Is this what we want? */
    if( start_date >= dates_rec_ref -> from_date &&
        to_date    <= dates_rec_ref -> to_date ) {
      if( plan_dates_ref != NULL )
        *plan_dates_ref = dates_rec_ref;

      return( EXISTS );
    }

    lst_status = LstLinkCurrentNext( plan_what );

  } /* while */

  /* Not found, let's make a new one. */
  dates_rec.from_date = start_date;
  dates_rec.to_date   = to_date;
  dates_rec.cals      = LstLinkNew( sizeof( PLAN_CAL ), NULL );

  (void) LstLinkInsertLast(  plan_what, &dates_rec );
  (void) LstLinkCurrentLast( plan_what );

  dates_rec_ref = (PLAN_DATES_REF) LstLinkGetCurrentRef( plan_what );

  if( plan_dates_ref != NULL )
    *plan_dates_ref = dates_rec_ref;


  return( IS_NEW );

} /* fetchDates */


/*----------------------------------------------------------------------*/

static void
  loadEntries( PLAN_REC_REF   plan_ref,
               TIM_TIME_REF   start_date,
               TIM_TIME_REF   to_date,
               char           *cal_name,
               LST_DESC_TYPE  entries_ref )
{

  /* Variables. */
  LST_DESC_TYPE         cal_entries;
  LST_STATUS            lst_status;
  PLAN_ENTRY            entry;
  TIM_DELTA_TYPE        delta;
  XTM_DB_ALL_ENTRY_DEF  entry_record;
  XTM_DB_STATUS         db_status;
  XTM_DM_FILTER_REC     filter_rec;


  /* Code. */

  /* Time to fetch the entries.... .*/
  filter_rec.flags            = XTM_DM_FLAG_ALL;
  filter_rec.text_string[ 0 ] = '\0';
  filter_rec.tag_string[ 0 ]  = '\0';
  
  db_status = xtmDmFetchFilteredEntriesRange(
                plan_ref -> appl_data_ref, cal_name,
                plan_ref -> appl_data_ref -> context,
                NULL,
                start_date, to_date, XTM_DB_FETCH_SKIP_INCLUDE,
                &filter_rec, &cal_entries );

  /* Fetch the entries and get rid of the 'scrap'. */
  lst_status = LstLinkCurrentFirst( cal_entries );

  while( lst_status == LST_OK ) {

    (void) LstLinkGetCurrent( cal_entries, &entry_record );

    if( entry_record.entry.entry_type != XTM_DB_DAY_ENTRY ||
        flagIsSet( entry_record.entry.flags, XTM_DB_FLAG_HIDE_IN_DAY_VIEW ) ) {

      lst_status = LstLinkCurrentNext( cal_entries );
      continue;
    }

    entry.duration = (int) entry_record.entry.duration;
    entry.flags    = 0;
    entry.start    = entry_record.entry.time_stamp;

    TimDelta( start_date, entry_record.entry.date_stamp, &delta );
    entry.day_no = delta.days;

    TimDelta( 0, entry_record.entry.time_stamp, &delta );
    entry.min_since_midnight = delta.minutes;

    if( entry.duration < 10 )
      entry.duration = 10;

    if( entry.min_since_midnight + entry.duration >= 24 * 60 )
      entry.duration = 24 * 60 - (entry.min_since_midnight);

    if( entry.duration < 1 )
      entry.duration = 1;

    lst_status = LstLinkInsertLast( entries_ref, &entry );
    lst_status = LstLinkCurrentNext( cal_entries );

  } /* while */

  /* Clear the list with entries. */
  xtmDmDeleteEntriesList( cal_entries );


  return;

} /* loadEntries */


/*----------------------------------------------------------------------*/

static void
  loadPlan( PLAN_REC_REF  plan_ref,
            TIM_TIME_REF  start_date,
            TIM_TIME_REF  to_date )
{

  /* Variables. */
  int             index;
  Widget          baseW;
  Widget          mainW;
  LST_DESC_TYPE   cals_ref;
  LST_STATUS      lst_status;
  PLAN_CAL_REF    cal_rec_ref;
  PLAN_DATES_REF  plan_dates;


  /* Code. */

  mainW = XtNameToWidget( plan_ref -> plannerW, "PlanTlBase.PlanTlFo" );


  /* We should have these dates. */
  (void) fetchDates( plan_ref, start_date, to_date, &plan_dates );

  /* Mark all calendars as 'not displayed'. */
  cals_ref = plan_dates -> cals;

  lst_status = LstLinkCurrentFirst( cals_ref );

  while( lst_status == LST_OK ) {

    /* Mark the calendar. */
    cal_rec_ref = (PLAN_CAL_REF) LstLinkGetCurrentRef( cals_ref );

    flagClear( cal_rec_ref -> flags, FLAG_CAL_DISPLAYED );

    lst_status = LstLinkCurrentNext( cals_ref );

  } /* while */


  /* For each calendar we have selected, make sure we have the entries. */
  baseW = XtNameToWidget( mainW,
            "PlanWhatFo.CalSelSw.ClipWindow.CalSelRc" );
  index = 1;

  do {
    char         buffer[ 50 ];
    char         *cal_name;
    Widget       tempW;
    FIND_RESULT  result;

    sprintf( buffer, "WhatCal%d", index );

    tempW = XtNameToWidget( baseW, buffer );
    if( tempW == NULL )
      break;

    if( XmToggleButtonGetState( tempW ) ) {
      cal_name = xitStringGetLabel( tempW );

      result = fetchCal( plan_dates -> cals, cal_name, &cal_rec_ref );
      if( result == IS_NEW )
        loadEntries( plan_ref, start_date, to_date,
                     cal_name, cal_rec_ref -> entries );

      flagSet( cal_rec_ref -> flags, FLAG_CAL_DISPLAYED );

      SysFree( cal_name );
    }

    index++;
  } while( True );


  return;

} /* loadPlan */


/*----------------------------------------------------------------------*/

static void 
  presentBusyMarker( PLAN_REC_REF  plan_ref,
                     Widget        parentW,
                     int           day_no,
                     int           start_min,
                     int           stop_min,
                     int           minute_offset,
                     int           *marker_index )
{

  /* Variables. */
  char       buffer[ 50 ];
  Dimension  height;
  Dimension  width;
  Position   x;
  Position   y;
  Arg        args[ 10 ];
  Cardinal   n;
  Widget     buttonW;

  static XIT_PUSH_STRUCT button_def[] = {
    { "", "  ", "", False, NULL }
  };


  /* Code. */

  /* Any old marker? */
  sprintf( buffer, "BusyMarker%d", *marker_index );

  buttonW = XtNameToWidget( parentW, buffer );
  if( buttonW == NULL ) {

    button_def[ 0 ].name = buffer;
    buttonW = xitCreatePushButton( parentW, &button_def[ 0 ] );

    width = (Dimension) plan_ref -> day_width - 20;

    n = 0;
    XtSetArg( args[ n ], XmNrecomputeSize, False ); n++;
    XtSetArg( args[ n ], XmNwidth, width ); n++;
    XtSetValues( buttonW, args, n );

  }

  /* Geometry for the marker. */
  x = (Position) day_no * plan_ref -> day_width + 10;
  y = (Position) ((start_min - minute_offset) * plan_ref -> pixel_per_min +
                  plan_ref -> time_line_height / 2) + 4;

  height = (Dimension) (stop_min - start_min + 1) * plan_ref -> pixel_per_min;
  if( height < 10 )
    height = 10;

  n = 0;
  XtSetArg( args[ n ], XmNx, x ); n++;
  XtSetArg( args[ n ], XmNy, y ); n++;
  XtSetArg( args[ n ], XmNheight, height ); n++;
  XtSetValues( buttonW, args, n );

  if( ! XtIsManaged( buttonW ) )
    XtManageChild( buttonW );
  else
    XtMapWidget( buttonW );


  *marker_index = *marker_index + 1;


  return;

} /* presentBusyMarker */


/*----------------------------------------------------------------------*/

static void 
  removeCached( PLAN_REC_REF  plan_ref )
{

  /* Variables. */
  LST_STATUS  lst_status;
  PLAN_CAL    cal_record;
  PLAN_DATES  date_record;


  /* Code. */

  /* Step through each date and calendar and release data. */
  lst_status = LstLinkCurrentFirst( plan_ref -> plan_what_ref );

  while( lst_status == LST_OK ) {

    LstLinkGetCurrent( plan_ref -> plan_what_ref, &date_record );

    /* Step through each calendar. */
    lst_status = LstLinkCurrentFirst( date_record.cals );

    while( lst_status == LST_OK ) {
      LstLinkGetCurrent( date_record.cals, &cal_record );

      LstLinkClear( cal_record.entries );

      lst_status = LstLinkCurrentNext( date_record.cals );
    } /* while */

    LstLinkClear( date_record.cals );

    lst_status = LstLinkCurrentNext( plan_ref -> plan_what_ref );

  } /* while */

  LstLinkClear( plan_ref -> plan_what_ref );


  return;

} /* removeCached */


/*----------------------------------------------------------------------*/

static void 
  scrollWindow( PLAN_REC_REF  plan_ref,
                Widget        scrollBar,
                int           value )
{

  /* Variables. */
  int       sb_increment;
  int       sb_maximum;
  int       sb_minimum;
  int       sb_page_increment;
  int       sb_slider_size;
  int       sb_value;
  Arg       args[ 10 ];
  Cardinal  n;


  /* Code. */

  /* Information about scroll bar. */
  n = 0;
  XtSetArg( args[ n ], XmNminimum, &sb_minimum ); n++;
  XtSetArg( args[ n ], XmNmaximum, &sb_maximum ); n++;
  XtSetArg( args[ n ], XmNsliderSize, &sb_slider_size ); n++;
  XtSetArg( args[ n ], XmNincrement, &sb_increment ); n++;
  XtSetArg( args[ n ], XmNpageIncrement, &sb_page_increment ); n++;
  XtSetArg( args[ n ], XmNvalue, &sb_value ); n++;
  XtGetValues( scrollBar, args, n );

  if( value < sb_minimum )
    value = sb_minimum;
  if( value > (sb_maximum - sb_slider_size) )
    value = (sb_maximum - sb_slider_size);

  /* Set position for scroll bar. */
  XmScrollBarSetValues( scrollBar, value, sb_slider_size,
                        sb_increment, sb_page_increment, True );

  /* The scrolled window does not react on valueChanged callback. */
  {
    XmScrollBarCallbackStruct call_data;

    /* Create the call data, I hope it is OK to skip event and pixel. */
    call_data.value  = value;
    call_data.reason = XmCR_DRAG;

    /* The drag callback makes the scrolled window to react. */
    XtCallCallbacks( scrollBar, XmNdragCallback, (XtPointer) &call_data );

  } /* block */

  
  return;

} /* scrollWindow */


/*----------------------------------------------------------------------*/

static void 
  sendMessage( PLAN_REC_REF  plan_ref,
               TIM_TIME_REF  to_date )
{

  /* Variables. */
  Boolean                 found;
  Boolean                 ok;
  int                     index;
  int                     index1;
  int                     grp_members_index;
  int                     msg_to_index;
  char                    buffer[ 50 ];
  char                    *char_ref;
  char                    *msg_to[ 500 ];
  char                    *grp_members[ 500 ];
  Widget                  baseW;
  Widget                  mainW;
  Widget                  tempW;
  XTM_GD_ENTRY            group_info;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  custom_data_ref = plan_ref -> appl_data_ref -> custom_data;

  mainW = XtNameToWidget( plan_ref -> plannerW, "PlanTlBase.PlanTlFo" );


  msg_to_index      = 0;
  grp_members_index = 0;

  /* Fetch all selected groups. */
  baseW = XtNameToWidget( mainW,
            "PlanWhatFo.GrpSelSw.ClipWindow.GrpSelRc" );

  index = 1;
  do {
    if( msg_to_index >= 500 )
      break;

    sprintf( buffer, "WhatGroup%d", index );

    tempW = XtNameToWidget( baseW, buffer );
    if( tempW == NULL )
      break;

    if( ! XmToggleButtonGetState( tempW ) ) {
      index++;
      continue;
    }

    char_ref = xitStringGetLabel( tempW );

    /* Group information. */
    ok = xtmGdFetchNamedGroup( custom_data_ref -> group_db_handle,
                               char_ref, &group_info );

    /* Save the information. */
    if( ok ) {
      grp_members[ grp_members_index ] = group_info.members;
      grp_members_index++;
    }

    msg_to[ msg_to_index ] = char_ref;
    msg_to_index++;

    index++;
  } while( True );


  /* Fetch all selected calendars. */
  baseW = XtNameToWidget( mainW,
            "PlanWhatFo.CalSelSw.ClipWindow.CalSelRc" );

  index = 1;
  do {
    if( msg_to_index >= 500 )
      break;

    sprintf( buffer, "WhatCal%d", index );

    tempW = XtNameToWidget( baseW, buffer );
    if( tempW == NULL )
      break;

    if( ! XmToggleButtonGetState( tempW ) ) {
      index++;
      continue;
    }

    char_ref = xitStringGetLabel( tempW );

    /* Is the calendar already present in one of the groups? */
    sprintf( buffer, "%s ", char_ref );
    found = False;

    for( index1 = 0; index1 < grp_members_index; index1++ ) {
      if( strstr( grp_members[ index1 ], buffer ) != NULL ) {
        found = True;
        break;
      }
    }

    /* Save the information. */
    if( ! found ) {
      msg_to[ msg_to_index ] = char_ref;
      msg_to_index++;
    }

    index++;
  } while( True );


  /* Message window. */
  if( plan_ref -> msg_handle == NULL ) {
    plan_ref -> msg_handle = xtmMsInitialize( plan_ref -> appl_data_ref,
                                              plan_ref -> plannerW );
    if( plan_ref -> msg_handle == NULL )
      return;
  }

  xtmMsSendMessage(   plan_ref -> msg_handle, "", 0 );
  xtmMsSetRecipients( plan_ref -> msg_handle, msg_to, msg_to_index );


  /* Free allocated memory. */
  for( index = 0; index < grp_members_index; index++ )
    SysFree( grp_members[ index ] );

  for( index = 0; index < msg_to_index; index++ )
    SysFree( msg_to[ index ] );


  /* Set specific date? */
  xtmMsSetToDate( plan_ref -> msg_handle, to_date );


  return;

} /* sendMessage */


/*----------------------------------------------------------------------*/

static void
  setAdditionalCals( PLAN_REC_REF  plan_ref )
{

  /* Variables. */
  char                    buffer[ 100 ];
  char                    *char_ref;
  char                    *db_names;
  Arg                     args[ 10 ];
  Cardinal                n;
  Widget                  mainW;
  Widget                  baseW;
  Widget                  buttonW;
  XmString                xstr;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  custom_data_ref = plan_ref -> appl_data_ref -> custom_data;

  mainW = XtNameToWidget( plan_ref -> plannerW, "PlanTlBase.PlanTlFo" );


  /* Set the calendars. */
  (void) xtmCdFetchDbNames( custom_data_ref -> cal_db_handle, &db_names );
  char_ref = db_names;

  baseW = XtNameToWidget( mainW,
            "PlanWhatFo.CalSelSw.ClipWindow.CalSelRc" );

  do {

    int              char_read;
    char             db_name[ XTM_GL_MAX_CAL_NAME + 1 ];
    XTM_CD_CAL_INFO  db_info;

    while( isspace( *char_ref ) )
      char_ref++;

    if( *char_ref == '\0' )
      break;

    char_read = strlen( char_ref );
    sscanf( char_ref, "%s%n", db_name, &char_read );
    char_ref = char_ref + char_read;


    /* Fetch information about the database. */
    (void) xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle, db_name,
                              &db_info, NULL );

    if( flagIsClear( db_info.operations, XTM_DB_FLAG_MODE_READ ) )
      continue;

    /* Do we have this? */
    if( fetchCalendarWidget( plan_ref, db_info.short_name ) != NULL )
      continue;

    /* Add the calendar to the list. */
    xstr = XmStringCreateLtoR( db_info.short_name, CS );

    plan_ref -> no_of_cals++;
    sprintf( buffer, "WhatCal%d", plan_ref -> no_of_cals );

    n = 0;
    XtSetArg( args[ n ], XmNlabelString, xstr ); n++;
    XtSetArg( args[ n ], XmNalignment, XmALIGNMENT_CENTER ); n++;
    XtSetArg( args[ n ], XmNset, False ); n++;
    XtSetArg( args[ n ], XmNuserData, (XtPointer) 0 ); n++;
    XtSetArg( args[ n ], XmNmarginHeight, 0 ); n++;
    buttonW = XmCreateToggleButton( baseW, buffer, args, n );

    XtAddCallback( buttonW, XmNvalueChangedCallback, 
                   (XtCallbackProc) calendarSelectedCB, (XtPointer) plan_ref );

    XtManageChild( buttonW );
    XmStringFree( xstr );

  } while( True );


  return;

} /* setAdditionalCals */


/*----------------------------------------------------------------------*/

static void
  setCalsAndGroups( PLAN_REC_REF  plan_ref )
{

  /* Variables. */
  int                     index;
  int                     item_count;
  char                    buffer[ 100 ];
  char                    *char_ref;
  char                    *db_names;
  Arg                     args[ 10 ];
  Cardinal                n;
  Widget                  mainW;
  Widget                  baseW;
  Widget                  buttonW;
  XmString                items[ 500 ];
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  custom_data_ref = plan_ref -> appl_data_ref -> custom_data;

  mainW = XtNameToWidget( plan_ref -> plannerW, "PlanTlBase.PlanTlFo" );


  /* Set the calendars. */
  (void) xtmCdFetchDbNames( custom_data_ref -> cal_db_handle, &db_names );
  char_ref   = db_names;
  index      = 1;
  item_count = 0;

  baseW = XtNameToWidget( mainW,
            "PlanWhatFo.CalSelSw.ClipWindow.CalSelRc" );

  do {

    int              char_read;
    char             db_name[ XTM_GL_MAX_CAL_NAME + 1 ];
    XTM_CD_CAL_INFO  db_info;

    while( isspace( *char_ref ) )
      char_ref++;

    if( *char_ref == '\0' )
      break;

    char_read = strlen( char_ref );
    sscanf( char_ref, "%s%n", db_name, &char_read );
    char_ref = char_ref + char_read;


    /* Fetch information about the database. */
    (void) xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle, db_name,
                              &db_info, NULL );

    if( flagIsClear( db_info.operations, XTM_DB_FLAG_MODE_READ ) )
      continue;

    items[ item_count ] = XmStringCreateLtoR( db_info.short_name, CS );

    item_count++;
    if( item_count >= 500 )
      break;

  } while( True );


  /* Sort the list. */
  xitSortStringList( items, item_count );


  /* Each calendar is a toggle button. */
  for( index = 0; index < item_count; index++ ) {

    sprintf( buffer, "WhatCal%d", index + 1 );

    n = 0;
    XtSetArg( args[ n ], XmNlabelString, items[ index ] ); n++;
    XtSetArg( args[ n ], XmNalignment, XmALIGNMENT_CENTER ); n++;
    XtSetArg( args[ n ], XmNset, False ); n++;
    XtSetArg( args[ n ], XmNuserData, (XtPointer) 0 ); n++;
    XtSetArg( args[ n ], XmNmarginHeight, 0 ); n++;
    buttonW = XmCreateToggleButton( baseW, buffer, args, n );


    XtAddCallback( buttonW, XmNvalueChangedCallback, 
                   (XtCallbackProc) calendarSelectedCB, (XtPointer) plan_ref );

    XtManageChild( buttonW );

  } /* loop */

  for( index = 0; index < item_count; index++ )
    XmStringFree( items[ index ] );

  SysFree( db_names );

  /* Keep track of the number of calendars. */
  plan_ref -> no_of_cals = item_count;


  /* Set the groups. */
  index      = 1;
  item_count = 0;

  baseW = XtNameToWidget( mainW,
            "PlanWhatFo.GrpSelSw.ClipWindow.GrpSelRc" );

  {
    int   char_read;
    char  group_name[ XTM_GD_MAX_GROUP_NAME + 1 ];
    char  *group_name_ref;
    char  *group_names;

    (void) xtmGdFetchGroupNames( custom_data_ref -> group_db_handle,
                                 &group_names );

    group_name_ref = group_names;

    do {

      while( isspace( *group_name_ref ) )
        group_name_ref++;

      if( *group_name_ref == '\0' )
        break;

      char_read = strlen( group_name_ref );
      sscanf( group_name_ref, "%s%n", group_name, &char_read );
      group_name_ref = group_name_ref + char_read;

      items[ item_count ] = XmStringCreateLtoR( group_name, CS );

      item_count++;
      if( item_count >= 500 )
        break;

    } while( True );


    /* Sort the list. */
    xitSortStringList( items, item_count );


    /* Each group is a toggle button. */
    for( index = 0; index < item_count; index++ ) {

      sprintf( buffer, "WhatGroup%d", index + 1 );

      n = 0;
      XtSetArg( args[ n ], XmNlabelString, items[ index ] ); n++;
      XtSetArg( args[ n ], XmNalignment, XmALIGNMENT_CENTER ); n++;
      XtSetArg( args[ n ], XmNset, False ); n++;
      XtSetArg( args[ n ], XmNuserData, (XtPointer) 0 ); n++;
      XtSetArg( args[ n ], XmNmarginHeight, 0 ); n++;
      buttonW = XmCreateToggleButton( baseW, buffer, args, n );


      XtAddCallback( buttonW, XmNvalueChangedCallback, 
                     (XtCallbackProc) groupSelectedCB, (XtPointer) plan_ref );

      XtManageChild( buttonW );

    } /* loop */

    for( index = 0; index < item_count; index++ )
      XmStringFree( items[ index ] );

    SysFree( group_names );

  } /* block */


  return;

} /* setCalsAndGroups */


/*----------------------------------------------------------------------*/

static void
  startOnDay( PLAN_REC_REF  plan_ref,
              TIM_TIME_REF  *start_time )
{

  /* Variables. */
  int  start_day;


  /* Code. */

  /* Start the day view on a specific day? */
  start_day = plan_ref -> appl_data_ref -> custom_data -> start_day_view_on;
  if( start_day == 0 )
    start_day = 1;

  while( True ) {
    if( TimIndexOfDayInIsoWeek( *start_time ) == start_day )
      break;

    TimAddDays( start_time, -1 );
  }


  return;

} /* startOnDay */


/*----------------------------------------------------------------------*/

static void
  updateDayNames( PLAN_REC_REF  plan_ref,
                  TIM_TIME_REF  start_date,
                  TIM_TIME_REF  stop_date )
{

  /* Variables. */
  Boolean                 change_labels = True;
  int                     day_index;
  char                    buffer[ 50 ];
  Arg                     args[ 5 ];
  Cardinal                n;
  Dimension               width;
  Pixel                   def_bg_color;
  Pixel                   def_fg_color;
  Position                day_pos;
  Widget                  buttonW;
  Widget                  dayDispBbW;
  Widget                  mainW;
  TIM_TIME_REF            curr_date;
  TIM_TIME_REF            now;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;

  static XIT_PUSH_STRUCT button_data[] = {
    { "", " ", "", True, NULL }
  };


  /* Code. */

  custom_data_ref = plan_ref -> appl_data_ref -> custom_data;

  mainW = XtNameToWidget( plan_ref -> plannerW, "PlanTlBase.PlanTlFo" );

  /* Widgets we will need. */
  dayDispBbW = XtNameToWidget( 
                 mainW, 
                 "DayDispSw.ClipWindow.DayDispBb" );

  now = TimLocalTime( TimMakeTimeNow() );


  /* Changes in the dates? */
  if( start_date == plan_ref -> old_planner_start &&
      stop_date  == plan_ref -> old_planner_stop )
    change_labels = False;

  plan_ref -> old_planner_start = start_date;
  plan_ref -> old_planner_stop  = stop_date;


  /* Set the correct size of the bulletin board. */
  width  = (Dimension) plan_ref -> show_no_days * plan_ref -> day_width;

  n = 0;
  XtSetArg( args[ n ], XmNwidth, width ); n++;
  XtSetValues( dayDispBbW, args, n );


  /* Loop through all days. */
  n = 0;
  XtSetArg( args[ n ], XmNforeground, &def_fg_color ); n++;
  XtSetArg( args[ n ], XmNbackground, &def_bg_color ); n++;
  XtGetValues( dayDispBbW, args, n );

  curr_date = start_date;
  day_index = 1;
  day_pos   = 0;

  while( curr_date <= stop_date ) {

    int    day_in_week;
    Pixel  bg_color;
    Pixel  fg_color;

    /* Does the day button exists? */
    sprintf( buffer, "DayName%dPb", day_index );

    buttonW = XtNameToWidget( dayDispBbW, buffer );

    if( buttonW == NULL ) {

      button_data[ 0 ].name  = buffer;
      button_data[ 0 ].title = " ";

      buttonW = xitCreatePushButton( dayDispBbW, &button_data[ 0 ] );

      XtAddCallback( buttonW, XmNactivateCallback, 
                     (XtCallbackProc) doMsgDateCB, (XtPointer) plan_ref );

      n = 0;
      XtSetArg( args[ n ], XmNalignment, XmALIGNMENT_BEGINNING ); n++;
      XtSetArg( args[ n ], XmNy, 0 ); n++;
      XtSetArg( args[ n ], XmNshadowThickness, 0 ); n++;
      XtSetArg( args[ n ], XmNtopShadowColor, def_bg_color ); n++;
      XtSetArg( args[ n ], XmNbottomShadowColor, def_bg_color ); n++;
      XtSetValues( buttonW, args, n );

      change_labels = True;

    } /* if */

    /* Name of the day. */
    if( change_labels ) {
      TimFormatStrTime( curr_date, "%a", buffer,  sizeof( buffer ) );
      xitStringSetLabel( buttonW, buffer );
    }

    /* Color mark the day? */
    bg_color = def_bg_color;
    fg_color = def_fg_color;

    day_in_week = TimIndexOfDayInIsoWeek( curr_date );

    if( day_in_week == 1 )
      fg_color = custom_data_ref -> monday_fg;
    else if( day_in_week == 2 )
      fg_color = custom_data_ref -> tuesday_fg;
    else if( day_in_week == 3 )
      fg_color = custom_data_ref -> wednesday_fg;
    else if( day_in_week == 4 )
      fg_color = custom_data_ref -> thursday_fg;
    else if( day_in_week == 5 )
      fg_color = custom_data_ref -> friday_fg;
    else if( day_in_week == 6 )
      fg_color = custom_data_ref -> saturday_fg;
    else if( day_in_week == 7 )
      fg_color = custom_data_ref -> sunday_fg;

    if( xtmHoIsHoliday( curr_date ) )
      fg_color = custom_data_ref -> holiday_fg;

    if( TimIsSameDate( now, curr_date ) == TIM_YES )
      bg_color = custom_data_ref -> today_bg;

    n = 0;
    XtSetArg( args[ n ], XmNx, day_pos ); n++;
    XtSetArg( args[ n ], XmNwidth, plan_ref -> day_width ); n++;
    XtSetArg( args[ n ], XmNforeground, fg_color ); n++;
    XtSetArg( args[ n ], XmNbackground, bg_color ); n++;
    XtSetArg( args[ n ], XmNuserData, (XtPointer) curr_date ); n++;
    XtSetValues( buttonW, args, n );

    /* Make sure it is displayed. */
    if( ! XtIsManaged( buttonW ) )
      XtManageChild( buttonW );
    else
      XtMapWidget( buttonW );


    /* Next day. */
    day_pos = day_pos + (Position) plan_ref -> day_width;

    day_index++;    
    TimAddDays( &curr_date, 1 );

  } /* while */

  /* Unmap days not used. */
  do {
    sprintf( buffer, "DayName%dPb", day_index );

    buttonW = XtNameToWidget( dayDispBbW, buffer );
    if( buttonW == NULL )
      break;

    XtUnmapWidget( buttonW );

    day_index++;
  } while( True );


  return;

} /* updateDayNames */


/*----------------------------------------------------------------------*/

static void
  updateEntries( PLAN_REC_REF  plan_ref,
                 TIM_TIME_REF  start_date,
                 TIM_TIME_REF  stop_date )
{

  /* Variables. */
  Boolean         is_busy;
  int             day;
  int             marker_index;
  int             minute = 0;
  int             minute_offset;
  int             start_busy_min = 0;
  int             stop_busy_min = 0;
  Widget          mainW;
  Widget          planBbW;
  LST_STATUS      lst_status;
  PLAN_CAL        cal_record;
  PLAN_DATES_REF  plan_dates_ref;
  PLAN_ENTRY      entry;
  TIM_DELTA_TYPE  delta;


  /* Code. */

  mainW = XtNameToWidget( plan_ref -> plannerW, "PlanTlBase.PlanTlFo" );

  planBbW = XtNameToWidget( mainW, 
                            "PlanSw.ClipWindow.PlanBb" );


  /* Noting is busy. */
  for( day = 0; day < plan_ref -> show_no_days; day++ ) {
    for( minute = 0; minute < 24 * 60; minute++ ) {
      *((plan_ref -> busy_days[ day ]) + minute) = False;
    }
  }


  /* Load the user entries. */
  loadPlan( plan_ref, start_date, stop_date );

  /* Grab the current dates. */
  fetchDates( plan_ref, start_date, stop_date, &plan_dates_ref );


  /* Find what is really busy. */
  lst_status = LstLinkCurrentFirst( plan_dates_ref -> cals );

  while( lst_status == LST_OK ) {

    /* Look at the calendar. */
    lst_status = LstLinkGetCurrent( plan_dates_ref -> cals, &cal_record );

    if( flagIsClear( cal_record.flags, FLAG_CAL_DISPLAYED ) ) {
      lst_status = LstLinkCurrentNext( plan_dates_ref -> cals );
      continue;
    }

    /* Look at the entries. */
    lst_status = LstLinkCurrentFirst( cal_record.entries );

    while( lst_status == LST_OK ) {

      /* Look at the entry. */
      lst_status = LstLinkGetCurrent( cal_record.entries, &entry );

      day = entry.day_no;

      for( minute = 0; minute < entry.duration &&
           entry.min_since_midnight + minute < 24 * 60; minute++ ) {
        *((plan_ref -> busy_days[ day ]) +
           entry.min_since_midnight + minute) = True;
      }

      /* Next entry. */
      lst_status = LstLinkCurrentNext( cal_record.entries );

    } /* while */

    /* Next calendar. */
    lst_status = LstLinkCurrentNext( plan_dates_ref -> cals );

  } /* while */


  /* Update the time scale. */
  updateTimeScale( plan_ref, plan_ref -> default_entry_start,
                   plan_ref -> default_entry_stop );
  
  /* Update the planner background. */
  updateEntriesBg( plan_ref, plan_ref -> planner_start, 
                   plan_ref -> planner_stop );


  /* Where do the time scale start? */
  TimDelta( 0, plan_ref -> entry_start, &delta );
  minute_offset = delta.minutes;

  /* When are we busy? */
  marker_index = 1;
  is_busy      = False;

  for( day = 0; day < plan_ref -> show_no_days; day++ ) {
    for( minute = 0; minute < 24 * 60; minute++ ) {

      /* Are we busy? */
      if( *((plan_ref -> busy_days[ day ]) + minute) ) {
        if( ! is_busy ) {
          start_busy_min = minute;
          stop_busy_min  = minute;
          is_busy = True;
        }

      /* but were we busy? */
      } else {
        if( is_busy ) {
          stop_busy_min = minute;
          is_busy       = False;

          presentBusyMarker( plan_ref, planBbW,
                             day, start_busy_min, stop_busy_min,
                             minute_offset,
                             &marker_index );
        }

      } /* if */

    } /* loop */

    /* Any 'left overs'? */
    if( is_busy ) {
      stop_busy_min = minute;
      is_busy       = False;

      presentBusyMarker( plan_ref, planBbW,
                         day, start_busy_min, stop_busy_min,
                         minute_offset,
                         &marker_index );
    }

  } /* loop */


  /* Hide any markers left over. */
  do {

    char    buffer[ 50 ];
    Widget  tempW;

    sprintf( buffer, "BusyMarker%d", marker_index );

    tempW = XtNameToWidget( planBbW, buffer );
    if( tempW == NULL )
      break;

    XtUnmapWidget( tempW );

    marker_index++;

  } while( True );


  return;

} /* updateEntries */


/*----------------------------------------------------------------------*/

static void
  updateEntriesBg( PLAN_REC_REF  plan_ref,
                   TIM_TIME_REF  start_date,
                   TIM_TIME_REF  stop_date )
{

  /* Variables. */
  int                     index;
  char                    buffer[ 50 ];
  Arg                     args[ 5 ];
  Cardinal                n;
  Dimension               height;
  Dimension               width;
  Display                 *display = XtDisplay( plan_ref -> plannerW );
  Position                day_pos;
  Position                pos;
  Position                stop_pos;
  Widget                  sepW;
  Widget                  planBbW;
  Widget                  mainW;
  TIM_TIME_REF            curr_date;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  custom_data_ref = plan_ref -> appl_data_ref -> custom_data;

  mainW = XtNameToWidget( plan_ref -> plannerW, "PlanTlBase.PlanTlFo" );

  /* Widgets we will need. */
  planBbW = XtNameToWidget( mainW, 
                            "PlanSw.ClipWindow.PlanBb" );

  /* Set the correct size of the bulletin board. */
  width  = (Dimension) plan_ref -> show_no_days * plan_ref -> day_width + 2;
  height = (Dimension) plan_ref -> time_label_height;

  n = 0;
  XtSetArg( args[ n ], XmNwidth, width + 10 ); n++;
  XtSetArg( args[ n ], XmNheight, height ); n++;
  XtSetValues( planBbW, args, n );


  /* Make the day separators. */
  curr_date = start_date;
  index     = 1;
  day_pos   = (Position) plan_ref -> day_width;

  while( curr_date <= stop_date ) {

    /* Does the separator exists? */
    sprintf( buffer, "DaySep%dSp", index );

    sepW = XtNameToWidget( planBbW, buffer );

    if( sepW == NULL ) {

      n = 0;
      XtSetArg( args[ n ], XmNy, 0 ); n++;
      XtSetArg( args[ n ], XmNorientation, XmVERTICAL );  n++;
      sepW = XmCreateSeparator( planBbW, buffer, args, n );

    } /* if */

    height = (Dimension) plan_ref -> time_label_height;

    n = 0;
    XtSetArg( args[ n ], XmNx, day_pos ); n++;
    XtSetArg( args[ n ], XmNheight, height ); n++;
    XtSetValues( sepW, args, n );

    /* Make sure it is displayed. */
    if( ! XtIsManaged( sepW ) )
      XtManageChild( sepW );
    else
    XtMapWidget( sepW );


    /* Next day. */
    day_pos = day_pos + (Position) plan_ref -> day_width;

    index++;    
    TimAddDays( &curr_date, 1 );

  } /* while */

  /* Unmap separators not used. */
  do {
    sprintf( buffer, "DaySep%dSp", index );

    sepW = XtNameToWidget( planBbW, buffer );
    if( sepW == NULL )
      break;

    XtUnmapWidget( sepW );

    index++;
  } while( True );


  /* Make the grid lines. */
  index    = 1;
  pos      = (Position) (plan_ref -> time_line_height / 2) + 2;
  stop_pos = (Position) plan_ref -> time_label_height;

  while( pos < stop_pos ) {

    /* Does the separator exists? */
    sprintf( buffer, "Grid%dSp", index );

    sepW = XtNameToWidget( planBbW, buffer );

    if( sepW == NULL ) {

      n = 0;
      XtSetArg( args[ n ], XmNx, 0 ); n++;
      XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL );  n++;
      XtSetArg( args[ n ], XmNseparatorType, XmSINGLE_DASHED_LINE );  n++;
      sepW = XmCreateSeparator( planBbW, buffer, args, n );

    } /* if */

    width = (Dimension) plan_ref -> show_no_days * plan_ref -> day_width + 2;

    n = 0;
    XtSetArg( args[ n ], XmNy, pos ); n++;
    XtSetArg( args[ n ], XmNwidth, width ); n++;
    XtSetValues( sepW, args, n );

    /* Make sure it is displayed. */
    if( ! XtIsManaged( sepW ) )
      XtManageChild( sepW );
    else
    XtMapWidget( sepW );

    XLowerWindow( display, XtWindow( sepW ) );

    /* Next grid. */
    pos = pos + (Position) plan_ref -> time_line_height * 2;

    index++;    
  } /* while */

  /* Unmap grid not used. */
  do {
    sprintf( buffer, "Grid%dSp", index );

    sepW = XtNameToWidget( planBbW, buffer );
    if( sepW == NULL )
      break;

    XtUnmapWidget( sepW );

    index++;
  } while( True );


  return;

} /* updateEntriesBg */


/*----------------------------------------------------------------------*/

static void
  updatePlanner( PLAN_REC_REF  plan_ref )
{

  /* Variables. */
  Cursor   wait_cursor;
  Display  *display = XtDisplay( plan_ref -> plannerW );
  Window   window   = XtWindow(  plan_ref -> plannerW );


  /* Code. */

  /* This might take time. */
  wait_cursor = XCreateFontCursor( display, XC_watch );
  XDefineCursor( display, window, wait_cursor );

  XFlush( display );


  /* What dates do we display? */
  updatePlannerDates( plan_ref, True );

  /* Update the day names. */
  updateDayNames( plan_ref,
                  plan_ref -> planner_start, plan_ref -> planner_stop );

  /* Update the planner with user entries. */
  updateEntries( plan_ref,
                 plan_ref -> planner_start, plan_ref -> planner_stop );

  /* Normal cursor again. */
  XUndefineCursor( display, window );


  return;

} /* updatePlanner */


/*----------------------------------------------------------------------*/

static void
  updatePlannerDates( PLAN_REC_REF  plan_ref,
                      Boolean       fetch )
{

  /* Variables. */
  Boolean                 ok;
  char                    buffer[ 50 ];
  Widget                  mainW;
  Widget                  tempW;
  TIM_TIME_REF            start_date;
  TIM_TIME_REF            stop_date;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  custom_data_ref = plan_ref -> appl_data_ref -> custom_data;

  mainW = XtNameToWidget( plan_ref -> plannerW, "PlanTlBase.PlanTlFo" );


  /* Fetch and check start date. */
  if( fetch ) {
    tempW = XtNameToWidget( mainW, "DateRc.DateTb" );

    ok = xtmFoFetchDate( mainW, tempW, XTM_FO_START_DATE, False, &start_date );
    if( ! ok )
      return;

    plan_ref -> planner_start = start_date;

    startOnDay( plan_ref, &plan_ref -> planner_start );

  } else {
    startOnDay( plan_ref, &plan_ref -> planner_start );

    tempW = XtNameToWidget( mainW, "DateRc.DateTb" );
    XmUbTimeBoxSetStartDate( tempW, plan_ref -> planner_start );

  } /* if */


  /* Some more information. */
  {
    int       start_week;
    int       stop_week;
    char      info_buffer[ 100 ];

    info_buffer[ 0 ] = '\0';
    strcat( info_buffer, " " );

    /* Stop date. */
    stop_date = plan_ref -> planner_start;
    TimAddDays( &stop_date, plan_ref -> show_no_days - 1 );

    plan_ref -> planner_stop = stop_date;

    xtmFoFormatDate( stop_date, buffer, sizeof( buffer ) );

    strcat( info_buffer, buffer );
    strcat( info_buffer, "  " );
    strcat( info_buffer, msgGetText( MXDI_WEEK_LABEL ) );

    start_week = TimIndexOfWeek( plan_ref -> planner_start );
    stop_week  = TimIndexOfWeek( plan_ref -> planner_stop );

    sprintf( buffer, " %d", start_week );
    strcat( info_buffer, buffer );

    if( start_week != stop_week ) {
      sprintf( buffer, "-%d", stop_week );
      strcat( info_buffer, buffer );
    }

    tempW = XtNameToWidget( mainW, "DateRc.DateInfoLa" );

    xitStringSetLabel( tempW, info_buffer );
  }


  return;

} /* updatePlannerDates */


/*----------------------------------------------------------------------*/

static void
  updateTimeScale( PLAN_REC_REF  plan_ref,
                   TIM_TIME_REF  start_time,
                   TIM_TIME_REF  stop_time )
{

  /* Variables. */
  Boolean               do_update = False;
  Boolean               first_time;
  int                   label_lines;
  char                  buffer[ 50 ];
  char                  time_buffer[ 5000 ];
  Arg                   args[ 5 ];
  Cardinal              n;
  Widget                mainW;
  Widget                timeLaW;
  TIM_TIME_REF          current_time;
  XTM_GL_BASE_DATA_REF  appl_data_ref;


  /* Code. */

  appl_data_ref = plan_ref -> appl_data_ref;

  mainW = XtNameToWidget( plan_ref -> plannerW, "PlanTlBase.PlanTlFo" );


  /* Get the widgets we need. */
  timeLaW = XtNameToWidget( mainW,
              "TimeDispSw.ClipWindow.TimeDispBb.TimeDispLa" );


  /* Find the 'true' start and 'stop' times. */
  {
    Boolean       is_busy = False;
    int           day;
    float         first_busy_min = 24 * 60;
    float         last_busy_min  = 0;
    int           minute;
    TIM_TIME_REF  first_time;
    TIM_TIME_REF  last_time;

    for( day = 0; day < plan_ref -> show_no_days; day++ ) {
      for( minute = 0; minute < 24 * 60; minute++ ) {

        /* Are we (or were we) busy? */
        if( *((plan_ref -> busy_days[ day ]) + minute) ) {
          if( ! is_busy ) {
            if( first_busy_min > minute )
              first_busy_min = minute;
            is_busy = True;
          }
        } else {
          if( is_busy ) {
            if( last_busy_min < minute )
              last_busy_min = minute;
            is_busy = False;
          }
        } /* if */
      }
    } /* loop */

    first_busy_min = first_busy_min / 60.0;
    last_busy_min  = last_busy_min  / 60.0 + 0.9;

    first_time = TimMakeTime( 1970, 1, 1, (int) first_busy_min, 0, 0 );
    last_time  = TimMakeTime( 1970, 1, 1, (int) last_busy_min, 0, 0 );

    if( first_time < start_time )
      start_time = first_time;

    if( last_time > stop_time )
      stop_time = last_time;

  } /* block */


  /* Any change? */
  if( start_time != plan_ref -> old_entry_start ||
      stop_time  != plan_ref -> old_entry_stop ||
      plan_ref -> entry_delta != plan_ref -> old_entry_delta )
    do_update = True;

  plan_ref -> old_entry_start = start_time;
  plan_ref -> old_entry_stop  = stop_time;
  plan_ref -> old_entry_delta = plan_ref -> entry_delta;

  plan_ref -> entry_start = start_time;
  plan_ref -> entry_stop  = stop_time;


  /* Create a label containing the times. */
  label_lines      = 0;
  current_time     = start_time;
  first_time       = True;
  time_buffer[ 0 ] = '\0';

  while( current_time <= stop_time ) {
    xtmFoFormatTime( current_time, buffer, sizeof( buffer ) );

    if( ! first_time ) {
      strcat( time_buffer, "\n-\n" );
      label_lines++;
    } else {
      first_time = False;
    }

    strcat( time_buffer, buffer );
    label_lines++;

    TimAddMinutes( &current_time, plan_ref -> entry_delta );
  } /* while */


  /* Set the time label. */
  if( do_update )
    xitStringSetLabel( timeLaW, time_buffer );


  /* The height of a single line in the time label. */
  {
    Dimension   width;
    XmString    xstr;
    XmFontList  font_list;

    n = 0;
    XtSetArg( args[ n ], XmNfontList, &font_list ); n++;
    XtGetValues( timeLaW, args, n );

    xstr = XmStringCreate( " ", CS );
    XmStringExtent( font_list, xstr, &width, &plan_ref -> time_line_height );
    XmStringFree( xstr );

    plan_ref -> pixel_per_min = (2.0 * plan_ref -> time_line_height) / 
                                 plan_ref -> entry_delta;
  }

  /* Set the size of the day separators. */
  n = 0;
  XtSetArg( args[ n ], XmNheight, &plan_ref -> time_label_height ); n++;
  XtGetValues( timeLaW, args, n );


  return;

} /* updateTimeScale */


/*----------------------------------------------------------------------*/

static void 
  addCalendarCB( Widget        widget,
                 PLAN_REC_REF  plan_ref,
                 XtPointer     call_data )
{

  /* Code. */

  /* Initialize? */
  if( plan_ref -> open_handle == NULL )
    plan_ref -> open_handle = xtmOvInitialize( plan_ref -> appl_data_ref,
                                               plan_ref -> plannerW,
                                               False,
                                               addCalendarApplyCB,
                                               (void *) plan_ref );
  if( plan_ref -> open_handle == NULL )
    return;

  /* Display the window. */
  xtmOvOpenView( plan_ref -> open_handle );


  return;

} /* addCalendarCB */


/*----------------------------------------------------------------------*/

static void
  addCalendarApplyCB( XTM_OV_REASON    reason,
                      XTM_CD_CAL_INFO  *db_info_ref,
                      void             *user_data )
{

  /* Variables. */
  PLAN_REC_REF  plan_ref;


  /* Code. */

  if( reason != XTM_OV_REASON_APPLY && reason != XTM_OV_REASON_OK )
    return;

  plan_ref = (PLAN_REC_REF) user_data;

  /* Additional calendar? */
  setAdditionalCals( plan_ref );


  return;

} /* addCalendarApplyCB */


/*----------------------------------------------------------------------*/

static void 
  applyCB( Widget        widget,
           PLAN_REC_REF  plan_ref,
           XtPointer     call_data )
{

  /* Code. */

  /* Update what we display. */
  updatePlanner( plan_ref );


  return;

} /* applyCB */


/*----------------------------------------------------------------------*/

static void 
  calendarSelectedCB( Widget                        widget,
                      PLAN_REC_REF                  plan_ref,
                      XmToggleButtonCallbackStruct  *call_data )
{

  /* Variables. */
  Arg        args[ 10 ];
  Cardinal   n;
  XtPointer  user_data_ref;


  /* Code. */

  /* Reset the current reference counter? */
  if( ! call_data -> set ) {
    user_data_ref = (XtPointer) 0;

    n = 0;
    XtSetArg( args[ n ], XmNuserData, user_data_ref ); n++;
    XtSetValues( widget, args, n );
  }


  /* Instant update? */
  if( plan_ref -> instant_update )
    updatePlanner( plan_ref );


  return;

} /* calendarSelectedCB */


/*----------------------------------------------------------------------*/

static void 
  clearCB( Widget        widget,
           PLAN_REC_REF  plan_ref,
           XtPointer     call_data )
{

  /* Code. */

  /* Clear and update. */
  clearCalsAndGroups( plan_ref, True );


  return;

} /* clearCB */


/*----------------------------------------------------------------------*/

static void 
  closeCB( Widget        widget,
           PLAN_REC_REF  plan_ref,
           XtPointer     call_data )
{

  /* Variables. */
  int  index;


  /* Code. */

  /* Do we have a user action callback registered? */
  if( plan_ref -> actionCB != NULL )
    (* plan_ref -> actionCB)( XTM_PL_REASON_EXIT,
                              plan_ref -> user_data );

  /* Keep the window in cache? */
  for( index = 0; index < MAX_CACHE_ENTRIES; index++ ) {
    if( cache_entries[ index ] == NULL ) {
      cache_entries[ index ] = plan_ref;

      destroyCB( NULL, plan_ref, NULL );
      XtPopdown( plan_ref -> plannerW );

      return;
    }
  }


  /* Window not kept in cache, really remove it. */
  XtDestroyWidget( plan_ref -> plannerW );


  return;

} /* closeCB */


/*----------------------------------------------------------------------*/

static void 
  ctrlMenuCB( Widget                     widget,
              PLAN_REC_REF               plan_ref,
              XmRowColumnCallbackStruct  *call_data )
{

  /* Variables. */
  Widget                  mainW;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  appl_data_ref   = plan_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;

  mainW = XtNameToWidget( plan_ref -> plannerW, "PlanTlBase.PlanTlFo" );


  /* Select what to do. */
  switch( (int) call_data -> data ) {

    /* Instant update. */
    case 0:
      if( XmToggleButtonGetState( call_data -> widget ) )
        plan_ref -> instant_update = True;
      else
        plan_ref -> instant_update = False;
      break;

  } /* switch */


  return;

} /* ctrlMenuCB */


/*----------------------------------------------------------------------*/

static void
  ctrlMenuMapCB( Widget        widget,
                 PLAN_REC_REF  plan_ref,
                 XtPointer     call_data )
{

  /* Variables. */
  Widget  tempW;


  /* Code. */

  /* All entries? */
  tempW = XtNameToWidget( widget, "InstantUpdateBu" );
  if( plan_ref -> instant_update )
    XmToggleButtonSetState( tempW, True, False );
  else
    XmToggleButtonSetState( tempW, False, False );


  return;

} /* ctrlMenuMapCB */


/*----------------------------------------------------------------------*/

static void 
  destroyCB( Widget        widget,
             PLAN_REC_REF  plan_ref,
             XtPointer     call_data )
{

  /* Code. */

  /* Do we have a user action callback registered? */
  if( plan_ref -> actionCB != NULL )
    (* plan_ref -> actionCB)( XTM_PL_REASON_DESTROY, 
                              plan_ref -> user_data );

  /* De-register updates. */
  if( plan_ref -> update_id != 0 )
    xtmUpRemove( plan_ref -> update_id );

  plan_ref -> update_id = 0;

  /* Destroy sub-windows. */
  if( plan_ref -> open_handle != NULL ) {
    xtmOvDestroy( plan_ref -> open_handle );
    plan_ref -> open_handle = NULL;
  }

  if( plan_ref -> nav_cal_handle != NULL ) {
    xtmNcDestroy( plan_ref -> nav_cal_handle );
    plan_ref -> nav_cal_handle = NULL;
  }

  if( plan_ref -> msg_handle != NULL ) {
    xtmMsDestroy( plan_ref -> msg_handle );
    plan_ref -> msg_handle = NULL;
  }


  /* Release the user data (only if not cached). */
  if( widget != NULL ) {
    SysFree( plan_ref );
  }


  return;

} /* destroyCB */


/*----------------------------------------------------------------------*/

static void 
  doMsgDateCB( Widget        widget,
               PLAN_REC_REF  plan_ref,
               XtPointer     call_data )
{

  /* Variables. */
  Arg           args[ 10 ];
  Cardinal      n;
  XtPointer     user_data;
  TIM_TIME_REF  to_date;


  /* Code. */

  /* Fetch the selected date. */
  n = 0;
  XtSetArg( args[ n ], XmNuserData, &user_data ); n++;
  XtGetValues( widget, args, n );

  to_date = (TIM_TIME_REF) user_data;

  sendMessage( plan_ref, to_date );


  return;

} /* doMsgDateCB */


/*----------------------------------------------------------------------*/

static void 
  fileMenuCB( Widget                     widget,
              PLAN_REC_REF               plan_ref,
              XmRowColumnCallbackStruct  *call_data )
{

  /* Variables. */
  Widget                  mainW;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  appl_data_ref   = plan_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;

  mainW = XtNameToWidget( plan_ref -> plannerW, "PlanTlBase.PlanTlFo" );


  /* Select what to do. */
  switch( (int) call_data -> data ) {

    /* Close the window. */
    case 0:
      closeCB( widget, plan_ref, NULL );
      break;

  } /* switch */


  return;

} /* fileMenuCB */


/*----------------------------------------------------------------------*/

static void 
  groupSelectedCB( Widget                        widget,
                   PLAN_REC_REF                  plan_ref,
                   XmToggleButtonCallbackStruct  *call_data )
{

  /* Variables. */
  Boolean                 ok;
  int                     char_read;
  int                     ref_count;
  char                    calendar[ XTM_GL_MAX_CAL_NAME + 1 ];
  char                    *group_name;
  char                    *members_ref;
  Arg                     args[ 10 ];
  Cardinal                n;
  Widget                  baseW;
  Widget                  mainW;
  Widget                  calW;
  XtPointer               user_data_ref;
  XTM_GD_ENTRY            group_info;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  custom_data_ref = plan_ref -> appl_data_ref -> custom_data;

  mainW = XtNameToWidget( plan_ref -> plannerW, "PlanTlBase.PlanTlFo" );

  baseW = XtNameToWidget( mainW,
            "PlanWhatFo.CalSelSw.ClipWindow.CalSelRc" );

  /* Fetch the name of the group. */
  group_name = xitStringGetLabel( widget );

  ok = xtmGdFetchNamedGroup( custom_data_ref -> group_db_handle,
                             group_name, &group_info );
  if( ! ok )
    return;

  members_ref = group_info.members;

  do {

    while( isspace( *members_ref ) )
      members_ref++;

    if( *members_ref == '\0' )
      break;

    char_read = strlen( members_ref );
    sscanf( members_ref, "%s%n", calendar, &char_read );
    members_ref = members_ref + char_read;

    /* Search the calendar. */
    calW = fetchCalendarWidget( plan_ref, calendar );
    if( calW == NULL )
      continue;

    /* Fetch the current reference counter. */
    n = 0;
    XtSetArg( args[ n ], XmNuserData, &user_data_ref ); n++;
    XtGetValues( calW, args, n );

    ref_count = (int) user_data_ref;

    if( call_data -> set ) {
      if( ref_count < 1 ) 
        XmToggleButtonSetState( calW, True, False );
      ref_count++;
    } else {
      if( ref_count == 1 ) 
        XmToggleButtonSetState( calW, False, False );
      if( ref_count > 0 )
        ref_count--;
    }

    user_data_ref = (XtPointer) ref_count;

    n = 0;
    XtSetArg( args[ n ], XmNuserData, user_data_ref ); n++;
    XtSetValues( calW, args, n );

  } while( True );

  SysFree( group_info.members );
  SysFree( group_name );


  /* Instant update? */
  if( plan_ref -> instant_update )
    updatePlanner( plan_ref );


  return;

} /* groupSelectedCB */


/*----------------------------------------------------------------------*/

static void 
  infoCB( Widget                     widget,
          PLAN_REC_REF               plan_ref,
          XmRowColumnCallbackStruct  *call_data )
{

  /* Code. */

  /* About window? */
  if( (int) call_data -> data == 6 ) {
    xtmHlDisplayAboutWindow( plan_ref -> plannerW );

    return;
  }

  /* Use the standard help. */
  xtmHlDisplayHelp( plan_ref -> appl_data_ref -> info_handle,
                    (int) call_data -> data,
                    planner_window_id, "" );

  return;

} /* infoCB */


/*----------------------------------------------------------------------*/

static void 
  navigateCB( Widget                        widget,
              PLAN_REC_REF                  plan_ref,
              XmUbArrowLabelCallbackStruct  *call_data )
{

  /* Variables. */
  Widget        mainW;
  Widget        tempW;
  TIM_TIME_REF  planner_start;


  /* Code. */

  mainW = XtNameToWidget( plan_ref -> plannerW, "PlanTlBase.PlanTlFo" );


  planner_start = plan_ref -> planner_start;


  /* Previous/next week? */
  tempW = XtNameToWidget( mainW, "NavFo.NavWeekAl" );
  if( tempW == widget ) {
    if( call_data -> reason == XmUbCR_BACK_ARROW_ACTIVATED )
      TimAddDays( &planner_start, -7 );
    else
      TimAddDays( &planner_start, 7 );
  }

  /* Previous/next month? */
  tempW = XtNameToWidget( mainW, "NavFo.NavMonthAl" );
  if( tempW == widget ) {
    if( call_data -> reason == XmUbCR_BACK_ARROW_ACTIVATED )
      TimAddMonths( &planner_start, -1 );
    else
      TimAddMonths( &planner_start, 1 );
  }


  /* Within limits? */
  if( TimIndexOfYear( planner_start ) < 1971 ) {
    XBell( XtDisplay( widget ), 100 );

    return;
  }

  /* Update the day view. */
  plan_ref -> planner_start = planner_start;

  /* Display. */
  updatePlannerDates( plan_ref, False );
  updatePlanner( plan_ref );


  return;

} /* navigateCB */


/*----------------------------------------------------------------------*/

static void 
  optMenuCB( Widget                     widget,
             PLAN_REC_REF               plan_ref,
             XmRowColumnCallbackStruct  *call_data )
{

  /* Variables. */
  Widget                  mainW;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  appl_data_ref   = plan_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;

  mainW = XtNameToWidget( plan_ref -> plannerW, "PlanTlBase.PlanTlFo" );


  /* Select what to do. */
  switch( (int) call_data -> data ) {

    /* Send message. */
    case 0:
      sendMessage( plan_ref, plan_ref -> planner_start );
      break;

    /* Redisplay. */
    case 2:
      refreshCB( NULL, plan_ref, NULL );
      break;

  } /* switch */


  return;

} /* optMenuCB */


/*----------------------------------------------------------------------*/

static void 
  refreshCB( Widget        widget,
             PLAN_REC_REF  plan_ref,
             XtPointer     call_data )
{

  /* Code. */

  /* Remove the cached data. */
  removeCached( plan_ref );

  /* Update what we display. */
  updatePlanner( plan_ref );


  return;

} /* applyCB */


/*----------------------------------------------------------------------*/

void 
  scrollHorizCB( Widget                     widget,
                 PLAN_REC_REF               plan_ref,
                 XmScrollBarCallbackStruct  *call_data )
{

  /* Variables. */
  Arg       args[ 5 ];
  Cardinal  n;
  Widget    mainW;
  Widget    scrollBar;
  Widget    tempW;


  /* Code. */

  mainW = XtNameToWidget( plan_ref -> plannerW, "PlanTlBase.PlanTlFo" );


  /* Scroll the day names window. */
  tempW = XtNameToWidget( mainW, "DayDispSw" );

  n = 0;
  XtSetArg( args[ n ], XmNhorizontalScrollBar, &scrollBar ); n++;
  XtGetValues( tempW, args, n );

  scrollWindow( plan_ref, scrollBar, call_data -> value );


  return;

} /* scrollHorizCB */


/*----------------------------------------------------------------------*/

void 
  scrollVertCB( Widget                     widget,
                PLAN_REC_REF               plan_ref,
                XmScrollBarCallbackStruct  *call_data )
{

  /* Variables. */
  Arg       args[ 5 ];
  Cardinal  n;
  Widget    mainW;
  Widget    scrollBar;
  Widget    tempW;


  /* Code. */

  mainW = XtNameToWidget( plan_ref -> plannerW, "PlanTlBase.PlanTlFo" );


  /* Scroll the times window. */
  tempW = XtNameToWidget( mainW, "TimeDispSw" );

  n = 0;
  XtSetArg( args[ n ], XmNverticalScrollBar, &scrollBar ); n++;
  XtGetValues( tempW, args, n );

  scrollWindow( plan_ref, scrollBar, call_data -> value );


  return;

} /* scrollVertCB */


/*----------------------------------------------------------------------*/

static void
  updateCB( UINT32  flags,
            void    *user_data,
            void    *update_user_data )
{

  /* Variables. */
  PLAN_REC_REF  plan_ref;


  /* Code. */

  /* Our private data. */
  plan_ref = (PLAN_REC_REF) user_data;


  /* Do refresh? */
  if( flagIsSet( flags, XTM_UP_PLANNER ) )
    refreshCB( NULL, plan_ref, NULL );


  return;

} /* updateCB */


/*----------------------------------------------------------------------*/

static void 
  zoomCB( Widget        widget,
          PLAN_REC_REF  plan_ref,
          XtPointer     call_data )
{

  /* Variables. */
  int     hours;
  int     minutes;
  char    *char_ref;
  Widget  mainW;
  Widget  tempW;


  /* Code. */

  mainW = XtNameToWidget( plan_ref -> plannerW, "PlanTlBase.PlanTlFo" );


  /* Fetch the zoom string. */
  tempW = XtNameToWidget( mainW, "ZoomFs" );

  xitFieldSelectGetCurrent( tempW, &char_ref );
  if( char_ref == NULL )
    return;

  sscanf( char_ref, "%d:%d", &hours, &minutes );

  plan_ref -> entry_delta = (hours * 60) + minutes;

  SysFree( char_ref );


  /* Display the changes. */
  updatePlanner( plan_ref );


  return;

} /* zoomCB */
