/*----------------------------------------------------------------------------
--
--  Module:           xtmMsgSend
--
--  Project:          Xdiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Send an XDiary message to one or more users.
--
--  Filename:         xtmMsgSend.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1992-04-01
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmMsgSend.c, Version: 1.1, Date: 95/02/18 15:52:34";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>

#include <X11/Intrinsic.h>
#include <X11/Shell.h>

#include <Xm/Protocols.h>

#include <Xm/Xm.h>
#include <Xm/Frame.h>
#include <Xm/List.h>
#include <Xm/RowColumn.h>
#include <Xm/SeparatoG.h>
#include <Xm/Text.h>
#include <Xm/ToggleB.h>


#include "System.h"
#include "LstLinked.h"
#include "Message.h"
#include "TimDate.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmCustBase.h"
#include "xtmDbTools.h"
#include "xtmFields.h"
#include "xtmFormat.h"
#include "xtmGroupDb.h"
#include "xtmHelp.h"
#include "xtmIcons.h"
#include "xtmLocate.h"
#include "xtmMsgBase.h"
#include "xtmTools.h"
#include "xitError.h"
#include "xitFieldSel.h"
#include "xitTools.h"
#include "XmUbTimeB.h"
#include "xtmMsgSend.h"
#include "xtmDbMisc.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Local widgets in the recipients window. */
#define dbLa                dataLocalW[  0 ]
#define dbLi                dataLocalW[  1 ]
#define genInfoLa           dataLocalW[  2 ]
#define groupLa             dataLocalW[  3 ]
#define groupLi             dataLocalW[  4 ]
#define ho1Sp               dataLocalW[  5 ]
#define menuBr              dataLocalW[  6 ]
#define nextInfoLa          dataLocalW[  7 ]
#define srcActionRc         dataLocalW[  8 ]
#define toActionRc          dataLocalW[  9 ]
#define toFieldLa           dataLocalW[ 10 ]
#define toFieldRc           dataLocalW[ 11 ]
#define toFieldTx           dataLocalW[ 12 ]
#define toLa                dataLocalW[ 13 ]
#define toLi                dataLocalW[ 14 ]


/* Local widgets in the message window. */
#define dateEntTb           dataLocalW[  0 ]
#define dateEntLa           dataLocalW[  1 ]
#define durEntFs            dataLocalW[  2 ]
#define durEntLa            dataLocalW[  3 ]
#define flagsRc             dataLocalW[  4 ]
#define mailFailTb          dataLocalW[  5 ]
#define msgMenuBr           dataLocalW[  6 ]
#define messageLa           dataLocalW[  7 ]
#define msgTx               dataLocalW[  8 ]
#define onlyMailTb          dataLocalW[  9 ]
#define startEntTb          dataLocalW[ 10 ]
#define startEntLa          dataLocalW[ 11 ]
#define textLa              dataLocalW[ 12 ]
#define textTx              dataLocalW[ 13 ]
#define timeEntFr           dataLocalW[ 14 ]
#define timeEntRc           dataLocalW[ 15 ]


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Record describing one instance of the send window. */
typedef struct {

  /* Any entry added in the message window? */
  Boolean  entry_added;

  /* Entry selected in the to list. */
  int  to_list_entry;

  /* Entry ID to send. */
  UINT32  entry_id;

  /* Entry ID to send comes from DB? */
  char  db_name[ XTM_GL_MAX_CAL_NAME + 1 ];

  /* File selection window. */
  Widget  fileSelectW;

  /* The message window. */
  Widget  messageW;

  /* The recipients window. */
  Widget  recipientW;

  /* Default date for appointment. */
  TIM_TIME_REF  default_entry_date;

  /* Application wide data. */
  XTM_GL_BASE_DATA_REF  appl_data_ref;

} SEND_REC, *SEND_REC_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmMsgSend";

/* IDs for the help windows. */
static char  *send_window_id = "MsgSend";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void 
  addFromListCB( Widget                widget,
                 SEND_REC_REF          send_ref,
                 XmListCallbackStruct  *call_data );

static void 
  clearToCB( Widget        widget,
             SEND_REC_REF  send_ref,
             XtPointer     call_data );

static void 
  closeCB( Widget        widget,
           SEND_REC_REF  send_ref,
           XtPointer     call_data );

static Widget 
  createMessageWindow( Widget        parent,
                       SEND_REC_REF  send_ref );

static Widget 
  createRecipientWindow( Widget        parent,
                         SEND_REC_REF  send_ref );

static void 
  deleteToCB( Widget        widget,
              SEND_REC_REF  send_ref,
              XtPointer     call_data );

static void 
  destroyCB( Widget        widget,
             SEND_REC_REF  send_ref,
             XtPointer     call_data );

static void 
  doAddRecipientsCB( Widget        widget,
                     SEND_REC_REF  send_ref,
                     XtPointer     call_data );

static void 
  includeFileCB( Widget        widget,
                 SEND_REC_REF  send_ref,
                 XtPointer     call_data );

static void 
  includeFileActionCB( XIT_FILE_SEL_REASON  reason,
                       char                 *file_name,
                       void                 *user_data );

static void 
  infoCB( Widget                     widget,
          SEND_REC_REF               send_ref,
          XmRowColumnCallbackStruct  *call_data );

static void 
  messageCB( Widget        widget,
             SEND_REC_REF  send_ref,
             XtPointer     call_data );

static void 
  msgCloseCB( Widget        widget,
              SEND_REC_REF  send_ref,
              XtPointer     call_data );

static void 
  sendCB( Widget        widget,
          SEND_REC_REF  send_ref,
          XtPointer     call_data );

static void
  setMessageWindowValues( SEND_REC_REF  send_ref,
                          char          *db_name,
                          UINT32        id );

static void
  setRecipientWindowValues( SEND_REC_REF  send_ref );

static void 
  toListCB( Widget                widget, 
            SEND_REC_REF          send_ref,
            XmListCallbackStruct  *call_data );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

XTM_MS_HANDLE
  xtmMsInitialize( XTM_GL_BASE_DATA_REF  appl_data_ref,
                   Widget                parent )
{

  /* Variables. */
  SEND_REC_REF  send_ref;


  /* Code. */

  /* Create and initialize our private data. */
  send_ref = SysNew( SEND_REC );
  if( send_ref == NULL )
    return( NULL );

  send_ref -> appl_data_ref = appl_data_ref;
  send_ref -> entry_added   = False;
  send_ref -> fileSelectW   = NULL;
  send_ref -> messageW      = NULL;
  send_ref -> recipientW    = NULL;
  send_ref -> entry_id      = 0;
  send_ref -> db_name[ 0 ]  = '\0';

  /* Create the message recipients window. */
  send_ref -> recipientW = createRecipientWindow( parent, send_ref );
  if( send_ref -> recipientW == NULL ) {
    SysFree( send_ref );

    return( NULL );
  }


  return( (XTM_MS_HANDLE) send_ref );

} /* xtmMsInitialize */


/*----------------------------------------------------------------------*/

void
  xtmMsDestroy( XTM_MS_HANDLE  send_handle )
{

  /* Variables. */
  SEND_REC_REF  send_ref;


  /* Code. */

  if( send_handle == NULL )
    return;

  /* Our private data. */
  send_ref = (SEND_REC_REF) send_handle;


  /* Get rid of all windows. */
  if( send_ref -> recipientW != NULL )
    XtDestroyWidget( send_ref -> recipientW );


  return;

} /* xtmMsDestroy */


/*---------------------------------------------------------------------------*/

void
  xtmMsSendMessage( XTM_MS_HANDLE  send_handle,
                    char           *db_name,
                    UINT32         id )
{

  /* Variables. */
  Arg           args[ 5 ];
  Cardinal      n;
  SEND_REC_REF  send_ref;


  /* Code. */

  /* Our private data. */
  send_ref = (SEND_REC_REF) send_handle;

  send_ref -> entry_added        = False;
  send_ref -> entry_id           = id;
  send_ref -> default_entry_date = (TIM_TIME_REF) 0;

  strcpy( send_ref -> db_name, db_name );


  /* Set the values in the recipient window. */
  setRecipientWindowValues( send_ref );


  /* Make sure the message window is visible. */
  XtPopup( send_ref -> recipientW, XtGrabNone );

  XRaiseWindow( XtDisplay( send_ref -> recipientW ), 
                XtWindow(  send_ref -> recipientW ) );

  XtMapWidget( send_ref -> recipientW );


  /* Make sure our children don't spoil our size. */
  n = 0;
  XtSetArg( args[ n ], XmNallowShellResize, False ); n++;
  XtSetValues( send_ref -> recipientW, args, n );


  return;

} /* xtmMsSendMessage */


/*---------------------------------------------------------------------------*/

void
  xtmMsSetRecipients( XTM_MS_HANDLE  send_handle,
                      char           *msg_to[],
                      int            msg_to_no )
{

  /* Variables. */
  int           index;
  int           index1;
  Arg           args[ 5 ];
  Cardinal      n;
  Widget        mainW;
  Widget        tempW;
  XmString      list_items[ 500 ];
  SEND_REC_REF  send_ref;


  /* Code. */

  /* Our private data. */
  send_ref = (SEND_REC_REF) send_handle;

  if( send_ref == NULL )
    return;

  mainW = XtNameToWidget( send_ref -> recipientW, 
                          "MsgSendTlBase.MsgSendTlFo" );


  /* Assign all recipients to the list. */
  if( msg_to_no > 500 )
    msg_to_no = 500;

  for( index = 0; index < msg_to_no; index++ )
    list_items[ index ] = XmStringCreate( msg_to[ index ], CS );

  tempW = XtNameToWidget( mainW, "ToLiSW.ToLi" );

  n = 0;
  XtSetArg( args[ n ], XmNitems, list_items ); n++;
  XtSetArg( args[ n ], XmNitemCount, index ); n++;
  XtSetValues( tempW, args, n );

  /* Free allocated memory. */
  for( index1 = 0; index1 < index; index1++ )
    XmStringFree( list_items[ index1 ] );


  return;

} /* xtmMsSetRecipients */


/*---------------------------------------------------------------------------*/

void
  xtmMsSetToDate( XTM_MS_HANDLE  send_handle,
                  TIM_TIME_REF   to_date )
{

  /* Variables. */
  SEND_REC_REF  send_ref;


  /* Code. */

  /* Our private data. */
  send_ref = (SEND_REC_REF) send_handle;

  if( send_ref == NULL )
    return;

  send_ref -> default_entry_date = to_date;


  return;

} /* xtmMsSetToDate */


/*----------------------------------------------------------------------*/

static Widget 
  createMessageWindow( Widget        parent,
                       SEND_REC_REF  send_ref )
{

  /* Variables. */
  int                     index;
  char                    *char_ref;
  Arg                     args[ 10 ];
  Cardinal                n;
  Widget                  dataLocalW[ 16 ];
  Widget                  menuCasc[ 2 ];
  Widget                  menuFileBu[ 3 ];
  Widget                  menuHelpBu[ 6 ];
  Widget                  msgFd;
  Widget                  pulldownMenu[ 2 ];
  Widget                  tempW;
  Widget                  workFo;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;

  static char  def_durations[] = 
    "0:10 0:15 0:30 0:45 1:00 1:30 2:00 4:00 6:00";

  static char  *pull_downs[] = { "pdown1", "pdown2" };

  static XIT_TEXT_STRUCT text_buffer_def[] = {
    { "MsgTx",  NULL, 2, True },
    { "TextTx", NULL, 2, True },
  };

  static XIT_CASCADE_STRUCT menupane[] = {
    { "", "", "FilePane" },
    { "", "", "HelpPane" },
  };

  static XIT_MENU_BUTTON_STRUCT file_casc[] = {
    { "",      "",  NULL, "IncludeBu", True,  False, False },
    { XIT_SEP, " ", NULL, "",          False, False, False },
    { "",      "",  NULL, "CloseBu",   True,  False, False },
  };

  static XIT_MENU_BUTTON_STRUCT help_casc[] = {
    { "", "", NULL, "ContextBu", True, False, False },
    { "", "", NULL, "WindowsBu", True, False, False },
    { "", "", NULL, "KeysBu",    True, False, False },
    { "", "", NULL, "IndexBu",   True, False, False },
    { "", "", NULL, "HelpBu",    True, False, False },
    { "", "", NULL, "VersionBu", True, False, False },
  };

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "",   sendCB,     NULL },
    { NULL, NULL,       NULL },
    { NULL, NULL,       NULL },
    { "",   msgCloseCB, NULL },
  };


  /* Code. */

  appl_data_ref   = send_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;

  /* Initialize the menu items. */
  action_buttons[ 0 ].label = msgGetText( MXDI_SEND_BUTTON );
  action_buttons[ 0 ].data  = send_ref;
  action_buttons[ 3 ].label = msgGetText( MXDI_CLOSE_BUTTON );
  action_buttons[ 3 ].data  = send_ref;

  menupane[ 0 ].title    = msgGetText( MXDI_FILE_MENU );
  menupane[ 0 ].mnemonic = msgGetText( MXDI_FILE_MENU_ACC );
  menupane[ 1 ].title    = msgGetText( MXDI_HELP_MENU );
  menupane[ 1 ].mnemonic = msgGetText( MXDI_HELP_MENU_ACC );

  file_casc[ 0 ].title    = msgGetText( MXDI_INCL_FILE_MENU );
  file_casc[ 0 ].mnemonic = msgGetText( MXDI_INCL_FILE_MENU_ACC );
  file_casc[ 2 ].title    = msgGetText( MXDI_CLOSE_MENU );
  file_casc[ 2 ].mnemonic = msgGetText( MXDI_CLOSE_MENU_ACC );

  help_casc[ 0 ].title    = msgGetText( MXDI_HELP_CONTEXT );
  help_casc[ 0 ].mnemonic = msgGetText( MXDI_HELP_CONTEXT_ACC );
  help_casc[ 1 ].title    = msgGetText( MXDI_HELP_WINDOWS );
  help_casc[ 1 ].mnemonic = msgGetText( MXDI_HELP_WINDOWS_ACC );
  help_casc[ 2 ].title    = msgGetText( MXDI_HELP_KEYS );
  help_casc[ 2 ].mnemonic = msgGetText( MXDI_HELP_KEYS_ACC );
  help_casc[ 3 ].title    = msgGetText( MXDI_HELP_INDEX );
  help_casc[ 3 ].mnemonic = msgGetText( MXDI_HELP_INDEX_ACC );
  help_casc[ 4 ].title    = msgGetText( MXDI_HELP_HELP );
  help_casc[ 4 ].mnemonic = msgGetText( MXDI_HELP_HELP_ACC );
  help_casc[ 5 ].title    = msgGetText( MXDI_HELP_VERSION );
  help_casc[ 5 ].mnemonic = msgGetText( MXDI_HELP_VERSION_ACC );


  /* Create a form dialog with buttons. */
  msgFd = xitCreateFormDialog( parent, "MsgFd",
                               1, 0,
                               action_buttons, 
                               XtNumber( action_buttons ) );

  char_ref = msgGetText( MXDI_COMPOSE_MSG_TITLE );

  n = 0;
  XtSetArg( args[ n ], XmNtitle, char_ref ); n++;
  XtSetValues( XtParent( msgFd ), args, n );


  /* Use this form as a working form. */
  workFo = XtNameToWidget( msgFd, "MsgFdFo" );


  /* Create the menubar and menu cascades. */
  msgMenuBr = XmCreateMenuBar( workFo, "MsgMenuBr", args, 0 );

  n = 0;
  for( index = 0; index < XtNumber( pulldownMenu ); index++ )
    pulldownMenu[ index ] = XmCreatePulldownMenu( msgMenuBr, 
                                                  pull_downs[ index ], 
                                                  NULL, n );

  for( index = 0; index < XtNumber( menuCasc ); index++ )
    menuCasc[ index ] = xitCreateCascadeButton( msgMenuBr, 
                                                pulldownMenu[ index ], 
                                                &menupane[ index ] );

  /* The help button should be placed to the right. */
  index = XtNumber( menuCasc ) - 1;
  n     = 0;
  XtSetArg( args[ n ], XmNmenuHelpWidget, menuCasc[ index ] ); n++;
  XtSetValues( menuBr, args, n );


  /* Create the file menu. */
  for( index = 0; index < XtNumber( menuFileBu ); index++ )
    menuFileBu[ index ] = xitCreateMenuPushButton( pulldownMenu[ 0 ],
                                                   &file_casc[ index ] );
  XtAddCallback( menuFileBu[ 0 ], XmNactivateCallback, 
                 (XtCallbackProc) includeFileCB, (XtPointer) send_ref );
  XtAddCallback( menuFileBu[ 2 ], XmNactivateCallback, 
                 (XtCallbackProc) msgCloseCB, (XtPointer) send_ref );


  /* Create the help menu. */
  XtAddCallback( pulldownMenu[ 1 ], XmNentryCallback, 
                 (XtCallbackProc) infoCB, (XtPointer) send_ref );

  for( index = 0; index < XtNumber( menuHelpBu ); index++ ) {
    menuHelpBu[ index ] = xitCreateMenuPushButton( pulldownMenu[ 1 ], 
                                                   &help_casc[ index ] );

    XtAddCallback( menuHelpBu[ index ], XmNactivateCallback, 
                   (XtCallbackProc) infoCB, (XtPointer) index );
  }

  /* We can't do context sensitive help. */
  XtSetSensitive( menuHelpBu[ 0 ], False );


  /* Create the time entry form. */
  timeEntFr = XmCreateFrame( workFo, "TimeEntFr", args, 0 );

  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL  ); n++;
  XtSetArg( args[ n ], XmNpacking,     XmPACK_COLUMN ); n++;
  XtSetArg( args[ n ], XmNnumColumns,  3             ); n++;
  timeEntRc = XmCreateRowColumn( timeEntFr, "TimeEntRc", args, n );


  /* Entry date. */
  dateEntLa = xitCreateLabel( timeEntRc, "DateEntLa", 
                              msgGetText( MXDI_DATE_LABEL ), -1 );

  dateEntTb = xtmFlCreateDateField( timeEntRc, "DateEntTb" );


  /* Start time. */
  startEntLa = xitCreateLabel( timeEntRc, "StartEntLa", 
                               msgGetText( MXDI_ENTRY_START_LABEL ), -1 );

  startEntTb = xtmFlCreateTimeField( timeEntRc, "StartEntTb" );


  /* Duration. */
  durEntLa = xitCreateLabel( timeEntRc, "DurEntLa",
                             msgGetText( MXDI_ENTRY_DURATION_LABEL ), -1 );

  durEntFs = xtmFlCreateSelField( timeEntRc, "DurEntFs",
                                  5, def_durations,
                                  ' ', True,
                                  NULL, NULL );

  tempW = xitFieldSelectGetChild( durEntFs, xitFIELD_SELECT_TEXT_FIELD );


  /* Create the flags form. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmVERTICAL ); n++;
  flagsRc = XmCreateRowColumn( workFo, "FlagsRc", args, n );

  onlyMailTb = xitCreateToggleButton( 
                 flagsRc, "OnlyMailTb", 
                 msgGetText( MXDI_ONLY_MAIL_LABEL ), False );

  mailFailTb = xitCreateToggleButton( 
                 flagsRc, "MailFailTb", 
                 msgGetText( MXDI_MAIL_IF_FAIL_LABEL ), True );


  /* Appointment text label. */
  textLa = xitCreateLabel( workFo, "TextLa", 
                           msgGetText( MXDI_TEXT_LABEL ), -1 );

  textTx = xitCreateTextScrolled( workFo, &text_buffer_def[ 1 ] );

  XtOverrideTranslations( textTx, xtm_di_newline_trans );


  /* Attached remark text. */
  messageLa = xitCreateLabel( workFo, "MessageLa", 
                              msgGetText( MXDI_REMARK_LABEL ), -1 );

  msgTx = xitCreateTextScrolled( workFo, &text_buffer_def[ 0 ] );

  XtOverrideTranslations( msgTx, xtm_di_newline_trans );

  n = 0;
  XtSetArg( args[ n ], XmNmaxLength, 160 ); n++;
  XtSetValues( msgTx, args, n );  


  /* Place the main elements together. */
  xitAttachWidget( msgMenuBr,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_FORM, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( timeEntFr,
                   XmATTACH_WIDGET, msgMenuBr, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,      XmATTACH_NONE, NULL );
  xitAttachWidget( flagsRc,
                   XmATTACH_WIDGET, msgMenuBr, XmATTACH_WIDGET, timeEntFr,
                   XmATTACH_NONE,   NULL,      XmATTACH_NONE, NULL );
  xitAttachWidget( textLa,
                   XmATTACH_WIDGET, timeEntFr, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,      XmATTACH_NONE, NULL );
  xitAttachWidget( XtParent( textTx ),
                   XmATTACH_WIDGET, textLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,   XmATTACH_NONE, NULL );
  xitAttachWidget( messageLa,
                   XmATTACH_WIDGET, XtParent( textTx ), XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,               XmATTACH_NONE, NULL );
  xitAttachWidget( XtParent( msgTx ),
                   XmATTACH_WIDGET, messageLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,      XmATTACH_NONE, NULL );


  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( flagsRc,             args, n );
  XtSetValues( messageLa,           args, n );
  XtSetValues( XtParent( msgTx ),   args, n );
  XtSetValues( timeEntFr,           args, n );
  XtSetValues( textLa,              args, n );
  XtSetValues( XtParent( textTx ),  args, n );


  /* Manage all the children. */
  XtManageChildren( menuCasc,   XtNumber( menuCasc ) );
  XtManageChildren( menuHelpBu, XtNumber( menuHelpBu ) );
  XtManageChildren( menuFileBu, XtNumber( menuFileBu ) );

  xitManageChildren( dataLocalW, XtNumber( dataLocalW ) );


  /* Set the size of the window. */
  xitSetSizeFormDialog( msgFd, True );


  /* Make the final attachments. */
  xitAttachWidget( XtParent( msgTx ),
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL );
  xitAttachWidget( messageLa,
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_WIDGET, XtParent( msgTx ) );
  xitAttachWidget( XtParent( textTx ),
                   XmATTACH_WIDGET, textLa, XmATTACH_FORM,   NULL,
                   XmATTACH_FORM,   NULL,   XmATTACH_WIDGET, messageLa );


  return( msgFd );

} /* createMessageWindow */


/*----------------------------------------------------------------------*/

static Widget 
  createRecipientWindow( Widget        parent,
                         SEND_REC_REF  send_ref )
{

  /* Variables. */
  int                     index;
  char                    *char_ref;
  Arg                     args[ 10 ];
  Cardinal                n;
  Widget                  dataLocalW[ 15 ];
  Widget                  menuCasc[ 2 ];
  Widget                  menuFileBu[ 1 ];
  Widget                  menuHelpBu[ 6 ];
  Widget                  msgSendTl;
  Widget                  pulldownMenu[ 2 ];
  Widget                  srcActionBu[ 1 ];
  Widget                  toplevel;
  Widget                  toActionBu[ 2 ];
  Widget                  workFo;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;

  static char  *pull_downs[] = { "pdown1", "pdown2" };

  static XIT_TEXT_STRUCT text_buffer_def[] = {
    { "ToFieldTx", NULL, 1, True },
  };

  static XIT_CASCADE_STRUCT menupane[] = {
    { "", "", "FilePane" },
    { "", "", "HelpPane" },
  };

  static XIT_MENU_BUTTON_STRUCT file_casc[] = {
    { "", "", NULL, "CloseBu", True, False, False },
  };

  static XIT_MENU_BUTTON_STRUCT help_casc[] = {
    { "", "", NULL, "ContextBu", True, False, False },
    { "", "", NULL, "WindowsBu", True, False, False },
    { "", "", NULL, "KeysBu",    True, False, False },
    { "", "", NULL, "IndexBu",   True, False, False },
    { "", "", NULL, "HelpBu",    True, False, False },
    { "", "", NULL, "VersionBu", True, False, False },
  };

  static XIT_PUSH_STRUCT source_action_def[] = {
    { "AddPb", "", "", True, NULL },
  };

  static XIT_PUSH_STRUCT to_action_def[] = {
    { "DeletePb", "", "", True, NULL },
    { "ClearPb",  "", "", True, NULL },
  };

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "",   messageCB, NULL },
    { NULL, NULL,      NULL },
    { NULL, NULL,      NULL },
    { "",   closeCB,   NULL },
  };


  /* Code. */

  appl_data_ref   = send_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;

  /* Initialize the menu items. */
  action_buttons[ 0 ].label = msgGetText( MXDI_COMPOSE_MESSAGE_BUTTON );
  action_buttons[ 0 ].data  = send_ref;
  action_buttons[ 3 ].label = msgGetText( MXDI_CLOSE_BUTTON );
  action_buttons[ 3 ].data  = send_ref;

  menupane[ 0 ].title    = msgGetText( MXDI_FILE_MENU );
  menupane[ 0 ].mnemonic = msgGetText( MXDI_FILE_MENU_ACC );
  menupane[ 1 ].title    = msgGetText( MXDI_HELP_MENU );
  menupane[ 1 ].mnemonic = msgGetText( MXDI_HELP_MENU_ACC );

  file_casc[ 0 ].title    = msgGetText( MXDI_CLOSE_MENU );
  file_casc[ 0 ].mnemonic = msgGetText( MXDI_CLOSE_MENU_ACC );

  help_casc[ 0 ].title    = msgGetText( MXDI_HELP_CONTEXT );
  help_casc[ 0 ].mnemonic = msgGetText( MXDI_HELP_CONTEXT_ACC );
  help_casc[ 1 ].title    = msgGetText( MXDI_HELP_WINDOWS );
  help_casc[ 1 ].mnemonic = msgGetText( MXDI_HELP_WINDOWS_ACC );
  help_casc[ 2 ].title    = msgGetText( MXDI_HELP_KEYS );
  help_casc[ 2 ].mnemonic = msgGetText( MXDI_HELP_KEYS_ACC );
  help_casc[ 3 ].title    = msgGetText( MXDI_HELP_INDEX );
  help_casc[ 3 ].mnemonic = msgGetText( MXDI_HELP_INDEX_ACC );
  help_casc[ 4 ].title    = msgGetText( MXDI_HELP_HELP );
  help_casc[ 4 ].mnemonic = msgGetText( MXDI_HELP_HELP_ACC );
  help_casc[ 5 ].title    = msgGetText( MXDI_HELP_VERSION );
  help_casc[ 5 ].mnemonic = msgGetText( MXDI_HELP_VERSION_ACC );

  source_action_def[ 0 ].title = msgGetText( MXDI_MSG_ADD_RECIPIENT );
  to_action_def[ 0 ].title     = msgGetText( MXDI_DELETE_ITEM );
  to_action_def[ 1 ].title     = msgGetText( MXDI_CLEAR_BUTTON );


  /* Create a separate window. */
  toplevel = xitGetToplevelWidget( parent );

  msgSendTl = xitCreateToplevelDialog( toplevel, "MsgSendTl",
                                       1, 0,
                                       action_buttons,
                                       XtNumber( action_buttons ) );

  char_ref = msgGetText( MXDI_RECIPIENT_MSG_TITLE );

  n = 0;
  XtSetArg( args[ n ], XmNtitle, char_ref ); n++;
  XtSetValues( msgSendTl, args, n );

  char_ref = msgGetText( MXDI_SEND_MSG_IC_TITLE );

  n = 0;
  XtSetArg( args[ n ], XmNiconName, char_ref ); n++;
  XtSetValues( msgSendTl, args, n );

  XtAddCallback( msgSendTl, XmNdestroyCallback, 
                 (XtCallbackProc) destroyCB, (XtPointer) send_ref );


  /* Exit from message send if this window is deleted. */
  {
    Atom  wm_delete_window;

    wm_delete_window = XmInternAtom( XtDisplay( msgSendTl ),
                                     "WM_DELETE_WINDOW", False );

    XmAddWMProtocols(        msgSendTl, &wm_delete_window, 1 );
    XmAddWMProtocolCallback( msgSendTl, wm_delete_window, 
                             (XtCallbackProc) closeCB, (XtPointer) send_ref );
  } /* block */


  /* Use this form as a working form. */
  workFo = XtNameToWidget( msgSendTl, "MsgSendTlBase.MsgSendTlFo" );


  /* Create the menubar and menu cascades. */
  menuBr = XmCreateMenuBar( workFo, "MenuBr", args, 0 );

  n = 0;
  for( index = 0; index < XtNumber( pulldownMenu ); index++ )
    pulldownMenu[ index ] = XmCreatePulldownMenu( menuBr, 
                                                  pull_downs[ index ], 
                                                  NULL, n );

  for( index = 0; index < XtNumber( menuCasc ); index++ )
    menuCasc[ index ] = xitCreateCascadeButton( menuBr, 
                                                pulldownMenu[ index ], 
                                                &menupane[ index ] );

  /* The help button should be placed to the right. */
  index = XtNumber( menuCasc ) - 1;
  n     = 0;
  XtSetArg( args[ n ], XmNmenuHelpWidget, menuCasc[ index ] ); n++;
  XtSetValues( menuBr, args, n );


  /* Create the file menu. */
  for( index = 0; index < XtNumber( menuFileBu ); index++ )
    menuFileBu[ index ] = xitCreateMenuPushButton( pulldownMenu[ 0 ],
                                                   &file_casc[ index ] );
  XtAddCallback( menuFileBu[ 0 ], XmNactivateCallback, 
                 (XtCallbackProc) closeCB, (XtPointer) send_ref );


  /* Create the help menu. */
  XtAddCallback( pulldownMenu[ 1 ], XmNentryCallback, 
                 (XtCallbackProc) infoCB, (XtPointer) send_ref );

  for( index = 0; index < XtNumber( menuHelpBu ); index++ ) {
    menuHelpBu[ index ] = xitCreateMenuPushButton( pulldownMenu[ 1 ], 
                                                   &help_casc[ index ] );

    XtAddCallback( menuHelpBu[ index ], XmNactivateCallback, 
                   (XtCallbackProc) infoCB, (XtPointer) index );
  }

  /* We can't do context sensitive help. */
  XtSetSensitive( menuHelpBu[ 0 ], False );


  /* General help text. */
  genInfoLa = xitCreateLabel( workFo, "GenInfoLa", 
                              msgGetText( MXDI_MSG_INFO_LABEL ), -1 );

  /* Create the to user list. */
  toLa = xitCreateLabel( workFo, "ToLa", msgGetText( MXDI_MSG_TO_LABEL ), -1 );

  n = 0;
  XtSetArg( args[ n ], XmNlistSizePolicy,         XmCONSTANT ); n++;
  XtSetArg( args[ n ], XmNscrollBarDisplayPolicy, XmSTATIC ); n++;
  XtSetArg( args[ n ], XmNselectionPolicy,        XmSINGLE_SELECT ); n++;
  XtSetArg( args[ n ], XmNlistMarginHeight,       5 ); n++;
  XtSetArg( args[ n ], XmNlistMarginWidth,        5 ); n++;
  toLi = XmCreateScrolledList( workFo, "ToLi", args, n );

  XtAddCallback( toLi, XmNsingleSelectionCallback,
                 (XtCallbackProc) toListCB, (XtPointer) send_ref );


  /* Delete or clear recipients. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation,  XmHORIZONTAL ); n++;
  XtSetArg( args[ n ], XmNspacing,      10 ); n++;
  XtSetArg( args[ n ], XmNmarginHeight, 10 ); n++;
  toActionRc = XmCreateRowColumn( workFo, "ToActionRc", args, n );

  for( index = 0; index < XtNumber( to_action_def ); index++ )
    toActionBu[ index ] = xitCreatePushButton( 
                            toActionRc, &to_action_def[ index ] );

  XtAddCallback( toActionBu[ 0 ], XmNactivateCallback, 
                 (XtCallbackProc) deleteToCB, (XtPointer) send_ref );
  XtAddCallback( toActionBu[ 1 ], XmNactivateCallback, 
                 (XtCallbackProc) clearToCB, (XtPointer) send_ref );


  /* Create calendar list. */
  dbLa = xitCreateLabel( workFo, "DbLa", 
                         msgGetText( MXDI_MSG_DB_LIST_LABEL ), -1 );

  n = 0;
  XtSetArg( args[ n ], XmNlistSizePolicy,         XmCONSTANT ); n++;
  XtSetArg( args[ n ], XmNscrollBarDisplayPolicy, XmSTATIC ); n++;
  XtSetArg( args[ n ], XmNselectionPolicy,        XmSINGLE_SELECT ); n++;
  XtSetArg( args[ n ], XmNlistMarginHeight,       5 ); n++;
  XtSetArg( args[ n ], XmNlistMarginWidth,        5 ); n++;
  dbLi = XmCreateScrolledList( workFo, "DbLi", args, n );

  XtAddCallback( dbLi, XmNdefaultActionCallback,
                 (XtCallbackProc) addFromListCB, (XtPointer) send_ref );



  /* Create the group list. */
  groupLa = xitCreateLabel( workFo, "GroupLa", 
                            msgGetText( MXDI_MSG_GROUP_LIST_LABEL ), -1 );

  n = 0;
  XtSetArg( args[ n ], XmNlistSizePolicy,         XmCONSTANT ); n++;
  XtSetArg( args[ n ], XmNscrollBarDisplayPolicy, XmSTATIC ); n++;
  XtSetArg( args[ n ], XmNselectionPolicy,        XmSINGLE_SELECT ); n++;
  XtSetArg( args[ n ], XmNlistMarginHeight,       5 ); n++;
  XtSetArg( args[ n ], XmNlistMarginWidth,        5 ); n++;
  groupLi = XmCreateScrolledList( workFo, "GroupLi", args, n );

  XtAddCallback( groupLi, XmNdefaultActionCallback,
                 (XtCallbackProc) addFromListCB, (XtPointer) send_ref );


  /* Calendar/Mail address. */
  toFieldRc = XmCreateRowColumn( workFo, "ToFieldRc", args, 0 );

  toFieldLa = xitCreateLabel( toFieldRc, "ToFieldLa", 
                              msgGetText( MXDI_MSG_CAL_MAIL_ADDR_LABEL ), -1 );
  toFieldTx = xitCreateTextCols( toFieldRc, &text_buffer_def[ 0 ], 25 );

  XtAddCallback( toFieldTx, XmNactivateCallback, 
                 (XtCallbackProc) doAddRecipientsCB, (XtPointer) send_ref );

  n = 0;
  XtSetArg( args[ n ], XmNmaxLength, 50 ); n++;
  XtSetValues( toFieldTx, args, n );


  /* Add recipients. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation,  XmHORIZONTAL ); n++;
  XtSetArg( args[ n ], XmNspacing,      10 ); n++;
  XtSetArg( args[ n ], XmNmarginHeight, 10 ); n++;
  srcActionRc = XmCreateRowColumn( workFo, "SrcActionRc", args, n );

  for( index = 0; index < XtNumber( source_action_def ); index++ )
    srcActionBu[ index ] = xitCreatePushButton( 
                             srcActionRc, &source_action_def[ index ] );

  XtAddCallback( srcActionBu[ 0 ], XmNactivateCallback, 
                 (XtCallbackProc) doAddRecipientsCB, (XtPointer) send_ref );


  /* Separator. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL );  n++;
  ho1Sp = XmCreateSeparatorGadget( workFo, "Ho1Sp", args, n );


  /* 'What to do next' help text. */
  nextInfoLa = xitCreateLabel( workFo, "NextInfoLa", 
                               msgGetText( MXDI_MSG_NEXT_INFO_LABEL ), -1 );


  /* Place the main elements together. */
  xitAttachWidget( menuBr,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_FORM, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( genInfoLa,
                   XmATTACH_WIDGET, menuBr, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,   XmATTACH_NONE, NULL );
  xitAttachWidget( toLa,
                   XmATTACH_WIDGET, genInfoLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,      XmATTACH_NONE, NULL );
  xitAttachWidget( XtParent( toLi ),
                   XmATTACH_WIDGET, toLa, XmATTACH_FORM,   NULL,
                   XmATTACH_NONE,   NULL, XmATTACH_WIDGET, toActionRc );
  xitAttachWidget( toActionRc,
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL,
                   XmATTACH_OPPOSITE_WIDGET, srcActionRc );
  xitAttachWidget( dbLa,
                   XmATTACH_WIDGET, genInfoLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,      XmATTACH_NONE, NULL );
  xitAttachWidget( XtParent( dbLi ),
                   XmATTACH_WIDGET, dbLa,
                   XmATTACH_OPPOSITE_WIDGET, dbLa,
                   XmATTACH_FORM, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( groupLa,
                   XmATTACH_WIDGET, XtParent( dbLi ), 
                   XmATTACH_OPPOSITE_WIDGET, dbLa,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( XtParent( groupLi ),
                   XmATTACH_WIDGET, groupLa,
                   XmATTACH_OPPOSITE_WIDGET, dbLa,
                   XmATTACH_FORM, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( toFieldRc,
                   XmATTACH_WIDGET, XtParent( groupLi ), 
                   XmATTACH_OPPOSITE_WIDGET, dbLa,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( srcActionRc,
                   XmATTACH_WIDGET, toFieldRc,
                   XmATTACH_OPPOSITE_WIDGET, dbLa,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( ho1Sp,
                   XmATTACH_WIDGET, srcActionRc, XmATTACH_FORM, NULL,
                   XmATTACH_FORM,   NULL,        XmATTACH_NONE, NULL );
  xitAttachWidget( nextInfoLa,
                   XmATTACH_WIDGET, ho1Sp, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,  XmATTACH_NONE, NULL );

  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( genInfoLa,           args, n );
  XtSetValues( toLa,                args, n );
  XtSetValues( XtParent( toLi ),    args, n );
  XtSetValues( toActionRc,          args, n );
  XtSetValues( dbLa,                args, n );
  XtSetValues( XtParent( dbLi ),    args, n );
  XtSetValues( groupLa,             args, n );
  XtSetValues( XtParent( groupLi ), args, n );
  XtSetValues( toFieldRc,           args, n );
  XtSetValues( srcActionRc,         args, n );
  XtSetValues( nextInfoLa,          args, n );

  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( ho1Sp, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNbottomOffset, 0 ); n++;
  XtSetValues( toActionRc,  args, n );
  XtSetValues( srcActionRc, args, n );


  /* Manage all the children. */
  XtManageChildren( menuCasc,    XtNumber( menuCasc ) );
  XtManageChildren( menuHelpBu,  XtNumber( menuHelpBu ) );
  XtManageChildren( menuFileBu,  XtNumber( menuFileBu ) );
  XtManageChildren( toActionBu,  XtNumber( toActionBu ) );
  XtManageChildren( srcActionBu, XtNumber( srcActionBu ) );

  xitManageChildren( dataLocalW, XtNumber( dataLocalW ) );


  /* Set icon for this window. */
  xtmIcSetSimpleIcon( msgSendTl, workFo, XTM_IC_ICON_MAIL );

  /* Set the size of the window. */
  xitSetSizeToplevelDialog( msgSendTl, True );


  /* Make sure our children don't spoil our size. */
  n = 0;
  XtSetArg( args[ n ], XmNallowShellResize, False ); n++;
  XtSetValues( msgSendTl, args, n );

  /* Make the final attachments. */
  n = 0;
  XtSetArg( args[ n ], XmNrightAttachment, XmATTACH_POSITION ); n++;
  XtSetArg( args[ n ], XmNrightPosition,   50 ); n++;
  XtSetValues( XtParent( toLi ), args, n );

  n = 0;
  XtSetArg( args[ n ], XmNleftAttachment, XmATTACH_POSITION ); n++;
  XtSetArg( args[ n ], XmNleftPosition,   50 ); n++;
  XtSetValues( dbLa, args, n );



  return( msgSendTl );

} /* createRecipientWindow */


/*----------------------------------------------------------------------*/

static void
  setRecipientWindowValues( SEND_REC_REF  send_ref )
{

  /* Variables. */
  int                     index;
  int                     index1;
  char                    buffer[ 100 ];
  char                    *char_ref;
  char                    *db_names;
  Arg                     args[ 5 ];
  Cardinal                n;
  Widget                  mainW;
  Widget                  tempW;
  XmString                list_items[ 100 ];
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  appl_data_ref   = send_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;

  mainW = XtNameToWidget( send_ref -> recipientW,
                          "MsgSendTlBase.MsgSendTlFo" );

  /* Set the to list. */
  tempW = XtNameToWidget( mainW, "ToLiSW.ToLi" );

  n = 0;
  XtSetArg( args[ n ], XmNitems, NULL ); n++;
  XtSetArg( args[ n ], XmNitemCount, 0 ); n++;
  XtSetValues( tempW, args, n );


  /* Set the calendars. */
  (void) xtmCdFetchDbNames( custom_data_ref -> cal_db_handle, &db_names );
  char_ref = db_names;
  index    = 0;

  do {

    int              char_read;
    char             db_name[ XTM_GL_MAX_CAL_NAME + 1 ];
    XTM_CD_CAL_INFO  db_info;

    while( isspace( *char_ref ) )
      char_ref++;

    if( *char_ref == '\0' )
      break;

    char_read = strlen( char_ref );
    sscanf( char_ref, "%s%n", db_name, &char_read );
    char_ref = char_ref + char_read;


    /* Fetch information about the database. */
    (void) xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle, db_name,
                              &db_info, NULL );

    sprintf( buffer, "%-15.15s", db_info.short_name );

    list_items[ index ] = XmStringCreate( buffer, CS );
    index++;

  } while( True );

  if( db_names != NULL )
    SysFree( db_names );


  /* The list is always sorted. */
  xitSortStringList( list_items, index );

  tempW = XtNameToWidget( mainW, "DbLiSW.DbLi" );

  n = 0;
  XtSetArg( args[ n ], XmNitems, list_items ); n++;
  XtSetArg( args[ n ], XmNitemCount, index ); n++;
  XtSetValues( tempW, args, n );

  /* Free allocated memory. */
  for( index1 = 0; index1 < index; index1++ )
    XmStringFree( list_items[ index1 ] );



  /* Set values for the groups. */
  index = 0;

  {
    int   char_read;
    char  group_name[ XTM_GD_MAX_GROUP_NAME + 1 ];
    char  *group_name_ref;
    char  *group_names;

    (void) xtmGdFetchGroupNames( custom_data_ref -> group_db_handle,
                                 &group_names );

    group_name_ref = group_names;

    do {

      while( isspace( *group_name_ref ) )
        group_name_ref++;

      if( *group_name_ref == '\0' )
        break;

      char_read = strlen( group_name_ref );
      sscanf( group_name_ref, "%s%n", group_name, &char_read );
      group_name_ref = group_name_ref + char_read;

      sprintf( buffer, "%s", group_name );

      list_items[ index ] = XmStringCreate( buffer, CS );
      index++;

    } while( True );

    SysFree( group_names );

  } /* block */


  /* The list is always sorted. */
  xitSortStringList( list_items, index );

  tempW = XtNameToWidget( mainW, "GroupLiSW.GroupLi" );

  n = 0;
  XtSetArg( args[ n ], XmNitems, list_items ); n++;
  XtSetArg( args[ n ], XmNitemCount, index ); n++;
  XtSetValues( tempW, args, n );

  /* Free allocated memory. */
  for( index1 = 0; index1 < index; index1++ )
    XmStringFree( list_items[ index1 ] );

  send_ref -> to_list_entry = -1;


  return;

} /* setRecipientWindowValues */


/*----------------------------------------------------------------------*/

static void
  setMessageWindowValues( SEND_REC_REF  send_ref,
                          char          *db_name,
                          UINT32        id )
{

  /* Variables. */
  Boolean                 ok;
  char                    buffer[ 100 ];
  Widget                  baseW;
  Widget                  mainW;
  Widget                  tempW;
  TIM_TIME_REF            default_time;
  XTM_DB_ALL_ENTRY_DEF    entry_record;
  XTM_DB_ENTRY_DATABASES  database;
  XTM_DB_STATUS           db_status;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  appl_data_ref   = send_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;

  mainW = XtNameToWidget( send_ref -> messageW, "MsgFdFo" );

  /* Set default appointment data? */
  if( id == 0 ) {

    baseW = XtNameToWidget( mainW, "TimeEntFr.TimeEntRc" );

    /* Appointment date. */
    tempW = XtNameToWidget( baseW, "DateEntTb" );

    if( send_ref -> default_entry_date == (TIM_TIME_REF) 0 )
      XmUbTimeBoxSetStartDate( tempW, TimLocalTime( TimMakeTimeNow() ) );
    else
      XmUbTimeBoxSetStartDate( tempW, send_ref -> default_entry_date );


    /* Appointment time. */
    default_time = TimMakeTime( 1970, 1, 1, 12, 0, 0 );

    tempW = XtNameToWidget( baseW, "StartEntTb" );
    XmUbTimeBoxSetStartTime( tempW, default_time );


    /* Duration. */
    sprintf( buffer, "%d:%02d", 
             custom_data_ref -> default_entry_delta / 60, 
             custom_data_ref -> default_entry_delta % 60 );

    tempW = XtNameToWidget( baseW, "DurEntFs" );
    xitFieldSelectSetCurrent( tempW, buffer, False );

    return;

  } /* if */


  /* Open the database and fetch the entry. */
  ok = xtmDmOpenDatabase( appl_data_ref, db_name, XTM_DB_FLAG_MODE_READ,
                          &database );
  if( ! ok )
    return;


  /* Fetch the entry. */
  db_status = xtmDbFetchEntry( &database, id, 
                               &entry_record, &entry_record.all_text );

  xtmDbCloseEntryDb( &database );


  if( db_status != XTM_DB_OK ) {
    xitErMessage( send_ref -> recipientW, XIT_ER_ERROR,
                  module_name, "setSendWindowValues",
                  msgGetText( MXDI_NO_ENTRY ) );

    return;
  }


  baseW = XtNameToWidget( mainW, "TimeEntFr.TimeEntRc" );


  /* Date for the appointment. */
  tempW = XtNameToWidget( baseW, "DateEntTb" );

  if( entry_record.entry.entry_type == XTM_DB_DAY_ENTRY )
    XmUbTimeBoxSetStartDate( tempW, entry_record.entry.date_stamp );
  else
    XmUbTimeBoxSetStartDateString( tempW, "" );


  /* Time for the appointment. */
  tempW = XtNameToWidget( baseW, "StartEntTb" );

  if( entry_record.entry.entry_type == XTM_DB_DAY_ENTRY )
    XmUbTimeBoxSetStartTime( tempW, entry_record.entry.time_stamp );
  else
    XmUbTimeBoxSetStartTimeString( tempW, "" );


  /* Duration of the appointment. */
  /* sprintf( buffer, "" ); */

  switch( entry_record.entry.entry_type ) {
    case XTM_DB_DAY_ENTRY:
      if( entry_record.entry.duration > 0 )
        sprintf( buffer, "%2d:%02d", 
                 (int) entry_record.entry.duration / 60,
                 (int) entry_record.entry.duration % 60 );
      break;
  } /* switch */

  tempW = XtNameToWidget( baseW, "DurEntFs" );
  xitFieldSelectSetCurrent( tempW, buffer, False );


  /* Appointment text. */
  tempW = XtNameToWidget( mainW, "TextTxSW.TextTx" );

  if( entry_record.all_text != NULL )
    XmTextSetString( tempW, entry_record.all_text );
  else
    XmTextSetString( tempW, entry_record.entry.text );


  if( entry_record.all_text != NULL )
    SysFree( entry_record.all_text );


  return;

} /* setMessageWindowValues */


/*----------------------------------------------------------------------*/

static void 
  addFromListCB( Widget                widget,
                 SEND_REC_REF          send_ref,
                 XmListCallbackStruct  *call_data )
{

  /* Code. */

  /* This is called from a list default selection, select the object! */
  XmListSelectPos( widget, call_data -> item_position, False );

  doAddRecipientsCB( widget, send_ref, NULL );


  return;

} /* addFromListCB */


/*----------------------------------------------------------------------*/

static void 
  clearToCB( Widget        widget,
             SEND_REC_REF  send_ref,
             XtPointer     call_data )
{

  /* Variables. */
  Arg       args[ 5 ];
  Cardinal  n;
  Widget    mainW;
  Widget    tempW;


  /* Code. */

  mainW = XtNameToWidget( send_ref -> recipientW,
                          "MsgSendTlBase.MsgSendTlFo" );

  /* Remove all items in the to list. */
  tempW = XtNameToWidget( mainW, "ToLiSW.ToLi" );

  n = 0;
  XtSetArg( args[ n ], XmNitems, NULL ); n++;
  XtSetArg( args[ n ], XmNitemCount, 0 ); n++;
  XtSetValues( tempW, args, n );


  return;

} /* clearCB */


/*----------------------------------------------------------------------*/

static void 
  closeCB( Widget        widget,
           SEND_REC_REF  send_ref,
           XtPointer     call_data )
{

  /* Code. */

  /* We don't need the message window anymore. */
  if( send_ref -> messageW != NULL )
    msgCloseCB( send_ref -> messageW, send_ref, NULL );


  XtPopdown( send_ref -> recipientW );


  return;

} /* closeCB */


/*----------------------------------------------------------------------*/

static void 
  deleteToCB( Widget        widget,
              SEND_REC_REF  send_ref,
              XtPointer     call_data )
{

  /* Variables. */
  Widget    mainW;
  Widget    tempW;


  /* Code. */

  mainW = XtNameToWidget( send_ref -> recipientW,
                          "MsgSendTlBase.MsgSendTlFo" );

  /* Any selection in the to list? */
  if( send_ref -> to_list_entry < 0 )
    return;

  /* Delete the entry. */
  tempW = XtNameToWidget( mainW, "ToLiSW.ToLi" );
  XmListDeletePos( tempW, send_ref -> to_list_entry );

  XmListDeselectAllItems( tempW );

  send_ref -> to_list_entry = -1;


  return;

} /* deleteToCB */


/*----------------------------------------------------------------------*/

static void 
  destroyCB( Widget        widget,
             SEND_REC_REF  send_ref,
             XtPointer     call_data )
{

  /* Code. */

  /* File selection. */
  if( send_ref -> fileSelectW != NULL )
    XtDestroyWidget( send_ref -> fileSelectW );

  /* Release the user data. */
  SysFree( send_ref );


  return;

} /* destroyCB */


/*----------------------------------------------------------------------*/

static void 
  doAddRecipientsCB( Widget        widget,
                     SEND_REC_REF  send_ref,
                     XtPointer     call_data )
{

  /* Variables. */
  Boolean                 ok;
  int                     items;
  int                     selected_item_count;
  char                    to_user[ 100 ];
  char                    buffer[ 51 ];
  char                    location[ 51 ];
  char                    *char_ref;
  Arg                     args[ 5 ];
  Cardinal                n;
  Widget                  mainW;
  Widget                  tempW;
  XmString                *selected_items;
  XmString                xstr;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;
  XTM_CD_CAL_INFO         db_info;


  /* Code. */

  custom_data_ref = send_ref -> appl_data_ref -> custom_data;
  mainW           = XtNameToWidget( send_ref -> recipientW,
                                    "MsgSendTlBase.MsgSendTlFo" );

  /* Any database selection? */
  tempW = XtNameToWidget( mainW, "DbLiSW.DbLi" );

  n = 0;
  XtSetArg( args[ n ], XmNselectedItemCount, &selected_item_count ); n++;
  XtSetArg( args[ n ], XmNselectedItems, &selected_items ); n++;
  XtGetValues( tempW, args, n );

  if( selected_item_count == 1 ) {
    tempW = XtNameToWidget( mainW, "ToLiSW.ToLi" );

    XmListAddItemUnselected( tempW, *selected_items, 0 );  

    tempW = XtNameToWidget( mainW, "DbLiSW.DbLi" );
    XmListDeselectAllItems( tempW );
  }


  /* Any group selection? */
  tempW = XtNameToWidget( mainW, "GroupLiSW.GroupLi" );

  n = 0;
  XtSetArg( args[ n ], XmNselectedItemCount, &selected_item_count ); n++;
  XtSetArg( args[ n ], XmNselectedItems, &selected_items ); n++;
  XtGetValues( tempW, args, n );

  if( selected_item_count == 1 ) {
    tempW = XtNameToWidget( mainW, "ToLiSW.ToLi" );

    XmListAddItemUnselected( tempW, *selected_items, 0 );  

    tempW = XtNameToWidget( mainW, "GroupLiSW.GroupLi" );
    XmListDeselectAllItems( tempW );
  }


  /* To mail address. */
  tempW    = XtNameToWidget( mainW, "ToFieldRc.ToFieldTx" );
  char_ref = xitStringGetText( tempW );

  items = sscanf( char_ref, "%s", to_user );
  SysFree( char_ref );

  if( items == 1 ) {

    XmTextSetString( tempW, "" );

    sprintf( buffer, "%s", to_user );

    /* Check if a 'real' calendar. */
    ok = xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle, to_user,
                            &db_info, NULL );

    /* Try to locate the calendar? */
    if( ! ok ) {
      ok = xtmLcLocateCalendar( send_ref -> recipientW, to_user,
                                custom_data_ref -> find_location_script,
                                location, sizeof( location ) );

      if( ok && strlen( location ) > 0 )
        sprintf( buffer, "%-15.15s (%s)", to_user, location );
      else
        sprintf( buffer, "%.40s (%s)", 
                 to_user, msgGetText( MXDI_SEND_AS_MAIL ) );
    }

    tempW = XtNameToWidget( mainW, "ToLiSW.ToLi" );

    xstr = XmStringCreateLtoR( buffer, CS );
    XmListAddItemUnselected( tempW, xstr, 0 );  
    XmStringFree( xstr );

  } /* if */


  /* No selections in the To list. */
  tempW = XtNameToWidget( mainW, "ToLiSW.ToLi" );

  XmListDeselectAllItems( tempW );
  XmListSetBottomPos( tempW, 0 );

  send_ref -> to_list_entry = -1;


  return;

} /* doAddRecipientsCB */


/*----------------------------------------------------------------------*/

static void 
  includeFileCB( Widget        widget,
                 SEND_REC_REF  send_ref,
                 XtPointer     call_data )
{

  /* Variables. */
  XIT_FILE_SEL_LABELS  labels;


  /* Code. */

  /* Do we need to create the dialog? */
  if( send_ref -> fileSelectW == NULL ) {

    labels.dir_list_label       = msgGetText( MXDI_DIRECTORY_LABEL );
    labels.file_list_label      = msgGetText( MXDI_FILES_LABEL );
    labels.filter_label         = msgGetText( MXDI_FILTER_ACTION_LABEL );
    labels.selection_label      = msgGetText( MXDI_SELECTION_LABEL );
    labels.cancel_button        = msgGetText( MXDI_CANCEL_BUTTON );
    labels.filter_button        = msgGetText( MXDI_FILTER_ACTION_LABEL );
    labels.ok_button            = msgGetText( MXDI_OK_BUTTON );
    labels.no_file_sel_msg      = msgGetText( MXDI_NO_FILE_SELECTED );
    labels.cannot_read_file_msg = msgGetText( MXDI_CANNOT_READ_FILE );

    /* The file selection dialog. */
    send_ref -> fileSelectW = 
      xitCreateFileSelect( send_ref -> messageW, "FileSelect",
                           0,
                           "", &labels,
                           includeFileActionCB, (void *) send_ref );

  } /* if */

  /* Display the dialog. */
  xitFileSelectDisplay( send_ref -> fileSelectW );


  return;

} /* includeFileCB */


/*----------------------------------------------------------------------*/

static void 
  includeFileActionCB( XIT_FILE_SEL_REASON  reason,
                       char                 *file_name,
                       void                 *user_data )
{

  /* Variables. */
  Widget        mainW;
  Widget        tempW;
  SEND_REC_REF  send_ref;


  /* Code. */

  send_ref = (SEND_REC_REF) user_data;

  mainW = XtNameToWidget( send_ref -> messageW, "MsgFdFo" );

  /* Only OK reasons. */
  if( reason != XIT_FILE_SELECT_OK )
    return;

  tempW = XtNameToWidget( mainW, "TextTxSW.TextTx" );
  xitStringSetTextFromFile( tempW, file_name );


  return;

} /* includeFileActionCB */


/*----------------------------------------------------------------------*/

static void 
  infoCB( Widget                     widget,
          SEND_REC_REF               send_ref,
          XmRowColumnCallbackStruct  *call_data )
{

  /* Code. */

  xtmHlDisplayHelp( send_ref -> appl_data_ref -> info_handle,
                    (int) call_data -> data,
                    send_window_id, "" );

  return;

} /* infoCB */


/*----------------------------------------------------------------------*/

static void 
  messageCB( Widget        widget,
             SEND_REC_REF  send_ref,
             XtPointer     call_data )
{

  /* Variables. */
  Widget  tempW;


  /* Code. */

  /* Is the message window created? */
  if( send_ref -> messageW == NULL )
    send_ref -> messageW = createMessageWindow( send_ref -> recipientW,
                                                send_ref );

  /* Add entry to the window. */
  if( ! send_ref -> entry_added ) {
    setMessageWindowValues( send_ref, 
                            send_ref -> db_name, send_ref -> entry_id );

    send_ref -> entry_added = True;
  }

  XtManageChild( send_ref -> messageW );


  /* Start at the text field. */
  tempW = XtNameToWidget( send_ref -> messageW, "MsgFdFo.TextTxSW.TextTx" );
  xitGrabFocus( tempW );


  return;

} /* messageCB */


/*----------------------------------------------------------------------*/

static void 
  msgCloseCB( Widget        widget,
              SEND_REC_REF  send_ref,
              XtPointer     call_data )
{

  /* Code. */

  XtUnmanageChild( send_ref -> messageW );


  return;

} /* msgCloseCB */


/*----------------------------------------------------------------------*/

static void 
  sendCB( Widget        widget,
          SEND_REC_REF  send_ref,
          XtPointer     call_data )
{

  /* Variables. */
  Boolean                 mail_fail;
  Boolean                 ok;
  Boolean                 only_mail;
  int                     duration = 0;
  int                     msg_to_count;
  char                    *char_ref;
  char                    *message;
  char                    *text;
  Arg                     args[ 5 ];
  Cardinal                n;
  Widget                  baseW;
  Widget                  mainW;
  Widget                  tempW;
  XmStringTable           msg_to;
  TIM_TIME_REF            entry_date;
  TIM_TIME_REF            entry_time;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  appl_data_ref   = send_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;


  /* Values from the recipients window. */
  mainW = XtNameToWidget( send_ref -> recipientW,
                          "MsgSendTlBase.MsgSendTlFo" );

  /* Addresses to selected. */
  tempW = XtNameToWidget( mainW, "ToLiSW.ToLi" );

  n = 0;
  XtSetArg( args[ n ], XmNitemCount, &msg_to_count ); n++;
  XtSetArg( args[ n ], XmNitems, &msg_to ); n++;
  XtGetValues( tempW, args, n );


  /* Anyone to send to? */
  if( msg_to_count < 1 ) {
    xitErMessage( send_ref -> recipientW, XIT_ER_ERROR,
                  module_name, "sendCB",
                  msgGetText( MXDI_NO_RECIPIENTS ) );
    return;
  }


  /* Values from the message window. */
  mainW = XtNameToWidget( send_ref -> messageW, "MsgFdFo" );

  baseW = XtNameToWidget( mainW, "TimeEntFr.TimeEntRc" );


  /* Fetch the appointment date. */
  tempW = XtNameToWidget( baseW, "DateEntTb" );

  ok = xtmFoFetchDate( mainW, tempW, XTM_FO_START_DATE, True, &entry_date );
  if( ! ok )
    return;


  /* Fetch the start time. */
  tempW = XtNameToWidget( baseW, "StartEntTb" );

  ok = xtmFoFetchTime( mainW, tempW, XTM_FO_START_TIME, True, &entry_time );
  if( ! ok )
    return;


  /* Fetch the duration. */
  tempW = XtNameToWidget( baseW, "DurEntFs" );
  xitFieldSelectGetCurrent( tempW, &char_ref );

  ok = xtmFoFetchMinuteTime( char_ref, True,
                             send_ref -> recipientW, 12 * 60, 
                             &duration );
  SysFree( char_ref );
  if( ! ok )
    return;

  if( duration < 0 )
    duration = 0;


  /* Flags. */
  tempW = XtNameToWidget( mainW, "FlagsRc.OnlyMailTb" );
  only_mail = XmToggleButtonGetState( tempW );

  tempW = XtNameToWidget( mainW, "FlagsRc.MailFailTb" );
  mail_fail = XmToggleButtonGetState( tempW );


  /* Appointment message. */
  tempW = XtNameToWidget( mainW, "MsgTxSW.MsgTx" );
  message = xitStringGetText( tempW );


  /* Appointment text. */
  tempW = XtNameToWidget( mainW, "TextTxSW.TextTx" );
  text  = xitStringGetText( tempW );


  /* Send the message (as mail or XDiary message). */
  ok = xtmMbSendMessage( appl_data_ref, send_ref -> messageW,
                         msg_to_count, msg_to,
                         only_mail, mail_fail,
                         entry_date, entry_time, duration,
                         message, text );

  /* Free occupied memory. */
  SysFree( message );
  SysFree( text );


  /* Close the window. */
  msgCloseCB( widget, send_ref, call_data );
  

  /* Tell we are done. */
  (void) xitCreateInformationDialog( 
           send_ref -> recipientW, "InformationDialog",
           msgGetText( MXDI_INFO_MESSAGE_LABEL ),
           msgGetText( MXDI_MESSAGE_SENT ),
           NULL, NULL );


  return;

} /* sendCB */


/*----------------------------------------------------------------------*/

static void 
  toListCB( Widget                widget, 
            SEND_REC_REF          send_ref,
            XmListCallbackStruct  *call_data )
{

  /* Code. */

  send_ref -> to_list_entry = call_data -> item_position;


  return;

} /* toListCB */
