/*----------------------------------------------------------------------------
--
--  Module:           xtmMsgRead
--
--  Project:          Xdiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Read an XDiary message.
--
--  Filename:         xtmMsgRead.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1991-04-01
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmMsgRead.c, Version: 1.1, Date: 95/02/18 15:52:32";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <X11/Intrinsic.h>

#include <Xm/Xm.h>
#include <Xm/List.h>
#include <Xm/Text.h>

#include "System.h"
#include "DirList.h"
#include "Message.h"
#include "TimDate.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmCalDb.h"
#include "xtmCustBase.h"
#include "xtmDbTools.h"
#include "xtmEditEntry.h"
#include "xtmFormat.h"
#include "xtmMsgAnswer.h"
#include "xtmTools.h"
#include "xitError.h"
#include "xitTools.h"
#include "xtmMsgRead.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Local widgets in the message select window. */
#define msgLa               dataLocalW[  0 ]
#define msgTx               dataLocalW[  1 ]


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/


/* Record describing one instance of the message select window. */
typedef struct {

  /* Application wide data. */
  XTM_GL_BASE_DATA_REF  appl_data_ref;

  /* Displaying this message id. */
  UINT32  msg_id;

  /* Save entry window. */
  XTM_ED_HANDLE  editor_handle;

  /* Answer window. */
  XTM_MA_HANDLE  answer_handle;

  /* Read window. */
  Widget  readW;

} READ_REC, *READ_REC_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmMsgRead";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void 
  answerCB( Widget        widget,
            READ_REC_REF  read_ref,
            XtPointer     call_data );

static void 
  closeCB( Widget        widget,
           READ_REC_REF  read_ref,
           XtPointer     call_data );

static Widget 
  createReadWindow( Widget        parent,
                    READ_REC_REF  read_ref );

static void 
  destroyCB( Widget        widget,
             READ_REC_REF  read_ref,
             XtPointer     call_data );

static void 
  mapCB( Widget        widget,
         READ_REC_REF  read_ref,
         XtPointer     call_data );

static void 
  saveCB( Widget        widget,
          READ_REC_REF  read_ref,
          XtPointer     call_data );

static void 
  saveEditorActionCB( XTM_ED_REASON reason,
                      void          *user_data );

static void
  setReadWindowValues( READ_REC_REF  read_ref,
                       UINT32        msg_id );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

XTM_MR_HANDLE
  xtmMrInitialize( XTM_GL_BASE_DATA_REF  appl_data_ref,
                   Widget                parent )
{

  /* Variables. */
  READ_REC_REF  read_ref;


  /* Code. */

  /* Create and initialize our private data. */
  read_ref = SysNew( READ_REC );
  if( read_ref == NULL )
    return( NULL );

  read_ref -> appl_data_ref = appl_data_ref;
  read_ref -> readW         = NULL;
  read_ref -> answer_handle = NULL;
  read_ref -> editor_handle = NULL;

  /* Create the message read window. */
  read_ref -> readW = createReadWindow( parent, read_ref );
  if( read_ref -> readW == NULL ) {
    SysFree( read_ref );

    return( NULL );
  }


  return( (XTM_MR_HANDLE) read_ref );

} /* xtmMrInitialize */


/*----------------------------------------------------------------------*/

void
  xtmMrDestroy( XTM_MR_HANDLE  read_handle )
{

  /* Variables. */
  READ_REC_REF  read_ref;


  /* Code. */

  if( read_handle == NULL )
    return;

  /* Our private data. */
  read_ref = (READ_REC_REF) read_handle;


  /* Get rid of all windows. */
  if( read_ref -> readW != NULL )
    XtDestroyWidget( read_ref -> readW );


  return;

} /* xtmMRDestroy */


/*---------------------------------------------------------------------------*/

void
  xtmMrReadMessage( XTM_MR_HANDLE  read_handle,
                    UINT32         msg_id )
{

  /* Variables. */
  READ_REC_REF  read_ref;


  /* Code. */

  /* Our private data. */
  read_ref = (READ_REC_REF) read_handle;

  read_ref -> msg_id = msg_id;


  /* Make sure the message window is visable. */
  if( ! XtIsManaged( read_ref -> readW ) )
    XtManageChild( read_ref -> readW );

  XtMapWidget( XtParent( read_ref -> readW ) );


  /* Set the values in the message read window. */
  setReadWindowValues( read_ref, msg_id );


  return;

} /* xtmMrReadMessage */


/*----------------------------------------------------------------------*/

static Widget 
  createReadWindow( Widget        parent,
                    READ_REC_REF  read_ref )
{

  /* Variables. */
  Arg       args[ 10 ];
  Cardinal  n;
  Widget    dataLocalW[ 2 ];
  Widget    msgReadFd;
  Widget    workFo;

  static XIT_TEXT_STRUCT text_buffer_def[] = {
    { "MsgTx",  NULL, 5, False },
  };

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "", saveCB,   NULL },
    { "", answerCB, NULL },
    { "", closeCB,  NULL },
  };


  /* Code. */

  /* Text items. */
  action_buttons[ 0 ].label = msgGetText( MXDI_SAVE_BUTTON );
  action_buttons[ 0 ].data  = read_ref;
  action_buttons[ 1 ].label = msgGetText( MXDI_ANSWER_BUTTON );
  action_buttons[ 1 ].data  = read_ref;
  action_buttons[ 2 ].label = msgGetText( MXDI_CLOSE_BUTTON );
  action_buttons[ 2 ].data  = read_ref;


  /* Create a form dialog with buttons. */
  msgReadFd= xitCreateFormDialog( parent, "MsgReadFd",
                                  1, 0,
                                  action_buttons,
                                  XtNumber( action_buttons ) );

  n = 0;
  XtSetArg( args[ n ], XmNtitle, msgGetText( MXDI_READ_MSG_TITLE ) ); n++;
  XtSetValues( XtParent( msgReadFd ), args, n );

  XtAddCallback( msgReadFd, XmNdestroyCallback, 
                 (XtCallbackProc) destroyCB, (XtPointer) read_ref );

  XtAddCallback( msgReadFd, XmNmapCallback, 
                 (XtCallbackProc) mapCB, (XtPointer) read_ref );


  /* Fetch container for the contents of the window. */
  workFo = XtNameToWidget( msgReadFd, "MsgReadFdFo" );


  /* Message label. */
  msgLa = xitCreateLabel( workFo, "MsgLa", " ", -1 );


  /* The message. */
  msgTx = xitCreateTextScrolled( workFo, &text_buffer_def[ 0 ] );

  n = 0;
  XtSetArg( args[ n ], XmNcursorPositionVisible, False ); n++;
  XtSetValues( msgTx, args, n );


  /* Place the main elements together. */
  xitAttachWidget( msgLa,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( XtParent( msgTx ),
                   XmATTACH_WIDGET, msgLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,  XmATTACH_NONE, NULL );

  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( msgLa,             args, n );
  XtSetValues( XtParent( msgTx ), args, n );


  /* Manage all the children. */
  xitManageChildren( dataLocalW, XtNumber( dataLocalW ) );

  /* Set the initial sizes. */
  xitSetSizeFormDialog( msgReadFd, True );


  /* Make the final attachments. */
  n = 0;
  XtSetArg( args[ n ], XmNrightAttachment,  XmATTACH_FORM ); n++;
  XtSetArg( args[ n ], XmNbottomAttachment, XmATTACH_FORM ); n++;
  XtSetValues( XtParent( msgTx ), args, n );


  return( msgReadFd );

} /* createReadWindow */


/*----------------------------------------------------------------------*/

static void
  setReadWindowValues( READ_REC_REF  read_ref,
                       UINT32        msg_id )
{

  /* Variables. */
  int                     items;
  int                     length;
  char                    *char_ref;
  char                    *message;
  char                    *msg_ref;
  char                    *text;
  char                    buffer[ 250 ];
  Widget                  mainW;
  Widget                  tempW;
  XTM_DB_MESSAGE_DEF      msg_info;
  XTM_DB_STATUS           db_status;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;
  XTM_CD_CAL_INFO         default_db;


  /* Code. */

  appl_data_ref   = read_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;

  mainW = XtNameToWidget( read_ref -> readW, "MsgReadFdFo" );


  /* All messages are saved in the default database. */
  (void) xtmCdFetchDefaultDb( custom_data_ref -> cal_db_handle,
                              &default_db, NULL );


  /* Fetch the message. */
  db_status = xtmDbFetchMessage( default_db.directory, 
                                 msg_id,
                                 &msg_info, &message, &text );
  if( db_status != XTM_DB_OK )
    return;


  /* Message from. */
  sprintf( buffer, "%s %s", msgGetText( MXDI_MESSAGE_FROM ), msg_info.from );

  tempW = XtNameToWidget( mainW, "MsgLa" );
  xitStringSetLabel( tempW, buffer );


  /* Allocate enough space for the message. */
  length = 250;
  if( message != NULL )
    length = length + strlen( message );
  if( text != NULL )
    length = length + strlen( text );

  msg_ref = (char *) SysMalloc( length );
  *msg_ref = '\0';


  /* Message from. */
  sprintf( buffer, "%s %s", msgGetText( MXDI_FROM_LABEL ), msg_info.from );
  strcat(  msg_ref, buffer );
  strcat(  msg_ref, "\n\n" );


  /* The introduction message. */
  sprintf( buffer, "%s", msgGetText( MXDI_REMARK_LABEL ) );
  strcat(  msg_ref, buffer );
  strcat(  msg_ref, "\n" );

  if( message != NULL ) {
    char_ref = message;

    length = strlen( char_ref );
    items = sscanf( char_ref, "%[^\n]%n", buffer, &length );
    while( items == 1 ) {

      strcat( msg_ref, "  " );
      strcat( msg_ref, buffer );
      strcat( msg_ref, "\n" );

      char_ref = char_ref + length;
      if( *char_ref == '\0' )
        break;
      char_ref++;

      length = strlen( char_ref );
      items = sscanf( char_ref, "%[^\n]%n", buffer, &length );

    } /* while */
  } /* if */

  strcat( msg_ref, "\n" );
  strcat( msg_ref, "----\n" );

  /* Date for the appointment. */
  sprintf( buffer, "%s ", msgGetText( MXDI_DATE_LABEL ) );
  strcat(  msg_ref, buffer );

  xtmFoFormatDate( msg_info.date, buffer, sizeof( buffer ) );

  strcat( msg_ref, buffer );
  strcat( msg_ref, "\n" );


  /* Time for the appointment. */
  sprintf( buffer, "%s ", msgGetText( MXDI_ENTRY_START_LABEL ) );
  strcat(  msg_ref, buffer );

  xtmFoFormatTime( msg_info.time, buffer, sizeof( buffer ) );

  strcat( msg_ref, buffer );
  strcat( msg_ref, "\n" );


  /* Duration of the appointment. */
  sprintf( buffer, "%s ", msgGetText( MXDI_ENTRY_DURATION_LABEL ) );
  strcat( msg_ref, buffer );

  if( msg_info.duration > 0 )
    sprintf( buffer, "%2d:%02d", 
             (int) msg_info.duration / 60, (int) msg_info.duration % 60 );
  else
    sprintf( buffer, "%s", "----" );

  strcat( msg_ref, buffer );
  strcat( msg_ref, "\n" );


  /* Appointment text. */
  sprintf( buffer, "%s", msgGetText( MXDI_TEXT_LABEL ) );
  strcat( msg_ref, buffer );
  strcat( msg_ref, "\n" );

  if( text == NULL )
    char_ref = "----\n";
  else
    char_ref = text;

  length = strlen( char_ref );
  items = sscanf( char_ref, "%[^\n]%n", buffer, &length );
  while( items == 1 ) {

    strcat( msg_ref, "  " );
    strcat( msg_ref, buffer );
    strcat( msg_ref, "\n" );

    char_ref = char_ref + length;
    if( *char_ref == '\0' )
      break;
    char_ref++;

    length = strlen( char_ref );
    items = sscanf( char_ref, "%[^\n]%n", buffer, &length );
  }

  strcat( msg_ref, "\n" );


  /* Assign the text. */
  tempW = XtNameToWidget( mainW, "MsgTxSW.MsgTx" );
  XmTextSetString( tempW, msg_ref );


  /* Free allocated data. */
  if( message != NULL )
    SysFree( message );

  if( text != NULL )
    SysFree( text );

  SysFree( msg_ref );


  return;

} /* setReadWindowValues */


/*----------------------------------------------------------------------*/

static void 
  answerCB( Widget        widget,
            READ_REC_REF  read_ref,
            XtPointer     call_data )
{

  /* Code. */

  /* Initialize the answer window? */
  if( read_ref -> answer_handle == NULL ) {
    read_ref -> answer_handle = xtmMaInitialize( read_ref -> appl_data_ref,
                                                 read_ref -> readW );
  }

  if( read_ref -> answer_handle == NULL )
    return;


  /* Display the answer window. */
  xtmMaAnswerMessage( read_ref -> answer_handle, read_ref -> msg_id );


  return;

} /* answerCB */


/*----------------------------------------------------------------------*/

static void 
  closeCB( Widget        widget,
           READ_REC_REF  read_ref,
           XtPointer     call_data )
{

  /* Code. */

  XtUnmapWidget( XtParent( read_ref -> readW ) );


  return;

} /* closeCB */


/*----------------------------------------------------------------------*/

static void 
  destroyCB( Widget        widget,
             READ_REC_REF  read_ref,
             XtPointer     call_data )
{

  /* Code. */

  /* Release the user data. */
  SysFree( read_ref );


  return;

} /* destroyCB */


/*----------------------------------------------------------------------*/

static void 
  mapCB( Widget        widget,
         READ_REC_REF  read_ref,
         XtPointer     call_data )
{

  /* Variables. */
  Arg       args[ 3 ];
  Cardinal  n;


  /* Code. */

  /* No default position anymore. */
  n = 0;
  XtSetArg( args[ n ], XmNdefaultPosition, False ); n++;
  XtSetValues( read_ref -> readW, args, n );


  return;

} /* mapCB */


/*----------------------------------------------------------------------*/

static void 
  saveCB( Widget        widget,
          READ_REC_REF  read_ref,
          XtPointer     call_data )
{

  /* Variables. */
  char                    *message;
  char                    *text;
  TIM_TIME_REF            default_time;
  XTM_DB_MESSAGE_DEF      msg_info;
  XTM_DB_STATUS           db_status;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;
  XTM_CD_CAL_INFO         default_db;


  /* Code. */

  appl_data_ref   = read_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;

  /* Initialize a new editor record? */
  if( read_ref -> editor_handle == NULL )
    read_ref -> editor_handle = xtmEdInitialize( 
                                  appl_data_ref,
                                  appl_data_ref   -> toplevel,
                                  custom_data_ref -> default_entry_delta,
                                  custom_data_ref -> start_hour,
                                  custom_data_ref -> stop_hour,
                                  custom_data_ref -> default_entry_delta,
                                  saveEditorActionCB, 
                                  (void *) read_ref );

  default_time = TimMakeTime( 1970, 1, 1, 12, 0, 0 );

  (void) xtmCdFetchDefaultDb( custom_data_ref -> cal_db_handle,
                              &default_db, NULL );


  /* Edit a default entry. */
  xtmEdEditEntry( read_ref -> editor_handle,
                  default_db.short_name,
                  0, 
                  TimLocalTime( TimMakeTimeNow() ), 
                  default_time );


  /* Fetch the message. */
  db_status = xtmDbFetchMessage( default_db.directory, 
                                 read_ref -> msg_id,
                                 &msg_info, &message, &text );
  if( db_status != XTM_DB_OK )
    return;


  /* Assign the message data to the editor window. */
  xtmEdSetValues( read_ref -> editor_handle,
                  (XTM_ED_SET_DATE     | XTM_ED_SET_TIME | 
                   XTM_ED_SET_DURATION | XTM_ED_SET_TEXT),
                  (TIM_TIME_REF) msg_info.date,
                  (TIM_TIME_REF) msg_info.time,
                  (UINT32)       msg_info.duration,
                  text );


  /* Free allocated data. */
  if( message != NULL )
    SysFree( message );

  if( text != NULL )
    SysFree( text );


  return;

} /* saveCB */


/*----------------------------------------------------------------------*/

static void 
  saveEditorActionCB( XTM_ED_REASON reason,
                      void          *user_data )
{

  /* Variables. */
  READ_REC_REF  read_ref;


  /* Code. */

  read_ref = (READ_REC_REF) user_data;

  if( reason == XTM_ED_REASON_DESTROY )
    read_ref -> editor_handle = NULL;


  return;

} /* saveEditorActionCB */
