/*----------------------------------------------------------------------------
--
--  Module:           xtmCustArch
--
--  Project:          XDiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Module to handle the archive customization.
--
--  Filename:         xtmCustArch.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1991-09-14
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmCustArch.c, Version: 1.1, Date: 95/02/18 15:52:01";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <X11/Intrinsic.h>

#include <Xm/Xm.h>
#include <Xm/BulletinB.h>
#include <Xm/Form.h>
#include <Xm/Frame.h>
#include <Xm/List.h>
#include <Xm/RowColumn.h>
#include <Xm/Text.h>

#include "System.h"
#include "LstLinked.h"
#include "Message.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmArchive.h"
#include "xtmCustom.h"
#include "xtmHelp.h"
#include "xitError.h"
#include "xitTools.h"
#include "xtmCustArch.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Local widgets in the archive window. */
#define actionRc            dataLocalW[  0 ]
#define archLa              dataLocalW[  1 ]
#define archLi              dataLocalW[  2 ]
#define fileLa              dataLocalW[  3 ]
#define fileRc              dataLocalW[  4 ]
#define fileTx              dataLocalW[  5 ]
#define nameLa              dataLocalW[  6 ]
#define nameRc              dataLocalW[  7 ]
#define nameTx              dataLocalW[  8 ]


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmCustArchive";

/* IDs for the help windows. */
static char  *cust_arch_window_id = "CustArch";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void 
  archiveListCB( Widget                widget, 
                 XTM_CU_BASE_DATA_REF  appl_data_ref,
                 XmListCallbackStruct  *call_data );

static void
  addCB( Widget                widget,
         XTM_CU_BASE_DATA_REF  custom_data,
         XtPointer             call_data );

static void
  clearArchCB( void  *element );

static void
  deleteCB( Widget                widget,
            XTM_CU_BASE_DATA_REF  custom_data,
            XtPointer             call_data );

static void 
  helpCB( Widget                widget,
          XTM_CU_BASE_DATA_REF  appl_data_ref,
          XtPointer             call_data );

static void 
  okCB( Widget                widget,
        XTM_CU_BASE_DATA_REF  appl_data_ref,
        XtPointer             call_data );

static void
  setAction( Widget   actionW,
             Boolean  add_key,
             Boolean  delete_key );

static void
  setArchiveData( Widget                baseW,
                  XTM_CU_BASE_DATA_REF  appl_data_ref );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

void 
  xtmCrEditArchiveData( Widget                parent,
                        XTM_CU_BASE_DATA_REF  appl_data_ref )
{

  /* Variables. */
  int       index;
  Arg       args[ 10 ];
  Cardinal  n;
  Widget    archiveFd;
  Widget    dataFo;
  Widget    dataLocalW[ 9 ];
  Widget    actionBu[ 2 ];

  static XIT_PUSH_STRUCT action_def[] = {
    { "AddPb",     "", "", True, NULL },
    { "DeletePb",  "", "", True, NULL },
  };

  static XIT_TEXT_STRUCT text_buffer[] = {
    { "NameTx", NULL, 1, True },
    { "FileTx", NULL, 1, True },
  };

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "",   okCB,   NULL },
    { "",   NULL,   NULL },
    { "",   helpCB, NULL },
  };


  /* Code. */

  action_buttons[ 0 ].label = msgGetText( MXDI_OK_BUTTON );
  action_buttons[ 0 ].data  = appl_data_ref;
  action_buttons[ 1 ].label = msgGetText( MXDI_CANCEL_BUTTON );
  action_buttons[ 1 ].data  = appl_data_ref;
  action_buttons[ 2 ].label = msgGetText( MXDI_HELP_BUTTON );
  action_buttons[ 2 ].data  = appl_data_ref;

  action_def[ 0 ].title = msgGetText( MXDI_ADD_CHANGE_ITEM );
  action_def[ 1 ].title = msgGetText( MXDI_DELETE_ITEM );


  /* Create a form dialog with buttons. */
  archiveFd = xitCreateFormDialog( parent, "ArchiveFd",
                                   1, 0,
                                   action_buttons,
                                   XtNumber( action_buttons ) );

  n = 0;
  XtSetArg( args[ n ], XmNtitle, msgGetText( MXDI_CUST_ARCHIVE_TITLE ) ); n++;
  XtSetValues( XtParent( archiveFd ), args, n );

  n = 0;
  XtSetArg( args[ n ], XmNdialogStyle, XmDIALOG_APPLICATION_MODAL ); n++;
  XtSetValues( archiveFd, args, n );


  /* Container for the contents of the window. */
  dataFo = XtNameToWidget( archiveFd, "ArchiveFdFo" );


  archLa = xitCreateLabel( dataFo, "ArchLa", 
                           msgGetText( MXDI_ARCH_LOCATION_LABEL ), -1 );

  /* List with database location. */
  n = 0;
  XtSetArg( args[ n ], XmNlistSizePolicy,         XmCONSTANT ); n++;
  XtSetArg( args[ n ], XmNscrollBarDisplayPolicy, XmSTATIC ); n++;
  XtSetArg( args[ n ], XmNselectionPolicy,        XmSINGLE_SELECT ); n++;
  XtSetArg( args[ n ], XmNlistMarginHeight,       5 ); n++;
  XtSetArg( args[ n ], XmNlistMarginWidth,        5 ); n++;
  archLi = XmCreateScrolledList( dataFo, "ArchLi", args, n );

  XtAddCallback( archLi, XmNsingleSelectionCallback,
                 (XtCallbackProc) archiveListCB, (XtPointer) appl_data_ref );
  XtAddCallback( archLi, XmNdefaultActionCallback,
                 (XtCallbackProc) archiveListCB, (XtPointer) appl_data_ref );

  /* Name of archive file. */
  nameRc = XmCreateRowColumn( dataFo, "NameRc", args, 0 );
  nameLa = xitCreateLabel( nameRc, "NameLa", 
                           msgGetText( MXDI_ARCH_NAME_LABEL ), -1 );

  nameTx = xitCreateTextCols( nameRc, &text_buffer[ 0 ], 15 );

  /* Archive file name. */
  fileRc = XmCreateRowColumn( dataFo, "FileRc", args, 0 );

  fileLa = xitCreateLabel( fileRc, "FileLa", 
                           msgGetText( MXDI_ARCH_FILE_LABEL ), -1 );

  fileTx = xitCreateTextCols( fileRc, &text_buffer[ 1 ], 30 );


  /* Set max lengths. */
  n = 0;
  XtSetArg( args[ n ], XmNmaxLength, 15 );  n++;
  XtSetValues( nameTx, args, n );  

  n = 0;
  XtSetArg( args[ n ], XmNmaxLength, PATH_MAX );  n++;
  XtSetValues( fileTx, args, n );  


  /* Action (add and delete). */
  n = 0;
  XtSetArg( args[ n ], XmNorientation,  XmHORIZONTAL ); n++;
  XtSetArg( args[ n ], XmNspacing,      10 ); n++;
  XtSetArg( args[ n ], XmNmarginHeight, 10 ); n++;
  actionRc = XmCreateRowColumn( dataFo, "ActionRc", args, n );

  for( index = 0; index < XtNumber( actionBu ); index++ )
    actionBu[ index ] = xitCreatePushButton( actionRc, 
                                             &action_def[ index ] );

  XtAddCallback( actionBu[ 0 ], XmNactivateCallback, 
                 (XtCallbackProc) addCB, (XtPointer) appl_data_ref );
  XtAddCallback( actionBu[ 1 ], XmNactivateCallback, 
                 (XtCallbackProc) deleteCB, (XtPointer) appl_data_ref );


  /* Put the elements together. */
  xitAttachWidget( archLa,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( XtParent( archLi ),
                   XmATTACH_WIDGET, archLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,   XmATTACH_NONE, NULL );
  xitAttachWidget( nameRc,
                   XmATTACH_WIDGET, XtParent( archLi ), XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,               XmATTACH_NONE, NULL );
  xitAttachWidget( fileRc,
                   XmATTACH_WIDGET, nameRc, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,   XmATTACH_NONE, NULL );
  xitAttachWidget( actionRc,
                   XmATTACH_WIDGET, fileRc, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,   XmATTACH_NONE, NULL );

  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( archLa,             args, n );
  XtSetValues( XtParent( archLi ), args, n );
  XtSetValues( nameRc,             args, n );
  XtSetValues( fileRc,             args, n );
  XtSetValues( actionRc,           args, n );

  /* Manage the widgets. */
  XtManageChildren( actionBu,  XtNumber( actionBu ) );

  xitManageChildren( dataLocalW, XtNumber( dataLocalW ) );

  /* Set the size of the window. */
  xitSetSizeFormDialog( archiveFd, True );


  /* Make the final attachments. */
  xitAttachWidget( actionRc,
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL );
  xitAttachWidget( fileRc,
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_WIDGET, actionRc );
  xitAttachWidget( nameRc,
                   XmATTACH_NONE, NULL, XmATTACH_FORM,   NULL,
                   XmATTACH_NONE, NULL, XmATTACH_WIDGET, fileRc );
  xitAttachWidget( XtParent( archLi ),
                   XmATTACH_WIDGET, archLa, XmATTACH_FORM, NULL,
                   XmATTACH_FORM,   NULL,   XmATTACH_WIDGET, nameRc );


  XtManageChild( archiveFd );


  /* Put defined archive files in the list. */
  setArchiveData( archiveFd, appl_data_ref );
  setAction(      actionRc, True, False );


  return;

} /* xtmCrEditArchiveData */


/*----------------------------------------------------------------------*/

static void
  setAction( Widget   actionW,
             Boolean  add_key,
             Boolean  delete_key )
{

  /* Variables. */
  Widget  tempW;


  /* Code. */

  tempW = XtNameToWidget( actionW, "AddPb" );
  XtSetSensitive( tempW, add_key );

  tempW = XtNameToWidget( actionW, "DeletePb" );
  XtSetSensitive( tempW, delete_key );

  return;

} /* setAction */


/*----------------------------------------------------------------------*/

static void
  setArchiveData( Widget                baseW,
                  XTM_CU_BASE_DATA_REF  appl_data_ref )
{

  /* Variables. */
  int                     index;
  int                     index1;
  char                    buffer[ 100 ];
  Arg                     args[ 5 ];
  Cardinal                n;
  Widget                  mainW;
  Widget                  tempW;
  XmString                list_items[ 100 ];
  LST_STATUS              lst_status;
  XTM_AR_ARCHIVE_FILE     archive_file;
  XTM_GL_CUSTOM_DATA_REF  custom_data;


  /* Code. */

  custom_data = appl_data_ref -> custom_data;

  mainW = XtNameToWidget( baseW, "ArchiveFdFo" );

  /* Set values for the archive files. */
  if( LstLinkElements( custom_data -> archive_files ) > 0 ) {

    index = 0;

    lst_status = LstLinkCurrentFirst( custom_data -> archive_files );
    while( lst_status == LST_OK ) {

      lst_status = LstLinkGetCurrent( custom_data -> archive_files, 
                                      &archive_file );

      sprintf( buffer, "%-15.15s %s", 
               archive_file.short_name, archive_file.file_name );

      list_items[ index ] = XmStringCreate( buffer, CS );
      index++;

      /* Next entry. */
      lst_status = LstLinkCurrentNext( custom_data -> archive_files );

    } /* while */


    /* The list is always sorted. */
    xitSortStringList( list_items, index );

    tempW = XtNameToWidget( mainW, "ArchLiSW.ArchLi" );

    /* Assign the database locations to the list. */
    n = 0;
    XtSetArg( args[ n ], XmNitems, list_items ); n++;
    XtSetArg( args[ n ], XmNitemCount, index ); n++;
    XtSetArg( args[ n ], XmNselectedItemCount, 0 ); n++;
    XtSetValues( tempW, args, n );

    /* Free allocated memory. */
    for( index1 = 0; index1 < index; index1++ )
      XmStringFree( list_items[ index1 ] );

  } /* if */


  /* Empty the name and directory field. */
  tempW = XtNameToWidget( mainW, "NameRc.NameTx" );
  XmTextSetString( tempW, "" );

  tempW = XtNameToWidget( mainW, "FileRc.FileTx" );
  XmTextSetString( tempW, "" );

  return;

} /* setArchiveData */


/*----------------------------------------------------------------------*/

static void
  addCB( Widget                widget,
         XTM_CU_BASE_DATA_REF  appl_data_ref,
         XtPointer             call_data )
{

  /* Variables. */
  int       found_position;
  int       index;
  int       item_count;
  char      buffer[ PATH_MAX + XTM_GL_MAX_ARCH_NAME + 10 ];
  char      file_name[ PATH_MAX + 1 ];
  char      short_name[ XTM_GL_MAX_ARCH_NAME + 1 ];
  char      *buffer_ref;
  Arg       args[ 3 ];
  Cardinal  n;
  Widget    baseW;
  Widget    listW;
  Widget    mainW;
  Widget    tempW;
  XmString  *items;
  XmString  xstr;


  /* Code. */

  baseW = xitGetParentWidget( widget, "ArchiveFd" );
  mainW = XtNameToWidget(     baseW,  "ArchiveFdFo" );

  /* Fetch the short name. */
  tempW = XtNameToWidget( mainW, "NameRc.NameTx" );

  buffer_ref = xitStringGetText( tempW );

  item_count = sscanf( buffer_ref, "%s", short_name );
  SysFree( buffer_ref );

  if( item_count == 0 )
    return;


  /* Fetch the file name. */
  tempW = XtNameToWidget( mainW, "FileRc.FileTx" );

  buffer_ref = xitStringGetText( tempW );

  item_count = sscanf( buffer_ref, "%s", file_name );
  SysFree( buffer_ref );

  if( item_count == 0 )
    return;


  /* Add the short name/directory to the scrolled list. */
  sprintf( buffer, "%-15.15s %s", short_name, file_name );

  xstr = XmStringCreate( buffer, CS );

  /* Check if we have an archive with this name. */
  listW = XtNameToWidget( mainW, "ArchLiSW.ArchLi" );

  n = 0;
  XtSetArg( args[ n ], XmNitemCount, &item_count ); n++;
  XtSetArg( args[ n ], XmNitems, &items ); n++;
  XtGetValues( listW, args, n );

  found_position = 0;

  for( index = 0; index < item_count; index++ ) {
    buffer_ref = xitStringGetString( *(items + index), CS );

    sscanf( buffer_ref, "%s", buffer );

    SysFree( buffer_ref );

    if( strcmp( short_name, buffer ) == 0 ) {
      found_position = index + 1;
      break;
    }
  } /* loop */


  /* Does the entry already exist? */
  if( found_position > 0 )
    XmListDeletePos( listW, found_position );

  /* Add the new entry. */
  XmListAddItem( listW, xstr, 0 );
  XmListSetBottomPos( listW, 0 );
  XmListDeselectAllItems( listW );

  XmStringFree( xstr );  

  /* Remove name and filename. */
  tempW = XtNameToWidget( mainW, "NameRc.NameTx" );
  XmTextSetString( tempW, "" );

  tempW = XtNameToWidget( mainW, "FileRc.FileTx" );
  XmTextSetString( tempW, "" );

  /* No items are selected in the list. */
  tempW = XtNameToWidget( mainW, "ArchLiSW.ArchLi" );
  XmListDeselectAllItems( tempW );

  tempW = XtNameToWidget( mainW, "ActionRc" );
  setAction( tempW, True, False );


  return;

} /* addCB */


/*----------------------------------------------------------------------*/

static void 
  archiveListCB( Widget                widget, 
                 XTM_CU_BASE_DATA_REF  appl_data_ref,
                 XmListCallbackStruct  *call_data )
{

  /* Variables. */
  char    file_name[ PATH_MAX + 1 ];
  char    short_name[ XTM_GL_MAX_ARCH_NAME + 1 ];
  char    *buffer_ref;
  Widget  baseW;
  Widget  mainW;
  Widget  tempW;


  /* Code. */

  baseW = xitGetParentWidget( widget, "ArchiveFd" );
  mainW = XtNameToWidget(     baseW,  "ArchiveFdFo" );

  /* Fetch the selected name/file line. */
  buffer_ref = xitStringGetString( call_data -> item, CS );

  sscanf( buffer_ref, "%[^ ] %[^ ]", short_name, file_name );

  /* Set short name and directory. */
  tempW = XtNameToWidget( mainW, "NameRc.NameTx" );
  XmTextSetString( tempW, short_name );

  tempW = XtNameToWidget( mainW, "FileRc.FileTx" );
  XmTextSetString( tempW, file_name );

  SysFree( buffer_ref );


  tempW = XtNameToWidget( mainW, "ActionRc" );
  setAction( tempW, True, True );


  return;

} /* archiveListCB */


/*----------------------------------------------------------------------*/

static void
  clearArchCB( void  *element )
{

  /* Variables. */
  XTM_AR_ARCHIVE_FILE  *entry_ref;


  /* Code. */

  entry_ref = (XTM_AR_ARCHIVE_FILE *) element;

  if( entry_ref -> file_name != NULL )
    SysFree( entry_ref -> file_name );


  return;

} /* clearArchCB */


/*----------------------------------------------------------------------*/

static void
  deleteCB( Widget                widget,
            XTM_CU_BASE_DATA_REF  appl_data_ref,
            XtPointer             call_data )
{

  /* Variables. */
  int       selected_item_count;
  Arg       args[ 3 ];
  Cardinal  n;
  Widget    baseW;
  Widget    mainW;
  Widget    tempW;
  Widget    listW;
  XmString  *selected_items;


  /* Code. */

  baseW = xitGetParentWidget( widget, "ArchiveFd" );
  mainW = XtNameToWidget(     baseW,  "ArchiveFdFo" );
  listW = XtNameToWidget(     mainW, "ArchLiSW.ArchLi" );

  /* Fetch selected items from the list. */
  n = 0;
  XtSetArg( args[ n ], XmNselectedItemCount, &selected_item_count ); n++;
  XtSetArg( args[ n ], XmNselectedItems, &selected_items ); n++;
  XtGetValues( listW, args, n );

  if( selected_item_count != 1 )
    return;  

  /* Delete the selected short name/directory. */
  XmListDeleteItem( listW, *selected_items );

  /* Remove name and filename. */
  tempW = XtNameToWidget( mainW, "NameRc.NameTx" );
  XmTextSetString( tempW, "" );

  tempW = XtNameToWidget( mainW, "FileRc.FileTx" );
  XmTextSetString( tempW, "" );

  /* No items are selected in the list. */
  XmListDeselectAllItems( listW );

  tempW = XtNameToWidget( mainW, "ActionRc" );
  setAction( tempW, True, False );


  return;

} /* deleteCB */


/*----------------------------------------------------------------------*/

static void 
  helpCB( Widget                widget,
          XTM_CU_BASE_DATA_REF  appl_data_ref,
          XtPointer             call_data )
{

  /* Code. */

  xtmHlDisplayHelp( appl_data_ref -> info_handle,
                    XTM_HL_WINDOW_INDEX,
                    cust_arch_window_id, "" );

  return;

} /* helpCB */


/*----------------------------------------------------------------------*/

static void 
  okCB( Widget                widget,
        XTM_CU_BASE_DATA_REF  appl_data_ref,
        XtPointer             call_data )
{

  /* Variables. */
  int                     index;
  int                     item_count;
  char                    file_name[ PATH_MAX + 1 ];
  char                    *buffer_ref;
  Arg                     args[ 3 ];
  Cardinal                n;
  Widget                  baseW;
  Widget                  mainW;
  Widget                  listW;
  XmString                *items;
  LST_STATUS              lst_status;
  XTM_AR_ARCHIVE_FILE     archive_file;
  XTM_GL_CUSTOM_DATA_REF  custom_data;


  /* Code. */

  custom_data = appl_data_ref -> custom_data;

  baseW = xitGetParentWidget( widget, "ArchiveFd" );
  mainW = XtNameToWidget(     baseW,  "ArchiveFdFo" );


  /* Fetch the archive files. */
  listW = XtNameToWidget( mainW, "ArchLiSW.ArchLi" );

  n = 0;
  XtSetArg( args[ n ], XmNitemCount, &item_count ); n++;
  XtSetArg( args[ n ], XmNitems, &items ); n++;
  XtGetValues( listW, args, n );

  /* Free old database location data and create a new list. */
  LstLinkClearDataAndList( custom_data -> archive_files, clearArchCB );

  custom_data -> archive_files = LstLinkNew( sizeof( XTM_AR_ARCHIVE_FILE ), 
                                             NULL );

  /* Fetch all archive files and assign them to the database list. */
  for( index = 0; index < item_count; index++ ) {

    buffer_ref = xitStringGetString( *(items + index), CS );

    sscanf( buffer_ref, "%s %s", archive_file.short_name, file_name );

    archive_file.file_name = SysNewString( file_name );

    SysFree( buffer_ref );

    lst_status = LstLinkInsertLast( custom_data -> archive_files, 
                                    &archive_file );

  } /* loop */


  /* Remove the archive window. */
  XtDestroyWidget( baseW );

  return;

} /* okCB */
