/*
 * Copyright(c) 1992 Bell Communications Research, Inc. (Bellcore)
 * Copyright(c) 1995-99 Andrew Lister
 * Copyright  1999, 2000, 2001, 2002, 2003, 2004 by the LessTif Developers.
 *
 *                        All rights reserved
 * Permission to use, copy, modify and distribute this material for
 * any purpose and without fee is hereby granted, provided that the
 * above copyright notice and this permission notice appear in all
 * copies, and that the name of Bellcore not be used in advertising
 * or publicity pertaining to this material without the specific,
 * prior written permission of an authorized representative of
 * Bellcore.
 *
 * BELLCORE MAKES NO REPRESENTATIONS AND EXTENDS NO WARRANTIES, EX-
 * PRESS OR IMPLIED, WITH RESPECT TO THE SOFTWARE, INCLUDING, BUT
 * NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR ANY PARTICULAR PURPOSE, AND THE WARRANTY AGAINST IN-
 * FRINGEMENT OF PATENTS OR OTHER INTELLECTUAL PROPERTY RIGHTS.  THE
 * SOFTWARE IS PROVIDED "AS IS", AND IN NO EVENT SHALL BELLCORE OR
 * ANY OF ITS AFFILIATES BE LIABLE FOR ANY DAMAGES, INCLUDING ANY
 * LOST PROFITS OR OTHER INCIDENTAL OR CONSEQUENTIAL DAMAGES RELAT-
 * ING TO THE SOFTWARE.
 *
 * MatrixWidget Author: Andrew Wason, Bellcore, aw@bae.bellcore.com
 *
 * $Id: Public.c,v 1.59 2004/11/29 02:23:11 tobiasoed Exp $
 */

/*
 * Public.c created by Andrew Lister (7 August, 1995)
 */

#ifdef HAVE_CONFIG_H
#include <XbaeConfig.h>
#endif

#include <stdio.h>
#include <stdlib.h>


#include <Xm/Xm.h>
#include <Xm/ScrollBar.h>
#include <Xbae/MatrixP.h>
#include <Xbae/Shadow.h>
#include <Xbae/Draw.h>
#include <Xbae/ScrollMgr.h>
#include <Xbae/Actions.h>
#include <Xbae/Utils.h>
#include <Xbae/Clip.h>
#include <Xbae/Create.h>
#include <Xbae/Methods.h>

#include <XbaeDebug.h>


/*
 * Public interface to set_cell method
 */
void XbaeMatrixSetCell(Widget w, int row, int column, const String value)
{
        xbaeObjectLock(w);
        /*
         * Make sure w is a Matrix or a subclass
         */
        XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

        /*
         * Call the set_cell method
         */
        ((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.set_cell((XbaeMatrixWidget) w, row,
                                                                    column, value, True);

        xbaeObjectUnlock(w);
}


/*
 * Public interface to edit_cell method
 */
void XbaeMatrixEditCell(Widget w, int row, int column)
{
        xbaeObjectLock(w);
        /*
         * Make sure w is a Matrix or a subclass
         */
        XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

        /*
         * Call the edit_cell method
         */
        ((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.edit_cell((XbaeMatrixWidget) w, NULL,
                                                                     row, column, NULL, 0);

        xbaeObjectUnlock(w);
}

/*
 * Public interface to select_cell method
 */
void XbaeMatrixSelectCell(Widget w, int row, int column)
{
        xbaeObjectLock(w);
        /*
         * Make sure w is a Matrix or a subclass
         */
        XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

        /*
         * Call the select_cell method
         */
        ((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.select_cell((XbaeMatrixWidget) w,
                                                                       row, column);

        xbaeObjectUnlock(w);
}

/*
 * Public interface to select_row method
 */
void XbaeMatrixSelectRow(Widget w, int row)
{
        xbaeObjectLock(w);
        /*
         * Make sure w is a Matrix or a subclass
         */
        XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

        /*
         * Call the select_row method
         */
        ((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.select_row((XbaeMatrixWidget) w, row);

        xbaeObjectUnlock(w);
}

/*
 * Public interface to select_column method
 */
void XbaeMatrixSelectColumn(Widget w, int column)
{
        xbaeObjectLock(w);
        /*
         * Make sure w is a Matrix or a subclass
         */
        XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

        /*
         * Call the select_column method
         */
        ((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.select_column((XbaeMatrixWidget) w, column);

        xbaeObjectUnlock(w);
}

/*
 * Public interface to deselect_all method
 */
void XbaeMatrixDeselectAll(Widget w)
{
        xbaeObjectLock(w);
        /*
         * Make sure w is a Matrix or a subclass
         */
        XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

        /*
         * Call the deselect_all method
         */
        ((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.deselect_all((XbaeMatrixWidget) w);

        xbaeObjectUnlock(w);
}

/*
 * Public interface to select_all method
 */
void XbaeMatrixSelectAll(Widget w)
{
        xbaeObjectLock(w);
        /*
         * Make sure w is a Matrix or a subclass
         */
        XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

        /*
         * Call the deselect_all method
         */
        ((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.select_all((XbaeMatrixWidget) w);

        xbaeObjectUnlock(w);
}

/*
 * Public interface to deselect_cell method
 */
void XbaeMatrixDeselectCell(Widget w, int row, int column)
{
        xbaeObjectLock(w);
        /*
         * Make sure w is a Matrix or a subclass
         */
        XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

        /*
         * Call the deselect_cell method
         */
        ((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.deselect_cell((XbaeMatrixWidget) w,
                                                                          row, column);

        xbaeObjectUnlock(w);
}

/*
 * Public interface to deselect_row method
 */
void XbaeMatrixDeselectRow(Widget w, int row)
{
        xbaeObjectLock(w);
        /*
         * Make sure w is a Matrix or a subclass
         */
        XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

        /*
         * Call the deselect_row method
         */
        ((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.deselect_row((XbaeMatrixWidget) w, row);

        xbaeObjectUnlock(w);
}

/*
 * Public interface to deselect_column method
 */
void XbaeMatrixDeselectColumn(Widget w, int column)
{
        xbaeObjectLock(w);
        /*
         * Make sure w is a Matrix or a subclass
         */
        XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

        /*
         * Call the deselect_column method
         */
        ((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.deselect_column((XbaeMatrixWidget) w, column);

        xbaeObjectUnlock(w);
}

void XbaeMatrixUnderlineCell(Widget w, int row, int column)
{
        XbaeMatrixWidget mw = (XbaeMatrixWidget) w;

        if (row >= mw->matrix.rows || row < 0 || column >= mw->matrix.columns || column < 0) {
                XtAppWarningMsg(XtWidgetToApplicationContext(w), "underlineCell", "badIndex",
                                "XbaeMatrix",
                                "XbaeMatrix: Row or column out of bounds for UnderlineCell.", NULL,
                                0);
                return;
        }

        /* If no cells have been underlined or deunderlined yet, allocate memory */
        if (!mw->matrix.per_cell)
                xbaeCreatePerCell(mw);

        /*
         * If the cell is not already underlined, underline it and redraw it
         */
        if (!mw->matrix.per_cell[row][column].underlined) {
                mw->matrix.per_cell[row][column].underlined = True;
                if (xbaeIsCellVisible(mw, row, column)) {
                        xbaeDrawCell(mw, row, column);
                }
        }
}

void XbaeMatrixUnderlineRow(Widget w, int row)
{
        XbaeMatrixWidget mw = (XbaeMatrixWidget) w;

        int j, lc, rc;

        if (row >= mw->matrix.rows || row < 0) {
                XtAppWarningMsg(XtWidgetToApplicationContext(w), "underlineRow", "badIndex",
                                "XbaeMatrix", "XbaeMatrix: Row out of bounds for UnderlineRow.",
                                NULL, 0);
                return;
        }

        /* If no cells have been underlined or deunderlined yet, allocate memory */
        if (!mw->matrix.per_cell)
                xbaeCreatePerCell(mw);

        /*
         * If the row is not visible, there's no need to redraw - but, we do
         * need to update the underlined cell resource
         */
        if (!xbaeIsRowVisible(mw, row)) {
                for (j = 0; j < mw->matrix.columns; j++)
                        if (!mw->matrix.per_cell[row][j].underlined) {
                                mw->matrix.per_cell[row][j].underlined = True;
                        }
                return;
        }

        /*
         * For each cell in the row, if the cell is not already underlined,
         * underline it and redraw it
         */
        xbaeGetVisibleColumns(mw, &lc, &rc);
        for (j = 0; j < mw->matrix.columns; j++) {
                if (!mw->matrix.per_cell[row][j].underlined) {
                        mw->matrix.per_cell[row][j].underlined = True;
                        if ((j >= lc && j <= rc) || IS_FIXED_COLUMN(mw, j)) {
                                xbaeDrawCell(mw, row, j);
                        }
                }
        }
}

void XbaeMatrixUnderlineColumn(Widget w, int column)
{
        XbaeMatrixWidget mw = (XbaeMatrixWidget) w;

        int i, tr, br;

        if (column >= mw->matrix.columns || column < 0) {
                XtAppWarningMsg(XtWidgetToApplicationContext(w), "underlineColumn", "badIndex",
                                "XbaeMatrix",
                                "XbaeMatrix: Column out of bounds for UnderlineColumn.", NULL, 0);
                return;
        }

        /* If no cells have been underlined or deunderlined yet, allocate memory */
        if (!mw->matrix.per_cell)
                xbaeCreatePerCell(mw);

        /*
         * No need to redraw unless the column is visible
         */
        if (!xbaeIsColumnVisible(mw, column)) {
                for (i = 0; i < mw->matrix.rows; i++)
                        if (!mw->matrix.per_cell[i][column].underlined) {
                                mw->matrix.per_cell[i][column].underlined = True;
                        }
                return;
        }

        /*
         * For each cell in the column, if the cell is not already underlined,
         * underline it and redraw it
         */
        xbaeGetVisibleRows(mw, &tr, &br);
        for (i = 0; i < mw->matrix.rows; i++) {
                if (!mw->matrix.per_cell[i][column].underlined) {
                        mw->matrix.per_cell[i][column].underlined = True;
                        if ((i >= tr && i <= br) || IS_FIXED_ROW(mw, i)) {
                                xbaeDrawCell(mw, i, column);
                        }
                }
        }
}

void XbaeMatrixDeunderlineCell(Widget w, int row, int column)
{
        XbaeMatrixWidget mw = (XbaeMatrixWidget) w;

        if (row >= mw->matrix.rows || row < 0 || column >= mw->matrix.columns || column < 0) {
                XtAppWarningMsg(XtWidgetToApplicationContext(w), "deunderlineCell", "badIndex",
                                "XbaeMatrix",
                                "XbaeMatrix: Row or column out of bounds for DeunderlineCell.",
                                NULL, 0);
                return;
        }

        if (!mw->matrix.per_cell)
                return;

        /*
         * If the cell is already underlined, deunderline it and redraw it
         */
        if (mw->matrix.per_cell[row][column].underlined) {
                mw->matrix.per_cell[row][column].underlined = False;
                if (xbaeIsCellVisible(mw, row, column)) {
                        xbaeDrawCell(mw, row, column);
                }
        }
}

void XbaeMatrixDeunderlineRow(Widget w, int row)
{
        XbaeMatrixWidget mw = (XbaeMatrixWidget) w;

        int j, lc, rc;

        if (row >= mw->matrix.rows || row < 0) {
                XtAppWarningMsg(XtWidgetToApplicationContext(w), "deunderlineRow", "badIndex",
                                "XbaeMatrix", "XbaeMatrix: Row out of bounds for DeunderlineRow.",
                                NULL, 0);
                return;
        }

        if (!mw->matrix.per_cell)
                return;

        /*
         * If the row is not visible, there's no need to redraw - but, we do
         * need to update the underlined cell resource
         */
        if (!xbaeIsRowVisible(mw, row)) {
                for (j = 0; j < mw->matrix.columns; j++)
                        if (mw->matrix.per_cell[row][j].underlined) {
                                mw->matrix.per_cell[row][j].underlined = False;
                        }
                return;
        }

        /*
         * For each cell in the row, if the cell is not already underlined,
         * underline it and redraw it
         */
        xbaeGetVisibleColumns(mw, &lc, &rc);
        for (j = 0; j < mw->matrix.columns; j++) {
                if (mw->matrix.per_cell[row][j].underlined) {
                        mw->matrix.per_cell[row][j].underlined = False;
                        if ((j >= lc && j <= rc) || IS_FIXED_COLUMN(mw, j)) {
                                xbaeDrawCell(mw, row, j);
                        }
                }
        }
}

void XbaeMatrixDeunderlineColumn(Widget w, int column)
{
        XbaeMatrixWidget mw = (XbaeMatrixWidget) w;

        int i, tr, br;

        if (column >= mw->matrix.columns || column < 0) {
                XtAppWarningMsg(XtWidgetToApplicationContext(w), "deunderlineColumn", "badIndex",
                                "XbaeMatrix",
                                "XbaeMatrix: Column out of bounds for DeunderlineColumn.", NULL, 0);
                return;
        }

        if (!mw->matrix.per_cell)
                return;

        /*
         * No need to redraw unless the column is visible
         */
        if (!xbaeIsColumnVisible(mw, column)) {
                for (i = 0; i < mw->matrix.rows; i++)
                        if (mw->matrix.per_cell[i][column].underlined) {
                                mw->matrix.per_cell[i][column].underlined = False;
                        }
                return;
        }

        /*
         * For each cell in the column, if the cell is already underlined,
         * deunderline it and redraw it
         */
        xbaeGetVisibleRows(mw, &tr, &br);
        for (i = 0; i < mw->matrix.rows; i++) {
                if (mw->matrix.per_cell[i][column].underlined) {
                        mw->matrix.per_cell[i][column].underlined = False;
                        if ((i >= tr && i <= br) || IS_FIXED_ROW(mw, i)) {
                                xbaeDrawCell(mw, i, column);
                        }
                }
        }
}

/*
 * Public interface to get_cell method
 */
String XbaeMatrixGetCell(Widget w, int row, int column)
{
        String s;

        xbaeObjectLock(w);
        /*
         * Make sure w is a Matrix or a subclass
         */
        XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

        /*
         * Call the get_cell method
         */
        s = ((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.get_cell((XbaeMatrixWidget) w,
                                                                        row, column);

        xbaeObjectUnlock(w);

        return s;
}

/*
 * Public interface to commit_edit method
 */
Boolean XbaeMatrixCommitEdit(Widget w, Boolean unmap)
{
        Boolean b;

        xbaeObjectLock(w);
        /*
         * Make sure w is a Matrix or a subclass
         */
        XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

        /*
         * Call the commit_edit method
         */
        b = ((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.commit_edit((XbaeMatrixWidget) w,
                                                                           NULL, unmap);

        xbaeObjectUnlock(w);

        return b;
}

/*
 * Public interface to cancel_edit method
 */
void XbaeMatrixCancelEdit(Widget w, Boolean unmap)
{
        xbaeObjectLock(w);
        /*
         * Make sure w is a Matrix or a subclass
         */
        XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

        /*
         * Call the cancel_edit method
         */
        ((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.cancel_edit((XbaeMatrixWidget) w,
                                                                       unmap);

        xbaeObjectUnlock(w);
}

/*
 * Public interface to add_rows method
 * ONLY use this method for rows with same heights
 */
void
XbaeMatrixAddRows(Widget w, int position, String * rows, String * labels, Pixel * colors,
                  int num_rows)
{
        xbaeObjectLock(w);

        /*
         * Make sure w is a Matrix or a subclass
         */
        XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

        /*
         * Call the add_rows method
         */
        ((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.add_rows((XbaeMatrixWidget) w,
                                                                    position, rows, labels,
                                                                    colors, NULL, num_rows);

        xbaeObjectUnlock(w);
}

/*
 * Public interface to add_rows method
 * This method should be used to support flexible height rows.
 */
void
XbaeMatrixAddVarRows(Widget w, int position, String * rows, String * labels, short *heights,
                     int *max_heights, unsigned char *alignments, unsigned char *label_alignments,
                     Pixel * colors, int num_rows)
{
        xbaeObjectLock(w);

        /*
         * Make sure w is a Matrix or a subclass
         */
        XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

        /*
         * Call the add_rows method
         */
        ((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.add_var_rows((XbaeMatrixWidget) w,
                                                                        position, rows, labels,
                                                                        heights, max_heights,
                                                                        alignments,
                                                                        label_alignments,
                                                                        colors, NULL, num_rows);

        xbaeObjectUnlock(w);
}

/*
 * Public interface to delete_rows method
 */
void XbaeMatrixDeleteRows(Widget w, int position, int num_rows)
{
        xbaeObjectLock(w);
        /*
         * Make sure w is a Matrix or a subclass
         */
        XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

        /*
         * Call the delete_rows method
         */
        ((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.delete_rows((XbaeMatrixWidget) w,
                                                                       position, num_rows);

        xbaeObjectUnlock(w);
}

/*
 * Public interface to add_columns method
 */
void
XbaeMatrixAddColumns(Widget w, int position, String * columns, String * labels, short *widths,
                     int *max_lengths, unsigned char *alignments, unsigned char *label_alignments,
                     Pixel * colors, int num_columns)
{
        xbaeObjectLock(w);
        /*
         * Make sure w is a Matrix or a subclass
         */
        XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

        /*
         * Call the add_columns method
         */
        ((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.add_columns((XbaeMatrixWidget) w,
                                                                       position, columns,
                                                                       labels, widths,
                                                                       max_lengths, alignments,
                                                                       label_alignments, colors,
                                                                       NULL, num_columns);

        xbaeObjectUnlock(w);
}

/*
 * Public interface to delete_columns method
 */
void XbaeMatrixDeleteColumns(Widget w, int position, int num_columns)
{
        xbaeObjectLock(w);
        /*
         * Make sure w is a Matrix or a subclass
         */
        XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

        /*
         * Call the delete_columns method
         */
        ((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.delete_columns((XbaeMatrixWidget) w,
                                                                          position,
                                                                          num_columns);

        xbaeObjectUnlock(w);
}

/*
 * Public interface to set_row_colors method
 */
void XbaeMatrixSetRowColors(Widget w, int position, Pixel * colors, int num_colors)
{
        xbaeObjectLock(w);
        /*
         * Make sure w is a Matrix or a subclass
         */
        XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

        /*
         * Call the set_row_colors method
         */
        ((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.set_row_colors((XbaeMatrixWidget) w,
                                                                          position, colors,
                                                                          num_colors, False);

        xbaeObjectUnlock(w);
}

/*
 * Public interface to set_column_colors method
 */
void XbaeMatrixSetColumnColors(Widget w, int position, Pixel * colors, int num_colors)
{
        xbaeObjectLock(w);
        /*
         * Make sure w is a Matrix or a subclass
         */
        XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

        /*
         * Call the set_column_colors method
         */
        ((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.set_column_colors((XbaeMatrixWidget) w, 
                                                                             position, colors, 
                                                                             num_colors, False);

        xbaeObjectUnlock(w);
}

/*
 * Public interface to set_cell_color method
 */
void XbaeMatrixSetCellColor(Widget w, int row, int column, Pixel color)
{
        xbaeObjectLock(w);
        /*
         * Make sure w is a Matrix or a subclass
         */
        XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

        /*
         * Call the set_cell_color method
         */
        ((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.set_cell_color((XbaeMatrixWidget) w,
                                                                          row, column, color,
                                                                          False);

        xbaeObjectUnlock(w);
}

/*
 * Public interface to set_row_colors method
 */
void XbaeMatrixSetRowBackgrounds(Widget w, int position, Pixel * colors, int num_colors)
{
        xbaeObjectLock(w);
        /*
         * Make sure w is a Matrix or a subclass
         */
        XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

        /*
         * Call the set_row_colors method
         */
        ((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.set_row_colors((XbaeMatrixWidget) w,
                                                                          position, colors,
                                                                          num_colors, True);

        xbaeObjectUnlock(w);
}

/*
 * Public interface to set_column_colors method
 */
void XbaeMatrixSetColumnBackgrounds(Widget w, int position, Pixel * colors, int num_colors)
{
        xbaeObjectLock(w);
        /*
         * Make sure w is a Matrix or a subclass
         */
        XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

        /*
         * Call the set_column_colors method
         */
        ((XbaeMatrixWidgetClass) XtClass(w))->matrix_class. set_column_colors((XbaeMatrixWidget) w,
                                                                              position, colors, 
                                                                              num_colors, True);

        xbaeObjectUnlock(w);
}

/*
 * Public interface to set_cell_color method
 */
void XbaeMatrixSetCellBackground(Widget w, int row, int column, Pixel color)
{
        xbaeObjectLock(w);
        /*
         * Make sure w is a Matrix or a subclass
         */
        XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

        /*
         * Call the set_cell_color method
         */
        ((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.set_cell_color((XbaeMatrixWidget) w,
                                                                          row, column, color,
                                                                          True);

        xbaeObjectUnlock(w);
}

/*
 * Help the user know what row & column he is in given an x & y (via an event).
 * Return True on success, False on failure.
 */
int XbaeMatrixGetEventRowColumn(Widget w, XEvent * event, int *row, int *column)
{
        XbaeMatrixWidget mw;
        int x, y;

        xbaeObjectLock(w);

        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return False;
        }

        /* Convert the event to the correct XY for the matrix widget. */
        mw = (XbaeMatrixWidget) w;
        if (!xbaeEventToMatrixXY(mw, event, &x, &y)) {
                xbaeObjectUnlock(w);
                return False;
        }

        /* Convert the point to a row,column. If it does not pick a valid cell,
           then return. */
        if (!xbaeMatrixXYToRowCol(mw, &x, &y, row, column)) {
                xbaeObjectUnlock(w);
                return False;
        }

        xbaeObjectUnlock(w);
        return True;
}

/*
 * Public interface for xbaeEventToXY()
 */
Boolean XbaeMatrixEventToXY(Widget w, XEvent * event, int *x, int *y)
{
        XbaeMatrixWidget mw;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return False;
        }

        mw = (XbaeMatrixWidget) w;

        if (!xbaeEventToMatrixXY(mw, event, x, y)) {
                xbaeObjectUnlock(w);
                return False;
        }

        /* Tobias: FIXME: Using xbaeEventToMatrixXY instead of the old xbaeEventToXY
         * changes this functions semantics and may break 3rd party code. 
         * I'm not sure if it's worth fixing as the old semantics are unusable (cell isn't returned)
         */

        xbaeObjectUnlock(w);
        return True;
}

/*
 * Public interface for xbaeRowColToXY().  From Philip Aston
 * (philipa@parallax.co.uk)
 */
Boolean XbaeMatrixRowColToXY(Widget w, int row, int column, int *x, int *y)
{
        XbaeMatrixWidget mw;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return False;
        }

        mw = (XbaeMatrixWidget) w;

        xbaeRowColToClipXY(mw, row, column, x, y);

        xbaeObjectUnlock(w);
        return True;
}

/*
 * Help the programmer to know what row & column we are currently at.
 * Set the row & column to -1 if bad widget.  Maybe the program will core. :)
 */
void XbaeMatrixGetCurrentCell(Widget w, int *row, int *column)
{
        XbaeMatrixWidget mw;

        if (!XtIsSubclass(w, xbaeMatrixWidgetClass))
                *row = *column = -1;
        else {
                mw = (XbaeMatrixWidget) w;

                /* 
                 * Make sure we return something valid. 
                 * At least Grace depends on it... 
                 */

                if (mw->matrix.current_row >= 0 && mw->matrix.current_row < mw->matrix.rows) {
                        *row = mw->matrix.current_row;
                } else {
                        *row = xbaeTopRow(mw);
                }
                
                if (mw->matrix.current_column >= 0 && mw->matrix.current_column < mw->matrix.columns) {
                        *column = mw->matrix.current_column;
                } else {
                        *column = xbaeLeftColumn(mw);
                }
        }
        xbaeObjectUnlock(w);
}

/*
 * Set current cell
 */
void XbaeMatrixSetCurrentCellPosition(Widget w, int current_row, int current_column)
{
        XbaeMatrixWidget mw = (XbaeMatrixWidget) w;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return;
        }
        if (current_row < 0 || current_column < 0 || current_row >= mw->matrix.rows
            || current_column >= mw->matrix.columns) {
                xbaeObjectUnlock(w);
                return;
        }

        mw->matrix.current_row = current_row;
        mw->matrix.current_column = current_column;

        xbaeObjectUnlock(w);
}


/*
 * Allow the programmer to call the Expose method directly if he feels
 * that it is really needed.
 */
void XbaeMatrixRefresh(Widget w)
{

        XbaeMatrixWidget mw = (XbaeMatrixWidget) w;
        int x, y;

        xbaeObjectLock(w);
        if (mw->matrix.disable_redisplay) {
                xbaeObjectUnlock(w);
                return;
        }

        if (!XtIsSubclass(w, xbaeMatrixWidgetClass) || !XtIsRealized((Widget) mw)) {
                xbaeObjectUnlock(w);
                return;
        }
        
        /*
         * Don't respond to exposures.
         */
        mw->matrix.disable_redisplay = 1;

        /*
         * Relayout.
         */
        xbaeRelayout(mw);

        /*
         * Flush pending expose events.
         */
        XmUpdateDisplay(w);

        /*
         * Respond to exposures.
         */
        mw->matrix.disable_redisplay = 0;

        #if 0
        /*
         * Generate expose events on the Matrix to force the redrawing.
         */
        x = 0;
        y = 0;
        if (mw->matrix.column_labels)
                XClearArea(XtDisplay(mw), XtWindow(mw), x, y, mw->core.width, FIXED_ROW_POSITION(mw),
                           True);
        y += FIXED_ROW_POSITION(mw);
        if (mw->matrix.row_labels)
                XClearArea(XtDisplay(mw), XtWindow(mw), x, y, FIXED_COLUMN_POSITION(mw),
                           mw->core.height - x, True);
        x += FIXED_COLUMN_POSITION(mw);
        if (mw->matrix.fixed_rows) {
                /* Clear the top clip completely */
                XClearArea(XtDisplay(mw), XtWindow(TopClip(mw)), 0, 0, 0, 0, True);

                /* Don't forget the corner areas! */
                if (mw->matrix.fixed_columns)
                        XClearArea(XtDisplay(mw), XtWindow(mw), x, y, LeftClip(mw)->core.width,
                                   TopClip(mw)->core.height, True);
                if (mw->matrix.trailing_fixed_columns)
                        XClearArea(XtDisplay(mw), XtWindow(mw), RightClip(mw)->core.x, y,
                                   RightClip(mw)->core.width, TopClip(mw)->core.height, True);
        }
        if (mw->matrix.fixed_columns)
                XClearArea(XtDisplay(mw), XtWindow(LeftClip(mw)), 0, 0, 0, 0, True);
        if (mw->matrix.trailing_fixed_columns)
                XClearArea(XtDisplay(mw), XtWindow(RightClip(mw)), 0, 0, 0, 0, True);
        if (mw->matrix.trailing_fixed_rows) {
                XClearArea(XtDisplay(mw), XtWindow(BottomClip(mw)), 0, 0, 0, 0, True);
                if (mw->matrix.fixed_columns)
                        XClearArea(XtDisplay(mw), XtWindow(mw), x, BottomClip(mw)->core.y,
                                   LeftClip(mw)->core.width, BottomClip(mw)->core.height, True);
                if (mw->matrix.trailing_fixed_columns)
                        XClearArea(XtDisplay(mw), XtWindow(mw), RightClip(mw)->core.x,
                                   BottomClip(mw)->core.y, RightClip(mw)->core.width,
                                   BottomClip(mw)->core.height, True);
        }
        /*
         * The areas to the right and bottom of the matrix also need to be
         * exposed. First to do is the right hand side.
         */
        /* SGO fix if columns = 0 */
        /* Tobias: FIXME this calculation is bogous */
        if (mw->matrix.columns) {
                x = COLUMN_POSITION(mw, mw->matrix.columns - 1) + COLUMN_WIDTH(mw,
                                                                               mw->matrix.columns -
                                                                               1);
        } else
                x = 0;
        XClearArea(XtDisplay(mw), XtWindow(mw), x, 0, mw->core.width - x, mw->core.height, True);
        y = ClipChild(mw)->core.y + ClipChild(mw)->core.height + TRAILING_FIXED_ROW_HEIGHT(mw);
        XClearArea(XtDisplay(mw), XtWindow(mw), 0, y, mw->core.width, mw->core.height - y, True);
        XbaeClipRedraw(ClipChild(mw));
        #else
        XClearArea(XtDisplay(mw), XtWindow(mw), 0, 0, 0, 0, True);

        if (XtIsManaged(TopClip(mw)))
                XClearArea(XtDisplay(mw), XtWindow(TopClip(mw)), 0, 0, 0, 0, True);
        if (XtIsManaged(LeftClip(mw)))
                XClearArea(XtDisplay(mw), XtWindow(LeftClip(mw)), 0, 0, 0, 0, True);
        if (XtIsManaged(RightClip(mw)))
                XClearArea(XtDisplay(mw), XtWindow(RightClip(mw)), 0, 0, 0, 0, True);
        if (XtIsManaged(BottomClip(mw)))
                XClearArea(XtDisplay(mw), XtWindow(BottomClip(mw)), 0, 0, 0, 0, True);
        if (XtIsManaged(RowLabelClip(mw))) 
                XClearArea(XtDisplay(mw), XtWindow(RowLabelClip(mw)), 0, 0, 0, 0, True);
        if (XtIsManaged(ColumnLabelClip(mw))) 
                XClearArea(XtDisplay(mw), XtWindow(ColumnLabelClip(mw)), 0, 0, 0, 0, True);
        
        XbaeClipRedraw(ClipChild(mw));
        #endif
        
        xbaeObjectUnlock(w);
}

/*
 * Public interface for redrawing one cell
 */
void XbaeMatrixRefreshCell(Widget w, int row, int column)
{
        XbaeMatrixWidget mw = (XbaeMatrixWidget) w;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return;
        }

        if (xbaeIsCellVisible(mw, row, column)) {
                xbaeDrawCell(mw, row, column);
        }

        xbaeObjectUnlock(w);
}

/*
 * Redraw an entire column
 */
void XbaeMatrixRefreshColumn(Widget w, int column)
{
        XbaeMatrixWidget mw = (XbaeMatrixWidget) w;
        int row;
        int found = 0;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return;
        }

        /* we attempt to be mildly efficient about this */
        if (xbaeIsColumnVisible(mw, column)) {
                /* fixed always visible */
                for (row = 0; row < mw->matrix.fixed_rows; row++)
                        xbaeDrawCell(mw, row, column);
                /* now the scrollable clip */
                for (; row < mw->matrix.rows - mw->matrix.trailing_fixed_rows; row++)
                        if (xbaeIsRowVisible(mw, row)) {
                                found = 1;
                                xbaeDrawCell(mw, row, column);
                        } else if (found)
                                break;  /* came to the end of the clip */
                /* and finally trailing fixed are always visible */
                for (row = mw->matrix.rows - mw->matrix.trailing_fixed_rows; row < mw->matrix.rows;
                     row++)
                        xbaeDrawCell(mw, row, column);
        }

        xbaeObjectUnlock(w);
}

/*
 * Redraw an entire row
 */
void XbaeMatrixRefreshRow(Widget w, int row)
{
        XbaeMatrixWidget mw = (XbaeMatrixWidget) w;
        int column;
        int found = 0;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return;
        }

        /* we attempt to be mildly efficient about this */
        if (xbaeIsRowVisible(mw, row)) {
                /* fixed always visible */
                for (column = 0; column < mw->matrix.fixed_columns; column++)
                        xbaeDrawCell(mw, row, column);
                /* now the scrollable clip */
                for (; column < mw->matrix.columns - mw->matrix.trailing_fixed_columns; column++)
                        if (xbaeIsColumnVisible(mw, column)) {
                                found = 1;
                                xbaeDrawCell(mw, row, column);
                        } else if (found)
                                break;  /* came to the end of the clip */
                /* and finally trailing fixed are always visible */
                for (column = mw->matrix.columns - mw->matrix.trailing_fixed_columns;
                     column < mw->matrix.columns; column++)
                        xbaeDrawCell(mw, row, column);
        }

        xbaeObjectUnlock(w);
}

/*
 *  XbaeMatrixVisibleRows()
 *
 *  This routine returns the number of rows that are visible in the matrix.
 */
int XbaeMatrixVisibleRows(Widget w)
{
        XbaeMatrixWidget matrix = (XbaeMatrixWidget) w;

        int top_row;
        int bottom_row;

        xbaeObjectLock(w);
        xbaeGetVisibleRows(matrix, &top_row, &bottom_row);
        xbaeObjectUnlock(w);

        return bottom_row - top_row + 1;

}                               /* XbaeMatrixVisibleRows */

/*
 *  XbaeMatrixVisibleColumns()
 *
 *  This routine returns the number of columns that are visible in the matrix.
 */
int XbaeMatrixVisibleColumns(Widget w)
{
        XbaeMatrixWidget matrix = (XbaeMatrixWidget) w;

        int left_column;
        int right_column;

        xbaeObjectLock(w);
        xbaeGetVisibleColumns(matrix, &left_column, &right_column);
        xbaeObjectUnlock(w);

        return right_column - left_column + 1;

}                               /* XbaeMatrixVisibleColumns */

/*
 * Get per-cell user data
 */
XtPointer XbaeMatrixGetCellUserData(Widget w, int row, int column)
{
        XbaeMatrixWidget mw;
        XtPointer data;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return NULL;
        }

        mw = (XbaeMatrixWidget) w;
        if (mw->matrix.per_cell)
                data = mw->matrix.per_cell[row][column].user_data;
        else {
                xbaeObjectUnlock(w);
                return NULL;
        }

        xbaeObjectUnlock(w);

        return data;
}

/*
 * Set per-cell user data
 */
void XbaeMatrixSetCellUserData(Widget w, int row, int column, XtPointer data)
{
        XbaeMatrixWidget mw;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return;
        }

        mw = (XbaeMatrixWidget) w;
        if (row < 0 || column < 0 || row >= mw->matrix.rows || column >= mw->matrix.columns) {
                xbaeObjectUnlock(w);
                return;
        }

        if (!mw->matrix.per_cell)
                xbaeCreatePerCell(mw);
        mw->matrix.per_cell[row][column].user_data = data;

        xbaeObjectUnlock(w);
        return;
}


/* 
 * ARCAD SYSTEMHAUS
 */
int XbaeMatrixGetCellPixmap(Widget w, int row, int column, Pixmap * pixmap, Pixmap * mask)
{
        XbaeMatrixWidget mw = (XbaeMatrixWidget) w;
        xbaeObjectLock(w);
        if (! XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return -1;
        }
        if (row < 0 || column < 0 || row >= mw->matrix.rows || column >= mw->matrix.columns) {
                xbaeObjectUnlock(w);
                return -1;
        }

        if (mw->matrix.per_cell) {
                *pixmap = mw->matrix.per_cell[row][column].pixmap;
                *mask = mw->matrix.per_cell[row][column].mask;
                xbaeObjectUnlock(w);
                return 0;
        }
        xbaeObjectUnlock(w);
        return -1;
}

void XbaeMatrixSetCellPixmap(Widget w, int row, int column, Pixmap pixmap, Pixmap mask)
{
        xbaeObjectLock(w);
        if (XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                XbaeMatrixWidget mw = (XbaeMatrixWidget) w;
                if (row >= 0 && column >= 0 && row < mw->matrix.rows && column < mw->matrix.columns) {
                        if (!mw->matrix.per_cell)
                                xbaeCreatePerCell(mw);
                        mw->matrix.per_cell[row][column].pixmap = pixmap;
                        mw->matrix.per_cell[row][column].mask = mask;
                }
        }
        xbaeObjectUnlock(w);
}

/*
 * Get per-row user data
 */
XtPointer XbaeMatrixGetRowUserData(Widget w, int row)
{
        XbaeMatrixWidget mw;
        XtPointer data;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return NULL;
        }

        mw = (XbaeMatrixWidget) w;
        if (mw->matrix.row_user_data)
                data = mw->matrix.row_user_data[row];
        else {
                xbaeObjectUnlock(w);
                return NULL;
        }

        xbaeObjectUnlock(w);

        return data;
}


/*
 * Set per-row user data
 */
void XbaeMatrixSetRowUserData(Widget w, int row, XtPointer data)
{
        XbaeMatrixWidget mw;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return;
        }

        mw = (XbaeMatrixWidget) w;
        if (!mw->matrix.row_user_data) {
                XtPointer *copy;

                copy = (XtPointer *) XtCalloc(mw->matrix.rows, sizeof(XtPointer));

                mw->matrix.row_user_data = copy;
        }

        mw->matrix.row_user_data[row] = data;

        xbaeObjectUnlock(w);
        return;
}

/*
 * Get per-column user data
 */
XtPointer XbaeMatrixGetColumnUserData(Widget w, int column)
{
        XbaeMatrixWidget mw;
        XtPointer data;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return NULL;
        }

        mw = (XbaeMatrixWidget) w;
        if (mw->matrix.column_user_data)
                data = mw->matrix.column_user_data[column];
        else {
                xbaeObjectUnlock(w);
                return NULL;
        }

        xbaeObjectUnlock(w);

        return data;
}

/*
 * Set per-column user data
 */
void XbaeMatrixSetColumnUserData(Widget w, int column, XtPointer data)
{
        XbaeMatrixWidget mw;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return;
        }

        mw = (XbaeMatrixWidget) w;
        if (!mw->matrix.column_user_data) {
                XtPointer *copy;

                copy = (XtPointer *) XtCalloc(mw->matrix.columns, sizeof(XtPointer));

                mw->matrix.column_user_data = copy;
        }

        mw->matrix.column_user_data[column] = data;

        xbaeObjectUnlock(w);
        return;
}

/*
 * Set per-cell widget
 */
void XbaeMatrixSetCellWidget(Widget w, int row, int column, Widget widget)
{
        XbaeMatrixWidget mw = (XbaeMatrixWidget) w;

        int num_widgets = mw->composite.num_children - XbaeNumChildren;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return;
        }
        if (row < 0 || column < 0 || row >= mw->matrix.rows || column >= mw->matrix.columns) {
                xbaeObjectUnlock(w);
                return;
        }

        if (!mw->matrix.per_cell) {
                xbaeCreatePerCell(mw);
        }
        mw->matrix.per_cell[row][column].widget = widget;

        /* If we're removing a cell widget, then we're done here */
        if (widget == NULL) {
                if (num_widgets == 1) {
                        /* This is the last user widget, reset the normal traversal behvour */
                        XtVaSetValues(TextChild(mw), XmNnavigationType, XmNONE, NULL);
                        XtVaSetValues(ClipChild(mw), XmNtraversalOn, True, NULL);
                }
                xbaeObjectUnlock(w);
                return;
        }

        if (num_widgets == 1) {
                /* This is the first user widget, start using standard TAB_GROUP traversal */
                XtVaSetValues(TextChild(mw), XmNnavigationType, XmTAB_GROUP, NULL);
                XtVaSetValues(ClipChild(mw), XmNtraversalOn, False, NULL);
        }

        XtVaSetValues(widget, XmNnavigationType, XmTAB_GROUP, NULL);
        
        if (XmIsGadget(widget)) {
                XtAppWarningMsg(XtWidgetToApplicationContext(w), "SetCellWidget",
                                "child is a gadget", "XbaeMatrix",
                                "XbaeMatrix: the child is a gadget - currently unsupported", NULL,
                                0);
                xbaeObjectUnlock(w);
                return;
        }

        /*
         * If we are adding after everything has been realized,
         * make sure that the widget is placed correctly.
         */
        if (!XtIsRealized(w)) {
                xbaeObjectUnlock(w);
                return;
        } else if(!XtIsRealized(widget))
                XtRealizeWidget(widget);

        xbaePositionCellWidget(mw, row, column);
        
        xbaeObjectUnlock(w);
}

Widget XbaeMatrixGetCellWidget(Widget w, int row, int column)
{
        XbaeMatrixWidget mw = (XbaeMatrixWidget) w;
        Widget r;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return NULL;
        }

        if (row < 0 || column < 0 || row >= mw->matrix.rows || column >= mw->matrix.columns) {
                xbaeObjectUnlock(w);
                return NULL;
        }

        if (!mw->matrix.per_cell) {
                xbaeObjectUnlock(w);
                return NULL;
        }
        r = mw->matrix.per_cell[row][column].widget;

        xbaeObjectUnlock(w);
        return r;
}

Boolean XbaeMatrixIsRowSelected(Widget w, int row)
{
        int col;
        XbaeMatrixWidget mw;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return False;
        }

        mw = (XbaeMatrixWidget) w;

        if (row < 0 || row >= mw->matrix.rows) {
                XtAppContext appcontext = XtWidgetToApplicationContext(w);
                XtAppError(appcontext, "Invalid row passed to XbaeMatrixIsRowSelected()");
                xbaeObjectUnlock(w);
                return False;
        }

        if (!mw->matrix.per_cell) {
                xbaeObjectUnlock(w);
                return False;
        }

        /*
         * Check all the cells in the row
         */
        for (col = 0; col < mw->matrix.columns; col++)
                if (!mw->matrix.per_cell[row][col].selected) {
                        xbaeObjectUnlock(w);
                        return False;
                }

        /*
         * Return success
         */
        xbaeObjectUnlock(w);
        return True;
}

Boolean XbaeMatrixIsColumnSelected(Widget w, int col)
{
        int row;
        XbaeMatrixWidget mw;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return False;
        }

        mw = (XbaeMatrixWidget) w;

        if (col < 0 || col >= mw->matrix.columns) {
                XtAppContext appcontext = XtWidgetToApplicationContext(w);
                XtAppError(appcontext, "Invalid column passed to XbaeMatrixIsColumnSelected()");
                xbaeObjectUnlock(w);
                return False;
        }

        if (!mw->matrix.per_cell) {
                xbaeObjectUnlock(w);
                return False;
        }

        /*
         * Check all the cells in the row
         */
        for (row = 0; row < mw->matrix.rows; row++)
                if (!mw->matrix.per_cell[row][col].selected) {
                        xbaeObjectUnlock(w);
                        return False;
                }

        /*
         * Return success
         */
        xbaeObjectUnlock(w);
        return True;
}

Boolean XbaeMatrixIsCellSelected(Widget w, int row, int column)
{
        XbaeMatrixWidget mw;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return False;
        }

        mw = (XbaeMatrixWidget) w;

        if (column < 0 || column >= mw->matrix.columns || row < 0 || row >= mw->matrix.rows) {
                XtAppContext appcontext = XtWidgetToApplicationContext(w);
                XtAppError(appcontext, "Invalid coordinates passed to XbaeMatrixIsCellSelected()");
                xbaeObjectUnlock(w);
                return False;
        }

        if (!mw->matrix.per_cell) {
                xbaeObjectUnlock(w);
                return False;
        }

        if (!mw->matrix.per_cell[row][column].selected) {
                xbaeObjectUnlock(w);
                return False;
        }

        /*
         * Return success
         */
        xbaeObjectUnlock(w);
        return True;
}

int XbaeMatrixFirstSelectedRow(Widget w)
{
        int i;
        XbaeMatrixWidget mw;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return -1;
        }

        mw = (XbaeMatrixWidget) w;

        if (!mw->matrix.per_cell) {
                xbaeObjectUnlock(w);
                return -1;
        }

        /*
         * Linear search for first selected
         */
        for (i = 0; i < mw->matrix.rows; i++)
                if (XbaeMatrixIsRowSelected(w, i)) {
                        xbaeObjectUnlock(w);
                        return i;
                }
        /*
         * No selection - return an invalid row id
         */
        xbaeObjectUnlock(w);
        return -1;
}

int XbaeMatrixFirstSelectedColumn(Widget w)
{
        int i;
        XbaeMatrixWidget mw;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return -1;
        }

        mw = (XbaeMatrixWidget) w;

        if (!mw->matrix.per_cell) {
                xbaeObjectUnlock(w);
                return -1;
        }

        /*
         * Linear search for first selected
         */
        for (i = 0; i < mw->matrix.columns; i++)
                if (XbaeMatrixIsColumnSelected(w, i)) {
                        xbaeObjectUnlock(w);
                        return i;
                }
        /*
         * No selection - return an invalid row id
         */
        xbaeObjectUnlock(w);
        return -1;
}

void XbaeMatrixFirstSelectedCell(Widget w, int *row, int *column)
{
        int i, j;
        XbaeMatrixWidget mw;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                *row = *column = -1;
                xbaeObjectUnlock(w);
                return;
        }

        mw = (XbaeMatrixWidget) w;

        if (!mw->matrix.per_cell) {
                *row = -1;
                *column = -1;
                xbaeObjectUnlock(w);
                return;
        }

        for (i = 0; i < mw->matrix.rows; i++)
                for (j = 0; j < mw->matrix.columns; j++)
                        if (mw->matrix.per_cell[i][j].selected) {
                                *row = i;
                                *column = j;
                                xbaeObjectUnlock(w);
                                return;
                        }
        *row = *column = -1;
        xbaeObjectUnlock(w);
}

int XbaeMatrixGetNumSelected(Widget w)
{
        XbaeMatrixWidget mw;
        int i;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return 0;
        }

        mw = (XbaeMatrixWidget) w;

        i = mw->matrix.num_selected_cells;
        xbaeObjectUnlock(w);
        return i;
}


int XbaeMatrixNumColumns(Widget w)
{
        XbaeMatrixWidget mw;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return 0;
        }

        mw = (XbaeMatrixWidget) w;

        xbaeObjectUnlock(w);
        return mw->matrix.columns;
}


int XbaeMatrixNumRows(Widget w)
{
        XbaeMatrixWidget mw;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return 0;
        }

        mw = (XbaeMatrixWidget) w;

        xbaeObjectUnlock(w);
        return mw->matrix.rows;
}


void XbaeMatrixUnhighlightAll(Widget w)
{
        XbaeMatrixWidget mw;
        int row, column;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return;
        }
        mw = (XbaeMatrixWidget) w;

        if (!mw->matrix.per_cell) {
                xbaeObjectUnlock(w);
                return;
        }

        for (row = 0; row < mw->matrix.rows; row++) {
                for (column = 0; column < mw->matrix.columns; column++) {
                        /*
                         * If the cell is visible and highlighted
                         */
                        if (mw->matrix.per_cell[row][column].highlighted) {
                                unsigned char new_hl = HighlightNone;
                                if (xbaeIsCellVisible(mw, row, column))
                                        xbaeChangeHighlight(mw, row, column, new_hl);
                                mw->matrix.per_cell[row][column].highlighted = new_hl;
                        }
                }
        }

        xbaeObjectUnlock(w);
}


void XbaeMatrixHighlightCell(Widget w, int row, int column)
{
        XbaeMatrixWidget mw;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return;
        }
        mw = (XbaeMatrixWidget) w;

        if (row >= mw->matrix.rows || row < 0 || column >= mw->matrix.columns || column < 0) {
                XtAppWarningMsg(XtWidgetToApplicationContext((Widget) mw), "highlightCell",
                                "badIndex", "XbaeMatrix",
                                "XbaeMatrix: Row or column out of bounds for HighlightCell.", NULL,
                                0);
                xbaeObjectUnlock(w);
                return;
        }

        /*
         * Scroll the cell onto the screen
         */
        if (mw->matrix.scroll_select)
                xbaeMakeCellVisible(mw, row, column);

        if (!mw->matrix.per_cell)
                xbaeCreatePerCell(mw);

        /*
         * If the cell is not already highlighted
         */
        if (!(mw->matrix.per_cell[row][column].highlighted & HighlightCell)) {
                unsigned char new_hl = mw->matrix.per_cell[row][column].highlighted | HighlightCell;
                if (xbaeIsCellVisible(mw, row, column)) 
                        xbaeChangeHighlight(mw, row, column, new_hl);
                mw->matrix.per_cell[row][column].highlighted = new_hl;
        }

        xbaeObjectUnlock(w);
}


void XbaeMatrixUnhighlightCell(Widget w, int row, int column)
{
        XbaeMatrixWidget mw;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return;
        }
        mw = (XbaeMatrixWidget) w;

        if (row >= mw->matrix.rows || row < 0 || column >= mw->matrix.columns || column < 0) {
                XtAppWarningMsg(XtWidgetToApplicationContext((Widget) mw), "unhighlightCell",
                                "badIndex", "XbaeMatrix",
                                "XbaeMatrix: Row or column out of bounds for UnhighlightCell.",
                                NULL, 0);
                xbaeObjectUnlock(w);
                return;
        }

        if (!mw->matrix.per_cell) {
                xbaeObjectUnlock(w);
                return;
        }

        if (mw->matrix.per_cell[row][column].highlighted & HighlightCell) {
                unsigned char new_hl = mw->matrix.per_cell[row][column].highlighted & ~HighlightCell;
                if (xbaeIsCellVisible(mw, row, column))
                        xbaeChangeHighlight(mw, row, column, new_hl);
                mw->matrix.per_cell[row][column].highlighted = new_hl;
        }

        xbaeObjectUnlock(w);
}


void XbaeMatrixHighlightRow(Widget w, int row)
{
        XbaeMatrixWidget mw;
        int column;
        Boolean visible;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return;
        }
        mw = (XbaeMatrixWidget) w;

        if (row >= mw->matrix.rows || row < 0) {
                XtAppWarningMsg(XtWidgetToApplicationContext((Widget) mw), "highlightRow",
                                "badIndex", "XbaeMatrix",
                                "XbaeMatrix: Row out of bounds for HighlightRow.", NULL, 0);
                xbaeObjectUnlock(w);
                return;
        }

        if (!mw->matrix.per_cell)
                xbaeCreatePerCell(mw);

        /*
         * Scroll the row onto the screen
         */
        if (mw->matrix.scroll_select)
                xbaeMakeRowVisible(mw, row);

        /*
         * For each cell in the row, if the cell is not already highlighted,
         * highlight it and redraw it if it is visible
         */

        visible = xbaeIsRowVisible(mw, row);

        for (column = 0; column < mw->matrix.columns; column++) {
                if (!(mw->matrix.per_cell[row][column].highlighted & HighlightRow)) {
                        unsigned char new_hl = mw->matrix.per_cell[row][column].highlighted | HighlightRow;
                        if (visible && xbaeIsColumnVisible(mw, column))
                                xbaeChangeHighlight(mw, row, column, new_hl);
                        mw->matrix.per_cell[row][column].highlighted = new_hl;
                }
        }

        xbaeObjectUnlock(w);
}


void XbaeMatrixUnhighlightRow(Widget w, int row)
{
        XbaeMatrixWidget mw;
        int column;
        Boolean visible;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return;
        }
        mw = (XbaeMatrixWidget) w;

        if (row >= mw->matrix.rows || row < 0) {
                XtAppWarningMsg(XtWidgetToApplicationContext((Widget) mw), "highlightRow",
                                "badIndex", "XbaeMatrix",
                                "XbaeMatrix: Row out of bounds for UnhighlightRow.", NULL, 0);
                xbaeObjectUnlock(w);
                return;
        }

        if (!mw->matrix.per_cell) {
                xbaeObjectUnlock(w);
                return;
        }

        visible = xbaeIsRowVisible(mw, row);

        /*
         * For each cell in the row, if the cell is highlighted,
         * unhighlight it and redraw it if it is visible
         */

        for (column = 0; column < mw->matrix.columns; column++) {
                if (mw->matrix.per_cell[row][column].highlighted & HighlightRow) {
                        unsigned char new_hl = mw->matrix.per_cell[row][column].highlighted & ~HighlightRow;
                        if (visible && xbaeIsColumnVisible(mw, column))
                                xbaeChangeHighlight(mw, row, column, new_hl);
                        mw->matrix.per_cell[row][column].highlighted = new_hl;
                }
        }

        xbaeObjectUnlock(w);
}


void XbaeMatrixHighlightColumn(Widget w, int column)
{
        XbaeMatrixWidget mw;
        int row;
        Boolean visible;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return;
        }
        mw = (XbaeMatrixWidget) w;

        if (column >= mw->matrix.columns || column < 0) {
                XtAppWarningMsg(XtWidgetToApplicationContext((Widget) mw), "highlightColumn",
                                "badIndex", "XbaeMatrix",
                                "XbaeMatrix: Column out of bounds for HighlightColumn.", NULL, 0);
                xbaeObjectUnlock(w);
                return;
        }

        if (!mw->matrix.per_cell)
                xbaeCreatePerCell(mw);

        /*
         * Scroll the row onto the screen
         */
        if (mw->matrix.scroll_select)
                xbaeMakeColumnVisible(mw, column);

        /*
         * For each cell in the column, if the cell is not already highlighted,
         * highlight it and redraw it if it is visible
         */

        visible = xbaeIsColumnVisible(mw, column);
        for (row = 0; row < mw->matrix.rows; row++) {
                if (!(mw->matrix.per_cell[row][column].highlighted & HighlightColumn)) {
                        unsigned char new_hl = mw->matrix.per_cell[row][column].highlighted | HighlightColumn;
                        if (visible && xbaeIsRowVisible(mw, row))
                                xbaeChangeHighlight(mw, row, column, new_hl);
                        mw->matrix.per_cell[row][column].highlighted = new_hl;
                }
        }
        xbaeObjectUnlock(w);
}


void XbaeMatrixUnhighlightColumn(Widget w, int column)
{
        XbaeMatrixWidget mw;
        int row;
        Boolean visible;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return;
        }
        mw = (XbaeMatrixWidget) w;

        if (column >= mw->matrix.columns || column < 0) {
                XtAppWarningMsg(XtWidgetToApplicationContext((Widget) mw), "highlightColumn",
                                "badIndex", "XbaeMatrix",
                                "XbaeMatrix: Column out of bounds for UnhighlightColumn.", NULL, 0);
                xbaeObjectUnlock(w);
                return;
        }

        if (!mw->matrix.per_cell) {
                xbaeObjectUnlock(w);
                return;
        }

        visible = xbaeIsColumnVisible(mw, column);

        /*
         * For each cell in the row, if the cell is highlighted,
         * unhighlight it and redraw it if it is visible.
         */
        for (row = 0; row < mw->matrix.rows; row++) {
                if (mw->matrix.per_cell[row][column].highlighted & HighlightColumn) {
                        unsigned char new_hl = mw->matrix.per_cell[row][column].highlighted & ~HighlightColumn;
                        if (visible && xbaeIsRowVisible(mw, row))
                                xbaeChangeHighlight(mw, row, column, new_hl);
                        mw->matrix.per_cell[row][column].highlighted = new_hl;
                }
        }
        xbaeObjectUnlock(w);
}

void XbaeMatrixDisableRedisplay(Widget w)
{
        XbaeMatrixWidget mw;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return;
        }
        mw = (XbaeMatrixWidget) w;

        mw->matrix.disable_redisplay++;
        xbaeObjectUnlock(w);
}


void XbaeMatrixEnableRedisplay(Widget w, Boolean redisplay)
{
        XbaeMatrixWidget mw;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return;
        }
        mw = (XbaeMatrixWidget) w;

        if (mw->matrix.disable_redisplay)
                mw->matrix.disable_redisplay--;

#undef FORCE_REDISPLAY_IF_TRUE
#ifndef FORCE_REDISPLAY_IF_TRUE
        if (redisplay && mw->matrix.disable_redisplay == 0)
                XbaeMatrixRefresh(w);
#else
        if (redisplay) {
                mw->matrix.disable_redisplay = 0;
                XbaeMatrixRefresh(w);
        }
#endif
        xbaeObjectUnlock(w);
}

/*
 * Public interface for xbaeMakeCellVisible()
 */
void XbaeMatrixMakeCellVisible(Widget w, int row, int column)
{
        XbaeMatrixWidget mw;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return;
        }

        mw = (XbaeMatrixWidget) w;

        xbaeMakeCellVisible(mw, row, column);
        xbaeObjectUnlock(w);
}

/*
 * Public interface for xbaeIsRowVisible()
 */
Boolean XbaeMatrixIsRowVisible(Widget w, int row)
{
        XbaeMatrixWidget mw;
        Boolean r;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return False;
        }

        mw = (XbaeMatrixWidget) w;

        r = xbaeIsRowVisible(mw, row);
        xbaeObjectUnlock(w);
        return r;
}

/*
 * Public interface for xbaeIsColumnVisible()
 */
Boolean XbaeMatrixIsColumnVisible(Widget w, int col)
{
        XbaeMatrixWidget mw;
        Boolean r;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return False;
        }

        mw = (XbaeMatrixWidget) w;

        r = xbaeIsColumnVisible(mw, col);
        xbaeObjectUnlock(w);
        return r;
}

/*
 * Public interface for xbaeIsCellVisible()
 */
Boolean XbaeMatrixIsCellVisible(Widget w, int row, int col)
{
        XbaeMatrixWidget mw;
        Boolean r;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return False;
        }

        mw = (XbaeMatrixWidget) w;

        r = xbaeIsCellVisible(mw, row, col);
        xbaeObjectUnlock(w);
        return r;
}

/*
 *  XbaeMatrixVisibleCells()
 *
 *  This routine returns the range of cells that are visible in the matrix.
 *
 */
void
XbaeMatrixVisibleCells(Widget w, int *top_row, int *bottom_row, int *left_column, int *right_column)
{
        XbaeMatrixWidget mw;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return;
        }

        mw = (XbaeMatrixWidget) w;

        xbaeGetVisibleRows(mw, top_row, bottom_row);
        xbaeGetVisibleColumns(mw, left_column, right_column);

        xbaeObjectUnlock(w);
}

/*
 * Get the label of the column passed here.
 */
String XbaeMatrixGetColumnLabel(Widget w, int column)
{
        XbaeMatrixWidget mw;
        String r;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return NULL;
        }

        mw = (XbaeMatrixWidget) w;

        if (!(mw->matrix.column_labels) || column > mw->matrix.columns) {
                xbaeObjectUnlock(w);
                return NULL;
        } else {
                r = mw->matrix.column_labels[column];
                xbaeObjectUnlock(w);
                return r;
        }
}

/*
 * Get the label of the row passed here.
 */
String XbaeMatrixGetRowLabel(Widget w, int row)
{
        XbaeMatrixWidget mw;
        String r;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return NULL;
        }

        mw = (XbaeMatrixWidget) w;

        if (!(mw->matrix.row_labels) || row > mw->matrix.rows) {
                xbaeObjectUnlock(w);
                return NULL;
        } else {
                r = mw->matrix.row_labels[row];
                xbaeObjectUnlock(w);
                return r;
        }
}

void XbaeMatrixSetColumnLabel(Widget w, int column, String value)
{
        XbaeMatrixWidget mw;
        Boolean first_label = False;
        int column_label_maxlines;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return;
        }

        mw = (XbaeMatrixWidget) w;
        column_label_maxlines = mw->matrix.column_label_maxlines;
        
        if (column < 0 || column >= mw->matrix.columns) {
                XtAppWarningMsg(XtWidgetToApplicationContext(w), "setColumnLabel",
                                "invalid column number", "XbaeMatrix",
                                "XbaeMatrix: invalid column number", NULL, 0);
                xbaeObjectUnlock(w);
                return;
        }

        /*
         * If we have no labels so far create empty ones for all the columns
         */
        if (!mw->matrix.column_labels) {
                int c;
                mw->matrix.column_labels = (String *) XtMalloc(mw->matrix.columns * sizeof(String));
                mw->matrix.column_label_lines = (ColumnLabelLines) XtMalloc(mw->matrix.columns * sizeof(ColumnLabelLinesRec));
                for (c = 0; c < mw->matrix.columns; c++) {
                        mw->matrix.column_labels[c] = NULL;
                        xbaeParseColumnLabel(NULL, &mw->matrix.column_label_lines[c]);
                }
                first_label = True;
        }
        
        /*
         * Free up the memory that was used up by the old label and copy the new value
         */
        if (mw->matrix.column_labels[column]) {
                XtFree((char *) mw->matrix.column_labels[column]);
                XtFree((char *) mw->matrix.column_label_lines[column].lengths);
        }

        mw->matrix.column_labels[column] = (value == NULL) ? NULL : XtNewString(value);
        xbaeParseColumnLabel(value, &mw->matrix.column_label_lines[column]);
        mw->matrix.column_label_maxlines = xbaeCalculateLabelMaxLines(mw->matrix.column_label_lines,
                                                                      mw->matrix.columns);

        /*
         * If this is the first column label or column_label_maxlines changed, redraw the whole matrix 
         * else redraw only the label that was changed
         */
        if (XtIsRealized(w) && !mw->matrix.disable_redisplay) {
                if (first_label || column_label_maxlines != mw->matrix.column_label_maxlines) {
                        XbaeMatrixRefresh(w);
                } else if (xbaeIsColumnVisible(mw, column)) {
                        xbaeDrawColumnLabel(mw, column, False);
                }
        }
        
        xbaeObjectUnlock(w);
}

void XbaeMatrixSetRowLabel(Widget w, int row, String value)
{
        XbaeMatrixWidget mw;
        Boolean first_label = False;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return;
        }

        mw = (XbaeMatrixWidget) w;

        if (row < 0 || row >= mw->matrix.rows) {
                XtAppWarningMsg(XtWidgetToApplicationContext(w), "setRowLabel",
                                "invalid row number", "XbaeMatrix",
                                "XbaeMatrix: invalid row number", NULL, 0);
                xbaeObjectUnlock(w);
                return;
        }

        /*
         * If we have no labels so far create empty ones for all the rows
         */
        if (!mw->matrix.row_labels) {
                int r;
                mw->matrix.row_labels = (String *) XtMalloc(mw->matrix.rows * sizeof(String));
                for (r = 0; r < mw->matrix.rows; r++) {
                        mw->matrix.row_labels[r] = NULL;
                }
                first_label = True;
        }

        /*
         * Free up the memory that was used up by the label and copy the new value
         */
        if (mw->matrix.row_labels[row]) {
            XtFree((XtPointer) mw->matrix.row_labels[row]);
        }

        mw->matrix.row_labels[row] = (value == NULL) ? NULL : XtNewString(value);

        /*
         * If this is the first row label, redraw the whole matrix 
         * else redraw only the label that was changed
         */
        if (1 || (XtIsRealized(w) && !mw->matrix.disable_redisplay)) {
                if (first_label) {
                        XbaeMatrixRefresh(w);
                } else if (xbaeIsRowVisible(mw, row)) {
                        xbaeDrawRowLabel(mw, row, False);
                }
        }
        xbaeObjectUnlock(w);
}

Widget XbaeCreateMatrix(Widget parent, String name, ArgList args, Cardinal ac)
{
        return XtCreateWidget(name, xbaeMatrixWidgetClass, parent, args, ac);
}

#ifndef	XtNwrongParameters
#define	XtNwrongParameters	"wrongParameters"
#endif

#ifndef	XtCXtToolkitError
#define	XtCXtToolkitError	"XtToolkitError"
#endif

void XbaeMatrixSetColumnWidth(Widget w, int column, int width)
{
        XbaeMatrixWidget mw = (XbaeMatrixWidget) w;

        if (mw && mw->matrix.column_widths && column >= 0 && column <= mw->matrix.columns) {
                if (width < 0)
                        mw->matrix.column_widths[column] = DEFAULT_COLUMN_WIDTH(mw);
                else
                        mw->matrix.column_widths[column] = width;

                xbaeGetColumnPositions(mw);
                XbaeMatrixRefresh(w);
        } else {
                XtAppWarningMsg(XtDisplayToApplicationContext(XtDisplay(w)), XtNwrongParameters,
                                "xbaeMatrixNoSuchColumn", XtCXtToolkitError,
                                "XbaeMatrix doesn't have this column", (String *) NULL,
                                (Cardinal *) NULL);
        }
}

int XbaeMatrixGetColumnWidth(Widget w, int column)
{
        XbaeMatrixWidget mw = (XbaeMatrixWidget) w;

        if (mw && column >= 0 && column < mw->matrix.columns)
                return mw->matrix.column_widths[column];
        return 0;
}

void XbaeMatrixSetRowHeight(Widget w, int row, int height)
{
        XbaeMatrixWidget mw = (XbaeMatrixWidget) w;

        if (mw && mw->matrix.row_heights && row >= 0 && row <= mw->matrix.rows) {
                if (height < 0)
                        mw->matrix.row_heights[row] = DEFAULT_ROW_HEIGHT(mw);
                else {
                        mw->matrix.row_heights[row] = height;
                }
                xbaeGetRowPositions(mw);
                XbaeMatrixRefresh(w);
        } else {
                XtAppWarningMsg(XtDisplayToApplicationContext(XtDisplay(w)), XtNwrongParameters,
                                "xbaeMatrixNoSuchRow", XtCXtToolkitError,
                                "XbaeMatrix doesn't have this row", (String *) NULL,
                                (Cardinal *) NULL);
        }
}

int XbaeMatrixGetRowHeight(Widget w, int row)
{
        XbaeMatrixWidget mw = (XbaeMatrixWidget) w;

        if (mw && row >= 0 && row < mw->matrix.rows)
                return mw->matrix.row_heights[row];
        return 0;
}


void XbaeMatrixShowColumnArrows(Widget w, int column, Boolean show)
{
        xbaeObjectLock(w);
        /*
         * Make sure w is a Matrix or a subclass
         */
        XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

        /*
         * Call the select_row method
         */
        ((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.set_show_column_arrows(
                                                                  (XbaeMatrixWidget) w, column, show);

        xbaeObjectUnlock(w);
}



struct sort_common {
        Widget w;
        int (*proc)(Widget, int, int, void *);
        void *user_data;
};

struct sort_index {
        int index;
        struct sort_common *common;
};

static int compare(const void *a, const void *b) {
        const struct sort_index *sort_index1 = a;
        const struct sort_index *sort_index2 = b;
        const struct sort_common *common = sort_index1->common;

        return common->proc(common->w, sort_index1->index, sort_index2->index, common->user_data);
}

void *reorder(void *array, size_t size, int n, struct sort_index *sort_indices) {
        void *ordered_array = NULL;

        if (array) {
                int i;
                ordered_array = XtMalloc(n * size);

                for(i = 0; i < n; i++) {
                       memcpy((char *) ordered_array + size * i, (char *) array + size * sort_indices[i].index, size);
                }

                XtFree(array);
        }
        
        return ordered_array;
}

void XbaeMatrixSortRows(Widget w, int (*proc)(Widget, int, int, void *), void *user_data)
{
        XbaeMatrixWidget mw;
        int row;
        int n_rows;

        struct sort_common common;
        struct sort_index *row_indices;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return;
        }

        mw = (XbaeMatrixWidget) w;

        if (mw->matrix.rows == 0 || mw->matrix.columns == 0 || mw->matrix.per_cell == NULL) {
                return;
        }

        n_rows = mw->matrix.rows;

        common.w = w;
        common.proc = proc;
        common.user_data = user_data;

        row_indices = (struct sort_index *) XtMalloc(n_rows * sizeof *row_indices);

        for(row = 0; row < n_rows; row++) {
                row_indices[row].index = row;
                row_indices[row].common = &common;
        }

        qsort(row_indices, n_rows, sizeof *row_indices, compare);

        mw->matrix.per_cell = reorder(mw->matrix.per_cell, 
                                      sizeof *mw->matrix.per_cell, n_rows, row_indices);
        mw->matrix.row_heights = reorder(mw->matrix.row_heights, 
                                         sizeof *mw->matrix.row_heights, n_rows, row_indices);
        mw->matrix.row_labels = reorder(mw->matrix.row_labels, 
                                        sizeof *mw->matrix.row_labels, n_rows, row_indices);
        mw->matrix.row_button_labels = reorder(mw->matrix.row_button_labels, 
                                               sizeof *mw->matrix.row_button_labels, n_rows, row_indices);
        mw->matrix.row_user_data = reorder(mw->matrix.row_user_data, 
                                           sizeof *mw->matrix.row_user_data, n_rows, row_indices);
        mw->matrix.row_shadow_types = reorder(mw->matrix.row_shadow_types, 
                                              sizeof *mw->matrix.row_shadow_types, n_rows, row_indices);

        XtFree((XtPointer) row_indices);

        xbaeGetRowPositions(mw);

        XbaeMatrixRefresh(w);

        xbaeObjectUnlock(w);
}


void XbaeMatrixSortColumns(Widget w, int (*proc)(Widget, int, int, void *), void *user_data)
{
        XbaeMatrixWidget mw;
        int row, column;
        int n_rows, n_columns;

        struct sort_common common;
        struct sort_index *column_indices;

        xbaeObjectLock(w);
        if (!XtIsSubclass(w, xbaeMatrixWidgetClass)) {
                xbaeObjectUnlock(w);
                return;
        }

        mw = (XbaeMatrixWidget) w;

        if (mw->matrix.rows == 0 || mw->matrix.columns == 0 || mw->matrix.per_cell == NULL) {
                return;
        }

        n_rows = mw->matrix.rows;
        n_columns = mw->matrix.columns;

        common.w = w;
        common.proc = proc;
        common.user_data = user_data;

        column_indices = (struct sort_index *) XtMalloc(n_columns * sizeof *column_indices);

        for(column = 0; column < n_columns; column++) {
                column_indices[column].index = column;
                column_indices[column].common = &common;
        }

        qsort(column_indices, n_columns, sizeof *column_indices, compare);

        for(row = 0; row < n_rows; row++) {
                mw->matrix.per_cell[row] = reorder(mw->matrix.per_cell[row], 
                                                   sizeof *mw->matrix.per_cell[row], n_columns, column_indices);
        }

        mw->matrix.column_widths = reorder(mw->matrix.column_widths, 
                                           sizeof *mw->matrix.column_widths, n_columns, column_indices);
        mw->matrix.column_labels = reorder(mw->matrix.column_labels, 
                                           sizeof *mw->matrix.column_labels, n_columns, column_indices);
        mw->matrix.column_label_lines = reorder(mw->matrix.column_label_lines, 
                                                sizeof *mw->matrix.column_label_lines, n_columns, column_indices);
        mw->matrix.column_button_labels = reorder(mw->matrix.column_button_labels, 
                                                  sizeof *mw->matrix.column_button_labels, n_columns, column_indices);
        mw->matrix.column_user_data = reorder(mw->matrix.column_user_data, 
                                              sizeof *mw->matrix.column_user_data, n_columns, column_indices);
        mw->matrix.column_shadow_types = reorder(mw->matrix.column_shadow_types, 
                                                 sizeof *mw->matrix.column_shadow_types, n_columns, column_indices);
        mw->matrix.column_alignments = reorder(mw->matrix.column_alignments, 
                                               sizeof *mw->matrix.column_alignments, n_columns, column_indices);
        mw->matrix.column_label_alignments = reorder(mw->matrix.column_label_alignments, 
                                                     sizeof *mw->matrix.column_label_alignments, n_columns, column_indices);
        mw->matrix.column_font_bold = reorder(mw->matrix.column_font_bold, 
                                              sizeof *mw->matrix.column_font_bold, n_columns, column_indices);
        mw->matrix.show_column_arrows = reorder(mw->matrix.show_column_arrows, 
                                                sizeof *mw->matrix.show_column_arrows, n_columns, column_indices);
        mw->matrix.column_max_lengths = reorder(mw->matrix.column_max_lengths, 
                                                sizeof *mw->matrix.column_max_lengths, n_columns, column_indices);
        mw->matrix.xmcolumn_labels = reorder(mw->matrix.xmcolumn_labels, 
                                             sizeof *mw->matrix.xmcolumn_labels, n_columns, column_indices);

        XtFree((XtPointer) column_indices);
        
        xbaeGetColumnPositions(mw);

        XbaeMatrixRefresh(w);

        xbaeObjectUnlock(w);
}
