/*
 * Copyright(c) 1992 Bell Communications Research, Inc. (Bellcore)
 * Copyright(c) 1995-99 Andrew Lister
 * Copyright  1999, 2000, 2001, 2002, 2003, 2004 by the LessTif Developers.
 *
 *                        All rights reserved
 * Permission to use, copy, modify and distribute this material for
 * any purpose and without fee is hereby granted, provided that the
 * above copyright notice and this permission notice appear in all
 * copies, and that the name of Bellcore not be used in advertising
 * or publicity pertaining to this material without the specific,
 * prior written permission of an authorized representative of
 * Bellcore.
 *
 * BELLCORE MAKES NO REPRESENTATIONS AND EXTENDS NO WARRANTIES, EX-
 * PRESS OR IMPLIED, WITH RESPECT TO THE SOFTWARE, INCLUDING, BUT
 * NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR ANY PARTICULAR PURPOSE, AND THE WARRANTY AGAINST IN-
 * FRINGEMENT OF PATENTS OR OTHER INTELLECTUAL PROPERTY RIGHTS.  THE
 * SOFTWARE IS PROVIDED "AS IS", AND IN NO EVENT SHALL BELLCORE OR
 * ANY OF ITS AFFILIATES BE LIABLE FOR ANY DAMAGES, INCLUDING ANY
 * LOST PROFITS OR OTHER INCIDENTAL OR CONSEQUENTIAL DAMAGES RELAT-
 * ING TO THE SOFTWARE.
 *
 * $Id: Draw.c,v 1.86 2004/11/24 01:19:25 tobiasoed Exp $
 */

#ifdef HAVE_CONFIG_H
#include <XbaeConfig.h>
#endif

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>

#include <Xm/Xm.h>
#include <Xm/XmP.h>
#include <Xm/DrawP.h>
#include <Xbae/MatrixP.h>
#include <Xbae/Utils.h>
#include <Xbae/Shadow.h>
#include <Xbae/Draw.h>

#include <XbaeDebug.h>

#if (XmVERSION < 2)
#define	XmUNSPECIFIED_PIXEL	((Pixel) (~0))
#endif

static void xbaeDrawCellString(XbaeMatrixWidget, Window, int, int, int, int, int, int, String, Pixel, Pixel);
static void xbaeDrawCellWidget(XbaeMatrixWidget, Window, int, int, int, int, int, int, Widget, Pixel, Pixel);
static void xbaeDrawCellPixmap(XbaeMatrixWidget, Window, int, int, int, int, int, int, Pixmap, Pixmap, int, int,
                               Pixel, Pixel, int);

static void xbaeDrawString(XbaeMatrixWidget mw, Window win, GC gc, String string, Boolean xm,
                           int length, int x, int y, int maxlen, int maxht, unsigned char alignment,
                           Boolean arrows, Boolean highlight, Boolean bold, Boolean underline,
                           Boolean rowLabel, Boolean colLabel, Pixel color);

/*
 * Draw a fixed or non-fixed cell. The coordinates are calculated relative
 * to the correct window and pixmap is copied to that window.
 */
static void
xbaeDrawCellPixmap(XbaeMatrixWidget mw, Window win, int row, int column, int x, int y, int width, int height, Pixmap pixmap,
                   Pixmap mask, int width_return, int height_return, Pixel bg, Pixel fg, int depth)
{
        int src_x = 0, src_y, dest_x, dest_y;
        int copy_width, copy_height;
        unsigned char alignment = mw->matrix.column_alignments 
                                  ? mw->matrix. column_alignments[column]
                                  : XmALIGNMENT_BEGINNING;
        Display *display = XtDisplay(mw);
        GC gc = mw->matrix.pixmap_gc;

        /*
         * Convert the row/column to the coordinates relative to the correct
         * window
         */
        dest_x = x + TEXT_WIDTH_OFFSET(mw);

        XSetForeground(display, gc, fg);
        XSetBackground(display, gc, bg);

        /*
         * Adjust the x and y drawing destinations as appropriate.  First the
         * y value....
         */
        dest_y = y;
        if (height_return > height) {
                /* Adjust the starting location in the src image */
                src_y = (height_return - height) / 2;
                copy_height = height;
        } else {
                /* Adjust the destination point */
                src_y = 0;
                dest_y += ((height - height_return) / 2);
                copy_height = height_return;
        }

        /*
         * Adjust the x value, paying attention to the columnAlignment
         */
        if (width_return > width)
                copy_width = width;
        else
                copy_width = width_return;

        switch (alignment) {
        case XmALIGNMENT_BEGINNING:
                src_x = 0;
                break;
        case XmALIGNMENT_CENTER:
                if (width_return > width)
                        src_x = (width_return - width) / 2;
                else {
                        src_x = 0;
                        dest_x += ((width - width_return) / 2);
                }
                break;
        case XmALIGNMENT_END:
                if (width_return > width)
                        src_x = width_return - width;
                else {
                        src_x = 0;
                        dest_x = x + width - TEXT_WIDTH_OFFSET(mw) - width_return;
                }
                break;
        }

        /*
         * Draw the pixmap.  Clip it, if necessary
         */
        if (pixmap && pixmap != XmUNSPECIFIED_PIXMAP) {
                if (depth > 1) {        /* A pixmap using xpm */
                        if (mask && mask != XmUNSPECIFIED_PIXMAP) {
                                XSetClipMask(display, gc, mask);

                                XSetClipOrigin(display, gc, dest_x - src_x, dest_y - src_y);
                        }
                        XCopyArea(display, pixmap, win, gc, src_x, src_y, copy_width, copy_height,
                                  dest_x, dest_y);
                        if (mask && mask != XmUNSPECIFIED_PIXMAP)
                                XSetClipMask(display, gc, None);
                } else          /* A plain old bitmap */
                        XCopyPlane(display, pixmap, win, gc, src_x, src_y, copy_width, copy_height,
                                   dest_x, dest_y, 1L);
        }
}

/*
 * Draw a fixed or non-fixed cell. The coordinates are calculated relative
 * to the correct window and the cell is drawn in that window.
 */
static void
xbaeDrawCellString(XbaeMatrixWidget mw, Window win, int row, int column, int x, int y, int width, int height,
                   String string, Pixel bg, Pixel fg)
{
        GC gc = mw->matrix.draw_gc;
        Boolean selected = False;
        Boolean underline = False;

        if (mw->matrix.per_cell) {
                selected = mw->matrix.per_cell[row][column].selected;
                underline = mw->matrix.per_cell[row][column].underlined;
        }
        
        DEBUGOUT(_XbaeDebug
                 (__FILE__, (Widget) mw, "%s[%d,%d] x %d y %d '%s'\n", __FUNCTION__, row, column, x,
                  y, string));

        if (height == 0)
                return;         /* Quick fix */

        /*
         * Draw the string in the cell.
         */
        xbaeDrawString(mw, win, gc, string, False, strlen(string), 
                       x + TEXT_X_OFFSET(mw), 
                       y + TEXT_Y_OFFSET(mw), 
                       mw->matrix.column_widths[column], 
                       height,
                       mw->matrix.column_alignments 
                        ? mw->matrix.column_alignments[column] 
                        : XmALIGNMENT_BEGINNING, 
                       mw->matrix.show_arrows && (mw->matrix.show_column_arrows == NULL
                                                  || mw->matrix.show_column_arrows[column]), 
                       selected,
                       mw->matrix.column_font_bold ? mw->matrix.column_font_bold[column] : 0,
                       underline, False, False, fg);
}

/*
 * Draw a user defined widget in the cell
 */
static void
xbaeDrawCellWidget(XbaeMatrixWidget mw, Window win, int row, int column, int x, int y, int width, int height, 
                   Widget widget, Pixel bg, Pixel fg)
{
        DEBUGOUT(_XbaeDebug2
                 (__FILE__, (Widget) mw, widget, "%s: rc %d,%d xy %d,%d\n", __FUNCTION__, row,
                  column, x, y));
}

/*
 * Width in pixels of a character in a given font
 */
#define charWidth(fs,c) ((fs)->per_char ? \
                         (fs)->per_char[((c) < (fs)->min_char_or_byte2 ? \
					 (fs)->default_char : \
					 (c) - \
					 (fs)->min_char_or_byte2)].width : \
			 (fs)->min_bounds.width)


/*
 * Draw a string with specified attributes. We want to avoid having to
 * use a GC clip_mask, so we clip by characters. This complicates the code.
 */

/* FIX ME this should be clipped with the row height for the last row */
static void xbaeDrawString(XbaeMatrixWidget mw, Window win, GC gc, String string, Boolean xm,   /* Is the above an XmString or a String ? */
                           int length, int x, int y, int maxlen, int maxht,     /* Added for row heights */
                           unsigned char alignment, Boolean arrows, Boolean highlight, Boolean bold,
                           Boolean underline, Boolean rowLabel, Boolean colLabel, Pixel color)
{

        int start, width, maxwidth, font_height;
        XFontStruct *font_struct;
        XFontSet font_set;
        Boolean choppedStart = False;
        Boolean choppedEnd = False;
        XRectangle *ink_array = NULL;
        XRectangle *logical_array = NULL;
        int num_chars, doclip;
        XRectangle overall_logical, clipht;
        Dimension i;            /* loop index */
        XmString xms = NULL;

        if (xm) {
                xms = (XmString) string;
                string = NULL;
        }

        DEBUGOUT(_XbaeDebug
                 (__FILE__, (Widget) mw, "%s(%s) x %d y %d, clip w %d h %d\n", __FUNCTION__,
                  xm ? "(some xmstring)" : string, x, y, maxlen, maxht));

        if (rowLabel || colLabel) {
                font_struct = mw->matrix.label_font_struct;
                font_set = mw->matrix.label_font_set;
                font_height = LABEL_HEIGHT(mw);
        } else {
                font_struct = mw->matrix.font_struct;
                font_set = mw->matrix.font_set;
                font_height = FONT_HEIGHT(mw);
        }

        /*
         * Initialize starting character in string
         */
        start = 0;
        
        if (!rowLabel) {
                if (mw->matrix.column_width_in_pixels) {
                        maxwidth = (maxlen - ((int) TEXT_WIDTH_OFFSET(mw) * 2));
                } else
                        maxwidth = maxlen * FONT_WIDTH(mw);
        } else {
                if (mw->matrix.column_width_in_pixels) {
                        maxwidth = (maxlen - ((int) TEXT_WIDTH_OFFSET(mw) * 2));
                } else
                        maxwidth = maxlen * LABEL_WIDTH(mw);
        }

        /*
         * Clip height
         */
        if (maxht < 0)
                doclip = 0;
        else
                doclip = 1;



        if (xm) {
                Dimension wid, ht;
                XmFontList fl;

                if (rowLabel || colLabel)
                        fl = mw->matrix.label_font_list;
                else
                        fl = mw->matrix.font_list;
                XmStringExtent(fl, xms, &wid, &ht);
                width = wid;

                printf("%s %s %d Hhe:%d Weite:%d\n", __FILE__, __FUNCTION__, __LINE__, ht, wid);


/* Start copied section */
                clipht.width = maxwidth + x;
                clipht.height = font_height + y;
                clipht.x = 0;
                clipht.y = 0;

                if (doclip) {
                        DEBUGOUT(_XbaeDebug
                                 (__FILE__, (Widget) mw, "%s 2 - x %d y %d w %d h %d\n",
                                  __FUNCTION__, x, y, clipht.width, clipht.height));
                        XSetClipRectangles(XtDisplay(mw), gc, 0, 0, &clipht, 1, Unsorted);
                }


                printf("%s %s %d Hhe:%d Weite:%d\n", __FILE__, __FUNCTION__, __LINE__, ht, wid);
                XmStringDraw(XtDisplay(mw), win, fl, xms, gc, x, y, width, alignment,
                             XmSTRING_DIRECTION_L_TO_R, &clipht);

/* End copied section */
        } else {
                /* FIX ME this should ideally all be using XmFontList (aka XmRenderTable) */
                if (font_set) {
                        printf("%s %s %d\n", __FILE__, __FUNCTION__, __LINE__);

                        ink_array = (XRectangle *) XtMalloc(length * sizeof(XRectangle));
                        logical_array = (XRectangle *) XtMalloc(length * sizeof(XRectangle));

                        XmbTextPerCharExtents(font_set, string, length, ink_array, logical_array,
                                              length, &num_chars, NULL, &overall_logical);

                        /*
                         * If the width of the string is greater than the width of this cell,
                         * we need to clip. We don't want to use the server to clip because
                         * it is slow, so we truncate characters if we exceed a cells pixel
                         * width.
                         */
                        if (overall_logical.width > maxwidth) {
                                switch (alignment) {

                                case XmALIGNMENT_CENTER:
                                        {
                                                int startx;
                                                int endx;
                                                int i;
                                                int end;

                                                /*
                                                 * If we are going to draw arrows at both ends, allow for them.
                                                 */


                                                if (arrows) {
                                                        maxwidth -= 2 * mw->matrix.font_width;
                                                        choppedStart = True;
                                                        choppedEnd = True;
                                                }

                                                /*
                                                 * Find limits of cell relative to the origin of the string.
                                                 */
                                                startx =
                                                    overall_logical.x + overall_logical.width / 2 -
                                                    maxwidth / 2;
                                                endx = startx + maxwidth - 1;

                                                /*
                                                 * Find the first character which fits into the cell.
                                                 */
                                                for (i = 0;
                                                     i < num_chars && logical_array[i].x < startx;
                                                     ++i) {
                                                        int cl = mblen(string + start, length);
                                                        start += cl;
                                                        length -= cl;
                                                }

                                                /*
                                                 * Find the last character which fits into the cell.
                                                 * At this point length represents the number of bytes
                                                 * between the end of the cell and the end of the full
                                                 * string. Note that the scan continues from above.
                                                 */
                                                for (end = start;
                                                     i < num_chars
                                                     && (logical_array[i].x +
                                                         logical_array[i].width) < endx; ++i) {
                                                        int cl = mblen(string + end, length);
                                                        end += cl;
                                                        length -= cl;
                                                }

                                                /*
                                                 * Now reset length so that it represents the number of bytes
                                                 * in the string.
                                                 */
                                                length = end - start;

                                                break;
                                        }

                                case XmALIGNMENT_END:
                                        {
                                                int startx;
                                                int i;

                                                /*
                                                 * If we are going to draw arrows at both ends, allow for them.
                                                 */


                                                if (arrows) {
                                                        maxwidth -= mw->matrix.font_width;
                                                        choppedEnd = True;
                                                }

                                                /*
                                                 * Find limits of cell relative to the origin of the string.
                                                 */
                                                startx =
                                                    overall_logical.x + overall_logical.width -
                                                    maxwidth;

                                                /*
                                                 * Find the first character which fits into the cell.
                                                 */
                                                for (i = 0;
                                                     i < num_chars && logical_array[i].x < startx;
                                                     ++i) {
                                                        int cl = mblen(string + start, length);
                                                        start += cl;
                                                        length -= cl;
                                                }

                                                break;
                                        }

                                case XmALIGNMENT_BEGINNING:
                                default:
                                        {
                                                int endx;
                                                int i;
                                                int end;

                                                /*
                                                 * If we are going to draw arrows at both ends, allow for them.
                                                 */

                                                if (arrows) {
                                                        maxwidth -= mw->matrix.font_width;
                                                        choppedStart = True;
                                                }

                                                /*
                                                 * Find limits of cell relative to the origin of the string.
                                                 */
                                                endx = overall_logical.x + maxwidth - 1;

                                                /*
                                                 * Find the last character which fits into the cell.
                                                 * At this point length represents the number of bytes
                                                 * between the end of the cell and the end of the full
                                                 * string.
                                                 */
                                                for (i = 0, end = start;
                                                     i < num_chars
                                                     && (logical_array[i].x +
                                                         logical_array[i].width) < endx; ++i) {
                                                        int cl = mblen(string + end, length);
                                                        end += cl;
                                                        length -= cl;
                                                        choppedEnd = True;
                                                        printf("String %02d => [%c]%d\n", i,
                                                               (char) string[i], (char) string[i]);
                                                }

                                                /*
                                                 * Now reset length so that it represents the number of bytes
                                                 * in the string.
                                                 */
                                                length = end - start;

                                                break;
                                        }
                                }

                                /*
                                 * Having truncated string recalculate extents to find origin
                                 */
                                XmbTextPerCharExtents(font_set, string, length, ink_array,
                                                      logical_array, length, &num_chars, NULL,
                                                      &overall_logical);
                        }
                        /*
                         * We fit inside our cell, so just compute the x of the start of
                         * our string
                         */
                        else {
                                switch (alignment) {

                                case XmALIGNMENT_CENTER:
                                        x += maxwidth / 2 - overall_logical.width / 2;
                                        break;

                                case XmALIGNMENT_END:
                                        x += maxwidth - overall_logical.width;
                                        break;

                                case XmALIGNMENT_BEGINNING:
                                default:
                                        /*
                                         * Leave x alone
                                         */
                                        break;
                                }
                        }

                        /*
                         * Don't worry, XSetForeground is smart about avoiding unnecessary
                         * protocol requests.
                         */
                        XSetForeground(XtDisplay(mw), gc, color);

                        if (arrows && choppedStart) {
                                XPoint points[3];
                                int aw = mw->matrix.font_width >> 1;
                                int ah = mw->matrix.font_height >> 1;
                                points[0].x = points[1].x = x + aw;
                                points[0].y = y + mw->matrix.font_y;
                                points[1].y = y + mw->matrix.font_y + ah;
                                points[2].x = x;
                                points[2].y = y + mw->matrix.font_y + ah / 2;
                                XFillPolygon(XtDisplay(mw), win, gc, points, 3, Convex,
                                             CoordModeOrigin);

                                /* Offset the start point so as to not draw on the triangle */
                                x += FONT_WIDTH(mw);
                        }

                        if (arrows && choppedEnd) {
                                XPoint points[3];
                                int aw = mw->matrix.font_width >> 1;
                                int ah = mw->matrix.font_height >> 1;
                                points[0].x = points[1].x = x + overall_logical.width;
                                points[0].y = y + mw->matrix.font_y;
                                points[1].y = y + mw->matrix.font_y + ah;
                                points[2].x = x + overall_logical.width + aw;
                                points[2].y = y + mw->matrix.font_y + ah / 2;
                                XFillPolygon(XtDisplay(mw), win, gc, points, 3, Convex,
                                             CoordModeOrigin);
                        }

                        /*
                         * Adjust x for origin of string.
                         */
                        x -= overall_logical.x;

                        clipht.height = y + font_height;
                        clipht.width = x + maxwidth;
                        clipht.x = 0;
                        clipht.y = 0;

                        if (doclip) {
                                DEBUGOUT(_XbaeDebug
                                         (__FILE__, (Widget) mw, "%s - x %d y %d w %d h %d\n",
                                          __FUNCTION__, x, y, clipht.width, clipht.height));
                                XSetClipRectangles(XtDisplay(mw), gc, 0, 0, &clipht, 1, Unsorted);
                        }
                        /*
                         * Now draw the string at x starting at char 'start' and of
                         * length 'length'
                         */
                        /*
                         * Scott MacHaffie, June 8, 2001
                         * Fixes for multiple row heights:
                         *
                         * To correctly display right-aligned text, we need to do
                         * the following:
                         *
                         * 1. shift y down to the bottom of the cell: y += maxht - font_height;
                         * 2. call xbaeDrawString() again with start reset to 0 and length
                         * set to: length = strlen(string) - length;
                         * and maxht set to: maxht -= font_height;
                         * Also, the code should bail out if maxht < font_height.
                         * This will draw subsequent rows above the current row.
                         *
                         * To correctly display center-aligned text, we need to figure out
                         * if maxht contains an even number of rows or an odd number of rows.
                         *
                         * If it contains an odd number of rows, then we draw as follows:
                         * 1. The given string is drawn at the center row of the cell.
                         * 2. Then we take the left part of the string (0 to start), and
                         * draw it right aligned from original y to the current y.
                         * 3. Then we take the right part of the string (length to strlen),
                         * and draw it left aligned from the current y + font_height to
                         * maxht.
                         *
                         * If the cell contains an even number of rows, then we
                         * conceptually do the same thing: treat the current string and
                         * row as null, then right-align the first half of the string
                         * in the upper half of the cell and left-align the second half
                         * of the string in the bottom half of the cell.  The middle point
                         * of the string needs to be determined, pixel-wise.
                         *
                         * Given: my time constraints, the fact that we only need left-
                         * aligned text to work, and the amount of work involved in
                         * testing all of these options, I have not implemented these
                         * other two alignments.
                         *
                         * See below for the multi-row left-aligned text.
                         */
                        XmbDrawString(XtDisplay(mw), win, font_set, gc, x, y, &string[start],
                                      length);

                        if (underline && length) {
                                for (i = 0; mw->matrix.underline_width > i; ++i)
                                        XDrawLine(XtDisplay(mw), win, gc, x,
                                                  y + mw->matrix.underline_position + i,
                                                  x + overall_logical.width,
                                                  y + mw->matrix.underline_position + i);
                        }

                        /*
                         * If bold is on, draw the string again offset by 1 pixel (overstrike)
                         */
                        if (bold)
                                XmbDrawString(XtDisplay(mw), win, font_set, gc, x - 1, y,
                                              &string[start], length);
                        if (ink_array)
                                XtFree((char *) ink_array);
                        if (logical_array)
                                XtFree((char *) logical_array);

                        if (start + length < strlen(string) && alignment == XmALIGNMENT_BEGINNING) {
                                /*
                                 * Scott MacHaffie, June 8, 2001
                                 * Fix for multiple row heights
                                 *
                                 * Only the first line of the string has been drawn.
                                 * Now we need to repeat the drawing, having shifted down one line.
                                 */
                                if (XbaeMultiLineCell(mw)) {
                                        if (length > 0)
                                                xbaeDrawString(mw, win, gc, &string[start + length],
                                                               False,
                                                               strlen(&string[start + length]), x,
                                                               y + font_height, maxlen,
                                                               maxht - font_height, alignment,
                                                               arrows, highlight, bold, underline,
                                                               rowLabel, colLabel, color);
                                }
                        }
                } else {
                        width = XTextWidth(font_struct, string, length);

                        /*
                         * If the width of the string is greater than the width of this cell,
                         * we need to clip. We don't want to use the server to clip because
                         * it is slow, so we truncate characters if we exceed a cells pixel
                         * width.
                         */
                        if (width > maxwidth) {
                                switch (alignment) {

                                case XmALIGNMENT_CENTER:
                                        {
                                                int startx = x;
                                                int endx = x + maxwidth - 1;
                                                int newendx;

                                                /*
                                                 * Figure out our x for the centered string.  Then loop
                                                 * and chop characters off the front until we are within
                                                 * the cell.
                                                 *
                                                 * Adjust x, the starting character and the length of the
                                                 * string for each char.
                                                 */
                                                x += maxwidth / 2 - width / 2;
                                                while (x < startx) {
                                                        int cw = charWidth(font_struct,
                                                                           (unsigned char)
                                                                           string[start]);

                                                        x += cw;
                                                        width -= cw;
                                                        length--;
                                                        start++;
                                                        choppedStart = True;
                                                }

                                                /*
                                                 * Now figure out the end x of the string.  Then loop and chop
                                                 * characters off the end until we are within the cell.
                                                 */
                                                newendx = x + width - 1;
                                                while (newendx > endx && *(string + start)) {
                                                        int cw = charWidth(font_struct,
                                                                           (unsigned char)
                                                                           string[start]);

                                                        newendx -= cw;
                                                        width -= cw;
                                                        length--;
                                                        choppedEnd = True;
                                                }

                                                break;
                                        }

                                case XmALIGNMENT_END:
                                        {

                                                /*
                                                 * Figure out our x for the right justified string.
                                                 * Then loop and chop characters off the front until we fit.
                                                 * Adjust x for each char lopped off. Also adjust the starting
                                                 * character and length of the string for each char.
                                                 */
                                                x += maxwidth - width;
                                                while (width > maxwidth) {
                                                        int cw = charWidth(font_struct,
                                                                           (unsigned char)
                                                                           string[start]);

                                                        width -= cw;
                                                        x += cw;
                                                        length--;
                                                        start++;
                                                        choppedStart = True;
                                                }
                                                break;
                                        }

                                case XmALIGNMENT_BEGINNING:
                                default:
                                        /*
                                         * Leave x alone, but chop characters off the end until we fit
                                         */
                                        while (width > maxwidth) {
                                                width -=
                                                    charWidth(font_struct,
                                                              (unsigned char) string[length - 1]);
                                                length--;
                                                choppedEnd = True;
                                        }
                                        break;
                                }
                        }

                        /*
                         * We fit inside our cell, so just compute the x of the start of
                         * our string
                         */
                        else {
                                switch (alignment) {
                                case XmALIGNMENT_CENTER:
                                        x += maxwidth / 2 - width / 2;
                                        break;

                                case XmALIGNMENT_END:
                                        x += maxwidth - width;
                                        break;

                                case XmALIGNMENT_BEGINNING:
                                default:
                                        /*
                                         * Leave x alone
                                         */
                                        break;
                                }
                        }

                        /*
                         * Don't worry, XSetForeground is smart about avoiding unnecessary
                         * protocol requests.
                         */
                        XSetForeground(XtDisplay(mw), gc, color);


                        /* if the show column arrows is set false we dont want to display the arrows
                         */
                        if (!arrows) {
                                choppedEnd = False;
                                choppedStart = False;
                        }

                        /*
                         * If we are going to draw arrows at both ends, allow for them.
                         */
                        if (arrows && choppedEnd) {
                                XPoint points[3];
                                int aw = mw->matrix.font_width >> 1;
                                int ah = mw->matrix.font_height >> 1;
                                points[0].x = points[1].x = x + width - aw;
                                points[0].y = y + mw->matrix.font_y;
                                points[1].y = y + mw->matrix.font_y + ah;
                                points[2].x = x + width;
                                points[2].y = y + mw->matrix.font_y + ah / 2;
                                XFillPolygon(XtDisplay(mw), win, gc, points, 3, Convex,
                                             CoordModeOrigin);

                                /* Reduce the length to allow for our foreign character */
                                length--;
                        }
                        if (arrows && choppedStart) {
                                XPoint points[3];
                                int aw = mw->matrix.font_width >> 1;
                                int ah = mw->matrix.font_height >> 1;
                                points[0].x = points[1].x = x + aw;
                                points[0].y = y + mw->matrix.font_y;
                                points[1].y = y + mw->matrix.font_y + ah;
                                points[2].x = x;
                                points[2].y = y + mw->matrix.font_y + ah / 2;
                                XFillPolygon(XtDisplay(mw), win, gc, points, 3, Convex,
                                             CoordModeOrigin);

                                /* Offset the start point so as to not draw on the triangle */
                                x += mw->matrix.font_width;
                                start++;
                                length--;
                        }

                        /*
                         * Now draw the string at x starting at char 'start' and of length
                         * 'length'
                         */
                        clipht.width = maxwidth + x;
                        clipht.height = font_height + y;
                        clipht.x = 0;
                        clipht.y = 0;

                        if (doclip) {
                                DEBUGOUT(_XbaeDebug
                                         (__FILE__, (Widget) mw, "%s 2 - x %d y %d w %d h %d\n",
                                          __FUNCTION__, x, y, clipht.width, clipht.height));
                                XSetClipRectangles(XtDisplay(mw), gc, 0, 0, &clipht, 1, Unsorted);
                        }
#ifdef NEED_WCHAR
                        if (TWO_BYTE_FONT(mw))
                                XDrawString16(XtDisplay(mw), win, gc, x, y, &string[start], length);
                        else
#endif




                                switch (alignment) {
                                case XmALIGNMENT_BEGINNING:
                                        if (XbaeMultiLineCell(mw)) {
                                                int l = strlen(string);
                                                char *p, *pointer = string;
                                                int count = 0;
/* printf ("\n%s %s %d [%s]\n", __FILE__,__FUNCTION__,__LINE__, pointer);	*/

                                                if ((p = strchr(pointer, 10)) != NULL) {
                                                        if ((char) *pointer != 10) {
                                                                /* 
                                                                 * Zerlege den String in einzelne saubere Abschnitte
                                                                 */
                                                                int l2 = (int) (p - pointer);
                                                                XDrawString(XtDisplay(mw), win, gc,
                                                                            x,
                                                                            y + font_height * count,
                                                                            pointer, l2);
                                                        }

                                                        xbaeDrawString(mw, win, gc, p + 1, False,
                                                                       l - (p - pointer) - 1, x,
                                                                       y + font_height, maxlen,
                                                                       maxht - font_height,
                                                                       alignment, arrows, highlight,
                                                                       bold, underline, rowLabel,
                                                                       colLabel, color);


                                                } else {
/* printf ("%s %s %d Rest:[%s]\n", __FILE__,__FUNCTION__,__LINE__, pointer);	*/
                                                        /* Rest */
                                                        XDrawString(XtDisplay(mw), win, gc, x,
                                                                    y + font_height * count,
                                                                    pointer, strlen(pointer));
                                                }

                                        } else
                                                XDrawString(XtDisplay(mw), win, gc, x, y,
                                                            &string[start], length);
                                        break;

                                default:
                                        XDrawString(XtDisplay(mw), win, gc, x, y, &string[start],
                                                    length);
                                        break;
                                }







                        /*
                         * If bold is on, draw the string again offset by 1 pixel (overstrike)
                         */
                        if (bold)
#ifdef NEED_WCHAR
                                if (TWO_BYTE_FONT(mw))
                                        XDrawString16(XtDisplay(mw), win, gc, x - 1, y,
                                                      &string[start], length);
                                else
#endif
                                        XDrawString(XtDisplay(mw), win, gc, x - 1, y,
                                                    &string[start], length);


                        if (underline && length) {
                                for (i = 0; mw->matrix.underline_width > i; ++i)
                                        XDrawLine(XtDisplay(mw), win, gc, x,
                                                  y + mw->matrix.underline_position + i, x + width,
                                                  y + mw->matrix.underline_position + i);
                        }

#ifdef OKOK
                        if (start + length < strlen(string) && alignment == XmALIGNMENT_BEGINNING) {
                                /*
                                 * Scott MacHaffie, June 8, 2001
                                 * Fix for multiple row heights
                                 *
                                 * Only the first line of the string has been drawn.
                                 * Now we need to repeat the drawing, having shifted
                                 * down one line.
                                 * Please see my comments above about how to fix
                                 * right- and center-aligned text.
                                 */
                                if (XbaeMultiLineCell(mw)) {
                                        if (length > 0)
                                                xbaeDrawString(mw, win, gc, &string[start + length],
                                                               False,
                                                               strlen(&string[start + length]), x,
                                                               y + font_height, maxlen,
                                                               maxht - font_height, alignment,
                                                               arrows, highlight, bold, underline,
                                                               rowLabel, colLabel, color);
                                }
                        }
#endif


                }
        }

        if (doclip) {
                XSetClipMask(XtDisplay(mw), gc, None);
        }
}

static void xbaeComputeCellColors(XbaeMatrixWidget mw, int row, int column, Pixel * fg, Pixel * bg)
{
        Boolean alt = mw->matrix.alt_row_count ? (row / mw->matrix.alt_row_count) % 2 : False;

        *bg = XmUNSPECIFIED_PIXEL;
        *fg = XmUNSPECIFIED_PIXEL;

        /* Compute the background color of the cell: */
        if (!mw->matrix.per_cell) {
                if (mw->matrix.alt_row_count) {
                        if (alt)
                                *bg = mw->matrix.odd_row_background;
                        else
                                *bg = mw->matrix.even_row_background;
                }

                /* fall back to core's background: */
                *bg = (XmUNSPECIFIED_PIXEL == *bg) ? mw->core.background_pixel : *bg;
        } else {
                if (mw->matrix.per_cell[row][column].selected) {
                        if (mw->matrix.reverse_select)
                                *bg = mw->matrix.per_cell[row][column].color;
                        else
                                *bg = mw->matrix.selected_background;

                        /* fall back to manager's foreground: */
                        *bg = (XmUNSPECIFIED_PIXEL == *bg) ? mw->manager.foreground : *bg;
                } else {
                        if (mw->matrix.per_cell[row][column].background != XmUNSPECIFIED_PIXEL)
                                *bg = mw->matrix.per_cell[row][column].background;
                        else if (mw->matrix.alt_row_count) {
                                if (alt)
                                        *bg = mw->matrix.odd_row_background;
                                else
                                        *bg = mw->matrix.even_row_background;
                        }

                        /* fall back to core's background: */
                        *bg = (XmUNSPECIFIED_PIXEL == *bg) ? mw->core.background_pixel : *bg;
                }
        }

        /* Compute the foreground color of the cell: */
        if (!mw->matrix.per_cell) {
                /* fall back to manager's foreground: */
                *fg = mw->manager.foreground;
        } else {
                if (mw->matrix.per_cell[row][column].selected) {
                        if (mw->matrix.reverse_select) {
                                if (mw->matrix.per_cell[row][column].background !=
                                    XmUNSPECIFIED_PIXEL)
                                        *fg = mw->matrix.per_cell[row][column].background;
                                else if (mw->matrix.alt_row_count) {
                                        if (alt)
                                                *fg = mw->matrix.odd_row_background;
                                        else
                                                *fg = mw->matrix.even_row_background;
                                }
                        } else
                                *fg = mw->matrix.selected_foreground;

                        /* fall back to core's background: */
                        *fg = (XmUNSPECIFIED_PIXEL == *fg) ? mw->core.background_pixel : *fg;
                } else {
                        if (mw->matrix.per_cell[row][column].color != XmUNSPECIFIED_PIXEL)
                                *fg = mw->matrix.per_cell[row][column].color;
                        else
                                /* fall back to manager's foreground: */
                                *fg = mw->manager.foreground;
                }
        }
}

static void DrawCellFill(XbaeMatrixWidget mw, Window win, int row, int column, int x, int y)
{
        Display *display = XtDisplay(mw);
        int column_width = COLUMN_WIDTH(mw, column);
        int row_height = ROW_HEIGHT(mw, row);
        int cell_width, cell_height;
        int clear_width, clear_height;
        int space = mw->matrix.cell_shadow_thickness;

        if (!win || mw->matrix.disable_redisplay || mw->matrix.rows == 0 || mw->matrix.columns == 0)
                return;

        assert(row < mw->matrix.rows && column < mw->matrix.columns);

        cell_width = column_width;
        clear_width = 0;
        if (IS_FILL_COLUMN(mw, column)) {
                if (mw->matrix.horz_fill) {
                        cell_width += EMPTY_WIDTH(mw);
                } else {
                        clear_width = EMPTY_WIDTH(mw);
                }
        }
        
        cell_height = row_height;
        clear_height = 0;
        if (IS_FILL_ROW(mw, row)) {
                if (mw->matrix.vert_fill) {
                        cell_height += EMPTY_HEIGHT(mw);
                } else {
                        clear_height = EMPTY_HEIGHT(mw);
                }
        }

        if (clear_width) {
                if (IN_GRID_ROW_MODE(mw)) {
                        /* Don't erase the shadow */
                        clear_width -= (column == mw->matrix.columns - 1) ? space : 0;
                        XClearArea(display, win, x + column_width, y + space, clear_width, cell_height - 2 * space, False);
                } else {
                        XClearArea(display, win, x + column_width, y, clear_width, cell_height, False);
                }
        }
        if (clear_height) {
                if (IN_GRID_COLUMN_MODE(mw)) {
                        /* Don't erase the shadow */
                        clear_height -= (row == mw->matrix.rows - 1) ? space : 0;
                        XClearArea(display, win, x + space, y + row_height, cell_width - 2 * space, clear_height, False);
                } else {
                        XClearArea(display, win, x, y + row_height, cell_width, clear_height, False);
                }
        }
}

static void DrawCellHighlight(XbaeMatrixWidget mw, Window win, GC gc,int row, int column, int x, int y, unsigned char hl)
{
        int column_width = COLUMN_WIDTH(mw, column);
        int row_height = ROW_HEIGHT(mw, row);
        int highlight_width, highlight_height;

        if (!win || mw->matrix.disable_redisplay || mw->matrix.rows == 0 || mw->matrix.columns == 0)
                return;

        assert(row < mw->matrix.rows && column < mw->matrix.columns);

        highlight_width = column_width;
        if (IS_FILL_COLUMN(mw, column)) {
                if (mw->matrix.horz_fill) {
                        highlight_width += EMPTY_WIDTH(mw);
                } else if (IN_GRID_ROW_MODE(mw)) {
                        if (hl & HighlightRow) {
                                highlight_width += EMPTY_WIDTH(mw);
                        }
                }
        }
        
        highlight_height = row_height;
        if (IS_FILL_ROW(mw, row)) {
                if (mw->matrix.vert_fill) {
                        highlight_height += EMPTY_HEIGHT(mw);
                } else if(IN_GRID_COLUMN_MODE(mw)) {
                        if (hl & HighlightColumn) {
                                highlight_height += EMPTY_HEIGHT(mw);
                        }
                }
        }
        xbaeDrawCellHighlight(mw, win, gc, row, column, x, y, highlight_width, highlight_height, hl);
}

void xbaeChangeHighlight(XbaeMatrixWidget mw, int row, int column, unsigned char new_hl)
{
        Pixel bg, fg;
        int x, y;
        Widget w = xbaeRowColToClipXY(mw, row, column, &x, &y);
        Window win = XtWindow(w);
        Display *display = XtDisplay(mw);

        if (!win || mw->matrix.disable_redisplay || mw->matrix.rows == 0 || mw->matrix.columns == 0)
                return;

        assert(row < mw->matrix.rows && column < mw->matrix.columns);

        /*
         * In order to erase the old background, we redraw it with the background color. 
         * Then, if there is some filled space we clear it as there is no background color there
         */
        xbaeComputeCellColors(mw, row, column, &fg, &bg);
        XSetForeground(display, mw->matrix.draw_gc, bg);
        DrawCellHighlight(mw, win, mw->matrix.draw_gc, row, column, x, y, mw->matrix.per_cell[row][column].highlighted);
        DrawCellFill(mw, win, row, column, x, y);
        
        /*
         * Draw the highlight
         */
        mw->matrix.per_cell[row][column].highlighted = new_hl;
        DrawCellHighlight(mw, win, mw->manager.highlight_GC, row, column, x, y, new_hl);
}

void xbaeDrawCell(XbaeMatrixWidget mw, int row, int column)
{
        Pixel bg, fg;
        String string;
        int cell_width, cell_height;
        int shadow_width, shadow_height;
        int x, y;
        Widget w = xbaeRowColToClipXY(mw, row, column, &x, &y);
        Window win = XtWindow(w);
        Display *display = XtDisplay(mw);

        #if 0
        fprintf(stderr,"drawcell: %d,%d", row, column); getchar();
        #endif

        if (!win || mw->matrix.disable_redisplay || mw->matrix.rows == 0 || mw->matrix.columns == 0)
                return;

        assert(row < mw->matrix.rows && column < mw->matrix.columns);

        cell_width = shadow_width = COLUMN_WIDTH(mw, column);
        if (IS_FILL_COLUMN(mw, column)) {
                if (mw->matrix.horz_fill) {
                        cell_width += EMPTY_WIDTH(mw);
                        shadow_width += EMPTY_WIDTH(mw);
                } else if (IN_GRID_ROW_MODE(mw)) {
                        shadow_width += EMPTY_WIDTH(mw);
                }
        }
        
        cell_height = shadow_height = ROW_HEIGHT(mw, row);
        if (IS_FILL_ROW(mw, row)) {
                if (mw->matrix.vert_fill) {
                        cell_height += EMPTY_HEIGHT(mw);
                        shadow_height += EMPTY_HEIGHT(mw);
                } else if(IN_GRID_COLUMN_MODE(mw)) {
                        shadow_height += EMPTY_HEIGHT(mw);
                }
        }

        /*
         * Calculate the forground and background colors
         */

        xbaeComputeCellColors(mw, row, column, &fg, &bg);

        /*
         * Fill the cell's background
         */
        #if 0
        {
            void usleep(unsigned long usec);
            XSetForeground(display, mw->matrix.draw_gc, fg);
            XFillRectangle(XtDisplay(mw), win, mw->matrix.draw_gc, x, y, cell_width, cell_height);
            usleep(50000);
        }
        #endif
         
         
        XSetForeground(display, mw->matrix.draw_gc, bg);
        XFillRectangle(XtDisplay(mw), win, mw->matrix.draw_gc, x, y, cell_width, cell_height);

        if (mw->matrix.per_cell && mw->matrix.per_cell[row][column].widget) {
                xbaeDrawCellWidget(mw, win, row, column, x, y, cell_width, cell_height, 
                                   mw->matrix.per_cell[row][column].widget, bg, fg);
        } else if (!mw->matrix.draw_cell_callback) {

                /* 
                 * ARCAD SYSTEMHAUS
                 */
                String string = xbaeGetCellStringValue(mw, row, column);
                Pixmap pixmap = xbaeGetCellPixmap(mw, row, column);

                /* 
                 * First check for pixmap; drawit
                 */
                if (pixmap != XmUNSPECIFIED_PIXMAP) {
                        Window root_return;
                        int x_return, y_return;
                        unsigned int width_return, height_return;
                        unsigned int border_width_return;
                        unsigned int depth;
                        if (XGetGeometry(XtDisplay(mw), pixmap, &root_return, &x_return, &y_return,
                                         &width_return, &height_return, &border_width_return, &depth)) {
                                /* printf ("%s %s %d pixmap(%d,%d,%d)\n", __FILE__, __FUNCTION__, __LINE__, width, height, depth); */
                                Pixmap mask = xbaeGetCellPixmapMask(mw, row, column);
                                xbaeDrawCellPixmap(mw, win, row, column, x, y, cell_width, cell_height, pixmap, mask,
                                                   width_return, height_return, bg, fg, depth);
                        }
                }

                /*
                 * Then check for text; drawit
                 */
                if (strcmp(string, "")) {
                        xbaeDrawCellString(mw, win, row, column, x, y, cell_width, cell_height, string, bg, fg);
                }
        } else {
                Pixmap pixmap;
                Pixmap mask;
                XbaeCellType type;
                int width_return, height_return;
                int depth;
                
                type = xbaeGetDrawCellValue(mw, row, column, &string, &pixmap, &mask, &width_return,
                                            &height_return, &bg, &fg, &depth);
                if (type == XbaeString)
                        xbaeDrawCellString(mw, win, row, column, x, y, cell_width, cell_height, string, bg, fg);
                else if (type == XbaePixmap)
                        xbaeDrawCellPixmap(mw, win, row, column, x, y, cell_width, cell_height, 
                                           pixmap, mask, width_return, height_return, bg, fg, depth);
        }

        /* 
         * Draw the cell highlight and the cell shadow
         */

        xbaeDrawCellShadow(mw, win, row, column, x, y, shadow_width, shadow_height);
        DrawCellFill(mw, win, row, column, x, y);
        if (mw->matrix.per_cell && mw->matrix.per_cell[row][column].highlighted) {
                DrawCellHighlight(mw, win, mw->manager.highlight_GC, row, column, x, y, mw->matrix.per_cell[row][column].highlighted);
        }
}

XbaeCellType
xbaeGetDrawCellValue(XbaeMatrixWidget mw, int row, int column, String * string, Pixmap * pixmap,
                     Pixmap * mask, int *width, int *height, Pixel * bg, Pixel * fg, int *depth)
{
        XbaeMatrixDrawCellCallbackStruct call_data;

        call_data.reason = XbaeDrawCellReason;
        call_data.event = (XEvent *) NULL;
        call_data.row = row;
        call_data.column = column;
        call_data.width = COLUMN_WIDTH(mw, column) - TEXT_WIDTH_OFFSET(mw) * 2;
        call_data.height = ROW_HEIGHT(mw, row) - TEXT_HEIGHT_OFFSET(mw) * 2;
        call_data.foreground = *fg;
        call_data.background = *bg;
        call_data.depth = 0;

	    /*
	     * Put a reasonable default in before the callback; if the callback does nothing,
	     * then the cell will be shown as usual.
	     */
        if (mw->matrix.per_cell) {
        	    call_data.pixmap = mw->matrix.per_cell[row][column].pixmap;
	            call_data.mask = mw->matrix.per_cell[row][column].mask;
	            call_data.type = (call_data.pixmap != XmUNSPECIFIED_PIXMAP) ? XbaePixmap : XbaeString;
	            call_data.string = mw->matrix.per_cell[row][column].CellValue;
        } else {
        	    call_data.pixmap = XmUNSPECIFIED_PIXMAP;
	            call_data.mask = XmUNSPECIFIED_PIXMAP;
	            call_data.type = XbaeString;
	            call_data.string = NULL;
        }

        /*
	     * Call the callback function
	     */
        XtCallCallbackList((Widget) mw, mw->matrix.draw_cell_callback, (XtPointer) & call_data);

	    /*
	     * Now do what the application told us
	     */
        *pixmap = call_data.pixmap;
        *mask = call_data.mask;
        *string = call_data.string ? call_data.string : "";     /* Handle NULLs */

        if (mw->matrix.reverse_select && mw->matrix.per_cell
            && mw->matrix.per_cell[row][column].selected) {
                /*
                 * if colours were set by the draw cell callback, handle reverse
                 * selection
                 */
                if (*bg != call_data.background) {
                        if (*fg != call_data.foreground)
                                *bg = call_data.foreground;
                        *fg = call_data.background;
                } else if (*fg != call_data.foreground)
                        *bg = call_data.foreground;
        } else {
                *fg = call_data.foreground;
                *bg = call_data.background;
        }
        *width = call_data.width;
        *height = call_data.height;
        *depth = call_data.depth;

        if (call_data.type == XbaePixmap) {
                if (*mask == XmUNSPECIFIED_PIXMAP || *mask == BadPixmap)
                        call_data.mask = 0;

                if (*pixmap == XmUNSPECIFIED_PIXMAP || *pixmap == BadPixmap) {
                        XtAppWarningMsg(XtWidgetToApplicationContext((Widget) mw),
                                        "drawCellCallback", "Pixmap", "XbaeMatrix",
                                        "XbaeMatrix: Bad pixmap passed from drawCellCallback", NULL,
                                        0);
                        call_data.type = XbaeString;
                        *string = "";
                } else if (!*depth) {
                        /*
                         * If we know the depth, width and height don't do a round
                         * trip to find the
                         * geometry
                         */
                        Window root_return;
                        int x_return, y_return;
                        unsigned int width_return, height_return;
                        unsigned int border_width_return;
                        unsigned int depth_return;

                        if (XGetGeometry
                            (XtDisplay(mw), *pixmap, &root_return, &x_return, &y_return,
                             &width_return, &height_return, &border_width_return, &depth_return)) {
                                *width = width_return;
                                *height = height_return;
                                *depth = depth_return;
                        }
                }
        }
        return (call_data.type);
}

/*
 * Draw the column label for the specified column.  Handles labels in
 * fixed and non-fixed columns.
 */
void xbaeDrawColumnLabel(XbaeMatrixWidget mw, int column, Boolean pressed)
{
        int labelX, labelY;
        int buttonX, buttonY;
        int i;
        GC gc = mw->matrix.label_gc;
        Widget w = xbaeRowColToClipXY(mw, -1, column, &buttonX, &buttonY);
        Window win = XtWindow(w);

        Boolean button = mw->matrix.button_labels || (mw->matrix.column_button_labels
                                                      && mw->matrix.column_button_labels[column]);

        DEBUGOUT(_XbaeDebug(__FILE__, (Widget) mw, "%s(%d)\n", __FUNCTION__, column));

        if (COLUMN_WIDTH(mw, column) <= 0) {    /* Arcad */
                return;
        }

        labelX = buttonX + TEXT_X_OFFSET(mw);
        labelY = buttonY + TEXT_HEIGHT_OFFSET(mw);
        
        if (button) {
                XSetForeground(XtDisplay(mw), gc, mw->matrix.button_label_background);
                XFillRectangle(XtDisplay(mw), win, gc, buttonX, buttonY,
                               COLUMN_WIDTH(mw, column), COLUMN_LABEL_HEIGHT(mw));
        } else {
                XClearArea(XtDisplay(mw), win, buttonX, buttonY,
                               COLUMN_WIDTH(mw, column), COLUMN_LABEL_HEIGHT(mw), False);
        }

        XSetForeground(XtDisplay(mw), gc, mw->matrix.column_label_color);
        XSetBackground(XtDisplay(mw), gc, mw->matrix.button_label_background);

        if (mw->matrix.xmcolumn_labels && mw->matrix.xmcolumn_labels[column]) {

                XmString label = mw->matrix.xmcolumn_labels[column];

                if (mw->matrix.column_labels) {
                        labelY += LABEL_HEIGHT(mw) * (mw->matrix.column_label_maxlines - 1);
                }
                
                xbaeDrawString(mw, win, gc, (String) label, True,      /* Hack !! */
                               1,
                               labelX, labelY,
                               mw->matrix.column_widths[column], -1,
                               mw->matrix.column_label_alignments 
                                ? mw->matrix.column_label_alignments[column] 
                                : XmALIGNMENT_BEGINNING,
                               False, False, mw->matrix.bold_labels, False, False, True,
                               mw->matrix.column_label_color);

        } else if (mw->matrix.column_labels && mw->matrix.column_labels[column] && mw->matrix.column_labels[column][0] != '\0') {
                String label = mw->matrix.column_labels[column];

                labelY += LABEL_HEIGHT(mw) * (mw->matrix.column_label_maxlines - mw->matrix.column_label_lines[column].lines)
                          - mw->matrix.label_font_y;

                for (i = 0; i < mw->matrix.column_label_lines[column].lines; i++) {
                        xbaeDrawString(mw, win, gc, label, False,
                                       mw->matrix.column_label_lines[column].lengths[i], 
                                       labelX, labelY, 
                                       mw->matrix.column_widths[column], -1,
                                       mw->matrix.column_label_alignments 
                                        ? mw->matrix.column_label_alignments[column] 
                                        : XmALIGNMENT_BEGINNING,
                                       False, False, mw->matrix.bold_labels, False, False, True,
                                       mw->matrix.column_label_color);

                        labelY += LABEL_HEIGHT(mw);
                        label += mw->matrix.column_label_lines[column].lengths[i] + 1;
                }
        } else {
                /* FIX ME */
        }

        if (button)
                xbaeDrawLabelShadow(mw, win, buttonX, buttonY,
                                   COLUMN_WIDTH(mw, column), COLUMN_LABEL_HEIGHT(mw), 
                                   pressed);
}

/*
 * Draw the row label for the specified row. Handles labels in fixed and
 * non-fixed rows.
 */
void xbaeDrawRowLabel(XbaeMatrixWidget mw, int row, Boolean pressed)
{
        int labelX, labelY;
        int buttonX, buttonY;
        GC gc = mw->matrix.label_gc;
        Widget w = xbaeRowColToClipXY(mw, row, -1, &buttonX, &buttonY);
        Window win = XtWindow(w);

        Boolean button = mw->matrix.button_labels || (mw->matrix.row_button_labels
                                                      && mw->matrix.row_button_labels[row]);

        DEBUGOUT(_XbaeDebug(__FILE__, (Widget) mw, "%s(%d)\n", __FUNCTION__, row));

        labelX = buttonX + TEXT_X_OFFSET(mw);
        labelY = buttonY + TEXT_Y_OFFSET(mw);
        
        if (button) {
                XSetForeground(XtDisplay(mw), gc, mw->matrix.button_label_background);
                XFillRectangle(XtDisplay(mw), win, gc, buttonX, buttonY,
                               ROW_LABEL_WIDTH(mw), ROW_HEIGHT(mw, row));
        } else {
                XClearArea(XtDisplay(mw), win, buttonX, buttonY,
                               ROW_LABEL_WIDTH(mw), ROW_HEIGHT(mw, row), False);
        }

        XSetForeground(XtDisplay(mw), gc, mw->matrix.row_label_color);
        XSetBackground(XtDisplay(mw), gc, mw->matrix.button_label_background);

        if (mw->matrix.row_labels && mw->matrix.row_labels[row] && mw->matrix.row_labels[row][0] != '\0') {
                xbaeDrawString(mw, win, gc, mw->matrix.row_labels[row], False, 
                               strlen(mw->matrix.row_labels[row]), 
                               labelX, labelY, 
                               mw->matrix.row_label_width, /* width clip */
                               -1,      /* height clip */
                               mw->matrix.row_label_alignment, False, False, mw->matrix.bold_labels,
                               False, True, False, mw->matrix.row_label_color);
        }

        if (button)
                xbaeDrawLabelShadow(mw, win, buttonX, buttonY,
                                   ROW_LABEL_WIDTH(mw), ROW_HEIGHT(mw, row), 
                                   pressed);
}
