/*
 * main.c -- Implementation of Scheme interpreter's main ()
 *
 * (C) m.b (Matthias Blume); May 1992, HUB; Jan 1993 PU/CS
 *         Humboldt-University of Berlin
 *         Princeton University, Dept. of Computer Science
 *
 * $Id: main.c,v 2.17 1994/11/12 22:23:06 blume Exp $
 */

# include "rcs.h"
RCSID ("$Id: main.c,v 2.17 1994/11/12 22:23:06 blume Exp $")

# include <setjmp.h>
# include <signal.h>
# include <locale.h>
# include <stdlib.h>
# include <string.h>
# include <errno.h>

# include "main.h"
# include "storage.h"
# include "storext.h"
# include "keyword.h"
# include "reader.h"
# include "except.h"
# include "Cont.h"
# include "Vector.h"
# include "Code.h"
# include "Boolean.h"
# include "Symbol.h"
# include "Cons.h"
# include "String.h"
# include "Corout.h"
# include "Numeric.h"
# include "io.h"
# include "m-except.h"
# include "speccont.h"
# include "type.h"
# include "mainloop.h"

# define ERRORSTRING (errno == 0 ? "unknown reason" : strerror (errno))

static jmp_buf recovery_point;
void *ScmSystemMainLoop = NULL;
static void *user_error_handler = NULL;
static void *user_error_data = NULL;
static void *error_continuation = NULL;
static void *argv_list = NULL;

# define NO_RECOVERY		0
# define ERROR_RECOVERY		1
# define ERROR_RE_RECOVERY	2
# define RESET_RECOVERY		3

# define VSCMBOOT "VSCMBOOT"

# ifndef DEFAULT_BOOTFILE
# define DEFAULT_BOOTFILE ".scheme-boot"
# endif

/*
 * Note that the default definition of dump_prefix is a UNIX-ism and should
 * be overridden by an appropriately chosen replacement if necessary.
 * This code was added according to an idea of Henry Cejtin (NEC).  It
 * provides the opportunity to ``run'' dump files directly using the
 * ``#!'' UNIX kernel hack.
 */
# ifndef DEFAULT_DUMP_PREFIX
# define DEFAULT_DUMP_PREFIX "#!/usr/local/bin/scheme -b"
# endif

const char *dump_prefix = DEFAULT_DUMP_PREFIX;

void ScmRaiseError (void *handler, void *data)
{
  user_error_handler = handler;
  user_error_data = data;
  longjmp (recovery_point, ERROR_RECOVERY);
}

void ScmReRaiseError (void *handler, void *data, void *cont)
{
  user_error_handler = handler;
  user_error_data = data;
  error_continuation = cont;
  longjmp (recovery_point, ERROR_RE_RECOVERY);
}

void ScmRaiseReset (void)
{
  longjmp (recovery_point, RESET_RECOVERY);
}

static
void init (void)
{
  setlocale (LC_ALL, "");
  MEM_root_var (ScmSystemMainLoop);
  MEM_root_var (user_error_handler);
  MEM_root_var (user_error_data);
  MEM_root_var (error_continuation);
  MEM_root_var (argv_list);
  MEM_init_all_modules ();
  ScmInitSymtab ();
  ScmInitializeKeywords ();
  ScmInitCoroutines ();
}

static void build_argv_list (int argc, char **argv)
{
  ScmCons *cons;
  ScmString *strg;
  size_t len;

  argv_list = &ScmNil;
  while (argc--) {
    SCM_NEW (cons, Cons);
    cons->cdr = argv_list;
    cons->car = NULL;
    argv_list = cons;
    len = strlen (argv [argc]);
    SCM_VNEW (strg, String, len, char);
    strncpy (strg->array, argv [argc], len);
    strg->length = len;
    ((ScmCons *) argv_list)->car = strg;
  }
}

static void interrupt (int sig)
{
  signal (SIGINT, interrupt);
  ScmRegisterAsynInterrupt ();
}

extern int main (int argc, char **argv);

MAIN
{
  char *volatile asmfile = NULL;
  char *volatile dumpfile = NULL;
  char *bootfile = NULL;
  FILE *volatile fp;
  void *dcl;
  volatile int initialized = 0;
  volatile int vargc;
  char **volatile vargv;
# ifdef ADJUSTABLE_SFIXNUM_RANGE
  int sfixnum_range = DEFAULT_SFIXNUM_RANGE;
# endif

  OPT
  ARG 'b': PARM
    bootfile = *argv;
    NEXTOPT
  ARG 'd': PARM
    dumpfile = *argv;
    NEXTOPT
  ARG 'a': PARM
    asmfile = *argv;
    NEXTOPT
  ARG 'p': PARM
    dump_prefix = *argv;
    NEXTOPT
  ARG 'r': PARM
# ifdef ADJUSTABLE_SFIXNUM_RANGE
    sfixnum_range = atoi (*argv);
# else
    fprintf (stderr,
	     "warning: command line option -r not available;\n"
	     "  the range of small fixnums is %d\n",
	     DEFAULT_SFIXNUM_RANGE);
# endif
    NEXTOPT
  OTHER
    fatal ("bad command line option");
  ENDOPT

# ifdef ADJUSTABLE_SFIXNUM_RANGE
  ScmInitSFixnums (sfixnum_range);
# endif

  vargc = argc;
  vargv = argv;

  /*
   * I found clock not working correctly without calling it at the
   * very beginning of the program... (ULTRIX ... 4.2 0 RISC).
   * Certainly, this hack doesn't hurt... :-|
   */
# ifndef CLOCK_IS_DEFINITELY_ANSI
  (void) clock ();
# endif

  init ();
  /*
   * Make two levels of ``dummy'' continuations -- this prevents other
   * routines from accessing members of the ``parent'' continuation when
   * there is no ``parent''.
   */
  ScmCC = NULL;
  ScmPushCContinuation (8, SCM_VM_TRAP_CONT, NULL);
  ScmPushCContinuation (64, SCM_VM_TRAP_CONT, NULL);
  PUSH (&ScmFalse);		/* allocate space for argv list */

  if (bootfile == NULL && (bootfile = getenv (VSCMBOOT)) == NULL)
    bootfile = DEFAULT_BOOTFILE;

  if (strcmp (bootfile, "-") != 0) {
    if ((fp = fopen (bootfile, "rb")) == NULL)
      fprintf (stderr, "cannot open bootfile ``%s'' (%s)\n",
	       bootfile, ERRORSTRING);
    else {
      MEM_restore_storage (fp);
      initialized = 1;
      errno = 0;
      if (fclose (fp) == EOF)
	fprintf (stderr, "problems when closing bootfile ``%s'' (%s)\n",
		 bootfile, ERRORSTRING);
    }
  }

  if (asmfile != NULL) {
    if (strcmp (asmfile, "-") == 0)
      fp = stdin;
    else if ((fp = fopen (asmfile, "r")) == NULL) {
      fprintf (stderr, "cannot open asmfile ``%s'' (%s)\n",
	       asmfile, ERRORSTRING);
      fp = NULL;
    }

    if (fp != NULL) {
      switch (setjmp (recovery_point)) {
      case NO_RECOVERY:
	while ((dcl = ScmRead (file_getc, file_ungetc, fp)) != &ScmEof) {
	  dcl = ScmAsmToProcedure (dcl);
	  if (initialized)
	    fatal ("more than one init procedure");
	  else {
	    ScmSystemMainLoop = dcl;
	    ScmPrepareProcedureCall (dcl, 1);
	    initialized = 1;
	  }
	}
	break;
      case ERROR_RECOVERY:
      case ERROR_RE_RECOVERY:
	display_object (user_error_data, file_putc, stderr);
	fputs (" ... during asm -- cannot recover\n", stderr);
	return EXIT_FAILURE;
      default:
	fputs ("system reset during asm -- cannot recover\n", stderr);
	return EXIT_FAILURE;
      }

      if (fp != stdin) {
	errno = 0;
	if (fclose (fp) == EOF)
	  fprintf (stderr, "problems when closing asmfile ``%s'' (%s)\n",
		   asmfile, ERRORSTRING);
      }
    }
  }

  build_argv_list (vargc, vargv);
  SET_TOP (argv_list);

  if (dumpfile != NULL) {
    if ((fp = fopen (dumpfile, "wb")) == NULL) {
      fprintf (stderr, "cannot open dumpfile ``%s'' (%s)\n",
	       dumpfile, ERRORSTRING);
      return EXIT_FAILURE;
    } else {
      MEM_dump_storage (fp, dump_prefix);
      errno = 0;
      if (fclose (fp) == EOF) {
	fprintf (stderr, "problems when closing dumpfile ``%s'' (%s)\n",
		 dumpfile, ERRORSTRING);
	return EXIT_FAILURE;
      }
      return EXIT_SUCCESS;
    }
  } else {

    if (signal (SIGINT, SIG_IGN) != SIG_IGN)
      signal (SIGINT, interrupt);

    if (initialized == 0)
      fatal ("no init procedure");

    switch (setjmp (recovery_point)) {
    case NO_RECOVERY:
      ScmDirtyModeCache (-1);
      break;
    case ERROR_RECOVERY:
      ScmPushCContinuation (1, SCM_VM_ERROR_CONT, NULL);
      error_continuation = ScmCC;
      /* fall through */
    case ERROR_RE_RECOVERY:
      ScmDirtyModeCache (-1);
      /* reset continuation */
      ScmCC = NULL;
      ScmPushCContinuation (8, SCM_VM_TRAP_CONT, NULL);
      ScmPushCContinuation (64, SCM_VM_TRAP_CONT, NULL);
      PUSH (&ScmTrue);		/* this is not the initial startup */
      if (user_error_handler == NULL) {
	display_object (user_error_data, file_putc, stderr);
	fputs ("\n... Standard error recovery (System reset)\n", stderr);
	ScmPrepareProcedureCall (ScmSystemMainLoop, 1);
	user_error_data = NULL;
	error_continuation = NULL;
      } else {
	Push (error_continuation);
	Push (user_error_data);
	ScmPrepareProcedureCall (user_error_handler, 2);
	user_error_handler = user_error_data = error_continuation = NULL;
      }
      break;
    default:
      ScmDirtyModeCache (-1);
      ScmCC = NULL;
      ScmPushCContinuation (8, SCM_VM_TRAP_CONT, NULL);
      ScmPushCContinuation (64, SCM_VM_TRAP_CONT, NULL);
      Push (&ScmTrue);
      ScmPrepareProcedureCall (ScmSystemMainLoop, 1);
      break;
    }
    ScmVM ();
  }
}
