/*
 * except.c -- Implementation of Scheme exception handling
 *
 * (C) m.b (Matthias Blume); May 1992, HUB; Jan 1993 PU/CS
 *         Humboldt-University of Berlin
 *         Princeton University, Dept. of Computer Science
 *
 * $Id: except.c,v 2.10 1994/11/12 22:20:38 blume Exp $
 */

# include "rcs.h"
RCSID ("$Id: except.c,v 2.10 1994/11/12 22:20:38 blume Exp $")

# include <stdio.h>
# include <stdarg.h>
# include <stdlib.h>
# include <string.h>

# include "Cont.h"
# include "String.h"
# include "except.h"
# include "m-except.h"
# include "io.h"
# include "storext.h"
# include "type.h"
# include "mode.h"
# include "tmpstring.h"


static void message (
  const char *prefix, const char *template, va_list ap, putc_proc pp, void *cd)
{
  void *x;
  char *s;
  int i;
  unsigned int u;
  double d;
  char buf [128];

  putc_string (prefix, pp, cd);
  while (template [0]) {
    if (template [0] == '%') {
      switch (template[1]) {
      case '%':
	(* pp) (template[1], cd);
	break;
      case 'w':
	x = va_arg (ap, void *);
	write_object (x, pp, cd);
	break;
      case 'd':
	x = va_arg (ap, void *);
	display_object (x, pp, cd);
	break;
      case 's':
	s = va_arg (ap, char *);
	putc_string (s, pp, cd);
	break;
      case 'i':
	i = va_arg (ap, int);
	sprintf (buf, "%d", i);
	putc_string (buf, pp, cd);
	break;
      case 'u':
	u = va_arg (ap, unsigned);
	sprintf (buf, "%u", u);
	putc_string (buf, pp, cd);
	break;
      case 'f':
	d = va_arg (ap, double);
	sprintf (buf, "%f", d);
	putc_string (buf, pp, cd);
	break;
      default:
	--template;
	break;
      }
      ++template;
    } else
      (* pp) (template [0], cd);
    ++template;
  }
  va_end (ap);
}

static ScmString *message_string (
  const char *prefix, const char *template, va_list ap)
{
  ScmString *string;
  const char *msg;
  size_t len;

  tmpbuf_reset ();
  message (prefix, template, ap, tmpbuf_putc, NULL);
  msg = tmpbuf_get (&len);
  SCM_VNEW (string, String, len, char);
  string->length = len;
  memcpy (string->array, msg, len);
  return string;
}

void warning (const char *text, ...)
{
  va_list ap;
  va_start (ap, text);
  message ("warning: ", text, ap, file_putc, stderr);
  putc ('\n', stderr);
}

void error (const char *text, ...)
{
  void *eh, *tmp;
  va_list ap;

  va_start (ap, text);
  tmp = message_string ("error: ", text, ap);
  eh = ScmMode (SCM_ERROR_HANDLER_MODE);
  ScmRaiseError (eh, tmp);
}

void fatal (const char *text)
{
# ifndef EXIT_VOLATILE
  extern NORETURN void exit (int) NRUTERON;
# endif

  fprintf (stderr, "fatal: %s\n", text);
  exit (EXIT_FAILURE);
}

void restriction (const char *text, ...)
{
  void *eh, *tmp;
  va_list ap;

  va_start (ap, text);
  tmp = message_string (
		"violation of an implementation restriction: ", text, ap);
  eh = ScmMode (SCM_ERROR_HANDLER_MODE);
  ScmRaiseError (eh, tmp);
}

void reset (const char *text)
{
  fprintf (stderr, "System reset: %s\n", text);
  ScmRaiseReset ();
}

void badarg (const char *fname, void *arg)
{
  error ("bad argument to primitive procedure %s: %w", fname, arg);
}
