(define (opt-functor
	 rounds-limit
	 lgi-mod error-mod varset-mod aux-mod foldint-mod
	 optutil-mod hack-mod fv-mod eff-mod)

  (module opt-sig (rounds-limit)

    (use scheme-sig)
    (use ee-sig)
    (use lgi-mod lgi-sig)
    (use error-mod error-sig)
    (use varset-mod varset-sig)
    (use aux-mod aux-sig)
    (use foldint-mod foldint-sig)
    (use optutil-mod optutil-sig)
    (use hack-mod hack-sig)
    (use fv-mod fv-sig)
    (use eff-mod eff-sig)

;;; Phases:
;;;  repeat
;;;      beta redexes -> letrec				BETA
;;;      value propagation				PROPAGATE-VALUES
;;;      constant folding				CONSTANT-FOLD
;;;      loop detection					LOOPIFY
;;;      dead variable elimination			ELIM-DEADVAR
;;;      variable usage count				COUNT-VAR-USAGE
;;;      procedure inlining				INLINE-EXPAND
;;;  until ``satisfactory''
;;;    BETA
;;;    PROPAGATE-VALUES
;;;    CONSTANT-FOLD
;;;    LOOPIFY
;;;    ELIM-DEADVAR
;;;    cse

    (define soft-rounds-limit rounds-limit) ; no inlining after that
    (define hard-rounds-limit (+ rounds-limit 2)) ; stop opt loop after that
    (define fold-integrable-disabled #f)

    (define debug #f)
    (define verification #f)		; perform sanity checks after each step
    (define count-the-lambdas #f)	; give # of lambdas before and after

    (define i-list (integrable 'list))
    (define i-car (integrable 'car))
    (define i-cdr (integrable 'cdr))
    (define (i-car/cdr sy) (if (eq? sy 'car) i-car i-cdr))

    ;; this simply converts beta-redexes into letrecs and integrates integrable
    ;; function calls
    (define (beta exp)
      (case (ee-type exp)
	((app)
	 (let* ((body (ee-body exp))
		(f (car body))
		(args (map beta (cdr body))))
	   (case (ee-type f)
	     ((lambda)
	      (let* ((fbody (ee-body f))
		     (fname (car fbody))
		     (fargs (cadr fbody)))
		(if (not (= (length args) (length fargs)))
		    ((semantic-error fname) "wrong number of arguments"))
		(click! "beta")
		(ee 'letrec
		    (list (map list fargs args)
			  (beta (caddr fbody))))))
	     ((vlambda)
	      (let* ((fbody (ee-body f))
		     (frarg (caddr fbody)))
		(define (build-bl f a)
		  (cond ((null? f)
			 (list (list frarg
				     (if (null? a)
					 (ee 'quote '())
					 (ee 'app (cons i-list a))))))
			((null? a)
			 ((semantic-error (car fbody))
			  "insufficient number of arguments"))
			(else
			 (cons (list (car f) (car a))
			       (build-bl (cdr f) (cdr a))))))
		(click! "beta")
		(ee 'letrec (list (build-bl (cadr fbody) args)
				  (beta (cadddr fbody))))))
	     ((integrable)
	      (let ((fsym (integrable-symbol f)))
		(cond ((pair? fsym)
		       (if (not (= (length args) 1))
			   ((semantic-error (symbol->string fsym))
			    "wrong number of arguments")
			   (do ((l fsym (cdr l))
				(e (car args)
				   (ee 'app (list (i-car/cdr (car l)) e))))
			       ((null? l)
				(click! "integrate")
				e))))
		      (else (ee 'app (cons (beta f) args))))))
	     (else (ee 'app (cons (beta f) args))))))
	(else (generic beta exp))))

    (define (ee-definition exp) (car (ee-info exp)))
    (define (ee-definition! exp def) (set-car! (ee-info exp) def))

    (define (ee-mark exp) (cadr (ee-info exp)))
    (define (ee-mark! exp) (set-car! (cdr (ee-info exp)) #t))

    (define (ee-uses exp) (caddr (ee-info exp)))
    (define (ee-use! exp)
      (let ((x (cddr (ee-info exp))))
	(set-car! x (+ (car x) 1))))

    (define (ee-calls exp) (cadddr (ee-info exp)))
    (define (ee-call! exp)
      (let ((x (cdddr (ee-info exp))))
	(set-car! x (+ (car x) 1))))

    (define (ee-reset-counts! exp)
      (let ((x (cddr (ee-info exp))))
	(set-car! x 0)
	(set-car! (cdr x) 0)))

    (define (ee-init-info! exp)
      (ee-info! exp (list #f #f 0 0)))

    ;; reset all info on local variables
    (define (reset-info exp)
      (visit-locals exp (lambda (e) (local-flag! e #f) (ee-init-info! e)))
      exp)

    ;; reset all info to #f
    (define (cleanup-info exp)

      (define (cleanup exp)
	(ee-info! exp #f)
	(walk cleanup exp))

      (cleanup exp)
      exp)

    ;; follow the definition chain of a local, return outmost local
    (define (master-var loc)
      (let ((d (ee-definition loc)))
	(if (and d (eq? (ee-type d) 'local))
	    (master-var d)
	    loc)))

    ;; set definition info on all locals to defining expression
    (define (propagate-values exp)

      (define (propagate exp)
	(if (eq? (ee-type exp) 'letrec)
	    (for-each (lambda (b)
			(let* ((v (car b))
			       (e (cadr b))
			       (t (ee-type e)))
			  (if (and (eq? t 'local)
				   (eq? (master-var e) v))
			      ((semantic-error "??")
			       "letrec cycle" (local-symbol v) "<->"
			       (local-symbol e)))
			  (ee-definition! v e)
			  (if (or (eq? t 'lambda)
				  (eq? t 'vlambda))
			      (ee-info! e v))))
		      (car (ee-body exp))))
	(walk propagate exp))

      (propagate exp)
      exp)

    ;; make a local variable
    (define (make-new-local s)
      (let ((n (new-local s)))
	(ee-init-info! n)
	n))

    ;; rename a local variable
    (define (new-tmp-local l)
      (make-new-local (local-symbol l)))

    ;; constant folding
    (define (constant-fold exp)

      (define (initial-cf exp) (cf exp '()))

      (define (cf exp allowed)

	(define (same-cf exp) (cf exp allowed))

	(case (ee-type exp)
	  ((local)
	   (let* ((m (master-var exp))
		  (d (ee-definition m)))
	     (if (and d
		      (let ((t (ee-type d)))
			(or (eq? t 'quote)
			    (eq? t 'integrable)
			    (and (= (ee-uses exp) 1)
				 (memq exp allowed)
				 (not (could-depend-on-effect? d))))))
		 (begin (click! "fold local") (alpha d))
		 m)))
	  ((app)
	   (let* ((body (ee-body exp))
		  (f (car body)))
	     (if (integrable? f)
		 (let ((i (integrable-symbol f)))
		   (if (or fold-integrable-disabled
			   (pair? i))
		       (generic same-cf exp)
		       (fold-integrable
			i (map same-cf (cdr body)) f click!
			make-new-local)))
		 (generic same-cf exp))))
	  ((if)
	   (let* ((body (ee-body exp))
		  (cnd (same-cf (car body)))
		  (cns (cadr body))
		  (alt (caddr body)))

	     (define (fold-if cnd cns alt)
	       (cond ((not (eq? (ee-type cnd) 'quote))
		      (ee 'if (list cnd (same-cf cns) (same-cf alt))))
		     ((not (ee-body cnd))
		      (click! "fold if (else)")
		      (same-cf alt))
		     (else
		      (click! "fold if (then)")
		      (same-cf cns))))

	     (case (ee-type cnd)
	       ((app)
		(let* ((cbody (ee-body cnd))
		       (f (car cbody)))
		  (if (and (integrable? f)
			   (eq? (integrable-symbol f) 'not))
		      (begin
			(if (not (= (length cbody) 2))
			    ((semantic-error "??") "wrong # of args for NOT"))
			(click! "if not")
			(fold-if (cadr cbody) alt cns))
		      (fold-if cnd cns alt))))
	       ((letrec)
		(click! "if <-> letrec")
		(let ((cbody (ee-body cnd)))
		  (same-cf
		   (ee 'letrec
		       (list (car cbody)
			     (ee 'if
				 (list (cadr cbody)
				       cns alt)))))))
	       (else
		(fold-if cnd cns alt)))))
	  ((letrec)
	   (let* ((body (ee-body exp))
		  (bl (car body))
		  (e (cadr body)))
	     (define (dont-allow x)
	       (do ((l allowed
		       (if (eq? (caar bl) x) l (cons (caar bl) l)))
		    (bl bl (cdr bl)))
		   ((not (pair? bl)) l)))
	     (ee 'letrec
		 (list
		  (map (lambda (b)
			 (list (car b) (cf (cadr b) (dont-allow (car b)))))
		       bl)
		  (cf e (append (map car bl) allowed))))))
	  ((delay lambda vlambda label)	(generic initial-cf exp))
	  (else (generic same-cf exp))))

      (initial-cf exp))

    ;; generate a new label
    (define new-label
      (let ((lab 0))
	(lambda ()
	  (set! lab (+ lab 1))
	  lab)))

    ;; find loops (direct tail recursions...)
    (define (loopify exp)

      (define (traverse exp t)

	(define (traverse-t exp)
	  (traverse exp t))

	(define (traverse-nt exp)
	  (traverse exp #f))

	;; body of traverse
	(let ((type (ee-type exp))
	      (body (ee-body exp)))

	  (case type
	    ((quote global integrable) exp)
	    ((local) (master-var exp))
	    ((set!)
	     (ee 'set! (list (car body) (traverse-nt (cadr body)))))
	    ((if)
	     (ee 'if (list (traverse-nt (car body))
			   (traverse-t (cadr body))
			   (traverse-t (caddr body)))))
	    ((app)
	     (let ((op (car body)))
	       (if (not (and t
			     (eq? (ee-type op) 'local)
			     (eq? (master-var op) (car t))))
		   (ee 'app (map traverse-nt body))
		   (let* ((lab (cadr t))
			  (d (ee-definition (car t)))
			  (d-body (ee-body d))
			  (d-fname (car d-body))
			  (fargs (cadr d-body))
			  (aargs (cdr body))
			  (lfargs (length fargs))
			  (laargs (length aargs)))
		     (set-cdr! (cdr t) #t)
		     (if (eq? (ee-type d) 'lambda)
			 (if (not (= lfargs laargs))
			     ((semantic-error d-fname)
			      "tail-recursive call with"
			      laargs "arguments (needs" lfargs "arguments)")
			     (begin 
			       (click! "goto lambda")
			       (ee 'goto
				   (cons lab (map traverse-nt aargs)))))
			 (letrec
			     ((r (lambda (f a)
				   (cond
				    ((not (pair? a))
				     (if (pair? f)
					 ((semantic-error d-fname)
					  "tail-recursive call with"
					  laargs "arguments (needs >="
					  lfargs "arguments)")
					 (list (ee 'quote '()))))
				    ((pair? f)
				     (cons (traverse-nt (car a))
					   (r (cdr f) (cdr a))))
				    (else
				     (list
				      (ee 'app
					  (cons i-list
						(map traverse-nt a)))))))))
			   (click! "goto vlambda")
			   (ee 'goto
			       (cons lab (r fargs aargs)))))))))
	    ((letrec)
	     (ee 'letrec
		 (list (map (bindgen traverse-nt) (car body))
		       (traverse-t (cadr body)))))
	    ((lambda)
	     (let ((i (ee-info exp)))
	       (if (not i)
		   (ee 'lambda
		       (list (car body)
			     (cadr body)
			     (traverse-nt (caddr body))))
		   (let* ((lab (new-label))
			  (t (cons i (cons lab #f)))
			  (e (traverse (caddr body) t))
			  (fname (car body))
			  (fargs (cadr body)))
		     (if (not (cddr t))
			 (ee 'lambda (list fname fargs e))
			 (let ((nfargs (map new-tmp-local fargs)))
			   (ee 'lambda
			       (list fname
				     nfargs
				     (ee 'label
					 (list lab
					       (map list fargs nfargs)
					       e))))))))))
	    ((vlambda)	
	     (let ((i (ee-info exp)))
	       (if (not i)
		   (ee 'vlambda
		       (list (car body)
			     (cadr body)
			     (caddr body)
			     (traverse-nt (cadddr body))))
		   (let* ((lab (new-label))
			  (t (cons i (cons lab #f)))
			  (e (traverse (cadddr body) t))
			  (fname (car body))
			  (fargs (cadr body))
			  (frarg (caddr body)))
		     (if (not (cddr t))
			 (ee 'vlambda (list fname fargs frarg e))
			 (let ((nfargs (map new-tmp-local fargs))
			       (nfrarg (new-tmp-local frarg)))
			   (ee 'vlambda
			       (list fname
				     nfargs
				     nfrarg
				     (ee 'label
					 (list lab
					       (cons (list frarg nfrarg)
						     (map list fargs nfargs))
					       e))))))))))
	    ((delay)
	     (ee 'delay (list (car body) (traverse-nt (cadr body)))))
	    ((goto)
	     (ee 'goto (cons (car body) (map traverse-nt (cdr body)))))
	    ((label)
	     (ee 'label
		 (list (car body)
		       (map (bindgen traverse-nt) (cadr body))
		       (traverse-t (caddr body)))))
	    (else
	     (bug "opt (loopify): funny expression type")))))

      ;; body of loopify
      (traverse exp #f))

    ;; eliminate dead variables
    (define (elim-deadvar exp)

      (define (binding-used? b) (ee-mark (car b)))

      (define (include n) (map (bindgen deadvar) n))

      (define (extract l c)
	(split l binding-used?
	       (lambda (used unknown)
		 (if (pair? used)
		     (let ((n (include used)))
		       (extract unknown
				(lambda (used not-used)
				  (c (append n used) not-used))))
		     (c '() unknown)))))

      (define (deadvar exp)
	(case (ee-type exp)
	  ((local) (ee-mark! exp) exp)
	  ((letrec)
	   (let* ((body (ee-body exp))
		  (e (deadvar (cadr body))))
	     (extract
	      (car body)
	      (lambda (used not-used)
		(split
		 not-used (bindapp could-have-effect?)
		 (lambda (may may-not)
		   (let ((keep (append (include may) used)))
		     (extract
		      may-not
		      (lambda (also-keep drop)
			(let ((to-be-kept (append also-keep keep)))
			  (if (pair? drop)
			      (click! "deadvar"))
			  (if (pair? to-be-kept)
			      (ee 'letrec (list to-be-kept e))
			      e)))))))))))
	  (else (generic deadvar exp))))

      (deadvar exp))

    (define (reset-counters exp)
      (visit-locals exp ee-reset-counts!)
      exp)

    ;; counting...
    (define (count-var-usage exp)

      (define (count! exp)
	(case (ee-type exp)
	  ((local) (ee-use! exp))
	  ((app)
	   (let*
	       ((b (ee-body exp))
		(f (car b)))
	     (if (eq? (ee-type f) 'local)
		 (ee-call! f))
	     (for-each count! b)))
	  (else (walk count! exp))))

      (count! exp)
      exp)

    ;; alpha-convert an expression (fresh local variables and labels)
    (define (alpha exp)

      (define (a exp b)

	(define (same-a exp)
	  (a exp b))

	(let ((body (ee-body exp)))
	  (case (ee-type exp)
	    ((local)
	     (cond ((assq exp b) => cdr)
		   (else exp)))
	    ((letrec)
	     (let* ((bl (car body))
		    (vl (map car bl))
		    (nvl (map new-tmp-local vl))
		    (nb (append (map cons vl nvl) b)))
	       (ee 'letrec
		   (list (map (lambda (nv b)
				(list nv (a (cadr b) nb)))
			      nvl bl)
			 (a (cadr body) nb)))))
	    ((lambda)
	     (let* ((vl (cadr body))
		    (nvl (map new-tmp-local vl))
		    (nb (append (map cons vl nvl) b)))
	       (ee 'lambda (list (car body) nvl (a (caddr body) nb)))))
	    ((vlambda)
	     (let* ((vl (cadr body))
		    (rv (caddr body))
		    (nvl (map new-tmp-local vl))
		    (nrv (new-tmp-local rv))
		    (nb (cons (cons rv nrv)
			      (append (map cons vl nvl) b))))
	       (ee 'vlambda (list (car body) nvl nrv (a (cadddr body) nb)))))
	    ((label)
	     (let* ((lab (car body))
		    (bl (cadr body))
		    (vl (map car bl))
		    (nvl (map new-tmp-local vl))
		    (nlab (new-label))
		    (nb (cons (cons lab nlab)
			      (append (map cons vl nvl) b))))
	       (ee 'label
		   (list nlab
			 (map (lambda (v b)
				(list v (same-a (cadr b))))
			      nvl bl)
			 (a (caddr body) nb)))))
	    ((goto)
	     (let ((lab (car body)))
	       (ee 'goto
		   (cons
		    (cond ((assv lab b) => cdr)
			  (else lab))
		    (map same-a (cdr body))))))
	    (else (generic same-a exp)))))

      (a exp '()))

    ;; assess the ``size'' of an expression... (that's a rough heuristic)
    (define (estimate-size var exp)

      ;; charges...
      (let ((c-quote 1)
	    (c-global 1)
	    (c-integrable 1)
	    (c-local 1)
	    (c-set! 3)
	    (c-if 2)
	    (c-arg 1)
	    (c-newloc 1)
	    (c-lambda 3)
	    (c-delay 2)
	    (c-closure 1)
	    (c-letrec 2)
	    (c-label 1)
	    (c-goto 1)
	    (c-call 3)
	    (c-car/cdr 1))

	(define (est-call exp)
	  (if (integrable? exp)
	      (let ((b (integrable-symbol exp)))
		(if (pair? b)
		    (* c-car/cdr (length b))
		    (case b
		      ((car cdr) c-car/cdr)
		      ((not) 1)
		      ((cons) 2)
		      (else c-call))))
	      c-call))

	(define (est-closure exp)
	  (* (length (ee-info exp)) c-closure))

	(define (est-binding b)
	  (+ c-newloc (est (cadr b))))

	(define (est exp)

	  (let ((body (ee-body exp)))
	    (case (ee-type exp)
	      ((quote) c-quote)
	      ((global) c-global)
	      ((integrable) c-integrable)
	      ((local) c-local)
	      ((set!)
	       (+ c-set! (est (cadr body))))
	      ((if)
	       (+ c-if (est (car body)) (est (cadr body)) (est (caddr body))))
	      ((app)
	       (+ (est-call (car body))
		  (* (length (cdr body)) c-arg)
		  (apply + (map est (cdr body)))))
	      ((letrec)
	       (+ c-letrec (apply + (map est-binding (car body)))
		  (est (cadr body))))
	      ((lambda vlambda) (+ c-lambda (est-closure exp)))
	      ((delay) (+ c-delay (est-closure exp)))
	      ((label)
	       (+ c-label
		  (apply + (map est-binding (cadr body)))
		  (est (caddr body))))
	      ((goto)
	       (+ c-goto
		  (apply + (map est (cdr body)))))
	      (else (error "opt: estimate-size: funny expression type")))))

	(cond ((local-flag var))
	      (else (let ((s (est exp)))
		      (local-flag! var s)
		      s)))))

    (define (inline-expand exp)

      (define (inline exp dont)

	(define (same-inline exp) (inline exp dont))

	(case (ee-type exp)
	  ((app)
	   (let* ((body (ee-body exp))
		  (f (car body)))
	     (if (and (eq? (ee-type f) 'local)
		      (not (memq f dont)))
		 (let ((d (ee-definition f))
		       (ucount (ee-uses f))
		       (ccount (ee-calls f)))
		   (define (decide ex)
		     (if (and (not (varset-in? f (freevar ex)))
			      (or (= ucount 1)
				  (let ((s (estimate-size f ex))
					(t1 13)
					(t2 30)
					(n (if (= ucount ccount)
					       ccount
					       (+ ccount 1)))
					(tn 2))
				    (or (<= s t1)
					(and (<= s t2) (<= n tn))))))
			 (let ((a (alpha d)))
			   (click! "inline")
			   (ee 'app
			       (cons a
				     (map same-inline (cdr body)))))
			 (generic same-inline exp)))
		 
		   (if d
		       (case (ee-type d)
			 ((lambda)
			  (decide (caddr (ee-body d))))
			 ((vlambda)
			  (decide (cadddr (ee-body d))))
			 (else (generic same-inline exp)))
		       (generic same-inline exp)))
		 (generic same-inline exp))))
	  ((letrec)
	   (let ((body (ee-body exp)))
	     (ee 'letrec
		 (list (map (lambda (cl)
			      (list (car cl)
				    (inline (cadr cl)
					    (cons (car cl) dont))))
			    (car body))
		       (same-inline (cadr body))))))
	  (else (generic same-inline exp))))

      (inline exp '()))

    (define (verify where exp)

      (define (v e b)
	(case (ee-type e)
	  ((local)
	   (if (not (memq e b))
	       (begin
		 (display "OUT OF SCOPE (")
		 (display where)
		 (display "): ") (write (hack e)) (newline))))
	  ((lambda)
	   (let ((body (ee-body e)))
	     (v (caddr body) (append (cadr body) b))))
	  ((vlambda)
	   (let ((body (ee-body e)))
	     (v (cadddr body) (cons (caddr body) (append (cadr body) b)))))
	  ((letrec)
	   (let* ((body (ee-body e))
		  (nb (append (map car (car body)) b))
		  (vv (lambda (x) (v x nb))))
	     (for-each (bindapp vv) (car body))
	     (vv (cadr body))))
	  ((label)
	   (let ((body (ee-body e)))
	     (for-each (bindapp (lambda (x) (v x b))) (cadr body))
	     (v (caddr body)
		(append (map car (cadr body)) b))))
	  (else (walk (lambda (x) (v x b)) e))))

      (define w
	(let ((m '()))
	  (define (reg! x)
	    (cond ((memq x m)
		   (display "MULTIPLE DECLARATION FOR: ")
		   (write (hack x))
		   (display " (")
		   (display where)
		   (display ")")
		   (newline))
		  (else
		   (set! m (cons x m)))))
	  (lambda (e)
	    (visit-locals e reg!))))

      (if verification
	  (begin
	    (w exp)
	    (v exp '())))
      exp)

    ;; optimizer pipeline (without inline expander)
    (define (pipeline exp)
      (verify
       "ELIM-DEADVAR"
       (elim-deadvar
	(verify
	 "LOOPIFY"
	 (loopify
	  (verify
	   "CONSTANT-FOLD"
	   (constant-fold
	    (count-var-usage
	     (propagate-values
	      (reset-info
	       (verify
		"BETA"
		(beta
		 (verify
		  "ORIG"
		  exp)))))))))))))

    ;; clicks
    (define click!
      (let ((clicks 0)
	    (details '()))

	(define (register! m)
	  (cond ((assv m details)
		 =>
		 (lambda (a) (set-cdr! a (+ (cdr a) 1))))
		(else
		 (set! details (cons (cons m 1) details)))))

	(define (show-and-reset! total)
	  (display "TOTAL CLICKS IN THIS ROUND: ") (write total) (newline)
	  (for-each (lambda (a)
		      (display "  ")
		      (display (car a))
		      (display ": ")
		      (write (cdr a))
		      (newline))
		    details)
	  (set! details '()))

	(lambda (m)
	  (if m
	      (begin
		(if debug
		    (register! m))
		(set! clicks (+ clicks 1)))
	      (let ((r clicks))
		(if debug
		    (show-and-reset! r))
		(set! clicks 0)
		r)))))

    ;; statistics
    (define (statistics-satisfactory?!)
      (zero? (click! #f)))

    ;; optimizer loop
    (define (optloop exp)

      (define (decide n exp)
	(cond ((statistics-satisfactory?!) exp)
	      ((>= n hard-rounds-limit)
	       (if debug
		   (begin (display "-> FINISH...") (newline)))
	       (let ((r (pipeline exp)))
		 (click! #f)
		 r))
	      (else (loop exp (+ n 1)))))

      (define (loop exp n)
	(if debug
	    (begin
	      (display "-> LOOP: ") (write n) (newline)))
	(if (>= n soft-rounds-limit)
	    (decide n (pipeline exp))
	    (decide n (verify
		       "INLINE-EXPAND"
		       (inline-expand
			(count-var-usage
			 (propagate-values
			  (reset-info
			   (pipeline exp)))))))))

      (loop exp 0))

    (define (show-lambda-count exp s)

      (define (count-lambdas exp)
	(let ((n 0))
	  (define (cnt exp)
	    (case (ee-type exp)
	      ((lambda vlambda) (set! n (+ n 1))))
	    (walk cnt exp))
	  (cnt exp)
	  n))

      (if count-the-lambdas
	  (begin
	    (display s)
	    (display " # of LAMBDAs: ")
	    (write (count-lambdas exp))
	    (newline)))
      exp)

    ;; the optimizer...
    (define (opt exp)
      (show-lambda-count exp "INITIAL")
      (show-lambda-count (cleanup-info (optloop exp)) "FINAL"))))
