(define (inspect-functor show-mod primv-mod)
  (module inspect-sig ()

    (use system-sig)
    (use show-mod show-sig)
    (use primv-mod primv-sig)

    (define (inspector cont . opt-level)
      (internal-inspector
       cont
       (if (pair? opt-level)
	   (car opt-level)
	   0)))

    (define (internal-inspector cont first-level)

      (if (not (and (integer? first-level)
		    (>= first-level 0)))
	  (error "inspector: bad level argument"))

      (let ((inspect-ccont-idx 0)
	    (inspect-shared-idx 1)
	    (inspect-stack-idx 2)
	    (inspect-prim-code-idx 3)
	    (inspect-modeid-pc-idx 4)
	    (inspect-mode-const-idx 5)
	    (inspect-env-inst-idx 6))

	(call-with-current-continuation
	 (lambda (exit)

	   (with-output-to-port
	       (standard-port 2)
	     (lambda ()

	       (define-syntax choose
		 (syntax-rules ()
		   ((_ (short long exp ...) ...)
		    (choices (list short long (lambda () exp ...)) ...))))

	       (define (choices . l)
		 (for-each (lambda (x)
			     (display (car x))
			     (display "/"))
			   l)
		 (display "help")
		 (display " --> ")
		 (let ((r (read (standard-port 0))))
		   (letrec ((ll (cons (list "help" "display this explanation"
					    (lambda () (help)))
				      l))
			    (help (lambda ()
				    (for-each
				     (lambda (x)
				       (display "   ")
				       (display (car x))
				       (display " - ")
				       (display (cadr x))
				       (newline))
				     ll))))
		     (if (eof-object? r)
			 (exit 'done-eof))
		     (if (symbol? r)
			 (let ((s (symbol->string r)))
			   (do ((cl ll (cdr cl))
				(c '()
				   (if (and (<= (string-length s)
						(string-length (caar cl)))
					    (string-ci=?
					     s
					     (substring
					      (caar cl) 0 (string-length s))))
				       (cons (caddar cl) c)
				       c)))
			       ((not (pair? cl))
				(if (and (pair? c) (not (pair? (cdr c))))
				    (begin ((car c))
					   (apply choices l)))))))
		     (display "... what?") (newline)
		     (help)
		     (apply choices l))))

	       (define (head s v l)
		 (display s)
		 (if (vector-ref v inspect-shared-idx)
		     (display "!"))
		 (display " {")
		 (write l)
		 (display "}; code = ")
		 (display (vector-ref v inspect-prim-code-idx))
		 (newline))

	       (define (show-stack v)
		 (let ((l (vector-length v)))
		   (do ((i 0 (+ i 1)))
		       ((>= i l) #f)
		     (display (if (> i 9) "     * " "     *  "))
		     (write i)
		     (display ": ")
		     (show (vector-ref v i))
		     (newline))))

	       (define (show-c-frame v l)
		 (head "% C-frame" v l)
		 (display "          mode = ")
		 (cond ((vector-ref v inspect-modeid-pc-idx)
			=>
			(lambda (modeid)
			  (display
			   (case modeid
			     ((0) "[INPUT-PORT: ")
			     ((1) "[OUTPUT-PORT: ")
			     ((2) "[ERROR-HANDLER: ")
			     ((3) "[GC-STRATEGY: ")
			     ((4) "[INTERRUPT-HANDLER: ")
			     ((5) "[TIMER-EXPIRATION-HANDLER: ")
			     (else "[?BOGUS?: ")))
			  (show (vector-ref v inspect-mode-const-idx))
			  (display "]")))
		       (else
			(display "[NONE]")))
		 (newline)
		 (display "          env = ")
		 (show (vector-ref v inspect-env-inst-idx))
		 (newline)
		 (choose
		  ("stack" "show stack contents"
			   (show-stack (vector-ref v inspect-stack-idx)))
		  ("up" "go to caller's frame"
			(loop (+ l 1)))
		  ("down" "go to callee's frame"
			  (loop (- l 1)))
		  ("quit" "exit inspector loop"
			  (exit 'done-c))))

	       (define (show-asm-stat stat cv)
		 (write (car stat))
		 (display "-")
		 (write (cadr stat))
		 (display ": ")
		 (display (caddr stat))
		 (case (caddr stat)
		   ((jump-forward
		     jump-backward true?jump false?jump true?jump+pop
		     false?jump+pop true?jump:pop false?jump:pop pop-true?jump
		     pop-false?jump get-loc get-loc-void
		     get-loc-cell put-loc put-loc-pop
		     put-loc-cell put-loc-cell-pop multi-pop make-cell
		     make-vector allocate-vector call call-exit
		     unchecked-call unchecked-call-exit void vector module
		     register-module fetch-module vec-ref vec-set)
		    (display " ")
		    (write (cadddr stat)))
		   ((take-true take-false take-nil pop exit cons append
			       list->vector car cdr not check ref deref
			       assign validate)
		    (display " -"))
		   ((eq?jump neq?jump eq?jump+pop neq?jump+pop
		     eq?jump:pop neq?jump:pop pop-eq?jump pop-neq?jump
		     eqv?jump neqv?jump eqv?jump+pop neqv?jump+pop
		     eqv?jump:pop neqv?jump:pop pop-eqv?jump pop-neqv?jump
		     memv?jump nmemv?jump memv?jump+pop nmemv?jump+pop
		     memv?jump:pop nmemv?jump:pop pop-memv?jump pop-nmemv?jump)
		    (display " ")
		    (show (vector-ref cv (cadddr stat)))
		    (display ", ")
		    (write (car (cddddr stat))))
		   ((get-vec get-vec-cell put-vec put-vec-pop put-vec-cell
			     put-vec-cell-pop get-vec-void make-closure)
		    (display " ")
		    (write (cadddr stat))
		    (display ", ")
		    (write (car (cddddr stat))))
		   ((get-glob put-glob put-glob-pop take lambda delay
			      unchecked-get-glob)
		    (display " ")
		    (show (vector-ref cv (cadddr stat))))
		   ((fetch)
		    (display
		     (case (cadddr stat)
		       ((0) "-constant ")
		       ((1) "-read-only ")
		       ((2) "-variable ")
		       (else "-??? ")))
		    (show (vector-ref cv (car (cddddr stat)))))
		   ((take-primitive)
		    (display " ")
		    (let ((i (cadddr stat)))
		      (display
		       (cond ((index->primitive i))
			     (else (string-append "?? (" (number->string  i)
						  ")"))))))
		   (else
		    (display "BOGUS: ")
		    (write stat)))
		 (newline))

	       (define (show-asm code start pc cv)
		 (do ((stat (disassemble code start)
			    (disassemble code (cadr stat)))
		      (prev start (car stat))
		      (i 0 (+ i 1)))
		     ((or (not stat) (> i 10))
		      prev)
		   (display (if (= (car stat) pc) "     % " "       "))
		   (show-asm-stat stat cv)))

	       (define (show-s-frame v l)
		 (head "$ S-frame" v l)
		 (let* ((cv (vector-ref v inspect-mode-const-idx))
			(code (vector-ref v inspect-prim-code-idx))
			(pc (vector-ref v inspect-modeid-pc-idx))
			(cstat (vector-ref v inspect-env-inst-idx))
			(start (car cstat)))
		   (display "          pc = ")
		   (write pc)
		   (display ", VM instruction = ")
		   (show-asm-stat cstat cv)
		   (choose
		    ("stack" "show stack contents"
			     (show-stack (vector-ref v inspect-stack-idx)))
		    ("up" "go to caller's frame" (loop (+ l 1)))
		    ("down" "go to callee's frame" (loop (- l 1)))
		    ("first" "show first instruction window"
			     (set! start (show-asm code 0 pc cv)))
		    ("next" "show next instruction window"
			    (set! start (show-asm code start pc cv)))
		    ("quit" "exit inspector loop" (exit 'done-s)))))

	       (define (show-info v l)
		 ((if (vector-ref v inspect-ccont-idx)
		      show-c-frame show-s-frame)
		  v l))

	       (define (loop level)
		 (if (zero? level)
		     (begin (display "*** INNERMOST LEVEL ***")
			    (newline)))
		 (cond ((negative? level)
			(loop 0))
		       ((inspect cont level)
			=>
			(lambda (info)
			  (loop (show-info info level))))
		       (else
			(display "*** NO FURTHER ANCESTORS ***")
			(newline)
			(loop (- level 1)))))

	       (loop first-level)))))))))
