/*
 * Numeric.c -- Implementation of Scheme's numeric data types
 *
 * (C) m.b (Matthias Blume); May 1992, HUB; Jan 1993 PU/CS
 *         Humboldt-University of Berlin
 *         Princeton University, Dept. of Computer Science
 *
 * $Id: Numeric.c,v 2.27 1999/02/09 03:07:39 blume Exp $
 */

# include "rcs.h"
RCSID ("$Id: Numeric.c,v 2.27 1999/02/09 03:07:39 blume Exp $")

# include <stdio.h>
# include <stdlib.h>
# include <string.h>
# include <ctype.h>
# include <assert.h>
# include <locale.h>

# include <math.h>
# include <limits.h>
# include <float.h>
# define PI 3.14159265358979323846264338

# include "Numeric.h"
# include "storage.h"
# include "storext.h"
# include "except.h"
# include "identifier.h"
# include "type.h"
# include "numops.h"

# include "realloc.h"

/* definitions of unsigned integral data type for a ``digit'' in bignums */
typedef unsigned short bigdig;

/* definition of an unsigned integral data type for a ``big register'' that */
/* can hold at least two ``bigdig''s */
typedef unsigned long bigreg;

/* number of bits in bigdig */
# define BIGBITS 16

/* ``bigreg'' 0 & 1 */
# define B0 ((bigreg)0)
# define B1 ((bigreg)1)

/* the base of the number system */
# define BASE (B1<<BIGBITS)

/* a bitmask with all bits of a ``bigdig'' on */
# define BIGMASK (~((~B0)<<BIGBITS))

/* the most significant bit of a ``bigdig'' */
# define BIGMSB (B1<<(BIGBITS-1))

/* an upper bound on the number of decimal digits for n bigdig's */
# define ndig_to_ndec(n) (5*(n))

/* an upper bound on the number of bigdig's per decimal digit */
# define ndec_to_ndig(n) ((21*(n)+99)/100)

/* number of bits in an unsigned long (lower bound) */
# define LONGBITS 32

struct ScmFixnum {
  MEM_descriptor _;
  unsigned int sign	: 1;
  unsigned int length	: 15;
  bigdig dig [1];
};

struct ScmFraction {
  MEM_descriptor _;
  unsigned int sign     : 1;
  unsigned int nlength  : 15;
  unsigned int dlength  : 16;
  bigdig dig [1];
};

struct ScmReal {
  MEM_descriptor _;
  double val;
};

struct ScmComplex {
  MEM_descriptor _;
  double re, im;
};

struct ScmSFixnum {
  MEM_descriptor _;
# ifdef SFIXNUM_FREQUENCY
  unsigned frequency;
# endif
};

# ifdef ADJUSTABLE_SFIXNUM_RANGE

# define SFIXNUM_RANGE_MAX 16384

static int sfixnum_range;	/* int for signed arithmetic */
static struct ScmSFixnum *sfixnum_array;

void ScmInitSFixnums (unsigned range)
{
  /* incl. low, excl. high; 0 and 1 must be included */

  if (range < 2 || range >= SFIXNUM_RANGE_MAX)
    fatal ("bad small fixnum range");

  sfixnum_range = (int) range;	/* cannot overflow */
  if ((sfixnum_array = malloc ((2 * range - 1) * sizeof (ScmSFixnum))) == NULL)
    fatal ("small fixnum range to broad (out of memory)");
}

# else

# define sfixnum_range DEFAULT_SFIXNUM_RANGE
static struct ScmSFixnum sfixnum_array [2 * sfixnum_range - 1];

# endif

# ifdef SFIXNUM_FREQUENCY
static void dump_sfixnum_frequency (void)
{
  unsigned freq;
  int i;
  fputs("sfixnum frequencies", stderr);
  for (i=0; i < 2*sfixnum_range-1; ++i)
    if ((freq = sfixnum_array [i].frequency) > 0)
      fprintf(stderr, "%6d: %u\n", i-sfixnum_range+1, freq);
}
# endif

static void init_sfixnum_array (void)
{
  int i;

  i = 2 * sfixnum_range - 1;
  while (i-- > 0) {
    sfixnum_array [i]._ = ScmType (SFixnum);
# ifdef SFIXNUM_FREQUENCY
    sfixnum_array [i].frequency = 0;
# endif
  }
# ifdef SFIXNUM_FREQUENCY
  atexit (dump_sfixnum_frequency);
# endif
}

# define SINT_VALUE(x) ((int)((ScmSFixnum *)(x)-sfixnum_array)-sfixnum_range+1)
# ifdef SFIXNUM_FREQUENCY
static inline void *MAKE_SINT(int x)
{
  ScmSFixnum *s = sfixnum_array + x + sfixnum_range-1;
  ++s->frequency;
  return (void *) s;
}
# else
# define MAKE_SINT(x) ((void *)(sfixnum_array+(x)+sfixnum_range-1))
# endif

# define NEW_FIXNUM(v,l) \
  (SCM_VNEW (v, Fixnum, l, bigdig), \
   (v)->length = (l))

# define NEW_FRACT(v,ln,ld) \
  (SCM_VNEW (v, Fraction, (ln) + (ld), bigdig), \
   (v)->nlength = (ln), \
   (v)->dlength = (ld))

static void *simple (ScmFixnum *f)
{
  int val;

  if (f->length > 1 || f->dig [0] >= sfixnum_range)
    return f;
  
  val = f->dig [0];
  if (f->sign)
    val = -val;

  return MAKE_SINT (val);
}

static ScmReal *new_real (double val)
{
  ScmReal *x;

  SCM_NEW (x, Real);
  x->val = val;
  return x;
}

static ScmComplex *new_cplx (double re, double im)
{
  ScmComplex *x;

  SCM_NEW (x, Complex);
  x->re = re;
  x->im = im;
  return x;
}


static MEM_cnt int_measure (void *x)
{
  ScmFixnum *a = x;
  return MEM_UNITS (sizeof (ScmFixnum) + (a->length - 1) * sizeof (bigdig));
}

static MEM_cnt fract_measure (void *x)
{
  ScmFraction *a = x;
  return MEM_UNITS (sizeof (ScmFraction) +
		    (a->nlength + a->dlength - 1) * sizeof (bigdig));
}

static void int_dumper (void *vfix, FILE *file);
static void fract_dumper (void *vfract, FILE *file);
static void real_dumper (void *vreal, FILE *file);
static void cplx_dumper (void *vcplx, FILE *file);
static void sint_dumper (void *vsfix, FILE *file);

static void *int_excavator (FILE *file);
static void *fract_excavator (FILE *file);
static void *real_excavator (FILE *file);
static void *cplx_excavator (FILE *file);

static void int_display (void *vint, putc_proc pp, void *cd);
static void fract_display (void *vfract, putc_proc pp, void *cd);
static void real_display (void *vrealt, putc_proc pp, void *cd);
static void cplx_display (void *vcplx, putc_proc pp, void *cd);
static void sint_display (void *vsfix, putc_proc pp, void *cd);


static void
  *the_fixnum_zero = NULL,
  *the_real_zero = NULL,
  *the_fixnum_one = NULL,
  *gcs1 = NULL,
  *gcs2 = NULL,
  *gcs3 = NULL;

static int num_equal (void *x, void *y)
{
  return ScmTypeOf (x) == ScmTypeOf (y) && ScmBinPred (SCM_EQ_PRED, x, y);
}

static double bignum_cvt_real (bigdig *a, int l)
{
  double r, p;
  bigdig b;

  r = 0.0;
  do {
    --l;
    p = r = ldexp (r, BIGBITS);
    r += b = a [l];
  } while (l > 0 && (b == 0 || r != p));
  if (l > 0)
    r = ldexp (r, l * BIGBITS);
  return r;
}

static double int_cvt_real (void *x)
{
  ScmFixnum *a = x;
  double r = bignum_cvt_real (a->dig, a->length);
  return a->sign ? -r : r;
}

static double fract_cvt_real (void *x)
{
  ScmFraction *a = x;
  double r =
    bignum_cvt_real (a->dig, a->nlength) /
      bignum_cvt_real (a->dig + a->nlength, a->dlength);
  return a->sign ? -r : r;
}

static double real_cvt_real (void *x)
{
  ScmReal *a = x;
  return a->val;
}

static double sint_cvt_real (void *x)
{
  return (double) SINT_VALUE (x);
}

static void int_module_init (void);


MEM_VECTOR (Fixnum,
	    0, int_measure,
	    MEM_NULL_iterator, int_dumper, int_excavator, MEM_NULL_revisor,
	    int_module_init, MEM_NULL_task, MEM_NULL_task,
	    EXT (SCM_INTEGER, int_cvt_real,
		 int_display, int_display, num_equal, num_equal));

MEM_VECTOR (Fraction,
	    0, fract_measure,
	    MEM_NULL_iterator, fract_dumper, fract_excavator, MEM_NULL_revisor,
	    MEM_NULL_task, MEM_NULL_task, MEM_NULL_task,
	    EXT (SCM_FRACTION, fract_cvt_real,
		 fract_display, fract_display, num_equal, num_equal));

MEM_VECTOR (Real,
	    MEM_UNITS (sizeof (ScmReal)), MEM_NULL_measure,
	    MEM_NULL_iterator, real_dumper, real_excavator, MEM_NULL_revisor,
	    MEM_NULL_task, MEM_NULL_task, MEM_NULL_task,
	    EXT (SCM_REAL, real_cvt_real,
		 real_display, real_display, num_equal, num_equal));

MEM_VECTOR (Complex,
	    MEM_UNITS (sizeof (ScmComplex)), MEM_NULL_measure,
	    MEM_NULL_iterator, cplx_dumper, cplx_excavator, MEM_NULL_revisor,
	    MEM_NULL_task, MEM_NULL_task, MEM_NULL_task,
	    EXT (SCM_COMPLEX, cannot_cvt_real,
		 cplx_display, cplx_display, num_equal, num_equal));

MEM_VECTOR (SFixnum,
	    0, MEM_NULL_measure,
	    MEM_NULL_iterator, sint_dumper, int_excavator, MEM_NULL_revisor,
	    MEM_NULL_task, MEM_NULL_task, MEM_NULL_task,
	    EXT (SCM_SINTEGER, sint_cvt_real,
		 sint_display, sint_display, num_equal, num_equal));

/*
 * The locale's decimal point...
 */

static char decimal_point;

/*
 * basic operations on arrays of bigdig's...
 */

static void bigdig_copy (bigdig *from, bigdig *to, int l)
{
  memcpy (to, from, (l) * sizeof (bigdig));
}

static void shift_left (bigdig *a, int l, int b)
{
  bigdig u, p;
  int i, rsh;

  u = 0;
  rsh = BIGBITS - b;
  for (i = 0; i < l; i++) {
    p = a [i];
    a [i] = ((p << b) | u) & BIGMASK;
    u = p >> rsh;
  }
}

static void shift_right (bigdig *a, int l, int b)
{
  bigdig p, u;
  int i, lsh;

  u = 0;
  lsh = BIGBITS - b;
  for (i = l - 1; i >= 0; i--) {
    p = a [i];
    a [i] = (p >> b) | u;
    u = (p << lsh) & BIGMASK;
  }
}

/* normalize returns sign or 0 if the number is zero */
static int normalize (bigdig *a, int *length, int sign)
{
  int i;

  i = *length - 1;
  while (a [i] == 0 && i > 0)
    --i;
  *length = i + 1;
  return (i == 0 && a [i] == 0) ? 0 : sign;
}

static int abs_cmp (bigdig *a, bigdig *b, int la, int lb)
{
  int i;

  if (la == lb) {
    for (i = la - 1; i >= 0; i--)
      if (a [i] < b [i])
	return -1;
      else if (a [i] > b [i])
	return 1;
    return 0;
  } else if (la < lb)
    return -1;
  return 1;
}

static void qadd (bigdig *a, bigreg b, bigdig *c, int la)
/* lc = la + 1 */
{
  int i;

  for (i = 0; b && i < la; i++) {
    b += a [i];
    c [i] = b & BIGMASK;
    b >>= BIGBITS;
  }
  for (; i < la; i++)
    c [i] = a [i];
  c [i] = b;			/* cannot be more that 1 */
}

static void sadd (bigdig *a, bigdig *b, bigdig *c, int la, int lb)
/* la >= lb, lc = la + 1 */
{
  bigreg s;
  int i;

  assert (la >= lb);

  s = 0;
  for (i = 0; i < lb; i++) {
    s += (bigreg) a [i] + (bigreg) b [i];
    c [i] = s & BIGMASK;
    s >>= BIGBITS;
  }
  qadd (a + lb, s, c + lb, la - lb);
}

static void qsub (bigdig *a, bigreg b, bigdig *c, int la)
{
  int i;
  bigdig tmp;

  for (i = 0; b && i < la; i++) {
    tmp = b & BIGMASK;
    b >>= BIGBITS;
    if (tmp > a [i]) {
      b++;
      c [i] = BASE + a [i] - tmp;
    } else
      c [i] = a [i] - tmp;
  }
  for (; i < la; i++)
    c [i] = a [i];
}

static void ssub (bigdig *a, bigdig *b, bigdig *c, int la, int lb)
/* la >= lb, lc = la */
{
  int i;
  bigreg s;
  bigdig tmp;

  assert (la >= lb);

  s = 0;
  for (i = 0; i < lb; i++) {
    s += b [i];
    tmp = s & BIGMASK;
    s >>= BIGBITS;
    if (tmp > a [i]) {
      s++;
      c [i] = BASE + a [i] - tmp;
    } else
      c [i] = a [i] - tmp;
  }
  qsub (a + lb, s, c + lb, la - lb);
}

/* returns the length of the result (normalized), sets *sc to the sign of c */
static int add (bigdig *a, bigdig *b, bigdig *c, int la, int lb,
		int sa, int sb, int *sc)
/* handles subtraction by setting sb = !sb */
{
  int cmp, tmpl, len;
  bigreg s;
  bigdig *tmpd;

  if (sa == sb) {
    if (la == 1)
      if (lb == 1) {
	s = (bigreg) a [0] + (bigreg) b [0];
	c [0] = s & BIGMASK;
	c [1] = s >> BIGBITS;
	len = 2;
      } else {
	qadd (b, a [0], c, lb);
	len = lb + 1;
      }
    else
      if (lb == 1) {
	qadd (a, b [0], c, la);
	len = la + 1;
      } else
	if (la > lb) {
	  sadd (a, b, c, la, lb);
	  len = la + 1;
	} else {
	  sadd (b, a, c, lb, la);
	  len = lb + 1;
	}
    *sc = normalize (c, &len, sa);
    return len;
  } else if ((cmp = abs_cmp (a, b, la, lb)) < 0) {
    tmpd = a; a = b; b = tmpd;
    tmpl = la; la = lb; lb = tmpl;
    sa = sb;
  } else if (cmp == 0) {
    c [0] = 0;
    *sc = 0;
    return 1;
  }

  /* subtraction */
  if (la == 1) {
    *sc = ((c [0] = a [0] - b [0]) == 0) ? 0 : sa;
    return 1;
  } else if (lb == 1)
    qsub (a, b [0], c, la);
  else
    ssub (a, b, c, la, lb);
  *sc = normalize (c, &la, sa);
  return la;
}

static void qmul (bigdig *a, bigreg b, bigdig *c, int la)
/* lc = la + 1 */
{
  int i;
  bigreg p;

  assert (b < BASE);

  p = 0;
  for (i = 0; i < la; i++) {
    p += (bigreg) a [i] * b;
    c [i] = p & BIGMASK;
    p >>= BIGBITS;
  }
  c [i] = p;
}

static void smul (bigdig *a, bigdig *b, bigdig *c, int la, int lb)
/* lc = la + lb */
{
  bigreg p;
  int i, k;

  memset (c, 0, (la + lb) * sizeof (bigdig));
  for (i = 0; i < la; i++) {
    p = 0;
    for (k = 0; k < lb; k++) {
      p += (bigreg) c [i + k] + (bigreg) a [i] * (bigreg) b [k];
      c [i + k] = p & BIGMASK;
      p >>= BIGBITS;
    }
    c [i + k] = p;
  }
}

/* returns the length of the result (normalized), sets *sc to the sign of c */
static int mul (bigdig *a, bigdig *b, bigdig *c, int la, int lb,
		int sa, int sb, int *sc)
{
  bigreg p;
  int len, sign;

  sign = (sa == sb) ? 0 : 1;
  len = la + lb;
  if (la == 1)
    if (lb == 1) {
      p = (bigreg) a [0] * (bigreg) b [0];
      c [0] = p & BIGMASK;
      c [1] = p >> BIGBITS;
    } else
      qmul (b, a [0], c, lb);
  else
    if (lb == 1)
      qmul (a, b [0], c, la);
    else
      smul (a, b, c, la, lb);
  *sc = normalize (c, &len, sign);
  return len;
}

static bigdig qdiv (bigdig *a, bigreg b, bigdig *c, bigreg *r, int la)
{
  int i;
  bigreg d;

  assert (b < BASE);

  d = 0;
  for (i = la - 1; i >= 0; i--) {
    d <<= BIGBITS;
    d |= a [i];
    c [i] = d / b;
    d %= b;
  }
  *r = d;
  /* tell caller whether length(quotient) == length(dividend) */
  return c [la - 1];
}

static bigdig qrem (bigdig *a, bigreg b, int la)
{
  int i;
  bigreg d;

  assert (b < BASE);

  d = 0;
  for (i = la - 1; i >= 0; i--) {
    d <<= BIGBITS;
    d |= a [i];
    d %= b;
  }
  return d;
}

/* a proof that the following works: see Knuth Vol. 2 */
static void sdiv (bigdig *a, bigdig *b, bigdig *q, bigdig *r, int la, int lb)
/* lq = la - lb + 1, lr = la + 1 */
/* q == NULL -> only the remainder will be computed */
/* r == a -> in place computation */
{
  int j, k, lshb;
  bigreg tmp, qh, b1, b2, tmp2;

  assert (la >= lb && lb >= 2);

  /* Find shift amount needed to normalize */
  for (lshb = 0, b1 = b [lb - 1]; (b1 & BIGMSB) == 0; b1 <<= 1)
    lshb++;

  /* initialize remainder */
  if (a != r)
    bigdig_copy (a, r, la);
  r [la] = 0;

  /* normalize */
  if (lshb) {
    shift_left (b, lb, lshb);
    shift_left (r, la + 1, lshb);
  }

  /* first two digits of (normalized) divisor */
  b1 = b [lb - 1];
  b2 = b [lb - 2];

  for (j = la; j >= lb; j--) {
    /* Educated guess... */
    tmp = (((bigreg) r [j]) << BIGBITS) | r [j - 1];
    if (r [j] == b1)
      qh = BASE - 1;
    else
      qh = tmp / b1;

    /* Fast verify by looking at next digit -> improving the guess*/
    tmp -= qh * b1; tmp2 = qh * b2;
    while (tmp < BASE && tmp2 > ((tmp << BIGBITS) | r [j - 2]))
      --qh, tmp2 -= b2, tmp += b1;

    /* multiply & subtract */
    tmp = 0;
    for (k = 0; k < lb; k++) {
      tmp += b [k] * qh;
      tmp2 = tmp & BIGMASK;
      tmp >>= BIGBITS;
      if (tmp2 > r [j - lb + k]) {
	r [j - lb + k] += BASE - tmp2;
	++tmp;
      } else
	r [j - lb + k] -= tmp2;
    }

    if (tmp > r [j]) {
      /* did not work out -- qh is still one too big */
      --qh;
      /* add back */
      tmp = 0;
      for (k = 0; k < lb; k++) {
	tmp += (bigreg) r [j - lb + k] + (bigreg) b [k];
	r [j - lb + k] = tmp & BIGMASK;
	tmp >>= BIGBITS;
      }
    }

    r [j] = 0;
    if (q != NULL)
      q [j - lb] = qh;
  }

  /* Denormalize */
  if (lshb) {
    shift_right (b, lb, lshb);
    shift_right (r, lb, lshb);
  }
}

/* ``registers'' for temporary bignums, maintained by provide_bignum */
static bigdig
  *tmp1_dig = NULL,
  *tmp2_dig = NULL,
  *tmp3_dig = NULL,
  *r1_dig = NULL,
  *r2_dig = NULL,
  *tmp_n_dig = NULL,
  *tmp_d_dig = NULL,
  *tmp_r_dig = NULL;
static int
  tmp1_len = 0,
  tmp2_len = 0,
  tmp3_len = 0,
  r1_len = 0,
  r2_len = 0,
  tmp_n_len = 0,
  tmp_d_len = 0,
  tmp_r_len = 0;

static bigdig **all_tmp_regs [] = {
  &tmp1_dig, &tmp2_dig, &tmp3_dig,
  &r1_dig, &r2_dig,
  &tmp_n_dig, &tmp_d_dig, &tmp_r_dig,
};

static bigdig *provide_bignum (bigdig *dig, int *l, int need)
{
  if (need > *l) {
    dig = REALLOC (dig, need * sizeof (bigdig));
    if (dig == NULL)
      reset ("Out of memory (provide_bigdig)");
    *l = need;
  }
  return dig;
}

# define PROVIDE_BIGNUM(v,l) (v##_dig = provide_bignum (v##_dig, &v##_len, l))

static bigdig *gcd2 (bigdig *a, bigdig *b, int la, int lb, int *lc)
/* computes the gcd of a and b, returns a pointer to static result area */
/* the length of the result is returned in *lc */
{
  /*
   * with thanks to Henry Cejtin... (for finding and fixing a nasty bug)
   */
  bigdig *bigt, digt, diga, digb;
  int lent;

  if (abs_cmp (a, b, la, lb) < 0) {
    bigt = a;
    lent = la;
    a = b;
    la = lb;
    b = bigt;
    lb = lent;
  }
  PROVIDE_BIGNUM (r1, la + 1);
  PROVIDE_BIGNUM (r2, lb + 1);
  bigdig_copy (a, r1_dig, la);
  bigdig_copy (b, r2_dig, lb);
  a = r1_dig;
  b = r2_dig;
  while (lb > 1) {
    /*
     * Special case: no quotient, in place.
     */
    sdiv (a, b, NULL, a, la, lb);
    (void) normalize (a, &la, 0);
    bigt = a;
    lent = la;
    a = b;
    la = lb;
    b = bigt;
    lb = lent;
  }
  digb = b [0];
  if (digb != 0) {
    digt = qrem (a, digb, la);
    diga = digb;
    digb = digt;
    while (digb != 0) {
      digt = diga % digb;
      diga = digb;
      digb = digt;
    }
    a [0] = diga;
    la = 1;
  }
  *lc = la;
  return a;
}

void *ScmNewRealComplex (double re, double im)
{
  return im == 0.0 ? (void *) new_real (re) : (void *) new_cplx (re, im);
}

# define new_real_cplx ScmNewRealComplex

/*
 * Implementation of various arithmetic functions (called from type
 * dispatch in storext.c)
 */

static double angle (double im, double re)
{
  if (im == 0.0 && re == 0.0)
    return 0.0;
  return atan2 (im, re);
}

int sint_sint_eq (void *x, void *y, const char *s)
{
  return x == y;
}

int sint_sint_cmp (void *x, void *y, const char *s)
{
  ScmSFixnum *a = x, *b = y;

  return (a < b) ? -1 : (a > b) ? 1 : 0;
}

int sint_int_cmp (void *x, void *y, const char *s)
{
  int a = SINT_VALUE (x);
  ScmFixnum *b = y;

  if (a < 0)
    if (b->sign)
      return 1;
    else
      return -1;
  else
    if (b->sign)
      return 1;
    else
      return -1;
}

int sint_fract_cmp (void *x, void *y, const char *s)
{
  int a = SINT_VALUE (x);
  ScmFraction *b = y;
  int rev, tmp;

  if (a < 0)
    if (b->sign) {
      rev = 1;
      a = -a;
    } else
      return -1;
  else
    if (b->sign)
      return 1;
    else
      rev = 0;

  tmp = 1 + b->dlength;  /* axe */
  PROVIDE_BIGNUM (tmp1, tmp);
  qmul (b->dig + b->nlength, a, tmp1_dig, b->dlength);
  (void) normalize (tmp1_dig, &tmp, 0);
  tmp = abs_cmp (tmp1_dig, b->dig, tmp, b->nlength);
  return rev ? -tmp : tmp;
}

int sint_real_eq (void *x, void *y, const char *s)
{
  double a = SINT_VALUE (x);
  ScmReal *b = y;
  return a == b->val;
}

int sint_real_cmp (void *x, void *y, const char *s)
{
  double a = SINT_VALUE (x);
  ScmReal *b = y;
  return a < b->val ? -1 : a == b->val ? 0 : 1;
}

int int_sint_cmp (void *x, void *y, const char *s)
{
  return -sint_int_cmp (y, x, s);
}

int int_int_eq (void *x, void *y, const char *s)
{
  ScmFixnum *a = x, *b = y;

  return
    a->sign == b->sign &&
      abs_cmp (a->dig, b->dig, a->length, b->length) == 0;
}

int int_int_cmp (void *x, void *y, const char *s)
{
  ScmFixnum *a = x, *b = y;

  if (a->sign)
    return b->sign
      ? abs_cmp (b->dig, a->dig, b->length, a->length)
      : -1;
  else
    return b->sign
      ? 1
      : abs_cmp (a->dig, b->dig, a->length, b->length);
}

int int_fract_cmp (void *x, void *y, const char *s)
{
  ScmFixnum *a = x;
  ScmFraction *b = y;
  int lc, sc, rev;

  if (a->sign)
    if (b->sign)
      rev = 1;
    else
      return -1;
  else
    if (b->sign)
      return 1;
    else
      rev = 0;

  PROVIDE_BIGNUM (tmp1, a->length + b->dlength);
  lc = mul (a->dig, b->dig + b->nlength, tmp1_dig, a->length, b->dlength,
	    0, 0, &sc);
  sc = abs_cmp (tmp1_dig, b->dig, lc, b->nlength);
  return rev ? -sc : sc;
}

int int_real_eq (void *x, void *y, const char *s)
{
  double a = int_cvt_real (x);
  ScmReal *b = y;
  return a == b->val;
}

int int_real_cmp (void *x, void *y, const char *s)
{
  double a = int_cvt_real (x);
  ScmReal *b = y;
  return a < b->val ? -1 : a == b->val ? 0 : 1;
}

int fract_sint_cmp (void *x, void *y, const char *s)
{
  return -sint_fract_cmp (y, x, s);
}

int fract_int_cmp (void *x, void *y, const char *s)
{
  return -int_fract_cmp (y, x, s);
}

int fract_fract_eq (void *x, void *y, const char *s)
{
  ScmFraction *a = x, *b = y;
  return
    a->sign == b->sign &&
    abs_cmp (a->dig, b->dig, a->nlength, b->nlength) == 0 &&
    abs_cmp (a->dig + a->nlength, b->dig + b->nlength,
	     a->dlength, b->dlength) == 0;
}

int fract_fract_cmp (void *x, void *y, const char *s)
{
  ScmFraction *a = x, *b = y;
  int rev = 0, lc1, lc2, sc;
  if (a->sign)
    if (b->sign)
      rev = 1;
    else
      return -1;
  else
    if (b->sign)
      return 1;
  PROVIDE_BIGNUM (tmp1, a->nlength + b->dlength);
  PROVIDE_BIGNUM (tmp2, b->nlength + a->dlength);
  lc1 = mul (a->dig, b->dig + b->nlength, tmp1_dig, a->nlength, b->dlength,
	     0, 0, &sc);
  lc2 = mul (a->dig + a->nlength, b->dig, tmp2_dig, a->dlength, b->nlength,
	     0, 0, &sc);
  sc = abs_cmp (tmp1_dig, tmp2_dig, lc1, lc2);
  return rev ? -sc : sc;
}

int fract_real_eq (void *x, void *y, const char *s)
{
  double a = fract_cvt_real (x);
  ScmReal *b = y;
  return a == b->val;
}

int fract_real_cmp (void *x, void *y, const char *s)
{
  double a = fract_cvt_real (x);
  ScmReal *b = y;
  return a < b->val ? -1 : a == b->val ? 0 : 1;
}

int real_sint_eq (void *x, void *y, const char *s)
{
  return sint_real_eq (y, x, s);
}

int real_sint_cmp (void *x, void *y, const char *s)
{
  return -sint_real_cmp (y, x, s);
}

int real_int_eq (void *x, void *y, const char *s)
{
  return int_real_eq (y, x, s);
}

int real_int_cmp (void *x, void *y, const char *s)
{
  return -int_real_cmp (y, x, s);
}

int real_fract_eq (void *x, void *y, const char *s)
{
  return fract_real_eq (y, x, s);
}

int real_fract_cmp (void *x, void *y, const char *s)
{
  return -fract_real_cmp (y, x, s);
}

int real_real_eq (void *x, void *y, const char *s)
{
  ScmReal *a = x, *b = y;
  return a->val == b->val;
}

int real_real_cmp (void *x, void *y, const char *s)
{
  ScmReal *a = x, *b = y;
  return a->val < b->val ? -1 : a->val == b->val ? 0 : 1;
}

int cplx_cplx_eq (void *x, void *y, const char *s)
{
  ScmComplex *a = x, *b = y;
  return a->re == b->re && a->im == b->im;
}

void *sint_sint_add (void *x, void *y, const char *s)
{
  return ScmLongToNumber ((long) SINT_VALUE (x) + (long) SINT_VALUE (y));
}

void *sint_sint_sub (void *x, void *y, const char *s)
{
  return ScmLongToNumber ((long) SINT_VALUE (x) - (long) SINT_VALUE (y));
}

void *sint_sint_mul (void *x, void *y, const char *s)
{
  return ScmLongToNumber ((long) SINT_VALUE (x) * (long) SINT_VALUE (y));
}

static int gcd (int x, int y)	/* 0 <= x < y */
{
  int r;

  while (x != 0)
    r = y % x,
    y = x,
    x = r;

  return y;
}

static int reduce (int n, int d, int *rn, int *rd, const char *s)
{
  int sign = 0, tmp;

  if (d == 0)
    error ("%s: zero divide", s);

  if (n < 0)
    n = -n,
    sign = 1 - sign;
  if (d < 0)
    d = -d,
    sign = 1 - sign;

  tmp = n < d ? gcd (n, d) : gcd (d, n);
  *rn = n / tmp;
  *rd = d / tmp;

  return sign;
}

void *sint_sint_div (void *x, void *y, const char *s)
{
  int n, d, sign;
  ScmFraction *f;

  sign = reduce (SINT_VALUE (x), SINT_VALUE (y), &n, &d, s);
  if (n == 0)
    return the_fixnum_zero;
  if (d == 1)
    return MAKE_SINT (n);
  NEW_FRACT (f, 1, 1);
  f->sign = sign;
  f->dig [0] = n;
  (f->dig + 1) [0] = d;
  return f;
}

static void *sint_int_addsub (int a, ScmFixnum *b, int subtract, int negate)
{
  int sa, sb, sc, lb = b->length, lc;
  ScmFixnum *c;

  if (a < 0)
    a = -a,
    sa = 1;
  else
    sa = 0;

  sb = b->sign;
  if (subtract)
    sb = 1 - sb;

  lc = (sa == sb) ? lb + 1 : lb;

  gcs1 = b;
  NEW_FIXNUM (c, lc);
  b = gcs1;
  gcs1 = NULL;

  if (sa == sb)
    qadd (b->dig, a, c->dig, lb);
  else
    qsub (b->dig, a, c->dig, lb);

  sc = normalize (c->dig, &lc, sb);
  c->length = lc;
  c->sign = negate ? 1 - sc : sc;
  return simple (c);
}

void *sint_int_add (void *x, void *y, const char *s)
{
  return sint_int_addsub (SINT_VALUE (x), y, 0, 0);
}

void *sint_int_sub (void *x, void *y, const char *s)
{
  return sint_int_addsub (SINT_VALUE (x), y, 1, 0);
}

void *sint_int_mul (void *x, void *y, const char *s)
{
  int a = SINT_VALUE (x);
  ScmFixnum *b = y, *c;
  int sign = b->sign;
  int lb = b->length, lc = lb + 1;

  if (a < 0)
    a = -a,
    sign = 1 - sign;

  gcs1 = b;
  NEW_FIXNUM (c, lc);
  b = gcs1;
  gcs1 = NULL;
  qmul (b->dig, a, c->dig, lb);
  c->sign = normalize (c->dig, &lc, sign);
  c->length = lc;
  return simple (c);
}

static void *
  lowest_term_fract (bigdig *n, bigdig *d, int ln, int ld, int sign,
		     const char *s)
{
  int lgcd;
  bigdig *gcd = gcd2 (n, d, ln, ld, &lgcd);
  bigreg r;

  if (ld == 1 && d [0] == 0)
    error ("%s: zero divide", s);
  if (ln == 1 && n [0] == 0)
    return exact_zero ();

  if (lgcd > 1) {
    PROVIDE_BIGNUM (tmp_n, ln - lgcd + 1);
    PROVIDE_BIGNUM (tmp_r, ln + 1);
    sdiv (n, gcd, tmp_n_dig, tmp_r_dig, ln, lgcd);
    n = tmp_n_dig; ln -= lgcd - 1;
    (void) normalize (n, &ln, 0);
    PROVIDE_BIGNUM (tmp_d, ld - lgcd + 1);
    PROVIDE_BIGNUM (tmp_r, ld + 1);
    sdiv (d, gcd, tmp_d_dig, tmp_r_dig, ld, lgcd);
    d = tmp_d_dig; ld -= lgcd - 1;
    (void) normalize (d, &ld, 0);
  } else if (gcd [0] != 1) {
    PROVIDE_BIGNUM (tmp_n, ln);
    if (qdiv (n, gcd [0], tmp_n_dig, &r, ln) == 0)
      --ln;
    n = tmp_n_dig;
    PROVIDE_BIGNUM (tmp_d, ld);
    if (qdiv (d, gcd [0], tmp_d_dig, &r, ld) == 0)
      --ld;
    d = tmp_d_dig;
  } else {
    /* this is necessary to avoid being confused by garbage collection */
    PROVIDE_BIGNUM (tmp_n, ln);
    bigdig_copy (n, tmp_n_dig, ln);
    PROVIDE_BIGNUM (tmp_d, ld);
    bigdig_copy (d, tmp_d_dig, ld);
    n = tmp_n_dig;
    d = tmp_d_dig;
  }
  if (ld == 1 && d [0] == 1) {
    ScmFixnum *f;
    NEW_FIXNUM (f, ln);
    bigdig_copy (n, f->dig, ln);
    f->sign = sign;
    return simple (f);
  } else {
    ScmFraction *f;
    NEW_FRACT (f, ln, ld);
    f->sign = sign;
    bigdig_copy (n, f->dig, ln);
    bigdig_copy (d, f->dig + ln, ld);
    return f;
  }
}

void *sint_int_div (void *x, void *y, const char *s)
{
  bigdig abd [1];
  int a = SINT_VALUE (x);
  ScmFixnum *b = y;
  int sign = b->sign;

  if (a < 0)
    abd [0] = -a,
    sign = 1 - sign;
  else
    abd [0] = a;

  return lowest_term_fract (abd, b->dig, 1, b->length, sign, s);
}

static void *sint_fract_addsub (void *x, void *y, const char *s,
				int sub1, int sub2)
{
  int a = SINT_VALUE (x);
  ScmFraction *b = y;
  int len, sign;

  if (a < 0)
    a = -1,
    sub1 = !sub1;

  if (b->sign)
    sub2 = !sub2;

  len = 1 + b->dlength;
  PROVIDE_BIGNUM (tmp1, len);
  qmul (b->dig + b->nlength, a, tmp1_dig, b->dlength);
  (void) normalize (tmp1_dig, &len, 0);
  PROVIDE_BIGNUM (tmp2, (len > b->nlength ? len : b->nlength) + 1);
  len = add (tmp1_dig, b->dig, tmp2_dig, len, b->nlength, sub1, sub2, &sign);
  return lowest_term_fract (tmp2_dig, b->dig + b->nlength, len, b->dlength,
			    sign, s);
}

void *sint_fract_add (void *x, void *y, const char *s)
{
  return sint_fract_addsub (x, y, s, 0, 0);
}

void *sint_fract_sub (void *x, void *y, const char *s)
{
  return sint_fract_addsub (x, y, s, 0, 1);
}

void *sint_fract_mul (void *x, void *y, const char *s)
{
  int a = SINT_VALUE (x);
  ScmFraction *b = y;
  int len, sign;

  if (a < 0)
    a = -a,
    sign = 1;
  else
    sign = 0;

  len = 1 + b->nlength;
  PROVIDE_BIGNUM (tmp1, len);
  qmul (b->dig, a, tmp1_dig, b->nlength);
  sign = normalize (tmp1_dig, &len, sign);
  return lowest_term_fract (tmp1_dig, b->dig + b->nlength, len, b->dlength,
			    b->sign != sign, s);
}

void *sint_fract_div (void *x, void *y, const char *s)
{
  int a = SINT_VALUE (x);
  ScmFraction *b = y;
  int len, sign;

  if (a < 0)
    a = -a,
    sign = 1;
  else
    sign = 0;

  len = 1 + b->dlength;
  PROVIDE_BIGNUM (tmp1, len);
  qmul (b->dig + b->nlength, a, tmp1_dig, b->dlength);
  sign = normalize (tmp1_dig, &len, sign);
  return lowest_term_fract (tmp1_dig, b->dig, len, b->nlength,
			    b->sign != sign, s);
}

void *int_sint_add (void *x, void *y, const char *s)
{
  return sint_int_addsub (SINT_VALUE (y), x, 0, 0);
}

void *int_sint_sub (void *x, void *y, const char *s)
{
  return sint_int_addsub (SINT_VALUE (y), x, 1, 1);
}

void *int_sint_mul (void *x, void *y, const char *s)
{
  return sint_int_mul (y, x, s);
}

void *int_sint_div (void *x, void *y, const char *s)
{
  bigdig bbd [1];
  int b = SINT_VALUE (y);
  ScmFixnum *a = x;
  int sign = a->sign;

  if (b < 0)
    bbd [0] = -b,
    sign = 1 - sign;
  else
    bbd [0] = b;

  return lowest_term_fract (a->dig, bbd, a->length, 1, sign, s);
}

void *int_int_add (void *x, void *y, const char *s)
{
  ScmFixnum *a = x, *b = y, *c;
  int la = a->length, lb = b->length, lc = (la > lb ? la : lb) + 1;
  int sc;

  gcs1 = a; gcs2 = b;
  NEW_FIXNUM (c, lc);
  a = gcs1; b = gcs2;
  gcs1 = gcs2 = NULL;
  c->length = add (a->dig, b->dig, c->dig, la, lb, a->sign, b->sign, &sc);
  c->sign = sc;
  return simple (c);
}

void *int_int_sub (void *x, void *y, const char *s)
{
  ScmFixnum *a = x, *b = y, *c;
  int la = a->length, lb = b->length, lc = (la > lb ? la : lb) + 1;
  int sc;

  gcs1 = a; gcs2 = b;
  NEW_FIXNUM (c, lc);
  a = gcs1; b = gcs2;
  gcs1 = gcs2 = NULL;
  c->length = add (a->dig, b->dig, c->dig, la, lb, a->sign, !b->sign, &sc);
  c->sign = sc;
  return simple (c);
}

void *int_int_mul (void *x, void *y, const char *s)
{
  ScmFixnum *a = x, *b = y, *c;
  int la = a->length, lb = b->length, lc = la + lb;
  int sc;

  gcs1 = a; gcs2 = b;
  NEW_FIXNUM (c, lc);
  a = gcs1; b = gcs2;
  gcs1 = gcs2 = NULL;
  c->length = mul (a->dig, b->dig, c->dig, la, lb, a->sign, b->sign, &sc);
  c->sign = sc;
  return c;			/* must be int, cannot be sint */
}

void *int_int_div (void *x, void *y, const char *s)
{
  ScmFixnum *a = x, *b = y;
  int sign = a->sign ? !b->sign : b->sign;
  return lowest_term_fract (a->dig, b->dig, a->length, b->length, sign, s);
}

static void *
  int_fract_addsub (void *x, void *y, const char *s, int sub1, int sub2)
{
  ScmFixnum *a = x;
  ScmFraction *b = y;
  int lc, sc;

  PROVIDE_BIGNUM (tmp1, a->length + b->dlength);
  lc = mul (a->dig, b->dig + b->nlength, tmp1_dig, a->length, b->dlength,
	    0, 0, &sc);
  PROVIDE_BIGNUM (tmp2, (lc > b->nlength ? lc : b->nlength) + 1);
  lc = add (tmp1_dig, b->dig, tmp2_dig, lc, b->nlength,
	    sub1 ? !a->sign : a->sign, sub2 ? !b->sign : b->sign, &sc);
  return lowest_term_fract (tmp2_dig, b->dig + b->nlength, lc, b->dlength,
			    sc, s);
}

void *int_fract_add (void *x, void *y, const char *s)
{
  return int_fract_addsub (x, y, s, 0, 0);
}

void *int_fract_sub (void *x, void *y, const char *s)
{
  return int_fract_addsub (x, y, s, 0, 1);
}

void *int_fract_mul (void *x, void *y, const char *s)
{
  ScmFixnum *a = x;
  ScmFraction *b = y;
  int lc, sc;

  PROVIDE_BIGNUM (tmp1, a->length + b->nlength);
  lc = mul (a->dig, b->dig, tmp1_dig, a->length, b->nlength,
	    a->sign, b->sign, &sc);
  return lowest_term_fract (tmp1_dig, b->dig + b->nlength, lc, b->dlength,
			    sc, s);
}

void *int_fract_div (void *x, void *y, const char *s)
{
  ScmFixnum *a = x;
  ScmFraction *b = y;
  int lc, sc;

  PROVIDE_BIGNUM (tmp1, a->length + b->dlength);
  lc = mul (a->dig, b->dig + b->nlength, tmp1_dig, a->length, b->dlength,
	    a->sign, b->sign, &sc);
  return lowest_term_fract (tmp1_dig, b->dig, lc, b->nlength, sc, s);
}

void *fract_sint_add (void *x, void *y, const char *s)
{
  return sint_fract_addsub (y, x, s, 0, 0);
}

void *fract_sint_sub (void *x, void *y, const char *s)
{
  return sint_fract_addsub (y, x, s, 1, 0);
}

void *fract_sint_mul (void *x, void *y, const char *s)
{
  return sint_fract_mul (y, x, s);
}

void *fract_sint_div (void *x, void *y, const char *s)
{
  int a = SINT_VALUE (y);   /* axe */
  ScmFraction *b = x;
  int len, sign;

  if (a < 0)
    a = -a,
    sign = 1;
  else
    sign = 0;

  len = 1 + b->dlength;
  PROVIDE_BIGNUM (tmp1, len);
  qmul (b->dig + b->nlength, a, tmp1_dig, b->dlength);
  sign = normalize (tmp1_dig, &len, sign);
  return lowest_term_fract (b->dig, tmp1_dig, b->nlength, len,
			    b->sign != sign, s);
}

void *fract_int_add (void *x, void *y, const char *s)
{
  return int_fract_add (y, x, s);
}

void *fract_int_sub (void *x, void *y, const char *s)
{
  return int_fract_addsub (y, x, s, 1, 0);
}

void *fract_int_mul (void *x, void *y, const char *s)
{
  return int_fract_mul (y, x, s);
}

void *fract_int_div (void *x, void *y, const char *s)
{
  ScmFraction *a = x;
  ScmFixnum *b = y;
  int lc, sc;

  PROVIDE_BIGNUM (tmp1, a->dlength + b->length);
  lc = mul (a->dig + a->nlength, b->dig, tmp1_dig, a->dlength, b->length,
	    a->sign, b->sign, &sc);
  return lowest_term_fract (a->dig, tmp1_dig, a->nlength, lc, sc, s);
}

static void *fract_fract_addsub (void *x, void *y, const char *s, int sub)
{
  ScmFraction *a = x, *b = y;
  int l1, l2, l3, s1, s2, s3;

  PROVIDE_BIGNUM (tmp1, a->nlength + b->dlength);
  l1 = mul (a->dig, b->dig + b->nlength, tmp1_dig, a->nlength, b->dlength,
	    a->sign, 0, &s1);
  PROVIDE_BIGNUM (tmp2, a->dlength + b->nlength);
  l2 = mul (a->dig + a->nlength, b->dig, tmp2_dig, a->dlength, b->nlength,
	    0, sub ? !b->sign : b->sign, &s2);
  PROVIDE_BIGNUM (tmp3, (l1 > l2 ? l1 : l2) + 1);
  l3 = add (tmp1_dig, tmp2_dig, tmp3_dig, l1, l2, s1, s2, &s3);
  PROVIDE_BIGNUM (tmp1, a->dlength + b->dlength);
  l1 = mul (a->dig + a->nlength, b->dig + b->nlength, tmp1_dig,
	    a->dlength, b->dlength, 0, 0, &s1);
  return lowest_term_fract (tmp3_dig, tmp1_dig, l3, l1, s3, s);
}

void *fract_fract_add (void *x, void *y, const char *s)
{
  return fract_fract_addsub (x, y, s, 0);
}

void *fract_fract_sub (void *x, void *y, const char *s)
{
  return fract_fract_addsub (x, y, s, 1);
}

void *fract_fract_mul (void *x, void *y, const char *s)
{
  ScmFraction *a = x, *b = y;
  int lc1, lc2, sc;

  PROVIDE_BIGNUM (tmp2, a->dlength + b->dlength);
  lc2 = mul (a->dig + a->nlength, b->dig + b->nlength, tmp2_dig,
	     a->dlength, b->dlength, 0, 0, &sc);
  PROVIDE_BIGNUM (tmp1, a->nlength + b->nlength);
  lc1 = mul (a->dig, b->dig, tmp1_dig, a->nlength, b->nlength,
	     a->sign, b->sign, &sc);
  return lowest_term_fract (tmp1_dig, tmp2_dig, lc1, lc2, sc, s);
}

void *fract_fract_div (void *x, void *y, const char *s)
{
  ScmFraction *a = x, *b = y;
  int lc1, lc2, sc;

  PROVIDE_BIGNUM (tmp2, a->dlength + b->nlength);
  lc2 = mul (a->dig + a->nlength, b->dig, tmp2_dig,
	     a->dlength, b->nlength, 0, 0, &sc);
  PROVIDE_BIGNUM (tmp1, a->nlength + b->dlength);
  lc1 = mul (a->dig, b->dig + b->nlength, tmp1_dig, a->nlength, b->dlength,
	     a->sign, b->sign, &sc);
  return lowest_term_fract (tmp1_dig, tmp2_dig, lc1, lc2, sc, s);
}

void *real_real_add (void *x, void *y, const char *s)
{
  return new_real (ScmGetReal (x) + ScmGetReal (y));
}

void *real_real_sub (void *x, void *y, const char *s)
{
  return new_real (ScmGetReal (x) - ScmGetReal (y));
}

void *real_real_mul (void *x, void *y, const char *s)
{
  return new_real (ScmGetReal (x) * ScmGetReal (y));
}

void *real_real_div (void *x, void *y, const char *s)
{
  double a = ScmGetReal (x), b = ScmGetReal (y);
  if (b == 0.0)
    error ("%s: %w %w (zero divide)", s, x, y);
  else
    return new_real (a / b);
}

void *real_cplx_add (void *x, void *y, const char *s)
{
  double a = ScmGetReal (x);
  ScmComplex *b = y;
  return new_cplx (a + b->re, b->im);
}

void *real_cplx_sub (void *x, void *y, const char *s)
{
  double a = ScmGetReal (x);
  ScmComplex *b = y;
  return new_cplx (a - b->re, -b->im);
}

void *real_cplx_mul (void *x, void *y, const char *s)
{
  double a = ScmGetReal (x);
  ScmComplex *b = y;
  return new_real_cplx (a * b->re, a * b->im);
}

void *real_cplx_div (void *x, void *y, const char *s)
{
  double a = ScmGetReal (x);
  ScmComplex *b = y;
  double f = a / (b->re * b->re + b->im * b->im);
  return new_real_cplx (f * b->re, - f * b->im);
}

void *cplx_real_add (void *x, void *y, const char *s)
{
  return real_cplx_add (y, x, s);
}

void *cplx_real_sub (void *x, void *y, const char *s)
{
  ScmComplex *a = x;
  double b = ScmGetReal (y);
  return new_cplx (a->re - b, a->im);
}

void *cplx_real_mul (void *x, void *y, const char *s)
{
  return real_cplx_mul (y, x, s);
}

void *cplx_real_div (void *x, void *y, const char *s)
{
  ScmComplex *a = x;
  double b = ScmGetReal (y);

  if (b == 0.0)
    error ("%s: %w %w (zero divide)", s, x, y);
  else
    return new_real_cplx (a->re / b, a->im / b);
}

void *cplx_cplx_add (void *x, void *y, const char *s)
{
  ScmComplex *a = x, *b = y;
  return new_real_cplx (a->re + b->re, a->im + b->im);
}

void *cplx_cplx_sub (void *x, void *y, const char *s)
{
  ScmComplex *a = x, *b = y;
  return new_real_cplx (a->re - b->re, a->im - b->im);
}

void *cplx_cplx_mul (void *x, void *y, const char *s)
{
  ScmComplex *a = x, *b = y;
  return new_real_cplx (a->re * b->re - a->im * b->im,
			a->re * b->im + a->im * b->re);
}

void *cplx_cplx_div (void *x, void *y, const char *s)
{
  ScmComplex *a = x, *b = y;
  double d = b->re * b->re + b->im * b->im;
  return new_real_cplx ((a->re * b->re + a->im * b->im) / d,
			(a->im * b->re - a->re * b->im) / d);
}

int sint_zero (void *x, const char *s)
{
  return SINT_VALUE (x) == 0;
}

int sint_positive (void *x, const char *s)
{
  return SINT_VALUE (x) > 0;
}

int sint_negative (void *x, const char *s)
{
  return SINT_VALUE (x) < 0;
}

int int_positive (void *x, const char *s)
{
  ScmFixnum *a = x;
  return a->sign == 0 && (a->length > 1 || a->dig [0] != 0);
}

int int_negative (void *x, const char *s)
{
  ScmFixnum *a = x;
  return a->sign == 1;
}

int fract_positive (void *x, const char *s)
{
  ScmFraction *a = x;
  return a->sign == 0;
}

int fract_negative (void *x, const char *s)
{
  ScmFraction *a = x;
  return a->sign == 1;
}

int real_zero (void *x, const char *s)
{
  ScmReal *a = x;
  return a->val == 0.0;
}

int real_positive (void *x, const char *s)
{
  ScmReal *a = x;
  return a->val > 0.0;
}

int real_negative (void *x, const char *s)
{
  ScmReal *a = x;
  return a->val < 0.0;
}

int real_is_int (void *x, const char *s)
{
  ScmReal *a = x;
  double tmp;

  return modf (a->val, &tmp) == 0.0;
}

void *no_cplx_angle (void *x, const char *s)
{
  if (ScmUPred (SCM_NEGATIVE_PRED, x))
    return new_real (PI);
  else
    return the_real_zero;
}

void *sint_abs (void *x, const char *s)
{
  int val = SINT_VALUE (x);
  return val >= 0 ? x : MAKE_SINT (-val);
}

void *sint_negate (void *x, const char *s)
{
  return MAKE_SINT (-SINT_VALUE (x));
}

void *sint_inverse (void *x, const char *s)
{
  int a = SINT_VALUE (x);
  ScmFraction *f;

  if (a == 0)
    error ("%s: zero divide", s);
  if (a == 1 || a == -1)
    return x;
  NEW_FRACT (f, 1, 1);
  f->dig [0] = 1;
  if (a < 0)
    (f->dig + 1) [0] = -a,
    f->sign = 1;
  else
    (f->dig + 1) [0] = a,
    f->sign = 0;
  return f;
}

void *sint_to_real (void *x, const char *s)
{
  return new_real (SINT_VALUE (x));
}

void *int_abs (void *x, const char *s)
{
  ScmFixnum *a = x;
  return a->sign ? int_negate (a, s) : a;
}

void *int_negate (void *x, const char *s)
{
  ScmFixnum *a = x, *b;

  gcs1 = a;
  NEW_FIXNUM (b, a->length);
  a = gcs1;
  gcs1 = NULL;
  bigdig_copy (a->dig, b->dig, a->length);
  b->sign = !a->sign;
  return b;			/* cannot be a sint after negation */
}

void *int_inverse (void *x, const char *s)
{
  ScmFixnum *a = x;
  ScmFraction *f;

  gcs1 = a;
  NEW_FRACT (f, 1, a->length);
  a = gcs1;
  gcs1 = NULL;
  f->sign = a->sign;
  f->dig [0] = 1;
  bigdig_copy (a->dig, f->dig + 1, a->length);
  return f;
}

void *int_one (void *x, const char *s)
{
  return the_fixnum_one;
}
 
void *int_to_real (void *x, const char *s)
{
  return new_real (int_cvt_real (x));
}

void *fract_negate (void *x, const char *s)
{
  ScmFraction *a = x, *f;

  gcs1 = a;
  NEW_FRACT (f, a->nlength, a->dlength);
  a = gcs1;
  gcs1 = NULL;
  f->sign = !a->sign;
  bigdig_copy (a->dig, f->dig, a->nlength + a->dlength);
  return f;
}

void *fract_abs (void *x, const char *s)
{
  ScmFraction *a = x;
  return a->sign ? fract_negate (x, s) : x;
}

void *fract_inverse (void *x, const char *s)
{
  ScmFraction *a = x;
  if (a->nlength == 1 && a->dig [0] == 1) {
    ScmFixnum *f;
    gcs1 = a;
    NEW_FIXNUM (f, a->dlength);
    a = gcs1;
    gcs1 = NULL;
    f->sign = a->sign;
    bigdig_copy (a->dig + 1, f->dig, a->dlength);
    return simple (f);
  } else {
    ScmFraction *f;
    gcs1 = a;
    NEW_FRACT (f, a->dlength, a->nlength);
    a = gcs1;
    gcs1 = NULL;
    f->sign = a->sign;
    bigdig_copy (a->dig, f->dig + a->dlength, a->nlength);
    bigdig_copy (a->dig + a->nlength, f->dig, a->dlength);
    return f;
  }
}

void *fract_numerator (void *x, const char *s)
{
  ScmFraction *a = x;
  ScmFixnum *f;
  gcs1 = a;
  NEW_FIXNUM (f, a->nlength);
  a = gcs1;
  gcs1 = NULL;
  f->sign = a->sign;
  bigdig_copy (a->dig, f->dig, a->nlength);
  return simple (f);
}

void *fract_denominator (void *x, const char *s)
{
  ScmFraction *a = x;
  ScmFixnum *f;
  gcs1 = a;
  NEW_FIXNUM (f, a->dlength);
  a = gcs1;
  gcs1 = NULL;
  f->sign = 0;
  bigdig_copy (a->dig + a->nlength, f->dig, a->dlength);
  return simple (f);
}

static void *fract_to_int (ScmFraction *a, int mode)
/* mode: 0 -> floor, 1 -> ceil, 2 -> trunc, 3 -> round */
{
  int l1, lr, add1 = 0, sign = a->sign;
  bigdig *r;
  bigreg rr;
  ScmFixnum *f;

  assert (mode >= 0 && mode <= 3);

  if (abs_cmp (a->dig, a->dig + a->nlength, a->nlength, a->dlength) < 0) {
    PROVIDE_BIGNUM (tmp1, 1);
    tmp1_dig [0] = 0;
    l1 = 1;
    r = a->dig;
    lr = a->nlength;
  } else if (a->dlength == 1) {
    PROVIDE_BIGNUM (tmp1, a->nlength);
    PROVIDE_BIGNUM (tmp2, 1);
    l1 = a->nlength;
    qdiv (a->dig, a->dig [a->nlength], tmp1_dig, &rr, a->nlength);
    tmp2_dig [0] = rr;
    (void) normalize (tmp1_dig, &l1, 0);
    r = tmp2_dig;
    lr = 1;
  } else {
    l1 = a->nlength - a->dlength + 1;
    PROVIDE_BIGNUM (tmp1, l1);
    lr = a->nlength + 1;
    PROVIDE_BIGNUM (tmp2, lr);
    r = tmp2_dig;
    sdiv (a->dig, a->dig + a->nlength, tmp1_dig, r, a->nlength, a->dlength);
    (void) normalize (tmp1_dig, &l1, 0);
    (void) normalize (r, &lr, 0);
  }
  if ((mode == 0 && sign) ||
      (mode == 1 && !sign))
    add1 = 1;
  else if (mode == 3) {
    int cmp;
    PROVIDE_BIGNUM (tmp3, lr + 1);
    bigdig_copy (r, tmp3_dig, lr);
    tmp3_dig [lr] = 0;
    shift_left (tmp3_dig, lr + 1, 1);
    if (tmp3_dig [lr] != 0)
      ++lr;
    cmp = abs_cmp (tmp3_dig, a->dig + a->nlength, lr, a->dlength);
    if (cmp > 0 || (cmp == 0 && (tmp1_dig [0] % 2)))
      add1 = 1;
  }
  if (add1) {
    PROVIDE_BIGNUM (tmp2, l1 + 1);
    qadd (tmp1_dig, 1, tmp2_dig, l1);
    if (tmp2_dig [l1] != 0)
      ++l1;
    r = tmp2_dig;
  } else
    r = tmp1_dig;
  NEW_FIXNUM (f, l1);
  f->sign = (l1 > 1 || r [0] != 0) ? sign : 0;
  bigdig_copy (r, f->dig, l1);
  return simple (f);
}

void *fract_floor (void *x, const char *s)
{
  return fract_to_int (x, 0);
}

void *fract_ceiling (void *x, const char *s)
{
  return fract_to_int (x, 1);
}

void *fract_truncate (void *x, const char *s)
{
  return fract_to_int (x, 2);
}

void *fract_round (void *x, const char *s)
{
  return fract_to_int (x, 3);
}

void *fract_to_real (void *x, const char *s)
{
  return new_real (fract_cvt_real (x));
}

void *real_abs (void *x, const char *s)
{
  ScmReal *a = x;
  return a->val >= 0 ? a : new_real (-a->val);
}

void *real_negate (void *x, const char *s)
{
  ScmReal *a = x;
  return new_real (-a->val);
}

void *real_inverse (void *x, const char *s)
{
  ScmReal *a = x;

  if (a->val == 0.0)
    error ("%s: %w (zero divide)", s, x);
  else
    return new_real (1.0 / a->val);
}

void *real_numerator (void *x, const char *s)
{
  ScmReal *a = x;
  double ign;
  if (modf (a->val, &ign) == 0.0)
    return x;
  else
    error ("%s: %w (not a fraction)", s, x);
}

void *real_denominator (void *x, const char *s)
{
  ScmReal *a = x;
  double ign;
  if (modf (a->val, &ign) == 0.0)
    return new_real (1.0);
  else
    error ("%s: %w (not a fraction)", s, x);
}

void *real_floor (void *x, const char *s)
{
  ScmReal *a = x;
  return new_real (floor (a->val));
}

void *real_ceiling (void *x, const char *s)
{
  ScmReal *a = x;
  return new_real (ceil (a->val));
}

void *real_truncate (void *x, const char *s)
{
  ScmReal *a = x;
  double tmp;

  modf (a->val, &tmp);
  return new_real (tmp);
}

void *real_round (void *x, const char *s)
{
  ScmReal *a = x;
  double f, n, tmp;

  f = modf (a->val, &n);
  if (f < 0.0) {
    if (f < -.5 || (f == -.5 && modf (n * .5, &tmp) != 0.0))
      n -= 1.0;
  } else {
    if (f > .5 || (f == .5 && modf (n * .5, &tmp) != 0.0))
      n += 1.0;
  }
  return new_real (n);
}

void *real_exp (void *x, const char *s)
{
  return new_real (exp (ScmGetReal (x)));
}

void *real_log (void *x, const char *s)
{
  double v = ScmGetReal (x);

  if (v > 0.0)
    return new_real (log (v));
  else if (v < 0.0)
    return new_cplx (log (-v), PI);
  else
    error ("%s: %w (zero logarithm)", s, x);
}

void *real_sin (void *x, const char *s)
{
  return new_real (sin (ScmGetReal (x)));
}

void *real_cos (void *x, const char *s)
{
  return new_real (cos (ScmGetReal (x)));
}

void *real_tan (void *x, const char *s)
{
  return new_real (tan (ScmGetReal (x)));
}

static double c_sqrt (double re, double im, double *im_r)
{
  double r;

  if (im == 0.0) {		/* singular case */
    if (re >= 0.0) {
      *im_r = 0.0;
      return sqrt (re);
    } else {
      *im_r = sqrt (-re);
      return 0.0;
    }
  }

  r = sqrt (.5 * (re + sqrt (re * re + im * im)));
  *im_r = .5 * im / r;
  return r;
}

static double c_log (double re, double im, double *im_r, const char *s)
{
  *im_r = angle (im, re);
  return log (sqrt (re * re + im * im));
}

static double c_asin (double re, double im, double *in_r, const char *s)
{
  double rr, ri, lr, li;
  rr = c_sqrt (1.0 - re * re + im * im, -2.0 * re * im, &ri);
  lr = c_log (rr - im, ri + re, &li, s);
  *in_r = -lr;
  return li;
}

void *real_asin (void *x, const char *s)
{
  double v = ScmGetReal (x);
  if (-1.0 <= v && v <= 1.0)
    return new_real (asin (v));
  else {
    double re, im;
    re = c_asin (v, 0.0, &im, s);
    return new_real_cplx (re, im);
  }
}

void *real_acos (void *x, const char *s)
{
  double v = ScmGetReal (x);
  if (-1.0 <= v && v <= 1.0)
    return new_real (acos (v));
  else {
    double re, im;
    re = c_asin (v, 0.0, &im, s);
    return new_real_cplx (.5 * PI - re, -im);
  }
}

void *real_atan (void *x, const char *s)
{
  return new_real (atan (ScmGetReal (x)));
}

void *real_sqrt (void *x, const char *s)
{
  double v = ScmGetReal (x);

  if (v >= 0.0)
    return new_real (sqrt (v));
  else
    return new_cplx (0.0, sqrt (-v));
}

void *real_to_fract (void *x, const char *s)
{
  ScmReal *a = x;
  double tmp, d, dd;
  int e, sign = 0, ll, len, i;

  if ((d = a->val) == 0.0)
    return the_fixnum_zero;
  if (d < 0.0) {
    sign = 1;
    d = -d;
  }
  d = frexp (d, &e);
  for (dd = d, ll = 0; dd != 0.0; dd = modf (ldexp (dd, 1), &tmp))
    ++ll;
  e -= ll;
  if (e >= 0) {
    ScmFixnum *f;
    len = (e + ll) / BIGBITS + 1;
    NEW_FIXNUM (f, len);
    while (ll > 0) {
      shift_left (f->dig, len, 1);
      d = modf (ldexp (d, 1), &tmp);
      if (tmp != 0.0)
	f->dig [0] |= 1;
      --ll;
    }
    if (e > 0) {
      shift_left (f->dig, len, e % BIGBITS);
      if ((e = e / BIGBITS)) {
	for (i = len; i-- > e; )
	  f->dig [i] = f->dig [i - e];
	for ( ; i-- > 0; )
	  f->dig [i] = 0;
      }
    }
    f->sign = normalize (f->dig, &len, sign);
    f->length = len;
    return simple (f);
  } else {
    int len2;
    len = (ll + BIGBITS - 1) / BIGBITS;
    PROVIDE_BIGNUM (tmp1, len);
    for (i = 0; i < len; i++)
      tmp1_dig [i] = 0;
    while (ll > 0) {
      shift_left (tmp1_dig, len, 1);
      d = modf (ldexp (d, 1), &tmp);
      if (tmp != 0.0)
	tmp1_dig [0] |= 1;
      --ll;
    }
    e = -e;
    len2 = e / BIGBITS + 1;
    PROVIDE_BIGNUM (tmp2, len2);
    for (i = 0; i < len2; i++)
      tmp2_dig [i] = 0;
    tmp2_dig [e / BIGBITS] |= (B1 << (e % BIGBITS));
    return lowest_term_fract (tmp1_dig, tmp2_dig, len, len2, sign,
			      "inexact->exact (real->fract)");
  }
}

void *cplx_negate (void *x, const char *s)
{
  ScmComplex *a = x;
  return new_cplx (-a->re, -a->im);
}

void *cplx_inverse (void *x, const char *s)
{
  ScmComplex *a = x;
  double d = 1.0 / (a->re * a->re + a->im * a->im);
  return new_real_cplx (a->re / d, -a->im / d);
}

void *cplx_exp (void *x, const char *s)
{
  ScmComplex *a = x;
  double f = exp (a->re);
  return new_real_cplx (f * cos (a->im), f * sin (a->im));
}

void *cplx_log (void *x, const char *s)
{
  ScmComplex *a = x;
  double re, im;
  re = c_log (a->re, a->im, &im, s);
  return new_real_cplx (re, im);
}

void *cplx_sin (void *x, const char *s)
{
  ScmComplex *a = x;
  return new_real_cplx (sin (a->re) * cosh (a->im),
			cos (a->re) * sinh (a->im));
}

void *cplx_cos (void *x, const char *s)
{
  ScmComplex *a = x;
  return new_real_cplx (cos (a->re) * cosh (a->im),
			- sin (a->re) * sinh (a->im));
}

void *cplx_tan (void *x, const char *s)
{
  ScmComplex *a = x;
  double cos_r = cos (a->re), sin_r = sin (a->re);
  double cosh_i = cosh (a->im), sinh_i = sinh (a->im);
  double cos_r_2 = cos_r * cos_r, sin_r_2 = 1.0 - cos_r_2;
  double sinh_i_2 = sinh_i * sinh_i, cosh_i_2 = 1.0 + sinh_i_2;
  double d = cos_r_2 * cosh_i_2 + sin_r_2 * sinh_i_2;
  if (d == 0.0)
    error ("%s: %w (complex tan)", s, x);
  else
    return new_real_cplx (sin_r * cos_r / d, sinh_i * cosh_i / d);
}

void *cplx_asin (void *x, const char *s)
{
  ScmComplex *a = x;
  double re, im;
  re = c_asin (a->re, a->im, &im, s);
  return new_real_cplx (re, im);
}

void *cplx_acos (void *x, const char *s)
{
  ScmComplex *a = x;
  double re, im;
  re = c_asin (a->re, a->im, &im, s);
  return new_real_cplx (.5 * PI - re, -im);
}

void *cplx_atan (void *x, const char *s)
{
  ScmComplex *a = x;
  double r1, i1, r2, i2;
  r1 = c_log (1.0 - a->im, a->re, &i1, s);
  r2 = c_log (1.0 + a->im, -a->re, &i2, s);
  return new_real_cplx (.5 * (i1 - i2), -.5 * (r1 - r2));
}

void *cplx_sqrt (void *x, const char *s)
{
  ScmComplex *a = x;
  double re, im;
  re = c_sqrt (a->re, a->im, &im);
  return new_real_cplx (re, im);
}

void *cplx_magnitude (void *x, const char *s)
{
  ScmComplex *a = x;
  return new_real (sqrt (a->re * a->re + a->im * a->im));
}

void *cplx_angle (void *x, const char *s)
{
  ScmComplex *a = x;
  return new_real (angle (a->im, a->re));
}

void *cplx_real_part (void *x, const char *s)
{
  ScmComplex *a = x;
  return new_real (a->re);
}

void *cplx_imag_part (void *x, const char *s)
{
  ScmComplex *a = x;
  return new_real (a->im);
}

void *exact_zero (void)
{
  return the_fixnum_zero;
}

void *inexact_zero (void)
{
  return the_real_zero;
}

static void deallocate_all (void);

static void int_module_init (void)
{
  atexit (deallocate_all);
  init_sfixnum_array ();
  MEM_root_var (the_fixnum_zero);
  MEM_root_var (the_fixnum_one);
  MEM_root_var (the_real_zero);
  MEM_root_var (gcs1);
  MEM_root_var (gcs2);
  MEM_root_var (gcs3);
  the_fixnum_zero = MAKE_SINT (0);
  the_fixnum_one = MAKE_SINT (1);
  the_real_zero = new_real (0.0);
  decimal_point = localeconv ()->decimal_point [0];
}

static void raw_bigdig_to_ascii_10 (bigdig *a, int la, char *strg)
{
  int i;
  char *s;
  bigreg r;
  char buf [5];

  PROVIDE_BIGNUM (tmp1, la);
  bigdig_copy (a, tmp1_dig, la);
  s = strg;
  while (la > 1 || tmp1_dig [0] > 0) {
    if (qdiv (tmp1_dig, 10000, tmp1_dig, &r, la) == 0)
      --la;
    sprintf (buf, "%04u", (unsigned) r);
    for (i = 3; i >= 0; i--)
      *s++ = buf [i];
  }
  /* strip off leading zeros */
  while (s > strg && s [-1] == '0')
    --s;
  *s = '\0';
  if (strg == s) {
    *s++ = '0';
    *s = '\0';
  } else {
    char *t = strg, c;
    while (t < --s) {
      c = *s;
      *s = *t;
      *t++ = c;
    }
  }
}

static char *unparse10_buf = NULL;
static int unparse10_len = 0;

static char *bigdig_to_ascii_10 (bigdig *a, int la)
/* returns pointer to static data */
{
  int l;

  /* allow room for leading zeros in raw_bigdig_to_ascii_10 */
  l = ndig_to_ndec (la) + 4;
  if (l > unparse10_len) {
    unparse10_buf = REALLOC (unparse10_buf, l);
    if (unparse10_buf == NULL) {
      unparse10_len = 0;
      reset ("Out of memory");
    }
    unparse10_len = l;
  }
  raw_bigdig_to_ascii_10 (a, la, unparse10_buf);
  return unparse10_buf;
}

static void raw_bigdig_to_ascii (bigdig *a, int la, int bits, char *strg)
/* bits == 0 -> base == 10 */
{
  int i;
  char *save, *s;
  unsigned r, mask;

  if (bits == 0) {
    raw_bigdig_to_ascii_10 (a, la, strg);
    return;
  }

  mask = ~(~0<<bits);

  save = s = strg;
  PROVIDE_BIGNUM (tmp1, la);
  bigdig_copy (a, tmp1_dig, la);
  while (la > 1 || tmp1_dig [0] != 0) {
    r = tmp1_dig [0] & 0xfff;
    shift_right (tmp1_dig, la, 12);
    if (tmp1_dig [la - 1] == 0)
      --la;
    for (i = 0; i < 12; i+= bits) {
      *s++ = "0123456789abcdef" [r & mask];
      r >>= bits;
    }
  }
  while (s > save && s [-1] == '0')
    --s;
  *s = '\0';
  if (save == s) {
    *s++ = '0';
    *s = '\0';
  } else
    while (save < --s) {
      i = *s;
      *s = *save;
      *save++ = i;
    }
}

static void restore_bignum (bigdig *tmp, FILE *file)
/* tmp must point to enough memory */
{
  int i, c, l;
  bigreg p;

  l = 0;
  while ((c = getc (file)) != EOF && isdigit (c)) {
    p = c - '0';
    for (i = 0; i < l; i++) {
      p += (bigreg) tmp [i] * 10;
      tmp [i] = p & BIGMASK;
      p >>= BIGBITS;
    }
    if (p)
      tmp [l++] = p;
  }
  if (l == 0)
    tmp [0] = 0;		/* thnx to Wolfgang Lux... */
  if (c != 'Y')
    fatal ("bad dump file format (restore_bignum)");
}

static void int_dumper (void *x, FILE *file)
{
  ScmFixnum *a = x;
  char *s;

  if (a->sign)
    putc ('-', file);
  MEM_dump_ul (a->length, file);
  s = bigdig_to_ascii_10 (a->dig, a->length);
  fputs (s, file);
  putc ('Y', file);
}

static void sint_dumper (void *x, FILE *file)
{
  int val = SINT_VALUE (x);

  if (val < 0) {
    putc ('-', file);
    val = -val;
  }
  MEM_dump_ul (1, file);
  fprintf (file, "%d", val);
  putc ('Y', file);
}

static void fract_dumper (void *x, FILE *file)
{
  ScmFraction *a = x;
  char *s;

  if (a->sign)
    putc ('-', file);
  MEM_dump_ul (a->nlength, file);
  MEM_dump_ul (a->dlength, file);
  s = bigdig_to_ascii_10 (a->dig, a->nlength);
  fputs (s, file);
  putc ('Y', file);
  s = bigdig_to_ascii_10 (a->dig + a->nlength, a->dlength);
  fputs (s, file);
  putc ('Y', file);
}

static void dump_dbl (double x, FILE *file)
/* split the number into machine-independend parts and dump those */
{
  int e;
  double ign;

  if (x < 0.0) {
    x = -x;
    putc ('-', file);
  }
  x = frexp (x, &e);
  while (modf (x, &ign) != 0.0) {
    e--;
    x = ldexp (x, 1);
  }
  fprintf (file, "%.0fY", x);
  if (e < 0) {
    e = -e;
    putc ('-', file);
  }
  MEM_dump_ul (e, file);
}

static double restore_dbl (FILE *file)
/* read what dump_dbl wrote */
{
  int c, e;
  int sign, esign;
  double x;

  if ((c = getc (file)) == EOF)
    fatal ("bad dump file format (restore_dbl 1)");
  if (c == '-')
    sign = 1;
  else {
    ungetc (c, file);
    sign = 0;
  }
  if (fscanf (file, "%lfY", &x) != 1 ||
      (c = getc (file)) == EOF)
    fatal ("bad dump file format (restore_dbl 2)");
  if (c == '-')
    esign = 1;
  else {
    ungetc (c, file);
    esign = 0;
  }
  e = MEM_restore_ul (file);
  if (esign)
    e = -e;
  x = ldexp (x, e);
  return sign ? -x : x;
}

static void real_dumper (void *x, FILE *file)
{
  ScmReal *a = x;
  dump_dbl (a->val, file);
}

static void cplx_dumper (void *x, FILE *file)
{
  ScmComplex *a = x;
  dump_dbl (a->re, file);
  dump_dbl (a->im, file);
}

static void *int_excavator (FILE *file)
{
  int c, sign, len;
  ScmFixnum *f;

  if ((c = getc (file)) == EOF)
    fatal ("bad dump file format (int_excavator I)");
  if (c == '-')
    sign = 1;
  else {
    ungetc (c, file);
    sign = 0;
  }
  len = MEM_restore_ul (file);
  if (len == 1) {
    long val;
    if (fscanf (file, "%ldY", &val) != 1)
      fatal ("bad dump file format (int_excavator II)");
    if (sign)
      val = -val;
    return ScmLongToNumber (val);
  } else {
    NEW_FIXNUM (f, len);
    f->sign = sign;
    restore_bignum (f->dig, file);
    return f;
  }
}

static void *fract_excavator (FILE *file)
{
  int c, sign, nlen, dlen;
  ScmFraction *f;

  if ((c = getc (file)) == EOF)
    fatal ("bad dump file format (fract_restore_init)");
  if (c == '-')
    sign = 1;
  else {
    ungetc (c, file);
    sign = 0;
  }
  nlen = MEM_restore_ul (file);
  dlen = MEM_restore_ul (file);
  NEW_FRACT (f, nlen, dlen);
  f->sign = sign;
  restore_bignum (f->dig, file);
  restore_bignum (f->dig + nlen, file);
  return f;
}

static void *real_excavator (FILE *file)
{
  return new_real (restore_dbl (file));
}

static void *cplx_excavator (FILE *file)
{
  double re, im;
  re = restore_dbl (file);
  im = restore_dbl (file);
  return new_cplx (re, im);
}

static void int_display (void *x, putc_proc pp, void *cd)
{
  ScmFixnum *a = x;

  if (a->sign)
    (* pp) ('-', cd);
  putc_string (bigdig_to_ascii_10 (a->dig, a->length), pp, cd);
}

static void sint_display (void *x, putc_proc pp, void *cd)
{
  char buf [8];

  sprintf (buf, "%d", SINT_VALUE (x));
  putc_string (buf, pp, cd);
}

static void fract_display (void *x, putc_proc pp, void *cd)
{
  ScmFraction *a = x;

  if (a->sign)
    (* pp) ('-', cd);
  putc_string (bigdig_to_ascii_10 (a->dig, a->nlength), pp, cd);
  (* pp) ('/', cd);
  putc_string (bigdig_to_ascii_10 (a->dig + a->nlength, a->dlength), pp, cd);
}

static void pf_double (double x, char *s)
{
  char *s1, *s2;
  sprintf (s, "%#.*g", DBL_DIG, x);
  if ((s1 = strchr (s, decimal_point)) != NULL) {
    if (decimal_point != '.')
      *s1 = '.';
    s1 += strspn (s1 + 1, "0123456789") + 1;
    /* s1 now points after the last digit (or dot) */
    for (s2 = s1; *--s2 == '0'; )
      ;
    s2++;
    if (s1 > s2)
      while ((*s2++ = *s1++) != '\0') /* strcpy would be illegal (overlap) */
	;
  }
}

static char *unparse_real (double x)
{
  static char buf [64];
  pf_double (x, buf);
  return buf;
}

static void real_display (void *x, putc_proc pp, void *cd)
{
  ScmReal *a = x;
  putc_string (unparse_real (a->val), pp, cd);
}

static void pf_imag (double im, char *s)
{
  if (im == -1.0)
    strcpy (s, "-i");
  else if (im == 1.0)
    strcpy (s, "+i");
  else {
    if (im < 0.0) {
      s [0] = '-';
      pf_double (-im, s + 1);
    } else {
      s [0] = '+';
      pf_double (im, s + 1);
    }
    strcat (s, "i");
  }
}

static char *unparse_cplx (double re, double im)
{
  static char buf [128];
  if (re == 0.0)
    pf_imag (im, buf);
  else {
    pf_double (re, buf);
    pf_imag (im, buf + strlen (buf));
  }
  return buf;
}

static void cplx_display (void *x, putc_proc pp, void *cd)
{
  ScmComplex *a = x;
  putc_string (unparse_cplx (a->re, a->im), pp, cd);
}

unsigned long ScmNumberToULong (void *x, const char *s)
{
  ScmFixnum *a;

  if (ScmTypeOf (x) == ScmType (SFixnum)) {
    int val = SINT_VALUE (x);
    if (val >= 0)
      return val;
  } else if (ScmTypeOf (x) == ScmType (Fixnum) && (a = x)->sign == 0)
    switch (a->length) {
    case 0:
    case 1:
      return a->dig [0];
    case 2:
      return a->dig [0] | (a->dig [1] << BIGBITS);
    default:
      error ("%s: %w (range)", s, x);
    }

  error ("%s: %w (not a non-negative exact integer)", s, x);
}

long ScmNumberToLong (void *x, const char *s)
{
  ScmFixnum *a;
  bigreg ul;
  long r;

  if (ScmTypeOf (x) == ScmType (SFixnum))
    return SINT_VALUE (x);

  if (ScmTypeOf (x) != ScmType (Fixnum))
    error ("%s: %w (not an exact integer)", s, x);

  a = x;
  switch (a->length) {
  case 0:
  case 1:
    ul = a->dig [0];
    break;
  case 2:
    ul = a->dig [0] | (a->dig [1] << BIGBITS);
    break;
  default:
    error ("%s: %w (range)", s, x);
  }
  if (ul > LONG_MAX)		/* misses LONG_MIN */
    error ("%s: %w (range)", s, x);
  r = ul;
  return a->sign ? -r : r;
}

unsigned int ScmNumberToUShort (void *x, const char *s)
{
  ScmFixnum *a;
  bigreg r;

  if (ScmTypeOf (x) == ScmType (SFixnum)) {
    int val = SINT_VALUE (x);
    if (val >= 0)
      return val;
    else
      error ("%s: %w (range)", s, x);
  }

  if (ScmTypeOf (x) != ScmType (Fixnum))
    error ("%s: %w (not an exact integer)", s, x);

  a = x;
  if (a->sign || a->length > 1)
    error ("%s: %w (range)", s, x);
  r = a->dig [0];
  if (BASE <= USHRT_MAX) {	/* constant expression */
    if (r > USHRT_MAX)
      error ("%s: %w (range)", s, x);
  }
  return r;
}

extern void *ScmLongToNumber (long l)
{
  unsigned long u, tmp;
  int sign, i;
  ScmFixnum *f;

  if (l > -sfixnum_range && l < sfixnum_range)
    return MAKE_SINT (l);

  if (l < 0) {
    sign = 1;
    u = -l;
  } else {
    sign = 0;
    u = l;
  }

  i = 0;
  tmp = u;
  do {
    i++;
    tmp >>= BIGBITS;
  } while (tmp != 0);

  NEW_FIXNUM (f, i);
  f->sign = sign;

  i = 0;
  do {
    f->dig [i++] = u & BIGMASK;
    u >>= BIGBITS;
  } while (u != 0);

  return f;
}

int ScmNumberIsEven (void *x, const char *s)
{
  if (ScmTypeOf (x) == ScmType (SFixnum))
      return (SINT_VALUE (x) & 0x1) == 0;
  if (ScmTypeOf (x) == ScmType (Fixnum))
    return (((ScmFixnum *) x)->dig [0] & 0x1) == 0;
  if (ScmTypeOf (x) == ScmType (Real)) {
    ScmReal *a = x;
    double ign;
    if (modf (a->val, &ign) == 0.0)
      return modf (.5 * a->val, &ign) == 0.0;
  }
  error ("%s: %w (not an integer)", s, x);
}

void *ScmAtan2 (void *y, void *x)
{
  return new_real (angle (ScmGetReal (y), ScmGetReal (x)));
}

# define EX_E 0
# define EX_I 1

static const char digits [] = "0123456789abcdef";

static int scan_digits (char *s, int l, int b, int *pp, int *ex)
/* returns the length of the sequence (0 on error) and stores the
   position of . in *pp (-1 -> no .), *ex = 1 -> # found in string */
{
  int i, f = 0;
  char *d;

  *ex = 0;
  *pp = -1;
  for (i = 0; i < l; i++)
    switch (s [i]) {
    case '.':
      if (b != 10)
	return 0;
      if (*pp >= 0)
	return 0;
      *pp = i;
      break;
    case '#':
      if (f == 0) {
	if (i == 0 || (i == 1 && s [0] == '.'))
	  return 0;
	*ex = 1;
	f = 1;
      }
      break;
    default:
      if ((d = strchr (digits, s [i])) == NULL || d - digits >= b)
	return (i == 1 && s [0] == '.') ? 0 : i;
      if (f)			/* a digit after '#' -> wrong */
	return 0;
      break;
    }
  if (l == 1 && s [0] == '.')
    return 0;
  return l;
}

static int ascii_to_bignum (bigdig *tmp, char *s, int l, int b)
/* returns the length */
{
  int i, j, len;
  bigreg p;
  char *x;

  len = 0;
  for (j = 0; j < l; j++) {
    if ((x = strchr (digits, s [j])) == NULL)
      p = 0;
    else
      p = x - digits;
    for (i = 0; i < len; i++) {
      p += (bigreg) tmp [i] * b;
      tmp [i] = p & BIGMASK;
      p >>= BIGBITS;
    }
    if (p)
      tmp [len++] = p;
  }
  if (len == 0)
    tmp [len++] = 0;
  return len;
}

static double ascii_to_double (char *s, int l, int b, int pp)
{ 
  double x = 0.0;
  int i;

  for (i = 0; i < l; i++)
    if (i != pp) {
      if (s [i] == '#')
	x *= b;
      else
	x = b * x + (strchr (digits, s [i]) - digits);
    }
  if (pp >= 0)
    for (i = l; --i > pp; )
      x /= b;
  return x;
}

static void *
  parse_ureal (char **sp, int *lp, int b, int ex)
/* advances *sp and decreases *lp accordingly, NULL -> failure */
{
  char *s = *sp;
  int l = *lp;
  int pp, iex, iex2;
  int l1, l2;

  if ((l1 = scan_digits (s, l, b, &pp, &iex)) == 0)
    return NULL;
  if (l1 == l)
    goto simple_case;
  switch (s [l1]) {
  case '/':
    if (pp >= 0)
      return NULL;
    if ((l2 = scan_digits (s + l1 + 1, l - l1 - 1, b, &pp, &iex2)) == 0)
      return NULL;
    if (pp >= 0)
      return NULL;
    *lp = l - l1 - l2 - 1;
    *sp = s + l1 + l2 + 1;
    if (iex || iex2 || ex == EX_I)
      return new_real (ascii_to_double (s, l1, b, -1) /
		       ascii_to_double (s + l1 + 1, l2, b, -1));
    else {
      int ln, ld;
      PROVIDE_BIGNUM (tmp1, ndec_to_ndig (l1));
      PROVIDE_BIGNUM (tmp2, ndec_to_ndig (l2));
      ln = ascii_to_bignum (tmp1_dig, s, l1, b);
      ld = ascii_to_bignum (tmp2_dig, s + l1 + 1, l2, b);
      return lowest_term_fract (tmp1_dig, tmp2_dig, ln, ld, 0, "read");
    }
  case 'e':
  case 's':
  case 'f':
  case 'd':
  case 'l':
    if (b != 10)
      return NULL;
    else {
      double x = ascii_to_double (s, l1, 10, pp);
      int e, i, sign = 0;
      if ((i = l1 + 1) >= l)
	return NULL;
      if (s [i] == '+' || s [i] == '-') {
	if (s [i] == '-')
	  sign = 1;
	if (++i >= l)
	  return NULL;
      }
      if (!isdigit ((int) (s [i])))
	return NULL;
      for (e = 0; i < l && isdigit ((int) (s [i])); i++)
	e = 10 * e + s [i] - '0';
      if (e != 0) {
	if (sign)
	  x /= pow (10.0, e);
	else
	  x *= pow (10.0, e);
      }
      *lp = l - i;
      *sp = s + i;
      return new_real (x);
    }
  default:
  simple_case:
    *lp = l - l1;
    *sp = s + l1;
    if (pp >= 0 || iex || ex == EX_I)
      return new_real (ascii_to_double (s, l1, b, pp));
    else {
      ScmFixnum *f;
      int diglen = ndec_to_ndig (l1);
      NEW_FIXNUM (f, diglen);
      f->sign = 0;
      f->length = ascii_to_bignum (f->dig, s, l1, b);
      return simple (f);
    }
  }
}

static char *parse_buf = NULL;
static int parse_buflen = 0;

static void provide_parse_buf (int len)
{
  if (len >= parse_buflen) {
    parse_buf = REALLOC (parse_buf, len);
    if (parse_buf == NULL) {
      parse_buflen = 0;
      reset ("Out of memory");
    }
    parse_buflen = len;
  }
}

void *ScmParseNumberString (const char *arg, int l, int b)
{
  int ex = EX_E, p, s1 = 0, s2 = 0, s1f = 0;
  void *ur1, *ur2;
  double dbl1, dbl2;
  char *s;

  /* copying and converting to lower case */
  provide_parse_buf (l);
  s = parse_buf;
  for (p = 0; p < l; p++)
    s [p] = tolower (arg [p]);

  /* scanning the prefix */
  if (l > 1 && s [0] == '#') {
    p = 0;
    switch (s [1]) {
    case 'i':
      ex = EX_I;
    case 'e':
      if (l > 3 && s [2] == '#') {
	p = 4;
	switch (s [3]) {
	case 'b': b = 2; break;
	case 'o': b = 8; break;
	case 'd': b = 10; break;
	case 'x': b = 16; break;
	default: return NULL;
	}
      } else
	p = 2;
      break;
    case 'b': b = 2; goto exactness2;
    case 'o': b = 8; goto exactness2;
    case 'd': b = 10; goto exactness2;
    case 'x': b = 16;
    exactness2:
      if (l > 3 && s [2] == '#') {
	p = 4;
	switch (s [3]) {
	case 'i': ex = EX_I; break;
	case 'e': break;
	default: return NULL;
	}
      } else
	p = 2;
      break;
    }
    s += p;
    l -= p;
  }

  if (l == 0)
    return NULL;

  if (s [0] == '+' || s [0] == '-') {
    s1f = 1;
    if (s [0] == '-')
      s1 = 1;
    s++;
    if (--l == 0)
      return NULL;
    if (s [0] == 'i')
      return new_cplx (0.0, s1 ? -1.0 : 1.0);
  }
  if ((ur1 = parse_ureal (&s, &l, b, ex)) == NULL)
    return NULL;
  if (l == 0) {
    if (s1) {
      if (ScmTypeOf (ur1) == ScmType (SFixnum))
	return MAKE_SINT (-SINT_VALUE (ur1));
      if (ScmTypeOf (ur1) == ScmType (Fixnum))
	((ScmFixnum *) ur1)->sign = 1;
      else if (ScmTypeOf (ur1) == ScmType (Fraction))
	((ScmFraction *) ur1)->sign = 1;
      else {			/* ScmReal */
	ScmReal *r = ur1;
	r->val = -r->val;
      }
    }
    return ur1;
  }
  switch (s [0]) {
  case 'i':
    if (l != 1 || s1f == 0)
      return NULL;
    dbl1 = ScmGetReal (ur1);
    return new_real_cplx (0.0, s1 ? -dbl1 : dbl1);
  case '-':
    s2 = 1;
    /* flow between cases! */
  case '+':
    s++;
    if (--l == 0)
      return NULL;
    if (s [0] == 'i') {
      if (l != 1)
	return NULL;
      dbl1 = ScmGetReal (ur1);
      return new_cplx (s1 ? -dbl1 : dbl1, s2 ? -1.0 : 1.0);
    }
    if ((ur2 = parse_ureal (&s, &l, b, ex)) == NULL ||
	l != 1 || s [0] != 'i')
      return NULL;
    dbl1 = ScmGetReal (ur1);
    dbl2 = ScmGetReal (ur2);
    return new_real_cplx (s1 ? -dbl1 : dbl1, s2 ? -dbl2 : dbl2);
  case '@':
    s++;
    if (--l == 0)
      return NULL;
    if (s [0] == '+' || s [0] == '-') {
      if (s [0] == '-')
	s2 = 1;
      s++;
      if (--l == 0)
	return NULL;
    }
    if ((ur2 = parse_ureal (&s, &l, b, ex)) == NULL || l != 0)
      return NULL;
    dbl1 = ScmGetReal (ur1);
    if (s1)
      dbl1 = -dbl1;
    dbl2 = ScmGetReal (ur2);
    if (s2)
      dbl2 = -dbl2;
    return new_real_cplx (dbl1 * cos (dbl2), dbl1 * sin (dbl2));
  default:
    return NULL;
  }
}

static void small_small_qr (long a, long b, void **q, void **r, const char *s)
{
  long qq, rr;

  if (b == 0)
    error ("%s: %w 0 (zero divide)", s, MAKE_SINT (a));

  qq = a / b;

  if (q != NULL)
    *q = ScmLongToNumber (qq);

  if (r != NULL) {
    rr = a - b * qq;
    *r = ScmLongToNumber (rr);
  }
}

static void small_big_qr (void *x, void **q, void **r)
{
  if (q != NULL)
    *q = the_fixnum_zero;

  if (r != NULL)
    *r = x;
}

static void big_small_qr (ScmFixnum *a, long b,
			  void **q, void **r, const char *s)
{
  bigreg bb;
  int sb;

  if (b == 0)
    error ("%s: %w 0 (zero divide)", s, (void *) a);

  if (b == 1) {
    if (q != NULL)
      *q = a;
    if (*r != NULL)
      *r = the_fixnum_zero;
    return;
  }

  if (b == -1) {
    if (r != NULL)
      *r = the_fixnum_zero;
    if (q != NULL)
      *q = int_negate (a, s);
    return;
  }

  if (b < 0)
    bb = -b,
    sb = 1;
  else
    bb = b,
    sb = 0;

  if (a->length > 2) {
    bigreg rr;
    bigdig *qq;
    int lq = a->length, qsign = (a->sign == sb) ? 0 : 1;

    if (q != NULL) {
      ScmFixnum *f;
      gcs1 = a;
      NEW_FIXNUM (f, lq);
      a = gcs1;
      gcs3 = f;
      qq = f->dig;
    } else {
      PROVIDE_BIGNUM (tmp1, lq);
      qq = tmp1_dig;
    }
    if (qdiv (a->dig, bb, qq, &rr, lq) == 0)
      --lq;
    if (r != NULL)
      *r = ScmLongToNumber (a->sign ? - (long) rr : rr);
    if (q != NULL) {
      ScmFixnum *f = gcs3;
      f->sign = qsign;
      f->length = lq;		/* >= 2, -> no need to call ``simple'' */
      *q = f;
      gcs3 = NULL;
    }
  } else {
    bigreg aa, qq, rr;
    int asign = a->sign;

    aa = a->dig [0];
    if (a->length > 1)
      aa |= (bigreg) a->dig [1] << BIGBITS;
    qq = aa / bb;
    rr = aa - qq * bb;
    if (q != NULL)
      gcs1 = ScmLongToNumber (asign == sb ? qq : - (long) qq);
    if (r != NULL)
      *r = ScmLongToNumber (asign ? - (long) rr : rr);
    if (q != NULL)
      *q = gcs1,
      gcs1 = NULL;
  }
}

void sint_sint_qr (void *x, void *y, void **q, void **r, const char *s)
{
  small_small_qr (SINT_VALUE (x), SINT_VALUE (y), q, r, s);
}

void sint_int_qr (void *x, void *y, void **q, void **r, const char *s)
{
  long a = SINT_VALUE (x);
  ScmFixnum *b = y;

  if (b->length == 1)
    small_small_qr (a, b->sign ? - (long) b->dig [0] : b->dig [0], q, r, s);
  else
    small_big_qr (x, q, r);
}

void int_sint_qr (void *x, void *y, void **q, void **r, const char *s)
{
  ScmFixnum *a = x;
  long b = SINT_VALUE (y);

  if (a->length == 1)
    small_small_qr (a->sign ? - (long) a->dig [0] : a->dig [0], b, q, r, s);
  else
    big_small_qr (a, b, q, r, s);
}

void int_int_qr (void *x, void *y, void **q, void **r, const char *s)
{
  ScmFixnum *a = x, *b = y;
  long aa, bb;

  
  if (b->length == 1) {
    bb = b->sign ? - (long) b->dig [0] : b->dig [0];
    if (a->length == 1) {
      small_small_qr (aa = a->sign ? - (long) a->dig [0] : a->dig [0], bb, q,
		      r, s);
    } else
      big_small_qr (a, bb, q, r, s);
  } else if (a->length == 1)
    small_big_qr (x, q, r);
  else {
    bigdig *qq, *rr;
    int lq, lr;
    int asign = a->sign, qsign = (a->sign == b->sign) ? 0 : 1;
    ScmFixnum *tq = NULL, *tr = NULL;

    int cmp = abs_cmp (a->dig, b->dig, a->length, b->length);

    if (cmp < 0) {
      if (q != NULL)
	*q = the_fixnum_zero;
      if (r != NULL)
	*r = a;
      return;
    }

    if (cmp == 0) {
      if (q != NULL)
	*q = qsign ? MAKE_SINT (-1) : MAKE_SINT (1);
      if (r != NULL)
	*r = the_fixnum_zero;
      return;
    }

    lq = a->length - b->length + 1;
    lr = a->length + 1;

    gcs1 = a; gcs2 = b;

    if (q != NULL) {
      ScmFixnum *f;
      NEW_FIXNUM (f, lq);
      gcs3 = f;
    }

    if (r != NULL) {
      NEW_FIXNUM (tr, lr);
      rr = tr->dig;
    } else {
      PROVIDE_BIGNUM (tmp2, lr);
      rr = tmp2_dig;
    }

    if (q != NULL) {
      tq = gcs3;
      qq = tq->dig;
      gcs3 = NULL;
    } else {
      PROVIDE_BIGNUM (tmp1, lq);
      qq = tmp1_dig;
    }
    a = gcs1; b = gcs2;
    gcs1 = NULL; gcs2 = NULL;

    sdiv (a->dig, b->dig, qq, rr, a->length, b->length);

    if (q != NULL) {
      tq->sign = normalize (tq->dig, &lq, qsign);
      tq->length = lq;
      *q = simple (tq);
    }

    if (r != NULL) {
      tr->sign = normalize (tr->dig, &lr, asign);
      tr->length = lr;
      *r = simple (tr);
    }
  }
}

void real_real_qr (void *x, void *y, void **q, void **r, const char *s)
{
  double a = ScmGetReal (x);
  double b = ScmGetReal (y);
  double tmp, qq, rr;

  if (modf (a, &tmp) != 0.0 || modf (b, &tmp) != 0.0)
    error ("%s: %w %w (no integer)", s, x, y);
  if (b == 0.0)
    error ("%s: %w %w (zero divide)", s, x, y);
  rr = modf (a / b, &qq) * b;
  if (q != NULL)
    gcs1 = new_real (qq);
  if (r != NULL)
    *r = new_real (rr);
  if (q != NULL) {
    *q = gcs1;
    gcs1 = NULL;
  }
}

static int base_to_bits (int base)
{
  if (base == 2)
    return 1;
  else if (base == 8)
    return 3;
  else if (base == 16)
    return 4;
  else
    error ("number->string: %i (bad radix, must be 2, 8, 10 or 16)", base);
}

char *ScmUnparseNumber (void *x, int base)
/* returns a pointer to static memory */
{
  int bits;
  char *s;

  if (ScmTypeOf (x) == ScmType (SFixnum)) {
    int a = SINT_VALUE (x);
    bigdig abd [1];
    if (base == 10) {
      bits = 0;
      provide_parse_buf (ndig_to_ndec (1) + 1);
    } else {
      bits = base_to_bits (base);
      provide_parse_buf (BIGBITS / bits + 2);
    }
    s = parse_buf;
    if (a < 0)
      *s++ = '-',
      abd [0] = -a;
    else
      abd [0] = a;
    raw_bigdig_to_ascii (abd, 1, bits, s);
    return parse_buf;
  } else if (ScmTypeOf (x) == ScmType (Fixnum)) {
    ScmFixnum *f = x;
    if (base == 10) {
      bits = 0;
      provide_parse_buf (ndig_to_ndec (f->length) + 1);
    } else {
      bits = base_to_bits (base);
      provide_parse_buf ((BIGBITS * f->length) / bits + 2);
    }
    s = parse_buf;
    if (f->sign)
      *s++ = '-';
    raw_bigdig_to_ascii (f->dig, f->length, bits, s);
    return parse_buf;
  } else if (ScmTypeOf (x) == ScmType (Fraction)) {
    ScmFraction *f = x;
    int p;
    if (base == 10) {
      bits = 0;
      provide_parse_buf (ndig_to_ndec (f->nlength) + ndig_to_ndec (f->dlength)
			 + 2);
    } else {
      bits = base_to_bits (base);
      provide_parse_buf ((BIGBITS * (f->nlength + f->dlength)) / bits + 3);
    }
    s = parse_buf;
    if (f->sign)
      *s++ = '-';
    raw_bigdig_to_ascii (f->dig, f->nlength, bits, s);
    p = strlen (s);
    s [p] = '/';
    raw_bigdig_to_ascii (f->dig + f->nlength, f->dlength, bits, s + p + 1);
    return parse_buf;
  } else if (ScmTypeOf (x) == ScmType (Real) ||
	     ScmTypeOf (x) == ScmType (Complex)) {
    if (base != 10)
      warning ("cannot cvt inexact number to strg using radix = %i", base);
    if (ScmTypeOf (x) == ScmType (Real))
      return unparse_real (((ScmReal *) x)->val);
    else
      return unparse_cplx (((ScmComplex *) x)->re, ((ScmComplex *) x)->im);
  } else
    badarg ("number->string", x);
}

static void safe_free (void *x)
{
  if (x != NULL)
    free (x);
}

static void deallocate_all (void)
{
  int i;

  safe_free (parse_buf);
  safe_free (unparse10_buf);
  for (i = 0; i < sizeof all_tmp_regs / sizeof all_tmp_regs [0]; i++)
    safe_free (*all_tmp_regs [i]);
}
