/*
 * Cont.h -- Declarations for Scheme Continuations
 *
 * (C) m.b (Matthias Blume); Apr 1992, HUB; Jan 1993 PU/CS
 *         Humboldt-University of Berlin
 *         Princeton University, Dept. of Computer Science
 *
 * $Id: Cont.h,v 2.7 1994/09/01 19:34:07 blume Exp $
 */

# ifndef CONT_H_
# define CONT_H_

# include <assert.h>

# include "storage.h"
# include "Vector.h"
# include "Code.h"

/*
 * A continuation (i.e. activation record) will be established for each
 * ``call'' of a Scheme procedure.
 * In order to achieve tail recursion elimination some primitive procedures
 * do not return a result but arrange for another combination to sit on top
 * of the stack.  This situation is signalled by returning the number of
 * elements (including the function to call) which are on the stack.
 * If the element on the stack is the result then a C (i.e. primitive)
 * procedure returns 0.
 */
typedef
struct ScmCont {
  MEM_descriptor _;
  struct ScmCont *father;	/* parent continuation */
  void (* result) (void);	/* how to store procedure call result */
  union {			/* discriminated by type field */
    struct {			/* ``C''-continuation */
      void *environ;		/* used arbitrarily */
      void *mode;		/* current ports, error handler etc. */
      unsigned short mode_id;	/* mode identifier */
      unsigned short prim_no;	/* seq. no. of this primitive routine */
    } c;
    struct {			/* ``Scheme''-continuation */
      ScmCode *code;		/* code of current procedure */
      ScmVector *constants;	/* copy of constant vector */
      unsigned long pc;		/* program counter */
    } s;
  } u;
  unsigned short length;       	/* stack length */
  unsigned short top;		/* stack top */
  void *array [1];		/* the stack */
} ScmCont;

DCL_MEM_TYPE (Cont);
DCL_MEM_TYPE (CCont);

# define CPOS(c,i)     ((c)->array [i])
# define POS(i)        CPOS (ScmCC, i)

/* PUSH assumes that there is enough space... */
# define PUSH(x)       (assert(ScmCC->top < ScmCC->length),\
			POS (ScmCC->top++) = (x))

/* the following operations assume that the stack is not empty */
/* CPOP's, CPEEK's and CSET_TOP's argument must *not* have side effects! */
# define CPOP(c)       (assert ((c)->top > 0), CPOS (c, --(c)->top))
# define CPEEK(c)      (assert ((c)->top > 0), CPOS (c, (c)->top - 1))
# define CSET_TOP(c,x) (assert ((c)->top > 0), CPOS (c, (c)->top - 1) = (x))
# define POP()         CPOP (ScmCC)
# define PEEK()        CPEEK (ScmCC)
# define SET_TOP(x)    CSET_TOP (ScmCC, x)

/* ``safe'' routine for pushing (automatically extends ScmCC when necessary) */
extern void Push (void *);

extern void ScmPushContinuation (unsigned stackreq);
extern void
  ScmPushCContinuation (unsigned stackreq, unsigned prim_no, void *environ);
extern ScmCont *ScmNewSCont (ScmCont *father, unsigned stackcnt);
extern void ScmDisposeCont (ScmCont *cont);
extern void ScmSetContinuation (ScmCont *cont, unsigned stackcnt);
extern void ScmRevertToFatherContinuation (unsigned stackcnt);
extern int ScmContIsShared (ScmCont *cont);
extern void ScmContSetShared (ScmCont *cont);

extern void *ScmMode (int mode);
extern void ScmSetMode (int mode, void *item);
extern void ScmDirtyModeCache (int mode); /* -1 dirties all modes */

/* does a continuation accept multiple arguments? */
extern int ScmMultiCont (ScmCont *cont);

extern ScmCont *ScmCC;	/* the CURRENT CONTINUATION */

# endif
