/* 
 *  Vnc2swf - writeswf.c (generates a SWF file)
 *  $Id: writeswf.c,v 1.4 2005/10/29 06:23:46 euske Exp $
 *
 *  Copyright (C) 2002 by Yusuke Shinyama (yusuke at cs . nyu . edu)
 *  All Rights Reserved.
 *
 *  This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 *  USA.
*/


#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <math.h>
#include <assert.h>

#include <X11/Xresource.h>
#include <X11/Intrinsic.h>
#include <X11/Xmu/Converters.h>
#include <X11/Xaw/Toggle.h>

#include <zlib.h>

#include "vncviewer.h"
#include "swf.h"


#define DEBUG_SWF 0
#define DEBUG_CLIP 0
#define DEBUG_ZBUFFER 0

static char* movie_filename;
static FILE* movie_dumpfp;
static int movie_width, movie_height;
static SWFWriter* movie_writer;

static unsigned long dblbuf_len;
static unsigned char* dblbuf_src;
static unsigned char* dblbuf_compressed;
static long intervalusec;
static struct timeval tv0;

#define img_size(w,h) ((unsigned long)(w)*(unsigned long)(h)*4)

/* ZBuffer */
typedef unsigned short zdepth;
typedef struct _DisplayItem {	/* normally 8bytes */
    int cells;
    unsigned short gx, gy, gw, gh;
} DisplayItem;
typedef struct _ZBuffer {
    int width, height;
    zdepth current_depth;
    zdepth lowest_depth;
    zdepth last_depth;
    unsigned int maplen;
    zdepth* map;
    DisplayItem* items;
} ZBuffer;

static ZBuffer zbuffer;

#define ZBUFF_GRID_SIZE 4


/* get_compressed_image:
 *   Convert the area (x,y),(w,h) in a framebuffer (ZPixmap) to a RGB array.
 */
static unsigned long get_compressed_image(int x, int y, int w, int h)
{
    unsigned long dbllen = dblbuf_len;

    int i, j;
    int skip = si.framebufferWidth * myFormat.bitsPerPixel / 8;
    char* src = image->data + y * skip + x * myFormat.bitsPerPixel / 8;
    char* dest = dblbuf_src;
    for(i = 0; i < h; i++) {
	unsigned char* linebuf = src;
	src += skip;
	for(j = 0; j < w; j++) {
	    unsigned long c = 0, c1, c2, c3;
	    switch(myFormat.bitsPerPixel) {
	    case 8:
		if (appData.useBGR233) {
		    c = BGR233ToPixel[*(linebuf++)];
		} else {
		    c = *(linebuf++);
		}
		break;
	    case 16:
		c1 = *(linebuf++);
		c = (c1<<8) | *(linebuf++);
		c = Swap16IfLE(c);
		break;
	    case 32:
		c1 = *(linebuf++);
		c2 = *(linebuf++);
		c3 = *(linebuf++);
		c = (c1<<24) | (c2<<16) | (c3<<8) | *(linebuf++);
		c = Swap32IfLE(c);
		break;
	    }
	    /* RGB */
	    *(dest++) = 0;
	    *(dest++) = redRv[(c>>myFormat.redShift) & myFormat.redMax]; /* R */
	    *(dest++) = greenRv[(c>>myFormat.greenShift) & myFormat.greenMax]; /* G */
	    *(dest++) = blueRv[(c>>myFormat.blueShift) & myFormat.blueMax]; /* B */
	}
    }
    
    compress(dblbuf_compressed, &dbllen, dblbuf_src, img_size(w, h));
    return dbllen;
}


/* ZBuffer_new:
 */
static Bool ZBuffer_new(void)
{
    zbuffer.width = (movie_width + ZBUFF_GRID_SIZE-1) / ZBUFF_GRID_SIZE;
    zbuffer.height = (movie_height + ZBUFF_GRID_SIZE-1) / ZBUFF_GRID_SIZE;
    zbuffer.maplen = zbuffer.width * zbuffer.height;
    zbuffer.map = (zdepth*) malloc(zbuffer.maplen * sizeof(zdepth));
    zbuffer.items = (DisplayItem*) malloc(65536 * sizeof(DisplayItem));
#if DEBUG_ZBUFFER
    fprintf(stderr, "ZBuffer_new: %dx%d\n", zbuffer.width, zbuffer.height);
#endif /* DEBUG_ZBUFFER */
    if (!zbuffer.map) return False;
    return True;
}

/* ZBuffer_init:
 */
static void ZBuffer_init(void)
{
    int i;
    for (i = 0; i < zbuffer.maplen; i++) {
	zbuffer.map[i] = 0;
    }
    zbuffer.current_depth = 1;
    zbuffer.lowest_depth = 1;
    zbuffer.last_depth = 1;
}

/* ZBuffer_paint:
 *   Fill the given rect with the current shape id.
 *   Check if every pixel in the given rect is covered with a newer object.
 */
static Bool ZBuffer_paint(int x, int y, int w, int h)
{
    int i, j, gx, gy, gw, gh;
    zdepth* p0;

    gx = x/ZBUFF_GRID_SIZE;
    gy = y/ZBUFF_GRID_SIZE;
    gw = (x+w+ZBUFF_GRID_SIZE-1) / ZBUFF_GRID_SIZE - gx;
    gh = (y+h+ZBUFF_GRID_SIZE-1) / ZBUFF_GRID_SIZE - gy;
    if (zbuffer.width < (gx+gw)) gw = zbuffer.width - gx;
    if (zbuffer.height < (gy+gh)) gh = zbuffer.height - gy;

#if DEBUG_ZBUFFER
    {
	fprintf(stderr, "ZBuffer_paint: (%d,%d) %dx%d -> (%d,%d) %dx%d, depth=%d\n",
		x,y,w,h, gx,gy,gw,gh, zbuffer.current_depth);
    }
#endif /* DEBUG_ZBUFFER */

    p0 = zbuffer.map + gy*zbuffer.width + gx;
    for (i = 0; i < gh; i++) {
	zdepth* p = p0;
	for (j = 0; j < gw; j++, p++) {
	    int depth = *p;
	    if (depth < zbuffer.last_depth) {
		/* this pixel is not yet covered! */
		goto endloop;
	    }
	}
	p0 += zbuffer.width;
    }
    /* the rect is completely covered by another object in the same frame. */
    return False;

  endloop:
    p0 = zbuffer.map + gy*zbuffer.width + gx;
    for (i = 0; i < gh; i++) {
	zdepth* p = p0;
	for (j = 0; j < gw; j++, p++) {
	    int depth = *p;
	    if (depth) {
		assert(0 < zbuffer.items[depth].cells);
		zbuffer.items[depth].cells--;
	    }
	    *p = zbuffer.current_depth;
	}
	p0 += zbuffer.width;
    }
    {
	DisplayItem* item1;
	item1 = &zbuffer.items[zbuffer.current_depth];
	item1->cells = gw * gh;
	item1->gx = gx;
	item1->gy = gy;
	item1->gw = gw;
	item1->gh = gh;
    }

    zbuffer.current_depth++;
    return True;
}

/* ZBuffer_write_frame:
 */
static void ZBuffer_write_frame(void)
{
    int depth, lowest_depth = 0;
    /* remove objects */
    for (depth = zbuffer.lowest_depth; depth < zbuffer.last_depth; depth++) {
	DisplayItem* item1 = &zbuffer.items[depth];
	if (item1->cells == 0) {
	    item1->cells = -1;
#if DEBUG_ZBUFFER
	    fprintf(stderr, "ZBuffer_write_frame: removed=%d\n", depth);
#endif /* DEBUG_ZBUFFER */
	    SWFWriter_remove_object(movie_writer, depth);
	} else if (!lowest_depth && 0 < item1->cells) {
	    lowest_depth = depth;
	}
    }
    if (lowest_depth) {
	zbuffer.lowest_depth = lowest_depth;
    }

    /* add objects */
    for (depth = zbuffer.last_depth; depth < zbuffer.current_depth; depth++) {
	DisplayItem* item1 = &zbuffer.items[depth];
	if (item1->cells == 0) {
	    item1->cells = -1;
	} else {
	    int x = item1->gx*ZBUFF_GRID_SIZE, y = item1->gy*ZBUFF_GRID_SIZE,
		w = item1->gw*ZBUFF_GRID_SIZE, h = item1->gh*ZBUFF_GRID_SIZE;
	    unsigned long dbllen = get_compressed_image(x, y, w, h);
	    int shapeid = SWFWriter_define_rect_shape_bitmap(movie_writer, dblbuf_compressed, dbllen, w, h);
#if DEBUG_ZBUFFER
	    fprintf(stderr, "ZBuffer_write_frame: added=%d, (%d,%d) %dx%d\n", depth, x, y, w, h);
#endif /* DEBUG_ZBUFFER */
	    SWFWriter_place_object(movie_writer, shapeid, depth, x, y);
	}
    }

    zbuffer.last_depth = zbuffer.current_depth;
    SWFWriter_next_frame(movie_writer);
}


/*  Lower level swf operations.
 */

static void set_recording_state(int state)
{
    if (appData.isRecording != state) {
	ZBuffer_write_frame();
	appData.isRecording = state;
	tv0.tv_sec = tv0.tv_usec = -1;
    }
    if (appData.showStatus) {
	if (appData.showDesktop) {
	    DisplayStatus();
	} else {
	    fprintf(stderr, "=== %s\n", (state)? "[Recording]" : "[Stopped]");
	}
    }
}

static void init_recording(void)
{
    intervalusec = 1000000L / appData.frameRate;
    ZBuffer_init();
    appData.isRecording = False;
    set_recording_state(appData.startRecording || (!appData.showDesktop));
}

/* return true if the whole rectangle is clipped. */
static int cliprect(int* x0, int* y0, int* w, int* h)
{
    if (*x0 < appData.cgx0) {
	int dx = appData.cgx0 - *x0;
	*w -= dx;
	*x0 += dx;
    } else if (appData.cgx1 <= *x0) {
	return 1;
    }

    if (*y0 < appData.cgy0) {
	int dy = appData.cgy0 - *y0;
	*h -= dy;
	*y0 += dy;
    } else if (appData.cgy1 <= *y0) {
	return 1;
    }

#if DEBUG_CLIP
    fprintf(stderr, "clipped: (%d,%d)-(%d,%d) (%d,%d) %dx%d\n", 
	    appData.cgx0, appData.cgy0, appData.cgx1, appData.cgy1, *x0, *y0, *w, *h);
#endif
    return (*w <= 0) || (*h <= 0);
}



/*******************************************
  Public interfaces
******************************************/

/* InitMovie:
 *   Initialize a movie file.
 */
Bool InitMovie(char* filename, float framerate, int w, int h)
{
    movie_filename = filename;
    movie_width = w;
    movie_height = h;

    {
	/* warn if the file is going to be overwritten */
	struct stat st;
	if (stat(movie_filename, &st) != -1) {
	    fprintf(stderr, "=== WriteInitMovie: Warning: file %s already exists, overwriting.\n", 
		    appData.dumpfileString);
	}
    }
    fprintf(stderr,"=== WriteInitMovie: Pid=%d, Opening file: \"%s\" for a movie size (%d, %d), frame rate %.2f...\n", 
	    getpid(), movie_filename, movie_width, movie_height, framerate);
    
    movie_dumpfp = fopen(movie_filename, "wb");
    if (!movie_dumpfp) {
	perror("InitMovie: Cannot open a file.");
	return False;
    }
    movie_writer = SWFWriter_open(movie_dumpfp, 5, movie_width, movie_height, framerate, 0);
    if (!movie_dumpfp) {
	perror("InitMovie: Cannot initialize a movie.");
	return False;
    }

    dblbuf_len = (unsigned long)floor(img_size(w, h)*1.01+12);
    dblbuf_src = malloc(dblbuf_len);
    dblbuf_compressed = malloc(dblbuf_len);
    if (!dblbuf_src || !dblbuf_compressed) {
	perror("InitMovie: Cannot initialize a image buffer.");
	return False;
    }
    if (!ZBuffer_new()) {
	perror("InitMovie: Cannot initialize a screen z-buffer.");
	return False;
    }
    
    init_recording();
    return True;
}


/* FinishMovie:
 *   Write the movie file.
 */
Bool FinishMovie(void)
{
    assert (movie_writer != NULL);

    /* flush */
    set_recording_state(False);
    
    {
	unsigned long len = SWFWriter_close(movie_writer);
	fprintf(stderr, "=== WriteFinishMovie: %ld bytes dumped into \"%s\".\n", len, movie_filename);

	/* generate html template */
	printf("<html><body>\n<object classid=\"clsid:D27CDB6E-AE6D-11cf-96B8-444553540000\"");
	printf(" width=\"%d\" height=\"%d\"", movie_width, movie_height);
	printf(" codebase=\"http://download.macromedia.com/pub/shockwave/cabs/flash/swflash.cab#version=5,0,0,0\">\n");
	printf("<param name=\"movie\" value=\"%s\">\n", movie_filename);
	printf("<param name=\"play\" value=\"true\">\n");
	printf("<param name=\"loop\" value=\"true\">\n");
	printf("<param name=\"quality\" value=\"high\">\n");
	printf("<embed src=\"%s\" width=\"%d\" height=\"%d\"", movie_filename, movie_width, movie_height);
	printf(" play=\"true\" loop=\"true\" quality=\"high\" type=\"application/x-shockwave-flash\"");
	printf(" pluginspage=\"http://www.macromedia.com/go/getflashplayer\"></embed>\n");
	printf("</object>\n</body>\n</html>\n");
    }

    return True;
}


void FlushChangedImages(void)
{
    if (appData.isRecording) {
	struct timeval tv;
	struct timezone dummytz;
	
	gettimeofday(&tv, &dummytz);
	while(tv0.tv_sec < tv.tv_sec || 
	      (tv0.tv_sec == tv.tv_sec && tv0.tv_usec < tv.tv_usec)) {
#if DEBUG_SWF
	    fprintf(stderr, "check_next_frame: next_frame.\n");
#endif /* DEBUG_SWF */
	    ZBuffer_write_frame();
	    if (tv0.tv_sec < 0) {
		/* initialize */
		tv0 = tv;
	    }
	    tv0.tv_usec += intervalusec;
	    if (1000000L <= tv0.tv_usec) {
		tv0.tv_usec -= 1000000L;
		tv0.tv_sec++;
	    }    
	}
    }
}

void ImageChanged(int x0, int y0, int w, int h)
{
#if DEBUG_SWF
    fprintf(stderr, "ImageChanged: (%d, %d) %dx%d\n", x0, y0, w, h);
#endif /* DEBUG_SWF */
    if (!cliprect(&x0, &y0, &w, &h)) {
	ZBuffer_paint(x0, y0, w, h);
    }
}

void SetRecordingState(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    XtVaSetValues(w, XtNstate, appData.isRecording, NULL);
}

void ToggleRecording(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    set_recording_state(!appData.isRecording);
    SetRecordingState(w, event, 0, 0);
}

void ResetRecording(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    if (movie_writer) {
	fprintf(stderr, "=== Reset Recording.\n");
	SWFWriter_initialize(movie_writer);
	init_recording();
    }
}

void TurnRecordingOn(int sig)
{
    fprintf(stderr,"TurnRecordingOn called\n");
    set_recording_state(True);
}
void TurnRecordingOff(int sig)
{
    fprintf(stderr,"TurnRecordingOff called\n");
    set_recording_state(False);
}
