/*
 * ufdbpeek.c - URLfilterDB
 *
 * ufdbGuard is copyrighted (C) 2005-2023 by URLfilterDB with all rights reserved.
 *
 * Parts of the ufdbGuard daemon are based on squidGuard.
 *
 * RCS: $Id: ufdbpeek.c,v 1.41 2022/12/27 08:21:05 root Exp root $
 */

#include "ufdb.h"
#include "ufdbchkport.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <signal.h>

#include "sg.h"

#ifdef __cplusplus
extern "C" {
#endif


static int loadDatabaseTables = 0;

static void usage( void )
{
   fprintf( stderr, "ufdbpeek is Copyright (C) 2011-2023 by URLfilterDB B.V. with all rights reserved.\n" );
   fprintf( stderr, "ufdbpeek version " UFDB_VERSION "\n" );
   fprintf( stderr, "usage: ufdbpeek [-p <port>] [-C CAcertsFile] [-D CAcertsDir] [-d] [-A] [-F] [-G] [-S] [-Y] <domain>\n" );
   exit( 1 );
}


static void doLoadDatabaseTables( char * dbhome )
{
   int    i;
   char   fname[1024];
   char * category;

   char * categories[] = {
      "safe",
      "adult",
      "alcohol",
      "arms",
      "ms-data-collection",
      "qmovies",
      "p2p",
      "proxies",
      "malware",
      "warez",
      "phishtank",
      "gambling",
      "violence",
      "drugs",
      "softdrugs",
      "illegal",
      "extappl",
      "socialnet",
      "youtube",
      "vimeo",
      "dailymotion",
      "webtv",
      "webradio",
      "chat",
      "dating",
      "games",
      "food",
      "entertain",
      "trading",
      "finance",
      "jobs",
      "shops",
      "sports",
      "travel",
      "news",
      "webmail",
      "forum",
      "private",
      "toolbars",
      "audio-video",
      "ads",
      "mozilla",
      "religion",
      "searchengine",
      "education",
      "health",
      "housing",
      "dnsoverhttps",
      "checked",
      NULL
   };

   // prevent compiler warning
   if (dbhome == NULL)
      { ; }

   for (i = 0;  (category = categories[i]) != NULL;  i++)
   {
      ufdbCategory( ufdbStrdup(category) );
      sprintf( fname, "%s/domains", category );
      ufdbCategoryDomainList( ufdbStrdup(fname) );
#if 0
      sprintf( fname, "%s/expressions", category );
      struct stat stbuf;
      if (stat( fname, &stbuf ) == 0  &&  stbuf.st_size > 0)
         ufdbCategoryExpressionList( ufdbStrdup(fname), "n" );          // XXX
#endif
      ufdbCategoryEnd();
   }
}


int main( int argc, char * argv[] )
{
   int    opt;
   int    port;
   int    certErrors;
   char * content;
   char * t;
   char * url;
   char   cn[1024];

   //ufdbGlobalSetLogging( 0 );

   UFDBinitializeGV( &ufdbNewGV );
   ufdbNewGV.Argv = argv;
   UFDBresetGV( &ufdbNewGV );

   ufdbNewGV.pid = ufdbGV.pid = getpid();
   strcpy( ufdbGV.progname, "ufdbpeek" );
   strcpy( ufdbNewGV.progname, "ufdbpeek" );
   strcpy( ufdbNewGV.logDirectory, "." );
   strcpy( ufdbGV.logDirectory, "." );
   //ufdbSetGlobalErrorLogFile( NULL, NULL, 0 );

   if (argc <= 1)
      usage();

   ufdbGV.debug = ufdbNewGV.debug = 0;
   ufdbNewGV.peek = 1;
   ufdbNewGV.httpsOfficialCertificate = 0;
   ufdbNewGV.httpsEnforceContentPeek = 1;
   ufdbNewGV.httpsWithHostname = 1;
   ufdbNewGV.YoutubeEdufilter = 0;
   ufdbNewGV.allowGoogleHTTPSusingIP = 1;
   ufdbNewGV.tunnelCheckMethod = UFDB_API_HTTPS_CHECK_AGRESSIVE;
   ufdbSetSignalHandler( SIGPIPE, SIG_IGN );
   ufdbNewGV.reconfig = UFDB_RECONFIGR_NONE;
   //ufdbSetGlobalErrorLogFile( NULL, NULL, 0 );

   port = 443;
   url = NULL;

   while ((opt = getopt( argc, argv, "?C:D:dhp:sAFGSY" )) > 0)
   {
      switch (opt) {
      case 'C':
      	 strcpy( ufdbNewGV.CAcertsFile, optarg );
	 break;
      case 'D':
      	 strcpy( ufdbNewGV.CAcertsDir, optarg );
	 break;
      case 'd':
	 loadDatabaseTables = 1;
	 ufdbNewGV.debug++;
         break;
      case 'p':
         port = atoi( optarg );
	 break;
      case 'A':
	 ufdbNewGV.debugAim = 1;
	 ufdbNewGV.AimOverHttps = 1;
         break;
      case 'F':
	 ufdbNewGV.debugFBchat = 1;
	 ufdbNewGV.FBchatOverHttps = 1;
         break;
      case 'G':
	 ufdbNewGV.debugGtalk = 1;
	 ufdbNewGV.GtalkOverHttps = 1;
         break;
      case 's':
	 strcpy( ufdbNewGV.peekServerName, "itunes.apple.com" );
         break;
      case 'S':
	 ufdbNewGV.debugSkype = 1;
	 ufdbNewGV.SkypeOverHttps = 1;
         break;
      case 'Y':
	 ufdbNewGV.debugYahooMsg = 1;
	 ufdbNewGV.YahooMsgOverHttps = 1;
         break;
      case '?':
      case 'h':
          usage();
	  exit( 0 );
	  break;
      }
   }

   if (optind < argc)
      url = argv[optind];
   else
      usage();

   ufdbLogMessage( "ufdbpeek v" UFDB_VERSION " is Copyright (C) 2011-2023 by URLfilterDB B.V. "
                   "with all rights reserved." );

   if (loadDatabaseTables)
      doLoadDatabaseTables( DEFAULT_DBHOME );

   ufdbGV = ufdbNewGV;
   ufdbGV.peek = 1;
   //ufdbSetGlobalErrorLogFile( NULL, NULL, 0 );

   UFDBinitHTTPSchecker();
   if (strncmp( url, "http://", 7 ) == 0)
      url += 7;
   if (strncmp( url, "https://", 8 ) == 0)
      url += 8;
   if ((t = strchr( url, '@' )) != NULL)
      *t = '\0';
   if ((t = strstr( url, ":443" )) != NULL)
      *t = '\0';
#if 0
   if ((t = strchr( url, ':' )) != NULL)
      *t = '\0';
#endif
   if ((t = strchr( url, '/' )) != NULL)
      *t = '\0';

   certErrors = 0;
   content = NULL;
   (void) UFDBsslPeekServer( url, port, cn, &certErrors, &content, 0 );

   if (content != NULL)
   {
      if (strlen(content) > 30000)
	 ufdbLogMessage( "%30000.00000s", content );
      else
         ufdbLogMessage( "%s", content );
   }

   if (ufdbGlobalLogfile != NULL)
   {
      fflush( ufdbGlobalLogfile );
      fclose( ufdbGlobalLogfile );
   }

   return 0;
}


#ifdef __cplusplus
}
#endif
