/*
 * ufdblib.h - URLfilterDB
 *
 * ufdbGuard is copyrighted (C) 2005-2020 by URLfilterDB with all rights reserved.
 *
 * RCS $Id: ufdblib.h,v 1.113 2022/12/27 08:17:33 root Exp root $
 */

#ifndef UFDB_UFDBLIB_H_INCLUDED
#define UFDB_UFDBLIB_H_INCLUDED

#ifdef __cplusplus
extern "C" {
#endif

#include "ufdb.h"
#include "ufdb_globals.h"

#include <time.h>
#include <stdio.h>
#include <sys/times.h>

#if !UFDB_BARE_METAL_SUPPORT
#include <sys/utsname.h>
#endif


void increment_UFDB_API_num_https( void );

void UFDBdropPrivileges( const char * username );
void UFDBraisePrivileges( const char * username, const char * why );

typedef unsigned long ufdb_memdb_hash_t;

typedef struct UFDBmemDBkv
{
   char *               key;
   char *               value;
   ufdb_memdb_hash_t    hash;
   struct UFDBmemDBkv * next;
} UFDBmemDBkv;

struct UFDBmemDB
{
   unsigned int      tableSize;
   unsigned int      nEntries;
   unsigned int      optimalMaxEntries;
   UFDBmemDBkv **    table;
};

/* The following functions are not all threadsafe !
 * A memDB is created by one thread while other threads have no access.
 * After the creation and initialisation a memDB is static and
 * multiple threads can search in the memDB.
 */
struct UFDBmemDB * UFDBmemDBinit( void );
void UFDBmemDBinsert( struct UFDBmemDB * db, const char * key, const char * value );
int  UFDBmemDBfind( struct UFDBmemDB * db, const char * key, char ** value );
int  UFDBmemDBfindDomain( struct UFDBmemDB * db, const char * domain );
void UFDBmemDBdeleteDB( struct UFDBmemDB * db );
void UFDBmemDBprintUserDB( const char * prefix, struct UFDBmemDB * db );

void UFDBappInit( void );
unsigned long UFDBappMemUsage( void );

void UFDBtimerInit( struct tms * t );
void UFDBtimerStop( struct tms * t );
void UFDBtimerPrint( struct tms * t, const char * tag );
void UFDBtimerPrintString( char * line, struct tms * t, const char * tag );
void UFDBlogTimer( struct tms * t, char * tag );

int    UFDBgettimeofday( struct timeval * tv, struct timezone * tz );
time_t UFDBtime( void );
int    UFDBhasRTC( void );

int UFDBloadExpressions( 
   struct ufdbRegExp ** exprList,
   char *               file     );

struct ufdbRegExp * UFDBoptimizeExprList( 
   char *               reSource,
   struct ufdbRegExp *  reList );

void ufdbFreeRegExprList( struct ufdbRegExp * re );

int    UFDBregexec( const void * preg, const char * string, size_t nmatch, regmatch_t pmatch[], int eflags );

struct Source * UFDBgetSourceList( void );
void UFDBsetSourceList( struct Source * sl );

int  ufdbSetCPU( char * CPUspec );
int  ufdbSetThreadCPUaffinity( int thread_num );
void ufdbResetCPUs( void );

const char * ufdbAPIstatusString( int api_code );
const char * ufdbDBstat2string( int status );
const char * ufdbStatus2string( int status );

#define URL_HIST_SIZE 180000

int    ufdbVerifyURLallCats( UFDBrevURL * revURL, char * URL );
void   ufdbRegisterUnknownURL( char * webserver, int portnumber, char * referer );
char * ufdbGetUnknownURLs( void );
void   ufdbResetUnknownURLs( void );

int UFDBopenSocket( const char * serverName, int port );

void UFDBregisterCountedIP( const char * address );
void UFDBinitializeIPcounters( void );
unsigned long UFDBgetNumberOfRegisteredIPs( void );

void UFDBregisterCountedUser( const char * username );
void UFDBinitializeUserCounters( void );
unsigned long UFDBgetNumberOfRegisteredUsers( void );

struct ufdbSetting {
  char *           name;
  char *           value;
  struct ufdbSetting * next;
};

void   ufdbLogFile( int, int, char * );
void   ufdbGlobalSetLogging( int logging );
void   ufdbRotateLogfile( char * );
void   UFDBrotateLogfile( void );
void   ufdbSetGlobalErrorLogFile( char * logdir, char * basename, int mutex_is_used );
void   UFDBcloseFilesNoLog( void );

void   ufdbLogMessage( const char * format, ... )
#ifdef __GNUC__
__attribute__ ((format(printf,1,2)))
#endif
;
void   ufdbLogError( const char *, ... )
#ifdef __GNUC__
__attribute__ ((format(printf,1,2)))
#endif
;
void   ufdbLogFatalError( const char *, ... )
#ifdef __GNUC__
__attribute__ ((format(printf,1,2)))
#endif
;

typedef void (*ufdb_crashfun_t)(void);
void ufdbRegisterFatalErrorCrashfun( ufdb_crashfun_t crfun );

#if !UFDB_BARE_METAL_SUPPORT
void ufdbGetSysInfo( struct utsname * si );
long ufdbGetNumCPUs( void );
#endif

void UFDBlogConfig( struct ufdbGV * gv );
int  UFDBchangeStatus( int status );

char * UFDBparseIPv6address( char * url, char * domain );
void   UFDBupdateURLwithNormalisedDomain( char * url, char * newDomain );
void   UFDBnormaliseIPv4( char * domain );

/*
 * strip a URL;
 * remove http:// prefix, remove www[0-9*]. prefix,
 * remove port number, remove username and password
 */
void UFDBstripURL(
   char * URL,                  /* input URL string */
   char * strippedUrl,          /* output char array (must be 1024 bytes) */
   char * domain,               /* output char array (must be 1024 bytes) */
   char * protocol,             /* output char array (must be 16 bytes) */
   int  * portnumber );         /* output integer */

void UFDBstripURL2( 
   char * URL, 			/* input URL string */
   int    stripwwwprefix,	/* input flag for stripping "www." prefix from URL */
   char * strippedUrl,  	/* output char array (must be UFDB_MAX_URL_LENGTH bytes) */
   char * domain,       	/* output char array (must be 1024 bytes) */
   char * protocol,		/* output char array (must be 16 bytes) */
   int  * portnumber );		/* output integer */

/*
 * Strip a domain (used by ufdbguard-for-bind) :
 * remove www[0-9*]. prefix,
 * convert characters to lower case.
 * cannot convert IP addresses
 */
UFDB_GCC_HOT
void UFDBstripDomain(
   const char * domain,         /* input domain */
   char *       stripped  );    /* output char array (must be 1024 bytes) */

char * UFDBprintable( char * string );

char * UFDBfgets( 
   char * requestBuffer, 
   int    bufferSize, 
   FILE * fp );

char * UFDBfgetsNoNL( char * s, int size, FILE * stream );

int UFDBcalcCksum( char * mem, long size );

void ufdbExecutePstack( const char * reason );

#ifdef __cplusplus
}
#endif

#endif

