/*
 * ufdbdb.c - URLfilterDB
 *
 * ufdbGuard is copyrighted (C) 2005-2021 by URLfilterDB B.V. with all rights reserved.
 *
 * Parts of ufdbGuard are based on squidGuard.
 * This module is NOT based on squidGuard.
 *
 * RCS $Id: ufdbdb.c,v 1.109 2022/12/26 23:17:39 root Exp root $
 */

/* This module is well tested and stable for a long time.
 * For maximum performance _FORTIFY_SOURCE is undefined.
 */
#undef _FORTIFY_SOURCE

/* to inline string functions with gcc : */
#if defined(__OPTIMIZE__) && defined(__GNUC__)  &&  defined(GCC_INLINE_STRING_FUNCTIONS_ARE_FASTER)
#define __USE_STRING_INLINES  1
#endif


#include "ufdb.h"
#if UFDBSS_SQUID
#include "sg.h"
#endif
#include "ufdblib.h"
#include "ufdbdb.h"

#if UFDB_API_USE_DP
#include "dpufdb.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>
#include <time.h>
#include <arpa/inet.h>
#if UFDB_PTHREAD_SUPPORT
#include <pthread.h>
#endif
#if !UFDB_BARE_METAL_SUPPORT
/* TO-DO: evaluate use of syslog() */
#include <syslog.h>
#include <sys/stat.h>
#include <fcntl.h>
#endif
#include <errno.h>
#include <sys/types.h>
#include <sys/time.h>

#if UFDB_BARE_METAL_SUPPORT && __OCTEON__
#include "dpufdb.h"
#endif

#if 0
#include <sys/socket.h>
#if HAVE_UNIX_SOCKETS
#include <sys/un.h>
#endif
#include <netdb.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#endif

#if __linux__  &&  HAVE_MADVISE && !UFDB_BARE_METAL_SUPPORT
#include <sys/mman.h>
#endif

#if UFDB_BZ2LIB_SUPPORT
#include "bzlib.h"
#endif
#include "zlib.h"

#ifdef __GNUC__
#define GCC_PREFETCH( mem, rw, locality )  __builtin_prefetch( mem, rw, locality )
#else
#define GCC_PREFETCH( mem, rw, locality )  /* void */
#endif

#ifdef __cplusplus
extern "C" {
#endif

#define UFDB_DO_DEBUG      0
#define UFDB_DEBUG_REVURL  0
#define UFDB_DEBUG_IPV6    0

#if UFDB_DO_DEBUG || 0
#define DEBUG(x) ufdbLogMessage x 
#define DEBUGP(x)
#else
#define DEBUG(x) 
#define DEBUGP(x)
#endif

// the following 4 variables are not declared UFDB_SHARED since only one thread/core will use them
static unsigned char *    tableMemStart;        // for debugging
static struct UFDBtable * tableIndex;
static int                tableIndex_i;

#if defined(__GNUC__)  &&  __GCC_HAVE_SYNC_COMPARE_AND_SWAP_4  &&  __SIZEOF_LONG__ == 4
   // nothing
#elif defined(__GNUC__)  &&  __GCC_HAVE_SYNC_COMPARE_AND_SWAP_8  &&  __SIZEOF_LONG__ == 8
   // nothing
#else
UFDB_SHARED static ufdb_mutex counterMutex = ufdb_mutex_initializer;
#endif


UFDBthreadAdmin * UFDBallocThreadAdmin( void )
{
   UFDBthreadAdmin * admin;

#if __OCTEON__
   if (ufdbGV.debug)
      ufdbLogMessage( "UFDBallocThreadAdmin: before ufdbMallocAligned(%d,%d)",
                      UFDB_CACHELINE_SIZE, (int) sizeof(UFDBthreadAdmin) );
#endif

   admin = (UFDBthreadAdmin *) ufdbMallocAligned( UFDB_CACHELINE_SIZE, sizeof(UFDBthreadAdmin) );

#if __OCTEON__
   if (ufdbGV.debug > 1)
      ufdbLogMessage( "UFDBallocThreadAdmin: after ufdbMallocAligned newmem=%p", admin );
#endif

   memset( admin, 0, sizeof(UFDBthreadAdmin) );

#if defined(__GNUC__)  &&  __GCC_HAVE_SYNC_COMPARE_AND_SWAP_4  &&  __SIZEOF_LONG__ == 4
   (void) __sync_add_and_fetch( &UFDB_API_num_threads, 1 );
#elif defined(__GNUC__)  &&  __GCC_HAVE_SYNC_COMPARE_AND_SWAP_8  &&  __SIZEOF_LONG__ == 8
   (void) __sync_add_and_fetch( &UFDB_API_num_threads, 1 );
#else
   ufdb_mutex_lock( &counterMutex );
   UFDB_API_num_threads++;
   ufdb_mutex_unlock( &counterMutex );
#endif

   return admin;
}


void UFDBfreeThreadAdmin( UFDBthreadAdmin * handle )
{
#if __OCTEON__
   if (ufdbGV.debug)
      ufdbLogMessage( "UFDBfreeThreadAdmin" );
#endif

   ufdbFree( (void*) handle );

#if defined(__GNUC__)  &&  __GCC_HAVE_SYNC_COMPARE_AND_SWAP_4  &&  __SIZEOF_LONG__ == 4
   (void) __sync_sub_and_fetch( &UFDB_API_num_threads, 1 );
#elif defined(__GNUC__)  &&  __GCC_HAVE_SYNC_COMPARE_AND_SWAP_8  &&  __SIZEOF_LONG__ == 8
   (void) __sync_sub_and_fetch( &UFDB_API_num_threads, 1 );
#else
   ufdb_mutex_lock( &counterMutex );
   UFDB_API_num_threads--;
   ufdb_mutex_unlock( &counterMutex );
#endif
}


void UFDBinitThreadAdmin( UFDBthreadAdmin * admin )
{
   memset( admin, 0, sizeof(UFDBthreadAdmin) );

#if defined(__GNUC__)  &&  __GCC_HAVE_SYNC_COMPARE_AND_SWAP_4  &&  __SIZEOF_LONG__ == 4
   (void) __sync_add_and_fetch( &UFDB_API_num_threads, 1 );
#elif defined(__GNUC__)  &&  __GCC_HAVE_SYNC_COMPARE_AND_SWAP_8  &&  __SIZEOF_LONG__ == 8
   (void) __sync_add_and_fetch( &UFDB_API_num_threads, 1 );
#else
   ufdb_mutex_lock( &counterMutex );
   UFDB_API_num_threads++;
   ufdb_mutex_unlock( &counterMutex );
#endif
}


UFDB_GCC_HOT
static inline char * findDomainEnd( char * url )
{
   /* works also URLs with IPv6: [a:b::0]/foo.html */
   return url + strcspn( url, "/?&;#" );
}


/* mystrchr() is faster than the standard strchr() since strings are generally short */
UFDB_GCC_HOT
static inline char * mystrchr( char * string, char letter )
{
   while (*string != '\0')
   {
      if (*string == letter)
         return string;
      string++;
   }
   return NULL;
}


/* Mutex locks for _allocRevURL() are a thread synchronisation bottleneck */
/* so it is better that each thread has its own array of UFDBrevURL* */

UFDB_GCC_HOT
static inline UFDBrevURL * _allocRevURL( 
   UFDBthreadAdmin * admin )
{
   UFDBrevURL *      newru;
   
   if (admin->iusage < MAX_REVURLS)
   {
      admin->myArray[admin->iusage].ipv4 = 0;
      admin->myArray[admin->iusage].ipv6 = 0;
      admin->myArray[admin->iusage].ippath = 0;
      admin->myArray[admin->iusage].dummy = 0;
      admin->iusage++;
      return &(admin->myArray[admin->iusage-1]);
   }

   newru = (UFDBrevURL *) ufdbMallocAligned( UFDB_CACHELINE_SIZE, sizeof(UFDBrevURL) );

   return newru;
}


#if UFDB_DBFORMAT_3_1
static inline int atendofIPvxURL( const unsigned char * end )
{
   if (*end == '\0')
      return 1;

   // also treat 1.2.3.4[:443]/ as an IP address
   if (*end == ':')
   {
      end++;
      if (isdigit(*end))
      {
         end++;
         while (isdigit(*end))
            end++;
      }
      else
         return 0;
   }
   if (*end == '/')
      end++;

   if (*end == '\0')
      return 1;

   return 0;
}
#endif


UFDB_GCC_HOT
static UFDBrevURL * parseIPv6URL( 
   UFDBthreadAdmin * admin,
   unsigned char *   URL )
{
   unsigned char *   oldBracket;
   unsigned int      size;
   UFDBrevURL *      newHead;
   UFDBrevURL *      newPath;

#if UFDB_DEBUG_IPV6
   ufdbLogMessage( " parseIPv6URL: %s", URL );
#endif

   oldBracket = (unsigned char *) mystrchr( (char *) URL, ']' );
   if (oldBracket == NULL)
   {
      oldBracket = URL;
      while (*oldBracket != '\0'  &&  *(oldBracket+1) != '/')
	 oldBracket++;
      if (oldBracket - URL > (int) sizeof(UFDBurlPart)-2)
         oldBracket = URL + sizeof(UFDBurlPart) - 2;
   }
   else
   {
#if UFDB_DBFORMAT_3_1
      struct in6_addr addr;
      *oldBracket = '\0';
      if (inet_pton( AF_INET6, (char*) URL+1, (void*) &addr ))
      {
         DEBUG(( "URL has an IPv6 address\n" ));
         *oldBracket = ']';
         newHead = _allocRevURL( admin );
         newHead->next = NULL;
         newHead->ipv6 = 1;
         newHead->ippath = !atendofIPvxURL( oldBracket+1 );
         memcpy( (char*) newHead->part, (char*) URL, oldBracket - URL + 1 );
         newHead->part[oldBracket - URL + 1] = '\0';
         newHead->hashval = ufdbTable3Hash( newHead->part );          // required for lookup in URL table (3.0 and 3.1+ippart)
         // data in in6_addr is in network format and we need to convert it to host format:
         uint64_t * n;
         n = (uint64_t*) &addr.s6_addr;
         n[0] = be64toh( n[0] );
         n[1] = be64toh( n[1] );
         memcpy( (void*) &newHead->part[64], (void*) &addr, sizeof(addr) );
         goto checkpath;
      }
      else
      {
         *oldBracket = ']';
         ufdbLogError( "parseIPv6URL: illegal IPv6 address in URL: %s", URL );
         return NULL;
      }
#else
      ;
#endif
   }

   newHead = _allocRevURL( admin );
   size = (unsigned char *) oldBracket - URL + 1;
   if (size > (int) sizeof(UFDBurlPart)-2)
      size = sizeof(UFDBurlPart) - 2;
   memcpy( newHead->part, URL, size );
   newHead->part[ size ] = '\0';
#if UFDB_DBFORMAT_3
   newHead->hashval = ufdbTable3Hash( newHead->part );   // required for lookup in URL table (3.0 and 3.1+ippart)
#endif
#if UFDB_DBFORMAT_3_1
checkpath:
#endif
   if (*(oldBracket+1) == '/')
   {
      newPath = _allocRevURL( admin );
      newHead->next = newPath;
      memccpy( newPath->part, oldBracket+1, '\0', sizeof(UFDBurlPart)-2 );
      newPath->part[ sizeof(UFDBurlPart)-2 ] = '\0';
      newPath->next = NULL;
   }
   else
   {
      newHead->next = NULL;
   }
   
   return newHead;
}


UFDB_GCC_HOT
static UFDBrevURL * parseFQDNandPath( 
   UFDBthreadAdmin * admin,
   char *            domain,
   char *            path,
   int               URL4table  )
{
   char *            d;
   char *            orig_domain;
   unsigned char *   newpart;
   UFDBrevURL *      head;
   UFDBrevURL *      prev;
   UFDBrevURL *      tail;

#if UFDB_DEBUG_REVURL
   DEBUG(( "   parseFQDNandPath  domain '%s'  path '%s'  URL4table %d", domain, path, URL4table ));
#endif

   prev = NULL;
   tail = NULL;				/* the hand-optimised unrolled loop is slightly faster */
   orig_domain = domain;

   while (1) 
   {
      head = _allocRevURL( admin );
      head->next = prev;
      if (tail == NULL)
         tail = head;

      d = (char *) head->part;
      while (1)						/* unrolled 6 times */
      {
	 if (*domain == '.'  ||  *domain == '\0')
	    break;
	 *d++ = *domain++;

	 if (*domain == '.'  ||  *domain == '\0')
	    break;
	 *d++ = *domain++;

	 if (*domain == '.'  ||  *domain == '\0')
	    break;
	 *d++ = *domain++;

	 if (*domain == '.'  ||  *domain == '\0')
	    break;
	 *d++ = *domain++;

	 if (*domain == '.'  ||  *domain == '\0')
	    break;
	 *d++ = *domain++;

	 if (*domain == '.'  ||  *domain == '\0')
	    break;
	 *d++ = *domain++;

         /* test for error: someone uses a label/path that is way too long */
	 if (d > (char *) &head->part[sizeof(UFDBurlPart) - 6 - 4])	
	 {
	    while (*domain != '.'  &&  *domain != '\0')
	       domain++;
	    break;						// do not report an error; just truncate
	 }
      }
      *d = '\0';

#if UFDB_DBFORMAT_3 || UFDB_DBFORMAT_3_1
      head->hashval = ufdbTable3Hash( head->part );
#endif

      if (*domain == '\0')
      {                                                         // parse path and parameters
	 if (*path != '\0')
	 {
	    int    n;
	    int    param_len;
	    int    value_len;
	    char * param_end;
	    char * orig_path;

	    tail->next = _allocRevURL( admin );
	    tail = tail->next;
	    tail->next = NULL;
	    orig_path = path;
	    newpart = tail->part;
	    *newpart = '\0';
	    while (*path != '\0')
	    {
               // UFDBstripURL2() and parseLine() already obey ufdbGV.parseURLparameters so
               // we do not have to deal with it here.
	       if (*path == '?') 
	       {
                  /* if no '/' in www.example.com?foo=bar, insert a '/'  */
		  if (path == orig_path)				
		  {
		     *newpart++ = '/';
		  }
		  *newpart = '\0';

                  /* allocate new part for the parameters */
		  tail->next = _allocRevURL( admin );		
		  tail = tail->next;
		  tail->next = NULL;
#if UFDB_DBFORMAT_3
                  tail->hashval = 0;
#endif
		  newpart = tail->part;
		  *newpart = '\0';
		  n = 0;

		  path++;                               // skip '?'
		  if (URL4table)                        // used in ufdbGenTable
		  {
                     char * nextqm;
                     if ((nextqm = strchr( path, '?' )) != NULL)
                     {
                        ufdbLogError( "URL '%s%s' has two or more '?'  (truncating second)",
                                      orig_domain, orig_path );
                        *nextqm = '\0';
                     }
		     while (*path != '\0')
		     {
			param_end = path;
			while (*param_end != '\0'  &&  *param_end != '='  &&  *param_end != '&')
			   param_end++;
			if (*param_end == '\0'  ||  *param_end == '&')
			{
			   if (ufdbGV.debug > 2)
			      ufdbLogMessage( "debug notice: URL '%s%s' parameter '%s' has no value", 
                                              orig_domain, orig_path, path );
			}
			param_len = param_end - path;
                        if (param_len == 0)     // bad parameter
                        {
                           if (ufdbGV.debug)
                              ufdbLogMessage( "URL '%s%s' :\n"
                                              "parameter list has a part without parameters "
                                              "(& at begin or end, <void>=value, or &&)", 
                                              orig_domain, orig_path );
                           path++;
                           continue;
                        }
			if (param_len >= (int) sizeof(UFDBurlPart))	// skip large parameter IDs
			{
                           if (ufdbGV.debug)
                              ufdbLogMessage( "URL '%s%s' :\n"
                                              "skipping too long parameter name '%s'", 
                                              orig_domain, orig_path, path );
			   path = param_end;
			   while (*path != '\0'  &&  *path != '&')
			      path++;
			   continue;
			}
			while (path < param_end)			// copy parameter ID to new part
			   *newpart++ = *path++;
			*newpart = '\0';
#if UFDB_DEBUG_REVURL
			ufdbLogMessage( "new part for parameter ID '%s'", tail->part );
#endif

			if (*param_end == '='  ||  *param_end == '&'  || *param_end == '\0')
			{
			   tail->next = _allocRevURL( admin );		// allocate new part for the value
			   tail = tail->next;
			   tail->next = NULL;
			   newpart = tail->part;
			   *newpart = '\0';

                           if (*param_end == '=')
                              path++;					// skip '='
                              
                           param_end = path;
                           while (*param_end != '\0'  &&  *param_end != '&')
                              param_end++;
                           value_len = (int) (param_end - path);
			   if (value_len >= (int) sizeof(UFDBurlPart))	// value is too large; truncate it
			   {
                              if (ufdbGV.debug)
                                 ufdbLogMessage( "URL '%s%s' truncating too long value '%s'"
                                                 " and skipping other parameters",
                                                 orig_domain, orig_path, path );
			      param_end = path + sizeof(UFDBurlPart) - 1;
			      *param_end = '\0';
			   }

			   while (path < param_end)			// copy value to new part
			      *newpart++ = *path++;
			   *newpart = '\0';
#if UFDB_DEBUG_REVURL
			   ufdbLogMessage( "new part for value '%s'  path '%s'", 
                                           tail->part, path );
#endif

			   if (*path != '\0')
			   {
			      path++;                                   /* skip '&' */
                              if (*path != '\0')
                              {
                                 tail->next = _allocRevURL( admin );	/* allocate the next parameter */
                                 tail = tail->next;
                                 tail->next = NULL;
                                 newpart = tail->part;
                                 *newpart = '\0';
                              }
			   }
			}
		     }
		     return head;
		  }
		  else							// !URL4table
		  {                                                     // this code is used in ufdbGuard (API)
		     while (*path != '\0')
		     {
			param_end = path;
			while (*param_end != '\0'  &&  *param_end != '&')
			   param_end++;
			param_len = (int) (param_end - path);
                        if (param_len == 0)                             // bad parameter
                        {
#if UFDB_DEBUG_REVURL
			   ufdbLogMessage( "URL '%s%s' :\n"
                                           "parameter list has a part without parameters "
                                           "(& at beginning or end or &&)", 
                                           orig_domain, orig_path );
#endif
                           path++;
                           continue;
                        }
			if (param_len >= (int) sizeof(UFDBurlPart))	// the parameter+value is too large
			{
#if UFDB_DEBUG_REVURL
			   ufdbLogMessage( "parseFQDNandPath: skipped parameter which is too long '%s'", path );
#endif
			   path = param_end;
			   if (*path != '\0')
			      path++;
			   continue;
			}
			else if (n + param_len >= (int) sizeof(UFDBurlPart))	// must allocate new RevUrl part
			{
#if UFDB_DEBUG_REVURL
			   ufdbLogMessage( "parseFQDNandPath: n=%d  strlen=%d  this part of URL: '%s'", 
                                           n, (int) strlen((char*)tail->part), tail->part );
			   ufdbLogMessage( "parseFQDNandPath: allocated new part for remaining part '%s'",
                                           path );
#endif
			   *newpart = '\0';
			   tail->next = _allocRevURL( admin );
			   tail = tail->next;
			   tail->next = NULL;
			   newpart = tail->part;
			   *newpart = '\0';
			   n = 0;
			}
                                                        // append the parameter+value to the current RevUrl part
                        if (n > 0)
                           *newpart++ = '&';
			n += param_len + 1;
			while (path < param_end)
			   *newpart++ = *path++;
                        if (*path == '&')
                           path++;
                        *newpart = '\0';
#if UFDB_DEBUG_REVURL
			ufdbLogMessage( "parseFQDNandPath: n=%d  copied one parameter to '%s'",
                                        n, tail->part );
#endif
		     }
		  }
	       }
	       else     // not yet seen a '?'
	       {
		  if (path == orig_path + sizeof(UFDBurlPart) - 1)
		     break;
		  *newpart++ = *path++;
	       }
	    }
	    *newpart = '\0';
	 }
         return head;
      }

      domain++;
      prev = head;
   }

   return head;		/* NOTREACHED */
}


/* Parse the URL and reverse it into a linked list 
 * e.g. my.sex.com becomes "com" -> "sex" -> "my" -> NULL
 *      youtube.com/watch?v=ahuJBU7hji2K  becomes "com" -> "youtube" -> "/watch" -> "v=ahuJBU7hji2K"
 *
 * The input parameter, URL, must be a writable array of characters (size = UFDB_MAX_URL_LENGTH).
 */
UFDB_GCC_HOT 
UFDBrevURL * UFDBgenRevURL( 
   UFDBthreadAdmin * admin,
   unsigned char *   URL )
{
#if UFDB_DBFORMAT_3_1
   struct in_addr    addr;
#endif
   char *            d;
   char              domain[UFDB_MAX_URL_LENGTH];

   if (URL[0] == '[')
      return parseIPv6URL( admin, URL );

   d = findDomainEnd( (char *) URL );

   if (d == (char *) URL)				/* domain is empty: illegal URL */
   {
      ufdbLogError( "URL has no domain: '%s'", URL );
      return NULL;
   }
   else
   {
      int dlen = d - (char *) URL;

      if (dlen >= UFDB_MAX_URL_LENGTH-1)
      {
         ufdbLogError( "URL exceeds maximum length: %s", URL );
         return NULL;
      }
      else
      {
	 /* copy the domainname (the dots will be replaced by \0) and find the start of the path */
	 memcpy( domain, (char *) URL, dlen );
	 domain[dlen] = '\0';
	 URL += dlen;

	 if (domain[dlen-1] == '.')		/* remove dot after FQDN: also match www.example.com. */
         {
	    domain[dlen-1] = '\0';
            dlen--;
         }

#if UFDB_DBFORMAT_3_1
         if (domain[dlen-1] <= '9'  &&  inet_pton( AF_INET, domain, &addr ))
         {
            UFDBrevURL * newHead;
            DEBUG(( "URL has an IPv4 address\n" ));
            newHead = parseFQDNandPath( admin, domain, (char *) URL, 0 );
            newHead->ipv4 = 1;
            newHead->ippath = !atendofIPvxURL( URL );
            // convert network format to host format
            addr.s_addr = ntohl( addr.s_addr );
            *((struct in_addr*) &newHead->part[64]) = addr;
            return newHead;
         }
#endif
      }
   }

   return parseFQDNandPath( admin, domain, (char *) URL, 0 );
}


/* Parse the URL and reverse it into a linked list suitable to be inserted in a table (used by ufdbGenTable)
 * e.g. my.sex.com becomes "com" -> "sex" -> "my" -> NULL
 *      youtube.com/watch?v=ahuJBU7hji2K  becomes "com" -> "youtube" -> "/watch" -> "v" -> "ahuJBU7hji2K"
 *      foo.net/cgi?a=1&b=2&c=3 becomes "net" -> "foo" -> "/cgi" -> "a" -> 1 -> "b" -> 2 -> "c" -> 3
 *
 * The input parameter, URL, must be a writable array of characters (size = UFDB_MAX_URL_LENGTH).
 */
UFDB_GCC_HOT 
UFDBrevURL * UFDBgenRevURL4table(
   UFDBthreadAdmin * admin,
   unsigned char *   URL )
{
   char *            d;
   char              domain[UFDB_MAX_URL_LENGTH];

   if (URL[0] == '[')
      return parseIPv6URL( admin, URL );

   /* copy the domainname (the dots will be replaced by \0) and find the start of the path */
   d = findDomainEnd( (char *) URL );

   if (d == (char *) URL)				/* domain is empty: illegal URL */
   {
      strcpy( domain, "empty-domain" );
   }
   else
   {
      int dlen = d - (char *) URL;

      strncpy( domain, (char *) URL, dlen );
      domain[dlen] = '\0';
      URL += dlen;

      if (domain[dlen-1] == '.')		/* remove dot after FQDN: also match www.example.com. */
	 domain[dlen-1] = '\0';
   }

   return parseFQDNandPath( admin, domain, (char *) URL, 1 );
}


/* DEBUG: log a revURL 
 */
void UFDBprintRevURL( UFDBrevURL * revURL )
{
   char buffer[UFDB_MAX_URL_LENGTH];

   buffer[0] = '\0';
   while (revURL != NULL)
   {
      strcat( buffer, (char *) revURL->part );
      if (revURL->next != NULL)
	 strcat( buffer, " . " );
      revURL = revURL->next;
   }
   ufdbLogMessage( "P  %s", buffer );
}


UFDB_GCC_HOT
void UFDBfreeRevURL( 
   UFDBthreadAdmin * admin,
   UFDBrevURL *      revURL )
{
   UFDBrevURL *      next;

   if (admin->iusage == MAX_REVURLS)
   {
      while (revURL != NULL)
      {
         next = revURL->next;
         if ((void*)revURL < (void*)admin  ||  (void*)revURL >= ((void*)admin)+sizeof(UFDBthreadAdmin))
         {
            ufdbFree( revURL );
         }
         revURL = next;
      }
   }
   admin->iusage = 0;
}


#if UFDB_DO_DEBUG
static void printMem( unsigned char * mem )
{
   unsigned char * m = tableMemStart;
   int i;

   ufdbLogMessage( "parsing error at %lu 0%07lo  memStart 0x%012lx  mem 0x%012lx", 
                   (unsigned long) (mem - m), (unsigned long) (mem - m),
                   (unsigned long) m, (unsigned long) mem  );
   for (i = 0; i < 16; i++)
   {
      ufdbLogMessage( "   %3d  %03o  0x%02x  %c", *mem, *mem, *mem, *mem );
      mem++;
   }
}
#endif


#define ROUNDUPBY      4
#define ROUNDUP(i)     ( (i) + (ROUNDUPBY - ((i)%ROUNDUPBY) ) )

#define BIGROUNDUPBY   (4 * ROUNDUPBY)
#define BIGROUNDUP(i)  ( (i) + (BIGROUNDUPBY - ((i)%BIGROUNDUPBY) ) )

UFDB_GCC_INLINE 
static struct UFDBtable * _reallocTableArray_1_2( 
   struct UFDBtable *  ptr,
   int                 nElem )
{
   if (nElem <= ROUNDUPBY)
   {
      ptr = (struct UFDBtable *) ufdbRealloc( ptr, (size_t) nElem * sizeof(struct UFDBtable) );
   }
   else if (nElem < 4*BIGROUNDUPBY)
   {
      if (nElem % ROUNDUPBY == 1)
	 ptr = (struct UFDBtable *) ufdbRealloc( ptr, (size_t) ROUNDUP(nElem) * sizeof(struct UFDBtable) );
   }
   else
   {
      if (nElem % BIGROUNDUPBY == 1)
	 ptr = (struct UFDBtable *) ufdbRealloc( ptr, (size_t) BIGROUNDUP(nElem) * sizeof(struct UFDBtable) );
   }

   return ptr;
}


void UFDBfreeTableIndex_1_2( struct UFDBtable * t )
{
   int i;

   if (t == NULL)
      return;

   for (i = 0; i < t->nNextLevels; i++)
   {
      UFDBfreeTableIndex_1_2( &(t->nextLevel[i]) );
   }

   ufdbFree( &(t->nextLevel[0]) );
}


#define HUGEPAGESIZE (2*1024*1024)

#if UFDB_BZ2LIB_SUPPORT
static int ufdbDecompressBZIP2Table( struct UFDBmemTable * memTable )
{
   char * new_mem;
   int    rv;

   if (ufdbGV.debug > 2)
      ufdbLogMessage( "decompressing BZIP2 table from %'ld original size of %d", 
                      memTable->fileSize - memTable->hdrSize, memTable->numEntries );

   // Here we can improve performance by doing an alignment of HUGEPAGESIZE.  
   // If Transparent Hugepages (THP) is configured, Linux uses hugepages.
   // If envars LD_PRELOAD=libhugetlbfs.so HUGETLB_MORECORE=yes are set, malloc
   // will allocate optimally aligned hugepages.
   size_t alignment = 4096;
   size_t tabsize = memTable->hdrSize + memTable->numEntries + 64; // add a safeguard for AVX2
   if (tabsize >= ((size_t) (HUGEPAGESIZE * 0.8)))
   {
      if (tabsize < HUGEPAGESIZE)
         tabsize = HUGEPAGESIZE;                        // round up if ~80% of 1 hugepage is requested
      else if (tabsize >= 3*HUGEPAGESIZE  ||  ufdbGV.madviseHugePages)
         tabsize += HUGEPAGESIZE - (tabsize % HUGEPAGESIZE);    // round up if size >= 3 hugepages
      if (ufdbGV.madviseHugePages)
#if __linux__  &&  HAVE_MADVISE && !UFDB_BARE_METAL_SUPPORT
         memTable->madvisedSize = tabsize;
#else
         ufdbLogMessage( "ufdbGV.madviseHugePages is set but madvise() is not available" );
#endif
   }
   else 
      tabsize += (tabsize + 4096 - 1) % 4096;           // round up to 4K 

   new_mem = (char *) ufdbMallocAligned( alignment, tabsize );
   if (new_mem == NULL)
   {
      ufdbLogFatalError( "ufdbDecompressBZIP2Table: cannot allocate %'ld bytes memory for BZIP2 table decompression", 
                         (long) tabsize );
      return UFDB_API_ERR_NOMEM;
   }

#if __linux__  &&  HAVE_MADVISE && !UFDB_BARE_METAL_SUPPORT
   if (memTable->madvisedSize)
   {
      rv = madvise( new_mem, tabsize, MADV_HUGEPAGE );
      if (rv != 0)
      {
         char   strbuf[128];
         if (strerror_r( errno, strbuf, sizeof(strbuf) )) {;}
         ufdbLogMessage( "WARNING: ufdbDecompressBZIP2Table: madvise( %p %'ld hugepage ) returns error %d: %s",
                         new_mem, tabsize, errno, strbuf );
      }
   }
#endif

   /* copy the table header */
   memcpy( new_mem, memTable->mem, memTable->hdrSize );

   rv = BZ2_bzBuffToBuffDecompress( new_mem + memTable->hdrSize,
                                    (unsigned int *) &(memTable->numEntries),
			            (char *) memTable->mem + memTable->hdrSize,
			            memTable->fileSize - memTable->hdrSize,
			            0,
			            0 );
   if (rv != BZ_OK)
   {
      ufdbLogFatalError( "ufdbDecompressBZIP2Table: BZIP2 decompression failed with code %d", rv );
      return UFDB_API_ERR_NOMEM;
   }

   if (memTable->memStatus == UFDB_MEMSTATUS_MALLOC)
      ufdbFree( memTable->mem );
   memTable->mem = new_mem;
   memTable->memStatus = UFDB_MEMSTATUS_MALLOC;

   return UFDB_API_OK;
}
#endif


static int ufdbDecompressZLIBTable( struct UFDBmemTable * memTable )
{
   char * new_mem;
   int    rv;
   z_stream  zs;

#if UFDB_BARE_METAL_SUPPORT  &&  __OCTEON__       &&  0
   if (!octeon_has_feature(OCTEON_FEATURE_ZIP))
   {
      ufdbLogFatalError( "ufdbDecompressZLIBTable: Octeon does not have ZIP feature: cannot decompress table" );
      return UFDB_API_ERR_FATAL;
   }
#endif

   if (ufdbGV.debug > 2)
      ufdbLogMessage( "decompressing ZLIB table from %'ld original size of %d",
                      memTable->fileSize - memTable->hdrSize, memTable->numEntries );

   // Here we can improve performance by doing an alignment of HUGEPAGESIZE.  
   // If Transparent Hugepages (THP) is configured, Linux uses hugepages.
   // If envars LD_PRELOAD=libhugetlbfs.so HUGETLB_MORECORE=yes are set, malloc
   // will allocate optimally aligned hugepages.
   // madvise() hints the kernel to use hugepages.
   size_t alignment = 4096;
   size_t tabsize = memTable->hdrSize + memTable->numEntries + 64;   // add a safeguard for AVX2
   if (tabsize >= ((size_t) (1600*1024)))
   {
      alignment = HUGEPAGESIZE;
      if (tabsize < HUGEPAGESIZE)
         tabsize = HUGEPAGESIZE;                        // round up if ~80% of 1 hugepage is requested
      else if (tabsize >= 3*HUGEPAGESIZE  ||  ufdbGV.madviseHugePages)
         tabsize += HUGEPAGESIZE - (tabsize % HUGEPAGESIZE);    // round up if size >= 3 hugepages
      if (ufdbGV.madviseHugePages)
#if __linux__  &&  HAVE_MADVISE && !UFDB_BARE_METAL_SUPPORT
         memTable->madvisedSize = tabsize;
#else
         ufdbLogMessage( "ufdbGV.madviseHugePages is set but madvise() is not available" );
#endif
   }
   else 
      tabsize += (tabsize + 4096 - 1) % 4096;           // round up to 4K 

   new_mem = (char *) ufdbMallocAligned( alignment, tabsize );
   if (new_mem == NULL)
   {
      ufdbLogFatalError( "ufdbDecompressZLIBTable: cannot allocate %'ld bytes memory for ZLIB table decompression", 
                         (long) tabsize );
      return UFDB_API_ERR_NOMEM;
   }

#if __linux__  &&  HAVE_MADVISE && !UFDB_BARE_METAL_SUPPORT
   if (memTable->madvisedSize)
   {
      rv = madvise( new_mem, tabsize, MADV_HUGEPAGE );
      if (rv != 0)
      {
         char   strbuf[128];
         if (strerror_r( errno, strbuf, sizeof(strbuf) )) {;}
         ufdbLogMessage( "WARNING: ufdbDecompressZLIBTable: madvise( %p %'ld hugepage ) returns error %d: %s",
                         new_mem, tabsize, errno, strbuf );
      }
   }
#endif

   zs.zalloc = ufdbZlibMalloc;
   zs.zfree = ufdbZlibFree;
   zs.opaque = Z_NULL;
   zs.zalloc = Z_NULL;
   zs.zfree  = Z_NULL;
   zs.next_in = Z_NULL;
   zs.avail_in = 0;
   rv = inflateInit( &zs );
   if (Z_OK != rv)
   {
      ufdbLogFatalError( "ufdbDecompressZLIBTable: ZLIB decompression initialisation failed: error %d  %s",
                         rv, zs.msg == NULL ? "" : zs.msg );
      return UFDB_API_ERR_NOMEM;
   }

   /* copy the table header */
   memcpy( new_mem, memTable->mem, memTable->hdrSize );

   zs.next_in = (unsigned char *) memTable->mem + memTable->hdrSize;
   zs.avail_in = memTable->fileSize - memTable->hdrSize;
   zs.next_out = (unsigned char *) new_mem + memTable->hdrSize;
   zs.avail_out = memTable->numEntries + 0;
   rv = inflate( &zs, Z_FINISH );
   if (rv != Z_STREAM_END)
   {
      ufdbLogFatalError( "ufdbDecompressZLIBTable: ZLIB decompression failed: error %d  %s",
                         rv, zs.msg == NULL ? "" : zs.msg );
      return UFDB_API_ERR_RANGE;
   }

   inflateEnd( &zs );

   if (memTable->memStatus == UFDB_MEMSTATUS_MALLOC)
      ufdbFree( memTable->mem );
   memTable->mem = new_mem;
   memTable->memStatus = UFDB_MEMSTATUS_MALLOC;

   return UFDB_API_OK;
}


/* Parse a binary table header from a memory table.
 */
int UFDBparseTableHeader( struct UFDBmemTable * mt )
{
   int  retval = UFDB_API_OK;
   int  rv;
   int  n;
   int  cksum;
   char prefix[32];
   char tableName[32];
   char key[32];
   char date[32];
   char flags[8+1];
   ufdbCrypt uc;
   unsigned char * mem;

   strcpy( date, "nodate" );
   strcpy( flags, "--------" );
   mt->version[0] = '\0';
   mt->numEntries = 0;
   mt->indexSize = 0;
   cksum = -1;
   n = sscanf( (char *) mt->mem, "%5s %7s %20s %11d key=%30s date=%20s %8s %d %d",
               prefix, &(mt->version[0]), tableName, &(mt->numEntries), 
	       key, date, flags, &(mt->indexSize), &cksum );

   memcpy( mt->flags, flags, 8 );
   if (mt->version[0] >= '3' || flags[3] == 'p')
      mt->hdrSize = sizeof(struct UFDBfileHeader21);      // 256
   else
      mt->hdrSize = sizeof(struct UFDBfileHeader);        // 99

#if UFDB_DO_DEBUG
   if (ufdbGV.debug > 1)
      ufdbLogMessage( "      UFDBparseTableHeader: n=%d prefix=%-5.5s version=%s name=%s num=%d key=%s "
                      "date=%s flags=%s indexsize=%d hdrsize=%d cksum=%05d", 
                      n, prefix, mt->version, tableName, mt->numEntries, key, 
		      date, flags, mt->indexSize, mt->hdrSize, cksum );
#endif

   if (n < 5  ||  strcmp(prefix,"UFDB") != 0)
   {
#if !UFDB_BARE_METAL_SUPPORT
      syslog( LOG_ALERT, "URL table %s has an invalid UFDB header", tableName );
#endif
      ufdbLogFatalError( "URL table %s has an invalid UFDB header", tableName );
      ufdbLogFatalError( "contact support@urlfilterdb.com" );
      return UFDB_API_ERR_INVALID_TABLE;
   }

   if (mt->indexSize < 0)
   {
#if !UFDB_BARE_METAL_SUPPORT
      syslog( LOG_ALERT, "URL table %s has an erroneous UFDB header: indexsize < 0", tableName );
#endif
      ufdbLogFatalError( "URL table %s has an erroneous UFDB header: indexsize < 0", tableName );
      ufdbLogFatalError( "contact support@urlfilterdb.com" );
      return UFDB_API_ERR_INVALID_TABLE;
   }

   if (strcmp( mt->version, UFDBdbVersion ) > 0)
   {
#if !UFDB_BARE_METAL_SUPPORT
      syslog( LOG_ALERT, "URL table %s has an unsupported data format (%s)", tableName, mt->version );
#endif
      ufdbLogFatalError( "UFDB file for table \"%s\" has data format version \"%s\" while "
      		         "this program\n"
                         "does not support versions higher than \"%s\"", 
		         tableName, mt->version, UFDBdbVersion );
      ufdbLogFatalError( "Download/install a new version of this program." );
      ufdbLogFatalError( "Go to http://www.urlfilterdb.com" );
      return UFDB_API_ERR_INVALID_TABLE;
   }

   /* starting with version 2.0 we need the indexSize */
   if (strcmp( "2.0", mt->version ) >= 0)
   {
      if (n < 7)
      {
#if !UFDB_BARE_METAL_SUPPORT
	 syslog( LOG_ALERT, "URL table %s has an invalid UFDB2+ header", tableName );
#endif
	 ufdbLogFatalError( "URL table %s has an invalid UFDB2+ header", tableName );
	 ufdbLogFatalError( "contact support@urlfilterdb.com" );
	 return UFDB_API_ERR_INVALID_TABLE;
      }
      if (strcmp( "2.1", mt->version ) >= 0)
      {
         /* starting with version 2.1 encrypted tables have
	  * an additional 64 random/dummy bytes at the start of the table.
	  */
         { ; }
      }
   }

   if (strlen( key ) < 19)
   {
      ufdbLogFatalError( "UFDB file for table \"%s\" has an invalid key\n"
                         "contact support@urlfilterdb.com",
                         tableName );
      return UFDB_API_ERR_INVALID_KEY;
   }

   mt->key[0] = key[0];
   mt->key[1] = key[1];
   mt->key[2] = key[2];
   mt->key[3] = key[3];
   /* skip '-' */
   mt->key[4] = key[5];
   mt->key[5] = key[6];
   mt->key[6] = key[7];
   mt->key[7] = key[8];
   /* skip '-' */
   mt->key[8] = key[10];
   mt->key[9] = key[11];
   mt->key[10] = key[12];
   mt->key[11] = key[13];
   /* skip '-' */
   mt->key[12] = key[15];
   mt->key[13] = key[16];
   mt->key[14] = key[17];
   mt->key[15] = key[18];

   strcpy( mt->date, date );

   mt->age = 0;
   if (strcmp( date, "nodate" ) != 0)
   {
      struct tm   tm;

      if (5 != sscanf( date, "%4d%2d%2d.%2d%2d", 
                       &tm.tm_year, &tm.tm_mon, &tm.tm_mday, &tm.tm_hour, &tm.tm_min ))
      {
	 mt->age = -1;
#if !UFDB_BARE_METAL_SUPPORT
	 syslog( LOG_ALERT, "URL table %s has an invalid date", tableName );
#endif
         ufdbLogFatalError( "URL table %s has an invalid date (date=%13.13s)", tableName, date );
         ufdbLogFatalError( "contact support@urlfilterdb.com" );
	 return UFDB_API_ERR_INVALID_TABLE;
      }
      else
      {
	 time_t      t_now;
	 time_t      t_db;
	 time_t      diff;

	 tm.tm_year -= 1900;
	 tm.tm_mon  -= 1;
	 tm.tm_isdst = 0;
	 tm.tm_sec   = 0;

	 t_db = mktime( &tm );
	 t_now = UFDBtime();
	 diff = t_now - t_db;
	 mt->age = diff / (24 * 60 * 60);
	 if (t_now < 36 * 60 * 60)
	 {
	    if (UFDBhasRTC())
	    {
	       ufdbLogFatalError( "The clock of this system is not set so the age of the database table "
	                          "%s cannot be verified", tableName );
	       ufdbLogFatalError( "Set the clock to the current date and load the database tables again" );
	       retval = UFDB_API_ERR_INVALID_TABLE;
	    }
	    else
	    {
	       ufdbLogMessage( "This system has no clock so cannot verify age of database table %s",
	                       tableName );
	       mt->age = 0;
	    }
	 }
	 else if (mt->age < -1)	/* allow 1 day back for various time zones */
	 {
	    ufdbLogFatalError( "table %s has an invalid date (%13.13s) or the clock is not correct",
	                       tableName, date );
	    ufdbLogFatalError( "the difference between current system time and database timestamp is "
	                       "%'ld seconds (%d days)", (long) diff, mt->age );
            ufdbLogFatalError( "contact support@urlfilterdb.com" );
	    retval = UFDB_API_ERR_INVALID_TABLE;
	 }
	 else if (mt->age > UFDB_MAX_TABLE_AGE  &&  flags[1] == 'P')
	 {
	    ufdbLogFatalError( "table %s is dated %13.13s and has EXPIRED.  *******************\n"
	                       "Check the clock, licenses and cron job for the update command.  "
                               "Run the update command with -v option to verify the URL database download.",
			       tableName, date );
	    retval = UFDB_API_STATUS_DATABASE_EXPIRED;
	    ufdbGV.databaseStatus = UFDB_API_STATUS_DATABASE_EXPIRED;
	 }
	 else if (mt->age > UFDB_WARN_TABLE_AGE  &&  flags[1] == 'P')
	 {
	    /* TODO: change state to DB_WARNING */
	    ufdbLogMessage( "WARNING: table %s is dated %13.13s and is %d days old "
	                    "(maximum allowed age is %d days).  *****\n"
			    "Check the clock, licenses and cron job for the update command.  "
                            "Run the update command with -v option to verify the URL database download.  *****",
			    tableName, date, mt->age, UFDB_MAX_TABLE_AGE );
	    retval = UFDB_API_STATUS_DATABASE_OLD;
	    if (ufdbGV.databaseStatus == UFDB_API_STATUS_DATABASE_OK)
	       ufdbGV.databaseStatus = UFDB_API_STATUS_DATABASE_OLD;
	 }
#if 0
	 ufdbLogMessage( "t_db  %12ld", t_db );
	 ufdbLogMessage( "t_now %12ld", t_now );
	 ufdbLogMessage( "diff  %12ld", t_now - t_db );
#endif
      }
   }

   if (retval != UFDB_API_OK  &&  retval != UFDB_API_STATUS_DATABASE_OLD)
   {
      mt->table.tag = (unsigned char *) "NEVER";
      mt->mem = NULL;
      mt->memStatus = UFDB_MEMSTATUS_UNKNOWN;
      mt->madvisedSize = 0;
      mt->hdrSize = 0;
      mt->fileSize = 0;
      mt->table.nNextLevels = 0;
      mt->table.nextLevel = NULL;
   }
   else
   {
      if (ufdbGV.debug > 1)
         ufdbLogMessage( "loading URL category %s with creation date %s", tableName, date );

#if 0
      if (cksum > 0 && strcmp( "2.1", mt->version ) >= 0)
      {
         int   mycksum;

	 mycksum = UFDBcalcCksum( (char *) mt->mem + mt->hdrSize,
                                  mt->fileSize - mt->hdrSize );
	 if (mycksum == 0)
	    mycksum = 1;

	 if (cksum >= 0  &&  cksum != mycksum)
	 {
	    mt->table.tag = (unsigned char *) "NEVER";
	    mt->mem = NULL;
            mt->memStatus = UFDB_MEMSTATUS_UNKNOWN;
            mt->madvisedSize = 0;
	    mt->fileSize = 0;
	    mt->hdrSize = 0;
	    mt->table.nNextLevels = 0;
	    mt->table.nextLevel = NULL;
	    ufdbLogError( "URL category %s has a file cksum of %05d but a different memory cksum of %05d",
                          tableName, cksum, mycksum );
	    return UFDB_API_ERR_CKSUM_NOT_VALID;
	 }
      }
#endif

      if (flags[2] == 'Q')			/* encrypted table */
      {
	 mem = (unsigned char *) mt->mem + mt->hdrSize;
	 ufdbCryptInit( &uc, (unsigned char *) mt->key, 16, mt->version );
	 ufdbEncryptText( &uc, mem, mem, mt->fileSize - mt->hdrSize );
	 if (ufdbGV.debug > 2)
         {
	    ufdbLogMessage( "table %s decrypted", tableName );
         }
      }
      // add AES encryption

      if (mt->flags[0] == 'Z')		/* ZLIB compressed table */
      {
	 rv = ufdbDecompressZLIBTable( mt );
         if (rv != UFDB_API_OK)
            retval = rv;
	 if (ufdbGV.debug > 2)
         {
            if (retval == UFDB_API_OK)
               ufdbLogMessage( "table %s uncompressed", tableName );
            else
               ufdbLogMessage( "table %s uncompression failed with error %d", tableName, retval );
         }
      }
      else if (mt->flags[0] == 'C')	/* BZIP2 compressed table */
      {
#if UFDB_BZ2LIB_SUPPORT
	 rv = ufdbDecompressBZIP2Table( mt );
#else
         ufdbLogFatalError( "table %s has unsupported BZIP2 compression", tableName );
	 return UFDB_API_ERR_INVALID_TABLE;
#endif
         if (rv != UFDB_API_OK)
            retval = rv;
	 if (ufdbGV.debug > 2)
         {
            if (retval == UFDB_API_OK)
               ufdbLogMessage( "table %s uncompressed", tableName );
            else
               ufdbLogError( "table %s uncompression failed with error %d", tableName, retval );
         }
      }
      else
      {
         if (mt->memStatus == UFDB_MEMSTATUS_DP)
         {
            if (ufdbGV.debug)
               ufdbLogMessage( "copying table (%'d bytes) to not use memory of DPufdbLoadFile",
                               mt->numEntries );
            char * newMem = ufdbMallocAligned( UFDB_CACHELINE_SIZE, mt->numEntries );
            memcpy( newMem, mt->mem, mt->numEntries );
            mt->mem = newMem;
            mt->memStatus = UFDB_MEMSTATUS_MALLOC;
         }
      }
   }

   return retval;
}


#if UFDB_OVERRIDE_GCC_OPT  && ((__GNUC__ > 4)  ||  (__GNUC__ == 4  &&  __GNUC_MINOR__ >= 4))
#pragma GCC push_options
#pragma GCC optimize ("O3")
#pragma GCC optimize ("align-functions=64")
#pragma GCC optimize ("no-stack-protector")
#endif


#include "strcmpurlpart.static.c"


#if !UFDB_BARE_METAL_SUPPORT
UFDB_GCC_INLINE
static unsigned char * parseNextTag_1_2( 
   struct UFDBtable * parent, 
   unsigned char *    mem )
{
   unsigned char *    tag;
   unsigned char      tagType;
   int                n;

   while (mem != NULL)
   {
      tag = mem;
      while (*mem >= ' ')
	 mem++;

      tagType = *mem;
      *mem++ = '\0';

      switch (tagType)
      {
      case UFDBsubLevel:
	    DEBUG(( "   parse  tag = %-10s  sub-level", tag )); /* */

	    n = parent->nNextLevels;
	    parent->nNextLevels++;
	    parent->nextLevel = _reallocTableArray_1_2( parent->nextLevel, parent->nNextLevels );
	    parent->nextLevel[n].tag = tag;
	    parent->nextLevel[n].nNextLevels = 0;
	    parent->nextLevel[n].nextLevel = NULL;

	    mem = parseNextTag_1_2( &(parent->nextLevel[n]), mem );
	    break;

      case UFDBsameLevel:
	    DEBUG(( "   parse  tag = %-10s  same-level", tag )); /* */

	    n = parent->nNextLevels;
	    parent->nNextLevels++;
	    parent->nextLevel = _reallocTableArray_1_2( parent->nextLevel, parent->nNextLevels );
	    parent->nextLevel[n].tag = tag;
	    parent->nextLevel[n].nNextLevels = 0;
	    parent->nextLevel[n].nextLevel = NULL;

	    break;

      case UFDBprevLevel:
	    if (tag[0] >= ' ')   /* is the code followed by a tag or another code ? */
	    {
	       DEBUG(( "   parse  tag = %-10s  prev-level", tag )); /* */

	       n = parent->nNextLevels;
	       parent->nNextLevels++;
	       parent->nextLevel = _reallocTableArray_1_2( parent->nextLevel, parent->nNextLevels );
	       parent->nextLevel[n].tag = tag;
	       parent->nextLevel[n].nNextLevels = 0;
	       parent->nextLevel[n].nextLevel = NULL;
	    }
	    else 
	    {
	       DEBUG(( "   parse  tag = %-10s  prev-level", "*" )); /* */
	       ;
	    }
	    return mem;
	    break;

      case UFDBendTable:
	    DEBUG(( "   parse  tag = %-10s  end-of-table", tag[0] >= ' ' ? (char*) tag : "NONE" )); /* */
	    if (tag[0] >= ' ')   /* is the code followed by a tag or another code ? */
	    {
	       DEBUG(( "   parse  tag = %-10s  end-of-table", tag )); /* */

	       n = parent->nNextLevels;
	       parent->nNextLevels++;
	       parent->nextLevel = _reallocTableArray_1_2( parent->nextLevel, parent->nNextLevels );
	       parent->nextLevel[n].tag = tag;
	       parent->nextLevel[n].nNextLevels = 0;
	       parent->nextLevel[n].nextLevel = NULL;
	    }
	    return NULL;
	    break;

      default:
            DEBUG(( "tagType = %d !", tagType ));
	    ;
      }
   }

   return NULL;
}
#endif


UFDB_GCC_INLINE
static struct UFDBtable * _allocTableIndex( int num )
{
   struct UFDBtable * t;
   
   DEBUGP(( "      _allocTableIndex( %3d )   ti = %d", num, tableIndex_i ));

   t = &tableIndex[tableIndex_i];
   tableIndex_i += num;

   return t;
}


UFDB_GCC_COLD
static unsigned char * parseNextTag_2_0(
   struct UFDBtable * parent, 
   unsigned char *    mem )
{
   unsigned char *    tag;
   unsigned char      tagType;
   int                n;
   unsigned int       num_children;

   while (mem != NULL)
   {
      tag = mem;
      while (*mem >= ' ')		/* find the end of the tag */
	 mem++;

      tagType = *mem;			/* the tagtype is after the tag and overwritten with a \0 */
      *mem++ = '\0';

      switch (tagType)
      {
      case UFDBsubLevel:
	    num_children = *mem++;
	    if (num_children == 0)
	    {
	       num_children = *mem  +  (*(mem+1) * 256)  +  (*(mem+2) * 65536);
	       mem += 3;
	    }

	    DEBUGP(( "   parse2_0 tag = %-18s  sub-level   %d child(ren)", 
	            tag, num_children ));   /* */

	    n = parent->nNextLevels;
	    parent->nNextLevels++;
	    parent->nextLevel[n].tag = tag;
	    parent->nextLevel[n].nNextLevels = 0;
	    parent->nextLevel[n].nextLevel = _allocTableIndex( num_children );

	    mem = parseNextTag_2_0( &(parent->nextLevel[n]), mem );
	    break;

      case UFDBsameLevel:
	    DEBUGP(( "   parse2_0 tag = %-18s  same-level", tag )); /* */

	    n = parent->nNextLevels;
	    parent->nNextLevels++;
	    parent->nextLevel[n].tag = tag;
	    parent->nextLevel[n].nNextLevels = 0;
	    parent->nextLevel[n].nextLevel = NULL;

	    break;

      case UFDBprevLevel:
	    if (tag[0] >= ' ')   /* is the code followed by a tag or another code ? */
	    {
	       DEBUGP(( "   parse2_0 tag = %-18s  prev-level", tag )); /* */

	       n = parent->nNextLevels;
	       parent->nNextLevels++;
	       parent->nextLevel[n].tag = tag;
	       parent->nextLevel[n].nNextLevels = 0;
	       parent->nextLevel[n].nextLevel = NULL;
	    }
	    else 
	    {
	       DEBUGP(( "   parse2_0 tag = %-18s  prev-level", "*" )); /* */
	       ;
	    }
	    return mem;
	    break;

      case UFDBendTable:
	    if (tag[0] >= ' ')   /* is the code followed by a tag or another code ? */
	    {
	       DEBUGP(( "   parse2_0 tag = %-18s  end-of-table", tag )); /* */
	       n = parent->nNextLevels;
	       parent->nNextLevels++;
	       parent->nextLevel[n].tag = tag;
	       parent->nextLevel[n].nNextLevels = 0;
	       parent->nextLevel[n].nextLevel = NULL;
	    }
	    else
	    {
	       DEBUGP(( "   parse2_0 tag = %-18s  end-of-table", "NO-TAG-HERE" )); /* */
	       ;
	    }
	    return NULL;
	    break;
      }
   }
   DEBUGP(( "parse2_0 end of parsing" ));

   return NULL;
}


UFDB_GCC_COLD
static unsigned char * parseNextTag_2_1(
   struct UFDBtable * parent, 
   unsigned char *    mem )
{
   unsigned char *    tag;
   unsigned char      tagType;
   int                n;
   unsigned int       num_children;

   while (mem != NULL)
   {
      tag = mem;
      while (*mem >= ' ')		/* find the end of the optional tag */
	 mem++;

      tagType = *mem;			/* the tagtype is after the tag and overwritten with a \0 */
      *mem++ = '\0';

      switch (tagType)
      {
      case UFDBsubLevel:
	    num_children = *mem++;	/* between 8 and 255 */

	    DEBUGP(( "   parse2_1 tag = %-18s  sub-level   %d child(ren)", tag, num_children ));

	    n = parent->nNextLevels;
	    parent->nNextLevels++;
	    parent->nextLevel[n].tag = tag;
	    parent->nextLevel[n].nNextLevels = 0;
	    parent->nextLevel[n].nextLevel = _allocTableIndex( num_children );

	    mem = parseNextTag_2_1( &(parent->nextLevel[n]), mem );
	    break;

      case UFDBsubLevel1:
            num_children = 1;
	    DEBUGP(( "   parse2_1 tag = %-18s  sub-level1  1 child", tag ));

	    n = parent->nNextLevels;
	    parent->nNextLevels++;
	    parent->nextLevel[n].tag = tag;
	    parent->nextLevel[n].nNextLevels = 0;
	    parent->nextLevel[n].nextLevel = _allocTableIndex( num_children );

	    mem = parseNextTag_2_1( &(parent->nextLevel[n]), mem );
	    break;

      case UFDBsubLevel2:
            num_children = 2;
	    DEBUGP(( "   parse2_1 tag = %-18s  sub-level2  2 children", tag ));

	    n = parent->nNextLevels;
	    parent->nNextLevels++;
	    parent->nextLevel[n].tag = tag;
	    parent->nextLevel[n].nNextLevels = 0;
	    parent->nextLevel[n].nextLevel = _allocTableIndex( num_children );

	    mem = parseNextTag_2_1( &(parent->nextLevel[n]), mem );
	    break;

      case UFDBsubLevel3:
            num_children = 3;
	    DEBUGP(( "   parse2_1 tag = %-18s  sub-level3  3 children", tag ));

	    n = parent->nNextLevels;
	    parent->nNextLevels++;
	    parent->nextLevel[n].tag = tag;
	    parent->nextLevel[n].nNextLevels = 0;
	    parent->nextLevel[n].nextLevel = _allocTableIndex( num_children );

	    mem = parseNextTag_2_1( &(parent->nextLevel[n]), mem );
	    break;

      case UFDBsubLevel4:
            num_children = 4;
	    DEBUGP(( "   parse2_1 tag = %-18s  sub-level4  4 children", tag ));

	    n = parent->nNextLevels;
	    parent->nNextLevels++;
	    parent->nextLevel[n].tag = tag;
	    parent->nextLevel[n].nNextLevels = 0;
	    parent->nextLevel[n].nextLevel = _allocTableIndex( num_children );

	    mem = parseNextTag_2_1( &(parent->nextLevel[n]), mem );
	    break;

      case UFDBsubLevel5:
            num_children = 5;
	    DEBUGP(( "   parse2_1 tag = %-18s  sub-level5  5 children", tag ));

	    n = parent->nNextLevels;
	    parent->nNextLevels++;
	    parent->nextLevel[n].tag = tag;
	    parent->nextLevel[n].nNextLevels = 0;
	    parent->nextLevel[n].nextLevel = _allocTableIndex( num_children );

	    mem = parseNextTag_2_1( &(parent->nextLevel[n]), mem );
	    break;

      case UFDBsubLevel6:
            num_children = 6;
	    DEBUGP(( "   parse2_1 tag = %-18s  sub-level6  6 children", tag ));

	    n = parent->nNextLevels;
	    parent->nNextLevels++;
	    parent->nextLevel[n].tag = tag;
	    parent->nextLevel[n].nNextLevels = 0;
	    parent->nextLevel[n].nextLevel = _allocTableIndex( num_children );

	    mem = parseNextTag_2_1( &(parent->nextLevel[n]), mem );
	    break;

      case UFDBsubLevel7:
            num_children = 7;
	    DEBUGP(( "   parse2_1 tag = %-18s  sub-level7  7 children", tag ));

	    n = parent->nNextLevels;
	    parent->nNextLevels++;
	    parent->nextLevel[n].tag = tag;
	    parent->nextLevel[n].nNextLevels = 0;
	    parent->nextLevel[n].nextLevel = _allocTableIndex( num_children );

	    mem = parseNextTag_2_1( &(parent->nextLevel[n]), mem );
	    break;

      case UFDBsubLevelNNN:
	    num_children = *mem  +  (*(mem+1) * 256);           /* between 256 and 65535 children */
	    mem += 2;

	    DEBUGP(( "   parse2_1 tag = %-18s  sub-level-NNN %d children", tag, num_children ));

	    n = parent->nNextLevels;
	    parent->nNextLevels++;
	    parent->nextLevel[n].tag = tag;
	    parent->nextLevel[n].nNextLevels = 0;
	    parent->nextLevel[n].nextLevel = _allocTableIndex( num_children );

	    mem = parseNextTag_2_1( &(parent->nextLevel[n]), mem );
	    break;

      case UFDBsubLevelNNNNN:
	    							/* between 65536 and 4 billion children */
	    num_children = *mem  +  (*(mem+1) * 256)  +  (*(mem+2) * 256*256)  +  (*(mem+3) * 256*256*256);	
	    mem += 4;

	    DEBUGP(( "   parse2_1 tag = %-18s  sub-level-NNNNN %d children", tag, num_children ));

	    n = parent->nNextLevels;
	    parent->nNextLevels++;
	    parent->nextLevel[n].tag = tag;
	    parent->nextLevel[n].nNextLevels = 0;
	    parent->nextLevel[n].nextLevel = _allocTableIndex( num_children );

	    mem = parseNextTag_2_1( &(parent->nextLevel[n]), mem );
	    break;

      case UFDBsameLevel:
	    DEBUGP(( "   parse2_1 tag = %-18s  same-level", tag )); /* */

	    n = parent->nNextLevels;
	    parent->nNextLevels++;
	    parent->nextLevel[n].tag = tag;
	    parent->nextLevel[n].nNextLevels = 0;
	    parent->nextLevel[n].nextLevel = NULL;

	    break;

      case UFDBprevLevel:
	    if (tag[0] >= ' ')   /* is the code followed by a tag or another code ? */
	    {
	       DEBUGP(( "   parse2_1 tag = %-18s  prev-level", tag )); /* */

	       n = parent->nNextLevels;
	       parent->nNextLevels++;
	       parent->nextLevel[n].tag = tag;
	       parent->nextLevel[n].nNextLevels = 0;
	       parent->nextLevel[n].nextLevel = NULL;
	    }
	    else 
	    {
	       DEBUGP(( "   parse2_1 tag = %-18s  prev-level", "*" )); /* */
	       ;
	    }
	    return mem;
	    break;

      case UFDBendTable:
	    if (tag[0] >= ' ')   /* is the code followed by a tag or another code ? */
	    {
	       DEBUGP(( "   parse2_1 tag = %-18s  end-of-table", tag )); /* */
	       n = parent->nNextLevels;
	       parent->nNextLevels++;
	       parent->nextLevel[n].tag = tag;
	       parent->nextLevel[n].nNextLevels = 0;
	       parent->nextLevel[n].nextLevel = NULL;
	    }
	    else
	    {
	       DEBUGP(( "   parse2_1 tag = %-18s  end-of-table", "END" )); /* */
	       ;
	    }
	    return NULL;
	    break;

      default:
            ufdbLogFatalError( "cannot parse 2.1 table: tag type is %03o", tagType );
	    return NULL;
	    break;
      }
   }

   return NULL;
}


static unsigned char * parseNextTag_2_2(
   struct UFDBtable * parent, 
   unsigned char *    mem )
{
   unsigned char *    tag;
   unsigned char      tagType;
   int                n;
   unsigned int       num_children;

   while (mem != NULL)
   {
      tag = mem;
      if (*tag == '\0')			/* do we have an empty tag ? */
      {
         ++mem;
      }
      else if (*mem >= ' ')		/* do we have a tag ? */
      {
	 while (*mem != '\0')		/* find the end of the tag */
	    ++mem;
	 ++mem;				/* skip the \0 delimiter of the tag */
      }

      tagType = *mem++;

      switch (tagType)
      {
      case UFDBsubLevel:
	    num_children = *mem++;	/* between 8 and 255 */

	    DEBUGP(( "   parse2_2 tag = %-18s  sub-level   %d child(ren)", tag, num_children ));

	    n = parent->nNextLevels;
	    parent->nNextLevels++;
	    parent->nextLevel[n].tag = tag;
	    parent->nextLevel[n].nNextLevels = 0;
	    parent->nextLevel[n].nextLevel = _allocTableIndex( num_children );

	    mem = parseNextTag_2_2( &(parent->nextLevel[n]), mem );
	    break;

      case UFDBsubLevel1:
            num_children = 1;
	    DEBUGP(( "   parse2_2 tag = %-18s  sub-level1  1 child", tag ));

	    n = parent->nNextLevels;
	    parent->nNextLevels++;
	    parent->nextLevel[n].tag = tag;
	    parent->nextLevel[n].nNextLevels = 0;
	    parent->nextLevel[n].nextLevel = _allocTableIndex( num_children );

	    mem = parseNextTag_2_2( &(parent->nextLevel[n]), mem );
	    break;

      case UFDBsubLevel2:
            num_children = 2;
	    DEBUGP(( "   parse2_2 tag = %-18s  sub-level2  2 children", tag ));

	    n = parent->nNextLevels;
	    parent->nNextLevels++;
	    parent->nextLevel[n].tag = tag;
	    parent->nextLevel[n].nNextLevels = 0;
	    parent->nextLevel[n].nextLevel = _allocTableIndex( num_children );

	    mem = parseNextTag_2_2( &(parent->nextLevel[n]), mem );
	    break;

      case UFDBsubLevel3:
            num_children = 3;
	    DEBUGP(( "   parse2_2 tag = %-18s  sub-level3  3 children", tag ));

	    n = parent->nNextLevels;
	    parent->nNextLevels++;
	    parent->nextLevel[n].tag = tag;
	    parent->nextLevel[n].nNextLevels = 0;
	    parent->nextLevel[n].nextLevel = _allocTableIndex( num_children );

	    mem = parseNextTag_2_2( &(parent->nextLevel[n]), mem );
	    break;

      case UFDBsubLevel4:
            num_children = 4;
	    DEBUGP(( "   parse2_2 tag = %-18s  sub-level4  4 children", tag ));

	    n = parent->nNextLevels;
	    parent->nNextLevels++;
	    parent->nextLevel[n].tag = tag;
	    parent->nextLevel[n].nNextLevels = 0;
	    parent->nextLevel[n].nextLevel = _allocTableIndex( num_children );

	    mem = parseNextTag_2_2( &(parent->nextLevel[n]), mem );
	    break;

      case UFDBsubLevel5:
            num_children = 5;
	    DEBUGP(( "   parse2_2 tag = %-18s  sub-level5  5 children", tag ));

	    n = parent->nNextLevels;
	    parent->nNextLevels++;
	    parent->nextLevel[n].tag = tag;
	    parent->nextLevel[n].nNextLevels = 0;
	    parent->nextLevel[n].nextLevel = _allocTableIndex( num_children );

	    mem = parseNextTag_2_2( &(parent->nextLevel[n]), mem );
	    break;

      case UFDBsubLevel6:
            num_children = 6;
	    DEBUGP(( "   parse2_2 tag = %-18s  sub-level6  6 children", tag ));

	    n = parent->nNextLevels;
	    parent->nNextLevels++;
	    parent->nextLevel[n].tag = tag;
	    parent->nextLevel[n].nNextLevels = 0;
	    parent->nextLevel[n].nextLevel = _allocTableIndex( num_children );

	    mem = parseNextTag_2_2( &(parent->nextLevel[n]), mem );
	    break;

      case UFDBsubLevel7:
            num_children = 7;
	    DEBUGP(( "   parse2_2 tag = %-18s  sub-level7  7 children", tag ));

	    n = parent->nNextLevels;
	    parent->nNextLevels++;
	    parent->nextLevel[n].tag = tag;
	    parent->nextLevel[n].nNextLevels = 0;
	    parent->nextLevel[n].nextLevel = _allocTableIndex( num_children );

	    mem = parseNextTag_2_2( &(parent->nextLevel[n]), mem );
	    break;

      case UFDBsubLevelNNN:
	    num_children = *mem  +  (*(mem+1) * 256);		/* between 256 and 65535 children */
	    mem += 2;

	    DEBUGP(( "   parse2_2 tag = %-18s  sub-level-NNN %d children", tag, num_children ));

	    n = parent->nNextLevels;
	    parent->nNextLevels++;
	    parent->nextLevel[n].tag = tag;
	    parent->nextLevel[n].nNextLevels = 0;
	    parent->nextLevel[n].nextLevel = _allocTableIndex( num_children );

	    mem = parseNextTag_2_2( &(parent->nextLevel[n]), mem );
	    break;

      case UFDBsubLevelNNNNN:
	    							/* between 65536 and 4 billion children */
	    num_children = *mem  +  (*(mem+1) * 256)  +  (*(mem+2) * 256*256)  +  (*(mem+3) * 256*256*256);	
	    mem += 4;

	    DEBUGP(( "   parse2_2 tag = %-18s  sub-level-NNNNN %d children", tag, num_children ));

	    n = parent->nNextLevels;
	    parent->nNextLevels++;
	    parent->nextLevel[n].tag = tag;
	    parent->nextLevel[n].nNextLevels = 0;
	    parent->nextLevel[n].nextLevel = _allocTableIndex( num_children );

	    mem = parseNextTag_2_2( &(parent->nextLevel[n]), mem );
	    break;

      case UFDBsameLevel:
	    DEBUGP(( "   parse2_2 tag = %-18s  same-level", tag )); /* */

	    n = parent->nNextLevels;
	    parent->nNextLevels++;
	    parent->nextLevel[n].tag = tag;
	    parent->nextLevel[n].nNextLevels = 0;
	    parent->nextLevel[n].nextLevel = NULL;

	    break;

      case UFDBprevLevel:
	    if (tag[0] >= ' ')   /* is the code followed by a tag or another code ? */
	    {
	       DEBUGP(( "   parse2_2 tag = %-18s  prev-level", tag )); /* */

	       n = parent->nNextLevels;
	       parent->nNextLevels++;
	       parent->nextLevel[n].tag = tag;
	       parent->nextLevel[n].nNextLevels = 0;
	       parent->nextLevel[n].nextLevel = NULL;
	    }
	    else 
	    {
	       DEBUGP(( "   parse2_2 tag = %-18s  prev-level", "*" )); /* */
	       ;
	    }
	    return mem;
	    break;

      case UFDBendTable:
	    if (tag[0] >= ' ')   /* is the code followed by a tag or another code ? */
	    {
	       DEBUGP(( "   parse2_2 tag = %-18s  end-of-table", tag )); /* */
	       n = parent->nNextLevels;
	       parent->nNextLevels++;
	       parent->nextLevel[n].tag = tag;
	       parent->nextLevel[n].nNextLevels = 0;
	       parent->nextLevel[n].nextLevel = NULL;
	    }
	    else
	    {
	       DEBUGP(( "   parse2_2 tag = %-18s  end-of-table", "END" )); /* */
	       ;
	    }
	    return NULL;
	    break;

      default:
            ufdbLogFatalError( "cannot parse 2.2 table: tag type is %03o", tagType );
#if UFDB_DO_DEBUG
	    printMem( mem );
#endif
	    return NULL;
	    break;
      }
   }

   return NULL;
}


#if UFDB_DBFORMAT_3
static void ufdbParseTable_3_0( struct UFDBmemTable * mt );
#endif
#if UFDB_DBFORMAT_3_1
static void ufdbParseTable_3_1( struct UFDBmemTable * mt );
#endif


/* Parse a binary table that is loaded into memory.
 */
void UFDBparseTable( struct UFDBmemTable * mt )
{
   unsigned char * memStart;

   mt->table.nNextLevels = 0;
   mt->table.nextLevel = (struct UFDBtable *) ufdbCalloc( 1, sizeof(struct UFDBtable) );

   if (ufdbGV.debug > 1)
      ufdbLogMessage( "UFDBparseTable: version=%s  flags=%8.8s  %d  %'ld  %d  hdrsize=%d",
                      mt->version, mt->flags, mt->numEntries, mt->fileSize, 
		      mt->indexSize, mt->hdrSize );

   tableMemStart = (unsigned char *) mt->mem;

#if UFDB_DBFORMAT_3_1
   if (strcmp( mt->version, "3.1" ) == 0)
   {
      mt->index = NULL;
      tableIndex = NULL;
      tableIndex_i = 0;
      memStart = (unsigned char *) mt->mem + mt->hdrSize + (mt->flags[2] == 'Q' ? 64 : 0);
      (void) ufdbParseTable_3_1( mt );
   }
   else
#endif
#if UFDB_DBFORMAT_3
   if (strcmp( mt->version, "3.0" ) == 0)
   {
      mt->index = NULL;
      tableIndex = NULL;
      tableIndex_i = 0;
      memStart = (unsigned char *) mt->mem + mt->hdrSize + (mt->flags[2] == 'Q' ? 64 : 0);
      (void) ufdbParseTable_3_0( mt );
   }
   else
#endif
   if (strcmp( mt->version, "2.2" ) >= 0)
   {
      mt->index = (struct UFDBtable *) ufdbCalloc( mt->indexSize+1, sizeof(struct UFDBtable) );
      tableIndex = (struct UFDBtable *) mt->index;
      tableIndex_i = 0;
      memStart = (unsigned char *) mt->mem + mt->hdrSize + 64 * (mt->flags[2] == 'Q');
      (void) parseNextTag_2_2( &(mt->table), memStart );
   }
   else if (strcmp( mt->version, "2.1" ) >= 0)
   {
      mt->index = (struct UFDBtable *) ufdbCalloc( mt->indexSize+1, sizeof(struct UFDBtable) );
      tableIndex = (struct UFDBtable *) mt->index;
      tableIndex_i = 0;
      (void) parseNextTag_2_1( &(mt->table),
                               (unsigned char *) mt->mem + mt->hdrSize + 
                                                                 64 * (mt->flags[2] == 'Q') );
   }
   else if (strcmp( mt->version, "2.0" ) >= 0)
   {
      mt->index = (struct UFDBtable *) ufdbCalloc( mt->indexSize+1, sizeof(struct UFDBtable) );
      tableIndex = (struct UFDBtable *) mt->index;
      tableIndex_i = 0;
      (void) parseNextTag_2_0( &(mt->table),
                               (unsigned char *) mt->mem + mt->hdrSize );
#if UFDB_DO_DEBUG
      ufdbLogMessage( "predicted index size: %d   last index: %d", mt->indexSize, tableIndex_i );
#endif
   }
   else 	/* version == "1.2" */
   {
#if UFDB_BARE_METAL_SUPPORT
      ufdbLogFatalError( "cannot parse database table with DB format %s", mt->version );
      mt->index = NULL;
#else
      mt->index = NULL;
      (void) parseNextTag_1_2( &(mt->table), 
                               (unsigned char *) mt->mem + mt->hdrSize );
#endif
   }
}


UFDB_GCC_HOT UFDB_GCC_INLINE
static int mystrn2cmp( unsigned char * s1, unsigned char * s2, const char * terminators )
{
   int diff;

   DEBUG(( "      mystrn2cmp( '%s' '%s' '%s' )", (char*) s1, (char*) s2, (char*) terminators ));
   // if (ufdbGV.debug)
   //    ufdbLogMessage( "      mystrn2cmp( '%s' '%s' '%s' )", (char*) s1, (char*) s2, (char*) terminators );

   diff = 0;
   while (*s2 != '\0')
   {
      diff = ((unsigned int) *s1) - ((unsigned int) *s2);
      if (diff != 0)
         return diff;
      s2++;
      s1++;
   }
   diff = ((unsigned int) *s1) - ((unsigned int) *s2);
   DEBUG(( "!        mystrn2cmp( '%s' '%s' '%s' )   diff %d", 
           (char*) s1, (char*) s2, terminators, diff ));
   while (*terminators != '\0')
   {
      if (*s1 == (unsigned char) *terminators)
      {
         DEBUG(( "         mystrn2cmp matched terminator %c  return 0", *terminators ));
         return 0;
      }
      terminators++;
   }

   DEBUG(( "         mystrn2cmp reached eos s2  return diff %d", diff ));
   return diff;
}


UFDB_GCC_HOT   /* recursive */
static int verifyURLparams( 
   struct UFDBtable * t, 
   UFDBrevURL *       revUrl )
{
   UFDBrevURL *	      origRevUrl;
   struct UFDBtable * tparam;
   unsigned char *    URLparam;
   unsigned char *    URLvalue;
   int                nl;
   int                tagLen;
   int                i, b, e, cmp;

   origRevUrl = revUrl;
   DEBUG(( "    verifyURLparams:  t.tag[0] '%s'  revURL '%s'", 
           (char*) t->nextLevel[0].tag, (char*) revUrl->part ));

   /* At this point revURL points to the list of parameters and
    * t points to a table of parameters and values with OR logic
    */
   for (nl = 0;  nl < t->nNextLevels;  nl++)
   {
      tparam = &t->nextLevel[nl];
      revUrl = origRevUrl;
      URLparam = revUrl->part;

      tagLen = 0;
      while (tparam->tag[tagLen] != '\0')	/* inlined strlen() */
         tagLen++;

      DEBUG(( "    verifyURLparams: table parameter %d '%s'  tagLen %d  nNextLevels %d  "
              "Matching against parameters %s", 
	      nl, tparam->tag, tagLen, tparam->nNextLevels, URLparam ));

      /* Find the parameter (tparam->tag) in the URL parameters (revUrl->part).
       * Note that a simple strstr() does not work since the parameter name must be right after BEGINNING or &.
       */
      while (URLparam != NULL)
      {
	 if (mystrn2cmp( URLparam, tparam->tag, "=&" ) == 0
             &&  (URLparam[tagLen] == '='  ||  URLparam[tagLen] == '&'  ||  URLparam[tagLen] == '\0'))
	 {
            /* we must return 1 iff the table has no values for this parameter !!!  */
            if (tparam->nNextLevels == 0)
            {
               DEBUG(( "    verifyURLparams: found parameter '%s' in URL and table has no values  return 1", 
                       tparam->tag ));
               return 1;
            }

            /* check valueless parameter */
            if (URLparam[tagLen] == '&'  ||  URLparam[tagLen] == '\0')
            {
               if (tparam->nNextLevels == 1  &&  tparam->nextLevel[0].tag[0] == '\0')
               {
                  DEBUG(( "    verifyURLparams: found valueless parameter '%s' check the next parameter",
                          tparam->tag ));
                  return verifyURLparams( &(tparam->nextLevel[0]), origRevUrl );
               }

               DEBUG(( "    verifyURLparams: found valueless parameter '%s' in URL and table has values  return 0",
                       tparam->tag ));
               return 0;
            }

	    URLvalue = URLparam + tagLen + 1;

	    DEBUG(( "    verifyURLparams: found parameter '%s' in URL.  value '%s'", 
                    tparam->tag, URLvalue ));

	    /* do a binary search for the URLvalue in tparam */
	    b = 0;
	    e = tparam->nNextLevels - 1;
	    DEBUG(( "    verifyURLparams: starting binary search  b=%d  e=%d", b, e ));
	    while (b <= e)
	    {
	       i = (b + e) >> 1;
	       cmp = mystrn2cmp( URLvalue, tparam->nextLevel[i].tag, "&" );
	       DEBUG(( "         mystrn2cmp[%d]  '%s'  '%s'  = %3d", 
                       i, URLvalue, tparam->nextLevel[i].tag, cmp ));
	       if (cmp < 0)
		  e = i - 1;
	       else if (cmp > 0)
		  b = i + 1;
	       else
	       {
		  /* see if there are more parameters to match */
		  if (tparam->nextLevel[i].nNextLevels > 0)
                  {
                     DEBUG(( "    verifyURLparams: check the next parameter" ));
		     return verifyURLparams( &(tparam->nextLevel[i]), origRevUrl );
                  }
                  DEBUG(( "    verifyURLparams: there are no more parameters: return 1" ));
		  return 1;
	       }
	    }
            DEBUG(( "    verifyURLparams: binary search finished: b=%d e=%d  return 0", b, e ));
	    return 0;
	 }
         else
	 {
	    DEBUG(( "    verifyURLparams: '%s' not first parameter in '%s'", 
                    (char*) tparam->tag, (char*) URLparam ));
         }

	 URLparam = (unsigned char *) mystrchr( (char*) URLparam, '&' );
	 if (URLparam != NULL)
	    URLparam++;
	 else
	    if (revUrl->next != NULL)
	    {
               DEBUG(( "      verifyURLparams: goto next revUrl part" ));
	       revUrl = revUrl->next;
	       URLparam = revUrl->part;
	    }
      }
   }

   return 0;
}


/* perform lookup of revUrl in table t.
 * return 1 iff found, 0 otherwise.
 */
UFDB_GCC_HOT 
int UFDBlookupRevUrl( 
   struct UFDBtable * t, 
   UFDBrevURL *       revUrl )
{
   int b, e;
#if UFDB_DO_DEBUG
   struct UFDBtable * origtable = t;
#endif

begin:
   DEBUG(( "    UFDBlookupRevUrl:  table %-14s [%d]  tag %s  :  %s  ipv4 %d  ipv6 %d",
           origtable->tag, origtable->nNextLevels, t->tag, revUrl->part, revUrl->ipv4, revUrl->ipv6 ));

   /* binary search */
   b = 0;
   e = t->nNextLevels - 1;

   while (b <= e)
   {
      int  cmp;
      int  i, is_path;

      i = (b + e) >> 1;
      cmp = strcmpURLpart( (char *) revUrl->part, (char *) t->nextLevel[i].tag );
      DEBUG(( "      strcmpURLpart[%d]  %s  %s  = %d   b %d  e %d",
              i, revUrl->part, t->nextLevel[i].tag, cmp, b, e ));
      if (cmp < 0)
	 e = i - 1;
      else if (cmp > 0)
	 b = i + 1;
      else					/* URL part and tag are equal */
      {
	 is_path = (revUrl->part[0] == '/');
         struct UFDBtable * t0 = t;
         UFDBrevURL * ru0 = revUrl;
	 t = &(t->nextLevel[i]);                /* advance to next level */
	 revUrl = revUrl->next;                 /* advance to next level */

	 DEBUG(( "      is_path %d  t->nNextLevels %d  t->nextLevel[%d].tag %s", is_path,
                 t->nNextLevels, t->nNextLevels-1,
                 t->nNextLevels > 0 ? t->nextLevel[t->nNextLevels-1].tag : (unsigned char*) "--" ));

	 if (t->nNextLevels == 0)		/* no more levels in table URL -> MATCH */
	    return 1;

         if (t->nextLevel[t->nNextLevels-1].tag[0] == '|'  &&  t->nextLevel[t->nNextLevels-1].tag[1] == '\0')
         {
	    struct UFDBtable * tc;
            tc = &(t->nextLevel[t->nNextLevels-1]);
            DEBUG(( "      UFDBlookupRevUrl: found |.  revUrl %s  tag[0] %s",
                            revUrl==NULL ? (char*)"NULL" : (char*)revUrl->part,
                            tc->nNextLevels==0 ? (char*)"NULL" : (char*)tc->nextLevel[0].tag ));
            if (tc->nNextLevels == 0)           /* no more levels in table URL */
            {
               if (revUrl == NULL  ||  revUrl->part[0] == '/')
	       {
		  DEBUG(( "      UFDBlookupRevUrl: found |  revUrl is NULL or /... and matches" ));
                  return 1;                     // no more subdomains in browser URL -> MATCH */
	       }
               else
	       {
		  DEBUG(( "      UFDBlookupRevUrl: found |  revUrl does not match '|'" ));
		  if (t->nNextLevels == 1)
		     return 0;
	       }
            }
         }

         if (is_path  &&  revUrl == NULL)
         {
            // in case that there is no match, it may happen that the next table entry 
            // has a matching path without parameters that matches so continue with the search.
            t = t0;
            revUrl = ru0;
            b = i + 1;
            DEBUG(( "      may have other paths that match.  continue search   b %d", b ));
            continue;
         }

	 if (revUrl == NULL)
	 {
	    DEBUG(( "      UFDBlookupRevUrl: revUrl is NULL.  no match" ));
	    return 0;                           /* no more levels in browser URL -> NO match */
	 }

	 if (is_path)
	 {
	    int rv = verifyURLparams( t, revUrl );
	    DEBUG(( "      strcmpURLpart[%d]   verifyURLparams returned %d", i, rv ));
            return rv;
	 }

	 /* optimise the recursive call : */
	 /* return UFDBlookupRevUrl( t, revUrl ); */
	 goto begin;
      }
   }

   return 0;  /* not found */
}


#if UFDB_DBFORMAT_3
#include "ufdbParseTable3.c"
#endif


#if UFDB_OVERRIDE_GCC_OPT  &&  ((__GNUC__ > 4)  ||  (__GNUC__ == 4  &&  __GNUC_MINOR__ >= 4))
#pragma GCC pop_options
#endif


/*
 *  UFDBlookup: lookup a domain/URL in domain and URL databases.  
 *  return 1 if found.
 */
int UFDBlookup( 
   UFDBthreadAdmin *     admin,
   struct UFDBmemTable * mt, 
   char *                request )
{
   int                   result;
   UFDBrevURL *          revUrl;
   
   if (admin == NULL)
   {
      ufdbLogFatalError( "UFDBlookup admin=NULL" );
      return 0;
   }

   revUrl = UFDBgenRevURL( admin, (unsigned char *) request );
   result = UFDBlookupRevUrl( &(mt->table.nextLevel[0]), revUrl );
   DEBUG(( "  UFDBlookup( %s, %s ) = %d", mt->table.nextLevel[0].tag, request, result ));

   UFDBfreeRevURL( admin, revUrl );

   return result;
}


#if UFDB_BARE_METAL_SUPPORT && __OCTEON__
int UFDBloadDatabase( 
   struct ufdbGV *       gv,
   struct UFDBmemTable * mt, 
   char *                file    )
{
   int     retval;
   int64_t length = 0;

   if (file == NULL)
   {
      mt->table.tag = (unsigned char *) "FILE-IS-NULL";
      mt->table.nNextLevels = 0;
      mt->table.nextLevel = NULL;
      mt->mem = NULL;
      mt->memStatus = UFDB_MEMSTATUS_UNKNOWN;
      mt->madvisedSize = 0;
      mt->fileSize = 0;
      mt->hdrSize = 0;
      mt->version[0] = '\0';
      strcpy( mt->flags, "--------" );
      mt->key[0] = '\0';
      mt->date[0] = '\0';
      mt->numEntries = 0;
      mt->indexSize = 0;
      mt->index = NULL;
      mt->db3p[0] = NULL;
      return UFDB_API_ERR_NULL;
   }

   mt->memStatus = UFDB_MEMSTATUS_DP;
   mt->mem = DPufdbLoadFile( file, &length );
   if (mt->mem == NULL)
   {
      ufdbLogFatalError( "UFDBloadDatabase: DPufdbLoadFile(%s) returned NULL; error %ld", file, length );
      mt->table.tag = (unsigned char *) "LOAD-FAILED";
      mt->table.nNextLevels = 0;
      mt->table.nextLevel = NULL;
      mt->mem = NULL;
      mt->memStatus = UFDB_MEMSTATUS_UNKNOWN;
      mt->madvisedSize = 0;
      mt->fileSize = 0;
      mt->hdrSize = 0;
      mt->version[0] = '\0';
      strcpy( mt->flags, "--------" );
      mt->key[0] = '\0';
      mt->date[0] = '\0';
      mt->numEntries = 0;
      mt->indexSize = 0;
      mt->index = NULL;
      mt->db3p[0] = NULL;
      return UFDB_API_ERR_NOFILE;
   }

   mt->table.tag = (unsigned char *) "UNIVERSE";
   mt->table.nNextLevels = 0;
   mt->table.nextLevel = NULL;
   mt->madvisedSize = 0;
   mt->fileSize = length;

   retval = UFDBparseTableHeader( mt );
#if 0
   ufdbLogMessage( "   UFDBparseTableHeader returns %d", retval );
#endif
   if (retval != UFDB_API_OK  &&  retval != UFDB_API_STATUS_DATABASE_OLD)
   {
      mt->table.tag = (unsigned char *) "READ-ERROR";
      mt->table.nNextLevels = 0;
      mt->table.nextLevel = NULL;
      if (mt->memStatus == UFDB_MEMSTATUS_MALLOC)
         ufdbFree( mt->mem );
      mt->mem = NULL;
      mt->memStatus = UFDB_MEMSTATUS_UNKNOWN;
      mt->madvisedSize = 0;
      mt->fileSize = 0;
      mt->hdrSize = 0; 
      mt->version[0] = '\0';
      strcpy( mt->flags, "--------" );
      mt->key[0] = '\0';
      mt->date[0] = '\0';
      mt->numEntries = 0;
      mt->indexSize = 0;
      mt->index = NULL;
      mt->db3p[0] = NULL;
      return retval;
   }
   UFDBparseTable( mt );

   gv->databaseLoadTime = UFDBtime();
   if (strstr( file, "checked" ) != NULL)
      strcpy( gv->dateOfCheckedDB, mt->date );

   return retval;
}

#else

/*
 *  Initialise a database category (open a .ufdb file) 
 */
int UFDBloadDatabase( 
   struct ufdbGV *       gv,
   struct UFDBmemTable * mt, 
   char *                file    )
{
   int                   n;
   int                   in;
   struct stat           fileStat;
   char                  f[1024];

   if (file == NULL)
   {
      mt->table.tag = (unsigned char *) "FILE-IS-NULL";
      mt->table.nNextLevels = 0;
      mt->table.nextLevel = NULL;
      mt->mem = NULL;
      mt->memStatus = UFDB_MEMSTATUS_UNKNOWN;
      mt->madvisedSize = 0;
      mt->fileSize = 0;
      mt->hdrSize = 0;
      mt->version[0] = '\0';
      strcpy( mt->flags, "--------" );
      mt->key[0] = '\0';
      mt->date[0] = '\0';
      mt->numEntries = 0;
      mt->indexSize = 0;
      mt->index = NULL;
      mt->db3p[0] = NULL;
      return UFDB_API_ERR_NULL;
   }

   strcpy( f, file );
   in = open( f, O_RDONLY );
   if (in < 0)
   {
      strcat( f, UFDBfileSuffix );
      in = open( f, O_RDONLY );
   }

   if (in < 0)
   {
      ufdbLogError( "UFDBloadDatabase: %s[" UFDBfileSuffix "] does not exist", file );
      mt->table.tag = (unsigned char *) "NO-FILE";
      mt->table.nNextLevels = 0;
      mt->table.nextLevel = NULL;
      mt->mem = NULL;
      mt->memStatus = UFDB_MEMSTATUS_UNKNOWN;
      mt->madvisedSize = 0;
      mt->fileSize = 0;
      mt->hdrSize = 0;
      mt->version[0] = '\0';
      strcpy( mt->flags, "--------" );
      mt->key[0] = '\0';
      mt->date[0] = '\0';
      mt->numEntries = 0;
      mt->indexSize = 0;
      mt->index = NULL;
      mt->db3p[0] = NULL;
      return UFDB_API_ERR_NOFILE;
   }
   if (ufdbGV.debug)
      ufdbLogMessage( "UFDBloadDatabase: %s", file );

   if (fstat(in,&fileStat) < 0)
   {
      close( in );
      mt->table.tag = (unsigned char *) "FSTAT-ERROR";
      mt->table.nNextLevels = 0;
      mt->table.nextLevel = NULL;
      mt->mem = NULL;
      mt->memStatus = UFDB_MEMSTATUS_UNKNOWN;
      mt->madvisedSize = 0;
      mt->fileSize = 0;
      mt->hdrSize = 0;
      mt->version[0] = '\0';
      strcpy( mt->flags, "--------" );
      mt->key[0] = '\0';
      mt->date[0] = '\0';
      mt->numEntries = 0;
      mt->indexSize = 0;
      mt->index = NULL;
      return UFDB_API_ERR_NOFILE;
   }

   mt->table.tag = (unsigned char *) "UNIVERSE";
   mt->table.nNextLevels = 0;
   mt->table.nextLevel = NULL;
   mt->madvisedSize = 0;
   mt->fileSize = fileStat.st_size;

   // Here we can improve performance by doing an alignment of HUGEPAGESIZE.  
   // If Transparent Hugepages (THP) is configured, Linux uses hugepages.
   // If envars LD_PRELOAD=libhugetlbfs.so HUGETLB_MORECORE=yes are set, malloc
   // will allocate optimally aligned hugepages.
   size_t alignment = 4096;
   size_t tabsize = fileStat.st_size + 1 + 32; // add 32-byte safeguard for AVX2
   if (tabsize >= ((size_t) (HUGEPAGESIZE * 0.8)))
   {
      alignment = HUGEPAGESIZE;
      if (tabsize < HUGEPAGESIZE)
         tabsize = HUGEPAGESIZE;                        // round up if ~80% of 1 hugepage is requested
      else if (tabsize >= 3*HUGEPAGESIZE  ||  gv->madviseHugePages)
         tabsize += HUGEPAGESIZE - (tabsize % HUGEPAGESIZE);    // round up if size >= 3 hugepages
      // *ONLY* madvise if not freed immediately by decompressors
      if (gv->madviseHugePages  &&  mt->flags[0] == '-')
#if __linux__  &&  HAVE_MADVISE && !UFDB_BARE_METAL_SUPPORT
         mt->madvisedSize = tabsize;
#else
         ufdbLogMessage( "madviseHugePages is set but madvise() is not available  ***" );
#endif
   }
   else 
      tabsize += (tabsize + 4096 - 1) % 4096;                    // round up to 4K 

   mt->mem = (char *) ufdbMallocAligned( alignment, tabsize );
   mt->memStatus = UFDB_MEMSTATUS_MALLOC;

#if __linux__  &&  HAVE_MADVISE && !UFDB_BARE_METAL_SUPPORT
   if (mt->madvisedSize)
   {
      int retval = madvise( mt->mem, tabsize, MADV_HUGEPAGE );
      if (retval != 0)
      {
         char   strbuf[128];
         if (strerror_r( errno, strbuf, sizeof(strbuf) )) {;}
         ufdbLogMessage( "WARNING: UFDBloadDatabase: madvise( %p %'ld hugepage ) returns error %d: %s",
                         mt->mem, tabsize, errno, strbuf );
      }
   }
#endif

   n = read( in, mt->mem, fileStat.st_size );
   close( in );
   if (n != fileStat.st_size)
   {
      mt->table.tag = (unsigned char *) "READ-ERROR";
      mt->table.nNextLevels = 0;
      mt->table.nextLevel = NULL;
      if (mt->memStatus == UFDB_MEMSTATUS_MALLOC)
         ufdbFree( mt->mem );
      mt->mem = NULL;
      mt->memStatus = UFDB_MEMSTATUS_UNKNOWN;
      mt->madvisedSize = 0;
      mt->fileSize = 0;
      mt->hdrSize = 0; 
      mt->version[0] = '\0';
      strcpy( mt->flags, "--------" );
      mt->key[0] = '\0';
      mt->date[0] = '\0';
      mt->numEntries = 0;
      mt->indexSize = 0;
      mt->index = NULL;
      mt->db3p[0] = NULL;
      return UFDB_API_ERR_READ;
   }

   n = UFDBparseTableHeader( mt );
#if 0
   ufdbLogMessage( "   UFDBparseTableHeader returns %d", n );
#endif
   if (n != UFDB_API_OK  &&  n != UFDB_API_STATUS_DATABASE_OLD)
   {
      mt->table.tag = (unsigned char *) "READ-ERROR";
      mt->table.nNextLevels = 0;
      mt->table.nextLevel = NULL;
      if (mt->memStatus == UFDB_MEMSTATUS_MALLOC)
         ufdbFree( mt->mem );
      mt->mem = NULL;
      mt->memStatus = UFDB_MEMSTATUS_UNKNOWN;
      mt->madvisedSize = 0;
      mt->fileSize = 0;
      mt->hdrSize = 0; 
      mt->version[0] = '\0';
      strcpy( mt->flags, "--------" );
      mt->key[0] = '\0';
      mt->date[0] = '\0';
      mt->numEntries = 0;
      mt->indexSize = 0;
      mt->index = NULL;
      mt->db3p[0] = NULL;
      return n;
   }
   UFDBparseTable( mt );

   gv->databaseLoadTime = UFDBtime();
#if UFDBSS_SQUID || UFDBSS_API || UFDBSS_BIND
   if (strstr( f, "checked/domains.ufdb" ) != NULL)
      strcpy( gv->dateOfCheckedDB, mt->date );
#endif

   return n;
}
#endif


const char * ufdbCategoryName( const char * domain )
{
#if UFDBSS_SQUID
   UFDBrevURL *      revurl;
   struct Category * cat;
   char *            buffer;
   UFDBthreadAdmin   myadm;

   if (domain == NULL)
      return "null";
   
   if (ufdbGV.reconfig || ufdbGV.terminating)
      return "unknown";

   UFDBinitThreadAdmin( &myadm );

   buffer = ufdbStrdup( domain );
   revurl = UFDBgenRevURL( &myadm, (unsigned char *) buffer );

   for (cat = ufdbGV.catList;  cat != NULL;  cat = cat->next)
   {
      if (cat->domainlistDb != NULL)
      {
	 struct UFDBmemTable * mt;
	 mt = (struct UFDBmemTable *) cat->domainlistDb->dbcp;
	 if (mt != NULL)
	 {
            // NOTE: UFDBlookupRevUrl() does not work with DB format 3.x
            // but this code is only inside ufdbGuard for Squid so it is OK.
	    if (UFDBlookupRevUrl( &(mt->table.nextLevel[0]), revurl ))
	       break;
	 }
      }
   }

   ufdbFree( buffer );
   UFDBfreeRevURL( &myadm, revurl );

   if (cat != NULL)
      return cat->name;
#else
   // prevent compiler warning
   if (domain == NULL)
      { ; }
#endif

   return "any";
}


#if UFDBSS_SQUID
/*
 * Initialise a DB (open and/or create a .ufdb file) 
 */
void sgDbInit( 
   struct sgDb *         Db, 
   char *                file )
{
   struct UFDBmemTable * mt;
   int                   n;
   int                   in;
   struct stat           fileStat;
   struct stat           basefileStat;
   char                  f[1024];

   if (file == NULL  ||  *file == '\0')
   {
      ufdbLogError( "sgDbInit: cannot load URL table since file is NULL" );
      Db->dbcp = NULL;
      Db->entries = -1;
      return;
   }

#if defined(UFDB_DEBUG)
   ufdbLogMessage( "  sgDbInit( 0x%08x, %s )", Db, file );
#endif

   if (strstr( file, UFDBfileSuffix ) == NULL)
      sprintf( f, "%s%s", file, UFDBfileSuffix );
   else
      strcpy( f, file );

   in = open( f, O_RDONLY );
   if (in < 0)
   {
      char   strbuf[128];
      if (strerror_r( errno, strbuf, sizeof(strbuf) )) {;}
      ufdbLogFatalError( "sgDbInit: cannot read from \"%s\" (read-only): %s", f, strbuf );
      Db->dbcp = NULL;
      Db->entries = -1;
      return;
   }

   if (ufdbGV.debug)
      ufdbLogMessage( "sgDbInit: %s", f );

   if (fstat(in,&fileStat) < 0)
   {
      char   strbuf[128];
      if (strerror_r( errno, strbuf, sizeof(strbuf) )) {;}
      ufdbLogFatalError( "fstat failed on \"%s\": %s", file, strbuf );
      Db->dbcp = NULL;
      Db->entries = -1;
      return;
   }

   if (stat( file, &basefileStat ) == 0)
   {
      if (basefileStat.st_mtime > fileStat.st_mtime)
         ufdbLogMessage( "WARNING: %s is newer than %s.  You should run ufdbGenTable *****", file, f );
   }

   mt = (struct UFDBmemTable *) ufdbMallocAligned( UFDB_CACHELINE_SIZE, sizeof(struct UFDBmemTable) );
   if (mt == NULL)
   {
      ufdbLogFatalError( "cannot allocate %'ld bytes memory for table header", (long) sizeof(struct UFDBmemTable) );
      Db->dbcp = NULL;
      Db->entries = -1;
      return;
   }
   mt->fileSize = fileStat.st_size;
   mt->table.tag = (unsigned char *) "UNIVERSE";
   mt->table.nNextLevels = 0;
   mt->table.nextLevel = NULL;
   mt->memStatus = UFDB_MEMSTATUS_MALLOC;
   mt->madvisedSize = 0;
   /* old user-defined tables may not have a 32-byte safeguard for SSE and AVX2 instructions */
   mt->mem = ufdbMallocAligned( 4096, fileStat.st_size + 1 + 32 );
   if (mt->mem == NULL)
   {
      ufdbLogFatalError( "cannot allocate %'ld bytes memory for table %s", (long) fileStat.st_size+1, file );
      ufdbFree( mt );
      Db->dbcp = NULL;
      Db->entries = -1;
      return;
   }
   n = read( in, mt->mem, fileStat.st_size );
   if (n != fileStat.st_size)
   {
      char   strbuf[128];
      if (strerror_r( errno, strbuf, sizeof(strbuf) )) {;}
      ufdbLogFatalError( "read failed on \"%s\" n=%d st_size=%'ld %s",
                         file, n, (long) fileStat.st_size, strbuf );
      if (mt->memStatus == UFDB_MEMSTATUS_MALLOC)
         ufdbFree( mt->mem );
      ufdbFree( mt );
      Db->dbcp = NULL;
      Db->entries = -1;
      return;
   }
   close( in );

   n = UFDBparseTableHeader( mt );
   if (n == UFDB_API_STATUS_DATABASE_OLD)
   {
      ufdbLogMessage( "warning: the URL table \"%s\" is over 4 days old  *****", file );
      n = UFDB_API_OK;
   }
   if (n == UFDB_API_OK)
      UFDBparseTable( mt );
   else
   {
      ufdbLogFatalError( "table \"%s\" could not be parsed. error code = %d", file, n );
      if (mt->memStatus == UFDB_MEMSTATUS_MALLOC)
         ufdbFree( mt->mem );
      ufdbFree( mt );
      Db->dbcp = NULL;
      Db->entries = -1;
      return;
   }

   Db->dbcp = (void *) mt;
   Db->entries = mt->numEntries;
}


/*
 *  sgDbLookup: lookup an item in a in-memory database.
 */
int sgDbLookup( struct sgDb * Db, char * request, char ** retval )
{
#if 0
   ufdbLogError( "  sgDbLookup( 0x%08x, %s, 0x%08x )", Db, request, retval );
#endif

   // prevent compiler warning
   if (retval == NULL)
      { ; }

   return UFDBlookup( NULL, (struct UFDBmemTable *) Db->dbcp, request );
}

#endif

#ifdef __cplusplus
}
#endif
