/*
 * ufdbdaemon.c - URLfilterDB
 *
 * ufdbGuard is copyrighted (C) 2005-2020 by URLfilterDB with all rights reserved.
 *
 * Parts of ufdbGuard are based on squidGuard.
 * This module is NOT based on squidGuard.
 * 
 * RCS $Id: ufdbdaemon.c,v 1.8 2022/12/26 23:16:12 root Exp root $
 */


#include "ufdb.h"
#include "ufdblib.h"

#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <ctype.h>
#include <signal.h>
#include <errno.h>
#if HAVE_SYS_SYSCALL_H
#include <sys/syscall.h>
#endif
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <pwd.h>

#if HAVE_PCRE_COMPILE
#include <pcre.h>
#endif

#if HAVE_PR_SET_TRACER
#include <sys/prctl.h>
#endif

#ifndef UFDB_API_NO_THREADS
#include <pthread.h>
#endif

#ifdef __cplusplus
extern "C" {
#endif

#if HAVE_SETRESUID
int setresuid( uid_t ruid, uid_t euid, uid_t suid );
int getresuid( uid_t * ruid, uid_t * euid, uid_t * suid );
#endif


static long          num_cpus;
static unsigned long cpu_mask = 0UL;
#define UFDB_MAX_CPUS 64


#if defined(__GLIBC__)
#if (__GLIBC__ > 2) || (__GLIBC__ == 2  &&  __GLIBC_MINOR__ >= 4) || HAVE_PCRE_COMPILE
#define NEED_REGEXEC_MUTEX 0
#else
#define NEED_REGEXEC_MUTEX 1
#endif
#else
#define NEED_REGEXEC_MUTEX 0
#endif

#if NEED_REGEXEC_MUTEX
pthread_mutex_t ufdb_regexec_mutex = UFDB_STATIC_MUTEX_INIT;
#endif

#ifdef UFDB_DEBUG
pthread_mutex_t ufdb_malloc_mutex = UFDB_STATIC_MUTEX_INIT;
#endif


/*
 * Linux support binding threads to a set of CPUs which prevents cache contention.
 * Freebsd has no support, but the 5.x series is recommended for best multithreaded performance.
 * On Solaris it is recommended to start ufdbguardd in a processor set.
 */
int ufdbSetThreadCPUaffinity( int thread_num )
{
   if (thread_num != 0)         // prevent compiler warning
      { ; }

#if defined(__linux__) && defined(SYS_sched_setaffinity)
   int retval;

   if (cpu_mask != 0UL)
   {
      /* TODO: use sched_setaffinity() */
      retval = syscall( SYS_sched_setaffinity, 0, 4, &cpu_mask );
      if (retval < 0)
         return UFDB_API_ERR_ERRNO;
   }
#endif

   return UFDB_API_OK;
}


/*
 * Bind my processes and threads to a set of CPUs.
 * This increases the cache efficiency for all programs.
 */
int ufdbSetCPU( 
   char * CPUspec )		/* comma-separated CPU numbers (starting with CPU 0) */
{
   int    cpu;

#if defined(_SC_NPROCESSORS_ONLN)
   num_cpus = sysconf( _SC_NPROCESSORS_ONLN );
   /* printf( "this system has %ld CPUs\n", num_cpus ); */

#elif defined(__linux__) && defined(SYS_sched_getaffinity)
   /* sched_getaffinity() is buggy on linux 2.4.x so we use syscall() instead */
   cpu = syscall( SYS_sched_getaffinity, getpid(), 4, &cpu_mask );
   /* printf( "sched_getaffinity returned %d %08lx\n", cpu, cpu_mask ); */
   if (cpu >= 0)
   {
      num_cpus = 0;
      for (cpu = 0; cpu < UFDB_MAX_CPUS; cpu++)
	 if (cpu_mask & (1 << cpu))
	    num_cpus++;
      /* printf( "   found %d CPUs in the cpu mask\n", num_cpus ); */
   }
   else
#else
      num_cpus = UFDB_MAX_CPUS;
#endif

   cpu_mask = 0;
   while (*CPUspec)
   {
      if (sscanf(CPUspec,"%d",&cpu) == 1)
      {
	 if (cpu >= 0 && cpu < num_cpus)
	 {
	    cpu_mask |= (1 << cpu);
	 }
	 else
	    return UFDB_API_ERR_RANGE;
      }
      else
         return UFDB_API_ERR_RANGE;

      /* find the next CPU number */
      while (isdigit( (int) *CPUspec))
         CPUspec++;
      while (*CPUspec == ' '  ||  *CPUspec == ',')
         CPUspec++;
   }

   return UFDB_API_OK;
}


void ufdbSetSignalHandler( int signal, void (*handler)(int) )
{
#if HAVE_SIGACTION
   struct sigaction act;

#ifndef SA_NODEFER
#define SA_NODEFER 0
#endif

#ifndef SA_NOCLDSTOP
#define SA_NOCLDSTOP 0
#endif

   act.sa_handler = handler;
   act.sa_flags = SA_RESTART;
   if (signal == SIGCHLD)
      act.sa_flags |= SA_NOCLDSTOP;
   if (signal == SIGALRM)
      act.sa_flags |= SA_NODEFER;
   sigemptyset( &act.sa_mask );
   sigaction( signal, &act, NULL );

#else

#if HAVE_SIGNAL
   signal( signal, handler );
#else
   ufdbLogError( "ufdbSetSignalHandler: cannot set handler for signal %d", signal );
#endif

#endif
}


static pid_t   ruid = -1;
static pid_t   euid;


void UFDBdropPrivileges( 
   const char *    username )
{
   struct passwd * fieldptrs;
#if HAVE_GETPWNAM_R
   struct passwd   pws;
   char            pws_fields[1024];
#endif

   if (ruid == -1)
   {
      ruid = getuid();
      euid = geteuid();

#ifdef UFDB_WORLD_READABLE_LOGFILES
      /* umask 133: no x for user, no wx for group, no wx for others */
      umask( S_IXUSR | S_IWGRP|S_IXGRP | S_IWOTH|S_IXOTH );
#else
      /* umask 137: no x for user, no wx for group, no rwx for others */
      umask( S_IXUSR | S_IWGRP|S_IXGRP | S_IROTH|S_IWOTH|S_IXOTH );
#endif
   }

   if (username == NULL  ||  username[0] == '\0')
      return;

#if HAVE_GETPWNAM_R
   if (getpwnam_r( username, &pws, pws_fields, sizeof(pws_fields)-1, &fieldptrs ) != 0)
#else
   fieldptrs = getpwnam( username );
   if (fieldptrs == NULL)
#endif
   {
      ufdbLogError( "Cannot get info on user '%s' so cannot change to this user  *****\n"
      		    "User '%s' probably does not exist.", 
		    username, username );
      return;
   }

   /* 
    * We have already written to the log file which may have been created as user root.
    * and need to change the ownership of the log file to be able to continue logging...
    */
   if (ufdbGlobalLogfile != NULL)
   {
      if (fchown( fileno(ufdbGlobalLogfile), fieldptrs->pw_uid, -1 ))
         { ; }
   }

   if (ufdbGV.debug > 1)
      ufdbLogMessage( "going to call seteuid(%d) and setegid(%d) to set euid to '%s'", 
                      fieldptrs->pw_uid, fieldptrs->pw_gid, username );

   if (setegid( fieldptrs->pw_gid ))
      { ; }

#if HAVE_SETRESUID  && 0
   if (setresuid( fieldptrs->pw_uid, fieldptrs->pw_uid, 0 ) != 0)
#endif

   if (seteuid( fieldptrs->pw_uid ) != 0)
   {
      ufdbLogError( "Cannot drop privileges and become user '%s': %s  *****", username, strerror(errno) );
      if (geteuid() != 0)
	 ufdbLogError( "I am not root so cannot drop/change privileges to user '%s'", username );
      return;
   }
   else
      ufdbLogMessage( "dropped privileges and became user '%s'", username );

#if 0 && defined(__linux__)
   if (ufdbGV.debug || ufdbGV.debugHttpd)
   {
      uid_t rid, eid, sid;
      (void) getresuid( &rid, &eid, &sid );
      ufdbLogMessage( "Privileges are dropped: now running as user '%s'  %d %d %d", username, rid, eid, sid );
      #define FF "/tmp/test.creat"
      unlink( FF );
      creat( FF, 0666 );
      ufdbLogMessage( "created test file %s", FF );
   }
#endif
}


void UFDBraisePrivileges( 
   const char *    username,
   const char *    why )
{
   if (username == NULL  ||  *username == '\0')
      return;

   ufdbLogMessage( "raising privileges to %s", why );

   if (seteuid( 0 ) != 0)	  
      ufdbLogError( "Cannot raise privileges *****" );

#if 0
   {
      uid_t rid, eid, sid;
      (void) getresuid( &rid, &eid, &sid );
      ufdbLogMessage( "Privileges are raised: now running with ids %d %d %d", rid, eid, sid );
      #define FF2 "/tmp/test.raised"
      unlink( FF2 );
      creat( FF2, 0666 );
      ufdbLogMessage( "created test file %s", FF );
   }
#endif
}


/*
 * A bitmap with IP addresses of clients is used to count the number of clients.
 * To keep the bitmap small (2 MB) the first octet of the IP address is ignored.
 *
 * This code assumes that there are 8 bits per byte.
 */

#define N_IPBITMAP   (256U * 256U * 256U)
#define IPBITMAPSIZE (N_IPBITMAP / 8)

static volatile unsigned char * IPbitmap = NULL;
static ufdb_mutex IPbitmapMutex = ufdb_mutex_initializer;

/* We can receive from Squid an IP address (most common) or a FQDN.
 * In case that we receive a FQDN, we calculate a hash and use this 
 * as a psuedo IP address.
 */
void UFDBregisterCountedIP( const char * address )
{
   unsigned char * a;
   unsigned int    i, o1, o2, o3;
   unsigned int    nshift;

   if (IPbitmap == (unsigned char *)NULL)
      UFDBinitializeIPcounters();

   a = (unsigned char *) address;

   /* extremely simple way of looking if the parameter is an IPv4 address... */
   if (*a <= '9' && *a >= '0')
   {
      /* first skip the first octect */
      while (*a != '.'  && *a != '\0')
	 a++;
      if (*a == '.') a++;

      o1 = 0;
      while (*a >= '0' && *a <= '9' && *a != '\0')
      {
	 o1 = o1 * 10U + (*a - '0');
	 a++;
      }
      if (*a == '.') a++;

      o2 = 0;
      while (*a >= '0' && *a <= '9' && *a != '\0')
      {
	 o2 = o2 * 10U + (*a - '0');
	 a++;
      }
      if (*a == '.') a++;

      o3 = 0;
      while (*a >= '0' && *a <= '9' && *a != '\0')
      {
	 o3 = o3 * 10U + (*a - '0');
	 a++;
      }
      o1 = (o1 << 16) + (o2 << 8) + o3;

      /* if we got a non-existent IP address, we might go out of bounds... */
      if (o1 >= N_IPBITMAP)
      {
	 o1 = N_IPBITMAP;
      }
   }
   else 	/* no IPv4 a, calculate a hash */
   {
      o1 = 104729U;
      while (*a != '\0')
      {
         o1 = (o1 << 4) + o1 + (*a - ' ') - 53U;
	 a++;
      }
      o1 = o1 % (256U * 256U * 256U);
   }

   i = o1 / 8U;

#if 0
   nshift = o1 - i * 8U;
#endif
   nshift = o1 & 07;

   /* 
    * To be thread-safe we should use a semaphore here.
    * But since this counter is not critical and a lost bit-set 
    * will probably be covered by another call to this function,
    * we choose for superfast code and skip the use of a semaphore
    * or __sync_fetch_and_or().
    */
   IPbitmap[i] |= (1 << nshift);
}


void UFDBinitializeIPcounters( void )
{
   unsigned int i;
   unsigned char * b;

   if (IPbitmap == NULL)
   {
      ufdb_mutex_lock( &IPbitmapMutex );
      if (IPbitmap == NULL)
         IPbitmap = (unsigned char *) ufdbMallocAligned( 2*1024*1024, IPBITMAPSIZE );   // IPBITMAPSIZE is 2 MB
      ufdb_mutex_unlock( &IPbitmapMutex );
   }

   b = (unsigned char *) IPbitmap;
   for (i = 0;  i < IPBITMAPSIZE;  i++)
      b[i] = 0;
   ufdbGV.lastIPcounterResetDate = time( NULL );
}


unsigned long UFDBgetNumberOfRegisteredIPs( void )
{
   unsigned char * b;
   unsigned long   n;
   unsigned char   v;
   unsigned int    i;

   b = (unsigned char *) IPbitmap;
   if (b == NULL)
      return 0;
   n = 0;
   for (i = 0;  i < IPBITMAPSIZE;  i++)
   {
      v = b[i];
      while (v != 0)
      {
         if (v & 1)
	    n++;
	 v >>= 1;
      }
   }

   return n;
}


/*
 * Usernames are hashed and the a bitmap for hashvalues is administered
 * to keep track of the number of users.
 * To support up to 50,000 users, a bitmap that supports 500,000 users
 * is used to prevent collisions where two or more users are counted as one.
 *
 * This code assumes that there are 8 bits per byte.
 */

/* USERBITMAPSIZE is 64KB for 512*1024 hashvalues */
#define N_USERBITMAP	(512 * 1024)
#define USERBITMAPSIZE  (N_USERBITMAP / 8)

static volatile unsigned char * UserBitmap = NULL;
static ufdb_mutex UserBitmapMutex = ufdb_mutex_initializer;

/* We can receive from Squid a username or '-'.
 * In case that we receive a username, we calculate a hashvalue and use this 
 * as an index in UserBitmap to set a bit to 1.
 */
void UFDBregisterCountedUser( const char * username )
{
   unsigned char * a;
   unsigned int    i, o1;
   unsigned int    nshift;

   if (UserBitmap == (unsigned char *)NULL)
      UFDBinitializeUserCounters();

   a = (unsigned char *) username;

   o1 = 104729U;
   while (*a != '\0')
   {
      //   ---- o1 * 17 ----
      o1 = (((o1 << 4) + o1) ^ (o1>>21))  +  *a - ' ' - 53U;
      a++;
   }
   o1 = o1 % N_USERBITMAP;

   i = o1 / 8U;

#if 0
   nshift = o1 - i * 8U;
#endif
   nshift = o1 & 07;

   /* 
    * To be thread-safe we should use a semaphore here.
    * But since this counter is not critical and a lost bit-set 
    * will probably be covered by another call to this function,
    * we choose for superfast code and skip the use of a semaphore
    * or __sync_fetch_and_or().
    */
   UserBitmap[i] |= (1 << nshift);
}


void UFDBinitializeUserCounters( void )
{
   int i;
   unsigned char * b;

   if (UserBitmap == (unsigned char *)NULL)
   {
      ufdb_mutex_lock( &UserBitmapMutex );
      if (UserBitmap == (unsigned char *)NULL)
         UserBitmap = (unsigned char *) ufdbMallocAligned( 4096, USERBITMAPSIZE );
      ufdb_mutex_unlock( &UserBitmapMutex );
   }

   b = (unsigned char *) UserBitmap;
   for (i = 0;  i < USERBITMAPSIZE;  i++)
      b[i] = 0;
}


unsigned long UFDBgetNumberOfRegisteredUsers( void )
{
   unsigned char * b;
   unsigned long   n;
   unsigned char   v;
   int             i;

   b = (unsigned char *) UserBitmap;
   if (b == NULL)
      return 0;
   n = 0;
   for (i = 0;  i < USERBITMAPSIZE;  i++)
   {
      v = b[i];
      while (v != 0)
      {
         if (v & 1)
	    n++;
	 v >>= 1;
      }
   }

   return n;
}


static void ufdbSendEmailToAdmin( int newStatus )
{
   int       mx;
   int       n;
   time_t    now_t;
   struct tm t;
   struct    timeval      tv;
   char      hostname[256+1];
   char      line[2048];

   if (ufdbGV.emailServer == NULL  ||  ufdbGV.adminEmail == NULL)
      return;

   if (ufdbGV.myHostname != NULL)
      strcpy( hostname, ufdbGV.myHostname );
   else
   {
      gethostname( hostname, sizeof(hostname) );
      hostname[256] = '\0';
   }

   mx = UFDBopenSocket( ufdbGV.emailServer, 25 );
   if (mx < 0)
   {
      ufdbLogError( "cannot open connection to mail server '%s' to inform status change: %s", 
                    ufdbGV.emailServer, strerror(errno) );
      return;
   }

   /*
    *  Prevent that the read() takes ages.  Use an agressive timeout.
    */
   tv.tv_sec = 6;
   tv.tv_usec = 0;
   setsockopt( mx, SOL_SOCKET, SO_RCVTIMEO, (void *) &tv, sizeof(tv) );
   tv.tv_sec = 5;
   tv.tv_usec = 0;
   setsockopt( mx, SOL_SOCKET, SO_SNDTIMEO, (void *) &tv, sizeof(tv) );

   if (ufdbGV.debug)
      ufdbLogMessage( "   mail: opened socket 25 to mail server '%s'", ufdbGV.emailServer );

   do {
      n = read( mx, line, 2048 );
   } while (n < 0  &&  errno == EINTR);
   if (n < 4  ||  strncmp( line, "220", 3 ) != 0)
   {
      ufdbLogError( "ufdbSendEmailToAdmin: mail server %s did not send 220 welcome message: %s",
      		    ufdbGV.emailServer, strerror(errno) );
      close( mx );
      return;
   }
   if (ufdbGV.debug > 1)
      ufdbLogMessage( "   mail: read welcome lines from mail server '%s'", ufdbGV.emailServer );

#if 0
   /* The welcome message may be long and sent in two chunks...
    * There is a 2 second timeout so no worries if there is no more input.
    */
   tv.tv_sec = 2;
   tv.tv_usec = 0;
   setsockopt( mx, SOL_SOCKET, SO_RCVTIMEO, (void *) &tv, sizeof(tv) );
   do {
      n = read( mx, line, 2048 );
   } while (n < 0  &&  errno == EINTR);
#endif

   /*
    * From now on we wait maximum 20 seconds for responses of the mail server
    */
   tv.tv_sec = 20;
   tv.tv_usec = 0;
   setsockopt( mx, SOL_SOCKET, SO_RCVTIMEO, (void *) &tv, sizeof(tv) );
   
   sprintf( line, "HELO %s\r\n", hostname );
   n = write( mx, line, strlen(line) );
   if (n != (int) strlen(line))
   {
      ufdbLogError( "ufdbSendEmailToAdmin: failed to send HELO to mail server '%s': %s", 
                    ufdbGV.emailServer, strerror(errno) );
      close( mx );
      return;
   }
   do {
      n = read( mx, line, 2047 );
   } while (n < 0  &&  errno == EINTR);
   if (n < 1)
   {
      ufdbLogError( "ufdbSendEmailToAdmin: mail server '%s' did not send reply to HELO: %s", 
      		    ufdbGV.emailServer, strerror(errno) );
      close( mx );
      return;
   }

   if (ufdbGV.debug > 1)
   {
      line[n] = '\0';
      ufdbLogMessage( "   mail: done with HELO handshake with mail server '%s'\nreply is %s", ufdbGV.emailServer, line );
   }
   
   sprintf( line, "MAIL FROM:<%s>\r\n", ufdbGV.senderEmail != NULL ? ufdbGV.senderEmail : ufdbGV.adminEmail );
   n = write( mx, line, strlen(line) );
   if (n != (int) strlen(line))
   {
      ufdbLogError( "ufdbSendEmailToAdmin: failed to send MAIL FROM to mail server '%s': %s", 
                    ufdbGV.emailServer, strerror(errno) );
      close( mx );
      return;
   }
   do {
      n = read( mx, line, 2048 );
   } while (n < 0  &&  errno == EINTR);
   if (n < 1)
   {
      ufdbLogError( "ufdbSendEmailToAdmin: mail server '%s' did not send reply to MAIL FROM: %s", 
      		    ufdbGV.emailServer, strerror(errno) );
      close( mx );
      return;
   }
   if (strncmp( line, "25", 2 ) != 0)
   {
      char * newline;

      newline = strchr( line, '\r' );
      if (newline == NULL)
	 newline = strchr( line, '\n' );
      if (newline != NULL)
         *newline ='\0';
      ufdbLogError( "ufdbSendEmailToAdmin: mail server '%s' did not send reply with OK to MAIL FROM:\n%s", 
      		    ufdbGV.emailServer, line );
      close( mx );
      return;
   }
   
   sprintf( line, "RCPT TO:<%s>\r\n", ufdbGV.adminEmail );
   n = write( mx, line, strlen(line) );
   if (n != (int) strlen(line))
   {
      ufdbLogError( "ufdbSendEmailToAdmin: failed to send RCPT TO to mail server '%s': %s", 
                    ufdbGV.emailServer, strerror(errno) );
      close( mx );
      return;
   }
   do {
      n = read( mx, line, 2048 );
   } while (n < 0  &&  errno == EINTR);
   if (n < 1)
   {
      ufdbLogError( "ufdbSendEmailToAdmin: mail server '%s' did not send reply to RCPT TO: %s", 
      		    ufdbGV.emailServer, strerror(errno) );
      close( mx );
      return;
   }
   if (strncmp( line, "25", 2 ) != 0)
   {
      char * newline;

      newline = strchr( line, '\r' );
      if (newline == NULL)
	 newline = strchr( line, '\n' );
      if (newline != NULL)
         *newline ='\0';
      ufdbLogError( "ufdbSendEmailToAdmin: mail server '%s' did not send reply with OK to RCPT TO:\n%s", 
      		    ufdbGV.emailServer, line );
      close( mx );
      return;
   }
   
   /* To support mail servers with plain authentication:
    *
    * According to RFC 2595 the client must send: [authorize-id] \0 authenticate-id \0 password. 
    * pwcheck_method has been set to sasldb for the following example.
    * 
    * >>> AUTH PLAIN dGVzdAB0ZXN0QHdpei5leGFtcGxlLmNvbQB0RXN0NDI=
    * 235 2.0.0 OK Authenticated
    * Decoded:
    * test\000test@wiz.example.com\000tEst42
    * With patch for lib/checkpw.c or a pwcheck_method that doesn't support realms:
    * >>> AUTH PLAIN dGVzdAB0ZXN0AHRFc3Q0Mg==
    * Decoded:
    * test\000test\000tEst42
    */

   if (ufdbGV.debug > 1)
      ufdbLogMessage( "   mail: sending DATA to mail server '%s'", ufdbGV.emailServer );
   
   sprintf( line, "DATA\r\n" );
   n = write( mx, line, strlen(line) );
   if (n != (int) strlen(line))
   {
      ufdbLogError( "ufdbSendEmailToAdmin: failed to send DATA to mail server '%s': %s", 
                    ufdbGV.emailServer, strerror(errno) );
      close( mx );
      return;
   }
   do {
      n = read( mx, line, 2048 );
   } while (n < 0  &&  errno == EINTR);
   if (n < 1)
   {
      ufdbLogError( "ufdbSendEmailToAdmin: mail server '%s' did not send reply to DATA: %s", 
      		    ufdbGV.emailServer, strerror(errno) );
      close( mx );
      return;
   }
   if (strncmp( line, "354", 3 ) != 0)
   {
      char * newline;

      newline = strchr( line, '\r' );
      if (newline == NULL)
	 newline = strchr( line, '\n' );
      if (newline != NULL)
         *newline ='\0';
      ufdbLogError( "ufdbSendEmailToAdmin: mail server '%s' did not send reply with OK to DATA:\n%s", 
      		    ufdbGV.emailServer, line );
      close( mx );
      return;
   }

   now_t = time( NULL );
   gmtime_r( &now_t, &t );

   sprintf( line, "From: ufdbGuard daemon <%s>\r\n"
		  "To: URL filter administrator <%s>\r\n"
                  "Subject: ufdbGuard on %s has a new status: %s\r\n"
		  "Date: %3.3s, %02d %3.3s %4d %02d:%02d:%02d GMT\r\n"
		  "X-Mailer: ufdbguardd " UFDB_VERSION
		  "\r\n"
		  "ufdbGuard with pid %d on %s has a new status: %s\n"
		  "database status: %s\n"
		  "license status: %s\n"
		  "configuration file: %s\n"
		  "version: " UFDB_VERSION "\n"
		  "\r\n.\r\n",
		  ufdbGV.senderEmail != NULL ? ufdbGV.senderEmail : ufdbGV.adminEmail,
		  ufdbGV.adminEmail,
		  hostname, ufdbStatus2string(newStatus),
		     &"SunMonTueWedThuFriSat"[t.tm_wday*3],
		     t.tm_mday,
		     &"JanFebMarAprMayJunJulAugSepOctNovDec"[t.tm_mon*3],
		     t.tm_year + 1900,
		     t.tm_hour, t.tm_min, t.tm_sec,
		  ufdbGV.pid, hostname, ufdbStatus2string(newStatus),
		  ufdbDBstat2string(ufdbGV.databaseStatus),
		  ufdbGV.licenseStatus,
		  ufdbGV.configFile
	  );
   n = write( mx, line, strlen(line) );
   if (n != (int) strlen(line))
   {
      ufdbLogError( "ufdbSendEmailToAdmin: failed to send CONTENT to mail server '%s': %s", 
                    ufdbGV.emailServer, strerror(errno) );
      close( mx );
      return;
   }
   do {
      n = read( mx, line, 2048 );
   } while (n < 0  &&  errno == EINTR);
   if (n < 1)
   {
      ufdbLogError( "ufdbSendEmailToAdmin: mail server '%s' did not send reply to DATA END: %s", 
      		    ufdbGV.emailServer, strerror(errno) );
      close( mx );
      return;
   }
   if (strncmp( line, "25", 2 ) != 0)
   {
      char * newline;

      newline = strchr( line, '\r' );
      if (newline == NULL)
	 newline = strchr( line, '\n' );
      if (newline != NULL)
         *newline ='\0';
      ufdbLogError( "ufdbSendEmailToAdmin: mail server '%s' did not send reply with OK to DATA END:\n%s", 
      		    ufdbGV.emailServer, line );
      close( mx );
      return;
   }

   if (ufdbGV.debug > 1)
      ufdbLogMessage( "   mail: sending QUIT to %s", ufdbGV.emailServer );
   
   sprintf( line, "QUIT\r\n" );
   n = write( mx, line, strlen(line) );
   if (n != (int) strlen(line))
   {
      ufdbLogError( "ufdbSendEmailToAdmin: failed to send QUIT to mail server '%s': %s", 
                    ufdbGV.emailServer, strerror(errno) );
      close( mx );
      return;
   }
#if 0
   n = read( mx, line, 2048 );
   /* ignore errors */
#endif
   close( mx );

   ufdbLogMessage( "sent email with status update to '%s' using mail server '%s'", ufdbGV.adminEmail, ufdbGV.emailServer );
}


static void ufdbExecuteExternalCommand( int status )
{
   pid_t pid;

   if (ufdbGV.externalStatusCommand == NULL)
      return;

   ufdbLogMessage( "going to execute: %s -s %s -d %s -l %s -v %s",
                   ufdbGV.externalStatusCommand, 
		   ufdbStatus2string(status), 
		   ufdbDBstat2string(ufdbGV.databaseStatus),
		   ufdbGV.licenseStatus,
		   UFDB_VERSION  );

   pid = fork();
   if (pid == 0)	/* we are the forked child */
   {
      int    i;
      const char * argv[12];

      for (i = 0; i < 2*UFDB_MAX_THREADS + 32;  i++)
         close( i );
      i = 0;
      argv[i++] = ufdbGV.externalStatusCommand;
      argv[i++] = "-s";
      argv[i++] = ufdbStatus2string( status );
      argv[i++] = "-d";
      argv[i++] = ufdbDBstat2string( ufdbGV.databaseStatus );
      argv[i++] = "-l";
      argv[i++] = ufdbGV.licenseStatus;
      argv[i++] = "-v";
      argv[i++] = UFDB_VERSION;
      argv[i] = NULL;
      execv( ufdbGV.externalStatusCommand, (char* const*) argv );
      _exit( 2 );       /* exit after failed execve */
   }
   else
   {
      if (ufdbGV.debug > 1)
         ufdbLogMessage( "   %s has pid %ld", ufdbGV.externalStatusCommand, (long) pid );
   }

   if (pid < 0)
   {
      ufdbLogError( "fork failed: cannot execute external status command '%s': %s", 
                    ufdbGV.externalStatusCommand, strerror(errno) );
      return;
   }

   /* we could do a waitpid() here but there is an other thread that does that */
}


int UFDBchangeStatus( int status )
{
   int    oldStatus;
   char * dbhome;
   FILE * fp;
   char   licStatFileName[1024];

   if (status == ufdbGV.status)
      return status;

   ufdbLogMessage( "Changing daemon status to \"%s\"", ufdbStatus2string(status) );

   /* prevent changing the ERROR state into an OK state */
   if (ufdbGV.status == UFDB_API_STATUS_FATAL_ERROR)
   {
      if (status == UFDB_API_STATUS_RELOAD_OK)
         return UFDB_API_STATUS_FATAL_ERROR;
   }

   /* update the license status */
   dbhome = ufdbGV.databaseDirectory;
   strcpy( licStatFileName, dbhome );
   strcat( licStatFileName, "/" );
   strcat( licStatFileName, UFDB_LICENSE_STATUS_FILENAME );
   fp = fopen( licStatFileName, "r" );
   if (fp != NULL)
   {
      char * newline;
      if (fgets( ufdbGV.licenseStatus, sizeof(ufdbGV.licenseStatus)-1, fp ))
         { ; }
      fclose( fp );
      if ((newline = strchr( ufdbGV.licenseStatus, '\n' )) != NULL)
         *newline = '\0';
      if (ufdbGV.licenseStatus[0] == '\0')
	 strcpy( ufdbGV.licenseStatus, "unknown" );
   }
   else
      strcpy( ufdbGV.licenseStatus, "unknown" );

   ufdbSendEmailToAdmin( status );
   ufdbExecuteExternalCommand( status );

   oldStatus = ufdbGV.status;
   ufdbGV.status = status;

   if (ufdbGV.debug)
      ufdbLogMessage( "UFDBchangeStatus done" );
   
   return oldStatus;
}


static char * my_fast_fgets_nonl( char * s, int size, FILE * stream )
{
   char * buf;
   int    ch;

   buf = s;
   while ((ch = getc_unlocked(stream)) != EOF  &&  --size > 0)
   {
      if (ch == '\n'  ||  ch == '\r')
         break;
      *buf++ = ch;
   }
   *buf = '\0';
   if (ch == EOF  &&  buf == s)
      return NULL;
   return s;
}


static void interruptPstack( int dummy )
{
   if (dummy)           // prevent compiler warning
      { ; }

   ufdbLogError( "waited too long for output of pstack.... aborting now!" );
   UFDBchangeStatus( UFDB_API_STATUS_FATAL_ERROR );
   _exit( 5 );  /* pstack took too long; exit now */
}


void ufdbExecutePstack( const char * reason )
{
   FILE * fp;
   char   cmd[512];
   char   buffer[2048];

#if 0
   /* popen causes SIGCHLD signals but we want to ignore them */
   ufdbSetSignalHandler( SIGCHLD, SIG_IGN );
#endif

   ufdbSetSignalHandler( SIGALRM, interruptPstack );
   alarm( 15 );

   if (ufdbGV.userName[0] != '\0')
      sprintf( cmd, DEFAULT_BINDIR "/ufdb-pstack %d -U %s %s 2>&1", ufdbGV.pid, ufdbGV.userName, reason );
   else
      sprintf( cmd, DEFAULT_BINDIR "/ufdb-pstack %d %s 2>&1", ufdbGV.pid, reason );

   /* gdb sometimes needs root permissions, so the popen() is done as root */
   UFDBraisePrivileges( ufdbGV.userName, "open pipe for ufdb-pstack" );

#if HAVE_PR_SET_TRACER
   // allow gdb process to attach to this process:
   if (prctl( PR_SET_PTRACER, PR_SET_PTRACER_ANY, 0, 0, 0 ) != 0)
      { ; }
#endif

   errno = 0;
   fp = popen( cmd, "r" );
   if (fp == NULL)
      ufdbLogError( "could not open pipe and execute \"%s\": %s", cmd, strerror(errno) );
   UFDBdropPrivileges( ufdbGV.userName );

   while (my_fast_fgets_nonl( buffer, 2040, fp ) != NULL)
   {
      ufdbLogMessage( "pstack  %s", buffer );
   }
   pclose( fp );
   ufdbLogMessage( "pstack  END" );
   alarm( 0 );
}


#ifdef __cplusplus
}
#endif
