/*
 * ufdbbase.c - URLfilterDB
 *
 * ufdbGuard is copyrighted (C) 2005-2021 by URLfilterDB with all rights reserved.
 *
 * Parts of ufdbGuard are based on squidGuard.
 * This module is NOT based on squidGuard.
 * 
 * RCS $Id: ufdbbase.c,v 1.142 2022/12/26 23:13:30 root Exp root $
 */

#undef _FORTIFY_SOURCE

#define UFDB_MALLOC_DEBUG     	 0


#include "ufdb.h"
#include "ufdblib.h"

#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#if UFDB_DPDK_SUPPORT
#include "rte_malloc.h"
#endif
#if HAVE_MEMALIGN
#include <malloc.h>
#endif
#include <ctype.h>
#include <signal.h>
#include <errno.h>
#if HAVE_SYS_SYSCALL_H
#include <sys/syscall.h>
#endif
#include <sys/time.h>
#include <sys/types.h>

#if UFDB_DP_DEV
#include "cvmx.h"
#include "cvmx-bootmem.h"
#include "cvmx-malloc.h"
#endif

#if !UFDB_BARE_METAL_SUPPORT
#include <sys/socket.h>
#include <sys/stat.h>
#include <pwd.h>
#endif

#if HAVE_PCRE_COMPILE
#include <pcre.h>
#endif

#if HAVE_PR_SET_TRACER
#include <sys/prctl.h>
#endif

#if !defined(UFDB_API_NO_THREADS) && UFDB_PTHREAD_SUPPORT
#include <pthread.h>
#endif


#if UFDB_DP_DEV
#include "dpmalloc.h"
void UFDBmallocInit( ufdb_config_t * config )
{
   ufdbMallocInitDPdevelopment();
}

#elif UFDB_BARE_METAL_SUPPORT && __OCTEON__

#include "dpmalloc.h"
UFDB_SHARED static ufdb_malloc_t malloc_wrapper = NULL;
UFDB_SHARED static ufdb_free_t   free_wrapper = NULL;

void UFDBmallocInit( ufdb_config_t * config )
{
   if (config->ufdb_alloc == NULL)
      ufdbLogFatalError( "UFDBmallocInit: ufdb_alloc is NULL" );
   else
      malloc_wrapper = config->ufdb_alloc;

   if (config->ufdb_free == NULL)
      ufdbLogFatalError( "UFDBmallocInit: ufdb_free is NULL" );
   else
      free_wrapper = config->ufdb_free;

   if (ufdbGV.debug)
      ufdbLogMessage( "UFDBmallocInit:  wrapper functions  malloc %p  free %p", malloc_wrapper, free_wrapper );
}
#endif


#ifdef __cplusplus
extern "C" {
#endif

#if HAVE_SETRESUID
int setresuid( uid_t ruid, uid_t euid, uid_t suid );
int getresuid( uid_t * ruid, uid_t * euid, uid_t * suid );
#endif


UFDB_SHARED static unsigned long cpu_mask = 0UL;
#define UFDB_MAX_CPUS 64


#if defined(__GLIBC__)
#if (__GLIBC__ > 2) || (__GLIBC__ == 2  &&  __GLIBC_MINOR__ >= 4) || HAVE_PCRE_COMPILE
#define NEED_REGEXEC_MUTEX 0
#else
#define NEED_REGEXEC_MUTEX 1
#endif
#else
#define NEED_REGEXEC_MUTEX 0
#endif

#if UFDB_REGEX_SUPPORT && NEED_REGEXEC_MUTEX
ufdb_mutex ufdb_regexec_mutex = ufdb_mutex_initializer;
#endif

#ifdef UFDB_DEBUG
ufdb_mutex ufdb_malloc_mutex = ufdb_mutex_initializer;
#endif


#if UFDB_DP_DEV
UFDB_SHARED static cvmx_arena_list_t alist = NULL;

void ufdbMallocInitDPdevelopment( void )
{
#define MYMEMSIZE (700 * 1024 * 1024)

   // allocate an arena and assign it to memory allocator
   void * mm;

   mm = cvmx_bootmem_alloc( MYMEMSIZE, 4096 );
   if (mm == NULL)
   {
      ufdbLogFatalError( "initDP: cvmx_bootmem_alloc failed" );
      exit( 1 );
   }

   if (cvmx_add_arena( &alist, mm, MYMEMSIZE ) != 0)
   {
      ufdbLogFatalError( "initDP: cvmx_add_arena failed" );
      exit( 1 );
   }
}
#endif


UFDB_GCC_MALLOC_ATTR
void * ufdbMalloc( size_t elsize )
{
   void * p;

#if UFDB_MALLOC_DEBUG
   if (ufdbGV.debug)
      ufdbLogMessage( "      ufdbMalloc %'ld", (long) elsize );
#endif

#if UFDB_DPDK_SUPPORT
   p = rte_malloc( NULL, elsize, 0 );
#elif UFDB_DP_DEV
   p = cvmx_malloc( alist, elsize );
#elif UFDB_BARE_METAL_SUPPORT && __OCTEON__
   p = malloc_wrapper( (uint32_t) elsize );
   if (ufdbGV.debug)
      ufdbLogMessage( "ufdbMalloc: malloc_wrapper(%ld) -> %p", (long)elsize, p );
#else
   p = (void *) malloc( elsize );
#endif
   if (p == NULL  &&  elsize > 0)
   {
      ufdbGV.memoryAllocationErrors++;
      ufdbLogFatalError( "cannot allocate %'ld bytes memory", (long) elsize );
   }
   else
   {
      ufdbGV.nmalloc++;
      // UFDB_API_num_bytes must be reset to zero at each reload
      // UFDB_API_num_bytes += elsize;
   }

#if UFDB_MALLOC_DEBUG
   if (ufdbGV.debug)
      ufdbLogMessage( "         ufdbMalloc %'ld -> 0x%08lx", (long) elsize, (long) p );
#endif

   return p;
}


#define UFDB_HUGEPAGESIZE (2*1024*1024)

void * UFDB_GCC_MALLOC_ATTR ufdbMallocAligned( size_t alignment, size_t elsize )
{
   void * p;

#if UFDB_MALLOC_DEBUG
   if (ufdbGV.debug)
      ufdbLogMessage( "      ufdbMallocAligned %'ld, %'ld", (long) alignment, (long) elsize );
#endif

   if (alignment < UFDB_HUGEPAGESIZE  &&  (elsize > ((size_t) (UFDB_HUGEPAGESIZE * 0.9))))
      alignment = UFDB_HUGEPAGESIZE;                 // the kernel may use a hugepage
   else if (alignment < 4096  &&  elsize > 3600)
      alignment = 4096;
   else
   {
      alignment = (alignment + (UFDB_CACHELINE_SIZE-1)) & ~(UFDB_CACHELINE_SIZE-1);
   }

#if UFDB_DPDK_SUPPORT
   p = rte_malloc( NULL, elsize, (unsigned int) alignment );
#elif UFDB_DP_DEV
   p = cvmx_memalign( alist, alignment, elsize );
#elif UFDB_BARE_METAL_SUPPORT && __OCTEON__
   if (ufdbGV.debug)
      ufdbLogMessage( "ufdbMallocAligned: going to call malloc_wrapper(%ld) wrapper=%p",
                      (long)elsize, malloc_wrapper );
   p = malloc_wrapper( (uint32_t) elsize );
   if (ufdbGV.debug)
      ufdbLogMessage( "ufdbMallocAligned: malloc_wrapper(%ld) -> %p", (long)elsize, p );
#else

   int    retval;
#if HAVE_POSIX_MEMALIGN
   p = NULL;
   retval = posix_memalign( &p, alignment, elsize );
#elif HAVE_MEMALIGN
   retval = 0;
   p = memalign( alignment, elsize );
#elif HAVE_ALIGNED_ALLOC
   retval = 0;
   p = aligned_alloc( alignment, elsize );
#else
   retval = 0;
   p = malloc( elsize );
#endif

#if UFDB_MALLOC_DEBUG
   if (ufdbGV.debug)
      ufdbLogMessage( "         ufdbMallocAligned(%ld):%d %'ld -> %p",
		      (long)alignment, retval, (long)elsize, p );
#endif

   if (retval == 0  &&  p != NULL)
      return p;

   p = (void *) ufdbMalloc( elsize );
#endif

   if (p == NULL  &&  elsize > 0)
   {
      ufdbGV.memoryAllocationErrors++;
      ufdbLogFatalError( "cannot allocate %'ld bytes %'ld-aligned memory", 
                         (long)elsize, (long)alignment );
   }
   else
      ufdbGV.nmalloc++;

#if UFDB_MALLOC_DEBUG
   if (ufdbGV.debug)
      ufdbLogMessage( "         ufdbMallocAligned not-aligned %'ld -> %p", (long)elsize, p );
#endif

   return p;
}


void * ufdbCalloc( size_t n, size_t num )
{
   void * p;

#if UFDB_MALLOC_DEBUG
   if (ufdbGV.debug)
      ufdbLogMessage( "      ufdbCalloc %ld %'ld", (long) n, (long) num );
#endif

   p = (void *) ufdbMalloc( n * num );
   if (p == NULL  &&  n > 0)
      ufdbLogFatalError( "cannot allocate %ld bytes zeroed memory", (long) (n*num) );
   else
      memset( p, 0, n * num );

#if UFDB_MALLOC_DEBUG
   if (ufdbGV.debug)
      ufdbLogMessage( "         ufdbCalloc %ld %ld  = %'ld  -> 0x%08lx", 
                      (long) n, (long) num, (long) n*num, (long) p );
#endif

   return p;
}


// realloc is only used to parse 1.2 tables and in ufdbGenTable and may be problematic on bare metal
#if UFDB_DP_DEV || !UFDB_BARE_METAL_SUPPORT
void * ufdbRealloc( void * ptr, size_t elsize )
{
   void * p;

#if UFDB_MALLOC_DEBUG
   if (ufdbGV.debug)
      ufdbLogMessage( "      ufdbRealloc 0x%08lx %'ld", (long) ptr, (long) elsize );
#endif

#if UFDB_DPDK_SUPPORT
   p = rte_realloc( ptr, (unsigned int) elsize, 0 );
#elif UFDB_DP_DEV
   ufdbLogFatalError( "ufdbRealloc: realloc is not implemented on OCTEON" );
   p = NULL;   // cvmx_realloc( alist, ptr, elsize );
#elif UFDB_BARE_METAL_SUPPORT && __OCTEON__
   ufdbLogFatalError( "ufdbRealloc: realloc is not implemented on OCTEON" );
   p = NULL;
#else
   p = (void *) realloc( ptr, elsize );
#endif
   if (p == NULL  &&  elsize > 0)
   {
      ufdbGV.memoryAllocationErrors++;
      ufdbLogFatalError( "cannot reallocate %'ld bytes memory", (long) elsize );
   }
   else
      ufdbGV.nrealloc++;

#if UFDB_MALLOC_DEBUG
   if (ufdbGV.debug)
      ufdbLogMessage( "         ufdbRealloc 0x%08lx %'ld  ->  %08lx", (long) ptr, (long) elsize, (long) p );
#endif

   return p;
}
#endif


void ufdbFree( void * ptr )
{
#if UFDB_MALLOC_DEBUG
   if (ufdbGV.debug)
      ufdbLogMessage( "      ufdbFree 0x%08lx", (long) ptr );
#endif

   if (ptr != NULL)
   {
#if UFDB_DPDK_SUPPORT
      rte_free( ptr );
#elif UFDB_DP_DEV
      cvmx_free( ptr );
#elif UFDB_BARE_METAL_SUPPORT && __OCTEON__
      if (ufdbGV.debug)
	 ufdbLogMessage( "ufdbFree: going to call free_wrapper(%p)", ptr );
      free_wrapper( ptr );
#else
      free( ptr );
#endif
      ufdbGV.nfree++;

#if UFDB_MALLOC_DEBUG
      if (ufdbGV.debug)
	 ufdbLogMessage( "      ufdbFree 0x%08lx freed", (long) ptr );
#endif
   }
}


void * ufdbZlibMalloc( UFDB_GCC_UNUSED void * opaque, unsigned int items, unsigned int size )
{
#if UFDB_BARE_METAL_SUPPORT && __OCTEON__
   if (ufdbGV.debug)
      ufdbLogMessage( "ufdbZlibMalloc: calling ufdbMalloc" );
#endif
   return ufdbMalloc( (size_t) items * size );
}


void ufdbZlibFree( UFDB_GCC_UNUSED void * opaque, void * address )
{
#if UFDB_BARE_METAL_SUPPORT && __OCTEON__
   if (ufdbGV.debug)
      ufdbLogMessage( "ufdbZlibFree: calling ufdbFree" );
#endif
   return ufdbFree( address );
}


char * ufdbStrdup( const char * s )
{
   int size;

#if defined(UFDB_DEBUG)
   ufdbLogMessage( "   ufdbStrdup %-60.60s", s );
#endif

   size = strlen( s ) + 1;
   return strcpy( (char *) ufdbMalloc(size), s );
}


int ufdbStrStrEnd( const char * s, const char * end )
{
   int n;

   n = strlen( end );
   while (s != NULL  &&  *s != '\0')
   {
      s = strstr( s, end );
      if (s != NULL)
      {
         if (*(s+n) == '\0')
	    return 1;
	 else
	    s++;
      }
   }
   return 0;
}


char * UFDBfgets( 
   char * requestBuffer, 
   int    bufferSize, 
   FILE * fp )
{
   char * b;
   int    ch;
   int    size;

   b = requestBuffer;
   size = 1;

   while ((ch = getc_unlocked(fp)) != EOF)
   {
      *b++ = ch;
      if (ch == '\n')
         goto end;
      if (++size == bufferSize)
         goto end;
   }

   if (b == requestBuffer  &&  (feof(fp) || ferror(fp)))
      return NULL;

end:
   *b = '\0';
   return requestBuffer;
}


char * UFDBfgetsNoNL( char * s, int size, FILE * stream )
{
   char * buf;
   int    ch;

   buf = s;
   while ((ch = getc_unlocked(stream)) != EOF  &&  --size > 0)
   {
      if (ch == '\r')
         continue;
      if (ch == '\n')
         break;
      *buf++ = ch;
   }
   *buf = '\0';
   if (ch == EOF  &&  buf == s)
      return NULL;

   /* remove trailing spaces */
   while (buf > s)
   {
      if (*(buf-1) == ' ')
      {
         buf--;
         *buf = '\0';
      }
      else
         break;
   }

   return s;
}


#if UFDB_REGEX_SUPPORT
static int UFDBregcomp( void * preg, const char * regex, int cflags )
{
   int     retval;
#if (HAVE_PCRE_COMPILE || HAVE_PCRE_COMPILE2)
   int     pcre_options;
   int     pcre_error;
   const char * pcre_errstr;
   int     pcre_erroffset;
   pcre ** ppreg;
   pcre *  pcre_comp_re;
#endif

#if HAVE_PCRE_COMPILE2
   pcre_options = PCRE_UTF8;
   if (cflags & REG_ICASE)
      pcre_options |= PCRE_CASELESS;
   pcre_comp_re = pcre_compile2( regex, pcre_options, &pcre_error, &pcre_errstr, &pcre_erroffset, NULL );
   retval = 0;
   if (pcre_comp_re == NULL)
   {
      if (ufdbGV.debug || ufdbGV.debugRegexp)
         ufdbLogError( "pcre_compile2: error with regular expression \"%s\": %s", regex, pcre_errstr );
      retval = pcre_error;
   }
   else
   {
      ppreg = (pcre**) preg;
      *ppreg = pcre_comp_re;
   }
#elif HAVE_PCRE_COMPILE
   pcre_options = PCRE_UTF8;
   if (cflags & REG_ICASE)
      pcre_options |= PCRE_CASELESS;
   pcre_comp_re = pcre_compile( regex, pcre_options, &pcre_errstr, &pcre_errofsset, NULL );
   retval = 0;
   if (pcre_comp_re == NULL)
   {
      if (ufdbGV.debug || ufdbGV.debugRegexp)
         ufdbLogError( "pcre_compile: error with regular expression \"%s\": %s", regex, pcre_errstr );
      retval = REG_ECOLLATE;
   }
   else
   {
      ppreg = (pcre**) preg;
      *ppreg = pcre_comp_re;
   }
#else
   retval = regcomp( (regex_t*) preg, regex, cflags );
#endif

   return retval;
}


UFDB_GCC_INLINE
static void UFDBregfree( void * preg )
{
#if HAVE_PCRE_COMPILE2 || HAVE_PCRE_COMPILE
   pcre_free( (pcre*) preg );
#else
   regfree( (regex_t*) preg );
#endif
}


/* TO-DO: make a macro for UFDBregexec() */
UFDB_GCC_HOT
int UFDBregexec( const void * preg, const char * string, size_t nmatch, regmatch_t pmatch[], int eflags )
{
   int retval;
#if HAVE_PCRE_COMPILE2 || HAVE_PCRE_COMPILE
   int pcre_ovector[2*3];
   int pcre_ovecsize;
#endif

#if NEED_REGEXEC_MUTEX
   ufdb_mutex_lock( &ufdb_regexec_mutex );
#endif

#if (HAVE_PCRE_COMPILE2 || HAVE_PCRE_COMPILE)
   pcre_ovecsize = nmatch * 3;
   if (pcre_ovecsize > (int) sizeof(pcre_ovector))
      pcre_ovecsize = sizeof(pcre_ovector);
   retval = pcre_exec( (pcre*) preg, NULL, string, strlen(string), 0,
                       PCRE_NO_UTF8_CHECK, pcre_ovector, pcre_ovecsize );
   if (retval == PCRE_ERROR_NOMATCH)
      retval = REG_NOMATCH;
   else if (nmatch > 0)
   {
      /* convert pcre ovector to regmatch_t */
      pmatch[0].rm_so = pcre_ovector[0];
      pmatch[0].rm_eo = pcre_ovector[1];
      if (nmatch > 1)
      {
         pmatch[1].rm_so = pcre_ovector[2];
         pmatch[1].rm_eo = pcre_ovector[3];
      }
   }
#else
   retval = regexec( (regex_t*) preg, string, nmatch, pmatch, eflags );
#endif

#if NEED_REGEXEC_MUTEX
   ufdb_mutex_unlock( &ufdb_regexec_mutex );
#endif

   return retval;
}


struct ufdbRegExp * ufdbNewPatternBuffer( 
   char * pattern, 
   int    flags )
{
   struct ufdbRegExp * re;
#if !(HAVE_PCRE_COMPILE2  ||  HAVE_PCRE_COMPILE)
   int i;
#endif

   re = (struct ufdbRegExp *) ufdbMallocAligned( 64, sizeof(struct ufdbRegExp) );

#if !(HAVE_PCRE_COMPILE2 || HAVE_PCRE_COMPILE)  &&  !defined(UFDB_API_NO_THREADS)
   re->next_nregex_i = 0;
   ufdb_mutex_init( &(re->lock) );
#endif

#if HAVE_PCRE_COMPILE2  ||  HAVE_PCRE_COMPILE
   re->compiled[0] = NULL;
   re->error = UFDBregcomp( &(re->compiled[0]), pattern, flags );
#else
   re->compiled[0] = ufdbCalloc( 1, sizeof(regex_t) );
   re->error = UFDBregcomp( re->compiled[0], pattern, flags );
   if (!re->error)
   {
      for (i = 1;  i < UFDB_NREGEX;  i++)
      {
	 re->compiled[i] = ufdbCalloc( 1, sizeof(regex_t) );
	 (void) UFDBregcomp( re->compiled[i], pattern, flags );
      }
   }
#endif
   re->pattern = ufdbStrdup( pattern );
   re->substitute = NULL;
   re->flags = flags;
   re->global = 0;
   re->httpcode = NULL;
   re->next = NULL;
 
   return re;
}


/* 
 *  optimise list of RE1 ... REn into one RE with (RE1)| ... |(REn) 
 */
struct ufdbRegExp * UFDBoptimizeExprList( 
   char *              reSource,
   struct ufdbRegExp * reList )
{
   struct ufdbRegExp * re;
   int    n;
   int    totalStrlen;

   n = 0;
   totalStrlen = 0;

   for (re = reList;  re != NULL;  re = re->next)
   {
      if (re->error == 0)
      {
         n++;
	 totalStrlen += strlen( re->pattern );
      }
   }

   if (n == 0)
   {
      if (ufdbGV.debugRegexp)
	 ufdbLogMessage( "   UFDBoptimizeExprList: %s has no REs.", reSource );
      return NULL;
   }

   if (ufdbGV.debug || ufdbGV.debugRegexp)
      ufdbLogMessage( "   UFDBoptimizeExprList: %s has %d REs", reSource, n );
   if (ufdbGV.debugRegexp > 1)
   {
      ufdbLogMessage( "   UFDBoptimizeExprList: %d REs of %s are not optimised due to expression debugging",
                      n, reSource );
      return reList;
   }

   if (n > 1)
   {
      char * newpattern;

      newpattern = (char *) ufdbMalloc( totalStrlen + 3*n + 1 );
      newpattern[0] = '\0';
      for (re = reList;  re != NULL;  re = re->next)
      {
         if (re->error == 0)
	 {
#ifdef UFDB_USE_OPTIM_RE_BRACKETS
	    strcat( newpattern, "(" );
	    strcat( newpattern, re->pattern );
	    if (re->next == NULL)
	       strcat( newpattern, ")" );
	    else
	       strcat( newpattern, ")|" );
#else
	    strcat( newpattern, re->pattern );
	    if (re->next != NULL)
	       strcat( newpattern, "|" );
#endif
	 }
      }
      if (ufdbGV.debug > 2  ||  ufdbGV.debugRegexp)
         ufdbLogMessage( "going to optimise complex expression of %d subexpressions and %d characters",
                         n, totalStrlen + 1*(n-1) );
      if (n > 500)
         ufdbLogMessage( "WARNING: the expressionlist has %d expressions and may use many resources  *****\n"
			 "Note that large numbers of expressions may impact performance considerably",
	                 n );
      re = ufdbNewPatternBuffer( newpattern, reList->flags );
      if (re->error) 
      {
         ufdbLogError( "UFDBoptimizeExprList: unable to optimise %d expressions of %s  (error %d)  *****",
                       n, reSource, re->error );
         if (n >= 20)
            ufdbLogMessage( "Since the %d expressions could not be optimised into one expression, "
                            "they will be evaluated one by one which impacts performance  *****", n );
	 ufdbFree( newpattern );
	 ufdbFreeRegExprList( re );
	 return reList;
      }

#if UFDB_GEN_API
      if (ufdbGV.debug || ufdbGV.debugRegexp)
#endif
         ufdbLogMessage( "the %d expressions of %s have been optimised", n, reSource );
      if (ufdbGV.debugRegexp)
         ufdbLogMessage( "optimised expression: %s", newpattern );

      ufdbFree( newpattern );
      ufdbFreeRegExprList( reList );
      return re;
   }
   else if (n == 1)
   {
      if (ufdbGV.debugRegexp || ufdbGV.debug)
	 ufdbLogMessage( "the expressions of %s has one RE and does not need optimisation", reSource );
   }

   return reList;
}


/*
 * initialize an expression list (read them from a file and do the regexp compilation)
 */
int UFDBloadExpressions( 
   struct ufdbRegExp ** exprList,
   char *               file  )
{
   FILE *               fin;
   char *               eoln;
   struct ufdbRegExp *  re;
   struct ufdbRegExp *  last;
   int                  retCode;
   char                 line[1024];

   if (exprList == NULL)
      return UFDB_API_ERR_NULL;
   *exprList = NULL;

   if (file == NULL)
      return UFDB_API_ERR_NULL;

   fin = fopen( file, "r" );
   if (fin == NULL)
      return UFDB_API_ERR_NOFILE;

   retCode = UFDB_API_OK;
   last = NULL;
   re = NULL;

   while (fgets( line, sizeof(line), fin ) != NULL)
   {
      if (line[0] == '#')         /* skip comments */
         continue;

      eoln = strchr( line, '\n' );
      if (eoln == NULL  ||  eoln == &line[0])
         continue;	/* skip empty lines and lines without a newline */
      else
      {
         if (*(eoln-1) == '\r')
	    eoln--;
      }
      *eoln = '\0';	/* get rid of the newline */

      /* URLs are folded to lowercase so we do not use REG_ICASE any more */
      re = ufdbNewPatternBuffer( line, REG_EXTENDED | REG_NOSUB );
      if (re->error)
      {
	 ufdbLogError( "could not compile regular expression from %s (error %d) : \"%s\"",
                       file, re->error, line );
         retCode = UFDB_API_ERR_EXPR;
      }

      re->next = last;
      last = re;
   }

   if (ufdbGV.expressionOptimisation)
   {
      *exprList = UFDBoptimizeExprList( file, re );
   }
   else
      *exprList = re;

   fclose( fin );

   return retCode;
}


/*
 * match a URL with a compiled RE.
 * return 0 if no match, 1 if there is a match.
 */
UFDB_GCC_HOT 
int ufdbRegExpMatch(   
   struct ufdbRegExp * regexp, 
   const char *        str )
{
   struct ufdbRegExp * rp;
   int                 error;
   int                 i;

   if (ufdbGV.debugRegexp)
      ufdbLogMessage( "      ufdbRegExpMatch \"%s\" %s", str, regexp==NULL ? "no REs" : "RE to test" );

   for (rp = regexp;  rp != NULL;  rp = rp->next)
   {
      if (ufdbGV.debugRegexp)
	 ufdbLogMessage( "      ufdbRegExpMatch  %s  %s  error=%d", str, rp->pattern, rp->error );
      if (rp->error)
         continue;

#if !(HAVE_PCRE_COMPILE2 || HAVE_PCRE_COMPILE)  &&  !defined(UFDB_API_NO_THREADS)
      ufdb_mutex_lock( &(rp->lock) );
      i = rp->next_nregex_i;
      rp->next_nregex_i = (i + 1) % UFDB_NREGEX;
      ufdb_mutex_unlock( &(rp->lock) );
#else
      i = 0;
#endif

#if HAVE_PCRE_COMPILE || HAVE_PCRE_COMPILE2
      error = UFDBregexec( (pcre*) rp->compiled[i], str, 0, NULL, 0 );
#else
      error = UFDBregexec( (regex_t*) rp->compiled[i], str, 0, NULL, 0 );
#endif
      if (error == 0) 	/* match */
      {
	 if (ufdbGV.debugRegexp)
	    ufdbLogMessage( "   RE match:  %s  %s", rp->pattern, str );
         return UFDB_API_MATCH;
      }
      if (error != REG_NOMATCH) 
      {
	 if (ufdbGV.debugRegexp)
	    ufdbLogMessage( "   RE error %d:  %s  %s  error=%d", error, rp->pattern, str, error );
         return UFDB_API_ERR_EXPR;
      }
   }

   return 0;
}


void ufdbFreeRegExprList( struct ufdbRegExp * re )
{
   struct ufdbRegExp * tmp;
   int                 i;

   while (re != NULL)
   {
      if (!re->error)
      {
	 for (i = 0;  i < UFDB_NREGEX;  i++)
	 {
	    UFDBregfree( re->compiled[i] );
#if !HAVE_PCRE_COMPILE  &&  !HAVE_PCRE_COMPILE2
	    ufdbFree( re->compiled[i] );
#endif
	 }
      }
      ufdbFree( re->pattern );
      ufdbFree( re->substitute );
      tmp = re->next;
      ufdbFree( re );
      re = tmp;
   }
}
#endif   /* UFDB_REGEX_SUPPORT */


void ufdbResetCPUs( void )
{
   cpu_mask = 0UL;
}


const char * ufdbAPIstatusString( int api_code )
{
   switch (api_code)
   {
   case UFDB_API_OK:                       return "OK";
   case UFDB_API_ERR_NULL:                 return "ERR_NULL";
   case UFDB_API_ERR_NOFILE:               return "ERR_NOFILE";
   case UFDB_API_ERR_READ:                 return "ERR_READ";
   case UFDB_API_ERR_EXPR:                 return "ERR_EXPR";
   case UFDB_API_ERR_RANGE:                return "ERR_RANGE";
   case UFDB_API_ERR_ERRNO:                return "ERR_ERRNO";
   case UFDB_API_ERR_SOCKET:               return "ERR_SOCKET";
   case UFDB_API_ERR_NOMEM:                return "ERR_NOMEM";
   case UFDB_API_REQ_QUEUED:               return "REQ_QUEUED";
   case UFDB_API_ERR_TUNNEL:               return "ERR_TUNNEL";
   case UFDB_API_ERR_INVALID_CERT:         return "ERR_INVALID_CERT";
   case UFDB_API_ERR_IP_ADDRESS:           return "ERR_IP_ADDRESS";
   case UFDB_API_ERR_OLD_TABLE:            return "ERR_OLD_TABLE";
   case UFDB_API_ERR_INVALID_TABLE:        return "ERR_INVALID_TABLE";
   case UFDB_API_ERR_INVALID_KEY:          return "ERR_INVALID_KEY";
   case UFDB_API_ERR_IS_SKYPE:             return "ERR_IS_SKYPE";
   case UFDB_API_ERR_FULL:                 return "ERR_FULL";
   case UFDB_API_ERR_UNKNOWN_PROTOCOL:     return "ERR_UNKNOWN_PROTOCOL";
   case UFDB_API_ERR_IS_GTALK:             return "ERR_IS_GTALK";
   case UFDB_API_ERR_IS_YAHOOMSG:          return "ERR_IS_YAHOOMSG";
   case UFDB_API_ERR_IS_AIM:               return "ERR_IS_AIM";
   case UFDB_API_ERR_IS_FBCHAT:            return "ERR_IS_FBCHAT";
   case UFDB_API_ERR_IS_CITRIXONLINE:      return "ERR_IS_CITRIXONLINE";
   case UFDB_API_ERR_IS_ANYDESK:           return "ERR_IS_ANYDESK";
   case UFDB_API_ERR_IS_TEAMVIEWER:        return "ERR_IS_TEAMVIEWER";
   case UFDB_API_ERR_CKSUM_NOT_VALID:      return "ERR_CKSUM_NOT_VALID";
   case UFDB_API_ERR_OUTDATED:             return "ERR_OUTDATED";
   case UFDB_API_ERR_FATAL:                return "ERR_FATAL";
   case UFDB_API_ERR_TLS:                  return "ERR_TLS";
   case UFDB_API_BEING_VERIFIED:           return "BEING_VERIFIED";
   case UFDB_API_MODIFIED_FOR_SAFESEARCH:  return "MODIFIED_FOR_SAFESEARCH";
   case -1:				   return "INTERNAL_ERROR_MINUS_ONE";
   }

   return "INTERNAL_ERROR_UNKNOWN_CODE";
}


const char * ufdbDBstat2string( int status )
{
   switch (status)
   {
      case UFDB_API_STATUS_DATABASE_OK:       return "up to date";
      case UFDB_API_STATUS_DATABASE_OLD:      return "one or more tables are more than 4 days old.  Check cron job for ufdbUpdate.";
      case UFDB_API_STATUS_DATABASE_EXPIRED:  return "one or more tables are EXPIRED.  Check licenses and cron job for ufdbUpdate.";
   }
   return "internal-error";
}


const char * ufdbStatus2string( int status )
{
   switch (status)
   {
      case UFDB_API_STATUS_VIRGIN:          return "virgin";
      case UFDB_API_STATUS_STARTED_OK:      return "started";
      case UFDB_API_STATUS_TERMINATED:      return "terminated";
      case UFDB_API_STATUS_RELOADING:       return "reloading";
      case UFDB_API_STATUS_RELOAD_OK:       return "reloaded";
      case UFDB_API_STATUS_FATAL_ERROR:     return "error";
      case UFDB_API_STATUS_ROLLING_LOGFILE: return "rolling-logfile";
      case UFDB_API_STATUS_UPDATE: 	    return "status-update";
      case UFDB_API_STATUS_CRASH_REPORT_UPLOADED:     	return "crash-report-uploaded";
      case UFDB_API_STATUS_CRASH_REPORT_NOT_UPLOADED:   return "upload-crash-reports-off-prohibits-upload";
   }
   return "internal-error-unknown-status";
}


#ifndef ufdbStrncpy

UFDB_GCC_HOT void ufdbStrncpy( char * dest, const char * src, size_t n )
{
   if (memccpy( dest, src, '\0', n ) == NULL)
      dest[n-1] = '\0';
}

#endif


#ifdef __cplusplus
}
#endif
