/*
 * ufdbUserlist.c - URLfilterDB
 *
 * ufdbGuard is copyrighted (C) 2005-2020 by URLfilterDB with all rights reserved.
 *
 * Parts of ufdbGuard are based on squidGuard.
 * This module is NOT based on squidGuard.
 *
 * RCS $Id: ufdbUserlist.c,v 1.13 2020/11/12 13:04:37 root Exp root $
 */


#include "ufdb.h"
#include "sg.h"
#include "ufdblib.h"
#include "ufdblocks.h"

#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <pthread.h>
#include <sys/stat.h>
#include <sys/types.h>

#ifdef __cplusplus
extern "C" {
#endif

extern pthread_rwlock_t  TheDynamicSourcesLock;   // TODO: put this in a header file



/* UFDBretrieveExecUserlist() uses a cache to support slow LDAP queries.
 * During a reload the execuserlist results are not deleted any more and an
 * unchanged command to retrieve a list of users is returned from the cache.
 * For the 15-minute refresh of a userlist, there is a new function: UFDBrefreshExecUserlist();
 */

typedef struct ulCacheElem {
   char *       command;
   struct sgDb  udb;
} ulCacheElem;

UFDB_GCC_ALIGN_CL
static ufdb_mutex   ul_mutex = ufdb_mutex_initializer;

static unsigned     NulCache = 0;
static ulCacheElem  ulCache[UFDB_MAX_USERLISTS];


static void encodeCommand( char * command, char * encodedCommand, int maxlen )
{
   int flen;

   flen = 0;
   while (*command != '\0')
   {
      if (*command == ' ' || *command == '\t')
      {
         *encodedCommand++ = '_';
      }
      else if (*command == '/' || *command == '\\')
      {
         *encodedCommand++ = '+';
      }
      else if (*command == '#' || *command == '|')
      {
         *encodedCommand++ = '+';
         *encodedCommand++ = '-';
         flen++;
      }
      else if (*command == '\'' || *command == '"' || *command == '`')
      {
         *encodedCommand++ = '+';
         *encodedCommand++ = '+';
         flen++;
      }
      else if (*command == '[' || *command == ']' || *command == '{' || *command == '}')
      {
         *encodedCommand++ = '-';
         *encodedCommand++ = '+';
         flen++;
      }
      else if (*command == '$' || *command == '%' || *command == '!' || *command == '~' || 
               *command == '(' || *command == ')')
      {
         *encodedCommand++ = '_';
         *encodedCommand++ = '_';
         flen++;
      }
      else if (*command == '<' || *command == '>' || *command == '?')
      {
         *encodedCommand++ = '_';
         *encodedCommand++ = '+';
         flen++;
      }
      else
	 *encodedCommand++ = *command;
      command++;

      flen++;
      if (flen == maxlen-8  &&  strlen(command) > 3)
      {
         /* The maximum filename length is maxlen characters and we are about to go over it.
	  * So calculate a hash for the rest of the command and put the hash value in the encoded command.
	  */
	 unsigned long h;
	 h = (69313 * *command) ^ *(command+1);
	 while (*command != '\0')
	 {
	    h = (h << 5) + ((*command * 7) ^ (h >> 3));
	    command++;
	 }
	 sprintf( encodedCommand, ".%6lu", h % 1000000 );
	 return;
      }
   }
   *encodedCommand = '\0';
}


static FILE * openCacheForWrite( struct ufdbGV * gv, char * command )
{
   int    ret;
   FILE * fc;
   char * dbhome;
   char   fileName[2048];

   dbhome = gv->databaseDirectory;
   strcpy( fileName, dbhome );
   strcat( fileName, "/cache.execlists" );
   errno = 0;
   ret = mkdir( fileName, 0770 );
   if (ret != 0  &&  errno != EEXIST)
   {
      ufdbLogError( "cannot create cache directory \"%s\": %s", fileName, strerror(errno) );
      encodeCommand( command, fileName+strlen(fileName), 255 - sizeof("/cache.execlists") );
   }
   else
   {
      (void) chmod( fileName, 0770 );
      strcat( fileName, "/" );
      encodeCommand( command, fileName+strlen(fileName), 255 );
   }

   fc = fopen( fileName, "w" );
   if (fc == NULL)
      ufdbLogError( "cannot write to userlist cache file \"%s\": %s", fileName, strerror(errno) );
   else if (gv->debug)
      ufdbLogMessage( "writing to userlist cache file \"%s\"", fileName );

   return fc;
}


static FILE * openCacheForRead( struct ufdbGV * gv, char * command )
{
   FILE * fc;
   char * dbhome;
   struct stat stbuf;
   char   fileName[2048];

   dbhome = gv->databaseDirectory;
   strcpy( fileName, dbhome );
   strcat( fileName, "/cache.execlists" );
   if (stat( fileName, &stbuf ) == 0)
   {
      strcat( fileName, "/" );
      encodeCommand( command, fileName+strlen(fileName), 255 );
   }
   else
      encodeCommand( command, fileName+strlen(fileName), 255 - sizeof("/cache.execlists") );

   fc = fopen( fileName, "r" );
   if (fc != NULL)
   {
      if (gv->debug)
	 ufdbLogMessage( "reading from userlist cache file \"%s\"", fileName );
   }

   return fc;
}


static void readUserlistFromCache( FILE * fin, struct sgDb * db )
{
   char          line[10000];

   db->dbhome = NULL;
   db->dbcp = (void *) UFDBmemDBinit();
   db->type = SGDBTYPE_EXECUSERLIST;

   while (UFDBfgetsNoNL(line,sizeof(line),fin) != NULL)
   {
      UFDBmemDBinsert( (struct UFDBmemDB *) db->dbcp, line, NULL );
   }
}


static int execUserlistCommand( struct ufdbGV * gv, char * command, struct sgDb * db )
{
   int           ullineno;
   FILE *        fin;
   FILE *        fcache;
   char *        lc;
   time_t        tb, te;
   char          line[10000];

   if (gv->terminating)
      return 0;

   /* make a new cache file for this command */
   fcache = openCacheForWrite( gv, command );
   
   db->dbhome = NULL;
   db->dbcp = (void *) UFDBmemDBinit();
   db->type = SGDBTYPE_EXECUSERLIST;

   tb = time( NULL );
   errno = 0;
   if ((fin = popen(command,"r")) == NULL)
   {
      ufdbLogError( "can't execute command of execuserlist \"%s\": %s  *****", command, strerror(errno) );
      return 0;
   }
   ullineno = 0;

   while (UFDBfgetsNoNL(line,sizeof(line),fin) != NULL)
   {
      ullineno++;

      if (gv->debug > 1  ||  gv->debugExternalScripts)
         ufdbLogMessage( "execuserlist: received \"%s\"", line );

      if (line[0] == '#')			/* skip comments */
         continue;

      if (line[0] == '\0')
      {
         ufdbLogError( "execuserlist \"%s\": line %d: line is empty", command, ullineno );
         continue;
      }

      for (lc = line;  *lc != '\0';  lc++)   /* convert username to lowercase chars */
      {
	 if (*lc <= 'Z'  &&  *lc >= 'A')
	    *lc += 'a' - 'A';
      }
      UFDBmemDBinsert( (struct UFDBmemDB *) db->dbcp, line, NULL );
      if (fcache != NULL)
	 fprintf( fcache, "%s\n", line );
   }
   (void) pclose( fin );
   if (fcache != NULL)
      fclose( fcache );
   te = time( NULL );

   db->entries = ullineno;
   ufdbLogMessage( "execuserlist: finished retrieving userlist (%d lines in %ld seconds) generated by \"%s\"",
                   ullineno, (long) (te - tb), command );

   return 1;
}


void UFDBdeleteUserlistCache( void )
{
   unsigned i;

   ufdb_mutex_lock( &ul_mutex );                // >>==========================

   for (i = 0; i < NulCache; i++)
   {
      ufdbFree( ulCache[i].command );
      UFDBmemDBdeleteDB( ulCache[i].udb.dbcp );
      ulCache[i].udb.dbcp = NULL;
   }
   NulCache = 0;

   ufdb_mutex_unlock( &ul_mutex );              // <<==========================
}


struct sgDb * UFDBretrieveExecUserlist( struct ufdbGV * gv, char * command )
{
   unsigned      i;
   FILE *        fcache;

   for (i = 0;  i < NulCache;  i++)
   {
      if (strcmp( ulCache[i].command, command ) == 0)
         return &(ulCache[i].udb);
   }

   /* TO-DO: remove old userlist commands from the cache */
   /* TO-DO: but do this only if in the last 2 reloads the cache is not used */

   ufdb_mutex_lock( &ul_mutex );
   i = NulCache;
   ulCache[i].command = ufdbStrdup( command );
   ulCache[i].udb.dbhome = NULL;
   NulCache++;
   ufdb_mutex_unlock( &ul_mutex );

   if (NulCache == UFDB_MAX_USERLISTS)
      ufdbLogFatalError( "UFDBrefreshExecUserlist: maximum number of %d dynamic userlists is reached", UFDB_MAX_USERLISTS );

   /* See if we have the command cached in a file */
   fcache = openCacheForRead( gv, command );
   if (fcache == NULL)
      execUserlistCommand( gv, command, &(ulCache[i].udb) );
   else
   {
      if (gv->debug)
         ufdbLogMessage( "reading cached userlist from file; command is \"%s\"", command );
      readUserlistFromCache( fcache, &(ulCache[i].udb) );
      fclose( fcache );
   }

   return &(ulCache[i].udb);
}


void UFDBrefreshExecUserlist( struct ufdbGV * gv, char * command )
{
   unsigned     i;
   int          found;
   int          ret;
   struct UFDBmemDB * oldUserlist;
   struct sgDb  tmpDb;

   found = 0;
   ufdb_mutex_lock( &ul_mutex );					// >>++++++++
   for (i = 0;  i < NulCache;  i++)
   {
      if (strcmp( ulCache[i].command, command ) == 0)
      {
         found = 1;
         break;
      }
   }
   ufdb_mutex_unlock( &ul_mutex );					// <<++++++++

   if (found)
   {
      execUserlistCommand( gv, command, &tmpDb );

      if (gv->debugExternalScripts  ||  gv->debug > 1)
         ufdbLogMessage( "UFDBrefreshExecUserlist: replacing userlist generated with command \"%s\"", command );

      ret = pthread_rwlock_wrlock( &TheDynamicSourcesLock );		// >-------------------------------------
      if (ret != 0)
	 ufdbLogError( "UFDBrefreshExecUserlist: pthread_rwlock_wrlock TheDynamicSourcesLock failed: code %d",
                       ret );

      (void) ufdb_mutex_lock( &ul_mutex );				// >>+++++++
      oldUserlist = ulCache[i].udb.dbcp;
      ulCache[i].udb.dbcp = tmpDb.dbcp;
      (void) ufdb_mutex_unlock( &ul_mutex );				// <<+++++++

      ret = pthread_rwlock_unlock( &TheDynamicSourcesLock );		// >-------------------------------------
      if (ret != 0)
	 ufdbLogError( "UFDBrefreshExecUserlist: pthread_rwlock_unlock TheDynamicSourcesLock failed: code %d",
                       ret );

      UFDBmemDBdeleteDB( oldUserlist );
   }
   else
   {
      ufdbLogError( "UFDBrefreshExecUserlist: could not find command in the cache: \"%s\"  *****", command );

      ufdb_mutex_lock( &ul_mutex );					// >>++++++++
      i = NulCache;
      ulCache[i].command = ufdbStrdup( command );
      ulCache[i].udb.dbhome = NULL;
      NulCache++;
      ufdb_mutex_unlock( &ul_mutex );				        // >>++++++++

      if (NulCache == UFDB_MAX_USERLISTS)
         ufdbLogFatalError( "UFDBrefreshExecUserlist: maximum number of %d dynamic userlists is reached",
                            UFDB_MAX_USERLISTS );

      execUserlistCommand( gv, command, &tmpDb );

      (void) ufdb_mutex_lock( &ul_mutex );				// >>++++++++
      ulCache[i].udb.dbcp = tmpDb.dbcp;
      (void) ufdb_mutex_unlock( &ul_mutex );				// <<++++++++
   }
}


#ifdef __cplusplus
}
#endif
