/*
 * ufdbIPlist.c - URLfilterDB
 *
 * ufdbGuard is copyrighted (C) 2005-2020 by URLfilterDB with all rights reserved.
 *
 * Parts of ufdbGuard are based on squidGuard.
 * This module is NOT based on squidGuard.
 *
 * RCS $Id: ufdbIPlist.c,v 1.11 2020/11/12 13:04:05 root Exp root $
 */


#include "ufdb.h"
#include "sg.h"
#include "ufdblib.h"

#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <pthread.h>
#include <sys/stat.h>
#include <sys/types.h>

#ifdef __cplusplus
extern "C" {
#endif

extern pthread_rwlock_t  TheDynamicSourcesLock;

#define USE_CACHE 0


/* UFDBretrieveExecIPlist() uses a cache to support slow LDAP queries.
 * During a reload the execiplist results are not deleted any more and an
 * unchanged command to retrieve a list of IPs is returned from the cache.
 * For the 15-minute refresh of an iplist, there is a new function: UFDBrefreshExecIPlist();
 */

#if USE_CACHE
typedef struct iplCacheElem {
   char *             command;
   struct UFDBmemDB * ipv4hosts;
   struct Ipv4 *      ipv4list;
   struct UFDBmemDB * ipv6hosts;
   struct Ipv6 *      ipv6list;
} iplCacheElem;

UFDB_GCC_ALIGN_CL
static ufdb_mutex    ipl_mutex = ufdb_mutex_initializer;

static unsigned      NiplCache = 0;
static iplCacheElem  iplCache[UFDB_MAX_IPLISTS];
#endif


#if USE_CACHE
static void encodeCommand( char * command, char * encodedCommand, int maxlen )
{
   int flen;

   flen = 0;
   while (*command != '\0')
   {
      if (*command == ' ' || *command == '\t')
      {
         *encodedCommand++ = '_';
      }
      else if (*command == '/' || *command == '\\')
      {
         *encodedCommand++ = '+';
      }
      else if (*command == '#' || *command == '|')
      {
         *encodedCommand++ = '+';
         *encodedCommand++ = '-';
         flen++;
      }
      else if (*command == '\'' || *command == '"' || *command == '`')
      {
         *encodedCommand++ = '+';
         *encodedCommand++ = '+';
         flen++;
      }
      else if (*command == '[' || *command == ']' || *command == '{' || *command == '}')
      {
         *encodedCommand++ = '-';
         *encodedCommand++ = '+';
         flen++;
      }
      else if (*command == '$' || *command == '%' || *command == '!' || *command == '~' || 
               *command == '(' || *command == ')')
      {
         *encodedCommand++ = '_';
         *encodedCommand++ = '_';
         flen++;
      }
      else if (*command == '<' || *command == '>' || *command == '?')
      {
         *encodedCommand++ = '_';
         *encodedCommand++ = '+';
         flen++;
      }
      else
	 *encodedCommand++ = *command;
      command++;

      flen++;
      if (flen == maxlen-8  &&  strlen(command) > 3)
      {
         /* The maximum filename length is maxlen characters and we are about to go over it.
	  * So calculate a hash for the rest of the command and put the hash value in the encoded command.
	  */
	 unsigned long h;
	 h = (69313 * *command) ^ *(command+1);
	 while (*command != '\0')
	 {
	    h = (h << 5) + ((*command * 7) ^ (h >> 3));
	    command++;
	 }
	 sprintf( encodedCommand, ".%6lu", h % 1000000 );
	 return;
      }
   }
   *encodedCommand = '\0';
}
#endif


#if USE_CACHE
static FILE * openCacheForWrite( struct ufdbGV * gv, char * command )
{
   int    ret;
   FILE * fc;
   char * dbhome;
   char   fileName[2048];

   dbhome = gv->databaseDirectory;
   strcpy( fileName, dbhome );
   strcat( fileName, "/cache.execlists" );
   errno = 0;
   ret = mkdir( fileName, 0770 );
   if (ret != 0  &&  errno != EEXIST)
   {
      ufdbLogError( "cannot create cache directory \"%s\": %s", fileName, strerror(errno) );
      encodeCommand( command, fileName+strlen(fileName), 255 - sizeof("/cache.execlists") );
   }
   else
   {
      (void) chmod( fileName, 0770 );
      strcat( fileName, "/" );
      encodeCommand( command, fileName+strlen(fileName), 255 );
   }

   fc = fopen( fileName, "w" );
   if (fc == NULL)
      ufdbLogError( "cannot write to iplist cache file \"%s\": %s", fileName, strerror(errno) );
   else if (gv->debug)
      ufdbLogMessage( "writing to iplist cache file \"%s\"", fileName );

   return fc;
}
#endif


#if USE_CACHE
static FILE * openCacheForRead( struct ufdbGV * gv, char * command )
{
   FILE * fc;
   char * dbhome;
   struct stat stbuf;
   char   fileName[2048];

   dbhome = gv->databaseDirectory;
   strcpy( fileName, dbhome );
   strcat( fileName, "/cache.execlists" );
   if (stat( fileName, &stbuf ) == 0)
   {
      strcat( fileName, "/" );
      encodeCommand( command, fileName+strlen(fileName), 255 );
   }
   else
      encodeCommand( command, fileName+strlen(fileName), 255 - sizeof("/cache.execlists") );

   fc = fopen( fileName, "r" );
   if (fc != NULL)
   {
      if (gv->debug)
	 ufdbLogMessage( "reading from iplist cache file \"%s\"", fileName );
   }

   return fc;
}
#endif


#if USE_CACHE
static void readIPlistFromCache( FILE * fin, struct Ipv4 ** i4, struct Ipv6 ** i6 )
{
   char          line[2048];

   while (UFDBfgetsNoNL(line,sizeof(line),fin) != NULL)
   {
      if (line[3] == '4')			// XXX verify this behaviour
         addIpv4( ip4, &line[5] );
      else
         addIpv6( ip6, &line[5] );
   }
}
#endif


static int execIPlistCommand(
   struct ufdbGV *     gv,
   const char *        command,
   struct UFDBmemDB ** i4h,
   struct Ipv4 **      i4,
   struct UFDBmemDB ** i6h,
   struct Ipv6 **      i6  )
{
   int                 ullineno;
   FILE *              fin;
   FILE *              fcache = NULL;
   char *              sep;
   time_t              tb, te;
   struct Ipv4 *       previ4 = NULL;
   struct Ipv6 *       previ6 = NULL;
   unsigned int        ipv4addr;
   struct in6_addr     ipv6addr;
   char                line[2048];
   char                origline[2048];

   *i4h = NULL;
   *i4 = NULL;
   *i6h = NULL;
   *i6 = NULL;

   if (gv->terminating)
      return 0;

#if USE_CACHE
   /* make a new cache file for this command */
   fcache = openCacheForWrite( command );
#endif

   tb = time( NULL );
   errno = 0;
   if ((fin = popen(command,"r")) == NULL)
   {
      ufdbLogError( "can't execute command of execiplist \"%s\": %s  *****", command, strerror(errno) );
      return 0;
   }
   ullineno = 0;

   if (gv->debug > 1  ||  gv->debugExternalScripts)
      ufdbLogMessage( "execiplist: executing \"%s\"", command );

   while (UFDBfgetsNoNL(line,sizeof(line),fin) != NULL)
   {
      ullineno++;

      if (line[0] == '#')			/* skip comments */
         continue;

      if (line[0] == '\0')                      /* skip empty lines */
         continue;

      if (fcache != NULL)
	 strcpy( origline, line );
      if (gv->debug > 1  ||  gv->debugExternalScripts)
         ufdbLogMessage( "execiplist: \"%s\" line %d: \"%s\"", command, ullineno, line );

      sep = strchr( line, '/' );
      if (sep != NULL)
         *sep = '\0';

      if (sgConvDot( line, &ipv4addr ))		// ipv4 ??
      {
         if (sep == NULL)       		// ipv4 host
         {
	    if (*i4h == NULL)
	       *i4h = UFDBmemDBinit();
	    UFDBmemDBinsert( *i4h, line, NULL );
         }
         else                   		// ipv4 subnet with cidr-notation
         {
            int cidr;
	    struct Ipv4 * newi4 = (struct Ipv4 *) ufdbMalloc( sizeof(struct Ipv4) );
	    if (*i4 == NULL)
	       *i4 = newi4;
	    else
	       previ4->next = newi4;

            *sep = '/';
            newi4->type = SG_IPTYPE_CIDR;
            newi4->net_is_set = 1;
            cidr = atoi( sep+1 );
            if (cidr <= 0  ||  cidr > 32)
            {
               ufdbLogError( "execiplist \"%s\": line %d: \"%s\": bad cidr. assuming 32", 
                             command, ullineno, line );
               cidr = 32;
            }
            if (cidr == 32)
               newi4->mask = (unsigned int) 0xffffffff;
            else
               newi4->mask = (unsigned int) 0xffffffff ^ ((unsigned int) 0xffffffff >> cidr);
            newi4->net = ipv4addr & newi4->mask;
            newi4->next = NULL;
	    previ4 = newi4;
         }
      }
      else      // not IPv4
      {
         if (!sgValidateIPv6( line, &ipv6addr ))	// ipv6 ??
         {
            if (sep != NULL)
               *sep = '/';
            ufdbLogError( "execiplist \"%s\": line %d: invalid IP address or IP subnet: %s",
                          command, ullineno, line );
            continue;
         }

         if (sep == NULL)				// ipv6 host
         {
	    if (*i6h == NULL)
	       *i6h = UFDBmemDBinit();
	    UFDBmemDBinsert( *i6h, line, NULL );
         }
         else						// ipv6 subnet
         {
	    struct Ipv6 * newi6 = (struct Ipv6 *) ufdbMalloc( sizeof(struct Ipv6) );
	    if (*i6 == NULL)
	       *i6 = newi6;
	    else
	       previ6->next = newi6;
            *sep = '/';
            newi6->type = SG_IPV6TYPE_CIDR;
            newi6->cidr = atoi( sep+1 );
            if (newi6->cidr <= 0  ||  newi6->cidr > 128)
            {
               ufdbLogError( "execiplist \"%s\" line %d: \"%s\": bad cidr. assuming 128",
                             command, ullineno, line );
               newi6->cidr = 128;
            }
            newi6->ipv6 = ipv6addr;
            newi6->next = NULL;
	    previ6 = newi6;
	 }
      }

      if (fcache != NULL)
	 fprintf( fcache, "%s\n", origline );
   }
   (void) pclose( fin );
   if (fcache != NULL)
      fclose( fcache );
   te = time( NULL );

   ufdbLogMessage( "execiplist: finished retrieving IP address list (%d lines in %ld seconds) "
                   "generated by \"%s\"",
                   ullineno, (long) (te - tb), command );

   return 1;
}


// UFDBretrieveExecIPlist() is called by readConfig() and must use ufdbNewGV
// UFDBrefreshExecIPlist() is called by a refresher thread and must use ufdbGV
void UFDBretrieveExecIPlist( 
   struct ufdbGV *     gv,
   char *              command,
   struct UFDBmemDB ** i4h,
   struct Ipv4 **      i4,
   struct UFDBmemDB ** i6h,
   struct Ipv6 **      i6  )
{
#if USE_CACHE
   unsigned      i;
   FILE *        fcache;

   for (i = 0;  i < NiplCache;  i++)
   {
      if (strcmp( iplCache[i].command, command ) == 0)
         return &(iplCache[i].udb);
   }

   /* TO-DO: remove old iplist commands from the cache */
   /* TO-DO: but do this only if in the last 2 reloads the cache is not used */

   pthread_mutex_lock( &ipl_mutex );
   i = NiplCache;
   iplCache[i].command = ufdbStrdup( command );
   iplCache[i].udb.dbhome = NULL;
   NiplCache++;
   pthread_mutex_unlock( &ipl_mutex );

   if (NiplCache == UFDB_MAX_IPLISTS)
      ufdbLogFatalError( "UFDBretrieveExecIPlist: maximum number of %d dynamic iplists is reached",
                         UFDB_MAX_IPLISTS );

   /* See if we have the command cached in a file */
   fcache = openCacheForRead( gv, command );
   if (fcache == NULL)
      execIPlistCommand( gv, s, command );
   else
   {
      if (gv->debug)
         ufdbLogMessage( "reading cached iplist from file; command is \"%s\"", command );
      readIPlistFromCache( fcache, &s->ipv4, &s->ipv6 );
      fclose( fcache );
   }

#else
   execIPlistCommand( gv, command, i4h, i4, i6h, i6 );
#endif
}


// UFDBretrieveExecIPlist() is called by readConfig() and must use ufdbNewGV
// UFDBrefreshExecIPlist() is called by a refresher thread and must use ufdbGV
void UFDBrefreshExecIPlist( 
   struct ufdbGV *     gv,
   const char *        command,
   struct UFDBmemDB ** i4h,
   struct Ipv4 **      i4, 
   struct UFDBmemDB ** i6h,
   struct Ipv6 **      i6 )
{
   int          ret;

#if USE_CACHE
   unsigned     i;
   int          found;
   struct UFDBmemDB * oldIPlist;
   struct sgDb  tmpDb;

   found = 0;
   pthread_mutex_lock( &ipl_mutex );
   for (i = 0;  i < NiplCache;  i++)
   {
      if (strcmp( iplCache[i].command, command ) == 0)
      {
         found = 1;
         break;
      }
   }
   pthread_mutex_unlock( &ipl_mutex );

   if (found)
   {
      execIPlistCommand( gv, command, i4h, i4, i6h, i6 );

      if (gv->debugExternalScripts  ||  gv->debug > 1)
         ufdbLogMessage( "UFDBrefreshExecIPlist: replacing iplist generated with command \"%s\"", command );

      ret = pthread_rwlock_wrlock( &TheDynamicSourcesLock );            /* >--------------------- */
      if (ret != 0)
	 ufdbLogError( "UFDBrefreshExecIPlist: pthread_rwlock_wrlock TheDynamicSourcesLock failed with code %d",
                       ret );

      (void) pthread_mutex_lock( &ipl_mutex );                          /* >===== */
      oldIPlist = iplCache[i].udb.dbcp;
      iplCache[i].udb.dbcp = tmpDb.dbcp;
      (void) pthread_mutex_unlock( &ipl_mutex );                        /* <===== */

      ret = pthread_rwlock_unlock( &TheDynamicSourcesLock );            /* <----------------------- */
      if (ret != 0)
	 ufdbLogError( "UFDBrefreshExecIPlist: pthread_rwlock_unlock TheDynamicSourcesLock failed with code %d",
                       ret );

      UFDBmemDBdeleteDB( oldIPlist );
   }
   else
   {
      ufdbLogError( "UFDBrefreshExecIPlist: could not find command in the cache: \"%s\"  *****", command );

      pthread_mutex_lock( &ipl_mutex );                                 /* >===== */
      i = NiplCache;
      iplCache[i].command = ufdbStrdup( command );
      iplCache[i].udb.dbhome = NULL;
      NiplCache++;
      pthread_mutex_unlock( &ipl_mutex );                               /* <===== */

      if (NiplCache == UFDB_MAX_IPLISTS)
         ufdbLogFatalError( "UFDBrefreshExecIPlist: maximum number of %d dynamic iplists is reached",
                            UFDB_MAX_IPLISTS );

      execIPlistCommand( gv, command, i4, i6 );

      (void) pthread_mutex_lock( &ipl_mutex );
      iplCache[i].udb.dbcp = tmpDb.dbcp;
      (void) pthread_mutex_unlock( &ipl_mutex );
   }

#else
   struct UFDBmemDB * newi4h;
   struct Ipv4 *      newi4;
   struct UFDBmemDB * tmpi4h;
   struct Ipv4 *      tmpi4;
   struct UFDBmemDB * newi6h;
   struct Ipv6 *      newi6;
   struct UFDBmemDB * tmpi6h;
   struct Ipv6 *      tmpi6;

   execIPlistCommand( gv, command, &newi4h, &newi4, &newi6h, &newi6 );

   ret = pthread_rwlock_wrlock( &TheDynamicSourcesLock );       // >-----------------------------------------
   if (ret != 0)
      ufdbLogError( "UFDBrefreshExecIPlist: pthread_rwlock_wrlock TheDynamicSourcesLock failed with code %d",
            ret );

   tmpi4h = *i4h;
   *i4h = newi4h;

   tmpi6h = *i6h;
   *i6h = newi6h;

   tmpi4 = *i4;
   *i4 = newi4;

   tmpi6 = *i6;
   *i6 = newi6;

   ret = pthread_rwlock_unlock( &TheDynamicSourcesLock );       // <-----------------------------------------
   if (ret != 0)
      ufdbLogError( "UFDBrefreshExecIPlist: pthread_rwlock_unlock TheDynamicSourcesLock failed with code %d",
                    ret );

   ufdbFreeIpv4List( tmpi4 );
   ufdbFreeIpv6List( tmpi6 );
   if (tmpi4h != NULL)
      UFDBmemDBdeleteDB( tmpi4h );
   if (tmpi6h != NULL)
      UFDBmemDBdeleteDB( tmpi6h );
#endif
}


#ifdef __cplusplus
}
#endif

