/*
 * ufdbGuard is copyrighted (C) 2005-2023 by URLfilterDB B.V. with all rights reserved.
 *
 * Parts of the ufdbGuard daemon are based on squidGuard.
 * squidGuard is copyrighted (C) 1998 by
 * ElTele st AS, Oslo, Norway, with all rights reserved.
 *
 * $Id: sgLog.c,v 1.133 2022/12/26 23:12:15 root Exp root $
 */


/* This module is well tested and stable for a long time AND
 * needs maximum performance, so we remove _FORTIFY_SOURCE.
 */
#undef _FORTIFY_SOURCE

#include "ufdb.h"
#include "ufdblib.h"
#include "ufdblocks.h"
#include "sg.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/utsname.h>
#include <unistd.h>
#include <string.h>
#include <stdarg.h>
#include <errno.h>
#include <pthread.h>
#include <syslog.h>

#ifdef __cplusplus
extern "C" {
#endif


UFDB_GCC_ALIGN_CL
static ufdb_mutex log_mutex = ufdb_mutex_initializer;
static int    globalLogging = 1;

static int    rotationError = 0;
static int    UFDBforceLogRotation = 0;

static unsigned long    logfilesize = 0;
static ufdb_crashfun_t  crashfun = (ufdb_crashfun_t) NULL;

static char   logFilename[1024];


/* Rotate log files.
 * NOTE: There is a race condition when multiple instances try to rotate.
 *       This may happen with ufdbgclient (N processes vs. N threads).
 */
static void RotateLogfile( 
   char *      filename )
{
   int         i;
   char        oldfile[1024+15];
   char        newfile[1024+15];

   /* rotate the log file:
    * file.log.7  ->  file.log.8
    * file.log.6  ->  file.log.7
    * file.log.5  ->  file.log.6
    * file.log.4  ->  file.log.5
    * file.log.3  ->  file.log.4
    * file.log.2  ->  file.log.3
    * file.log.1  ->  file.log.2
    * file.log    ->  file.log.1
    */

   for (i = 8; i > 1; i--)
   {
      snprintf( newfile, sizeof(newfile), "%s.%d", filename, i );
      snprintf( oldfile, sizeof(oldfile), "%s.%d", filename, i-1 );
      rename( oldfile, newfile );
   }
   sync();	/* sync() may resolve some borderline disk space issues */

   snprintf( newfile, sizeof(newfile), "%s.%d", filename, 1 );
   if (rename( filename, newfile ) != 0)
   {
      /* OOPS, rename did not work. maybe no space or a permission problem
       * we need to do damage control: remove the original file.
       */
      if (unlink( filename ) != 0)
      {
         /* it is getting worse... */
	 if (truncate( filename, 0 ))
            { ; }
      }
      if (++rotationError < 5)
      {
	 syslog( LOG_ALERT, "%s cannot rotate its log files !!  Check permissions and file system space.  *****",
	         ufdbGV.progname );
      }
   }
}


UFDB_GCC_HOT static void ufdbLog( 
   char *    msg  )
{
   char *    nl;
   int       multiline;
   char      date[28];
   unsigned long  logmsglen;
#ifdef UFDB_DEBUG_LOGGING
   int       newlog = 0;
#endif
   char      logmsg[UFDB_MAX_LOGMSG_SZ+2048+2048];

#if 0
   if (!globalLogging)
      return;
#endif

   niso( 0, date );

   multiline = 0;
   logmsg[0] = '\0';
   logmsglen = 0;
   while ((nl = strchr( msg, '\n' )) != NULL)
   {
      *nl = '\0';
      logmsglen += (unsigned long) snprintf( logmsg+logmsglen, sizeof(logmsg)-logmsglen-30, 
                                             "%s [%d] %s%s\n", date, ufdbGV.pid, multiline?"      ":"", msg );
      msg = nl + 1;
      multiline = 1;
      if (logmsglen > sizeof(logmsg)-2048)
      {
	 logmsglen += (unsigned long) snprintf( logmsg+logmsglen, sizeof(logmsg)-logmsglen-30, 
                                                "%s [%d] ... ... ...\n", date, ufdbGV.pid );
         goto msg_too_long;
      }
   }

   if (*msg != '\0')
      logmsglen += (unsigned long) snprintf( logmsg+logmsglen, sizeof(logmsg)-logmsglen-30, 
                                             "%s [%d] %s%s\n", date, ufdbGV.pid, multiline?"      ":"", msg );

msg_too_long:
   ufdb_mutex_lock( &log_mutex );			/* LOG_MUTEX *******************/

   if (UFDBforceLogRotation)
   {
      UFDBforceLogRotation = 0;
      if (ufdbGlobalLogfile != NULL)  
      {
	 RotateLogfile( logFilename );
	 ufdbSetGlobalErrorLogFile( NULL, NULL, 1 );
#ifdef UFDB_DEBUG_LOGGING
         newlog = 1;
#endif
      }
   }

   if (ufdbGlobalLogfile == NULL) 
   {
      fputs( logmsg, stderr );
      fflush( stderr );
   }
   else
   {
      if (logfilesize == 0)
	 logfilesize = ftell( ufdbGlobalLogfile );

      /* write is *much* faster than fputs */
      if (write( fileno(ufdbGlobalLogfile), logmsg, (size_t) logmsglen ) > 0)
	 logfilesize += logmsglen;

      if (logfilesize > (size_t) ufdbGV.maxLogfileSize)
      {
	 RotateLogfile( logFilename );
	 ufdbSetGlobalErrorLogFile( NULL, NULL, 1 );
#ifdef UFDB_DEBUG_LOGGING
         newlog = 1;
#endif
	 logfilesize = 1;
      }
   }

   ufdb_mutex_unlock( &log_mutex );			/* LOG_MUTEX *******************/
}


void ufdbGlobalSetLogging( int logging )
{
   globalLogging = logging;
}


void UFDBrotateLogfile( void )
{
   UFDBforceLogRotation = 1;
}


void ufdbSetGlobalErrorLogFile( char * logdir, char * basename, int mutex_is_used )
{
   struct stat s;
   int         retval;

   if (!globalLogging)
      return;

   if (ufdbGlobalLogfile == stderr)
      return;

   if (logdir == NULL  ||  logdir[0] == '\0')
   {
      if (ufdbGV.logDirectory[0] == '\0')
      {
         if (ufdbNewGV.logDirectory[0] == '\0')
            strcpy( logFilename, DEFAULT_LOGDIR );
         else
            strcpy( logFilename, ufdbNewGV.logDirectory );
      }
      else
         strcpy( logFilename, ufdbGV.logDirectory );
   }
   else
      strcpy( logFilename, logdir );
   strcat( logFilename, "/" );

   if (basename == NULL  ||  basename[0] == '\0')
   {
      if (ufdbGV.progname[0] != '\0')
      {
         strcat( logFilename, ufdbGV.progname );
         strcat( logFilename, ".log" );
      }
      else
         strcat( logFilename, DEFAULT_LOGFILE );
   }
   else
   {
      strcat( logFilename, basename );
      strcat( logFilename, ".log" );
   }

   retval = stat( logFilename, &s );
   if (retval == 0)
   {
      if (s.st_size > (off_t) ufdbGV.maxLogfileSize)
	 RotateLogfile( logFilename );
   }

   if (!mutex_is_used)
      ufdb_mutex_lock( &log_mutex );            /* critical section */

   if (ufdbGlobalLogfile != NULL)
      fclose( ufdbGlobalLogfile );

   ufdbGlobalLogfile = fopen( logFilename, "a" );
   if (ufdbGlobalLogfile == NULL)
   {
      fprintf( stderr, "%s: cannot write to logfile %s: %s (uid=%d,euid=%d)\n",
      	 	       ufdbGV.progname, logFilename, strerror(errno), getuid(), geteuid() );
      if (retval == 0)
      {
         fprintf( stderr, "%s: the logfile exists, the owner has uid %d and the file mode is %04o\n",
	                  ufdbGV.progname, s.st_uid, s.st_mode );
      }
      else
      {
         fprintf( stderr, "%s: the logfile does not exist and I cannot create it\n",
	                  ufdbGV.progname );
      }

      /*
       * We insist in having a logfile so try an other directory...
       */
      strcpy( logFilename, "/tmp/" );
      strcat( logFilename, ufdbGV.progname );
      strcat( logFilename, ".log" );
      ufdbGlobalLogfile = fopen( logFilename, "a" );
      if (ufdbGlobalLogfile == NULL)
	 fprintf( stderr, "%s: cannot write to logfile %s: %s\n", ufdbGV.progname, logFilename, strerror(errno) );
   }

   if (ufdbGlobalLogfile == NULL)
      logfilesize = 1;
   else
   {
      fprintf( ufdbGlobalLogfile, "\n"
                               "The ufdbGuard (" UFDB_VERSION ") software suite is free and Open Source Software.\n"
                               "Copyright (C) 2005-2023 by URLfilterDB B.V. and others.\n"
			       "\n" );
      fflush( ufdbGlobalLogfile );
      logfilesize = ftell( ufdbGlobalLogfile );
   }

   if (!mutex_is_used)
      ufdb_mutex_unlock( &log_mutex );          /* end critical section */
}


UFDB_GCC_HOT void niso( 
   time_t    t, 
   char *    buf )
{
   struct tm lc;

   if (t == 0)
      t = UFDBtime();
   localtime_r( &t, &lc );
   sprintf( buf, "%04d-%02d-%02d %02d:%02d:%02d", 
            lc.tm_year + 1900, lc.tm_mon + 1, lc.tm_mday, lc.tm_hour, lc.tm_min, lc.tm_sec );
}


UFDB_GCC_HOT 
void ufdbLogError( const char * format, ... )
{
   va_list ap;
   char    msg[UFDB_MAX_LOGMSG_SZ];

   if (!globalLogging)
      return;

   strcpy( msg, "ERROR: " );

   va_start( ap, format );
   if (vsnprintf( msg+7, UFDB_MAX_LOGMSG_SZ-32-7, format, ap ) > UFDB_MAX_LOGMSG_SZ-7-34)
   {
      msg[UFDB_MAX_LOGMSG_SZ-34] = ' ';
      msg[UFDB_MAX_LOGMSG_SZ-33] = '.';
      msg[UFDB_MAX_LOGMSG_SZ-32] = '.';
      msg[UFDB_MAX_LOGMSG_SZ-31] = '.';
      msg[UFDB_MAX_LOGMSG_SZ-30] = '\0';
   }
   va_end( ap );

   ufdbLog( msg );
}


UFDB_GCC_HOT 
void ufdbLogMessage( const char * format, ... )
{
   va_list ap;
   char    msg[UFDB_MAX_LOGMSG_SZ];

   if (!globalLogging)
      return;

   va_start( ap, format );
   if (vsnprintf( msg, UFDB_MAX_LOGMSG_SZ-32, format, ap ) > UFDB_MAX_LOGMSG_SZ-34)
   {
      msg[UFDB_MAX_LOGMSG_SZ-34] = ' ';
      msg[UFDB_MAX_LOGMSG_SZ-33] = '.';
      msg[UFDB_MAX_LOGMSG_SZ-32] = '.';
      msg[UFDB_MAX_LOGMSG_SZ-31] = '.';
      msg[UFDB_MAX_LOGMSG_SZ-30] = '\0';
   }
   va_end( ap );

   ufdbLog( msg );
}


UFDB_GCC_HOT 
void ufdbLogFatalError( const char * format, ... )
{
   va_list ap;
   char    msg[UFDB_MAX_LOGMSG_SZ];
   char    logmsg[UFDB_MAX_LOGMSG_SZ+100];

   va_start( ap, format );
   if (vsnprintf( msg, UFDB_MAX_LOGMSG_SZ-64, format, ap ) > UFDB_MAX_LOGMSG_SZ-64-2)
   {
      msg[UFDB_MAX_LOGMSG_SZ-64] = ' ';
      msg[UFDB_MAX_LOGMSG_SZ-63] = '.';
      msg[UFDB_MAX_LOGMSG_SZ-62] = '.';
      msg[UFDB_MAX_LOGMSG_SZ-61] = '.';
      msg[UFDB_MAX_LOGMSG_SZ-60] = '\0';
   }
   va_end( ap );

   sprintf( logmsg, "\nFATAL ERROR: %s  *****\n ", msg );
   ufdbLog( logmsg );
   if (ufdbGV.memoryAllocationErrors)
   {
      sprintf( logmsg, "FATAL ERROR: there were %d memory allocation errors  *****", 
               ufdbGV.memoryAllocationErrors );
      ufdbLog( logmsg );
   }

   if (ufdbGV.debug > 1)
   {
      /* only print to tty */
      if (fileno(stderr) >= 0  &&  isatty(fileno(stderr)))
      {
         fprintf( stderr, "%s\n", logmsg );
         fflush( stderr );
      }
   }

   ufdbGV.fatalError++;

   if (ufdbGV.crashOnFatal  &&  crashfun != (ufdb_crashfun_t) NULL)
   {
      fprintf( stderr, "ufdbGV.crashOnFatal is set.  crashing...\n" );
      fflush( stderr );
      (crashfun)();
   }
}


void ufdbRegisterFatalErrorCrashfun( ufdb_crashfun_t crfun )
{
   crashfun = crfun;
}


void UFDBcloseFilesNoLog( void )
{
   int i;
   int logfile;

   if (ufdbGV.debug)
      ufdbLogMessage( "UFDBcloseFilesNoLog: closing all files" );

   if (ufdbGlobalLogfile == NULL)
      logfile = fileno( stderr );
   else
      logfile = fileno( ufdbGlobalLogfile );

   for (i = 0;  i < 2*UFDB_MAX_THREADS + 32;  i++)
   {
      if (i != logfile)
	 (void) close( i );
   }
}


#ifdef __cplusplus
}
#endif
