/****** genTable3_1.c is optionally included by genTable.c ******/

// RCS $Id: genTable3_1.c,v 1.12 2023/02/06 16:31:40 root Exp root $


// XXX TODO: optimize all *.blogspot.* entries: keep blogspot.com and all other domains refer to blogspot.com


static uint32_t createMemTableIPv4_3_1( const struct Ip4Table * table )
{
   DEBUG(( stderr, "createMemTableIPv4_3_1\n" ));

   if (verbose || ufdbGV.debug)
      fprintf( stderr, "IPv4 table has %ld entries.\n", numIPv4 );

#if UFDB_DO_DEBUG
   if (ufdbGV.debug > 1)
   {
      // print content of whole IPv4 table
      int               i, chunki;
      int               n;
      struct Ip4Chunk * chunk;
      char              txt[32];

      n = 0;
      for (chunki = 0;  chunki < table->numChunks;  chunki++)
      {
         chunk = table->chunks[chunki];
         fprintf( stderr, "# chunk %d with %d IPs\n", chunki, chunk->numIps );
         for (i = 0;  i < chunk->numIps;  i++)
         {
            // convert host format to network format
            IPv4_txt( &chunk->ips[i], txt );
            fprintf( stderr, "%3d  %s\n", n, txt );
            n++;
         }
      }
      fprintf( stderr, "\n" );
   }
#endif

   int               i, j;
   uint32_t          totalIP;
   struct Ip4Chunk * chunk;

   totalIP = 0;
   for (i = 0;  i < table->numChunks;  i++)
      totalIP += table->chunks[i]->numIps;

   if (totalIP == 0)
      return 0;

   uint64_t bo = mem_alloc8( 8 +                                // table size padded to 8 bytes
                             totalIP * 2 * sizeof(uint32_t) );  // N * (IP address + mask)

   uint32_t * ts = (uint32_t*) (mem + bo);
   *ts++ = totalIP;
   *ts++ = 0;

   uint32_t * t = ts;

   for (j = 0;  j < table->numChunks;  j++)
   {
      chunk = table->chunks[j];
      for (i = 0;  i < chunk->numIps;  i++)
      {
         *t++ = chunk->ips[i].ip.s_addr;
         *t++ = 0xFFFFFFFF << (32 - chunk->ips[i].prefix);
      }
   }

   return (uint32_t) (bo >> 2);
}


// make optional index for IPv4 table with size 256 
// do not make an index iff there is a prefix < 8
static uint32_t createIPv4index_3_1( uint32_t ip4offset )
{
   unsigned char *     iptable;
   unsigned char *     ipindex;
   uint32_t   tableSize;
   uint32_t * ipp;
   int        numValues;
   int        i, j;
   uint8_t    values[256];
   uint32_t   ndx[256];

   if (ip4offset == 0)
      return 0;

   iptable = mem + (((uint64_t) ip4offset) << 2);
   tableSize = *((uint32_t*) iptable);
   DEBUG(( stderr, "createIPv4index_3_1  table size is %d\n", tableSize ));

   if (tableSize < 32)
   {
      DEBUG(( stderr, "   table size is %u so no index\n", tableSize ));
      return 0;
   }
   
   for (i = 0;  i < 256;  i++)
      ndx[i] = 0;

   ipp = (uint32_t*) (iptable + 2 * sizeof(uint32_t));
   numValues = 1;
   values[0] = (*ipp & 0xFF000000) >> 24;
   ndx[0] = 0;
   ipp++;
   if ((*ipp & 0xFF000000) != 0xFF000000)      // mask must have at least 8 bits
   {
      DEBUG(( stderr, "   first IP has mask <8 bits so no index\n" ));
      return 0;
   }
   ipp++;

   for (i = 1;  i < (int) tableSize;  i++)
   {
      values[numValues] = (*ipp & 0xFF000000) >> 24;
      ipp++;
      if ((*ipp & 0xFF000000) != 0xFF000000)      // mask must have at least 8 bits
      {
         DEBUG(( stderr, "   IP %d has mask <8 bits so no index\n", i+1 ));
         return 0;
      }
      ipp++;

      if (values[numValues] != values[numValues-1])     // only collect unique values
      {
         ndx[numValues] = (uint32_t) i;
         numValues++;
      }
   }
   for (i = 0;  i < numValues;  i++)
      DEBUG(( stderr, "      scanned:  value %-3u  ndx %05u\n", values[i], ndx[i] ));

   if (numValues < 8)
   {
      DEBUG(( stderr, "   number of unique first byte values is %d so no index\n", numValues ));
      return 0;
   }

   if (verbose || ufdbGV.debug)
      fprintf( stderr, "creating IPv4 index-256 for %d unique 1st byte values.  %d total #IPs\n",
                       numValues, tableSize );

   ipindex = (void*) (mem + mem_alloc4( (256+1) * sizeof(uint32_t)  ));
   ipp = (uint32_t*) ipindex;

   *ipp++ = 0;                       // first index elem points to first table elem
   j = 0;
   if (values[0] == 0)
      j = 1;
   DEBUG(( stderr, "      index-256 %-3d  ndx %05u   *start*\n", 0, 0 ));
   for (i = 1;  i < 256;  i++)
   {
      if (j == numValues)
      {
         *ipp = tableSize - 1;       // reached end-of-table
         DEBUG(( stderr, "      index-256 %-3d  ndx %05u   #%d   end\n", i, *ipp, j ));
      }
      else if (i == values[j])
      {
         *ipp = ndx[j];
         DEBUG(( stderr, "      index-256 %-3d  ndx %05u   #%d   == values[%d] %d\n", i, *ipp, j, j, values[j] ));
         j++;
      }
      else
      {
         *ipp = ndx[j];
         DEBUG(( stderr, "      index-256 %-3d  ndx %05u   #%d    < values[%d] %d\n", i, *ipp, j, j, values[j] ));
      }
      ipp++;
   }
   *ipp = tableSize - 1;        // the indices have one extra entry pointing to the end of table
   DEBUG(( stderr, "      index-256 %-3d  ndx %05u   *extra*\n", 256, *ipp ));

   return (uint32_t) ((ipindex - mem) >> 2);
}


// make optional index for IPv6 table with size 16, 32 or 256 
// do not make an index iff there is a prefix < 8
static uint32_t createIPv6index_3_1( uint32_t ip6offset )
{
   unsigned char *     iptable;
   unsigned char *     ipindex;
   uint32_t   tableSize;
   uint64_t * ip6p;
   uint32_t * ipp;
   int        numValues;
   int        i, j;
   uint8_t    values[256];
   uint32_t   ndx[256];

   if (ip6offset == 0)
      return 0;

   iptable = mem + (((uint64_t) ip6offset) << 2);
   tableSize = *((uint32_t*) iptable);
   DEBUG(( stderr, "createIPv6index_3_1  table size is %d\n", tableSize ));

   if (tableSize < 32)
   {
      DEBUG(( stderr, "   table size is %u so no index\n", tableSize ));
      return 0;
   }
   
   for (i = 0;  i < 256;  i++)
      ndx[i] = 0;

   ip6p = (uint64_t*) (iptable + 16);
   numValues = 1;
   values[0] = (*ip6p & 0xFF00000000000000) >> 56;
   ndx[0] = 0;
   ip6p += 2;
   if ((*ip6p & 0xFF00000000000000) != 0xFF00000000000000)      // mask must have at least 8 bits
   {
      DEBUG(( stderr, "   first IP has mask <8 bits so no index\n" ));
      return 0;
   }
   ip6p += 2;    // skip mask

   for (i = 1;  i < (int) tableSize;  i++)
   {
      values[numValues] = (*ip6p & 0xFF00000000000000) >> 56;
      ip6p += 2;
      if ((*ip6p & 0xFF00000000000000) != 0xFF00000000000000)      // mask must have at least 8 bits
      {
         DEBUG(( stderr, "   IP %d has mask <8 bits so no index\n", i+1 ));
         return 0;
      }
      ip6p += 2;    // skip mask

      if (values[numValues] != values[numValues-1])     // only collect unique values
      {
         ndx[numValues] = (uint32_t) i;
         numValues++;
      }
   }
   for (i = 0;  i < numValues;  i++)
      DEBUG(( stderr, "      scanned:  value %02x  ndx %-3u\n", values[i], ndx[i] ));

   if (numValues < 8)
   {
      DEBUG(( stderr, "   number of unique first byte values is %d so no index\n", numValues ));
      return 0;
   }

   if (verbose || ufdbGV.debug)
      fprintf( stderr, "creating IPv6 index-256 for %d unique 1st byte values.  %d total #IPs\n",
                       numValues, tableSize );

   DEBUG(( stderr, "   index-size is 256  (%d values)\n", numValues ));
   ipindex = (void*) (mem + mem_alloc4( (256+1) * sizeof(uint32_t)  ));
   ipp = (uint32_t*) ipindex;

   *ipp++ = 0;                 // first index elem points to first table elem
   DEBUG(( stderr, "      index-256 %02x  ndx %-3u\n", 0, 0 ));
   for (i = 1, j = 0;  i < 256;  i++)
   {
      if (j == numValues)
         *ipp = tableSize - 1;       // reached end-of-table
      else if (i < values[j])
         *ipp = ndx[j];
      else
      {
         *ipp = ndx[j];
         j++;
      }
      DEBUG(( stderr, "      index-256 %02x  ndx %-3u   #%d\n", i, *ipp, j ));
      ipp++;
   }
   *ipp = tableSize - 1;        // the indices have one extra entry pointing to the end of table

   if (verbose || ufdbGV.debug)
      fprintf( stderr, "IPv6 table index-256 created.\n" );

   return (uint32_t) ((ipindex - mem) >> 2);
}


static uint32_t createMemTableIPv6_3_1( const struct Ip6Table * table )
{
   DEBUG(( stderr, "createMemTableIPv6_3_1\n" ));

   if (verbose || ufdbGV.debug)
      fprintf( stderr, "IPv6 table has %ld entries.\n", numIPv6 );

#if UFDB_DO_DEBUG
   if (ufdbGV.debug > 1)
   {
      // print content of whole IPv6 table
      int               i, chunki;
      int               n;
      struct Ip6Chunk * chunk;
      char              txt[64];

      n = 0;
      for (chunki = 0;  chunki < table->numChunks;  chunki++)
      {
         chunk = table->chunks[chunki];
         fprintf( stderr, "# chunk %d with %d IPs\n", chunki, chunk->numIps );
         for (i = 0;  i < chunk->numIps;  i++)
         {
            // convert host format to network format
            IPv6_txt( &chunk->ips[i], txt );
            fprintf( stderr, "%3d  %s\n", n, txt );
            n++;
         }
      }
      fprintf( stderr, "\n" );
   }
#endif

   int               i, j;
   uint32_t          totalIP;
   struct Ip6Chunk * chunk;

   totalIP = 0;
   for (i = 0;  i < table->numChunks;  i++)
      totalIP += table->chunks[i]->numIps;

   if (totalIP == 0)
      return 0;

   uint64_t bo = mem_alloc32( 32 +                       // table size padded to 32 bytes
                              totalIP * 2*16 );          // N * (IP address + mask)

   uint32_t * ts = (uint32_t*) (mem + bo);
   *ts = totalIP;
   uint64_t * t = (uint64_t*) (mem + bo + 32);

   for (j = 0;  j < table->numChunks;  j++)
   {
      chunk = table->chunks[j];
      for (i = 0;  i < chunk->numIps;  i++)
      {
         uint64_t * p;
         p = (uint64_t*) &chunk->ips[i].ip;
         *t++ = p[0];
         *t++ = p[1];

         uint64_t mask1, mask2;
         int prefix = chunk->ips[i].prefix;
         mask1 = prefix > 64  ? 0xFFFFFFFFFFFFFFFF : 0xFFFFFFFFFFFFFFFF << (64 - prefix);
         mask2 = prefix <= 64 ? 0 : 0xFFFFFFFFFFFFFFFF << (128 - prefix);
         *t++ = mask1;
         *t++ = mask2;
      }
   }

   return (uint32_t) (bo >> 2);
}


static void createMemTable_3_1(
   const struct UFDBgentableNode * URLtable,
   const struct Ip4Table *         ip4table,
   const struct Ip6Table *         ip6table  )
{
   uint64_t bo = mem_alloc16( 8 * sizeof(uint32_t) );
   if (ufdbGV.debug)
      fprintf( stderr, "table offsets are at %ld\n", bo );

   uint32_t ip4offset = createMemTableIPv4_3_1( ip4table );
   uint32_t ip4ndxoffset = createIPv4index_3_1( ip4offset );
   uint32_t ip6offset = createMemTableIPv6_3_1( ip6table );
   uint32_t ip6ndxoffset = createIPv6index_3_1( ip6offset );

   if (verbose)
      fprintf( stderr, "URL table has %u nodes\n", URLtable->totalSubNodes );
   uint32_t urloffset = createMemTableURLs_3_0( URLtable, 0, UFDBunknownType, 0 );
   // MAYBE: make optional index for 1st level of URL table with size 32, 64 or 128 

   // store the offsets at the bo
   uint32_t * offsets = (uint32_t*) (mem + bo);
   offsets[0] = ip4offset;
   offsets[1] = ip4ndxoffset;
   offsets[2] = ip6offset;
   offsets[3] = ip6ndxoffset;
   offsets[4] = urloffset;
   offsets[5] = 0;
   offsets[6] = 0;
   offsets[7] = 0;
}

