/*
Copyright (C) 2003, Nik Reiman - nik@aboleo.net

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/****
  This module acts as a frontend to all other UBS daemons.  One ring to unite them all!  Ehhgh.  Anyways, it provides a somewhat user friendly shell which lets one view the status of the UBS, start and stop different modules, view log files, and other such things.  The shell can either run in interactive mode (if no arguments are given), in which case it takes in user input, produces output, and goes through a standard REPL loop (read, evaluate, print, loop) like a normall shell.  It can also run in non-interactive mode if arguments are given on the command line, in which case it runs the given command, prints the output to standard out, and exits.
  Web based frontends will probably end up using the non-interactive functionality of this program.  Graphical or system frontends (such as GTK) will probably be best to re-implement the functionality of this code on their own.
****/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <sys/types.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef WIN32
#include <windows.h>
#endif

#include "ubs-main.h"
#include "ubs-sched.h"
#include "ubs-event.h"

/*+
  Main program function.  If arguments are given, this program will exec them, placing the output on stdout.  Otherwise, it will launch an interactive shell.

  int main Returns OK on success

  int argc Number of arguments on the command line, including program name

  char *argv[] Arguments to the program
*/
int main(int argc, char *argv[])
{
 char buf[BIGBUF], *cmdv[MAXARGS];
 int i, cmdc;

 ubs_table_init(&GLOBAL);
 ubs_table_init(&SCHED);
 ubs_table_init(&EVENT);

 ubs_init(argv[0]);

 // set default values for stuff
 ubs_table_add(&SCHED, "miscdir", DEF_MISCDIR);
 ubs_table_add(&SCHED, "numshows", DEF_NUMSHOWS);
 ubs_table_add(&EVENT, "numevents", DEF_NUMEVENTS);

 if(read_config(NULL, "global", &GLOBAL)) {
  console_error("Can't read global settings in configuration file", FAIL);
 }
 // event specific context
 if(read_config(NULL, "event", &EVENT)) {
  console_error("Can't read event settings in configuration file", FAIL);
 }
 // schedule specific context
 if(read_config(NULL, "sched", &SCHED)) {
  console_error("Can't read sched settings in configuration file", FAIL);
 }

 // read in individual events
 if(atoi(ubs_table_data(&EVENT, "numevents")) != 0) {
  if(read_events(NULL)) {
   console_error("Error reading events from configuration file", FAIL);
  }
 }

 // read in the schedule of shows
 if(atoi(ubs_table_data(&SCHED, "numshows")) != 0) {
  if(read_sched(NULL)) {
   console_error("Error reading shows from configuration file", FAIL);
  }
 }

 chdir(ubs_table_data(&GLOBAL, "prefix"));

 init_week();

 for(i = 0; i < MAXARGS; i++) {
  cmdv[i] = (char *)malloc(STRBUF * sizeof(char));
 }

 // If there are arguments on the command line, then execute the
 // command and quit
 log_error_msg(LOG_DEBUG, "Launching non-interactive command");
 if(argc > 1) {
  for(i = 1; i < argc; i++) {
   strncpy(cmdv[i - 1], argv[i], STRBUF);
  }
  parse_command(argc - 1, cmdv);
  return OK;
 }

 fprintf(stdout, "Welcome to the UBS.  Use the 'help' command for more info.\n");
 while(1) {
#ifdef HAVE_CONFIG_H
  fprintf(stdout, "ubs-%s> ", VERSION);
#else
  fprintf(stdout, "ubs-win32> ");
#endif

  // clear out the command vector array
  for(i = 0; i < MAXARGS; i++) {
   bzero(cmdv[i], sizeof(char) * STRBUF);
  }
  cmdc = 0;
  optind = 1;
  if(fgets(buf, BIGBUF, stdin)) {
   chomp_line(buf);
   cmdc = split_line(cmdv, buf, STRBUF);
   if(parse_command(cmdc, cmdv) == QUIT) {
    break;
   }
  }
  else {
   break;
  }
 }

 for(i = 0; i < MAXARGS; i++) {
  free(cmdv[i]);
 }
 return OK;
}

/*+
  The UBS shell just reads in user input, and passes it off to this function to do the brunt of the parsing work.  This function expects to be given an array of values, and from there, it figures out which function to call.

  int parse_command Returns OK on success, QUIT if the command was "exit" or "quit", or otherwise simply returns whatever the respective command is set to return
+*/
int parse_command(int cmdc, char **cmdv)
{
 if(cmdc == 0) {
  return OK;
 }
 else if(!strcasecmp(cmdv[0], "exit") || !strcasecmp(cmdv[0], "quit")) {
  return QUIT;
 }
 else if(!strcasecmp(cmdv[0], "version")) {
  version();
  return OK;
 }
 else if(!strcasecmp(cmdv[0], "start")) {
  return cmd_start(cmdc, cmdv);
 }
 else if(!strcasecmp(cmdv[0], "stop")) {
  return cmd_stop(cmdc, cmdv);
 }
 else if(!strcasecmp(cmdv[0], "restart")) {
  return cmd_restart(cmdc, cmdv);
 }
 else if(!strcasecmp(cmdv[0], "status")) {
  return cmd_status(cmdc, cmdv);
 }
 else if(!strcasecmp(cmdv[0], "log")) {
  return cmd_log(cmdc, cmdv);
 }
 else if(!strcasecmp(cmdv[0], "queue")) {
  return cmd_queue(cmdc, cmdv);
 }
 else if(!strcasecmp(cmdv[0], "tag")) {
  return cmd_tag(cmdc, cmdv);
 }
 else if(!strcasecmp(cmdv[0], "shows")) {
  return cmd_shows(cmdc, cmdv);
 }
 else if(!strcasecmp(cmdv[0], "sched")) {
  return cmd_sched(cmdc, cmdv);
 }
 else if(!strcasecmp(cmdv[0], "search")) {
  return cmd_search(cmdc, cmdv);
 }
 else if(!strcasecmp(cmdv[0], "events")) {
  return cmd_events(cmdc, cmdv);
 }
 else if(!strcasecmp(cmdv[0], "help")) {
  return cmd_help(cmdc, cmdv);
 }
 else {
  fprintf(stdout, "Unknown command '%s'. Try 'help' for a list of known commands.\n",
         cmdv[0]);
 }

 return OK;
}
