/*
Copyright (C) 2003, Nik Reiman - nik@aboleo.net

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/****
  This file contains functions relevant to the schedule of shows.  This includes finding shows, printing things out, and adding shows.
****/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include "ubs-sched.h"

/*+
  Returns the index in the ALLSHOWS array for a given timeslot (weekday and time of day)

  int find_show Returns the index of the show, or -1 if no show is found

  int wday The weekday (0-6 for sunday-saturday, respectively)

  int timeslot The time of day.  Time is handled kind of strangely by the UBS.  Basically, it sees each day as being divided into 48 30-minute durations, and indexes them from 0-47.  So slot 0 would correspond to 12.00am-12.30am, slot 1 would be 12.30am-1.00am, and so on
+*/
int find_show(int wday, int timeslot)
{
 int i, id = (int)WEEK[wday][timeslot] / 10;

 for(i = 0; i < atoi(ubs_table_data(&SCHED, "numshows")); i++) {
  if(ALLSHOWS[i].show_id == id)
   return i;
 }

 return FAIL;
}

/*+
  Prints the current schedule to standard output.  Again, not used internally by the UBS.  This will go through each day, and for each timeslot will print out the show that occupies that slot.

  int dump_sched Returns OK on success 

  int format The output format
+*/
int dump_sched(int format) {
 int i, j, show_num = -1;

 for(i = 0; i < 7; i++) {
  for(j = 0; j < 48; j++) {
   if(format == ENGLISH) {
    show_num = find_show(i, j);
    if(show_num != -1) {
     fprintf(stdout, "Day %d, %02d:%02d: %s\n", i, j / 2, (j % 2) * 30, 
       ALLSHOWS[show_num].show_name);
    }
    else {
     fprintf(stdout, "Day %d, %02d:%02d: Default\n", i, j / 2, (j % 2) * 30);
    }
   }
   else if(format == CSV) {
    fprintf(stdout, "%d,%d,%d\n", i, j, WEEK[i][j]);
   }
   else if(format == HTML) {
    fprintf(stdout, "<tr><td>%d</td><td>%d:%02d</td><td>%d</td><td>%s</td></tr>\n",
      i, j / 2, (j % 2) * 30, WEEK[i][j], ALLSHOWS[show_num].show_name);
   }
  }
 }

 return OK;
}

/*+
  This sets a slot in the global schedule (the 'week' array).  The way the show ID's are done is kind of weird, and now that I'm going through this code much later, I realize that it is somewhat counter-intuitive.  Anyways, the ALLSHOWS array is 0-indexed, and corresponds to the "pure" show ID.  That is, if you want to get the information on the second show, you just access ALLSHOWS[1].  That being said, this information is also stored within the ALLSHOWS data structure in ALLSHOWS[].show_id .  This represents the "true" ID, _which starts at 1_ (that's rather important, mind you).  However, this information is stored differently within the actual schedule (yes, it would be just toooo easy to do it the obvious way, wouldn't it?).  The week data structure is simply a large 2-D matrix of single integer values, where each value is: (ID * 10) + (Frequency) . Thus, everything in the matrix is a 2-digit number, with the first digit being the actual ID of the show in question, and the second digit being the priority of the show.  Keep in mind that this ID is going to be one higher than the index in the ALLSHOWS array.  Yes, this is dumb.  Sorry. :/ This was done so that the scheduling algorithm could go through the week matrix and if two shows overlapped, then it would find out the priority and overwrite if necessary.

  int add_show Returns OK on success.

  int show_num The index of the show to add to the week.  This index corresponds to the show's place in the ALLSHOWS array.

  int wday The weekday to add the show in

  int timeslot The timeslot to add the show in
+*/
int add_show(int show_num, int wday, int timeslot)
{
 int cur = WEEK[wday][timeslot];

 if(cur == 0 || ALLSHOWS[show_num].repeat <= cur % 10) {
  WEEK[wday][timeslot] = (ALLSHOWS[show_num].show_id * 10) +
   ALLSHOWS[show_num].repeat;
 }

 return OK;
}

/*+
  Initializes the week array, setting everything to default values.

  int init_week Returns OK on success.
+*/
int init_week(void)
{
 int i, j;

 log_error_msg(LOG_DEBUG, "Initializing WEEK array");
 for(i = 0; i <= 6; i++) {
  memset(WEEK[i], 0, 47);
 }

 for(i = 0; i < atoi(ubs_table_data(&SCHED, "numshows")); i++) {
  for(j = ALLSHOWS[i].start; j < ALLSHOWS[i].end; j++) {
   if(ALLSHOWS[i].repeat == daily) {
    add_show(i, sun, j);
    add_show(i, mon, j);
    add_show(i, tues, j);
    add_show(i, wed, j);
    add_show(i, thurs, j);
    add_show(i, fri, j);
    add_show(i, sat, j);
   }
   else if(ALLSHOWS[i].repeat == weekly) {
    add_show(i, ALLSHOWS[i].when.weekday, j);
   }
   else if(ALLSHOWS[i].repeat == once) {
    // erm do this later
   }
  }
 }

 return OK;
}
