/*
Copyright (C) 2003, Nik Reiman - nik@aboleo.net

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/****
  Runs an event for the ubs-event daemon.  This consists of finding the event, exec'ing the event out, and sending a signal to the ubs-play daemon to let it know it has to play a special event.
****/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <errno.h>
#include <signal.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include "ubs-event.h"

/*+
  Runs an event, based on the index in the ALLEVENTS array.

  int run_event Returns OK on success, NO_FORK for forking errors, NO_FILE if the .pid file can't be opened

  int event_num The index of the ALLSHOWS array to run
+*/
int run_event(int event_num)
{
 int pid;
 char buf[FILEBUF];
 FILE *fp;

 if((pid = fork()) < 0) {
  log_error_msg(LOG_ERROR, "Can't create fork");
  return NO_FORK;
 }
 // child process
 else if(pid == 0) {
  snprintf(buf, FILEBUF, "bin/%s", ALLEVENTS[event_num].name);
  exec_proc(buf, NULL);
  log_error_msg(LOG_DEBUG, "exec_proc returned");
  return NO_FORK;
 }
 // parent process
 else {
  if(waitpid(pid, NULL, 0) < 0) {
   log_error_msg(LOG_DEBUG, "Waitpid failed");
   return NO_FORK;
  }
  // signal the ubs-play daemon
  // eventually this will not be hardcoded
  if((fp = fopen("var/ubs-play.pid", "r")) == NULL) {
   log_error_msg(LOG_ERROR, "Unable to open ubs-play.pid");
   return NO_FILE;
  }

  if(fgets(buf, BUFSIZ, fp) == NULL) {
   log_error_msg(LOG_ERROR, "Unable to read from ubs-play.pid");
   return NO_FILE;
  }

  pid = (pid_t)atoi(buf);
  fclose(fp);

  if(kill(pid, SIGUSR1) < 0) {
   if(errno == ESRCH) {
    log_error_msg(LOG_ERROR, "Could not send event signal to ubs-play");
   }
   else if(errno == EPERM) {
    log_error_msg(LOG_ERROR, "Permission denied in sending event signal to ubs-play");
   }
  }
  else {
   log_error_msg(LOG_DEBUG, "Sent event signal to ubs-play");
  }

  // log this event in the event's specific logfile,
  // not the global logfiles
  log_event(ALLEVENTS[event_num].logfile, "status", ALLEVENTS[event_num].name, "Event completed", LOG_STATUS);
 }

 return OK;
}
