/*
Copyright (C) 2003, Nik Reiman - nik@aboleo.net

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/****
  Shows the status of the UBS daemons.
****/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <dirent.h>
#include <sys/types.h>
#include <errno.h>

#include "ubs-main.h"
#include "ubs.h"

/*+
  Shows the status.  Basically equivilent to displaying all the "ubs-*.status" files.

  int cmd_status Returns OK on success, NO_DIR if the bin/ directory can't be opened, or otherwise, whatever sub_status_proc() returns.

  int cmdc Number of arguments

  char **cmdv Argument vector
+*/
int cmd_status(int cmdc, char **cmdv) {
 DIR *dp;
 struct dirent *ds;
 int ch, format = ENGLISH;

 while((ch = getopt(cmdc, cmdv, "hwecs")) != -1) {
  switch(ch) {
   case 'e':
    format = ENGLISH;
    break;
   case 'c':
    format = CSV;
    break;
   case 'w':
    format = HTML;
    break;
   case 's':
    format = SILENT;
   case 'h':
    sub_help_cmd("status");
    return OK;
    break;
   default:
    sub_help_cmd("status");
    return OK;
    break;
  }
 }

 cmdc -= optind;
 cmdv += optind;

 if(cmdc == 0 || !strcasecmp(cmdv[0], "all")) {
  // Status for all.
  if((dp = opendir("bin")) == NULL) {
   printf("Unable to open bin/ directory\n");
   return NO_DIR;
  }

  while((ds = readdir(dp)) != NULL) {
   if(!strncmp(ds->d_name, "ubs-", 4)) {
    sub_status_proc(ds->d_name, format);
   }
  }
  closedir(dp);
 }
 else {
  sub_status_proc(cmdv[0], format);
 }
 
 return OK;
}

/*+
  Queries a ubs daemon to see what the current status is.  This is done by first opening up the status file (var/PROCNAME.pid), and then attempting to ping_pid() that process.  If this can be done, then the nifty little status file (var/PROCNAME.status) is read and redisplayed.

  int sub_status_proc Returns OK on success.  Success is relative, here.  Even if the pid file can't be opened, technically, this function has figured out the status, and thus, it was somewhat successful.

  char *proc The process name to query.

  int format The output format.
+*/
int sub_status_proc(char *proc, int format) {
 FILE *fp;
 char buf[STRBUF];

 snprintf(buf, STRBUF, "var/%s.pid", proc);
 if((fp = fopen(buf, "r")) == NULL) {
  if(format == ENGLISH) {
   printf("%s: Not Running\n", proc);
  }
  else if(format == CSV) {
   printf("%s,Not Running\n", proc);
  }
  else if(format == HTML) {
   printf("<tr><td>%s</td><td>Not Running</td></tr>\n", proc);
  }

  return OK;
 }
 if(fgets(buf, BUFSIZ, fp) == NULL) {
  if(format == ENGLISH) {
   printf("%s: Not Running\n", proc);
  }
  else if(format == CSV) {
   printf("%s,Not Running\n", proc);
  }
  else if(format == HTML) {
   printf("<tr><td>%s</td><td>Not Running</td></tr>\n", proc);
  }

  fclose(fp);
  return OK;
 }
 if(ping_pid(proc) != OK) {
  if(errno == ESRCH) {
   if(format == ENGLISH) {
    printf("%s: Exited Unexpectedly\n", proc);
   }
   else if(format == CSV) {
    printf("%s,Exited Unexpectedly\n", proc);
   }
   else if(format == HTML) {
    printf("<tr><td>%s</td><td>Exited Unexpectedly</td></tr>\n", proc);
   }
  }
  else if(errno == EPERM) {
   if(format == ENGLISH) {
    printf("%s: Permission Denied\n", proc);
   }
   else if(format == CSV) {
    printf("%s,Permission Denied\n", proc);
   }
   else if(format == HTML) {
    printf("<tr><td>%s</td><td>Permission Denied</td></tr>\n", proc);
   }
  }
 }
 else {
  snprintf(buf, STRBUF, "var/%s.status", proc);
  if((fp = fopen(buf, "r")) == NULL) {
   if(format == ENGLISH) {
    printf("%s: No Information\n", proc);
   }
   else if(format == CSV) {
    printf("%s,No Information\n", proc);
   }
   else if(format == HTML) {
    printf("<tr><td>%s</td><td>No Information</td></tr>\n", proc);
   }
   return OK;
  }
  if(fgets(buf, BUFSIZ, fp) == NULL) {
   if(format == ENGLISH) {
    printf("%s: No Information\n", proc);
   }
   else if(format == CSV) {
    printf("%s,No Information\n", proc);
   }
   else if(format == HTML) {
    printf("<tr><td>%s</td><td>No Information</td></tr>\n", proc);
   }
   fclose(fp);
   return OK;
  }
  if(format == ENGLISH) {
   printf("%s: %s\n", proc, buf);
  }
  else if(format == CSV) {
   printf("%s,%s\n", proc, buf);
  }
  else if(format == HTML) {
   printf("<tr><td>%s</td><td>%s</td></tr>\n", proc, buf);
  }
  fclose(fp);
  return OK;
 }
 fclose(fp);
 return OK;
}
