/*
Copyright (C) 2003, Nik Reiman - nik@aboleo.net

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/****
  This will start the UBS daemons.  If a specific daemon is specified on the command line, it will start that one.  Otherwise, it starts all known daemons (which are known by their name, which is in the form ubs-*).
****/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <dirent.h>
#include <sys/types.h>
#include <errno.h>

#include "ubs-main.h"
#include "ubs.h"

/*+
  Starts up one of the UBS modules, forking it into the background.

  int cmd_start Returns OK on success, NO_DIR if bin/ can't be read, or otherwise returns whatever sub_start_proc() returns.

  int cmdc Number of arguments

  char **cmdv Argument vector
+*/
int cmd_start(int cmdc, char **cmdv) {
 DIR *dp;
 struct dirent *ds;
 int ch, format = ENGLISH;
 
 while((ch = getopt(cmdc, cmdv, "hwces")) != -1) {
  switch(ch) {
   case 'e':
    format = ENGLISH;
    break;
   case 'c':
    format = CSV;
    break;
   case 'w':
    format = HTML;
    break;
   case 's':
    format = SILENT;
    break;
   case 'h':
    sub_help_cmd("start");
    return OK;
    break;
   default:
    sub_help_cmd("start");
    return OK;
    break;
  }
 }

 cmdc -= optind;
 cmdv += optind;

 if(cmdc == 0 || !strcasecmp(cmdv[0], "all")) {
  // start all.  To do this, we basically find all the files
  // in the bin directory that match some form of ubs-*, and exec
  // each one of those.  Kind of weird.  But it works, so shut up.
  if((dp = opendir("bin")) == NULL) {
   printf("Unable to open bin/ directory\n");
   return NO_DIR;
  }

  while((ds = readdir(dp)) != NULL) {
   if(!strncmp(ds->d_name, "ubs-", 4)) {
    sub_start_proc(ds->d_name, format);
   }
  }

  closedir(dp);
 }
 // just start a process as specified on the command line
 else {
  return sub_start_proc(cmdv[0], format);
 }
 
 return OK;
}

/*+
  Subfunction to start a process, where process is considered to be in the form PREFIX/bin/NAME.

  int sub_start_proc Returns OK on success, FAIL if the process is already running, NO_FORK if the process can't be exec'd.

  char *proc The process name
+*/
int sub_start_proc(char *proc, int format) {
 char target[FILEBUF];
 int res;

 res = ping_pid(proc);
 if(res == OK) {
  if(format == ENGLISH) {
   printf("%s: Already Running\n", proc);
  }
  else if(format == CSV) {
   printf("%s,Already Running\n", proc);
  }
  else if(format == HTML) {
   printf("<tr><td>%s</td><td>Already Running</td></tr>\n", proc);
  }
  return FAIL;
 }
 else {
  snprintf(target, FILEBUF, "bin/%s", proc);
  if(!fork()) {
   execlp(target, target, NULL);

   // should not reach here
   printf("Error: Could not exec process for %s\n", proc);
   exit(NO_FORK);
  }
  else {
   // kill the zombie process
   signal(SIGCHLD, SIG_IGN);
   if(format == ENGLISH) {
    printf("%s: Started\n", proc);
   }
   else if(format == CSV) {
    printf("%s,Started\n", proc);
   }
   else if(format == HTML) {
    printf("<tr><td>%s</td><td>Started</td></tr>\n", proc);
   }
   return OK;
  }
 }
}
