/*
Copyright (C) 2003, Nik Reiman - nik@aboleo.net

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/****
  Functions for a table data structure, which is similar to a hash (only with no indexing).
****/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include "ubs.h"

/*+
  Creates memory and initializes a new ubs_table data structure.  This function must be called before the ubs_table can be used.

  int ubs_table Returns OK on success, NO_MEM if memory allocation failed

  ubs_table *t Pointer to the table to be initialized
+*/
int ubs_table_init(ubs_table *t) {
 t->num_values = 0;
 if((t->values = (keypair *)malloc(sizeof(keypair))) == NULL) {
  return NO_MEM;
 }

 return OK;
}

/*+
  Adds a new keypair (key, value) to a table

  int ubs_table_add Returns OK on success, NO_MEM if memory allocation fails

  ubs_table *t Table to add the keypair to

  char *key The key

  char *value Value for the key
+*/
int ubs_table_add(ubs_table *t, char *key, char *value) {
 int i;

 for(i = 0; i < t->num_values; i++) {
  // if the key already exists in the table, overwrite.  Tough luck.
  if(!strcasecmp(t->values[i].key, key)) {
   strncpy(t->values[i].key, key, STRBUF);
   strncpy(t->values[i].value, value, BIGSTR);
   return OK;
  }
 }

 // this should only be encountered if the loop finishes running
 // and finds no existing key
 if((t->values = (keypair *)realloc(t->values, (t->num_values + 1) *
     (sizeof(keypair)))) == NULL) {
  return NO_MEM;
 }
 strncpy(t->values[t->num_values].key, key, STRBUF);
 strncpy(t->values[t->num_values].value, value, BIGSTR);
 t->num_values++;

 return OK;
}

/*+
  Returns the value for a key in a ubs_table

  char *ubs_table_data Returns the string corresponding to the key passed in, or NULL if no such key exists

  ubs_table *t Table to read from

  char *key The key to seek
+*/
char *ubs_table_data(ubs_table *t, char *key) {
 int i;
 
 for(i = 0; i < t->num_values; i++) {
  if(!strcasecmp(t->values[i].key, key)) {
   return t->values[i].value;
  }
 }

 return NULL;
}

/*+
  Checks to see if a key is present in the table

  int ubs_table_exists Returns YES if the key is found, NO if not

  ubs_table *t Table to read from

  char *key The key to seek
+*/
int ubs_table_exists(ubs_table *t, char *key) {
 int i;

 for(i = 0; i < t->num_values; i++) {
  if(!strcasecmp(t->values[i].key, key)) {
   return YES;
  }
 }

 return NO;
}

/*+
  Changes the value of a key in a ubs_table structure

  int ubs_table_change Returns OK if successful, FAIL if the key does not exist in the table already

  ubs_table *t Table to read from

  char *key The key to change the value for

  char *newval New value for this key
+*/
int ubs_table_change(ubs_table *t, char *key, char *newval) {
 int i;

 for(i = 0; i < t->num_values; i++) {
  if(!strcasecmp(t->values[i].key, key)) {
   strncpy(t->values[i].value, newval, BIGSTR);
   return OK;
  }
 }

 return FAIL;
}

int ubs_table_remove(ubs_table *t, char *key) {
 // heh

 return OK;
}
