/*
Copyright (C) 2003, Nik Reiman - nik@aboleo.net

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/****
  Functions for controlling the queue.  This includes, viewing, popping, pushing, chopping, and other fun stuff.
****/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include "ubs.h"

/*+
  Grabs the first line from the queue, but does not remove this line from the queue file

  void queue_pop Returns nothing

  char *str A character array where the first line in the queue will be stored
+*/
void queue_pop(char *str) {
 FILE *fp;

 if((fp = fopen(ubs_table_data(&GLOBAL, "queue"), "r")) == NULL) {
  log_error_msg(LOG_ERROR, "Unable to open queue for reading in queue_pop");
  return;
 }
 fgets(str, BIGBUF, fp);
 fclose(fp);

 return;
}

/*+
  Removes the top line in the queue

  void queue_chop Returns nothing; takes no arguments
+*/
void queue_chop(void) {
 FILE *fp_orig, *fp_new;
 char line_buf[FILEBUF], tmpqueue[FILEBUF];

 if((fp_orig = fopen(ubs_table_data(&GLOBAL, "queue"), "r")) == NULL) {
  log_error_msg(LOG_ERROR, "Unable to open queue for reading in queue_chop");
  return;
 }

 snprintf(tmpqueue, FILEBUF, "tmp/%d", rand());
 log_error_msg(LOG_DEBUG, "Creating temporary file '%s'", tmpqueue);
 if((fp_new = fopen(tmpqueue, "w")) == NULL) {
  log_error_msg(LOG_ERROR, "Error creating temporary file");
  fclose(fp_orig);
  return;
 }

 // grab the first line
 fgets(line_buf, FILEBUF, fp_orig);
 while(!feof(fp_orig)) {
  if(fgets(line_buf, FILEBUF, fp_orig) == NULL) {
   break;
  }
  fprintf(fp_new, "%s", line_buf);
 }

 fclose(fp_new);
 fclose(fp_orig);
 log_error_msg(LOG_DEBUG, "Moving temporary file to queue");
 rename(tmpqueue, ubs_table_data(&GLOBAL, "queue"));
 return;
}

/*+
  Initializes the queue, by zero'ing it out

  void queue_init Returns nothing, takes no arguments
+*/
void queue_init(void) {
 FILE *fp;

 if((fp = fopen(ubs_table_data(&GLOBAL, "queue"), "w")) == NULL) {
  log_error_msg(LOG_ERROR, "Cannot open queue for writing in queue_init");
  return;
 }

 fclose(fp);
 return;
}

/*+
  Returns the number of files in the queue

  int queue_length Returns number of files in the queue, or -1 if the queue can't be opened.
+*/
int queue_length(void) {
 FILE *fp;
 int num = 0;
 char junk[BIGBUF];

 if((fp = fopen(ubs_table_data(&GLOBAL, "queue"), "r")) == NULL) {
  log_error_msg(LOG_ERROR, "Cannot open queue for reading in queue_length");
  return NO_FILE;
 }

 while(fgets(junk, BIGBUF, fp)) {
  num++;
 }

 fclose(fp);
 return num;
}

/*+
  Adds a line to the end of the queue

  int queue_push Returns OK on success, NO_FILE if the queue can't be opened

  char *filename The string to be added to the queue
+*/
int queue_push(char *filename) {
 FILE *fp;

 if((fp = fopen(ubs_table_data(&GLOBAL, "queue"), "a")) == NULL) {
  log_error_msg(LOG_ERROR, "Cannot open queue for writing in queue_push");
  return NO_FILE;
 }

 fprintf(fp, "%s\n", filename);

 fclose(fp);
 return OK;
}

/*+
  Works like push, but adds the string to the very front of the queue rather than the end.

  int queue_prepush Returns OK on success, NO_FILE if the queue can't be opened

  char *filename The string to be added to the queue
+*/
int queue_prepush(char *filename) {
 FILE *fp_orig, *fp_new;
 char junk[BIGBUF], tmpqueue[FILEBUF];

 if((fp_orig = fopen(ubs_table_data(&GLOBAL, "queue"), "r")) == NULL) {
  log_error_msg(LOG_ERROR, "Cannot open queue for reading in queue_prepush");
  return NO_FILE;
 }

 snprintf(tmpqueue, FILEBUF, "tmp/%d", rand());
 log_error_msg(LOG_DEBUG, "Creating temporary file '%s'\n", tmpqueue);
 if((fp_new = fopen(tmpqueue, "w")) == NULL) {
  log_error_msg(LOG_ERROR, "Cannot open queue for writing in queue_prepush");
  fclose(fp_orig);
  return NO_FILE;
 }

 fprintf(fp_new, "%s\n", filename);
 while(fgets(junk, BIGBUF, fp_orig)) {
  fprintf(fp_new, "%s", junk);
 }
 
 fclose(fp_new);
 fclose(fp_orig);
 log_error_msg(LOG_DEBUG, "Moving temporary file to queue");
 rename(tmpqueue, ubs_table_data(&GLOBAL, "queue"));

 return OK;
}
