/*
Copyright (C) 2003, Nik Reiman - nik@aboleo.net

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/****
  This file contains functions for breaking up lines, which is used primarily in parsing the configuration files.  It also contains other random string functions, like chomp and such.
****/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "ubs-sched.h"

/*+
  Breaks up a line into two parts, which are separated by a '=' character.  An arbitrary amount of whitespace is permitted.  If the line is blank, or it is a comment (contains a '#' character...a slight bug), it returns.

  int parse_line Returns OK if the line is blank, OK if the line is a comment, OK if the line could be parsed, FAIL if nothing was found 

  char *line_buf The line to parse

  char *directive The character array to store our results in for the directive

  char *argument The character array to store our results in for the argument
+*/
int parse_line(char *line_buf, char *directive, char *argument)
{
 char *p1;

 bzero(directive, STRBUF);
 bzero(argument, STRBUF);

 // look for comments or blank lines
 if((p1 = strchr(line_buf, '#')) != NULL) {
  return OK;
 }
 else if(line_buf[0] == '\0') {
  return OK;
 }
 // look for a directive/argument combination, like
 // foo = bar
 // where "foo" is some kind of directive, and "bar" is the argument
 else if((p1 = strchr(line_buf, '=')) != NULL) {
  // grab the directive and arg
  strncpy(directive, line_buf, p1 - line_buf - 1);
  strncpy(argument, p1 + 1, (strlen(line_buf) + line_buf) - p1 - 1);
  if(argument[0] == '\0' || directive[0] == '\0') {
   // found nothing
   return FAIL;
  }
  // get rid of pesky whitespace
  weed_white(directive);
  weed_white(argument);

  // code for finding a directive/argument combo
  return YES;
 }
 else { 
  // error code
  return QUIT;
 }

 return OK;
}

/*+
  Strips leading and trailing whitespace from a string (I think).

  int weed_white Returns OK on success

  char *str A string to strip.  Will be modified directly
+*/
int weed_white(char *str) {
 char buf[BIGBUF];
 int i, word_st = 0, quote = 0;

 bzero(buf, BIGBUF);

 for(i = 0; i < strlen(str); i++) {
  if(word_st == 0 && !blank(str[i])) {
   if(str[i] == '\'' || str[i] == '"') {
    quote = 1;
   }
   else {
    buf[0] = str[i];
   }
   word_st = 1;
  }
  else if(word_st == 1 && !blank(str[i])) {
   if(quote) {
    if(str[i] == '\'' || str[i] == '"') {
     break;
    }
    else {
     strncat(buf, str + i, 1);
    }
   }
   else {
    strncat(buf, str + i, 1);
   }
  }
  else if(word_st == 1 && blank(str[i])) {
   if(quote) {
    strncat(buf, str + i, 1);
   }
   else {
    break;
   }
  }
 }
 strncpy(str, buf, strlen(str));

 return OK;
}

/*+
  Behaves like perl's 'chomp' function.  If the last character of a string is a newline, it removes it.

  void chomp_line Returns nothing

  char *line The line to chomp
+*/
void chomp_line(char *line) {
 if(line[strlen(line) - 1] == '\n') {
  line[strlen(line) - 1] = '\0';
 }
 return;
}

/*+
  This does the same as "isblank()", but is included here for portability

  int blank Returns YES if the character is a space, tab, or newline, NO for any other character

  char ch A character
+*/
int blank(char ch)
{
 if(ch == ' ' || ch == '\t' || ch == '\n') {
  return YES;
 }
 else {
  return NO;
 }
}

/*+
  Tries to emulate perl's split function.  It's a bit sloppy, but it seems to work.

  int split_line Returns the number of arguments
  
  char **dest_arr A pointer to a character array where the results of the split will be stored.  This function does not malloc any of this memory out; the caller needs to do that ahead of time.

  char *line The line to be parsed

  int max Maximum number of characters to copy into each array entry
+*/
int split_line(char **dest_arr, char *line, int max) {
 int i, state = 0, start_pt = 0, num = 0;

 for(i = 0; i < strlen(line); i++) {
  if(state == 0 && !blank(line[i])) {
   state = 1;
   start_pt = i;
  }
  else if(state == 1 && blank(line[i])) {
   state = 0;
   strncpy(dest_arr[num], line + start_pt, i - start_pt);
   num++;
  }
  else {
   // do nothing
  }
 }

 if(state) {
  strncpy(dest_arr[num], line + start_pt, max);
  num++;
 }

 return num;
}

/*+
  A cheap hack for a function that should really be in string.h (only the GNU libc seems to have it).  This function does the same thing as strstr, except that it is case insensitive.

  int ubs_strcasestr Returns NO if no match is found, YES if a match is found
  
  char *haystack The string to be searched
  
  char *needle The string to locate within the 'haystack'
+*/
int ubs_strcasestr(char *haystack, char *needle) {
 int i;
 for(i = 0; i < strlen(haystack); i++) {
  if(!strncasecmp(haystack + i, needle, strlen(needle))) {
   return YES;
  }
 }
 return NO;
}
