/****************************************************************************/
/*                                                                          */
/*               TPM Specific HMAC routines                                 */
/*                                                                          */
/* This file is copyright 2003 IBM. See "License" for details               */
/****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <netinet/in.h>
#include <tcpa.h>
#include <hmac.h>
#include <openssl/sha.h>
#include <openssl/hmac.h>

/****************************************************************************/
/*                                                                          */
/* Validate the HMAC in an AUTH1 response                                   */
/*                                                                          */
/* This function validates the Authorization Digest for all AUTH1           */
/* responses.                                                               */
/*                                                                          */
/* The arguments are...                                                     */
/*                                                                          */
/* buffer - a pointer to response buffer                                    */
/* command - the command code from the original request                     */
/* ononce - a pointer to a 20 byte array containing the oddNonce            */
/* key    - a pointer to the key used in the request HMAC                   */
/* keylen - the size of the key                                             */
/* followed by a variable length set of arguments, which must come in       */
/* pairs.                                                                   */
/* The first value in each pair is the length of the data in the second     */
/*   argument of the pair                                                   */
/* The second value in each pair is an offset in the buffer to the data     */
/*   to be included in the hash for the paramdigest                         */
/* There should NOT be pairs for the TCPA_RESULT or TCPA_COMMAND_CODE       */
/* The last pair must be followed by a pair containing 0,0                  */
/*                                                                          */
/****************************************************************************/
int checkhmac1(unsigned char *buffer, uint32_t command,
               unsigned char *ononce, unsigned char *key, int keylen, ...)
{
    uint32_t bufsize;
    uint16_t tag;
    uint32_t ordinal;
    uint32_t result;
    unsigned char *enonce;
    unsigned char *continueflag;
    unsigned char *authdata;
    unsigned char testhmac[20];
    unsigned char paramdigest[20];
    SHA_CTX sha;
    unsigned int dlen;
    unsigned int dpos;
    va_list argp;

    bufsize = htonl(*(uint32_t *) (buffer + 2));
    tag = htons(*(uint16_t *) (buffer + 0));
    ordinal = ntohl(command);
    result = *(uint32_t *) (buffer + TCPA_RETURN_OFFSET);

    if (tag == TPM_TAG_RSP_COMMAND)
        return 0;
    if (tag != TPM_TAG_RSP_AUTH1_COMMAND)
        return -1;
    authdata = buffer + bufsize - TCPA_HASH_SIZE;
    continueflag = authdata - 1;
    enonce = continueflag - TCPA_NONCE_SIZE;
    SHA1_Init(&sha);
    SHA1_Update(&sha, &result, 4);
    SHA1_Update(&sha, &ordinal, 4);
    va_start(argp, keylen);
    for (;;) {
        dlen = (unsigned int) va_arg(argp, unsigned int);
        if (dlen == 0)
            break;
        dpos = (unsigned int) va_arg(argp, unsigned int);
        SHA1_Update(&sha, buffer + dpos, dlen);
    }
    va_end(argp);
    SHA1_Final(paramdigest, &sha);
    rawhmac(testhmac, key, keylen, TCPA_HASH_SIZE, paramdigest,
            TCPA_NONCE_SIZE, enonce,
            TCPA_NONCE_SIZE, ononce, 1, continueflag, 0, 0);
    if (memcmp(testhmac, authdata, dlen) != 0)
        return -1;
    return 0;
}

/****************************************************************************/
/*                                                                          */
/* Calculate HMAC value for an AUTH1 command                                */
/*                                                                          */
/* This function calculates the Authorization Digest for all OIAP           */
/* commands.                                                                */
/*                                                                          */
/* The arguments are...                                                     */
/*                                                                          */
/* digest - a pointer to a 20 byte array that will receive the result       */
/* key    - a pointer to the key to be used in the HMAC calculation         */
/* keylen - the size of the key in bytes                                    */
/* h1     - a pointer to a 20 byte array containing the evenNonce           */
/* h2     - a pointer to a 20 byte array containing the oddNonce            */
/* h3     - an unsigned character containing the continueAuthSession value  */
/* followed by a variable length set of arguments, which must come in       */
/* pairs.                                                                   */
/* The first value in each pair is the length of the data in the second     */
/*   argument of the pair                                                   */
/* The second value in each pair is a pointer to the data to be hashed      */
/*   into the paramdigest.                                                  */
/* The last pair must be followed by a pair containing 0,0                  */
/*                                                                          */
/****************************************************************************/
int authhmac(unsigned char *digest, unsigned char *key,
             unsigned int keylen, unsigned char *h1, unsigned char *h2,
             unsigned char h3, ...)
{
    unsigned char paramdigest[TCPA_HASH_SIZE];
    SHA_CTX sha;
    unsigned int dlen;
    unsigned char *data;
    unsigned char c;

    va_list argp;

    SHA1_Init(&sha);
    if (h1 == NULL || h2 == NULL)
        return -1;
    c = h3;
    va_start(argp, h3);
    for (;;) {
        dlen = (unsigned int) va_arg(argp, unsigned int);
        if (dlen == 0)
            break;
        data = (unsigned char *) va_arg(argp, int);
        if (data == NULL)
            return -1;
        SHA1_Update(&sha, data, dlen);
    }
    va_end(argp);
    SHA1_Final(paramdigest, &sha);
    rawhmac(digest, key, keylen, TCPA_HASH_SIZE, paramdigest,
            TCPA_NONCE_SIZE, h1, TCPA_NONCE_SIZE, h2, 1, &c, 0, 0);
    return 0;
}

/****************************************************************************/
/*                                                                          */
/* Calculate Raw HMAC value                                                 */
/*                                                                          */
/* This function calculates an HMAC digest                                  */
/*                                                                          */
/* The arguments are...                                                     */
/*                                                                          */
/* digest - a pointer to a 20 byte array that will receive the result       */
/* key    - a pointer to the key to be used in the HMAC calculation         */
/* keylen - the size of the key in bytes                                    */
/* followed by a variable length set of arguments, which must come in       */
/* pairs.                                                                   */
/* The first value in each pair is the length of the data in the second     */
/*   argument of the pair                                                   */
/* The second value in each pair is a pointer to the data to be hashed      */
/*   into the paramdigest.                                                  */
/* The last pair must be followed by a pair containing 0,0                  */
/*                                                                          */
/****************************************************************************/
int rawhmac(unsigned char *digest, unsigned char *key,
            unsigned int keylen, ...)
{
    HMAC_CTX hmac;
    unsigned int dlen;
    unsigned char *data;
    va_list argp;

    HMAC_Init(&hmac, key, keylen, EVP_sha1());
    va_start(argp, keylen);
    for (;;) {
        dlen = (unsigned int) va_arg(argp, unsigned int);
        if (dlen == 0)
            break;
        data = (unsigned char *) va_arg(argp, int);
        if (data == NULL)
            return -1;
        HMAC_Update(&hmac, data, dlen);
    }
    HMAC_Final(&hmac, digest, &dlen);
    HMAC_cleanup(&hmac);
    va_end(argp);
    return 0;
}

/****************************************************************************/
/*                                                                          */
/* Perform a SHA1 hash on a single buffer                                   */
/*                                                                          */
/****************************************************************************/
void sha1(unsigned char *input, int len, unsigned char *output)
{
    SHA_CTX sha;

    SHA1_Init(&sha);
    SHA1_Update(&sha, input, len);
    SHA1_Final(output, &sha);
}
