/* tn5250 -- an implentation of the 5250 telnet protocol.
 * Copyright (C) 1997 Michael Madore
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "config.h"
#include <windows.h>
#include <winsock.h>
#include <stdlib.h>
#include <stdio.h>
#include "utility.h"
#include "displaybuf.h"
#include "terminal.h"
#include "winterm.h"

#define WM_SOCKET_RECV (WM_USER+101)

static HINSTANCE g_hInstance;

#define A_5250_GREEN    0x000
#define A_5250_WHITE    0x100
#define A_5250_RED      0x200
#define A_5250_TURQ     0x300
#define A_5250_YELLOW   0x400
#define A_5250_PINK     0x500
#define A_5250_BLUE     0x600
#define A_5250_BLACK	0x700

#define A_COLOR_MASK	0xf00

#define A_REVERSE	0x1000
#define A_UNDERLINE	0x2000
#define A_BLINK		0x4000
#define A_VERTICAL	0x8000

static int attribute_map[] = { A_5250_GREEN,
                   A_5250_GREEN | A_REVERSE,
                   A_5250_WHITE,
                   A_5250_WHITE | A_REVERSE,
                   A_5250_GREEN | A_UNDERLINE,
                   A_5250_GREEN | A_UNDERLINE | A_REVERSE,
                   A_5250_WHITE | A_UNDERLINE,
                   0x00,
                   A_5250_RED,
                   A_5250_RED | A_REVERSE,
                   A_5250_RED | A_BLINK,
                   A_5250_RED | A_BLINK | A_REVERSE,
                   A_5250_RED | A_UNDERLINE,
                   A_5250_RED | A_UNDERLINE | A_REVERSE,
                   A_5250_RED | A_UNDERLINE | A_BLINK,
                   0x00,
                   A_5250_TURQ | A_VERTICAL,
                   A_5250_TURQ | A_VERTICAL | A_REVERSE,
                   A_5250_YELLOW | A_VERTICAL,
                   A_5250_YELLOW | A_VERTICAL | A_REVERSE,
                   A_5250_TURQ | A_UNDERLINE | A_VERTICAL,
                   A_5250_TURQ | A_UNDERLINE | A_REVERSE | A_VERTICAL,
                   A_5250_YELLOW | A_UNDERLINE | A_VERTICAL,
                   0x00,
                   A_5250_PINK,
                   A_5250_PINK | A_REVERSE,
                   A_5250_BLUE,
                   A_5250_BLUE | A_REVERSE,
                   A_5250_PINK | A_UNDERLINE,
                   A_5250_PINK | A_UNDERLINE | A_REVERSE,
                   A_5250_BLUE | A_UNDERLINE,
                   0x00};

int WinTerminal::register_stuff (HINSTANCE hInst)
{
        WNDCLASS wc;

        g_hInstance = hInst;

        ZeroMemory (&wc, sizeof (WNDCLASS));
        wc.style = CS_VREDRAW | CS_HREDRAW;
        wc.lpfnWndProc = wnd_proc;
        wc.cbClsExtra = 0;
        wc.cbWndExtra = sizeof (WinTerminal*);
        wc.hInstance = hInst;
        wc.hIcon = LoadIcon (NULL, IDI_APPLICATION);
        wc.hCursor = LoadCursor (NULL, IDC_ARROW);
        wc.hbrBackground = (HBRUSH)GetStockObject (BLACK_BRUSH);
        wc.lpszMenuName = NULL;
        wc.lpszClassName = "tn5250WinTerminal";

        if (!RegisterClass (&wc)) {
                MessageBox (NULL, "Cannot register class!", "tn5250",
                        MB_ICONEXCLAMATION);
                return -1;
        }

        return 0;
}

LRESULT WINAPI WinTerminal::wnd_proc (HWND hWnd, UINT cmd, WPARAM wp,
        LPARAM lp)
{
        WinTerminal *pWinTerminal =
                (WinTerminal*)GetWindowLong (hWnd,0);
        switch (cmd) {
        case WM_CREATE:
                {
                        LPCREATESTRUCT lpcs = (LPCREATESTRUCT)lp;
                        pWinTerminal =
                                (WinTerminal*)lpcs->lpCreateParams;
                        SetWindowLong (hWnd, 0, (LPARAM)pWinTerminal);
                }
                return 0;
        }
        if (pWinTerminal != NULL)
                return pWinTerminal->wnd_proc (cmd, wp, lp);
        else
                return DefWindowProc (hWnd, cmd, wp, lp);
}

LRESULT WinTerminal::wnd_proc (UINT cmd, WPARAM wp, LPARAM lp)
{
        switch (cmd) {
        case WM_PAINT:
                OnPaint ();
                return 0;

	case WM_GETMINMAXINFO:
		{
			LPMINMAXINFO lpmmi = (LPMINMAXINFO)lp;

			// Get the default take on these
			DefWindowProc (hWnd, cmd, wp, lp);
			GetWindowMax (lpmmi->ptMaxSize);

			if (pcached != NULL) {
				// Don't allow tracking larger than size
				lpmmi->ptMaxTrackSize.y = lpmmi->ptMaxSize.y;
				lpmmi->ptMaxTrackSize.x = lpmmi->ptMaxSize.x;
			}
		}
		return 0;

	case WM_KEYDOWN:
		switch (wp) {
		case VK_UP: next_char = K_UP; break;
		case VK_DOWN: next_char = K_DOWN; break;
		case VK_LEFT: next_char = K_LEFT; break;
		case VK_RIGHT: next_char = K_RIGHT; break;
		case VK_DELETE: next_char = K_DELETE; break;
		case VK_BACK: next_char = K_BACKSPACE; break;
		case VK_HOME: next_char = K_HOME; break;
		case VK_END: next_char = K_END; break;
		case VK_PRIOR: next_char = K_ROLLDN; break;
		case VK_NEXT: next_char = K_ROLLUP; break;
		case VK_PRINT: next_char = K_PRINT; break;
		case VK_F1: next_char = K_F1; break;
		case VK_F2: next_char = K_F2; break;
		case VK_F3: next_char = K_F3; break;
		case VK_F4: next_char = K_F4; break;
		case VK_F5: next_char = K_F5; break;
		case VK_F6: next_char = K_F6; break;
		case VK_F7: next_char = K_F7; break;
		case VK_F8: next_char = K_F8; break;
		case VK_F9: next_char = K_F9; break;
		case VK_F10: next_char = K_F10; break;
		case VK_F11: next_char = K_F11; break;
		case VK_F12: next_char = K_F12; break;
		case VK_F13: next_char = K_F13; break;
		case VK_F14: next_char = K_F14; break;
		case VK_F15: next_char = K_F15; break;
		case VK_F16: next_char = K_F16; break;
		case VK_F17: next_char = K_F17; break;
		case VK_F18: next_char = K_F18; break;
		case VK_F19: next_char = K_F19; break;
		case VK_F20: next_char = K_F20; break;
		case VK_F21: next_char = K_F21; break;
		case VK_F22: next_char = K_F22; break;
		case VK_F23: next_char = K_F23; break;
		case VK_F24: next_char = K_F24; break;
		case VK_INSERT: next_char = K_INSERT; break;
		}
		break;

	case WM_CHAR:
		next_char = (char)wp;
		break;

        case WM_CLOSE:
                PostQuitMessage (0);
                break;
        }
        return DefWindowProc (hWnd, cmd, wp, lp);
}

void WinTerminal::init ()
{
        quit_flag = false;
	pcached = NULL;
	prendered = NULL;

	hDisplayBmp = (HBITMAP)NULL;

        hWnd = CreateWindow ("tn5250WinTerminal",
                "tn5250 -",
                WS_OVERLAPPEDWINDOW,
                CW_USEDEFAULT,
                CW_USEDEFAULT,
                CW_USEDEFAULT,
                CW_USEDEFAULT,
                NULL,
                NULL,
                g_hInstance,
                (LPVOID)this);

	hFont = (HFONT)GetStockObject (OEM_FIXED_FONT);

        if (hWnd == NULL) {
                MessageBox (NULL, "Can't create window!", "tn5250",
                        MB_ICONEXCLAMATION);
                exit (1);
        }

	HFONT hFont = (HFONT)GetStockObject (ANSI_FIXED_FONT);
	SendMessage (hWnd, WM_SETFONT, (WPARAM)hFont, 0);

        //ShowWindow (hWnd, SW_SHOWDEFAULT);
}

void WinTerminal::term ()
{
	if (pcached)
		delete pcached;
	if (prendered)
		delete prendered;
	if (hDisplayBmp)
		DeleteObject (hDisplayBmp);
        DestroyWindow (hWnd);
}

int WinTerminal::getkey ()
{
        int r = next_char;
        next_char = -1;
        return r;
}

int WinTerminal::waitevent ()
{
        MSG msg;

        if (quit_flag)
                return EVENT_QUIT;

        while (GetMessage (&msg, NULL, 0, 0)) {
                TranslateMessage (&msg);

                switch (msg.message) {
                case WM_SOCKET_RECV:
			if (LOWORD(msg.lParam) == FD_CLOSE) {
				quit_flag = true;
				return EVENT_QUIT;
			}
                        return EVENT_DATA;

                default:
                        DispatchMessage (&msg);
                }

		if (next_char != -1)
			return EVENT_KEY;
        }

        quit_flag = true;
        return EVENT_QUIT;
}

void WinTerminal::GetCellSize (HDC hDC)
{
	// Get size of font.
	TEXTMETRIC tm;
	GetTextMetrics (hDC, &tm);
	cellw = (int)(tm.tmMaxCharWidth);
	cellh = (int)(tm.tmHeight);
	cellh += 2;

}

void WinTerminal::update (const DisplayBuffer& buffer)
{
	HFONT hFontOld;
	HDC hDC;
	bool first_flag = false;

	if (pcached == NULL) {
		pcached = new DisplayBuffer (buffer);
		first_flag = true;
	} else
		*pcached = buffer;

        hDC = GetDC (hWnd);
	hFontOld = (HFONT)SelectObject (hDC, hFont);

	GetCellSize (hDC);
	if (hDisplayBmp == NULL)
		hDisplayBmp = CreateCompatibleBitmap (hDC, 80*cellw, 25*cellh + 2);

	HDC hBmpDC = CreateCompatibleDC (hDC);
	HBITMAP hOldBmp = (HBITMAP)SelectObject (hBmpDC, hDisplayBmp);
	HFONT hBmpDCOldFont = (HFONT)SelectObject (hBmpDC, hFont);
	OnPaint (hBmpDC);

	// Get size of bitmap.
	BITMAP bm;
	GetObject (hDisplayBmp, sizeof (BITMAP), &bm);

	// Blt it.
	BitBlt (hDC, 0, 0, bm.bmWidth, bm.bmHeight, hBmpDC, 0, 0, SRCCOPY);

	SelectObject (hBmpDC, hBmpDCOldFont);
	SelectObject (hBmpDC, hOldBmp);
	DeleteDC (hBmpDC);

	//OnPaint (hDC);

	SelectObject (hDC, hFontOld);
	ReleaseDC (hWnd, hDC);

	// Resize window (and show it) if this is the first update.
	if (first_flag) {
		POINT pt;
		GetWindowMax (pt);
		SetWindowPos (hWnd, NULL, 0, 0, pt.x, pt.y,
			SWP_NOACTIVATE | SWP_NOMOVE | SWP_NOZORDER);

		ShowWindow (hWnd, SW_SHOWDEFAULT);
	}
}

void WinTerminal::OnPaint (HDC hDC)
{
	if  (pcached == NULL)
		return;

        int my = height (), mx = width ();
        int y, x;
	unsigned char attr = 0x20, newattr = 0x20;

        SetTextColor (hDC, RGB(0,255,0));
        for (y = 0; y < pcached->height (); y++) {
                if (y >= my)
                        break;

                for (x = 0; x < pcached->width (); x++) {
                        if (x >= mx)
                                break;

			bool damaged = false;
			if (prendered) {	
				unsigned char foo = (*pcached)[y][x];
				unsigned char newfoo = (*prendered)[y][x];

				// Check for attributes
				if ((foo & 0xe0) == 0x20) {
					attr = foo;
				}
				if ((newfoo & 0xe0) == 0x20) {
					newattr = foo;
				}

				// Has the cursor moved and is this the to or from cell?
				if (x == pcached->cursor_x () || x == prendered->cursor_x ()) {
					if (y == pcached->cursor_y () || y == prendered->cursor_y ()) {
						if (pcached->cursor_x () != prendered->cursor_x ()
							|| pcached->cursor_y () != prendered->cursor_y ())
							damaged = true;
					}
				}

				// If the character or the attribute is different
				if (foo != newfoo || attr != newattr)
					damaged = true;
			} else
				damaged = true;

			if (damaged)
				DrawCell (hDC, attr, *pcached, y, x);
                }
        }

	if (prendered == NULL)
		prendered = new DisplayBuffer (*pcached);
	else
		*prendered = *pcached;
}

void WinTerminal::DrawCell (HDC hDC, unsigned char attr,
	const DisplayBuffer& buffer,
	int y, int x)
{
        RECT rect;
	HBRUSH hBrBg;
	unsigned char cdata;
	char ascii_char;
	int win_attrs;
	COLORREF fg, bg;

	cdata = buffer[y][x];

	rect.top = y*cellh;
	rect.bottom = rect.top + cellh;
	rect.left = x*cellw;
	rect.right = rect.left + cellw;

	win_attrs = attribute_map[attr - 0x20];
	if ((cdata & 0xe0) == 0x20) {
		ascii_char = ' ';
		win_attrs = attribute_map[0];
	} else
		ascii_char = ebcdic2ascii (cdata);

	// Nondisplay
	if (attr == 0x27) {
		ascii_char = ' ';
		win_attrs = attribute_map[0];
	}

	bg = RGB(0,0,0);
	switch (win_attrs & A_COLOR_MASK) {
	case A_5250_WHITE:
		fg = RGB(255,255,255);
		break;
	case A_5250_GREEN:
		fg = RGB(0,255,0);
		break;
	case A_5250_BLUE:
		fg = RGB(0,255,255);
		break;
	case A_5250_RED:
		fg = RGB(255,0,0);
		break;
	case A_5250_TURQ:
		fg = RGB(192,192,255);
		break;
	case A_5250_YELLOW:
		fg = RGB(255,255,0);
		break;
	case A_5250_PINK:
		fg = RGB(255,128,192);
		break;
	case A_5250_BLACK:
		fg = RGB(0,0,0);
		break;
	}

	// Reverse the colors for reverse attributes (or cursor)
	if ((win_attrs & A_REVERSE) != 0) {
		COLORREF i = fg;
		fg = bg;
		bg = i;
	}

	// Color for cursor
	if (buffer.cursor_x () == x && buffer.cursor_y () == y) {
		bg = RGB(0,0,255);
		fg = RGB(0,0,0);
	}

	// Clear the cell first
	hBrBg = CreateSolidBrush (bg);
	FillRect (hDC, &rect, hBrBg);
	DeleteObject (hBrBg);

	// Handle underline attributes
	if ((win_attrs & A_UNDERLINE) != 0) {
		HPEN hPenUL, hPenOld;

		hPenUL = CreatePen (PS_SOLID, 1, fg);
		hPenOld = (HPEN)SelectObject (hDC, hPenUL);

		MoveToEx (hDC, rect.left, rect.bottom - 1, NULL);
		LineTo (hDC, rect.right, rect.bottom - 1);

		SelectObject (hDC, hPenOld);
		DeleteObject (hPenUL);
	}

	// Render the character
	SetTextColor (hDC, fg);
	SetBkColor (hDC, bg);
	DrawText (hDC, &ascii_char, 1, &rect, DT_CENTER | DT_TOP 
		| DT_NOPREFIX);
}

void WinTerminal::update_indicators (const DisplayBuffer& buffer)
{
	HDC hDC = GetDC (hWnd);	
	HDC hBmpDC = CreateCompatibleDC (hDC);
	HBITMAP hOldBmp = (HBITMAP)SelectObject (hBmpDC, hDisplayBmp);
	HFONT hOldFont = (HFONT)SelectObject (hBmpDC, hFont);

	GetCellSize (hBmpDC);

	// Clear the indicator area
	BITMAP bm;
	HBRUSH hBr = (HBRUSH)GetStockObject (BLACK_BRUSH);
	GetObject (hDisplayBmp, sizeof (BITMAP), &bm);
	RECT rect;

	rect.top = cellh * (height () - 1);
	rect.bottom = bm.bmHeight;
	rect.right = bm.bmWidth;
	rect.left = 0;
	FillRect (hBmpDC, &rect, hBr);

	// Draw the separator line.
	HPEN hPenSep = (HPEN)CreatePen (PS_SOLID, 1, RGB(0,255,255));
	HPEN hPenOld = (HPEN)SelectObject (hBmpDC, hPenSep);
	MoveToEx (hBmpDC, rect.left, rect.top + 1, NULL);
	LineTo (hBmpDC, rect.right, rect.top + 1);
	SelectObject (hBmpDC, hPenOld);

	// Compose status line
	char ind_buf [80];
	int inds = buffer.indicators ();

	memset (ind_buf , ' ', sizeof (ind_buf));
	memcpy (ind_buf + 4, "5250", 4);
	if ((inds & DisplayBuffer::IND_MESSAGE_WAITING) != 0) {
		memcpy (ind_buf + 23, "MW", 2);
	}
	if ((inds & DisplayBuffer::IND_INHIBIT) != 0) {
		memcpy (ind_buf + 9, "X II", 4);
	} else if ((inds & DisplayBuffer::IND_X_CLOCK) != 0) {
		memcpy (ind_buf + 9, "X CLOCK", 7);
	} else if ((inds & DisplayBuffer::IND_X_SYSTEM) != 0) {
		memcpy (ind_buf + 9, "X SYSTEM", 8);
	}
	if ((inds & DisplayBuffer::IND_INSERT) != 0) {
		memcpy (ind_buf + 30, "IM", 2);
	}

	// Draw the status line
	SetTextColor (hBmpDC, RGB(255,255,255));
	SetBkColor (hBmpDC, RGB(0,0,0));
	rect.top += 2;
	DrawText (hBmpDC, ind_buf, sizeof (ind_buf), &rect, DT_LEFT | DT_NOPREFIX | DT_VCENTER);

	// Update the display.
	BitBlt (hDC, 0, 0, bm.bmWidth, bm.bmHeight, hBmpDC, 0, 0, SRCCOPY);

	// Clean up
	SelectObject (hBmpDC, hOldBmp);
	SelectObject (hBmpDC, hOldFont);
	DeleteDC (hBmpDC);
	ReleaseDC (hWnd, hDC);
}

int WinTerminal::flags () const
{
        return HAS_COLOR;
}

int WinTerminal::height () const
{
        return 25; //FIXME:
}

int WinTerminal::width () const
{
        return 80; //FIXME:
}

void WinTerminal::connection_fd (SOCKET_TYPE fd)
{
        int ec;

        conn_fd = fd;
        ec = WSAAsyncSelect (conn_fd, hWnd, WM_SOCKET_RECV, FD_READ | FD_CLOSE);
        if (ec == SOCKET_ERROR) {
                char msg[128];
                sprintf (msg, "WSAAsyncSelect failed: %d",
                        (int)WSAGetLastError ());
                MessageBox (NULL, msg, "tn5250", 0);
        }
}

void WinTerminal::OnPaint ()
{
        PAINTSTRUCT ps;
        HDC hDC = BeginPaint (hWnd, &ps);
	HDC hBmpDC = CreateCompatibleDC (hDC);
	HBITMAP hBmpOld = (HBITMAP)SelectObject (hBmpDC, hDisplayBmp);

	// Get bitmap size.
	BITMAP bm;
	GetObject (hDisplayBmp, sizeof (BITMAP), &bm);

	// Copy bitmap to window
	BitBlt (hDC, 0, 0, bm.bmWidth, bm.bmHeight, hBmpDC, 0, 0, SRCCOPY);
	
	// Clean up
	SelectObject (hBmpDC, hBmpOld);
	DeleteDC (hBmpDC);
        EndPaint (hWnd, &ps);
}

void WinTerminal::GetWindowMax (POINT& pt)
{
	int bwidth, bheight;
	RECT rectCl, rectWnd;

	if (pcached != NULL) {
		GetWindowRect (hWnd, &rectWnd);
		GetClientRect (hWnd, &rectCl);

		// Get the Windows border width/height
		bwidth = (rectWnd.right - rectWnd.left) - 
			(rectCl.right - rectCl.left);
		bheight = (rectWnd.bottom - rectWnd.top) -
			(rectCl.bottom - rectCl.top);

		pt.y = cellh * height () + 2 + bheight;
		pt.x = cellw * width () + bwidth;
	}
}
