/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.streaming.connectors.kinesis;

import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import java.nio.ByteBuffer;
import java.util.List;
import java.util.Properties;
import org.apache.flink.annotation.PublicEvolving;
import org.apache.flink.annotation.VisibleForTesting;
import org.apache.flink.api.common.serialization.SerializationSchema;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.kinesis.shaded.com.amazonaws.services.kinesis.producer.Attempt;
import org.apache.flink.kinesis.shaded.com.amazonaws.services.kinesis.producer.KinesisProducer;
import org.apache.flink.kinesis.shaded.com.amazonaws.services.kinesis.producer.KinesisProducerConfiguration;
import org.apache.flink.kinesis.shaded.com.amazonaws.services.kinesis.producer.UserRecordFailedException;
import org.apache.flink.kinesis.shaded.com.amazonaws.services.kinesis.producer.UserRecordResult;
import org.apache.flink.metrics.Counter;
import org.apache.flink.metrics.MetricGroup;
import org.apache.flink.runtime.state.FunctionInitializationContext;
import org.apache.flink.runtime.state.FunctionSnapshotContext;
import org.apache.flink.streaming.api.checkpoint.CheckpointedFunction;
import org.apache.flink.streaming.api.functions.sink.RichSinkFunction;
import org.apache.flink.streaming.api.functions.sink.SinkFunction;
import org.apache.flink.streaming.connectors.kinesis.KinesisPartitioner;
import org.apache.flink.streaming.connectors.kinesis.serialization.KinesisSerializationSchema;
import org.apache.flink.streaming.connectors.kinesis.util.KinesisConfigUtil;
import org.apache.flink.streaming.connectors.kinesis.util.TimeoutLatch;
import org.apache.flink.util.InstantiationUtil;
import org.apache.flink.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@PublicEvolving
public class FlinkKinesisProducer<OUT>
extends RichSinkFunction<OUT>
implements CheckpointedFunction {
    public static final String KINESIS_PRODUCER_METRIC_GROUP = "kinesisProducer";
    public static final String METRIC_BACKPRESSURE_CYCLES = "backpressureCycles";
    public static final String METRIC_OUTSTANDING_RECORDS_COUNT = "outstandingRecordsCount";
    private static final long serialVersionUID = 6447077318449477846L;
    private static final Logger LOG = LoggerFactory.getLogger(FlinkKinesisProducer.class);
    private final Properties configProps;
    private boolean failOnError = false;
    private int queueLimit = Integer.MAX_VALUE;
    private String defaultStream;
    private String defaultPartition;
    private final KinesisSerializationSchema<OUT> schema;
    private KinesisPartitioner<OUT> customPartitioner = null;
    private transient KinesisProducer producer;
    private transient TimeoutLatch backpressureLatch;
    private transient FutureCallback<UserRecordResult> callback;
    private transient Counter backpressureCycles;
    private volatile transient Throwable thrownException;

    public FlinkKinesisProducer(final SerializationSchema<OUT> schema, Properties configProps) {
        this(new KinesisSerializationSchema<OUT>(){

            @Override
            public ByteBuffer serialize(OUT element) {
                return ByteBuffer.wrap(schema.serialize(element));
            }

            @Override
            public String getTargetStream(OUT element) {
                return null;
            }
        }, configProps);
    }

    public FlinkKinesisProducer(KinesisSerializationSchema<OUT> schema, Properties configProps) {
        Preconditions.checkNotNull((Object)configProps, (String)"configProps can not be null");
        this.configProps = KinesisConfigUtil.replaceDeprecatedProducerKeys(configProps);
        Preconditions.checkNotNull(schema, (String)"serialization schema cannot be null");
        Preconditions.checkArgument((boolean)InstantiationUtil.isSerializable(schema), (Object)("The provided serialization schema is not serializable: " + schema.getClass().getName() + ". Please check that it does not contain references to non-serializable instances."));
        this.schema = schema;
    }

    public void setFailOnError(boolean failOnError) {
        this.failOnError = failOnError;
    }

    public void setQueueLimit(int queueLimit) {
        Preconditions.checkArgument((queueLimit > 0 ? 1 : 0) != 0, (Object)"queueLimit must be a positive number");
        this.queueLimit = queueLimit;
    }

    public void setDefaultStream(String defaultStream) {
        this.defaultStream = defaultStream;
    }

    public void setDefaultPartition(String defaultPartition) {
        this.defaultPartition = defaultPartition;
    }

    public void setCustomPartitioner(KinesisPartitioner<OUT> partitioner) {
        Preconditions.checkNotNull(partitioner, (String)"partitioner cannot be null");
        Preconditions.checkArgument((boolean)InstantiationUtil.isSerializable(partitioner), (Object)("The provided custom partitioner is not serializable: " + partitioner.getClass().getName() + ". Please check that it does not contain references to non-serializable instances."));
        this.customPartitioner = partitioner;
    }

    public void open(Configuration parameters) throws Exception {
        super.open(parameters);
        KinesisProducerConfiguration producerConfig = KinesisConfigUtil.getValidatedProducerConfiguration(this.configProps);
        this.producer = this.getKinesisProducer(producerConfig);
        MetricGroup kinesisMectricGroup = this.getRuntimeContext().getMetricGroup().addGroup(KINESIS_PRODUCER_METRIC_GROUP);
        this.backpressureCycles = kinesisMectricGroup.counter(METRIC_BACKPRESSURE_CYCLES);
        kinesisMectricGroup.gauge(METRIC_OUTSTANDING_RECORDS_COUNT, this.producer::getOutstandingRecordsCount);
        this.backpressureLatch = new TimeoutLatch();
        this.callback = new FutureCallback<UserRecordResult>(){

            @Override
            public void onSuccess(UserRecordResult result) {
                FlinkKinesisProducer.this.backpressureLatch.trigger();
                if (!result.isSuccessful()) {
                    if (FlinkKinesisProducer.this.failOnError) {
                        if (FlinkKinesisProducer.this.thrownException == null) {
                            FlinkKinesisProducer.this.thrownException = new RuntimeException("Record was not sent successful");
                        }
                    } else {
                        LOG.warn("Record was not sent successful");
                    }
                }
            }

            @Override
            public void onFailure(Throwable t) {
                FlinkKinesisProducer.this.backpressureLatch.trigger();
                if (FlinkKinesisProducer.this.failOnError) {
                    FlinkKinesisProducer.this.thrownException = t;
                } else {
                    LOG.warn("An exception occurred while processing a record", t);
                }
            }
        };
        if (this.customPartitioner != null) {
            this.customPartitioner.initialize(this.getRuntimeContext().getIndexOfThisSubtask(), this.getRuntimeContext().getNumberOfParallelSubtasks());
        }
        LOG.info("Started Kinesis producer instance for region '{}'", (Object)producerConfig.getRegion());
    }

    public void invoke(OUT value, SinkFunction.Context context) throws Exception {
        if (this.producer == null) {
            throw new RuntimeException("Kinesis producer has been closed");
        }
        this.checkAndPropagateAsyncError();
        boolean didWaitForFlush = this.enforceQueueLimit();
        if (didWaitForFlush) {
            this.checkAndPropagateAsyncError();
        }
        String stream = this.defaultStream;
        String partition = this.defaultPartition;
        ByteBuffer serialized = this.schema.serialize(value);
        String customStream = this.schema.getTargetStream(value);
        if (customStream != null) {
            stream = customStream;
        }
        String explicitHashkey = null;
        if (this.customPartitioner != null) {
            partition = this.customPartitioner.getPartitionId(value);
            explicitHashkey = this.customPartitioner.getExplicitHashKey(value);
        }
        if (stream == null) {
            if (this.failOnError) {
                throw new RuntimeException("No target stream set");
            }
            LOG.warn("No target stream set. Skipping record");
            return;
        }
        ListenableFuture<UserRecordResult> cb = this.producer.addUserRecord(stream, partition, explicitHashkey, serialized);
        Futures.addCallback(cb, this.callback);
    }

    public void close() throws Exception {
        LOG.info("Closing producer");
        super.close();
        if (this.producer != null) {
            LOG.info("Flushing outstanding {} records", (Object)this.producer.getOutstandingRecordsCount());
            this.flushSync();
            LOG.info("Flushing done. Destroying producer instance.");
            this.producer.destroy();
            this.producer = null;
        }
        this.checkAndPropagateAsyncError();
    }

    public void initializeState(FunctionInitializationContext context) throws Exception {
    }

    public void snapshotState(FunctionSnapshotContext context) throws Exception {
        this.checkAndPropagateAsyncError();
        this.flushSync();
        if (this.producer.getOutstandingRecordsCount() > 0) {
            throw new IllegalStateException("Number of outstanding records must be zero at this point: " + this.producer.getOutstandingRecordsCount());
        }
        this.checkAndPropagateAsyncError();
    }

    @VisibleForTesting
    protected KinesisProducer getKinesisProducer(KinesisProducerConfiguration producerConfig) {
        return new KinesisProducer(producerConfig);
    }

    private void checkAndPropagateAsyncError() throws Exception {
        if (this.thrownException != null) {
            String errorMessages = "";
            if (this.thrownException instanceof UserRecordFailedException) {
                List<Attempt> attempts = ((UserRecordFailedException)this.thrownException).getResult().getAttempts();
                for (Attempt attempt : attempts) {
                    if (attempt.getErrorMessage() == null) continue;
                    errorMessages = errorMessages + attempt.getErrorMessage() + "\n";
                }
            }
            if (this.failOnError) {
                throw new RuntimeException("An exception was thrown while processing a record: " + errorMessages, this.thrownException);
            }
            LOG.warn("An exception was thrown while processing a record: {}", (Object)this.thrownException, (Object)errorMessages);
            this.thrownException = null;
        }
    }

    private boolean enforceQueueLimit() {
        int attempt = 0;
        while (this.producer.getOutstandingRecordsCount() >= this.queueLimit) {
            this.backpressureCycles.inc();
            if (attempt >= 10) {
                LOG.warn("Waiting for the queue length to drop below the limit takes unusually long, still not done after {} attempts.", (Object)attempt);
            }
            ++attempt;
            try {
                this.backpressureLatch.await(100L);
            }
            catch (InterruptedException e) {
                LOG.warn("Flushing was interrupted.");
                break;
            }
        }
        return attempt > 0;
    }

    private void flushSync() throws Exception {
        while (this.producer.getOutstandingRecordsCount() > 0) {
            this.producer.flush();
            try {
                Thread.sleep(500L);
            }
            catch (InterruptedException e) {
                LOG.warn("Flushing was interrupted.");
                break;
            }
        }
    }
}

