/* Copyright (c) 1998 Joe Rumsey (mrogre@mediaone.net) */
#include "copyright.h"

/* packet.h for tanked

   Contains definitions of all the packet types for both directions.

   Packets the server sends start with s_
   Packets the client sends start with c_

   */

#ifndef packet_h
#define packet_h

#include "defs.h"
typedef unsigned char UCHAR;
typedef char CHAR;
typedef short SHORT;
typedef unsigned short USHORT;
typedef int INT;
typedef unsigned int UINT;

enum s_packet_types {
    S_NULL,
    S_MAPINFO,
    S_PLAYERINFO,
    S_POSITIONS,
    S_PERSONAL,
    S_LOGIN,
    S_SHELL,
    S_MESSAGE,
    S_PLAYERDATA,
    S_PLAYERSTATS,
    S_SHELLINFO,
    S_MAPSQUARE,
    S_FLAG,
    S_UDPREPLY,
    S_MOTDLINE,
    S_TEAMOK,
    S_MINE,
    S_PING
};

/* mapinfo - the header for the map, but not the raw data */
typedef struct {
    UCHAR type;                 /* S_MAPINFO */
    UCHAR width, height;
    UCHAR pad;
    CHAR name[32];

    /* data goes here, width x height array of bytes */
} s_mapinfo;

/* playerinfo, status and team a player belongs to */
typedef struct {
    UCHAR type;                 /* S_PLAYERINFO */
    UCHAR num;
    UCHAR status, team;
} s_playerinfo;

/* one s_position followed by multiple s_oneposition packets send the coords of every
   player */
typedef struct {
    UCHAR type;                 /* S_POSITIONS */
    UCHAR num;                  /* how many players contained in this packet, MSB = tactical or strategic */
    UCHAR pad1, pad2;
} s_positions;

typedef struct {                /* not a packet by itself, combined with s_positions */
    UCHAR num, lx, ly, dir;
} s_oneposition;

typedef struct {                /* Info only for the player being sent to */
    UCHAR type;                 /* S_PERSONAL */
    UCHAR status;
    USHORT x, y;                /* coords */

    UCHAR dir;                  /* heading */
    UCHAR damage;               /* How hurt you are.  0 = full strength, 255 = dead */
    UCHAR ammo;                 /* how many shells you have left */
    UCHAR mines;                /* only need 3 bits of this (with <= 8 MINESPERPLAYER) */
    USHORT fuel;                /* amount of fuel left */
} s_personal;

typedef struct {
    UCHAR type;                 /* S_LOGIN */
    UCHAR num;                  /* who you are */
    UCHAR pad1, pad2;
} s_login;

/* shell: sends just coords for a shell */
typedef struct {
    UCHAR type;                 /* S_SHELL */
    UCHAR num;
    UCHAR lx, ly;
} s_shell;

/* message: used for all text messages, including warnings currently */
typedef struct {
    UCHAR type;                 /* S_MESSAGE */
    UCHAR from;
    UCHAR to;
    UCHAR flags;
    CHAR message[80];
} s_message;

/* playerdata: sends the name of a particular player */
typedef struct {
    UCHAR type;
    UCHAR num;
    UCHAR pad1, pad2;
    CHAR name[16];
} s_playerdata;

/* playerstats: sends wins, losses, kills for a player
   wins = total number of tanks killed
   losses = total number of times he's died
   kills = tanks killed since the last time he died
   */
typedef struct {
    UCHAR type;
    UCHAR num;
    UCHAR pad1, pad2;
    UINT wins;
    UINT losses;
    UINT kills;
} s_playerstats;

/* shellinfo: status of a shell (not coords) */
typedef struct {
    UCHAR type;
    UCHAR num;
    UCHAR status;
    UCHAR pad;
} s_shellinfo;

/* sends the terrain for a single map square */
typedef struct {
    UCHAR type;
    UCHAR x, y;
    UCHAR value;
} s_mapsquare;

/* flag: sends the location and team of a particular flag */
typedef struct {
    UCHAR type;
    UCHAR num;
    UCHAR team;
    UCHAR pad;
    USHORT x;
    USHORT y;
} s_flag;

#define MODE_UDP 1
#define MODE_TCP 2
#define SWITCH_DENIED 1
#define SWITCH_UDP_OK 2

/* udpreply: sent when a player requests a UDP connection (or requests it be closed) */
typedef struct {
    UCHAR type;
    UCHAR reply;
    USHORT port;
} s_udpreply;

/* sends one line of the motd */
typedef struct {
    UCHAR type;
    UCHAR pad;
    USHORT line;
    CHAR text[80];
} s_motdline;

/* sends a verification that the team a player wants to join is ok or not */
typedef struct {
    UCHAR type;
    UCHAR teamok;
    UCHAR pad1, pad2;
} s_teamok;

/* sends a mine's status & location */
typedef struct {
    UCHAR type;
    UCHAR status;
    USHORT num;
    USHORT x, y;
} s_mine;

/* reply to a client's request for a ping (lag timing) */
typedef struct {
    UCHAR type;
    UCHAR seq;
    UCHAR pad1, pad2;
} s_ping;

enum c_packet_types {
    C_NULL,
    C_LOGIN,
    C_STEERING,
    C_SHELL,
    C_MESSAGE,
    C_SPEED,
    C_COURSE,
    C_UDPREQUEST,
    C_TEAM,
    C_MINE,
    C_BUILD,
    C_PING
};

/* send the player's name, login, and client version to the server */
typedef struct {
    UCHAR type;                 /* C_LOGIN */
    UCHAR pad1;
    USHORT version;
    CHAR name[16];
    CHAR login[16];
} c_login;

/* obsolete, c_course is current */
typedef struct {
    UCHAR type;                 /* C_STEERING */
    UCHAR keys;
    UCHAR pad1, pad2;
} c_steering;

/* player wants to fire a shell */
typedef struct {
    UCHAR type;                 /* C_SHELL*/
    UCHAR pad1, pad2, pad3;     /* Just the packet itself is enough info. */
} c_shell;

/* player sent a message */
typedef struct {
    UCHAR type;                 /* C_MESSAGE */
    UCHAR to;
    USHORT flags;
    CHAR message[80];
} c_message;

/* player wants to change his speed */
typedef struct {
    UCHAR type;
    CHAR speed;
    UCHAR pad1, pad2;
} c_speed;

/* player wants to change course */
typedef struct {
    UCHAR type;
    UCHAR dir;
    UCHAR pad1, pad2;
} c_course;

/* player wants to open or close his UDP connection */
typedef struct {
    UCHAR type;
    UCHAR mode;
    USHORT port;
} c_udprequest;

/* player wants to join this team */
typedef struct {
    UCHAR type;
    UCHAR team;
    UCHAR pad1, pad2;
} c_team;

/* player wants to lay a mine */
typedef struct {
    UCHAR type;
    UCHAR pad1, pad2, pad3;     /* don't need any info, just getting the packet is enough */
} c_mine;

/* player wants to build something */
typedef struct {
    UCHAR type;
    UCHAR terrain;
    UCHAR pad1, pad2;
} c_build;

/* player wants to check out his lag */
typedef struct {
    UCHAR type;
    UCHAR seq;
    UCHAR pad1, pad2;
} c_ping;

#endif
