#include <inttypes.h>
#include <stdlib.h>
#include "utf8_codepages.h"

struct ciolib_cpmap {
	uint32_t	unicode;
	uint8_t		cpchar;
};

// Sorted by unicode codepoint...
const static struct ciolib_cpmap cp437_table[162] = {
	{0x0000, 0},   {0x00A0, 255}, {0x00A1, 173}, {0x00A2, 155},
	{0x00A3, 156}, {0x00A5, 157}, {0x00A6, 124}, {0x00A7, 21},  {0x00AA, 166},
	{0x00AB, 174}, {0x00AC, 170}, {0x00B0, 248}, {0x00B1, 241},
	{0x00B2, 253}, {0x00B5, 230}, {0x00B6, 20},  {0x00B7, 250},
	{0x00BA, 167}, {0x00BB, 175}, {0x00BC, 172}, {0x00BD, 171},
	{0x00BF, 168}, {0x00C4, 142}, {0x00C5, 143}, {0x00C6, 146},
	{0x00C7, 128}, {0x00C9, 144}, {0x00D1, 165}, {0x00D6, 153},
	{0x00DC, 154}, {0x00DF, 225}, {0x00E0, 133}, {0x00E1, 160},
	{0x00E2, 131}, {0x00E4, 132}, {0x00E5, 134}, {0x00E6, 145},
	{0x00E7, 135}, {0x00E8, 138}, {0x00E9, 130}, {0x00EA, 136},
	{0x00EB, 137}, {0x00EC, 141}, {0x00ED, 161}, {0x00EE, 140},
	{0x00EF, 139}, {0x00F1, 164}, {0x00F2, 149}, {0x00F3, 162},
	{0x00F4, 147}, {0x00F6, 148}, {0x00F7, 246}, {0x00F9, 151},
	{0x00FA, 163}, {0x00FB, 150}, {0x00FC, 129}, {0x00FF, 152},
	{0x0192, 159}, {0x0393, 226}, {0x0398, 233}, {0x03A3, 228},
	{0x03A6, 232}, {0x03A9, 234}, {0x03B1, 224}, {0x03B4, 235},
	{0x03B5, 238}, {0x03C0, 227}, {0x03C3, 229}, {0x03C4, 231},
	{0x03C6, 237}, {0x2022, 7},   {0x203C, 19},  {0x207F, 252},
	{0x20A7, 158}, {0x2190, 27},  {0x2191, 24},  {0x2192, 26},
	{0x2193, 25},  {0x2194, 29},  {0x2195, 18},  {0x21A8, 23},
	{0x2219, 249}, {0x221A, 251}, {0x221E, 236}, {0x221F, 28},
	{0x2229, 239}, {0x2248, 247}, {0x2261, 240}, {0x2264, 243},
	{0x2265, 242}, {0x2310, 169}, {0x2320, 244}, {0x2321, 245},
	{0x2500, 196}, {0x2502, 179}, {0x250C, 218}, {0x2510, 191},
	{0x2514, 192}, {0x2518, 217}, {0x251C, 195}, {0x2524, 180},
	{0x252C, 194}, {0x2534, 193}, {0x253C, 197}, {0x2550, 205},
	{0x2551, 186}, {0x2552, 213}, {0x2553, 214}, {0x2554, 201},
	{0x2555, 184}, {0x2556, 183}, {0x2557, 187}, {0x2558, 212},
	{0x2559, 211}, {0x255A, 200}, {0x255B, 190}, {0x255C, 189},
	{0x255D, 188}, {0x255E, 198}, {0x255F, 199}, {0x2560, 204},
	{0x2561, 181}, {0x2562, 182}, {0x2563, 185}, {0x2564, 209},
	{0x2565, 210}, {0x2566, 203}, {0x2567, 207}, {0x2568, 208},
	{0x2569, 202}, {0x256A, 216}, {0x256B, 215}, {0x256C, 206},
	{0x2580, 223}, {0x2584, 220}, {0x2588, 219}, {0x258C, 221},
	{0x2590, 222}, {0x2591, 176}, {0x2592, 177}, {0x2593, 178},
	{0x25A0, 254}, {0x25AC, 22},  {0x25B2, 30},  {0x25BA, 16},
	{0x25BC, 31},  {0x25C4, 17},  {0x25CB, 9},   {0x25D8, 8},
	{0x25D9, 10},  {0x263A, 1},   {0x263B, 2},   {0x263C, 15},
	{0x2640, 12},  {0x2642, 11},  {0x2660, 6},   {0x2663, 5},
	{0x2665, 3},   {0x2666, 4},   {0x266A, 13},  {0x266B, 14},
	{0xfffd, '?'}
};

const static uint32_t cp437_ext_table[32] = {
	0x0000, 0x263A, 0x263B, 0x2665, 0x2666, 0x2663, 0x2660, 0x2022,
	0x25D8, 0x25CB, 0x25D9, 0x2642, 0x2640, 0x266A, 0x266B, 0x263C,
	0x25BA, 0x25C4, 0x2195, 0x203C, 0x00B6, 0x00A7, 0x25AC, 0x21A8,
	0x2191, 0x2193, 0x2192, 0x2190, 0x221F, 0x2194, 0x25B2, 0x25BC
};

const static uint32_t cp437_unicode_table[128] = {
	0x00C7, 0x00FC, 0x00E9, 0x00E2, 0x00E4, 0x00E0, 0x00E5, 0x00E7,
	0x00EA, 0x00EB, 0x00E8, 0x00EF, 0x00EE, 0x00EC, 0x00C4, 0x00C5,
	0x00C9, 0x00E6, 0x00C6, 0x00F4, 0x00F6, 0x00F2, 0x00FB, 0x00F9,
	0x00FF, 0x00D6, 0x00DC, 0x00A2, 0x00A3, 0x00A5, 0x20A7, 0x0192,
	0x00E1, 0x00ED, 0x00F3, 0x00FA, 0x00F1, 0x00D1, 0x00AA, 0x00BA,
	0x00BF, 0x2310, 0x00AC, 0x00BD, 0x00BC, 0x00A1, 0x00AB, 0x00BB,
	0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x2561, 0x2562, 0x2556,
	0x2555, 0x2563, 0x2551, 0x2557, 0x255D, 0x255C, 0x255B, 0x2510,
	0x2514, 0x2534, 0x252C, 0x251C, 0x2500, 0x253C, 0x255E, 0x255F,
	0x255A, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256C, 0x2567,
	0x2568, 0x2564, 0x2565, 0x2559, 0x2558, 0x2552, 0x2553, 0x256B,
	0x256A, 0x2518, 0x250C, 0x2588, 0x2584, 0x258C, 0x2590, 0x2580,
	0x03B1, 0x00DF, 0x0393, 0x03C0, 0x03A3, 0x03C3, 0x00B5, 0x03C4,
	0x03A6, 0x0398, 0x03A9, 0x03B4, 0x221E, 0x03C6, 0x03B5, 0x2229,
	0x2261, 0x00B1, 0x2265, 0x2264, 0x2320, 0x2321, 0x00F7, 0x2248,
	0x00B0, 0x2219, 0x00B7, 0x221A, 0x207F, 0x00B2, 0x25A0, 0x00A0
};

const static struct ciolib_cpmap cp1251_table[156] = {
	{0x00a0, 160}, {0x00a4, 164}, {0x00a6, 166}, {0x00a7, 167},
	{0x00a9, 169}, {0x00ab, 171}, {0x00ac, 172}, {0x00ad, 173},
	{0x00ae, 174}, {0x00b0, 176}, {0x00b1, 177}, {0x00b5, 181},
	{0x00b6, 20}, {0x00b7, 183}, {0x00bb, 187}, {0x0401, 168},
	{0x0402, 128}, {0x0403, 129}, {0x0404, 170}, {0x0405, 189},
	{0x0406, 178}, {0x0407, 175}, {0x0408, 163}, {0x0409, 138},
	{0x040a, 140}, {0x040b, 142}, {0x040c, 141}, {0x040e, 161},
	{0x040f, 143}, {0x0410, 192}, {0x0411, 193}, {0x0412, 194},
	{0x0413, 195}, {0x0414, 196}, {0x0415, 197}, {0x0416, 198},
	{0x0417, 199}, {0x0418, 200}, {0x0419, 201}, {0x041a, 202},
	{0x041b, 203}, {0x041c, 204}, {0x041d, 205}, {0x041e, 206},
	{0x041f, 207}, {0x0420, 208}, {0x0421, 209}, {0x0422, 210},
	{0x0423, 211}, {0x0424, 212}, {0x0425, 213}, {0x0426, 214},
	{0x0427, 215}, {0x0428, 216}, {0x0429, 217}, {0x042a, 218},
	{0x042b, 219}, {0x042c, 220}, {0x042d, 221}, {0x042e, 222},
	{0x042f, 223}, {0x0430, 224}, {0x0431, 225}, {0x0432, 226},
	{0x0433, 227}, {0x0434, 228}, {0x0435, 229}, {0x0436, 230},
	{0x0437, 231}, {0x0438, 232}, {0x0439, 233}, {0x043a, 234},
	{0x043b, 235}, {0x043c, 236}, {0x043d, 237}, {0x043e, 238},
	{0x043f, 239}, {0x0440, 240}, {0x0441, 241}, {0x0442, 242},
	{0x0443, 243}, {0x0444, 244}, {0x0445, 245}, {0x0446, 246},
	{0x0447, 247}, {0x0448, 248}, {0x0449, 249}, {0x044a, 250},
	{0x044b, 251}, {0x044c, 252}, {0x044d, 253}, {0x044e, 254},
	{0x044f, 255}, {0x0451, 184}, {0x0452, 144}, {0x0453, 131},
	{0x0454, 186}, {0x0455, 190}, {0x0456, 179}, {0x0457, 191},
	{0x0458, 188}, {0x0459, 154}, {0x045a, 156}, {0x045b, 158},
	{0x045c, 157}, {0x045e, 162}, {0x045f, 159}, {0x0490, 165},
	{0x0491, 180}, {0x2013, 150}, {0x2014, 151}, {0x2018, 145},
	{0x2019, 146}, {0x201a, 130}, {0x201c, 147}, {0x201d, 148},
	{0x201e, 132}, {0x2020, 134}, {0x2021, 135}, {0x2022, 7},
	{0x2026, 133}, {0x2030, 137}, {0x2039, 139}, {0x203a, 155},
	{0x203c, 19}, {0x20ac, 136}, {0x2116, 185}, {0x2122, 153},
	{0x2190, 27}, {0x2191, 24}, {0x2192, 26}, {0x2193, 25},
	{0x2194, 29}, {0x2195, 18}, {0x21a8, 23}, {0x221f, 28},
	{0x25ac, 22}, {0x25b2, 30}, {0x25ba, 16}, {0x25bc, 31},
	{0x25c4, 17}, {0x25cb, 9}, {0x25d8, 8}, {0x25d9, 10},
	{0x263a, 1}, {0x263b, 2}, {0x263c, 15}, {0x2640, 12},
	{0x2642, 11}, {0x2660, 6}, {0x2663, 5}, {0x2665, 3},
	{0x2666, 4}, {0x266a, 13}, {0x266b, 14}, {0xfffd, 63}
};

const static uint32_t cp1251_unicode_table[128] = {
	0x0402, 0x0403, 0x201a, 0x0453, 0x201e, 0x2026, 0x2020, 0x2021,
	0x20ac, 0x2030, 0x0409, 0x2039, 0x040a, 0x040c, 0x040b, 0x040f,
	0x0452, 0x2018, 0x2019, 0x201c, 0x201d, 0x2022, 0x2013, 0x2014,
	0xfffd, 0x2122, 0x0459, 0x203a, 0x045a, 0x045c, 0x045b, 0x045f,
	0x00a0, 0x040e, 0x045e, 0x0408, 0x00a4, 0x0490, 0x00a6, 0x00a7,
	0x0401, 0x00a9, 0x0404, 0x00ab, 0x00ac, 0x00ad, 0x00ae, 0x0407,
	0x00b0, 0x00b1, 0x0406, 0x0456, 0x0491, 0x00b5, 0x00b6, 0x00b7,
	0x0451, 0x2116, 0x0454, 0x00bb, 0x0458, 0x0405, 0x0455, 0x0457,
	0x0410, 0x0411, 0x0412, 0x0413, 0x0414, 0x0415, 0x0416, 0x0417,
	0x0418, 0x0419, 0x041a, 0x041b, 0x041c, 0x041d, 0x041e, 0x041f,
	0x0420, 0x0421, 0x0422, 0x0423, 0x0424, 0x0425, 0x0426, 0x0427,
	0x0428, 0x0429, 0x042a, 0x042b, 0x042c, 0x042d, 0x042e, 0x042f,
	0x0430, 0x0431, 0x0432, 0x0433, 0x0434, 0x0435, 0x0436, 0x0437,
	0x0438, 0x0439, 0x043a, 0x043b, 0x043c, 0x043d, 0x043e, 0x043f,
	0x0440, 0x0441, 0x0442, 0x0443, 0x0444, 0x0445, 0x0446, 0x0447,
	0x0448, 0x0449, 0x044a, 0x044b, 0x044c, 0x044d, 0x044e, 0x044f
};

const static struct ciolib_cpmap koi8_r_table[160] = {
	{0x00a0, 154}, {0x00a7,  21}, {0x00a9, 191}, {0x00b0, 156},
	{0x00b2, 157}, {0x00b6,  20}, {0x00b7, 158}, {0x00f7, 159},
	{0x0401, 179}, {0x0410, 225}, {0x0411, 226}, {0x0412, 247},
	{0x0413, 231}, {0x0414, 228}, {0x0415, 229}, {0x0416, 246},
	{0x0417, 250}, {0x0418, 233}, {0x0419, 234}, {0x041a, 235},
	{0x041b, 236}, {0x041c, 237}, {0x041d, 238}, {0x041e, 239},
	{0x041f, 240}, {0x0420, 242}, {0x0421, 243}, {0x0422, 244},
	{0x0423, 245}, {0x0424, 230}, {0x0425, 232}, {0x0426, 227},
	{0x0427, 254}, {0x0428, 251}, {0x0429, 253}, {0x042a, 255},
	{0x042b, 249}, {0x042c, 248}, {0x042d, 252}, {0x042e, 224},
	{0x042f, 241}, {0x0430, 193}, {0x0431, 194}, {0x0432, 215},
	{0x0433, 199}, {0x0434, 196}, {0x0435, 197}, {0x0436, 214},
	{0x0437, 218}, {0x0438, 201}, {0x0439, 202}, {0x043a, 203},
	{0x043b, 204}, {0x043c, 205}, {0x043d, 206}, {0x043e, 207},
	{0x043f, 208}, {0x0440, 210}, {0x0441, 211}, {0x0442, 212},
	{0x0443, 213}, {0x0444, 198}, {0x0445, 200}, {0x0446, 195},
	{0x0447, 222}, {0x0448, 219}, {0x0449, 221}, {0x044a, 223},
	{0x044b, 217}, {0x044c, 216}, {0x044d, 220}, {0x044e, 192},
	{0x044f, 209}, {0x0451, 163}, {0x2022,   7}, {0x203c,  19},
	{0x2190,  27}, {0x2191,  24}, {0x2192,  26}, {0x2193,  25},
	{0x2194,  29}, {0x2195,  18}, {0x21a8,  23}, {0x2219, 149},
	{0x221a, 150}, {0x221f,  28}, {0x2248, 151}, {0x2264, 152},
	{0x2265, 153}, {0x2320, 147}, {0x2321, 155}, {0x2500, 128},
	{0x2502, 129}, {0x250c, 130}, {0x2510, 131}, {0x2514, 132},
	{0x2518, 133}, {0x251c, 134}, {0x2524, 135}, {0x252c, 136},
	{0x2534, 137}, {0x253c, 138}, {0x2550, 160}, {0x2551, 161},
	{0x2552, 162}, {0x2553, 164}, {0x2554, 165}, {0x2555, 166},
	{0x2556, 167}, {0x2557, 168}, {0x2558, 169}, {0x2559, 170},
	{0x255a, 171}, {0x255b, 172}, {0x255c, 173}, {0x255d, 174},
	{0x255e, 175}, {0x255f, 176}, {0x2560, 177}, {0x2561, 178},
	{0x2562, 180}, {0x2563, 181}, {0x2564, 182}, {0x2565, 183},
	{0x2566, 184}, {0x2567, 185}, {0x2568, 186}, {0x2569, 187},
	{0x256a, 188}, {0x256b, 189}, {0x256c, 190}, {0x2580, 139},
	{0x2584, 140}, {0x2588, 141}, {0x258c, 142}, {0x2590, 143},
	{0x2591, 144}, {0x2592, 145}, {0x2593, 146}, {0x25a0, 148},
	{0x25ac,  22}, {0x25b2,  30}, {0x25ba,  16}, {0x25bc,  31},
	{0x25c4,  17}, {0x25cb,   9}, {0x25d8,   8}, {0x25d9,  10},
	{0x263a,   1}, {0x263b,   2}, {0x263c,  15}, {0x2640,  12},
	{0x2642,  11}, {0x2660,   6}, {0x2663,   5}, {0x2665,   3},
	{0x2666,   4}, {0x266a,  13}, {0x266b,  14}, {0xfffd, '?'}
};

const static uint32_t koi8_r_unicode_table[128] = {
	0x2500, 0x2502, 0x250c, 0x2510, 0x2514, 0x2518, 0x251c, 0x2524,
	0x252c, 0x2534, 0x253c, 0x2580, 0x2584, 0x2588, 0x258c, 0x2590,
	0x2591, 0x2592, 0x2593, 0x2320, 0x25a0, 0x2219, 0x221a, 0x2248,
	0x2264, 0x2265, 0x00a0, 0x2321, 0x00b0, 0x00b2, 0x00b7, 0x00f7,
	0x2550, 0x2551, 0x2552, 0x0451, 0x2553, 0x2554, 0x2555, 0x2556,
	0x2557, 0x2558, 0x2559, 0x255a, 0x255b, 0x255c, 0x255d, 0x255e,
	0x255f, 0x2560, 0x2561, 0x0401, 0x2562, 0x2563, 0x2564, 0x2565,
	0x2566, 0x2567, 0x2568, 0x2569, 0x256a, 0x256b, 0x256c, 0x00a9,
	0x044e, 0x0430, 0x0431, 0x0446, 0x0434, 0x0435, 0x0444, 0x0433,
	0x0445, 0x0438, 0x0439, 0x043a, 0x043b, 0x043c, 0x043d, 0x043e,
	0x043f, 0x044f, 0x0440, 0x0441, 0x0442, 0x0443, 0x0436, 0x0432,
	0x044c, 0x044b, 0x0437, 0x0448, 0x044d, 0x0449, 0x0447, 0x044a,
	0x042e, 0x0410, 0x0411, 0x0426, 0x0414, 0x0415, 0x0424, 0x0413,
	0x0425, 0x0418, 0x0419, 0x041a, 0x041b, 0x041c, 0x041d, 0x041e,
	0x041f, 0x042f, 0x0420, 0x0421, 0x0422, 0x0423, 0x0416, 0x0412,
	0x042c, 0x042b, 0x0417, 0x0428, 0x042d, 0x0429, 0x0427, 0x042a
};

const static uint32_t empty_ext_table[32] = {
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd
};

const static struct ciolib_cpmap iso8859_2_table[127] = {
	{0x00a0, 160}, {0x00a4, 164}, {0x00a7, 167}, {0x00a8, 168},
	{0x00ad, 173}, {0x00b0, 176}, {0x00b4, 180}, {0x00b6, 20},
	{0x00b8, 184}, {0x00c1, 193}, {0x00c2, 194}, {0x00c4, 196},
	{0x00c7, 199}, {0x00c9, 201}, {0x00cb, 203}, {0x00cd, 205},
	{0x00ce, 206}, {0x00d3, 211}, {0x00d4, 212}, {0x00d6, 214},
	{0x00d7, 215}, {0x00da, 218}, {0x00dc, 220}, {0x00dd, 221},
	{0x00df, 223}, {0x00e1, 225}, {0x00e2, 226}, {0x00e4, 228},
	{0x00e7, 231}, {0x00e9, 233}, {0x00eb, 235}, {0x00ed, 237},
	{0x00ee, 238}, {0x00f3, 243}, {0x00f4, 244}, {0x00f6, 246},
	{0x00f7, 247}, {0x00fa, 250}, {0x00fc, 252}, {0x00fd, 253},
	{0x0102, 195}, {0x0103, 227}, {0x0104, 161}, {0x0105, 177},
	{0x0106, 198}, {0x0107, 230}, {0x010c, 200}, {0x010d, 232},
	{0x010e, 207}, {0x010f, 239}, {0x0110, 208}, {0x0111, 240},
	{0x0118, 202}, {0x0119, 234}, {0x011a, 204}, {0x011b, 236},
	{0x0139, 197}, {0x013a, 229}, {0x013d, 165}, {0x013e, 181},
	{0x0141, 163}, {0x0142, 179}, {0x0143, 209}, {0x0144, 241},
	{0x0147, 210}, {0x0148, 242}, {0x0150, 213}, {0x0151, 245},
	{0x0154, 192}, {0x0155, 224}, {0x0158, 216}, {0x0159, 248},
	{0x015a, 166}, {0x015b, 182}, {0x015e, 170}, {0x015f, 186},
	{0x0160, 169}, {0x0161, 185}, {0x0162, 222}, {0x0163, 254},
	{0x0164, 171}, {0x0165, 187}, {0x016e, 217}, {0x016f, 249},
	{0x0170, 219}, {0x0171, 251}, {0x0179, 172}, {0x017a, 188},
	{0x017b, 175}, {0x017c, 191}, {0x017d, 174}, {0x017e, 190},
	{0x02c7, 183}, {0x02d8, 162}, {0x02d9, 255}, {0x02db, 178},
	{0x02dd, 189}, {0x2022, 7}, {0x203c, 19}, {0x2190, 27},
	{0x2191, 24}, {0x2192, 26}, {0x2193, 25}, {0x2194, 29},
	{0x2195, 18}, {0x21a8, 23}, {0x221f, 28}, {0x25ac, 22},
	{0x25b2, 30}, {0x25ba, 16}, {0x25bc, 31}, {0x25c4, 17},
	{0x25cb, 9}, {0x25d8, 8}, {0x25d9, 10}, {0x263a, 1},
	{0x263b, 2}, {0x263c, 15}, {0x2640, 12}, {0x2642, 11},
	{0x2660, 6}, {0x2663, 5}, {0x2665, 3}, {0x2666, 4},
	{0x266a, 13}, {0x266b, 14}, {0xfffd, 63}
};

const static uint32_t iso8859_2_unicode_table[128] = {
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0x00a0, 0x0104, 0x02d8, 0x0141, 0x00a4, 0x013d, 0x015a, 0x00a7,
	0x00a8, 0x0160, 0x015e, 0x0164, 0x0179, 0x00ad, 0x017d, 0x017b,
	0x00b0, 0x0105, 0x02db, 0x0142, 0x00b4, 0x013e, 0x015b, 0x02c7,
	0x00b8, 0x0161, 0x015f, 0x0165, 0x017a, 0x02dd, 0x017e, 0x017c,
	0x0154, 0x00c1, 0x00c2, 0x0102, 0x00c4, 0x0139, 0x0106, 0x00c7,
	0x010c, 0x00c9, 0x0118, 0x00cb, 0x011a, 0x00cd, 0x00ce, 0x010e,
	0x0110, 0x0143, 0x0147, 0x00d3, 0x00d4, 0x0150, 0x00d6, 0x00d7,
	0x0158, 0x016e, 0x00da, 0x0170, 0x00dc, 0x00dd, 0x0162, 0x00df,
	0x0155, 0x00e1, 0x00e2, 0x0103, 0x00e4, 0x013a, 0x0107, 0x00e7,
	0x010d, 0x00e9, 0x0119, 0x00eb, 0x011b, 0x00ed, 0x00ee, 0x010f,
	0x0111, 0x0144, 0x0148, 0x00f3, 0x00f4, 0x0151, 0x00f6, 0x00f7,
	0x0159, 0x016f, 0x00fa, 0x0171, 0x00fc, 0x00fd, 0x0163, 0x02d9
};

const static struct ciolib_cpmap iso8859_4_table[127] = {
	{0x00a0, 160}, {0x00a4, 164}, {0x00a7, 167}, {0x00a8, 168},
	{0x00ad, 173}, {0x00af, 175}, {0x00b0, 176}, {0x00b4, 180},
	{0x00b6, 20}, {0x00b8, 184}, {0x00c1, 193}, {0x00c2, 194},
	{0x00c3, 195}, {0x00c4, 196}, {0x00c5, 197}, {0x00c6, 198},
	{0x00c9, 201}, {0x00cb, 203}, {0x00cd, 205}, {0x00ce, 206},
	{0x00d4, 212}, {0x00d5, 213}, {0x00d6, 214}, {0x00d7, 215},
	{0x00d8, 216}, {0x00da, 218}, {0x00db, 219}, {0x00dc, 220},
	{0x00df, 223}, {0x00e1, 225}, {0x00e2, 226}, {0x00e3, 227},
	{0x00e4, 228}, {0x00e5, 229}, {0x00e6, 230}, {0x00e9, 233},
	{0x00eb, 235}, {0x00ed, 237}, {0x00ee, 238}, {0x00f4, 244},
	{0x00f5, 245}, {0x00f6, 246}, {0x00f7, 247}, {0x00f8, 248},
	{0x00fa, 250}, {0x00fb, 251}, {0x00fc, 252}, {0x0100, 192},
	{0x0101, 224}, {0x0104, 161}, {0x0105, 177}, {0x010c, 200},
	{0x010d, 232}, {0x0110, 208}, {0x0111, 240}, {0x0112, 170},
	{0x0113, 186}, {0x0116, 204}, {0x0117, 236}, {0x0118, 202},
	{0x0119, 234}, {0x0122, 171}, {0x0123, 187}, {0x0128, 165},
	{0x0129, 181}, {0x012a, 207}, {0x012b, 239}, {0x012e, 199},
	{0x012f, 231}, {0x0136, 211}, {0x0137, 243}, {0x0138, 162},
	{0x013b, 166}, {0x013c, 182}, {0x0145, 209}, {0x0146, 241},
	{0x014a, 189}, {0x014b, 191}, {0x014c, 210}, {0x014d, 242},
	{0x0156, 163}, {0x0157, 179}, {0x0160, 169}, {0x0161, 185},
	{0x0166, 172}, {0x0167, 188}, {0x0168, 221}, {0x0169, 253},
	{0x016a, 222}, {0x016b, 254}, {0x0172, 217}, {0x0173, 249},
	{0x017d, 174}, {0x017e, 190}, {0x02c7, 183}, {0x02d9, 255},
	{0x02db, 178}, {0x2022, 7}, {0x203c, 19}, {0x2190, 27},
	{0x2191, 24}, {0x2192, 26}, {0x2193, 25}, {0x2194, 29},
	{0x2195, 18}, {0x21a8, 23}, {0x221f, 28}, {0x25ac, 22},
	{0x25b2, 30}, {0x25ba, 16}, {0x25bc, 31}, {0x25c4, 17},
	{0x25cb, 9}, {0x25d8, 8}, {0x25d9, 10}, {0x263a, 1},
	{0x263b, 2}, {0x263c, 15}, {0x2640, 12}, {0x2642, 11},
	{0x2660, 6}, {0x2663, 5}, {0x2665, 3}, {0x2666, 4},
	{0x266a, 13}, {0x266b, 14}, {0xfffd, 63}
};

const static uint32_t iso8859_4_unicode_table[128] = {
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0x00a0, 0x0104, 0x0138, 0x0156, 0x00a4, 0x0128, 0x013b, 0x00a7,
	0x00a8, 0x0160, 0x0112, 0x0122, 0x0166, 0x00ad, 0x017d, 0x00af,
	0x00b0, 0x0105, 0x02db, 0x0157, 0x00b4, 0x0129, 0x013c, 0x02c7,
	0x00b8, 0x0161, 0x0113, 0x0123, 0x0167, 0x014a, 0x017e, 0x014b,
	0x0100, 0x00c1, 0x00c2, 0x00c3, 0x00c4, 0x00c5, 0x00c6, 0x012e,
	0x010c, 0x00c9, 0x0118, 0x00cb, 0x0116, 0x00cd, 0x00ce, 0x012a,
	0x0110, 0x0145, 0x014c, 0x0136, 0x00d4, 0x00d5, 0x00d6, 0x00d7,
	0x00d8, 0x0172, 0x00da, 0x00db, 0x00dc, 0x0168, 0x016a, 0x00df,
	0x0101, 0x00e1, 0x00e2, 0x00e3, 0x00e4, 0x00e5, 0x00e6, 0x012f,
	0x010d, 0x00e9, 0x0119, 0x00eb, 0x0117, 0x00ed, 0x00ee, 0x012b,
	0x0111, 0x0146, 0x014d, 0x0137, 0x00f4, 0x00f5, 0x00f6, 0x00f7,
	0x00f8, 0x0173, 0x00fa, 0x00fb, 0x00fc, 0x0169, 0x016b, 0x02d9
};

#if 0
const static struct ciolib_cpmap cp866_table[160] = {
	{0x00a0, 255}, {0x00a4, 253}, {0x00a7,  21}, {0x00b0, 248},
	{0x00b6,  20}, {0x00b7, 250}, {0x0401, 240}, {0x0404, 242},
	{0x0407, 244}, {0x040e, 246}, {0x0410, 128}, {0x0411, 129},
	{0x0412, 130}, {0x0413, 131}, {0x0414, 132}, {0x0415, 133},
	{0x0416, 134}, {0x0417, 135}, {0x0418, 136}, {0x0419, 137},
	{0x041a, 138}, {0x041b, 139}, {0x041c, 140}, {0x041d, 141},
	{0x041e, 142}, {0x041f, 143}, {0x0420, 144}, {0x0421, 145},
	{0x0422, 146}, {0x0423, 147}, {0x0424, 148}, {0x0425, 149},
	{0x0426, 150}, {0x0427, 151}, {0x0428, 152}, {0x0429, 153},
	{0x042a, 154}, {0x042b, 155}, {0x042c, 156}, {0x042d, 157},
	{0x042e, 158}, {0x042f, 159}, {0x0430, 160}, {0x0431, 161},
	{0x0432, 162}, {0x0433, 163}, {0x0434, 164}, {0x0435, 165},
	{0x0436, 166}, {0x0437, 167}, {0x0438, 168}, {0x0439, 169},
	{0x043a, 170}, {0x043b, 171}, {0x043c, 172}, {0x043d, 173},
	{0x043e, 174}, {0x043f, 175}, {0x0440, 224}, {0x0441, 225},
	{0x0442, 226}, {0x0443, 227}, {0x0444, 228}, {0x0445, 229},
	{0x0446, 230}, {0x0447, 231}, {0x0448, 232}, {0x0449, 233},
	{0x044a, 234}, {0x044b, 235}, {0x044c, 236}, {0x044d, 237},
	{0x044e, 238}, {0x044f, 239}, {0x0451, 241}, {0x0454, 243},
	{0x0457, 245}, {0x045e, 247}, {0x2022,   7}, {0x203c,  19},
	{0x2116, 252}, {0x2190,  27}, {0x2191,  24}, {0x2192,  26},
	{0x2193,  25}, {0x2194,  29}, {0x2195,  18}, {0x21a8,  23},
	{0x2219, 249}, {0x221a, 251}, {0x221f,  28}, {0x2500, 196},
	{0x2502, 179}, {0x250c, 218}, {0x2510, 191}, {0x2514, 192},
	{0x2518, 217}, {0x251c, 195}, {0x2524, 180}, {0x252c, 194},
	{0x2534, 193}, {0x253c, 197}, {0x2550, 205}, {0x2551, 186},
	{0x2552, 213}, {0x2553, 214}, {0x2554, 201}, {0x2555, 184},
	{0x2556, 183}, {0x2557, 187}, {0x2558, 212}, {0x2559, 211},
	{0x255a, 200}, {0x255b, 190}, {0x255c, 189}, {0x255d, 188},
	{0x255e, 198}, {0x255f, 199}, {0x2560, 204}, {0x2561, 181},
	{0x2562, 182}, {0x2563, 185}, {0x2564, 209}, {0x2565, 210},
	{0x2566, 203}, {0x2567, 207}, {0x2568, 208}, {0x2569, 202},
	{0x256a, 216}, {0x256b, 215}, {0x256c, 206}, {0x2580, 223},
	{0x2584, 220}, {0x2588, 219}, {0x258c, 221}, {0x2590, 222},
	{0x2591, 176}, {0x2592, 177}, {0x2593, 178}, {0x25a0, 254},
	{0x25ac,  22}, {0x25b2,  30}, {0x25ba,  16}, {0x25bc,  31},
	{0x25c4,  17}, {0x25cb,   9}, {0x25d8,   8}, {0x25d9,  10},
	{0x263a,   1}, {0x263b,   2}, {0x263c,  15}, {0x2640,  12},
	{0x2642,  11}, {0x2660,   6}, {0x2663,   5}, {0x2665,   3},
	{0x2666,   4}, {0x266a,  13}, {0x266b,  14}, {0xfffd, '?'}
};

const static uint32_t cp866_unicode_table[128] = {
	0x0410, 0x0411, 0x0412, 0x0413, 0x0414, 0x0415, 0x0416, 0x0417,
	0x0418, 0x0419, 0x041a, 0x041b, 0x041c, 0x041d, 0x041e, 0x041f,
	0x0420, 0x0421, 0x0422, 0x0423, 0x0424, 0x0425, 0x0426, 0x0427,
	0x0428, 0x0429, 0x042a, 0x042b, 0x042c, 0x042d, 0x042e, 0x042f,
	0x0430, 0x0431, 0x0432, 0x0433, 0x0434, 0x0435, 0x0436, 0x0437,
	0x0438, 0x0439, 0x043a, 0x043b, 0x043c, 0x043d, 0x043e, 0x043f,
	0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x2561, 0x2562, 0x2556,
	0x2555, 0x2563, 0x2551, 0x2557, 0x255d, 0x255c, 0x255b, 0x2510,
	0x2514, 0x2534, 0x252c, 0x251c, 0x2500, 0x253c, 0x255e, 0x255f,
	0x255a, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256c, 0x2567,
	0x2568, 0x2564, 0x2565, 0x2559, 0x2558, 0x2552, 0x2553, 0x256b,
	0x256a, 0x2518, 0x250c, 0x2588, 0x2584, 0x258c, 0x2590, 0x2580,
	0x0440, 0x0441, 0x0442, 0x0443, 0x0444, 0x0445, 0x0446, 0x0447,
	0x0448, 0x0449, 0x044a, 0x044b, 0x044c, 0x044d, 0x044e, 0x044f,
	0x0401, 0x0451, 0x0404, 0x0454, 0x0407, 0x0457, 0x040e, 0x045e,
	0x00b0, 0x2219, 0x00b7, 0x221a, 0x2116, 0x00a4, 0x25a0, 0x00a0
};
#endif

const static struct ciolib_cpmap cp866m_table[160] = {
	{0x00a0, 255}, {0x00a7,  21}, {0x00b0, 248}, {0x00B2, 253},
	{0x00b6,  20}, {0x00b7, 250}, {0x00F7, 246}, {0x0401, 240},
	{0x0410, 128}, {0x0411, 129}, {0x0412, 130}, {0x0413, 131},
	{0x0414, 132}, {0x0415, 133}, {0x0416, 134}, {0x0417, 135},
	{0x0418, 136}, {0x0419, 137}, {0x041a, 138}, {0x041b, 139},
	{0x041c, 140}, {0x041d, 141}, {0x041e, 142}, {0x041f, 143},
	{0x0420, 144}, {0x0421, 145}, {0x0422, 146}, {0x0423, 147},
	{0x0424, 148}, {0x0425, 149}, {0x0426, 150}, {0x0427, 151},
	{0x0428, 152}, {0x0429, 153}, {0x042a, 154}, {0x042b, 155},
	{0x042c, 156}, {0x042d, 157}, {0x042e, 158}, {0x042f, 159},
	{0x0430, 160}, {0x0431, 161}, {0x0432, 162}, {0x0433, 163},
	{0x0434, 164}, {0x0435, 165}, {0x0436, 166}, {0x0437, 167},
	{0x0438, 168}, {0x0439, 169}, {0x043a, 170}, {0x043b, 171},
	{0x043c, 172}, {0x043d, 173}, {0x043e, 174}, {0x043f, 175},
	{0x0440, 224}, {0x0441, 225}, {0x0442, 226}, {0x0443, 227},
	{0x0444, 228}, {0x0445, 229}, {0x0446, 230}, {0x0447, 231},
	{0x0448, 232}, {0x0449, 233}, {0x044a, 234}, {0x044b, 235},
	{0x044c, 236}, {0x044d, 237}, {0x044e, 238}, {0x044f, 239},
	{0x0451, 241}, {0x045F, 252}, {0x2022,   7}, {0x203c,  19},
	{0x2190,  27}, {0x2191,  24}, {0x2192,  26}, {0x2193,  25},
	{0x2194,  29}, {0x2195,  18}, {0x21a8,  23}, {0x2219, 249},
	{0x221a, 251}, {0x221f,  28}, {0x2264, 243}, {0x2265, 242},
	{0x2320, 244}, {0x2321, 245}, {0x2248, 247}, {0x2500, 196},
	{0x2502, 179}, {0x250c, 218}, {0x2510, 191}, {0x2514, 192},
	{0x2518, 217}, {0x251c, 195}, {0x2524, 180}, {0x252c, 194},
	{0x2534, 193}, {0x253c, 197}, {0x2550, 205}, {0x2551, 186},
	{0x2552, 213}, {0x2553, 214}, {0x2554, 201}, {0x2555, 184},
	{0x2556, 183}, {0x2557, 187}, {0x2558, 212}, {0x2559, 211},
	{0x255a, 200}, {0x255b, 190}, {0x255c, 189}, {0x255d, 188},
	{0x255e, 198}, {0x255f, 199}, {0x2560, 204}, {0x2561, 181},
	{0x2562, 182}, {0x2563, 185}, {0x2564, 209}, {0x2565, 210},
	{0x2566, 203}, {0x2567, 207}, {0x2568, 208}, {0x2569, 202},
	{0x256a, 216}, {0x256b, 215}, {0x256c, 206}, {0x2580, 223},
	{0x2584, 220}, {0x2588, 219}, {0x258c, 221}, {0x2590, 222},
	{0x2591, 176}, {0x2592, 177}, {0x2593, 178}, {0x25a0, 254},
	{0x25ac,  22}, {0x25b2,  30}, {0x25ba,  16}, {0x25bc,  31},
	{0x25c4,  17}, {0x25cb,   9}, {0x25d8,   8}, {0x25d9,  10},
	{0x263a,   1}, {0x263b,   2}, {0x263c,  15}, {0x2640,  12},
	{0x2642,  11}, {0x2660,   6}, {0x2663,   5}, {0x2665,   3},
	{0x2666,   4}, {0x266a,  13}, {0x266b,  14}, {0xfffd, '?'}
};

const static uint32_t cp866m_unicode_table[128] = {
	0x0410, 0x0411, 0x0412, 0x0413, 0x0414, 0x0415, 0x0416, 0x0417,
	0x0418, 0x0419, 0x041a, 0x041b, 0x041c, 0x041d, 0x041e, 0x041f,
	0x0420, 0x0421, 0x0422, 0x0423, 0x0424, 0x0425, 0x0426, 0x0427,
	0x0428, 0x0429, 0x042a, 0x042b, 0x042c, 0x042d, 0x042e, 0x042f,
	0x0430, 0x0431, 0x0432, 0x0433, 0x0434, 0x0435, 0x0436, 0x0437,
	0x0438, 0x0439, 0x043a, 0x043b, 0x043c, 0x043d, 0x043e, 0x043f,
	0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x2561, 0x2562, 0x2556,
	0x2555, 0x2563, 0x2551, 0x2557, 0x255d, 0x255c, 0x255b, 0x2510,
	0x2514, 0x2534, 0x252c, 0x251c, 0x2500, 0x253c, 0x255e, 0x255f,
	0x255a, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256c, 0x2567,
	0x2568, 0x2564, 0x2565, 0x2559, 0x2558, 0x2552, 0x2553, 0x256b,
	0x256a, 0x2518, 0x250c, 0x2588, 0x2584, 0x258c, 0x2590, 0x2580,
	0x0440, 0x0441, 0x0442, 0x0443, 0x0444, 0x0445, 0x0446, 0x0447,
	0x0448, 0x0449, 0x044a, 0x044b, 0x044c, 0x044d, 0x044e, 0x044f,
	0x0401, 0x0451, 0x2265, 0x2264, 0x2320, 0x2321, 0x00F7, 0x2248,
	0x00b0, 0x2219, 0x00b7, 0x221a, 0x045F, 0x00B2, 0x25a0, 0x00a0
};

const static struct ciolib_cpmap iso8859_9_table[126] = {
	{0x00a0, 160}, {0x00a1, 161}, {0x00a2, 162}, {0x00a3, 163},
	{0x00a4, 164}, {0x00a5, 165}, {0x00a6, 166}, {0x00a7, 167},
	{0x00a8, 168}, {0x00a9, 169}, {0x00aa, 170}, {0x00ab, 171},
	{0x00ac, 172}, {0x00ad, 173}, {0x00ae, 174}, {0x00af, 175},
	{0x00b0, 176}, {0x00b1, 177}, {0x00b2, 178}, {0x00b3, 179},
	{0x00b4, 180}, {0x00b5, 181}, {0x00b6, 182}, {0x00b7, 183},
	{0x00b8, 184}, {0x00b9, 185}, {0x00ba, 186}, {0x00bb, 187},
	{0x00bc, 188}, {0x00bd, 189}, {0x00be, 190}, {0x00bf, 191},
	{0x00c0, 192}, {0x00c1, 193}, {0x00c2, 194}, {0x00c3, 195},
	{0x00c4, 196}, {0x00c5, 197}, {0x00c6, 198}, {0x00c7, 199},
	{0x00c8, 200}, {0x00c9, 201}, {0x00ca, 202}, {0x00cb, 203},
	{0x00cc, 204}, {0x00cd, 205}, {0x00ce, 206}, {0x00cf, 207},
	{0x00d1, 209}, {0x00d2, 210}, {0x00d3, 211}, {0x00d4, 212},
	{0x00d5, 213}, {0x00d6, 214}, {0x00d7, 215}, {0x00d8, 216},
	{0x00d9, 217}, {0x00da, 218}, {0x00db, 219}, {0x00dc, 220},
	{0x00df, 223}, {0x00e0, 224}, {0x00e1, 225}, {0x00e2, 226},
	{0x00e3, 227}, {0x00e4, 228}, {0x00e5, 229}, {0x00e6, 230},
	{0x00e7, 231}, {0x00e8, 232}, {0x00e9, 233}, {0x00ea, 234},
	{0x00eb, 235}, {0x00ec, 236}, {0x00ed, 237}, {0x00ee, 238},
	{0x00ef, 239}, {0x00f1, 241}, {0x00f2, 242}, {0x00f3, 243},
	{0x00f4, 244}, {0x00f5, 245}, {0x00f6, 246}, {0x00f7, 247},
	{0x00f8, 248}, {0x00f9, 249}, {0x00fa, 250}, {0x00fb, 251},
	{0x00fc, 252}, {0x00ff, 255}, {0x011e, 208}, {0x011f, 240},
	{0x0130, 221}, {0x0131, 253}, {0x015e, 222}, {0x015f, 254},
	{0x2022, 7}, {0x203c, 19}, {0x2190, 27}, {0x2191, 24},
	{0x2192, 26}, {0x2193, 25}, {0x2194, 29}, {0x2195, 18},
	{0x21a8, 23}, {0x221f, 28}, {0x25ac, 22}, {0x25b2, 30},
	{0x25ba, 16}, {0x25bc, 31}, {0x25c4, 17}, {0x25cb, 9},
	{0x25d8, 8}, {0x25d9, 10}, {0x263a, 1}, {0x263b, 2},
	{0x263c, 15}, {0x2640, 12}, {0x2642, 11}, {0x2660, 6},
	{0x2663, 5}, {0x2665, 3}, {0x2666, 4}, {0x266a, 13},
	{0x266b, 14}, {0xfffd, 63}
};

const static uint32_t iso8859_9_unicode_table[128] = {
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0x00a0, 0x00a1, 0x00a2, 0x00a3, 0x00a4, 0x00a5, 0x00a6, 0x00a7,
	0x00a8, 0x00a9, 0x00aa, 0x00ab, 0x00ac, 0x00ad, 0x00ae, 0x00af,
	0x00b0, 0x00b1, 0x00b2, 0x00b3, 0x00b4, 0x00b5, 0x00b6, 0x00b7,
	0x00b8, 0x00b9, 0x00ba, 0x00bb, 0x00bc, 0x00bd, 0x00be, 0x00bf,
	0x00c0, 0x00c1, 0x00c2, 0x00c3, 0x00c4, 0x00c5, 0x00c6, 0x00c7,
	0x00c8, 0x00c9, 0x00ca, 0x00cb, 0x00cc, 0x00cd, 0x00ce, 0x00cf,
	0x011e, 0x00d1, 0x00d2, 0x00d3, 0x00d4, 0x00d5, 0x00d6, 0x00d7,
	0x00d8, 0x00d9, 0x00da, 0x00db, 0x00dc, 0x0130, 0x015e, 0x00df,
	0x00e0, 0x00e1, 0x00e2, 0x00e3, 0x00e4, 0x00e5, 0x00e6, 0x00e7,
	0x00e8, 0x00e9, 0x00ea, 0x00eb, 0x00ec, 0x00ed, 0x00ee, 0x00ef,
	0x011f, 0x00f1, 0x00f2, 0x00f3, 0x00f4, 0x00f5, 0x00f6, 0x00f7,
	0x00f8, 0x00f9, 0x00fa, 0x00fb, 0x00fc, 0x0131, 0x015f, 0x00ff
};

const static struct ciolib_cpmap iso8859_8_table[90] = {
	{0x00a0, 160}, {0x00a2, 162}, {0x00a3, 163}, {0x00a4, 164},
	{0x00a5, 165}, {0x00a6, 166}, {0x00a7, 167}, {0x00a8, 168},
	{0x00a9, 169}, {0x00ab, 171}, {0x00ac, 172}, {0x00ad, 173},
	{0x00ae, 174}, {0x00af, 175}, {0x00b0, 176}, {0x00b1, 177},
	{0x00b2, 178}, {0x00b3, 179}, {0x00b4, 180}, {0x00b5, 181},
	{0x00b6, 182}, {0x00b7, 183}, {0x00b8, 184}, {0x00b9, 185},
	{0x00bb, 187}, {0x00bc, 188}, {0x00bd, 189}, {0x00be, 190},
	{0x00d7, 170}, {0x00f7, 186}, {0x05d0, 224}, {0x05d1, 225},
	{0x05d2, 226}, {0x05d3, 227}, {0x05d4, 228}, {0x05d5, 229},
	{0x05d6, 230}, {0x05d7, 231}, {0x05d8, 232}, {0x05d9, 233},
	{0x05da, 234}, {0x05db, 235}, {0x05dc, 236}, {0x05dd, 237},
	{0x05de, 238}, {0x05df, 239}, {0x05e0, 240}, {0x05e1, 241},
	{0x05e2, 242}, {0x05e3, 243}, {0x05e4, 244}, {0x05e5, 245},
	{0x05e6, 246}, {0x05e7, 247}, {0x05e8, 248}, {0x05e9, 249},
	{0x05ea, 250}, {0x200e, 253}, {0x200f, 254}, {0x2017, 223},
	{0x2022, 7}, {0x203c, 19}, {0x2190, 27}, {0x2191, 24},
	{0x2192, 26}, {0x2193, 25}, {0x2194, 29}, {0x2195, 18},
	{0x21a8, 23}, {0x221f, 28}, {0x25ac, 22}, {0x25b2, 30},
	{0x25ba, 16}, {0x25bc, 31}, {0x25c4, 17}, {0x25cb, 9},
	{0x25d8, 8}, {0x25d9, 10}, {0x263a, 1}, {0x263b, 2},
	{0x263c, 15}, {0x2640, 12}, {0x2642, 11}, {0x2660, 6},
	{0x2663, 5}, {0x2665, 3}, {0x2666, 4}, {0x266a, 13},
	{0x266b, 14}, {0xfffd, 63}
};

const static uint32_t iso8859_8_unicode_table[128] = {
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0x00a0, 0xfffd, 0x00a2, 0x00a3, 0x00a4, 0x00a5, 0x00a6, 0x00a7,
	0x00a8, 0x00a9, 0x00d7, 0x00ab, 0x00ac, 0x00ad, 0x00ae, 0x00af,
	0x00b0, 0x00b1, 0x00b2, 0x00b3, 0x00b4, 0x00b5, 0x00b6, 0x00b7,
	0x00b8, 0x00b9, 0x00f7, 0x00bb, 0x00bc, 0x00bd, 0x00be, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0x2017,
	0x05d0, 0x05d1, 0x05d2, 0x05d3, 0x05d4, 0x05d5, 0x05d6, 0x05d7,
	0x05d8, 0x05d9, 0x05da, 0x05db, 0x05dc, 0x05dd, 0x05de, 0x05df,
	0x05e0, 0x05e1, 0x05e2, 0x05e3, 0x05e4, 0x05e5, 0x05e6, 0x05e7,
	0x05e8, 0x05e9, 0x05ea, 0xfffd, 0xfffd, 0x200e, 0x200f, 0xfffd
};

const static struct ciolib_cpmap koi8_u_table[160] = {
	{0x00a0, 154}, {0x00a7,  21}, {0x00a9, 191}, {0x00b0, 156},
	{0x00b2, 157}, {0x00b6,  20}, {0x00b7, 158}, {0x00f7, 159},
	{0x0401, 179}, {0x0404, 180}, {0x0406, 182}, {0x0407, 183},
	{0x0410, 225}, {0x0411, 226}, {0x0412, 247}, {0x0413, 231},
	{0x0414, 228}, {0x0415, 229}, {0x0416, 246}, {0x0417, 250},
	{0x0418, 233}, {0x0419, 234}, {0x041a, 235}, {0x041b, 236},
	{0x041c, 237}, {0x041d, 238}, {0x041e, 239}, {0x041f, 240},
	{0x0420, 242}, {0x0421, 243}, {0x0422, 244}, {0x0423, 245},
	{0x0424, 230}, {0x0425, 232}, {0x0426, 227}, {0x0427, 254},
	{0x0428, 251}, {0x0429, 253}, {0x042a, 255}, {0x042b, 249},
	{0x042c, 248}, {0x042d, 252}, {0x042e, 224}, {0x042f, 241},
	{0x0430, 193}, {0x0431, 194}, {0x0432, 215}, {0x0433, 199},
	{0x0434, 196}, {0x0435, 197}, {0x0436, 214}, {0x0437, 218},
	{0x0438, 201}, {0x0439, 202}, {0x043a, 203}, {0x043b, 204},
	{0x043c, 205}, {0x043d, 206}, {0x043e, 207}, {0x043f, 208},
	{0x0440, 210}, {0x0441, 211}, {0x0442, 212}, {0x0443, 213},
	{0x0444, 198}, {0x0445, 200}, {0x0446, 195}, {0x0447, 222},
	{0x0448, 219}, {0x0449, 221}, {0x044a, 223}, {0x044b, 217},
	{0x044c, 216}, {0x044d, 220}, {0x044e, 192}, {0x044f, 209},
	{0x0451, 163}, {0x0454, 164}, {0x0456, 166}, {0x0457, 167},
	{0x0490, 189}, {0x0491, 173}, {0x2022,   7}, {0x203c,  19},
	{0x2190,  27}, {0x2191,  24}, {0x2192,  26}, {0x2193,  25},
	{0x2194,  29}, {0x2195,  18}, {0x21a8,  23}, {0x2219, 149},
	{0x221a, 150}, {0x221f,  28}, {0x2248, 151}, {0x2264, 152},
	{0x2265, 153}, {0x2320, 147}, {0x2321, 155}, {0x2500, 128},
	{0x2502, 129}, {0x250c, 130}, {0x2510, 131}, {0x2514, 132},
	{0x2518, 133}, {0x251c, 134}, {0x2524, 135}, {0x252c, 136},
	{0x2534, 137}, {0x253c, 138}, {0x2550, 160}, {0x2551, 161},
	{0x2552, 162}, {0x2554, 165}, {0x2557, 168}, {0x2558, 169},
	{0x2559, 170}, {0x255a, 171}, {0x255b, 172}, {0x255d, 174},
	{0x255e, 175}, {0x255f, 176}, {0x2560, 177}, {0x2561, 178},
	{0x2563, 181}, {0x2566, 184}, {0x2567, 185}, {0x2568, 186},
	{0x2569, 187}, {0x256a, 188}, {0x256c, 190}, {0x2580, 139},
	{0x2584, 140}, {0x2588, 141}, {0x258c, 142}, {0x2590, 143},
	{0x2591, 144}, {0x2592, 145}, {0x2593, 146}, {0x25a0, 148},
	{0x25ac,  22}, {0x25b2,  30}, {0x25ba,  16}, {0x25bc,  31},
	{0x25c4,  17}, {0x25cb,   9}, {0x25d8,   8}, {0x25d9,  10},
	{0x263a,   1}, {0x263b,   2}, {0x263c,  15}, {0x2640,  12},
	{0x2642,  11}, {0x2660,   6}, {0x2663,   5}, {0x2665,   3},
	{0x2666,   4}, {0x266a,  13}, {0x266b,  14}, {0xfffd, '?'}
};

const static uint32_t koi8_u_unicode_table[128] = {
	0x2500, 0x2502, 0x250c, 0x2510, 0x2514, 0x2518, 0x251c, 0x2524,
	0x252c, 0x2534, 0x253c, 0x2580, 0x2584, 0x2588, 0x258c, 0x2590,
	0x2591, 0x2592, 0x2593, 0x2320, 0x25a0, 0x2219, 0x221a, 0x2248,
	0x2264, 0x2265, 0x00a0, 0x2321, 0x00b0, 0x00b2, 0x00b7, 0x00f7,
	0x2550, 0x2551, 0x2552, 0x0451, 0x0454, 0x2554, 0x0456, 0x0457,
	0x2557, 0x2558, 0x2559, 0x255a, 0x255b, 0x0491, 0x255d, 0x255e,
	0x255f, 0x2560, 0x2561, 0x0401, 0x0404, 0x2563, 0x0406, 0x0407,
	0x2566, 0x2567, 0x2568, 0x2569, 0x256a, 0x0490, 0x256c, 0x00a9,
	0x044e, 0x0430, 0x0431, 0x0446, 0x0434, 0x0435, 0x0444, 0x0433,
	0x0445, 0x0438, 0x0439, 0x043a, 0x043b, 0x043c, 0x043d, 0x043e,
	0x043f, 0x044f, 0x0440, 0x0441, 0x0442, 0x0443, 0x0436, 0x0432,
	0x044c, 0x044b, 0x0437, 0x0448, 0x044d, 0x0449, 0x0447, 0x044a,
	0x042e, 0x0410, 0x0411, 0x0426, 0x0414, 0x0415, 0x0424, 0x0413,
	0x0425, 0x0418, 0x0419, 0x041a, 0x041b, 0x041c, 0x041d, 0x041e,
	0x041f, 0x042f, 0x0420, 0x0421, 0x0422, 0x0423, 0x0416, 0x0412,
	0x042c, 0x042b, 0x0417, 0x0428, 0x042d, 0x0429, 0x0427, 0x042a
};

const static struct ciolib_cpmap iso8859_15_table[126] = {
	{0x00a0, 160}, {0x00a1, 161}, {0x00a2, 162}, {0x00a3, 163},
	{0x00a5, 165}, {0x00a7, 167}, {0x00a9, 169}, {0x00aa, 170},
	{0x00ab, 171}, {0x00ac, 172}, {0x00ad, 173}, {0x00ae, 174},
	{0x00af, 175}, {0x00b0, 176}, {0x00b1, 177}, {0x00b2, 178},
	{0x00b3, 179}, {0x00b5, 181}, {0x00b6, 20}, {0x00b7, 183},
	{0x00b9, 185}, {0x00ba, 186}, {0x00bb, 187}, {0x00bf, 191},
	{0x00c0, 192}, {0x00c1, 193}, {0x00c2, 194}, {0x00c3, 195},
	{0x00c4, 196}, {0x00c5, 197}, {0x00c6, 198}, {0x00c7, 199},
	{0x00c8, 200}, {0x00c9, 201}, {0x00ca, 202}, {0x00cb, 203},
	{0x00cc, 204}, {0x00cd, 205}, {0x00ce, 206}, {0x00cf, 207},
	{0x00d0, 208}, {0x00d1, 209}, {0x00d2, 210}, {0x00d3, 211},
	{0x00d4, 212}, {0x00d5, 213}, {0x00d6, 214}, {0x00d7, 215},
	{0x00d8, 216}, {0x00d9, 217}, {0x00da, 218}, {0x00db, 219},
	{0x00dc, 220}, {0x00dd, 221}, {0x00de, 222}, {0x00df, 223},
	{0x00e0, 224}, {0x00e1, 225}, {0x00e2, 226}, {0x00e3, 227},
	{0x00e4, 228}, {0x00e5, 229}, {0x00e6, 230}, {0x00e7, 231},
	{0x00e8, 232}, {0x00e9, 233}, {0x00ea, 234}, {0x00eb, 235},
	{0x00ec, 236}, {0x00ed, 237}, {0x00ee, 238}, {0x00ef, 239},
	{0x00f0, 240}, {0x00f1, 241}, {0x00f2, 242}, {0x00f3, 243},
	{0x00f4, 244}, {0x00f5, 245}, {0x00f6, 246}, {0x00f7, 247},
	{0x00f8, 248}, {0x00f9, 249}, {0x00fa, 250}, {0x00fb, 251},
	{0x00fc, 252}, {0x00fd, 253}, {0x00fe, 254}, {0x00ff, 255},
	{0x0152, 188}, {0x0153, 189}, {0x0160, 166}, {0x0161, 168},
	{0x0178, 190}, {0x017d, 180}, {0x017e, 184}, {0x2022, 7},
	{0x203c, 19}, {0x20ac, 164}, {0x2190, 27}, {0x2191, 24},
	{0x2192, 26}, {0x2193, 25}, {0x2194, 29}, {0x2195, 18},
	{0x21a8, 23}, {0x221f, 28}, {0x25ac, 22}, {0x25b2, 30},
	{0x25ba, 16}, {0x25bc, 31}, {0x25c4, 17}, {0x25cb, 9},
	{0x25d8, 8}, {0x25d9, 10}, {0x263a, 1}, {0x263b, 2},
	{0x263c, 15}, {0x2640, 12}, {0x2642, 11}, {0x2660, 6},
	{0x2663, 5}, {0x2665, 3}, {0x2666, 4}, {0x266a, 13},
	{0x266b, 14}, {0xfffd, 63}
};

const static uint32_t iso8859_15_unicode_table[128] = {
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0x00a0, 0x00a1, 0x00a2, 0x00a3, 0x20ac, 0x00a5, 0x0160, 0x00a7,
	0x0161, 0x00a9, 0x00aa, 0x00ab, 0x00ac, 0x00ad, 0x00ae, 0x00af,
	0x00b0, 0x00b1, 0x00b2, 0x00b3, 0x017d, 0x00b5, 0x00b6, 0x00b7,
	0x017e, 0x00b9, 0x00ba, 0x00bb, 0x0152, 0x0153, 0x0178, 0x00bf,
	0x00c0, 0x00c1, 0x00c2, 0x00c3, 0x00c4, 0x00c5, 0x00c6, 0x00c7,
	0x00c8, 0x00c9, 0x00ca, 0x00cb, 0x00cc, 0x00cd, 0x00ce, 0x00cf,
	0x00d0, 0x00d1, 0x00d2, 0x00d3, 0x00d4, 0x00d5, 0x00d6, 0x00d7,
	0x00d8, 0x00d9, 0x00da, 0x00db, 0x00dc, 0x00dd, 0x00de, 0x00df,
	0x00e0, 0x00e1, 0x00e2, 0x00e3, 0x00e4, 0x00e5, 0x00e6, 0x00e7,
	0x00e8, 0x00e9, 0x00ea, 0x00eb, 0x00ec, 0x00ed, 0x00ee, 0x00ef,
	0x00f0, 0x00f1, 0x00f2, 0x00f3, 0x00f4, 0x00f5, 0x00f6, 0x00f7,
	0x00f8, 0x00f9, 0x00fa, 0x00fb, 0x00fc, 0x00fd, 0x00fe, 0x00ff
};

const static struct ciolib_cpmap iso8859_5_table[127] = {
	{0x00a0, 160}, {0x00a7, 21}, {0x00ad, 173}, {0x00b6, 20},
	{0x0401, 161}, {0x0402, 162}, {0x0403, 163}, {0x0404, 164},
	{0x0405, 165}, {0x0406, 166}, {0x0407, 167}, {0x0408, 168},
	{0x0409, 169}, {0x040a, 170}, {0x040b, 171}, {0x040c, 172},
	{0x040e, 174}, {0x040f, 175}, {0x0410, 176}, {0x0411, 177},
	{0x0412, 178}, {0x0413, 179}, {0x0414, 180}, {0x0415, 181},
	{0x0416, 182}, {0x0417, 183}, {0x0418, 184}, {0x0419, 185},
	{0x041a, 186}, {0x041b, 187}, {0x041c, 188}, {0x041d, 189},
	{0x041e, 190}, {0x041f, 191}, {0x0420, 192}, {0x0421, 193},
	{0x0422, 194}, {0x0423, 195}, {0x0424, 196}, {0x0425, 197},
	{0x0426, 198}, {0x0427, 199}, {0x0428, 200}, {0x0429, 201},
	{0x042a, 202}, {0x042b, 203}, {0x042c, 204}, {0x042d, 205},
	{0x042e, 206}, {0x042f, 207}, {0x0430, 208}, {0x0431, 209},
	{0x0432, 210}, {0x0433, 211}, {0x0434, 212}, {0x0435, 213},
	{0x0436, 214}, {0x0437, 215}, {0x0438, 216}, {0x0439, 217},
	{0x043a, 218}, {0x043b, 219}, {0x043c, 220}, {0x043d, 221},
	{0x043e, 222}, {0x043f, 223}, {0x0440, 224}, {0x0441, 225},
	{0x0442, 226}, {0x0443, 227}, {0x0444, 228}, {0x0445, 229},
	{0x0446, 230}, {0x0447, 231}, {0x0448, 232}, {0x0449, 233},
	{0x044a, 234}, {0x044b, 235}, {0x044c, 236}, {0x044d, 237},
	{0x044e, 238}, {0x044f, 239}, {0x0451, 241}, {0x0452, 242},
	{0x0453, 243}, {0x0454, 244}, {0x0455, 245}, {0x0456, 246},
	{0x0457, 247}, {0x0458, 248}, {0x0459, 249}, {0x045a, 250},
	{0x045b, 251}, {0x045c, 252}, {0x045e, 254}, {0x045f, 255},
	{0x2022, 7}, {0x203c, 19}, {0x2116, 240}, {0x2190, 27},
	{0x2191, 24}, {0x2192, 26}, {0x2193, 25}, {0x2194, 29},
	{0x2195, 18}, {0x21a8, 23}, {0x221f, 28}, {0x25ac, 22},
	{0x25b2, 30}, {0x25ba, 16}, {0x25bc, 31}, {0x25c4, 17},
	{0x25cb, 9}, {0x25d8, 8}, {0x25d9, 10}, {0x263a, 1},
	{0x263b, 2}, {0x263c, 15}, {0x2640, 12}, {0x2642, 11},
	{0x2660, 6}, {0x2663, 5}, {0x2665, 3}, {0x2666, 4},
	{0x266a, 13}, {0x266b, 14}, {0xfffd, 63}
};

const static uint32_t iso8859_5_unicode_table[128] = {
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0x00a0, 0x0401, 0x0402, 0x0403, 0x0404, 0x0405, 0x0406, 0x0407,
	0x0408, 0x0409, 0x040a, 0x040b, 0x040c, 0x00ad, 0x040e, 0x040f,
	0x0410, 0x0411, 0x0412, 0x0413, 0x0414, 0x0415, 0x0416, 0x0417,
	0x0418, 0x0419, 0x041a, 0x041b, 0x041c, 0x041d, 0x041e, 0x041f,
	0x0420, 0x0421, 0x0422, 0x0423, 0x0424, 0x0425, 0x0426, 0x0427,
	0x0428, 0x0429, 0x042a, 0x042b, 0x042c, 0x042d, 0x042e, 0x042f,
	0x0430, 0x0431, 0x0432, 0x0433, 0x0434, 0x0435, 0x0436, 0x0437,
	0x0438, 0x0439, 0x043a, 0x043b, 0x043c, 0x043d, 0x043e, 0x043f,
	0x0440, 0x0441, 0x0442, 0x0443, 0x0444, 0x0445, 0x0446, 0x0447,
	0x0448, 0x0449, 0x044a, 0x044b, 0x044c, 0x044d, 0x044e, 0x044f,
	0x2116, 0x0451, 0x0452, 0x0453, 0x0454, 0x0455, 0x0456, 0x0457,
	0x0458, 0x0459, 0x045a, 0x045b, 0x045c, 0x00a7, 0x045e, 0x045f
};

const static struct ciolib_cpmap cp850_table[158] = {
	{0x00a0, 255}, {0x00a1, 173}, {0x00a2, 189}, {0x00a3, 156},
	{0x00a4, 207}, {0x00a5, 190}, {0x00a6, 221}, {0x00a7, 21},
	{0x00a8, 249}, {0x00a9, 184}, {0x00aa, 166}, {0x00ab, 174},
	{0x00ac, 170}, {0x00ad, 240}, {0x00ae, 169}, {0x00af, 238},
	{0x00b0, 248}, {0x00b1, 241}, {0x00b2, 253}, {0x00b3, 252},
	{0x00b4, 239}, {0x00b5, 230}, {0x00b6, 20}, {0x00b7, 250},
	{0x00b8, 247}, {0x00b9, 251}, {0x00ba, 167}, {0x00bb, 175},
	{0x00bc, 172}, {0x00bd, 171}, {0x00be, 243}, {0x00bf, 168},
	{0x00c0, 183}, {0x00c1, 181}, {0x00c2, 182}, {0x00c3, 199},
	{0x00c4, 142}, {0x00c5, 143}, {0x00c6, 146}, {0x00c7, 128},
	{0x00c8, 212}, {0x00c9, 144}, {0x00ca, 210}, {0x00cb, 211},
	{0x00cc, 222}, {0x00cd, 214}, {0x00ce, 215}, {0x00cf, 216},
	{0x00d0, 209}, {0x00d1, 165}, {0x00d2, 227}, {0x00d3, 224},
	{0x00d4, 226}, {0x00d5, 229}, {0x00d6, 153}, {0x00d7, 158},
	{0x00d8, 157}, {0x00d9, 235}, {0x00da, 233}, {0x00db, 234},
	{0x00dc, 154}, {0x00dd, 237}, {0x00de, 232}, {0x00df, 225},
	{0x00e0, 133}, {0x00e1, 160}, {0x00e2, 131}, {0x00e3, 198},
	{0x00e4, 132}, {0x00e5, 134}, {0x00e6, 145}, {0x00e7, 135},
	{0x00e8, 138}, {0x00e9, 130}, {0x00ea, 136}, {0x00eb, 137},
	{0x00ec, 141}, {0x00ed, 161}, {0x00ee, 140}, {0x00ef, 139},
	{0x00f0, 208}, {0x00f1, 164}, {0x00f2, 149}, {0x00f3, 162},
	{0x00f4, 147}, {0x00f5, 228}, {0x00f6, 148}, {0x00f7, 246},
	{0x00f8, 155}, {0x00f9, 151}, {0x00fa, 163}, {0x00fb, 150},
	{0x00fc, 129}, {0x00fd, 236}, {0x00fe, 231}, {0x00ff, 152},
	{0x0131, 213}, {0x0192, 159}, {0x2017, 242}, {0x2022, 7},
	{0x203c, 19}, {0x2190, 27}, {0x2191, 24}, {0x2192, 26},
	{0x2193, 25}, {0x2194, 29}, {0x2195, 18}, {0x21a8, 23},
	{0x221f, 28}, {0x2500, 196}, {0x2502, 179}, {0x250c, 218},
	{0x2510, 191}, {0x2514, 192}, {0x2518, 217}, {0x251c, 195},
	{0x2524, 180}, {0x252c, 194}, {0x2534, 193}, {0x253c, 197},
	{0x2550, 205}, {0x2551, 186}, {0x2554, 201}, {0x2557, 187},
	{0x255a, 200}, {0x255d, 188}, {0x2560, 204}, {0x2563, 185},
	{0x2566, 203}, {0x2569, 202}, {0x256c, 206}, {0x2580, 223},
	{0x2584, 220}, {0x2588, 219}, {0x2591, 176}, {0x2592, 177},
	{0x2593, 178}, {0x25a0, 254}, {0x25ac, 22}, {0x25b2, 30},
	{0x25ba, 16}, {0x25bc, 31}, {0x25c4, 17}, {0x25cb, 9},
	{0x25d8, 8}, {0x25d9, 10}, {0x263a, 1}, {0x263b, 2},
	{0x263c, 15}, {0x2640, 12}, {0x2642, 11}, {0x2660, 6},
	{0x2663, 5}, {0x2665, 3}, {0x2666, 4}, {0x266a, 13},
	{0x266b, 14}, {0xfffd, 63}
};

const static uint32_t cp850_unicode_table[128] = {
	0x00c7, 0x00fc, 0x00e9, 0x00e2, 0x00e4, 0x00e0, 0x00e5, 0x00e7,
	0x00ea, 0x00eb, 0x00e8, 0x00ef, 0x00ee, 0x00ec, 0x00c4, 0x00c5,
	0x00c9, 0x00e6, 0x00c6, 0x00f4, 0x00f6, 0x00f2, 0x00fb, 0x00f9,
	0x00ff, 0x00d6, 0x00dc, 0x00f8, 0x00a3, 0x00d8, 0x00d7, 0x0192,
	0x00e1, 0x00ed, 0x00f3, 0x00fa, 0x00f1, 0x00d1, 0x00aa, 0x00ba,
	0x00bf, 0x00ae, 0x00ac, 0x00bd, 0x00bc, 0x00a1, 0x00ab, 0x00bb,
	0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x00c1, 0x00c2, 0x00c0,
	0x00a9, 0x2563, 0x2551, 0x2557, 0x255d, 0x00a2, 0x00a5, 0x2510,
	0x2514, 0x2534, 0x252c, 0x251c, 0x2500, 0x253c, 0x00e3, 0x00c3,
	0x255a, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256c, 0x00a4,
	0x00f0, 0x00d0, 0x00ca, 0x00cb, 0x00c8, 0x0131, 0x00cd, 0x00ce,
	0x00cf, 0x2518, 0x250c, 0x2588, 0x2584, 0x00a6, 0x00cc, 0x2580,
	0x00d3, 0x00df, 0x00d4, 0x00d2, 0x00f5, 0x00d5, 0x00b5, 0x00fe,
	0x00de, 0x00da, 0x00db, 0x00d9, 0x00fd, 0x00dd, 0x00af, 0x00b4,
	0x00ad, 0x00b1, 0x2017, 0x00be, 0x00b6, 0x00a7, 0x00f7, 0x00b8,
	0x00b0, 0x00a8, 0x00b7, 0x00b9, 0x00b3, 0x00b2, 0x25a0, 0x00a0
};

const static struct ciolib_cpmap cp865_table[160] = {
	{0x00a0, 255}, {0x00a1, 173}, {0x00a3, 156}, {0x00a4, 175},
	{0x00a7,  21}, {0x00aa, 166}, {0x00ab, 174}, {0x00ac, 170},
	{0x00b0, 248}, {0x00b1, 241}, {0x00b2, 253}, {0x00b5, 230},
	{0x00b6,  20}, {0x00b7, 250}, {0x00ba, 167}, {0x00bc, 172},
	{0x00bd, 171}, {0x00bf, 168}, {0x00c4, 142}, {0x00c5, 143},
	{0x00c6, 146}, {0x00c7, 128}, {0x00c9, 144}, {0x00d1, 165},
	{0x00d6, 153}, {0x00d8, 157}, {0x00dc, 154}, {0x00df, 225},
	{0x00e0, 133}, {0x00e1, 160}, {0x00e2, 131}, {0x00e4, 132},
	{0x00e5, 134}, {0x00e6, 145}, {0x00e7, 135}, {0x00e8, 138},
	{0x00e9, 130}, {0x00ea, 136}, {0x00eb, 137}, {0x00ec, 141},
	{0x00ed, 161}, {0x00ee, 140}, {0x00ef, 139}, {0x00f1, 164},
	{0x00f2, 149}, {0x00f3, 162}, {0x00f4, 147}, {0x00f6, 148},
	{0x00f7, 246}, {0x00f8, 155}, {0x00f9, 151}, {0x00fa, 163},
	{0x00fb, 150}, {0x00fc, 129}, {0x00ff, 152}, {0x0192, 159},
	{0x0393, 226}, {0x0398, 233}, {0x03a3, 228}, {0x03a6, 232},
	{0x03a9, 234}, {0x03b1, 224}, {0x03b4, 235}, {0x03b5, 238},
	{0x03c0, 227}, {0x03c3, 229}, {0x03c4, 231}, {0x03c6, 237},
	{0x2022,   7}, {0x203c,  19}, {0x207f, 252}, {0x20a7, 158},
	{0x2190,  27}, {0x2191,  24}, {0x2192,  26}, {0x2193,  25},
	{0x2194,  29}, {0x2195,  18}, {0x21a8,  23}, {0x2219, 249},
	{0x221a, 251}, {0x221e, 236}, {0x221f,  28}, {0x2229, 239},
	{0x2248, 247}, {0x2261, 240}, {0x2264, 243}, {0x2265, 242},
	{0x2310, 169}, {0x2320, 244}, {0x2321, 245}, {0x2500, 196},
	{0x2502, 179}, {0x250c, 218}, {0x2510, 191}, {0x2514, 192},
	{0x2518, 217}, {0x251c, 195}, {0x2524, 180}, {0x252c, 194},
	{0x2534, 193}, {0x253c, 197}, {0x2550, 205}, {0x2551, 186},
	{0x2552, 213}, {0x2553, 214}, {0x2554, 201}, {0x2555, 184},
	{0x2556, 183}, {0x2557, 187}, {0x2558, 212}, {0x2559, 211},
	{0x255a, 200}, {0x255b, 190}, {0x255c, 189}, {0x255d, 188},
	{0x255e, 198}, {0x255f, 199}, {0x2560, 204}, {0x2561, 181},
	{0x2562, 182}, {0x2563, 185}, {0x2564, 209}, {0x2565, 210},
	{0x2566, 203}, {0x2567, 207}, {0x2568, 208}, {0x2569, 202},
	{0x256a, 216}, {0x256b, 215}, {0x256c, 206}, {0x2580, 223},
	{0x2584, 220}, {0x2588, 219}, {0x258c, 221}, {0x2590, 222},
	{0x2591, 176}, {0x2592, 177}, {0x2593, 178}, {0x25a0, 254},
	{0x25ac,  22}, {0x25b2,  30}, {0x25ba,  16}, {0x25bc,  31},
	{0x25c4,  17}, {0x25cb,   9}, {0x25d8,   8}, {0x25d9,  10},
	{0x263a,   1}, {0x263b,   2}, {0x263c,  15}, {0x2640,  12},
	{0x2642,  11}, {0x2660,   6}, {0x2663,   5}, {0x2665,   3},
	{0x2666,   4}, {0x266a,  13}, {0x266b,  14}, {0xfffd, '?'}
};

const static uint32_t cp865_unicode_table[128] = {
	0x00c7, 0x00fc, 0x00e9, 0x00e2, 0x00e4, 0x00e0, 0x00e5, 0x00e7,
	0x00ea, 0x00eb, 0x00e8, 0x00ef, 0x00ee, 0x00ec, 0x00c4, 0x00c5,
	0x00c9, 0x00e6, 0x00c6, 0x00f4, 0x00f6, 0x00f2, 0x00fb, 0x00f9,
	0x00ff, 0x00d6, 0x00dc, 0x00f8, 0x00a3, 0x00d8, 0x20a7, 0x0192,
	0x00e1, 0x00ed, 0x00f3, 0x00fa, 0x00f1, 0x00d1, 0x00aa, 0x00ba,
	0x00bf, 0x2310, 0x00ac, 0x00bd, 0x00bc, 0x00a1, 0x00ab, 0x00a4,
	0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x2561, 0x2562, 0x2556,
	0x2555, 0x2563, 0x2551, 0x2557, 0x255d, 0x255c, 0x255b, 0x2510,
	0x2514, 0x2534, 0x252c, 0x251c, 0x2500, 0x253c, 0x255e, 0x255f,
	0x255a, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256c, 0x2567,
	0x2568, 0x2564, 0x2565, 0x2559, 0x2558, 0x2552, 0x2553, 0x256b,
	0x256a, 0x2518, 0x250c, 0x2588, 0x2584, 0x258c, 0x2590, 0x2580,
	0x03b1, 0x00df, 0x0393, 0x03c0, 0x03a3, 0x03c3, 0x00b5, 0x03c4,
	0x03a6, 0x0398, 0x03a9, 0x03b4, 0x221e, 0x03c6, 0x03b5, 0x2229,
	0x2261, 0x00b1, 0x2265, 0x2264, 0x2320, 0x2321, 0x00f7, 0x2248,
	0x00b0, 0x2219, 0x00b7, 0x221a, 0x207f, 0x00b2, 0x25a0, 0x00a0

};

const static struct ciolib_cpmap iso8859_7_table[124] = {
	{0x00a0, 160}, {0x00a3, 163}, {0x00a6, 166}, {0x00a7, 21},
	{0x00a8, 168}, {0x00a9, 169}, {0x00ab, 171}, {0x00ac, 172},
	{0x00ad, 173}, {0x00b0, 176}, {0x00b1, 177}, {0x00b2, 178},
	{0x00b3, 179}, {0x00b6, 20}, {0x00b7, 183}, {0x00bb, 187},
	{0x00bd, 189}, {0x037a, 170}, {0x0384, 180}, {0x0385, 181},
	{0x0386, 182}, {0x0388, 184}, {0x0389, 185}, {0x038a, 186},
	{0x038c, 188}, {0x038e, 190}, {0x038f, 191}, {0x0390, 192},
	{0x0391, 193}, {0x0392, 194}, {0x0393, 195}, {0x0394, 196},
	{0x0395, 197}, {0x0396, 198}, {0x0397, 199}, {0x0398, 200},
	{0x0399, 201}, {0x039a, 202}, {0x039b, 203}, {0x039c, 204},
	{0x039d, 205}, {0x039e, 206}, {0x039f, 207}, {0x03a0, 208},
	{0x03a1, 209}, {0x03a3, 211}, {0x03a4, 212}, {0x03a5, 213},
	{0x03a6, 214}, {0x03a7, 215}, {0x03a8, 216}, {0x03a9, 217},
	{0x03aa, 218}, {0x03ab, 219}, {0x03ac, 220}, {0x03ad, 221},
	{0x03ae, 222}, {0x03af, 223}, {0x03b0, 224}, {0x03b1, 225},
	{0x03b2, 226}, {0x03b3, 227}, {0x03b4, 228}, {0x03b5, 229},
	{0x03b6, 230}, {0x03b7, 231}, {0x03b8, 232}, {0x03b9, 233},
	{0x03ba, 234}, {0x03bb, 235}, {0x03bc, 236}, {0x03bd, 237},
	{0x03be, 238}, {0x03bf, 239}, {0x03c0, 240}, {0x03c1, 241},
	{0x03c2, 242}, {0x03c3, 243}, {0x03c4, 244}, {0x03c5, 245},
	{0x03c6, 246}, {0x03c7, 247}, {0x03c8, 248}, {0x03c9, 249},
	{0x03ca, 250}, {0x03cb, 251}, {0x03cc, 252}, {0x03cd, 253},
	{0x03ce, 254}, {0x2015, 175}, {0x2018, 161}, {0x2019, 162},
	{0x2022, 7}, {0x203c, 19}, {0x20ac, 164}, {0x20af, 165},
	{0x2190, 27}, {0x2191, 24}, {0x2192, 26}, {0x2193, 25},
	{0x2194, 29}, {0x2195, 18}, {0x21a8, 23}, {0x221f, 28},
	{0x25ac, 22}, {0x25b2, 30}, {0x25ba, 16}, {0x25bc, 31},
	{0x25c4, 17}, {0x25cb, 9}, {0x25d8, 8}, {0x25d9, 10},
	{0x263a, 1}, {0x263b, 2}, {0x263c, 15}, {0x2640, 12},
	{0x2642, 11}, {0x2660, 6}, {0x2663, 5}, {0x2665, 3},
	{0x2666, 4}, {0x266a, 13}, {0x266b, 14}, {0xfffd, 63}
};

const static uint32_t iso8859_7_unicode_table[128] = {
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0x00a0, 0x2018, 0x2019, 0x00a3, 0x20ac, 0x20af, 0x00a6, 0x00a7,
	0x00a8, 0x00a9, 0x037a, 0x00ab, 0x00ac, 0x00ad, 0xfffd, 0x2015,
	0x00b0, 0x00b1, 0x00b2, 0x00b3, 0x0384, 0x0385, 0x0386, 0x00b7,
	0x0388, 0x0389, 0x038a, 0x00bb, 0x038c, 0x00bd, 0x038e, 0x038f,
	0x0390, 0x0391, 0x0392, 0x0393, 0x0394, 0x0395, 0x0396, 0x0397,
	0x0398, 0x0399, 0x039a, 0x039b, 0x039c, 0x039d, 0x039e, 0x039f,
	0x03a0, 0x03a1, 0xfffd, 0x03a3, 0x03a4, 0x03a5, 0x03a6, 0x03a7,
	0x03a8, 0x03a9, 0x03aa, 0x03ab, 0x03ac, 0x03ad, 0x03ae, 0x03af,
	0x03b0, 0x03b1, 0x03b2, 0x03b3, 0x03b4, 0x03b5, 0x03b6, 0x03b7,
	0x03b8, 0x03b9, 0x03ba, 0x03bb, 0x03bc, 0x03bd, 0x03be, 0x03bf,
	0x03c0, 0x03c1, 0x03c2, 0x03c3, 0x03c4, 0x03c5, 0x03c6, 0x03c7,
	0x03c8, 0x03c9, 0x03ca, 0x03cb, 0x03cc, 0x03cd, 0x03ce, 0xfffd
};

const static struct ciolib_cpmap iso8859_1_table[126] = {
	{0x00a0, 160}, {0x00a1, 161}, {0x00a2, 162}, {0x00a3, 163},
	{0x00a4, 164}, {0x00a5, 165}, {0x00a6, 166}, {0x00a7, 167},
	{0x00a8, 168}, {0x00a9, 169}, {0x00aa, 170}, {0x00ab, 171},
	{0x00ac, 172}, {0x00ad, 173}, {0x00ae, 174}, {0x00af, 175},
	{0x00b0, 176}, {0x00b1, 177}, {0x00b2, 178}, {0x00b3, 179},
	{0x00b4, 180}, {0x00b5, 181}, {0x00b6, 182}, {0x00b7, 183},
	{0x00b8, 184}, {0x00b9, 185}, {0x00ba, 186}, {0x00bb, 187},
	{0x00bc, 188}, {0x00bd, 189}, {0x00be, 190}, {0x00bf, 191},
	{0x00c0, 192}, {0x00c1, 193}, {0x00c2, 194}, {0x00c3, 195},
	{0x00c4, 196}, {0x00c5, 197}, {0x00c6, 198}, {0x00c7, 199},
	{0x00c8, 200}, {0x00c9, 201}, {0x00ca, 202}, {0x00cb, 203},
	{0x00cc, 204}, {0x00cd, 205}, {0x00ce, 206}, {0x00cf, 207},
	{0x00d0, 208}, {0x00d1, 209}, {0x00d2, 210}, {0x00d3, 211},
	{0x00d4, 212}, {0x00d5, 213}, {0x00d6, 214}, {0x00d7, 215},
	{0x00d8, 216}, {0x00d9, 217}, {0x00da, 218}, {0x00db, 219},
	{0x00dc, 220}, {0x00dd, 221}, {0x00de, 222}, {0x00df, 223},
	{0x00e0, 224}, {0x00e1, 225}, {0x00e2, 226}, {0x00e3, 227},
	{0x00e4, 228}, {0x00e5, 229}, {0x00e6, 230}, {0x00e7, 231},
	{0x00e8, 232}, {0x00e9, 233}, {0x00ea, 234}, {0x00eb, 235},
	{0x00ec, 236}, {0x00ed, 237}, {0x00ee, 238}, {0x00ef, 239},
	{0x00f0, 240}, {0x00f1, 241}, {0x00f2, 242}, {0x00f3, 243},
	{0x00f4, 244}, {0x00f5, 245}, {0x00f6, 246}, {0x00f7, 247},
	{0x00f8, 248}, {0x00f9, 249}, {0x00fa, 250}, {0x00fb, 251},
	{0x00fc, 252}, {0x00fd, 253}, {0x00fe, 254}, {0x00ff, 255},
	{0x2022, 7}, {0x203c, 19}, {0x2190, 27}, {0x2191, 24},
	{0x2192, 26}, {0x2193, 25}, {0x2194, 29}, {0x2195, 18},
	{0x21a8, 23}, {0x221f, 28}, {0x25ac, 22}, {0x25b2, 30},
	{0x25ba, 16}, {0x25bc, 31}, {0x25c4, 17}, {0x25cb, 9},
	{0x25d8, 8}, {0x25d9, 10}, {0x263a, 1}, {0x263b, 2},
	{0x263c, 15}, {0x2640, 12}, {0x2642, 11}, {0x2660, 6},
	{0x2663, 5}, {0x2665, 3}, {0x2666, 4}, {0x266a, 13},
	{0x266b, 14}, {0xfffd, 63}
};

const static uint32_t iso8859_1_unicode_table[128] = {
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0x00a0, 0x00a1, 0x00a2, 0x00a3, 0x00a4, 0x00a5, 0x00a6, 0x00a7,
	0x00a8, 0x00a9, 0x00aa, 0x00ab, 0x00ac, 0x00ad, 0x00ae, 0x00af,
	0x00b0, 0x00b1, 0x00b2, 0x00b3, 0x00b4, 0x00b5, 0x00b6, 0x00b7,
	0x00b8, 0x00b9, 0x00ba, 0x00bb, 0x00bc, 0x00bd, 0x00be, 0x00bf,
	0x00c0, 0x00c1, 0x00c2, 0x00c3, 0x00c4, 0x00c5, 0x00c6, 0x00c7,
	0x00c8, 0x00c9, 0x00ca, 0x00cb, 0x00cc, 0x00cd, 0x00ce, 0x00cf,
	0x00d0, 0x00d1, 0x00d2, 0x00d3, 0x00d4, 0x00d5, 0x00d6, 0x00d7,
	0x00d8, 0x00d9, 0x00da, 0x00db, 0x00dc, 0x00dd, 0x00de, 0x00df,
	0x00e0, 0x00e1, 0x00e2, 0x00e3, 0x00e4, 0x00e5, 0x00e6, 0x00e7,
	0x00e8, 0x00e9, 0x00ea, 0x00eb, 0x00ec, 0x00ed, 0x00ee, 0x00ef,
	0x00f0, 0x00f1, 0x00f2, 0x00f3, 0x00f4, 0x00f5, 0x00f6, 0x00f7,
	0x00f8, 0x00f9, 0x00fa, 0x00fb, 0x00fc, 0x00fd, 0x00fe, 0x00ff
};

const static struct ciolib_cpmap cp866m2_table[160] = {
	{0x00a0, 255}, {0x00a7,  21}, {0x00A9, 252}, {0x00b0, 248},
	{0x00B2, 253}, {0x00b6,  20}, {0x00b7, 250}, {0x00F7, 246},
	{0x0401, 240}, {0x0410, 128}, {0x0411, 129}, {0x0412, 130},
	{0x0413, 131}, {0x0414, 132}, {0x0415, 133}, {0x0416, 134},
	{0x0417, 135}, {0x0418, 136}, {0x0419, 137}, {0x041a, 138},
	{0x041b, 139}, {0x041c, 140}, {0x041d, 141}, {0x041e, 142},
	{0x041f, 143}, {0x0420, 144}, {0x0421, 145}, {0x0422, 146},
	{0x0423, 147}, {0x0424, 148}, {0x0425, 149}, {0x0426, 150},
	{0x0427, 151}, {0x0428, 152}, {0x0429, 153}, {0x042a, 154},
	{0x042b, 155}, {0x042c, 156}, {0x042d, 157}, {0x042e, 158},
	{0x042f, 159}, {0x0430, 160}, {0x0431, 161}, {0x0432, 162},
	{0x0433, 163}, {0x0434, 164}, {0x0435, 165}, {0x0436, 166},
	{0x0437, 167}, {0x0438, 168}, {0x0439, 169}, {0x043a, 170},
	{0x043b, 171}, {0x043c, 172}, {0x043d, 173}, {0x043e, 174},
	{0x043f, 175}, {0x0440, 224}, {0x0441, 225}, {0x0442, 226},
	{0x0443, 227}, {0x0444, 228}, {0x0445, 229}, {0x0446, 230},
	{0x0447, 231}, {0x0448, 232}, {0x0449, 233}, {0x044a, 234},
	{0x044b, 235}, {0x044c, 236}, {0x044d, 237}, {0x044e, 238},
	{0x044f, 239}, {0x0451, 241}, {0x2022,   7}, {0x203c,  19},
	{0x2190,  27}, {0x2191,  24}, {0x2192,  26}, {0x2193,  25},
	{0x2194,  29}, {0x2195,  18}, {0x21a8,  23}, {0x2219, 249},
	{0x221a, 251}, {0x221f,  28}, {0x2264, 243}, {0x2265, 242},
	{0x2320, 244}, {0x2321, 245}, {0x2248, 247}, {0x2500, 196},
	{0x2502, 179}, {0x250c, 218}, {0x2510, 191}, {0x2514, 192},
	{0x2518, 217}, {0x251c, 195}, {0x2524, 180}, {0x252c, 194},
	{0x2534, 193}, {0x253c, 197}, {0x2550, 205}, {0x2551, 186},
	{0x2552, 213}, {0x2553, 214}, {0x2554, 201}, {0x2555, 184},
	{0x2556, 183}, {0x2557, 187}, {0x2558, 212}, {0x2559, 211},
	{0x255a, 200}, {0x255b, 190}, {0x255c, 189}, {0x255d, 188},
	{0x255e, 198}, {0x255f, 199}, {0x2560, 204}, {0x2561, 181},
	{0x2562, 182}, {0x2563, 185}, {0x2564, 209}, {0x2565, 210},
	{0x2566, 203}, {0x2567, 207}, {0x2568, 208}, {0x2569, 202},
	{0x256a, 216}, {0x256b, 215}, {0x256c, 206}, {0x2580, 223},
	{0x2584, 220}, {0x2588, 219}, {0x258c, 221}, {0x2590, 222},
	{0x2591, 176}, {0x2592, 177}, {0x2593, 178}, {0x25a0, 254},
	{0x25ac,  22}, {0x25b2,  30}, {0x25ba,  16}, {0x25bc,  31},
	{0x25c4,  17}, {0x25cb,   9}, {0x25d8,   8}, {0x25d9,  10},
	{0x263a,   1}, {0x263b,   2}, {0x263c,  15}, {0x2640,  12},
	{0x2642,  11}, {0x2660,   6}, {0x2663,   5}, {0x2665,   3},
	{0x2666,   4}, {0x266a,  13}, {0x266b,  14}, {0xfffd, '?'}
};

const static uint32_t cp866m2_unicode_table[128] = {
	0x0410, 0x0411, 0x0412, 0x0413, 0x0414, 0x0415, 0x0416, 0x0417,
	0x0418, 0x0419, 0x041a, 0x041b, 0x041c, 0x041d, 0x041e, 0x041f,
	0x0420, 0x0421, 0x0422, 0x0423, 0x0424, 0x0425, 0x0426, 0x0427,
	0x0428, 0x0429, 0x042a, 0x042b, 0x042c, 0x042d, 0x042e, 0x042f,
	0x0430, 0x0431, 0x0432, 0x0433, 0x0434, 0x0435, 0x0436, 0x0437,
	0x0438, 0x0439, 0x043a, 0x043b, 0x043c, 0x043d, 0x043e, 0x043f,
	0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x2561, 0x2562, 0x2556,
	0x2555, 0x2563, 0x2551, 0x2557, 0x255d, 0x255c, 0x255b, 0x2510,
	0x2514, 0x2534, 0x252c, 0x251c, 0x2500, 0x253c, 0x255e, 0x255f,
	0x255a, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256c, 0x2567,
	0x2568, 0x2564, 0x2565, 0x2559, 0x2558, 0x2552, 0x2553, 0x256b,
	0x256a, 0x2518, 0x250c, 0x2588, 0x2584, 0x258c, 0x2590, 0x2580,
	0x0440, 0x0441, 0x0442, 0x0443, 0x0444, 0x0445, 0x0446, 0x0447,
	0x0448, 0x0449, 0x044a, 0x044b, 0x044c, 0x044d, 0x044e, 0x044f,
	0x0401, 0x0451, 0x2265, 0x2264, 0x2320, 0x2321, 0x00F7, 0x2248,
	0x00b0, 0x2219, 0x00b7, 0x221a, 0x00A9, 0x00B2, 0x25a0, 0x00a0
};

const static struct ciolib_cpmap cp866u_table[160] = {
	{0x00a0, 255}, {0x00a7,  21}, {0x00A9, 252}, {0x00B2, 253},
	{0x00b6,  20}, {0x00b7, 250}, {0x0401, 240}, {0x0404, 244},
	{0x0406, 246}, {0x0407, 248}, {0x0410, 128}, {0x0411, 129},
	{0x0412, 130}, {0x0413, 131}, {0x0414, 132}, {0x0415, 133},
	{0x0416, 134}, {0x0417, 135}, {0x0418, 136}, {0x0419, 137},
	{0x041a, 138}, {0x041b, 139}, {0x041c, 140}, {0x041d, 141},
	{0x041e, 142}, {0x041f, 143}, {0x0420, 144}, {0x0421, 145},
	{0x0422, 146}, {0x0423, 147}, {0x0424, 148}, {0x0425, 149},
	{0x0426, 150}, {0x0427, 151}, {0x0428, 152}, {0x0429, 153},
	{0x042a, 154}, {0x042b, 155}, {0x042c, 156}, {0x042d, 157},
	{0x042e, 158}, {0x042f, 159}, {0x0430, 160}, {0x0431, 161},
	{0x0432, 162}, {0x0433, 163}, {0x0434, 164}, {0x0435, 165},
	{0x0436, 166}, {0x0437, 167}, {0x0438, 168}, {0x0439, 169},
	{0x043a, 170}, {0x043b, 171}, {0x043c, 172}, {0x043d, 173},
	{0x043e, 174}, {0x043f, 175}, {0x0440, 224}, {0x0441, 225},
	{0x0442, 226}, {0x0443, 227}, {0x0444, 228}, {0x0445, 229},
	{0x0446, 230}, {0x0447, 231}, {0x0448, 232}, {0x0449, 233},
	{0x044a, 234}, {0x044b, 235}, {0x044c, 236}, {0x044d, 237},
	{0x044e, 238}, {0x044f, 239}, {0x0451, 241}, {0x0454, 245},
	{0x0456, 247}, {0x0457, 249}, {0x0490, 242}, {0x0491, 243},
	{0x2022,   7}, {0x203c,  19}, {0x2190,  27}, {0x2191,  24},
	{0x2192,  26}, {0x2193,  25}, {0x2194,  29}, {0x2195,  18},
	{0x21a8,  23}, {0x221a, 251}, {0x221f,  28}, {0x2500, 196},
	{0x2502, 179}, {0x250c, 218}, {0x2510, 191}, {0x2514, 192},
	{0x2518, 217}, {0x251c, 195}, {0x2524, 180}, {0x252c, 194},
	{0x2534, 193}, {0x253c, 197}, {0x2550, 205}, {0x2551, 186},
	{0x2552, 213}, {0x2553, 214}, {0x2554, 201}, {0x2555, 184},
	{0x2556, 183}, {0x2557, 187}, {0x2558, 212}, {0x2559, 211},
	{0x255a, 200}, {0x255b, 190}, {0x255c, 189}, {0x255d, 188},
	{0x255e, 198}, {0x255f, 199}, {0x2560, 204}, {0x2561, 181},
	{0x2562, 182}, {0x2563, 185}, {0x2564, 209}, {0x2565, 210},
	{0x2566, 203}, {0x2567, 207}, {0x2568, 208}, {0x2569, 202},
	{0x256a, 216}, {0x256b, 215}, {0x256c, 206}, {0x2580, 223},
	{0x2584, 220}, {0x2588, 219}, {0x258c, 221}, {0x2590, 222},
	{0x2591, 176}, {0x2592, 177}, {0x2593, 178}, {0x25a0, 254},
	{0x25ac,  22}, {0x25b2,  30}, {0x25ba,  16}, {0x25bc,  31},
	{0x25c4,  17}, {0x25cb,   9}, {0x25d8,   8}, {0x25d9,  10},
	{0x263a,   1}, {0x263b,   2}, {0x263c,  15}, {0x2640,  12},
	{0x2642,  11}, {0x2660,   6}, {0x2663,   5}, {0x2665,   3},
	{0x2666,   4}, {0x266a,  13}, {0x266b,  14}, {0xfffd, '?'}
};

const static uint32_t cp866u_unicode_table[128] = {
	0x0410, 0x0411, 0x0412, 0x0413, 0x0414, 0x0415, 0x0416, 0x0417,
	0x0418, 0x0419, 0x041a, 0x041b, 0x041c, 0x041d, 0x041e, 0x041f,
	0x0420, 0x0421, 0x0422, 0x0423, 0x0424, 0x0425, 0x0426, 0x0427,
	0x0428, 0x0429, 0x042a, 0x042b, 0x042c, 0x042d, 0x042e, 0x042f,
	0x0430, 0x0431, 0x0432, 0x0433, 0x0434, 0x0435, 0x0436, 0x0437,
	0x0438, 0x0439, 0x043a, 0x043b, 0x043c, 0x043d, 0x043e, 0x043f,
	0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x2561, 0x2562, 0x2556,
	0x2555, 0x2563, 0x2551, 0x2557, 0x255d, 0x255c, 0x255b, 0x2510,
	0x2514, 0x2534, 0x252c, 0x251c, 0x2500, 0x253c, 0x255e, 0x255f,
	0x255a, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256c, 0x2567,
	0x2568, 0x2564, 0x2565, 0x2559, 0x2558, 0x2552, 0x2553, 0x256b,
	0x256a, 0x2518, 0x250c, 0x2588, 0x2584, 0x258c, 0x2590, 0x2580,
	0x0440, 0x0441, 0x0442, 0x0443, 0x0444, 0x0445, 0x0446, 0x0447,
	0x0448, 0x0449, 0x044a, 0x044b, 0x044c, 0x044d, 0x044e, 0x044f,
	0x0401, 0x0451, 0x0490, 0x0491, 0x0404, 0x0454, 0x0406, 0x0456,
	0x0407, 0x0457, 0x00b7, 0x221a, 0x00A9, 0x00B2, 0x25a0, 0x00a0
};

const static struct ciolib_cpmap cp1131_table[160] = {
	{0x00a0, 255}, {0x00a7,  21}, {0x00b6,  20}, {0x00b7, 250},
	{0x0401, 240}, {0x0404, 242}, {0x0406, 248}, {0x0407, 244},
	{0x040e, 246}, {0x0410, 128}, {0x0411, 129}, {0x0412, 130},
	{0x0413, 131}, {0x0414, 132}, {0x0415, 133}, {0x0416, 134},
	{0x0417, 135}, {0x0418, 136}, {0x0419, 137}, {0x041a, 138},
	{0x041b, 139}, {0x041c, 140}, {0x041d, 141}, {0x041e, 142},
	{0x041f, 143}, {0x0420, 144}, {0x0421, 145}, {0x0422, 146},
	{0x0423, 147}, {0x0424, 148}, {0x0425, 149}, {0x0426, 150},
	{0x0427, 151}, {0x0428, 152}, {0x0429, 153}, {0x042a, 154},
	{0x042b, 155}, {0x042c, 156}, {0x042d, 157}, {0x042e, 158},
	{0x042f, 159}, {0x0430, 160}, {0x0431, 161}, {0x0432, 162},
	{0x0433, 163}, {0x0434, 164}, {0x0435, 165}, {0x0436, 166},
	{0x0437, 167}, {0x0438, 168}, {0x0439, 169}, {0x043a, 170},
	{0x043b, 171}, {0x043c, 172}, {0x043d, 173}, {0x043e, 174},
	{0x043f, 175}, {0x0440, 224}, {0x0441, 225}, {0x0442, 226},
	{0x0443, 227}, {0x0444, 228}, {0x0445, 229}, {0x0446, 230},
	{0x0447, 231}, {0x0448, 232}, {0x0449, 233}, {0x044a, 234},
	{0x044b, 235}, {0x044c, 236}, {0x044d, 237}, {0x044e, 238},
	{0x044f, 239}, {0x0451, 241}, {0x0454, 243}, {0x0456, 249},
	{0x0457, 245}, {0x045e, 247}, {0x0490, 252}, {0x0491, 253},
	{0x2022,   7}, {0x203c,  19}, {0x2190,  27}, {0x2191,  24},
	{0x2192,  26}, {0x2193,  25}, {0x2194,  29}, {0x2195,  18},
	{0x21a8,  23}, {0x2219, 254}, {0x221a, 251}, {0x221f,  28},
	{0x2500, 196}, {0x2502, 179}, {0x250c, 218}, {0x2510, 191},
	{0x2514, 192}, {0x2518, 217}, {0x251c, 195}, {0x2524, 180},
	{0x252c, 194}, {0x2534, 193}, {0x253c, 197}, {0x2550, 205},
	{0x2551, 186}, {0x2552, 213}, {0x2553, 214}, {0x2554, 201},
	{0x2555, 184}, {0x2556, 183}, {0x2557, 187}, {0x2558, 212},
	{0x2559, 211}, {0x255a, 200}, {0x255b, 190}, {0x255c, 189},
	{0x255d, 188}, {0x255e, 198}, {0x255f, 199}, {0x2560, 204},
	{0x2561, 181}, {0x2562, 182}, {0x2563, 185}, {0x2564, 209},
	{0x2565, 210}, {0x2566, 203}, {0x2567, 207}, {0x2568, 208},
	{0x2569, 202}, {0x256a, 216}, {0x256b, 215}, {0x256c, 206},
	{0x2580, 223}, {0x2584, 220}, {0x2588, 219}, {0x258c, 221},
	{0x2590, 222}, {0x2591, 176}, {0x2592, 177}, {0x2593, 178},
	{0x25ac,  22}, {0x25b2,  30}, {0x25ba,  16}, {0x25bc,  31},
	{0x25c4,  17}, {0x25cb,   9}, {0x25d8,   8}, {0x25d9,  10},
	{0x263a,   1}, {0x263b,   2}, {0x263c,  15}, {0x2640,  12},
	{0x2642,  11}, {0x2660,   6}, {0x2663,   5}, {0x2665,   3},
	{0x2666,   4}, {0x266a,  13}, {0x266b,  14}, {0xfffd, '?'}
};

const static uint32_t cp1131_unicode_table[128] = {
	0x0410, 0x0411, 0x0412, 0x0413, 0x0414, 0x0415, 0x0416, 0x0417,
	0x0418, 0x0419, 0x041a, 0x041b, 0x041c, 0x041d, 0x041e, 0x041f,
	0x0420, 0x0421, 0x0422, 0x0423, 0x0424, 0x0425, 0x0426, 0x0427,
	0x0428, 0x0429, 0x042a, 0x042b, 0x042c, 0x042d, 0x042e, 0x042f,
	0x0430, 0x0431, 0x0432, 0x0433, 0x0434, 0x0435, 0x0436, 0x0437,
	0x0438, 0x0439, 0x043a, 0x043b, 0x043c, 0x043d, 0x043e, 0x043f,
	0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x2561, 0x2562, 0x2556,
	0x2555, 0x2563, 0x2551, 0x2557, 0x255d, 0x255c, 0x255b, 0x2510,
	0x2514, 0x2534, 0x252c, 0x251c, 0x2500, 0x253c, 0x255e, 0x255f,
	0x255a, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256c, 0x2567,
	0x2568, 0x2564, 0x2565, 0x2559, 0x2558, 0x2552, 0x2553, 0x256b,
	0x256a, 0x2518, 0x250c, 0x2588, 0x2584, 0x258c, 0x2590, 0x2580,
	0x0440, 0x0441, 0x0442, 0x0443, 0x0444, 0x0445, 0x0446, 0x0447,
	0x0448, 0x0449, 0x044a, 0x044b, 0x044c, 0x044d, 0x044e, 0x044f,
	0x0401, 0x0451, 0x0404, 0x0454, 0x0407, 0x0457, 0x040e, 0x045e,
	0x0406, 0x0456, 0x00b7, 0x221a, 0x0490, 0x0491, 0x2219, 0x00a0
};

const static struct ciolib_cpmap armscii8_table[122] = {
	{0x00a0, 160}, {0x00a7, 162}, {0x00ab, 167}, {0x00b6, 20},
	{0x00bb, 166}, {0x02bc, 254}, {0x0531, 178}, {0x0532, 180},
	{0x0533, 182}, {0x0534, 184}, {0x0535, 186}, {0x0536, 188},
	{0x0537, 190}, {0x0538, 192}, {0x0539, 194}, {0x053a, 196},
	{0x053b, 198}, {0x053c, 200}, {0x053d, 202}, {0x053e, 204},
	{0x053f, 206}, {0x0540, 208}, {0x0541, 210}, {0x0542, 212},
	{0x0543, 214}, {0x0544, 216}, {0x0545, 218}, {0x0546, 220},
	{0x0547, 222}, {0x0548, 224}, {0x0549, 226}, {0x054a, 228},
	{0x054b, 230}, {0x054c, 232}, {0x054d, 234}, {0x054e, 236},
	{0x054f, 238}, {0x0550, 240}, {0x0551, 242}, {0x0552, 244},
	{0x0553, 246}, {0x0554, 248}, {0x0555, 250}, {0x0556, 252},
	{0x055b, 176}, {0x055c, 175}, {0x055d, 170}, {0x055e, 177},
	{0x0561, 179}, {0x0562, 181}, {0x0563, 183}, {0x0564, 185},
	{0x0565, 187}, {0x0566, 189}, {0x0567, 191}, {0x0568, 193},
	{0x0569, 195}, {0x056a, 197}, {0x056b, 199}, {0x056c, 201},
	{0x056d, 203}, {0x056e, 205}, {0x056f, 207}, {0x0570, 209},
	{0x0571, 211}, {0x0572, 213}, {0x0573, 215}, {0x0574, 217},
	{0x0575, 219}, {0x0576, 221}, {0x0577, 223}, {0x0578, 225},
	{0x0579, 227}, {0x057a, 229}, {0x057b, 231}, {0x057c, 233},
	{0x057d, 235}, {0x057e, 237}, {0x057f, 239}, {0x0580, 241},
	{0x0581, 243}, {0x0582, 245}, {0x0583, 247}, {0x0584, 249},
	{0x0585, 251}, {0x0586, 253}, {0x0589, 163}, {0x058a, 173},
	{0x058e, 161}, {0x2013, 172}, {0x2014, 168}, {0x2022, 7},
	{0x2026, 174}, {0x203c, 19}, {0x2190, 27}, {0x2191, 24},
	{0x2192, 26}, {0x2193, 25}, {0x2194, 29}, {0x2195, 18},
	{0x21a8, 23}, {0x221f, 28}, {0x25ac, 22}, {0x25b2, 30},
	{0x25ba, 16}, {0x25bc, 31}, {0x25c4, 17}, {0x25cb, 9},
	{0x25d8, 8}, {0x25d9, 10}, {0x263a, 1}, {0x263b, 2},
	{0x263c, 15}, {0x2640, 12}, {0x2642, 11}, {0x2660, 6},
	{0x2663, 5}, {0x2665, 3}, {0x2666, 4}, {0x266a, 13},
	{0x266b, 14}, {0xfffd, 63}
};

const static uint32_t armscii8_unicode_table[128] = {
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0x00a0, 0x058e, 0x00a7, 0x0589, 0x0029, 0x002a, 0x00bb, 0x00ab,
	0x2014, 0x002e, 0x055d, 0x002c, 0x2013, 0x058a, 0x2026, 0x055c,
	0x055b, 0x055e, 0x0531, 0x0561, 0x0532, 0x0562, 0x0533, 0x0563,
	0x0534, 0x0564, 0x0535, 0x0565, 0x0536, 0x0566, 0x0537, 0x0567,
	0x0538, 0x0568, 0x0539, 0x0569, 0x053a, 0x056a, 0x053b, 0x056b,
	0x053c, 0x056c, 0x053d, 0x056d, 0x053e, 0x056e, 0x053f, 0x056f,
	0x0540, 0x0570, 0x0541, 0x0571, 0x0542, 0x0572, 0x0543, 0x0573,
	0x0544, 0x0574, 0x0545, 0x0575, 0x0546, 0x0576, 0x0547, 0x0577,
	0x0548, 0x0578, 0x0549, 0x0579, 0x054a, 0x057a, 0x054b, 0x057b,
	0x054c, 0x057c, 0x054d, 0x057d, 0x054e, 0x057e, 0x054f, 0x057f,
	0x0550, 0x0580, 0x0551, 0x0581, 0x0552, 0x0582, 0x0553, 0x0583,
	0x0554, 0x0584, 0x0555, 0x0585, 0x0556, 0x0586, 0x02bc, 0xfffd
};

const static struct ciolib_cpmap haik8_table[122] = {
	{0x00a0, 128}, {0x00a7, 21}, {0x00ab, 135}, {0x00b6, 20},
	{0x00bb, 134}, {0x02bc, 254}, {0x0531, 146}, {0x0532, 148},
	{0x0533, 150}, {0x0534, 152}, {0x0535, 154}, {0x0536, 156},
	{0x0537, 158}, {0x0538, 160}, {0x0539, 162}, {0x053a, 164},
	{0x053b, 166}, {0x053c, 168}, {0x053d, 170}, {0x053e, 172},
	{0x053f, 174}, {0x0540, 176}, {0x0541, 178}, {0x0542, 180},
	{0x0543, 182}, {0x0544, 184}, {0x0545, 186}, {0x0546, 188},
	{0x0547, 190}, {0x0548, 224}, {0x0549, 226}, {0x054a, 228},
	{0x054b, 230}, {0x054c, 232}, {0x054d, 234}, {0x054e, 236},
	{0x054f, 238}, {0x0550, 240}, {0x0551, 242}, {0x0552, 244},
	{0x0553, 246}, {0x0554, 248}, {0x0555, 250}, {0x0556, 252},
	{0x055b, 144}, {0x055c, 143}, {0x055d, 138}, {0x055e, 145},
	{0x0561, 147}, {0x0562, 149}, {0x0563, 151}, {0x0564, 153},
	{0x0565, 155}, {0x0566, 157}, {0x0567, 159}, {0x0568, 161},
	{0x0569, 163}, {0x056a, 165}, {0x056b, 167}, {0x056c, 169},
	{0x056d, 171}, {0x056e, 173}, {0x056f, 175}, {0x0570, 177},
	{0x0571, 179}, {0x0572, 181}, {0x0573, 183}, {0x0574, 185},
	{0x0575, 187}, {0x0576, 189}, {0x0577, 191}, {0x0578, 225},
	{0x0579, 227}, {0x057a, 229}, {0x057b, 231}, {0x057c, 233},
	{0x057d, 235}, {0x057e, 237}, {0x057f, 239}, {0x0580, 241},
	{0x0581, 243}, {0x0582, 245}, {0x0583, 247}, {0x0584, 249},
	{0x0585, 251}, {0x0586, 253}, {0x0589, 131}, {0x058a, 141},
	{0x058e, 129}, {0x2013, 140}, {0x2014, 136}, {0x2022, 7},
	{0x2026, 142}, {0x203c, 19}, {0x2190, 27}, {0x2191, 24},
	{0x2192, 26}, {0x2193, 25}, {0x2194, 29}, {0x2195, 18},
	{0x21a8, 23}, {0x221f, 28}, {0x25ac, 22}, {0x25b2, 30},
	{0x25ba, 16}, {0x25bc, 31}, {0x25c4, 17}, {0x25cb, 9},
	{0x25d8, 8}, {0x25d9, 10}, {0x263a, 1}, {0x263b, 2},
	{0x263c, 15}, {0x2640, 12}, {0x2642, 11}, {0x2660, 6},
	{0x2663, 5}, {0x2665, 3}, {0x2666, 4}, {0x266a, 13},
	{0x266b, 14}, {0xfffd, 63}
};

const static uint32_t haik8_unicode_table[128] = {
	0x00a0, 0x058e, 0x00a7, 0x0589, 0x0029, 0x002a, 0x00bb, 0x00ab,
	0x2014, 0x002e, 0x055d, 0x002c, 0x2013, 0x058a, 0x2026, 0x055c,
	0x055b, 0x055e, 0x0531, 0x0561, 0x0532, 0x0562, 0x0533, 0x0563,
	0x0534, 0x0564, 0x0535, 0x0565, 0x0536, 0x0566, 0x0537, 0x0567,
	0x0538, 0x0568, 0x0539, 0x0569, 0x053a, 0x056a, 0x053b, 0x056b,
	0x053c, 0x056c, 0x053d, 0x056d, 0x053e, 0x056e, 0x053f, 0x056f,
	0x0540, 0x0570, 0x0541, 0x0571, 0x0542, 0x0572, 0x0543, 0x0573,
	0x0544, 0x0574, 0x0545, 0x0575, 0x0546, 0x0576, 0x0547, 0x0577,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0x0548, 0x0578, 0x0549, 0x0579, 0x054a, 0x057a, 0x054b, 0x057b,
	0x054c, 0x057c, 0x054d, 0x057d, 0x054e, 0x057e, 0x054f, 0x057f,
	0x0550, 0x0580, 0x0551, 0x0581, 0x0552, 0x0582, 0x0553, 0x0583,
	0x0554, 0x0584, 0x0555, 0x0585, 0x0556, 0x0586, 0x02bc, 0xfffd
};

const static struct ciolib_cpmap atascii_table[58] = {
	{0x0007, 253}, {0x0008, 126}, {0x0009, 127}, {0x000d, 155},
	{0x007f, 254}, {0x00a0, 155}, {0x2022, 148}, {0x2190, 30},
	{0x2191, 28}, {0x2192, 159}, {0x2193, 157}, {0x1f8b0, 125},
	{0x21b0, 125}, {0x241b, 27}, {0x2500, 18}, {0x2502, 124},
	{0x250c, 145}, {0x2510, 5}, {0x2514, 26}, {0x2518, 131},
	{0x251c, 129}, {0x2524, 132}, {0x252c, 23}, {0x2534, 24},
	{0x253c, 147}, {0x2571, 6}, {0x2572, 135}, {0x2580, 149},
	{0x2582, 142}, {0x2584, 21}, {0x2586, 141}, {0x2588, 160},
	{0x258a, 130}, {0x258c, 25}, {0x258e, 150}, {0x2590, 153},
	{0x2596, 143}, {0x2597, 137}, {0x2598, 140}, {0x2599, 139},
	{0x259b, 137}, {0x259c, 143}, {0x259d, 139}, {0x259f, 140},
	{0x25b6, 127}, {0x25c0, 126}, {0x25d8, 148}, {0x25e2, 136},
	{0x25e3, 10}, {0x25e4, 136}, {0x25e5, 138}, {0x2660, 123},
	{0x2663, 144}, {0x2665, 63}, {0x2666, 96}, {0xfffd, 63},
	{0x1fb82, 13}, {0x1fb87, 2}
};

const static uint32_t atascii_unicode_table[256] = {
	0x2665, 0x251c,0x1fb87, 0x2518, 0x2524, 0x2510, 0x2571, 0x2572,
	0x25e2, 0x2597, 0x25e3, 0x259d, 0x2598,0x1fb82, 0x2582, 0x2596,
	0x2663, 0x250c, 0x2500, 0x253c, 0x2022, 0x2584, 0x258e, 0x252c,
	0x2534, 0x258c, 0x2514, 0x001b, 0x2191, 0x2193, 0x2190, 0x2192,
	0x0020, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x0027,
	0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
	0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
	0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
	0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
	0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
	0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
	0x0058, 0x0059, 0x005a, 0x005b, 0x005c, 0x005d, 0x005e, 0x005f,
	0x2666, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
	0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
	0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
	0x0078, 0x0079, 0x007a, 0x2660, 0x2502,0x1f8b0, 0x0008, 0x0009,
	0x2665, 0x251c, 0x258a, 0x2518, 0x2524, 0x2510, 0x2571, 0x2572,
	0x25e4, 0x259b, 0x25e5, 0x2599, 0x259f, 0x2586, 0x2582, 0x259c,
	0x2663, 0x250c, 0x2500, 0x253c, 0x25d8, 0x2580, 0x258e, 0x252c,
	0x2534, 0x2590, 0x2514, 0x000d, 0x2191, 0x2193, 0x2190, 0x2192,
	0x2588, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x0027,
	0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
	0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
	0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
	0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
	0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
	0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
	0x0058, 0x0059, 0x005a, 0x005b, 0x005c, 0x005d, 0x005e, 0x005f,
	0x2666, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
	0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
	0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
	0x0078, 0x0079, 0x007a, 0x2660, 0x2502, 0x0007, 0x007f, 0x25b6
};

const static uint32_t atascii_ext_table[256] = {
	0x2665, 0x251c,0x1fb87, 0x2518, 0x2524, 0x2510, 0x2571, 0x2572,
	0x25e2, 0x2597, 0x25e3, 0x259d, 0x2598,0x1fb82, 0x2582, 0x2596,
	0x2663, 0x250c, 0x2500, 0x253c, 0x2022, 0x2584, 0x258e, 0x252c,
	0x2534, 0x258c, 0x2514, 0x241b, 0x2191, 0x2193, 0x2190, 0x2192,
	0x0020, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x0027,
	0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
	0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
	0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
	0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
	0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
	0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
	0x0058, 0x0059, 0x005a, 0x005b, 0x005c, 0x005d, 0x005e, 0x005f,
	0x2666, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
	0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
	0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
	0x0078, 0x0079, 0x007a, 0x2660, 0x2502,0x1f8b0, 0x25c0, 0x25b6,
	0x2665, 0x251c,0x1fb87, 0x2518, 0x2524, 0x2510, 0x2571, 0x2572,
	0x25e2, 0x2597, 0x25e3, 0x259d, 0x2598,0x1fb82, 0x2582, 0x2596,
	0x2663, 0x250c, 0x2500, 0x253c, 0x2022, 0x2584, 0x258e, 0x252c,
	0x2534, 0x258c, 0x2514, 0x00a0, 0x2191, 0x2193, 0x2190, 0x2192,
	0x0020, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x0027,
	0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
	0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
	0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
	0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
	0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
	0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
	0x0058, 0x0059, 0x005a, 0x005b, 0x005c, 0x005d, 0x005e, 0x005f,
	0x2666, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
	0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
	0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
	0x0078, 0x0079, 0x007a, 0x2660, 0x2502,0x1f8b0, 0x25c0, 0x25b6
};

const static struct ciolib_cpmap petsciiu_table[129] = {
	{0x0020, 32}, {0x0021, 33}, {0x0022, 34}, {0x0023, 35},
	{0x0024, 36}, {0x0025, 37}, {0x0026, 38}, {0x0027, 39},
	{0x0028, 40}, {0x0029, 41}, {0x002a, 42}, {0x002b, 43},
	{0x002c, 44}, {0x002d, 45}, {0x002e, 46}, {0x002f, 47},
	{0x0030, 48}, {0x0031, 49}, {0x0032, 50}, {0x0033, 51},
	{0x0034, 52}, {0x0035, 53}, {0x0036, 54}, {0x0037, 55},
	{0x0038, 56}, {0x0039, 57}, {0x003a, 58}, {0x003b, 59},
	{0x003c, 60}, {0x003d, 61}, {0x003e, 62}, {0x003f, 63},
	{0x0040, 64}, {0x0041, 65}, {0x0042, 66}, {0x0043, 67},
	{0x0044, 68}, {0x0045, 69}, {0x0046, 70}, {0x0047, 71},
	{0x0048, 72}, {0x0049, 73}, {0x004a, 74}, {0x004b, 75},
	{0x004c, 76}, {0x004d, 77}, {0x004e, 78}, {0x004f, 79},
	{0x0050, 80}, {0x0051, 81}, {0x0052, 82}, {0x0053, 83},
	{0x0054, 84}, {0x0055, 85}, {0x0056, 86}, {0x0057, 87},
	{0x0058, 88}, {0x0059, 89}, {0x005a, 90}, {0x005b, 91},
	{0x005d, 93}, {0x00a0, 160}, {0x00a3, 92}, {0x03c0, 126},
	{0x2190, 95}, {0x2191, 94}, {0x2500, 96}, {0x2502, 125},
	{0x250c, 176}, {0x2510, 174}, {0x2514, 173}, {0x2518, 189},
	{0x251c, 171}, {0x2524, 179}, {0x252c, 178}, {0x2534, 177},
	{0x253c, 123}, {0x256d, 117}, {0x256e, 105}, {0x256f, 107},
	{0x2570, 106}, {0x2571, 110}, {0x2572, 109}, {0x2573, 118},
	{0x2581, 164}, {0x2582, 175}, {0x2583, 185}, {0x2584, 162},
	{0x258c, 161}, {0x258d, 181}, {0x258e, 180}, {0x258f, 165},
	{0x2592, 166}, {0x2594, 163}, {0x2595, 167}, {0x2596, 187},
	{0x2597, 172}, {0x2598, 190}, {0x259a, 191}, {0x259d, 188},
	{0x25cb, 119}, {0x25cf, 113}, {0x25e4, 169}, {0x25e5, 127},
	{0x2660, 97}, {0x2663, 120}, {0x2665, 115}, {0x2666, 122},
	{0xfffd, 63}, {0x1fb70, 116}, {0x1fb71, 103}, {0x1fb72, 98},
	{0x1fb74, 104}, {0x1fb75, 121}, {0x1fb76, 101}, {0x1fb77, 100},
	{0x1fb78, 99}, {0x1fb7a, 102}, {0x1fb7b, 114}, {0x1fb7c, 108},
	{0x1fb7d, 111}, {0x1fb7e, 112}, {0x1fb7f, 186}, {0x1fb82, 183},
	{0x1fb83, 184}, {0x1fb87, 170}, {0x1fb88, 182}, {0x1fb8c, 124},
	{0x1fb8f, 168}
};

const static uint32_t petsciiu_unicode_table[256] = {
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0x0020, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x0027,
	0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
	0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
	0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
	0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
	0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
	0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
	0x0058, 0x0059, 0x005a, 0x005b, 0x00a3, 0x005d, 0x2191, 0x2190,
	0x2500, 0x2660,0x1fb72,0x1fb78,0x1fb77,0x1fb76,0x1fb7a,0x1fb71,
	0x1fb74,0x256e, 0x2570, 0x256f,0x1fb7c, 0x2572, 0x2571,0x1fb7d,
	0x1fb7e,0x25cf,0x1fb7b, 0x2665,0x1fb70, 0x256d, 0x2573, 0x25cb,
	0x2663,0x1fb75, 0x2666, 0x253c,0x1fb8c, 0x2502, 0x03c0, 0x25e5,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0x00a0, 0x258c, 0x2584, 0x2594, 0x2581, 0x258f, 0x2592, 0x2595,
	0x1fb8f,0x25e4,0x1fb87, 0x251c, 0x2597, 0x2514, 0x2510, 0x2582,
	0x250c, 0x2534, 0x252c, 0x2524, 0x258e, 0x258d,0x1fb88,0x1fb82,
	0x1fb83,0x2583,0x1fb7f, 0x2596, 0x259d, 0x2518, 0x2598, 0x259a,
	0x2500, 0x2660,0x1fb72,0x1fb78,0x1fb77,0x1fb76,0x1fb7a,0x1fb71,
	0x1fb74,0x256e, 0x2570, 0x256f,0x1fb7c, 0x2572, 0x2571,0x1fb7d,
	0x1fb7e,0x25cf,0x1fb7b, 0x2665,0x1fb70, 0x256d, 0x2573, 0x25cb,
	0x2663,0x1fb75, 0x2666, 0x253c,0x1fb8c, 0x2502, 0x03c0, 0x25e5,
	0x00a0, 0x258c, 0x2584, 0x2594, 0x2581, 0x258f, 0x2592, 0x2595,
	0x1fb8f,0x25e4,0x1fb87, 0x251c, 0x2597, 0x2514, 0x2510, 0x2582,
	0x250c, 0x2534, 0x252c, 0x2524, 0x258e, 0x258d,0x1fb88,0x1fb82,
	0x1fb83,0x2583,0x1fb7f, 0x2596, 0x259d, 0x2518, 0x2598, 0x03c0
};

const static struct ciolib_cpmap petsciil_table[129] = {
	{0x0020, 32}, {0x0021, 33}, {0x0022, 34}, {0x0023, 35},
	{0x0024, 36}, {0x0025, 37}, {0x0026, 38}, {0x0027, 39},
	{0x0028, 40}, {0x0029, 41}, {0x002a, 42}, {0x002b, 43},
	{0x002c, 44}, {0x002d, 45}, {0x002e, 46}, {0x002f, 47},
	{0x0030, 48}, {0x0031, 49}, {0x0032, 50}, {0x0033, 51},
	{0x0034, 52}, {0x0035, 53}, {0x0036, 54}, {0x0037, 55},
	{0x0038, 56}, {0x0039, 57}, {0x003a, 58}, {0x003b, 59},
	{0x003c, 60}, {0x003d, 61}, {0x003e, 62}, {0x003f, 63},
	{0x0040, 64}, {0x0041, 97}, {0x0042, 98}, {0x0043, 99},
	{0x0044, 100}, {0x0045, 101}, {0x0046, 102}, {0x0047, 103},
	{0x0048, 104}, {0x0049, 105}, {0x004a, 106}, {0x004b, 107},
	{0x004c, 108}, {0x004d, 109}, {0x004e, 110}, {0x004f, 111},
	{0x0050, 112}, {0x0051, 113}, {0x0052, 114}, {0x0053, 115},
	{0x0054, 116}, {0x0055, 117}, {0x0056, 118}, {0x0057, 119},
	{0x0058, 120}, {0x0059, 121}, {0x005a, 122}, {0x005b, 91},
	{0x005d, 93}, {0x0061, 65}, {0x0062, 66}, {0x0063, 67},
	{0x0064, 68}, {0x0065, 69}, {0x0066, 70}, {0x0067, 71},
	{0x0068, 72}, {0x0069, 73}, {0x006a, 74}, {0x006b, 75},
	{0x006c, 76}, {0x006d, 77}, {0x006e, 78}, {0x006f, 79},
	{0x0070, 80}, {0x0071, 81}, {0x0072, 82}, {0x0073, 83},
	{0x0074, 84}, {0x0075, 85}, {0x0076, 86}, {0x0077, 87},
	{0x0078, 88}, {0x0079, 89}, {0x007a, 90}, {0x00a0, 160},
	{0x00a3, 92}, {0x2190, 95}, {0x2191, 94}, {0x2500, 96},
	{0x2502, 125}, {0x250c, 176}, {0x2510, 174}, {0x2514, 173},
	{0x2518, 189}, {0x251c, 171}, {0x2524, 179}, {0x252c, 178},
	{0x2534, 177}, {0x253c, 123}, {0x2581, 164}, {0x2582, 175},
	{0x2583, 185}, {0x2584, 162}, {0x258c, 161}, {0x258d, 181},
	{0x258e, 180}, {0x258f, 165}, {0x2592, 166}, {0x2594, 163},
	{0x2595, 167}, {0x2596, 187}, {0x2597, 172}, {0x2598, 190},
	{0x259a, 191}, {0x259d, 188}, {0x2713, 186}, {0xfffd, 63},
	{0x1fb82, 183}, {0x1fb83, 184}, {0x1fb87, 170}, {0x1fb88, 182},
	{0x1fb8c, 124}, {0x1fb8f, 168}, {0x1fb96, 126}, {0x1fb98, 127},
	{0x1fb99, 169}
};

const static uint32_t petsciil_unicode_table[256] = {
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0x0020, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x0027,
	0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
	0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037,
	0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
	0x0040, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
	0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
	0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077,
	0x0078, 0x0079, 0x007a, 0x005b, 0x00a3, 0x005d, 0x2191, 0x2190,
	0x2500, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
	0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
	0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
	0x0058, 0x0059, 0x005a, 0x253c,0x1fb8c, 0x2502,0x1fb96,0x1fb98,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
	0x00a0, 0x258c, 0x2584, 0x2594, 0x2581, 0x258f, 0x2592, 0x2595,
	0x1fb8f,0x1fb99,0x1fb87,0x251c, 0x2597, 0x2514, 0x2510, 0x2582,
	0x250c, 0x2534, 0x252c, 0x2524, 0x258e, 0x258d,0x1fb88,0x1fb82,
	0x1fb83,0x2583, 0x2713, 0x2596, 0x259d, 0x2518, 0x2598, 0x259a,
	0x2500, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
	0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
	0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057,
	0x0058, 0x0059, 0x005a, 0x253c,0x1fb8c, 0x2502,0x1fb96,0x1fb98,
	0x00a0, 0x258c, 0x2584, 0x2594, 0x2581, 0x258f, 0x2592, 0x2595,
	0x1fb8f,0x1fb99,0x1fb87,0x251c, 0x2597, 0x2514, 0x2510, 0x2582,
	0x250c, 0x2534, 0x252c, 0x2524, 0x258e, 0x258d,0x1fb88,0x1fb82,
	0x1fb83,0x2583, 0x2713, 0x2596, 0x259d, 0x2518, 0x2598,0x1fb96
};

const static struct ciolib_cpmap prestel_table[] = {
	{0x00080, 128}, {0x00081, 129},	{0x00082, 130}, {0x00083, 131},
	{0x00084, 132}, {0x00085, 133}, {0x00086, 134}, {0x00087, 135},
	{0x00088, 136}, {0x00089, 137}, {0x0008a, 138}, {0x0008b, 139},
	{0x0008c, 140}, {0x0008d, 141}, {0x0008e, 142}, {0x0008f, 143},
	{0x00090, 144}, {0x00091, 145}, {0x00092, 146}, {0x00093, 147},
	{0x00094, 148}, {0x00095, 149}, {0x00096, 150}, {0x00097, 151},
	{0x00098, 152}, {0x00099, 153}, {0x0009a, 154}, {0x0009b, 155},
	{0x0009c, 156}, {0x0009d, 157}, {0x0009e, 158}, {0x0009f, 159},

	{0x000a3,  35}, {0x000bc, 123}, {0x000bd,  92}, {0x000be, 125},
	{0x000f7, 126}, {0x02015,  96}, {0x02190,  91}, {0x02191,  94},
	{0x02192,  93}, {0x02317,  95}, {0x023f8, 124}, {0x02588, 255},
	{0x0258c, 181}, {0x02590, 234}, {0x025e2, 213}, {0x025e3, 197},
	{0x1fb00, 161}, {0x1fb01, 162}, {0x1fb02, 163}, {0x1fb03, 164},
	{0x1fb04, 165}, {0x1fb05, 166}, {0x1fb06, 167}, {0x1fb07, 168},
	{0x1fb08, 169}, {0x1fb09, 170}, {0x1fb0a, 171}, {0x1fb0b, 172},
	{0x1fb0c, 173}, {0x1fb0d, 174}, {0x1fb0e, 175}, {0x1fb0f, 176},
	{0x1fb10, 177}, {0x1fb11, 178}, {0x1fb12, 179}, {0x1fb13, 180},
	{0x1fb14, 182}, {0x1fb15, 183}, {0x1fb16, 184}, {0x1fb17, 185},
	{0x1fb18, 186}, {0x1fb19, 187}, {0x1fb1a, 188}, {0x1fb1b, 189},
	{0x1fb1c, 190}, {0x1fb1d, 191}, 
	{0x1fb1e, 224}, {0x1fb1f, 225}, {0x1fb20, 226}, {0x1fb21, 227},
	{0x1fb22, 228}, {0x1fb23, 229}, {0x1fb24, 230}, {0x1fb25, 231},
	{0x1fb26, 232}, {0x1fb27, 233}, {0x1fb28, 235}, {0x1fb29, 236},
	{0x1fb2a, 237}, {0x1fb2b, 238}, {0x1fb2c, 239}, {0x1fb2d, 240},
	{0x1fb2e, 241}, {0x1fb2f, 242}, {0x1fb30, 243}, {0x1fb31, 244},
	{0x1fb32, 245}, {0x1fb33, 246}, {0x1fb34, 247}, {0x1fb35, 248},
	{0x1fb36, 249}, {0x1fb37, 250}, {0x1fb38, 251}, {0x1fb39, 252},
	{0x1fb3a, 253}, {0x1fb3b, 254},
	
	{0x1fb3c, 192}, {0x1fb3d, 193}, {0x1fb3e, 194}, {0x1fb3f, 195},
	{0x1fb40, 196}, {0x1fb41, 198}, {0x1fb42, 199}, {0x1fb43, 200},
	{0x1fb44, 201}, {0x1fb45, 202}, {0x1fb46, 203}, {0x1fb47, 208},
	{0x1fb48, 209}, {0x1fb49, 210}, {0x1fb4a, 211}, {0x1fb4b, 212},
	{0x1fb4c, 214}, {0x1fb4d, 215}, {0x1fb4e, 216}, {0x1fb4f, 217},
	{0x1fb50, 218}, {0x1fb51, 219}, 
	{0x1fb68, 204}, {0x1fb69, 205},
	{0x1fb6a, 220}, {0x1fb6b, 221},
};

const static uint32_t prestel_unicode_table[256] = {
	0x00000, 0x00001, 0x00002, 0x00003,
	0x00004, 0x00005, 0x00006, 0x00007,
	0x00008, 0x00009, 0x0000a, 0x0000b,
	0x0000c, 0x0000d, 0x0000e, 0x0000f,
	0x00010, 0x00011, 0x00012, 0x00013,
	0x00014, 0x00015, 0x00016, 0x00017,
	0x00018, 0x00019, 0x0001a, 0x0001b,
	0x0001c, 0x0001d, 0x0001e, 0x0001f,
	0x00020, 0x00021, 0x00022, 0x000a3,
	0x00024, 0x00025, 0x00026, 0x00027,
	0x00028, 0x00029, 0x0002a, 0x0002b,
	0x0002c, 0x0002d, 0x0002e, 0x0002f,
	0x00030, 0x00031, 0x00032, 0x00033,
	0x00034, 0x00035, 0x00036, 0x00037,
	0x00038, 0x00039, 0x0003a, 0x0003b,
	0x0003c, 0x0003d, 0x0003e, 0x0003f,
	0x00040, 0x00041, 0x00042, 0x00043,
	0x00044, 0x00045, 0x00046, 0x00047,
	0x00048, 0x00049, 0x0004a, 0x0004b,
	0x0004c, 0x0004d, 0x0004e, 0x0004f,
	0x00050, 0x00051, 0x00052, 0x00053,
	0x00054, 0x00055, 0x00056, 0x00057,
	0x00058, 0x00059, 0x0005a, 0x02190,
	0x000bd, 0x02192, 0x02191, 0x02317,
	0x02015, 0x00061, 0x00062, 0x00063,
	0x00064, 0x00065, 0x00066, 0x00067,
	0x00068, 0x00069, 0x0006a, 0x0006b,
	0x0006c, 0x0006d, 0x0006e, 0x0006f,
	0x00070, 0x00071, 0x00072, 0x00073,
	0x00074, 0x00075, 0x00076, 0x00077,
	0x00078, 0x00079, 0x0007a, 0x000bc,
	0x023f8, 0x000be, 0x000f7, 0x0007f,

	0x00080, 0x00081, 0x00082, 0x00083,
	0x00084, 0x00085, 0x00086, 0x00087,
	0x00088, 0x00089, 0x0008a, 0x0008b,
	0x0008c, 0x0008d, 0x0008e, 0x0008f,
	0x00090, 0x00091, 0x00092, 0x00093,
	0x00094, 0x00095, 0x00096, 0x00097,
	0x00098, 0x00099, 0x0009a, 0x0009b,
	0x0009c, 0x0009d, 0x0009e, 0x0009f,

	0x00020, 0x1fb00, 0x1fb01, 0x1fb02,
	0x1fb03, 0x1fb04, 0x1fb05, 0x1fb06,
	0x1fb07, 0x1fb08, 0x1fb09, 0x1fb0a,
	0x1fb0b, 0x1fb0c, 0x1fb0d, 0x1fb0e,
	0x1fb0f, 0x1fb10, 0x1fb11, 0x1fb12,
	0x1fb13,  0x258c, 0x1fb14, 0x1fb15,
	0x1fb16, 0x1fb17, 0x1fb18, 0x1fb19,
	0x1fb1a, 0x1fb1b, 0x1fb1c, 0x1fb1d,

	0x1fb3c, 0x1fb3d, 0x1fb3e, 0x1fb3f,
	0x1fb40,  0x25e3, 0x1fb41, 0x1fb42,
	0x1fb43, 0x1fb44, 0x1fb45, 0x1fb46,
	0x1fb68, 0x1fb69, 0x00020, 0x00020,

	0x1fb47, 0x1fb48, 0x1fb49, 0x1fb4a,
	0x1fb4b,  0x25e2, 0x1fb4c, 0x1fb4d,
	0x1fb4e, 0x1fb4f, 0x1fb50, 0x1fb51,
	0x1fb6a, 0x1fb6b, 0x00020, 0x00020, 

	0x1fb1e, 0x1fb1f, 0x1fb20, 0x1fb21,
	0x1fb22, 0x1fb23, 0x1fb24, 0x1fb25,
	0x1fb26, 0x1fb27,  0x2590, 0x1fb28,
	0x1fb29, 0x1fb2a, 0x1fb2b, 0x1fb2c,
	0x1fb2d, 0x1fb2e, 0x1fb2f, 0x1fb30,
	0x1fb31, 0x1fb32, 0x1fb33, 0x1fb34,
	0x1fb35, 0x1fb36, 0x1fb37, 0x1fb38,
	0x1fb39, 0x1fb3a, 0x1fb3b,  0x2588,

/*
	// Contiguous
	0x00020, 0x1fb00, 0x1fb01, 0x1fb02,
	0x1fb03, 0x1fb04, 0x1fb05, 0x1fb06,
	0x1fb07, 0x1fb08, 0x1fb09, 0x1fb0a,
	0x1fb0b, 0x1fb0c, 0x1fb0d, 0x1fb0e,
	0x1fb0f, 0x1fb10, 0x1fb11, 0x1fb12,
	0x1fb13,  0x258c, 0x1fb14, 0x1fb15,
	0x1fb16, 0x1fb17, 0x1fb18, 0x1fb19,
	0x1fb1a, 0x1fb1b, 0x1fb1c, 0x1fb1d,
	0x1fb1e, 0x1fb1f, 0x1fb20, 0x1fb21,
	0x1fb22, 0x1fb23, 0x1fb24, 0x1fb25,
	0x1fb26, 0x1fb27,  0x2590, 0x1fb28,
	0x1fb29, 0x1fb2a, 0x1fb2b, 0x1fb2c,
	0x1fb2d, 0x1fb2e, 0x1fb2f, 0x1fb30,
	0x1fb31, 0x1fb32, 0x1fb33, 0x1fb34,
	0x1fb35, 0x1fb36, 0x1fb37, 0x1fb38,
	0x1fb39, 0x1fb3a, 0x1fb3b,  0x2588,

	// Separated
	0x00020, 0x1ce51, 0x1ce52, 0x1ce53,
	0x1ce54, 0x1ce55, 0x1ce56, 0x1ce57,
	0x1ce58, 0x1ce59, 0x1ce5a, 0x1ce5b,
	0x1ce5c, 0x1ce5d, 0x1ce5e, 0x1ce5f,
	0x1ce60, 0x1ce61, 0x1ce62, 0x1ce63,
	0x1ce64, 0x1ce65, 0x1ce66, 0x1ce67,
	0x1ce68, 0x1ce69, 0x1ce6a, 0x1ce6b,
	0x1ce6c, 0x1ce6d, 0x1ce6e, 0x1ce6f,
	0x1ce70, 0x1ce71, 0x1ce72, 0x1ce73,
	0x1ce74, 0x1ce75, 0x1ce76, 0x1ce77,
	0x1ce78, 0x1ce79, 0x1ce7a, 0x1ce7b,
	0x1ce7c, 0x1ce7d, 0x1ce7e, 0x1ce7f,
	0x1ce80, 0x1ce81, 0x1ce82, 0x1ce83,
	0x1ce84, 0x1ce85, 0x1ce86, 0x1ce87,
	0x1ce88, 0x1ce89, 0x1ce8a, 0x1ce8b,
	0x1ce8c, 0x1ce8d, 0x1ce8e, 0x1ce8f
*/
};

const static struct ciolib_cpmap prestel_table_sep[] = {
	{0x00080, 128}, {0x00081, 129}, {0x00082, 130}, {0x00083, 131},
	{0x00084, 132}, {0x00085, 133}, {0x00086, 134}, {0x00087, 135},
	{0x00088, 136}, {0x00089, 137}, {0x0008a, 138}, {0x0008b, 139},
	{0x0008c, 140}, {0x0008d, 141}, {0x0008e, 142}, {0x0008f, 143},
	{0x00090, 144}, {0x00091, 145}, {0x00092, 146}, {0x00093, 147},
	{0x00094, 148}, {0x00095, 149}, {0x00096, 150}, {0x00097, 151},
	{0x00098, 152}, {0x00099, 153}, {0x0009a, 154}, {0x0009b, 155},
	{0x0009c, 156}, {0x0009d, 157}, {0x0009e, 158}, {0x0009f, 159},

	{0x000a3, 35},
	{0x000bc, 123},
	{0x000bd, 92},
	{0x000be, 125},	{0x000f7, 126},
	{0x02015, 96},
	{0x02190, 91}, {0x02191, 94}, {0x02192, 93},
	{0x02317, 95},

	{0x023f8, 124},
	{0x025e2, 213},

	{0x025e3, 197},
	{0x1ce51, 161}, {0x1ce52, 162}, {0x1ce53, 163}, {0x1ce54, 164},
	{0x1ce55, 165}, {0x1ce56, 166}, {0x1ce57, 167}, {0x1ce58, 168},
	{0x1ce59, 169}, {0x1ce5a, 170}, {0x1ce5b, 171}, {0x1ce5c, 172},
	{0x1ce5d, 173}, {0x1ce5e, 174}, {0x1ce5f, 175}, {0x1ce60, 176},
	{0x1ce61, 177}, {0x1ce62, 178}, {0x1ce63, 179}, {0x1ce64, 180},
	{0x1ce65, 181}, {0x1ce66, 182}, {0x1ce67, 183}, {0x1ce68, 184},
	{0x1ce69, 185}, {0x1ce6a, 186}, {0x1ce6b, 187}, {0x1ce6c, 188},
	{0x1ce6d, 189}, {0x1ce6e, 190}, {0x1ce6f, 191},

	{0x1ce70, 224}, {0x1ce71, 225}, {0x1ce72, 226}, {0x1ce73, 227},
	{0x1ce74, 228}, {0x1ce75, 229}, {0x1ce76, 230}, {0x1ce77, 231},
	{0x1ce78, 232}, {0x1ce79, 233}, {0x1ce7a, 234}, {0x1ce7b, 235},
	{0x1ce7c, 236}, {0x1ce7d, 237}, {0x1ce7e, 238}, {0x1ce7f, 239},
	{0x1ce80, 240}, {0x1ce81, 241}, {0x1ce82, 242}, {0x1ce83, 243},
	{0x1ce84, 244}, {0x1ce85, 245}, {0x1ce86, 246}, {0x1ce87, 247},
	{0x1ce88, 248}, {0x1ce89, 249}, {0x1ce8a, 250}, {0x1ce8b, 251},
	{0x1ce8c, 252}, {0x1ce8d, 253}, {0x1ce8e, 254}, {0x1ce8f, 255},

	{0x1fb3c, 192}, {0x1fb3d, 193}, {0x1fb3e, 194}, {0x1fb3f, 195},
	{0x1fb40, 196},
	{0x1fb41, 198}, {0x1fb42, 199},
	{0x1fb43, 200}, {0x1fb44, 201}, {0x1fb45, 202}, {0x1fb46, 203},

	{0x1fb47, 208}, {0x1fb48, 209}, {0x1fb49, 210},
	{0x1fb4a, 211}, {0x1fb4b, 212},
	{0x1fb4c, 214}, {0x1fb4d, 215}, {0x1fb4e, 216}, {0x1fb4f, 217},
	{0x1fb50, 218}, {0x1fb51, 219},
	{0x1fb68, 204}, {0x1fb69, 205},
	{0x1fb6a, 220}, {0x1fb6b, 221},
};

const static uint32_t prestel_unicode_table_sep[256] = {
	0x00000, 0x00001, 0x00002, 0x00003,
	0x00004, 0x00005, 0x00006, 0x00007,
	0x00008, 0x00009, 0x0000a, 0x0000b,
	0x0000c, 0x0000d, 0x0000e, 0x0000f,
	0x00010, 0x00011, 0x00012, 0x00013,
	0x00014, 0x00015, 0x00016, 0x00017,
	0x00018, 0x00019, 0x0001a, 0x0001b,
	0x0001c, 0x0001d, 0x0001e, 0x0001f,
	0x00020, 0x00021, 0x00022, 0x000a3,
	0x00024, 0x00025, 0x00026, 0x00027,
	0x00028, 0x00029, 0x0002a, 0x0002b,
	0x0002c, 0x0002d, 0x0002e, 0x0002f,
	0x00030, 0x00031, 0x00032, 0x00033,
	0x00034, 0x00035, 0x00036, 0x00037,
	0x00038, 0x00039, 0x0003a, 0x0003b,
	0x0003c, 0x0003d, 0x0003e, 0x0003f,
	0x00040, 0x00041, 0x00042, 0x00043,
	0x00044, 0x00045, 0x00046, 0x00047,
	0x00048, 0x00049, 0x0004a, 0x0004b,
	0x0004c, 0x0004d, 0x0004e, 0x0004f,
	0x00050, 0x00051, 0x00052, 0x00053,
	0x00054, 0x00055, 0x00056, 0x00057,
	0x00058, 0x00059, 0x0005a, 0x02190,
	0x000bd, 0x02192, 0x02191, 0x02317,
	0x02015, 0x00061, 0x00062, 0x00063,
	0x00064, 0x00065, 0x00066, 0x00067,
	0x00068, 0x00069, 0x0006a, 0x0006b,
	0x0006c, 0x0006d, 0x0006e, 0x0006f,
	0x00070, 0x00071, 0x00072, 0x00073,
	0x00074, 0x00075, 0x00076, 0x00077,
	0x00078, 0x00079, 0x0007a, 0x000bc,
	0x023f8, 0x000be, 0x000f7, 0x0007f,

	0x00080, 0x00081, 0x00082, 0x00083,
	0x00084, 0x00085, 0x00086, 0x00087,
	0x00088, 0x00089, 0x0008a, 0x0008b,
	0x0008c, 0x0008d, 0x0008e, 0x0008f,
	0x00090, 0x00091, 0x00092, 0x00093,
	0x00094, 0x00095, 0x00096, 0x00097,
	0x00098, 0x00099, 0x0009a, 0x0009b,
	0x0009c, 0x0009d, 0x0009e, 0x0009f,

	0x00020, 0x1ce51, 0x1ce52, 0x1ce53,
	0x1ce54, 0x1ce55, 0x1ce56, 0x1ce57,
	0x1ce58, 0x1ce59, 0x1ce5a, 0x1ce5b,
	0x1ce5c, 0x1ce5d, 0x1ce5e, 0x1ce5f,
	0x1ce60, 0x1ce61, 0x1ce62, 0x1ce63,
	0x1ce64, 0x1ce65, 0x1ce66, 0x1ce67,
	0x1ce68, 0x1ce69, 0x1ce6a, 0x1ce6b,
	0x1ce6c, 0x1ce6d, 0x1ce6e, 0x1ce6f,

	0x1fb3c, 0x1fb3d, 0x1fb3e, 0x1fb3f,
	0x1fb40,  0x25e3, 0x1fb41, 0x1fb42,
	0x1fb43, 0x1fb44, 0x1fb45, 0x1fb46,
	0x1fb68, 0x1fb69, 0x00020, 0x00020,

	0x1fb47, 0x1fb48, 0x1fb49, 0x1fb4a,
	0x1fb4b,  0x25e2, 0x1fb4c, 0x1fb4d,
	0x1fb4e, 0x1fb4f, 0x1fb50, 0x1fb51,
	0x1fb6a, 0x1fb6b, 0x00020, 0x00020, 

	0x1ce70, 0x1ce71, 0x1ce72, 0x1ce73,
	0x1ce74, 0x1ce75, 0x1ce76, 0x1ce77,
	0x1ce78, 0x1ce79, 0x1ce7a, 0x1ce7b,
	0x1ce7c, 0x1ce7d, 0x1ce7e, 0x1ce7f,
	0x1ce80, 0x1ce81, 0x1ce82, 0x1ce83,
	0x1ce84, 0x1ce85, 0x1ce86, 0x1ce87,
	0x1ce88, 0x1ce89, 0x1ce8a, 0x1ce8b,
	0x1ce8c, 0x1ce8d, 0x1ce8e, 0x1ce8f

/*
	// Contiguous
	0x00020, 0x1fb00, 0x1fb01, 0x1fb02,
	0x1fb03, 0x1fb04, 0x1fb05, 0x1fb06,
	0x1fb07, 0x1fb08, 0x1fb09, 0x1fb0a,
	0x1fb0b, 0x1fb0c, 0x1fb0d, 0x1fb0e,
	0x1fb0f, 0x1fb10, 0x1fb11, 0x1fb12,
	0x1fb13,  0x258c, 0x1fb14, 0x1fb15,
	0x1fb16, 0x1fb17, 0x1fb18, 0x1fb19,
	0x1fb1a, 0x1fb1b, 0x1fb1c, 0x1fb1d,
	0x1fb1e, 0x1fb1f, 0x1fb20, 0x1fb21,
	0x1fb22, 0x1fb23, 0x1fb24, 0x1fb25,
	0x1fb26, 0x1fb27,  0x2590, 0x1fb28,
	0x1fb29, 0x1fb2a, 0x1fb2b, 0x1fb2c,
	0x1fb2d, 0x1fb2e, 0x1fb2f, 0x1fb30,
	0x1fb31, 0x1fb32, 0x1fb33, 0x1fb34,
	0x1fb35, 0x1fb36, 0x1fb37, 0x1fb38,
	0x1fb39, 0x1fb3a, 0x1fb3b,  0x2588,

	// Separated
	0x00020, 0x1ce51, 0x1ce52, 0x1ce53,
	0x1ce54, 0x1ce55, 0x1ce56, 0x1ce57,
	0x1ce58, 0x1ce59, 0x1ce5a, 0x1ce5b,
	0x1ce5c, 0x1ce5d, 0x1ce5e, 0x1ce5f,
	0x1ce60, 0x1ce61, 0x1ce62, 0x1ce63,
	0x1ce64, 0x1ce65, 0x1ce66, 0x1ce67,
	0x1ce68, 0x1ce69, 0x1ce6a, 0x1ce6b,
	0x1ce6c, 0x1ce6d, 0x1ce6e, 0x1ce6f,
	0x1ce70, 0x1ce71, 0x1ce72, 0x1ce73,
	0x1ce74, 0x1ce75, 0x1ce76, 0x1ce77,
	0x1ce78, 0x1ce79, 0x1ce7a, 0x1ce7b,
	0x1ce7c, 0x1ce7d, 0x1ce7e, 0x1ce7f,
	0x1ce80, 0x1ce81, 0x1ce82, 0x1ce83,
	0x1ce84, 0x1ce85, 0x1ce86, 0x1ce87,
	0x1ce88, 0x1ce89, 0x1ce8a, 0x1ce8b,
	0x1ce8c, 0x1ce8d, 0x1ce8e, 0x1ce8f
*/
};

const static struct ciolib_cpmap atarist_table[162] = {
	{0x000a1, 173}, {0x000a2, 155}, {0x000a3, 156}, {0x000a5, 157},
	{0x000a7, 221}, {0x000a8, 185}, {0x000a9, 189}, {0x000aa, 166},
	{0x000ab, 174}, {0x000ac, 170}, {0x000ae, 190}, {0x000af, 255},
	{0x000b0, 248}, {0x000b1, 241}, {0x000b2, 253}, {0x000b3, 254},
	{0x000b4, 186}, {0x000b5, 230}, {0x000b6, 188}, {0x000b7, 250},
	{0x000ba, 167}, {0x000bb, 175}, {0x000bc, 172}, {0x000bd, 171},
	{0x000bf, 168}, {0x000c0, 182}, {0x000c3, 183}, {0x000c4, 142},
	{0x000c5, 143}, {0x000c6, 146}, {0x000c7, 128}, {0x000c9, 144},
	{0x000d1, 165}, {0x000d5, 184}, {0x000d6, 153}, {0x000d8, 178},
	{0x000dc, 154}, {0x000df, 158}, {0x000e0, 133}, {0x000e1, 160},
	{0x000e2, 131}, {0x000e3, 176}, {0x000e4, 132}, {0x000e5, 134},
	{0x000e6, 145}, {0x000e7, 135}, {0x000e8, 138}, {0x000e9, 130},
	{0x000ea, 136}, {0x000eb, 137}, {0x000ec, 141}, {0x000ed, 161},
	{0x000ee, 140}, {0x000ef, 139}, {0x000f1, 164}, {0x000f2, 149},
	{0x000f3, 162}, {0x000f4, 147}, {0x000f5, 177}, {0x000f6, 148},
	{0x000f7, 246}, {0x000f8, 179}, {0x000f9, 151}, {0x000fa, 163},
	{0x000fb, 150}, {0x000fc, 129}, {0x000ff, 152}, {0x00132, 193},
	{0x00133, 192}, {0x00152, 181}, {0x00153, 180}, {0x00192, 159},
	{0x00259,  26}, {0x00393, 226}, {0x00398, 233}, {0x003a3, 228},
	{0x003a6, 232}, {0x003a9, 234}, {0x003b1, 224}, {0x003b2, 225},
	{0x003b4, 235}, {0x003c0, 227}, {0x003c3, 229}, {0x003c4, 231},
	{0x003c6, 237}, {0x005d0, 194}, {0x005d1, 195}, {0x005d2, 196},
	{0x005d3, 197}, {0x005d4, 198}, {0x005d5, 199}, {0x005d6, 200},
	{0x005d7, 201}, {0x005d8, 202}, {0x005d9, 203}, {0x005da, 217},
	{0x005db, 204}, {0x005dc, 205}, {0x005dd, 218}, {0x005de, 206},
	{0x005df, 216}, {0x005e0, 207}, {0x005e1, 208}, {0x005e2, 209},
	{0x005e3, 219}, {0x005e4, 210}, {0x005e5, 220}, {0x005e6, 211},
	{0x005e7, 212}, {0x005e8, 213}, {0x005e9, 214}, {0x005ea, 215},
	{0x02020, 187}, {0x0207f, 252}, {0x02122, 191}, {0x021e6,   4},
	{0x021e7,   1}, {0x021e8,   3}, {0x021e9,   2}, {0x02208, 238},
	{0x02219, 249}, {0x0221a, 251}, {0x0221e, 223}, {0x02227, 222},
	{0x02229, 239}, {0x0222e, 236}, {0x02248, 247}, {0x02261, 240},
	{0x02264, 243}, {0x02265, 242}, {0x02310, 169}, {0x02320, 244},
	{0x02321, 245}, {0x0240c,  12}, {0x0240d,  13}, {0x0241b,  27},
	{0x0266a,  11}, {0x02713,   8}, {0x0fffd,   0}, {0x0fffd,  14}, 
	{0x0fffd,  15}, {0x0fffd,  28}, {0x0fffd,  29}, {0x0fffd,  30},
	{0x0fffd,  31}, {0x1f514,  10}, {0x1f552,   9}, {0x1fbbd,   5},
	{0x1fbbe,   6}, {0x1fbbf,   7}, {0x1fbf0,  16}, {0x1fbf1,  17},
	{0x1fbf2,  18}, {0x1fbf3,  19}, {0x1fbf4,  20}, {0x1fbf5,  21},
	{0x1fbf6,  22}, {0x1fbf7,  23}, {0x1fbf8,  24}, {0x1fbf9,  25},
};

const static uint32_t atarist_ext_table[32] = {
	0xfffd, 0x21e7, 0x21e9, 0x21e8, 0x21e6, 0x1fbbd, 0x1fbbe, 0x1fbbf,
	0x2713,	0x1F552, 0x1F514, 0x266A, 0x240C, 0x240D, 0xfffd, 0xfffd, 
	0x1FBF0, 0x1FBF1, 0x1FBF2, 0x1FBF3, 0x1FBF4, 0x1FBF5, 0x1FBF6, 0x1FBF7,
	0x1FBF8, 0x1FBF9, 0x0259, 0x241B, 0xfffd, 0xfffd, 0xfffd, 0xfffd
};

const static uint32_t atarist_unicode_table[128] = {
	0x00C7, 0x00FC, 0x00E9, 0x00E2, 0x00E4, 0x00E0, 0x00E5, 0x00E7,
	0x00EA, 0x00EB, 0x00E8, 0x00EF, 0x00EE, 0x00EC, 0x00C4, 0x00C5,
	0x00C9, 0x00E6, 0x00C6, 0x00F4, 0x00F6, 0x00F2, 0x00FB, 0x00F9,
	0x00FF, 0x00D6, 0x00DC, 0x00A2, 0x00A3, 0x00A5, 0x00DF, 0x0192,
	0x00E1, 0x00ED, 0x00F3, 0x00FA, 0x00F1, 0x00D1, 0x00AA, 0x00BA,
	0x00BF, 0x2310, 0x00AC, 0x00BD, 0x00BC, 0x00A1, 0x00AB, 0x00BB,
	0x00E3, 0x00F5, 0x00D8, 0x00F8, 0x0153, 0x0152, 0x00C0, 0x00C3,
	0x00D5, 0x00A8, 0x00B4, 0x2020, 0x00B6, 0x00A9, 0x00AE, 0x2122,
	0x0133, 0x0132, 0x05D0, 0x05D1, 0x05D2, 0x05D3, 0x05D4, 0x05D5,
	0x05D6, 0x05D7, 0x05D8, 0x05D9, 0x05DB, 0x05DC, 0x05DE, 0x05E0,
	0x05E1, 0x05E2, 0x05E4, 0x05E6, 0x05E7, 0x05E8, 0x05E9, 0x05EA,
	0x05DF, 0x05DA, 0x05DD, 0x05E3, 0x05E5, 0x00A7, 0x2227, 0x221E,
	0x03B1, 0x03B2, 0x0393, 0x03C0, 0x03A3, 0x03C3, 0x00B5, 0x03C4,
	0x03A6, 0x0398, 0x03A9, 0x03B4, 0x222E, 0x03C6, 0x2208, 0x2229,
	0x2261, 0x00B1, 0x2265, 0x2264, 0x2320, 0x2321, 0x00F7, 0x2248,
	0x00B0, 0x2219, 0x00B7, 0x221A, 0x207F, 0x00B2, 0x00B3, 0x00AF,
};

static int
cmptab(const void *key, const void *entry)
{
	const uint32_t *pkey = key;
	const struct ciolib_cpmap *pentry = entry;

	if (*pkey == pentry->unicode)
		return 0;
	if (*pkey < pentry->unicode)
		return -1;
	return 1;
}

/*
 * ASCII is unchanged, no mapping attempted.
 */
static uint8_t
cptable_from_unicode_cpoint(uint32_t cpoint, char unmapped, const struct codepage_def *cpdef)
{
	struct ciolib_cpmap *mapped;

	if (cpoint < 128)
		return cpoint;
	mapped = bsearch(&cpoint, cpdef->cp_table, cpdef->cp_table_sz, sizeof(cpdef->cp_table[0]), cmptab);
	if (mapped == NULL)
		return unmapped;
	return mapped->cpchar;
}

/*
 * Fallback table... tries mapping first, ASCII that doesn't map is
 * returned unchanged.
 */
static uint8_t
fb_cptable_from_unicode_cpoint(uint32_t cpoint, char unmapped, const struct codepage_def *cpdef)
{
	struct ciolib_cpmap *mapped;

	mapped = bsearch(&cpoint, cpdef->cp_table, cpdef->cp_table_sz, sizeof(cpdef->cp_table[0]), cmptab);
	if (mapped == NULL) {
		if (cpoint < 128)
			return cpoint;
		return unmapped;
	}
	return mapped->cpchar;
}

static uint32_t
cpoint_from_cptable(uint8_t ch, const struct codepage_def *cpdef)
{
	if (ch < 128)
		return ch;
	return cpdef->cp_unicode_table[ch - 128];
}

static uint32_t
cpoint_from_cptable_ext(uint8_t ch, const struct codepage_def *cpdef)
{
	if (ch < 32)
		return cpdef->cp_ext_unicode_table[ch];
	return ciolib_cp[cpdef->cp].from_cpchar(ch, cpdef);
}

static uint32_t
ft_cpoint_from_cptable(uint8_t ch, const struct codepage_def *cpdef)
{
	return cpdef->cp_unicode_table[ch];
}

static uint32_t
ft_cpoint_from_cptable_ext(uint8_t ch, const struct codepage_def *cpdef)
{
	return cpdef->cp_ext_unicode_table[ch];
}

static int
write_cp(uint8_t *str, uint32_t cp)
{
	if (cp < 128) {
		*str = cp;
		return 1;
	}
	if (cp < 0x800) {
		*(str++) = 0xc0 | (cp >> 6);
		*(str++) = 0x80 | (cp & 0x3f);
		return 2;
	}
	if (cp < 0x10000) {
		*(str++) = 0xe0 | (cp >> 12);
		*(str++) = 0x80 | ((cp >> 6) & 0x3f);
		*(str++) = 0x80 | (cp & 0x3f);
		return 3;
	}
	if (cp < 0x110000) {
		*(str++) = 0xf0 | (cp >> 18);
		*(str++) = 0x80 | ((cp >> 12) & 0x3f);
		*(str++) = 0x80 | ((cp >> 6) & 0x3f);
		*(str++) = 0x80 | (cp & 0x3f);
		return 4;
	}
	return -1;
}

static int
read_cp(const uint8_t *str, uint32_t *cp)
{
	int incode;
	const uint8_t *p = str;

	if (cp == NULL)
		goto error;

	if (str == NULL)
		goto error;

	if (*p & 0x80) {
		if ((*p & 0xe0) == 0xc0) {
			incode = 1;
			*cp = *p & 0x1f;
		}
		else if ((*p & 0xf0) == 0xe0) {
			incode = 2;
			*cp = *p & 0x0f;
		}
		else if ((*p & 0xf8) == 0xf0) {
			incode = 3;
			*cp = *p & 0x07;
		}
		else
			goto error;

		while (incode) {
			p++;
			incode--;
			if ((*p & 0xc0) != 0x80)
				goto error;
			*cp <<= 6;
			*cp |= (*p & 0x3f);
		}
	}
	else {
		*cp = *p;
	}
	return p - str + 1;

error:
	if (cp)
		*cp = 0xffff;
	return -1;
}

static int
utf8_bytes(uint32_t cp)
{
	if (cp < 0x80)
		return 1;
	if (cp < 0x800)
		return 2;
	if (cp < 0x10000)
		return 3;
	if (cp < 0x110000)
		return 4;
	return -1;
}

static uint8_t *
cpstr_to_utf8(const char *cpstr, size_t buflen, size_t *outlen, const struct codepage_def *cpdef)
{
	size_t needed = 0;
	int cplen;
	uint8_t *ret = NULL;
	uint8_t *rp;
	size_t idx;
	uint8_t ch;

	// Calculate the number of bytes needed
	for (idx = 0; idx < buflen; idx++) {
		ch = cpstr[idx];
		if (ch == 0)
			cplen = 4;
		else if (ch < 128)
			cplen = 1;
		else
			cplen = utf8_bytes(cpdef->cp_unicode_table[ch - 128]);
		if (cplen == -1)
			goto error;
		needed += cplen;
	}

	ret = malloc(needed + 1);
	if (ret == NULL)
		goto error;

	rp = ret;
	for (idx = 0; idx < buflen; idx++) {
		ch = cpstr[idx];
		if (ch == 0) {
			*(rp++) = 0xef;
			*(rp++) = 0xbf;
			*(rp++) = 0xbe;
			cplen = 0;
		}
		else if (ch < 128) {
			*rp = ch;
			cplen = 1;
		}
		else {
			cplen = write_cp(rp, cpdef->cp_unicode_table[ch - 128]);
			if (cplen < 1)
				goto error;
		}
		rp += cplen;
	}
	*rp = 0;
	if (outlen)
		*outlen = rp - ret;
	return ret;

error:
	free(ret);
	return NULL;
}

/*
 * Full table... all unicode -> codepage translations must be in the
 * table.  For non-ascii codepages.
 */
static uint8_t
ft_from_unicode_cpoint(uint32_t cpoint, char unmapped, const struct codepage_def *cpdef)
{
	struct ciolib_cpmap *mapped;

	mapped = bsearch(&cpoint, cpdef->cp_table, cpdef->cp_table_sz, sizeof(cpdef->cp_table[0]), cmptab);
	if (mapped == NULL)
		return unmapped;
	return mapped->cpchar;
}

static uint8_t *
ftstr_to_utf8(const char *cpstr, size_t buflen, size_t *outlen, const struct codepage_def *cpdef)
{
	size_t needed = 0;
	int cplen;
	uint8_t *ret = NULL;
	uint8_t *rp;
	size_t idx;
	uint8_t ch;

	// Calculate the number of bytes needed
	for (idx = 0; idx < buflen; idx++) {
		ch = cpstr[idx];
		cplen = utf8_bytes(cpdef->cp_unicode_table[ch]);
		if (cplen == -1)
			goto error;
		needed += cplen;
	}

	ret = malloc(needed + 1);
	if (ret == NULL)
		goto error;

	rp = ret;
	for (idx = 0; idx < buflen; idx++) {
		ch = cpstr[idx];
		cplen = write_cp(rp, cpdef->cp_unicode_table[ch]);
		if (cplen < 1)
			goto error;
		rp += cplen;
	}
	*rp = 0;
	if (outlen)
		*outlen = rp - ret;
	return ret;

error:
	free(ret);
	return NULL;
}

/*
 * Converts UTF-8 to defined codepage, replacing unmapped characters
 * with unmapped if unmapped is zero, unmapped characters are stripped.
 *
 * Returns NULL if there are invalid sequences or codepoints.
 * Does not normalize the unicode, just a simple mapping
 * (TODO: Normalize into combined chars etc)
 */
static char *
utf8_to_cpstr(const uint8_t *utf8str, char unmapped, size_t inlen, size_t *outlen, const struct codepage_def *cpdef)
{
	size_t idx;
	char *rp;
	size_t outsz = 0;
	int incode = 0;
	uint32_t codepoint;
	char *ret = NULL;

	// TODO: Normalize UTF-8...

	// Calculate the number of code points and validate.
	for (idx = 0; idx < inlen; idx++) {
		if (incode) {
			switch (utf8str[idx] & 0xc0) {
				case 0x80:
					incode--;
					if (incode == 0)
						outsz++;
					break;
				default:
					goto error;
			}
		}
		else {
			if (utf8str[idx] & 0x80) {
				if ((utf8str[idx] & 0xe0) == 0xc0)
					incode = 1;
				else if ((utf8str[idx] & 0xf0) == 0xe0)
					incode = 2;
				else if ((utf8str[idx] & 0xf8) == 0xf0)
					incode = 3;
				else
					goto error;
			}
			else
				outsz++;
		}
	}
	ret = malloc(outsz + 1);
	if (ret == NULL)
		goto error;
	rp = ret;

	// Fill the string...
	for (idx = 0; idx < inlen;) {
		idx += read_cp(&utf8str[idx], &codepoint);
		if (codepoint == 0xffff || codepoint == 0xfffe)
			goto error;
		*(rp++) = cpdef->from_unicode_cpoint(codepoint, unmapped, cpdef);
	}
	*rp = 0;
	if (outlen)
		*outlen = rp - ret;

	return ret;
error:
	free(ret);
	return NULL;
}

const struct codepage_def ciolib_cp[CIOLIB_CP_COUNT] = {
	// 0
	{"CP437", CIOLIB_CP437, cpstr_to_utf8, utf8_to_cpstr, cptable_from_unicode_cpoint, cpoint_from_cptable,  cpoint_from_cptable_ext,
		cp437_table, sizeof(cp437_table) / sizeof(cp437_table[0]),
		cp437_unicode_table, cp437_ext_table},
	{"CP1251", CIOLIB_CP1251, cpstr_to_utf8, utf8_to_cpstr, cptable_from_unicode_cpoint, cpoint_from_cptable,  cpoint_from_cptable_ext,
		cp1251_table, sizeof(cp1251_table) / sizeof(cp1251_table[0]),
		cp1251_unicode_table, cp437_ext_table},
	{"KOI8-R", CIOLIB_KOI8_R, cpstr_to_utf8, utf8_to_cpstr, cptable_from_unicode_cpoint, cpoint_from_cptable,  cpoint_from_cptable_ext,
		koi8_r_table, sizeof(koi8_r_table) / sizeof(koi8_r_table[0]),
		koi8_r_unicode_table, empty_ext_table},
	{"ISO-8859-2", CIOLIB_ISO_8859_2, cpstr_to_utf8, utf8_to_cpstr, cptable_from_unicode_cpoint, cpoint_from_cptable,  cpoint_from_cptable_ext,
		iso8859_2_table, sizeof(iso8859_2_table) / sizeof(iso8859_2_table[0]),
		iso8859_2_unicode_table, empty_ext_table},
	{"ISO-8859-4", CIOLIB_ISO_8859_4, cpstr_to_utf8, utf8_to_cpstr, cptable_from_unicode_cpoint, cpoint_from_cptable,  cpoint_from_cptable_ext,
		iso8859_4_table, sizeof(iso8859_4_table) / sizeof(iso8859_4_table[0]),
		iso8859_4_unicode_table, empty_ext_table},
	// 5
	{"CP866M", CIOLIB_CP866M, cpstr_to_utf8, utf8_to_cpstr, cptable_from_unicode_cpoint, cpoint_from_cptable,  cpoint_from_cptable_ext,
		cp866m_table, sizeof(cp866m_table) / sizeof(cp866m_table[0]),
		cp866m_unicode_table, cp437_ext_table},
	{"ISO-8859-9", CIOLIB_ISO_8859_9, cpstr_to_utf8, utf8_to_cpstr, cptable_from_unicode_cpoint, cpoint_from_cptable,  cpoint_from_cptable_ext,
		iso8859_9_table, sizeof(iso8859_9_table) / sizeof(iso8859_9_table[0]),
		iso8859_9_unicode_table, empty_ext_table},
	{"ISO-8859-8", CIOLIB_ISO_8859_8, cpstr_to_utf8, utf8_to_cpstr, cptable_from_unicode_cpoint, cpoint_from_cptable,  cpoint_from_cptable_ext,
		iso8859_8_table, sizeof(iso8859_8_table) / sizeof(iso8859_8_table[0]),
		iso8859_8_unicode_table, empty_ext_table},
	{"KOI8-U", CIOLIB_KOI8_U, cpstr_to_utf8, utf8_to_cpstr, cptable_from_unicode_cpoint, cpoint_from_cptable,  cpoint_from_cptable_ext,
		koi8_u_table, sizeof(koi8_u_table) / sizeof(koi8_u_table[0]),
		koi8_u_unicode_table, empty_ext_table},
	{"ISO-8859-15", CIOLIB_ISO_8859_15, cpstr_to_utf8, utf8_to_cpstr, cptable_from_unicode_cpoint, cpoint_from_cptable,  cpoint_from_cptable_ext,
		iso8859_15_table, sizeof(iso8859_15_table) / sizeof(iso8859_15_table[0]),
		iso8859_15_unicode_table, empty_ext_table},
	// 10
	{"ISO-8859-5", CIOLIB_ISO_8859_5, cpstr_to_utf8, utf8_to_cpstr, cptable_from_unicode_cpoint, cpoint_from_cptable,  cpoint_from_cptable_ext,
		iso8859_5_table, sizeof(iso8859_5_table) / sizeof(iso8859_5_table[0]),
		iso8859_5_unicode_table, empty_ext_table},
	{"CP850", CIOLIB_CP850, cpstr_to_utf8, utf8_to_cpstr, cptable_from_unicode_cpoint, cpoint_from_cptable,  cpoint_from_cptable_ext,
		cp850_table, sizeof(cp850_table) / sizeof(cp850_table[0]),
		cp850_unicode_table, cp437_ext_table},
	{"CP865", CIOLIB_CP865, cpstr_to_utf8, utf8_to_cpstr, cptable_from_unicode_cpoint, cpoint_from_cptable,  cpoint_from_cptable_ext,
		cp865_table, sizeof(cp865_table) / sizeof(cp865_table[0]),
		cp865_unicode_table, cp437_ext_table},
	{"ISO-8859-7", CIOLIB_ISO_8859_7, cpstr_to_utf8, utf8_to_cpstr, cptable_from_unicode_cpoint, cpoint_from_cptable,  cpoint_from_cptable_ext,
		iso8859_7_table, sizeof(iso8859_7_table) / sizeof(iso8859_7_table[0]),
		iso8859_7_unicode_table, empty_ext_table},
	{"ISO-8859-1", CIOLIB_ISO_8859_1, cpstr_to_utf8, utf8_to_cpstr, cptable_from_unicode_cpoint, cpoint_from_cptable,  cpoint_from_cptable_ext,
		iso8859_1_table, sizeof(iso8859_1_table) / sizeof(iso8859_1_table[0]),
		iso8859_1_unicode_table, empty_ext_table},
	// 15
	{"CP866M2", CIOLIB_CP866M2, cpstr_to_utf8, utf8_to_cpstr, cptable_from_unicode_cpoint, cpoint_from_cptable,  cpoint_from_cptable_ext,
		cp866m2_table, sizeof(cp866m2_table) / sizeof(cp866m2_table[0]),
		cp866m2_unicode_table, cp437_ext_table},
	{"CP866U", CIOLIB_CP866U, cpstr_to_utf8, utf8_to_cpstr, cptable_from_unicode_cpoint, cpoint_from_cptable,  cpoint_from_cptable_ext,
		cp866u_table, sizeof(cp866u_table) / sizeof(cp866u_table[0]),
		cp866u_unicode_table, cp437_ext_table},
	{"CP1131", CIOLIB_CP1131, cpstr_to_utf8, utf8_to_cpstr, cptable_from_unicode_cpoint, cpoint_from_cptable,  cpoint_from_cptable_ext,
		cp1131_table, sizeof(cp1131_table) / sizeof(cp1131_table[0]),
		cp1131_unicode_table, cp437_ext_table},
	{"ARMSCII-8", CIOLIB_ARMSCII8, cpstr_to_utf8, utf8_to_cpstr, cptable_from_unicode_cpoint, cpoint_from_cptable,  cpoint_from_cptable_ext,
		armscii8_table, sizeof(armscii8_table) / sizeof(armscii8_table[0]),
		armscii8_unicode_table, cp437_ext_table},
	{"HAIK8", CIOLIB_HAIK8, cpstr_to_utf8, utf8_to_cpstr, cptable_from_unicode_cpoint, cpoint_from_cptable,  cpoint_from_cptable_ext,
		haik8_table, sizeof(haik8_table) / sizeof(haik8_table[0]),
		haik8_unicode_table, cp437_ext_table},
	// 20
	{"ATASCII", CIOLIB_ATASCII, ftstr_to_utf8, utf8_to_cpstr, fb_cptable_from_unicode_cpoint, ft_cpoint_from_cptable,  ft_cpoint_from_cptable_ext,
		atascii_table, sizeof(atascii_table) / sizeof(atascii_table[0]),
		atascii_unicode_table, atascii_ext_table},
	{"PETSCIIU", CIOLIB_PETSCIIU, ftstr_to_utf8, utf8_to_cpstr, ft_from_unicode_cpoint, ft_cpoint_from_cptable,  cpoint_from_cptable_ext,
		petsciiu_table, sizeof(petsciiu_table) / sizeof(petsciiu_table[0]),
		petsciiu_unicode_table, empty_ext_table},
	{"PETSCIIL", CIOLIB_PETSCIIL, ftstr_to_utf8, utf8_to_cpstr, ft_from_unicode_cpoint, ft_cpoint_from_cptable,  cpoint_from_cptable_ext,
		petsciil_table, sizeof(petsciil_table) / sizeof(petsciil_table[0]),
		petsciil_unicode_table, empty_ext_table},

	{"PRESTEL", CIOLIB_PRESTEL, ftstr_to_utf8, utf8_to_cpstr, cptable_from_unicode_cpoint, ft_cpoint_from_cptable,  cpoint_from_cptable_ext,
		prestel_table, sizeof(prestel_table) / sizeof(prestel_table[0]),
		prestel_unicode_table, empty_ext_table},
	{"PRESTEL-SEP", CIOLIB_PRESTEL_SEP, ftstr_to_utf8, utf8_to_cpstr, cptable_from_unicode_cpoint, ft_cpoint_from_cptable,  cpoint_from_cptable_ext,
		prestel_table_sep, sizeof(prestel_table_sep) / sizeof(prestel_table_sep[0]),
		prestel_unicode_table_sep, empty_ext_table},
	{"ATARI-ST", CIOLIB_ATARIST, cpstr_to_utf8, utf8_to_cpstr, cptable_from_unicode_cpoint, cpoint_from_cptable,  cpoint_from_cptable_ext,
		atarist_table, sizeof(atarist_table) / sizeof(atarist_table[0]),
		atarist_unicode_table, atarist_ext_table},
};

uint8_t *cp_to_utf8(enum ciolib_codepage cp, const char *cpstr, size_t buflen, size_t *outlen)
{
	if (cp < 0 || cp >= CIOLIB_CP_COUNT)
		return NULL;

	return ciolib_cp[cp].to_utf8(cpstr, buflen, outlen, &ciolib_cp[cp]);
}

char *utf8_to_cp(enum ciolib_codepage cp, const uint8_t *utf8str, char unmapped, size_t buflen, size_t *outlen)
{
	if (cp < 0 || cp >= CIOLIB_CP_COUNT)
		return NULL;

	return ciolib_cp[cp].utf8_to(utf8str, unmapped, buflen, outlen, &ciolib_cp[cp]);
}

uint8_t cpchar_from_unicode_cpoint(enum ciolib_codepage cp, uint32_t cpoint, char unmapped)
{
	if (cp < 0 || cp >= CIOLIB_CP_COUNT)
		return unmapped;

	return ciolib_cp[cp].from_unicode_cpoint(cpoint, unmapped, &ciolib_cp[cp]);
}

uint32_t cpoint_from_cpchar(enum ciolib_codepage cp, uint8_t ch)
{
	if (cp < 0 || cp >= CIOLIB_CP_COUNT)
		return 0;

	return ciolib_cp[cp].from_cpchar(ch, &ciolib_cp[cp]);
}

uint32_t cpoint_from_cpchar_ext(enum ciolib_codepage cp, uint8_t ch)
{
	if (cp < 0 || cp >= CIOLIB_CP_COUNT)
		return 0;

	return ciolib_cp[cp].from_cpchar_ext(ch, &ciolib_cp[cp]);
}
