open Types
open Activebuffer

(****************************)
(* Operations on connexions *)
(****************************)

(* bind the server socket *)
let bind_server port =
  let fds = Unix.socket Unix.PF_INET Unix.SOCK_STREAM 0 in
  Unix.setsockopt fds Unix.SO_REUSEADDR true;
  Unix.bind fds (Unix.ADDR_INET (Unix.inet_addr_any, port));
  Unix.listen fds 5;
  fds

(* catch all exceptions made by a function *)
let trywith f =
  try
    f ()
  with _ -> ()

(* Closes a connexion *)
let close_connexion conn =
  (match conn.state with
  | DNS -> Dns.forget conn
  | _ -> ());
  trywith
    (fun () -> Unix.shutdown conn.client Unix.SHUTDOWN_ALL);
  trywith
    (fun () -> Unix.close conn.client);
  if conn.server <> Unix.stdin then
    trywith (fun () -> Unix.close conn.server)

(* put a connexion in a state where it just has to
 * finish writing data to the client and then exit *)
let finish conn str =
  (match conn.state with
  | DNS -> Dns.forget conn
  | _ -> ());
  if str <> ""
  then conn.write_ans <- Activebuffer.activebuffer_of_string str
  else
    if conn.state_ans == CMD_LINE
    then conn.write_ans <- Activebuffer.activebuffer_of_string not_avail;
  if conn.server <> Unix.stdin then
    (trywith (fun () -> Unix.close conn.server);
     conn.server <- Unix.stdin);
  conn.state <- FINISHING

(***********************)
(* Operations on files *)
(***********************)

(* read a whole file *)
let read_file name =
  let chan = open_in_bin name in
  let buf_size = ref 1024 in
  let buf = ref (String.create !buf_size) in
  let nb_read = ref 0 in
  let cont = ref true in
  while !cont do
    let tmp = input chan !buf !nb_read (!buf_size - !nb_read) in
    if tmp = 0
    then cont := false
    else
      (nb_read := !nb_read + tmp;
       if !nb_read = !buf_size
       then
         (buf := !buf ^ (String.create !buf_size);
          buf_size := 2 * !buf_size))
  done;
  close_in chan;
  String.sub !buf 0 !nb_read

(* hack for using the parser *)
let pars = ref (fun x -> assert false)
let lex = ref 0
let read_string_list file =
  let chan = open_in file in
  let lexbuf = Lexing.from_channel chan in
  let res = !pars !lex lexbuf in
  close_in chan;
  res

(* does the string match the pattern *)
let match_pattern str pat =
  let size_s = String.length str in
  let size_p = String.length pat in
  let rec match_aux pos_s pos_p =
    if pos_p == size_p then
      pos_s == size_s
    else
      ((pat.[pos_p] == '*' && match_aux pos_s (pos_p+1))
     ||
       (pos_s != size_s &&
        (match pat.[pos_p] with
        | '*' ->
            match_aux (pos_s+1) pos_p
        | '?' -> match_aux (pos_s+1) (pos_p+1)
        | x -> x = (Char.lowercase str.[pos_s])
              && match_aux (pos_s+1) (pos_p+1))))
  in
  match_aux 0 0

(* decode password for authentication *)
let decode64 s =
  if s = ""
  then ""
  else
    let val64 c =
      match c with
      | 'A' .. 'Z' -> (Char.code c) - (Char.code 'A')
      | 'a' .. 'z' -> (Char.code c) - (Char.code 'a') + 26
      | '0' .. '9' -> (Char.code c) - (Char.code '0') + 52
      | '+' -> 62 | '/' -> 63 | '=' -> 0
      | _ -> failwith "not a base64 string" in
    let len = String.length s in
    let len_res = len * 3 / 4 in
    let res = String.create len_res in
    for i=0 to len/4 - 1 do
      let i1 = 4*i and i2 = 3*i in
      let v1 = (val64 s.[i1]) lsl 18 in
      let v2 = (val64 s.[i1 + 1]) lsl 12 in
      let v3 = (val64 s.[i1 + 2]) lsl 6 in
      let v4 = val64 s.[i1 + 3] in
      let v = v1 lor v2 lor v3 lor v4 in
      res.[i2] <- Char.chr (v lsr 16);
      res.[i2 + 1] <- Char.chr (v lsr 8 land 0xFF);
      res.[i2 + 2] <- Char.chr (v land 0xFF)
    done;
    let nb_keep =
      if s.[len-1] = '=' then
        if s.[len-2] = '=' then len_res - 2 else len_res - 1
      else len_res in
    String.sub res 0 nb_keep

(* encode passwd for authentication *)
let encode64 s =
  let b64 =
    [| 'A'; 'B'; 'C'; 'D'; 'E'; 'F'; 'G'; 'H'; 'I'; 'J'; 'K'; 'L'; 'M';
       'N'; 'O'; 'P'; 'Q'; 'R'; 'S'; 'T'; 'U'; 'V'; 'W'; 'X'; 'Y'; 'Z';
       'a'; 'b'; 'c'; 'd'; 'e'; 'f'; 'g'; 'h'; 'i'; 'j'; 'k'; 'l'; 'm';
       'n'; 'o'; 'p'; 'q'; 'r'; 's'; 't'; 'u'; 'v'; 'w'; 'x'; 'y'; 'z';
       '0'; '1'; '2'; '3'; '4'; '5'; '6'; '7'; '8'; '9'; '+'; '/' |] in
  let len = String.length s in
  let lenres = if len = 0 then 0 else ((len - 1) / 3 + 1) * 4 in
  let res = String.make lenres '=' in
  for i = 0 to len / 3 - 1 do
    let v = (Char.code (s.[3*i]) lsl 16) +
        (Char.code (s.[3*i+1]) lsl 8) + (Char.code (s.[3*i+2])) in
    res.[4*i] <- b64.(v lsr 18);
    res.[4*i+1] <- b64.((v lsr 12) mod 64);
    res.[4*i+2] <- b64.((v lsr 6) mod 64);
    res.[4*i+3] <- b64.(v mod 64)
  done;
  (match len mod 3 with
  | 0 -> ()
  | 1 ->
      let v = Char.code s.[len-1] in
      res.[lenres-4] <- b64.(v lsr 2);
      res.[lenres-3] <- b64.((v mod 4) lsl 4)
  | 2 ->
      let v = (Char.code (s.[len-2]) lsl 8) + (Char.code (s.[len-1])) in
      res.[lenres-4] <- b64.(v lsr 10);
      res.[lenres-3] <- b64.((v lsr 4) mod 64);
      res.[lenres-2] <- b64.((v mod 16) lsl 2)
  | _ -> assert false);
  res
