/*
 * firewallstate.c
 *
 * Functions to add firewall entries, and restore to previously existing
 * settings.
 *
 * Copyright (c) 2003 Todd MacDermid <tmacd@synacklabs.net>
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "packetp.h"

/* add_tcp_block sets up a firewall entry to block all incoming TCP packets
 * on all interfaces, except loopback. Note to self: how do we ensure that
 * holes poked for existing services don't let things through before
 * we block here? 
 *
 * This iterates over each interface, building a linked list as it
 * goes along.
 */

int add_tcp_block(const struct intf_entry *entry, void *ctx_arg) {
  struct packetp_ctx *ctx;
  struct fw_node *new_node;
  struct addr addr_any;

  if(entry->intf_type == INTF_TYPE_LOOPBACK) {
    return(0);
  }

  addr_any.addr_type = ADDR_TYPE_IP;
  addr_any.addr_bits = 0;
  addr_any.addr_ip = IP_ADDR_ANY;

  ctx = ctx_arg;
    
  if ((new_node = (struct fw_node *)calloc(1, sizeof(struct fw_node))) == NULL) {
    fprintf(stderr, "add_tcp_block: Failed to allocate memory\n");
    return(-1);
  }
    
  new_node->next = ctx->fw_head;
  strncpy(new_node->rule.fw_device, entry->intf_name, FWDEVSZ-1);
  new_node->rule.fw_op = FW_OP_BLOCK;
  new_node->rule.fw_dir = FW_DIR_IN;
  new_node->rule.fw_proto = IP_PROTO_TCP;
  memcpy(&(new_node->rule.fw_src), &addr_any, sizeof(struct addr));
  memcpy(&(new_node->rule.fw_dst), &(entry->intf_addr), sizeof(struct addr));
  new_node->rule.fw_sport[0] = 0;
  new_node->rule.fw_sport[1] = TCP_PORT_MAX;
  new_node->rule.fw_dport[0] = 0;
  new_node->rule.fw_dport[1] = TCP_PORT_MAX;

  if(fw_add(ctx->fw_handle, &(new_node->rule)) < 0) {
    fprintf(stderr, "add_tcp_block: fw_add failed\n");
    free(new_node);
    return(-1);
  }
  
  ctx->fw_head = new_node;
  return(0);
}

/* packetp_fw_init adds in firewall entries as appropriate to block traffic 
 * either on all interfaces in the case of a target of IP_ADDR_ANY, or to
 * the appropriate interface if an individual target is set. There is
 * no firewall set in FAKEIP mode.
 *
 * NOTE: This makes no attempt to play nicely with preexisting firewall
 * rules. As a matter of fact, if you depend on a firewall for security
 * on a server, and run this, you will probably be bypassing the rules
 * already in existence! (Pcap pulls the packets off the wire before they
 * hit the firewall, and packetp_process may reinject them). Just warning
 * ya...
 *
 * Alternatively, if you have specific hole punched through a default
 * deny firewall, this may not be able to stop packets, and thus, 
 * packet purgatory won't stop the kernel from interfering. This whole 
 * section needs some  intelligence added to become more generally useful. 
 * Ah well. Later. Perfect is the enemy of good enough, and I wanna get 
 * this working.
 */

int packetp_fw_init(struct packetp_ctx *ctx) {
  struct fw_node *new_node;
  struct route_node *route_info;

  if(ctx->target_ip.addr_ip == IP_ADDR_ANY) {
    if((intf_loop(ctx->intf_handle, add_tcp_block, ctx)) < 0) {
      fprintf(stderr, "packetp_fw_init: Error in intf_loop\n");
      return(-1);
    }
  } else {  /* If we've only been given a single CIDR block of target_ip */
    if((route_info = packetp_route_find(ctx, &(ctx->target_ip))) == NULL) {
      fprintf(stderr, "packetp_fw_init: Could not find interface for %s\n", 
	      addr_ntoa(&(ctx->target_ip)));
      return(-1);
    } 
    
    if ((new_node = (struct fw_node *)calloc(1, sizeof(struct fw_node))) == NULL) {
      fprintf(stderr, "packetp_fw_init: Failed to allocate memory\n");
      return(-1);
    }
    
    new_node->next = NULL;
    strncpy(new_node->rule.fw_device, route_info->interface.intf_name, FWDEVSZ-1);
    new_node->rule.fw_op = FW_OP_BLOCK;
    new_node->rule.fw_dir = FW_DIR_IN;
    new_node->rule.fw_proto = IP_PROTO_TCP;
    memcpy(&(new_node->rule.fw_src), &(ctx->target_ip), sizeof(struct addr));
    memcpy(&(new_node->rule.fw_dst), &(route_info->interface.intf_addr), sizeof(struct addr));
    new_node->rule.fw_sport[0] = 0;
    new_node->rule.fw_sport[1] = TCP_PORT_MAX;
    new_node->rule.fw_dport[0] = 0;
    new_node->rule.fw_dport[1] = TCP_PORT_MAX;

    if(fw_add(ctx->fw_handle, &(new_node->rule)) < 0) {
      fprintf(stderr, "packetp_fw_init: fw_add failed\n");
      free(new_node);
      return(-1);
    }
    
    ctx->fw_head = new_node;
  }
  return(0);
}

/* packetp_fw_end deletes all rules that were added to the firewall 
 * by packetp_fw_init, and frees the memory allocated for the linked list. 
 * Returns -1 on failure (which means you probably have to go fix your 
 * firewall entries by hand), 0 on success.
 */

int packetp_fw_end(struct packetp_ctx *ctx) {
  struct fw_node *walk_node, *del_node;

  walk_node = ctx->fw_head;

  while(walk_node != NULL) {
    del_node = walk_node;
    walk_node = walk_node->next;

    if((fw_delete(ctx->fw_handle, &(del_node->rule))) < 0) {
      fprintf(stderr, "Error removing firewall rule.\n");
      return(-1);
    }
    free(del_node);
  }
  return(0);
}
