#ifdef WIN32
#include <windows.h> 
#endif
#include <GL/gl.h>
#include <GL/glu.h>
#include "stk_window.h"

/* --------------------------------------------------------- */
/* ----------------------- Rectangle ----------------------- */
/* --------------------------------------------------------- */
Stk_Gui_Rect::Stk_Gui_Rect(float xa,float ya,float size_x,float size_y)
{
    x=xa;
    y=ya;
    width=size_x;
    height=size_y;
}

Stk_Gui_Rect::Stk_Gui_Rect(const Stk_Gui_Rect &r)
{
    x=r.x;
    y=r.y;
    width=r.width;
    height=r.height;
}

Stk_Gui_Rect::Stk_Gui_Rect()
{
    x=y=width=height=0;
}

void Stk_Gui_Rect::Resize(const Stk_Gui_Rect &r)
{
    x=r.x;
    y=r.y;
    width=r.width;
    height=r.height;  
}

void Stk_Gui_Rect::Resize(float xa,float ya,float size_x,float size_y)
{
    x=xa;
    y=ya;
    width=size_x;
    height=size_y;
}


float Stk_Gui_Rect::GetX() const
{
    return x;
}

float Stk_Gui_Rect::GetY() const
{
    return y;
}

float Stk_Gui_Rect::GetWidth() const
{
    return width;
}

float Stk_Gui_Rect::GetHeight() const
{
    return height;
}


/* ---------------------------------------------------------- */
/* ----------------------- GContext ------------------------- */
/* ---------------------------------------------------------- */

Stk_Gui_GContext::Stk_Gui_GContext(const Stk_Gui_Rect &r)
        :Stk_Gui_Rect(r)
{
}

Stk_Gui_GContext::~Stk_Gui_GContext()
{}


void Stk_Gui_GContext::Lock()
{
        //okiglViewport(1,width,height-1,0);
    
    glMatrixMode(GL_PROJECTION);
    glPushMatrix();
    glLoadIdentity();
    gluOrtho2D(0, GetWidth(),0,GetHeight());
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    float mat[16];
    mat[0]=1.0f;  mat[4]=0.0f;  mat[8]=0.0f;   mat[12]=0.0f;
    mat[1]=0.0f;  mat[5]=-1.0f; mat[9]=0.0f;   mat[13]=GetHeight();
    mat[2]=0.0f;  mat[6]=0.0f;  mat[10]=1.0f;  mat[14]=0.0f;
    mat[3]=0.0f;  mat[7]=0.0f;  mat[11]=0.0f;  mat[15]=1.0f;
    glLoadMatrixf(mat);
    glEnable(GL_BLEND);
    glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);

}

void Stk_Gui_GContext::Free()
{
    glDisable(GL_BLEND);
    glMatrixMode(GL_PROJECTION);
    glPopMatrix();glPopMatrix();
    glMatrixMode(GL_MODELVIEW);
    glPopMatrix();
}

/* -------------------------------------------------------- */
/* ----------------------- Widget ------------------------- */
/* -------------------------------------------------------- */
Stk_Gui_Widget::Stk_Gui_Widget()
        :Stk_Gui_Rect()
{}

Stk_Gui_Widget::Stk_Gui_Widget(const Stk_Gui_Rect &r)
        :Stk_Gui_Rect(r)
{}

Stk_Gui_Widget::~Stk_Gui_Widget()
{}

void Stk_Gui_Widget::Draw()
{}

void Stk_Gui_Widget::Attach(Stk_Handler *son)
{
    AddHandler(son);
}

void Stk_Gui_Widget::Unfasten()
{
    if(father!=0){
        father->RemHandler(this);
    }
}


/* --------------------- Button -------------------------- */
Stk_Gui_Button::Stk_Gui_Button(const Stk_Gui_Rect &r)
        :Stk_Gui_Widget(r)
{}


Stk_Gui_Button::~Stk_Gui_Button() 
{}


void Stk_Gui_Button::Draw()
{
    glPushMatrix();
    glTranslatef(GetX(),GetY(),0);
    DrawButton();
    glPopMatrix();
}

void Stk_Gui_Button::DrawButton()
{}


void Stk_Gui_Button::MessageReceived(Stk_Message *msg) 
{
    int *valeur;
        // we have interest only in mouse event
    if(msg->what!=STK_MOU_EVENT)
        return;
        // we have interest in type 1 message
    msg->FindInt("type",valeur);
     if(*valeur!=1)
         return;
        // we have interest in keypressed message
    msg->FindInt("pressed",valeur);
    if(*valeur==0)
        return;
        // we are interesting by click left
    msg->FindInt("button",valeur);
    if(*valeur!=STK_GUI_LEFT)
        return;
    int *x,*y;
    msg->FindInt("x",x);
    msg->FindInt("y",y);
    if( ((*x)>=GetX()) && ((*x)<GetX()+GetWidth()) &&
        ((*y)>=GetY()) && ((*y)<GetY()+GetHeight()) ){
        Click();
    }
    
}


void Stk_Gui_Button::Click() 
{
    printf("click on button\n");
}



/* --------------------- Container ------------------------ */
Stk_Gui_Container::Stk_Gui_Container(const Stk_Gui_Rect &r)
        :Stk_Gui_Widget(r)
{
    widget=0;
}

Stk_Gui_Container::~Stk_Gui_Container()
{
    if(widget!=0){
        delete widget;
    }
    widget=0;
}

void Stk_Gui_Container::Attach(Stk_Gui_Widget *w)
{
    Stk_Gui_Widget::Attach(w);
    widget=w;
        // take the size of the container
    w->Resize(*this);
}

void Stk_Gui_Container::Unfasten()
{
    Stk_Gui_Widget::Unfasten();
    widget=0;
}

void Stk_Gui_Container::Draw()
{
    glPushMatrix();
        //glTranslatef(GetX(),GetY(),0);
    if(widget!=0){
        widget->Draw();
    }
    glPopMatrix();
}

void Stk_Gui_Container::Resize(const Stk_Gui_Rect &r)
{
    Stk_Gui_Rect::Resize(r);
    if(widget!=0){
        widget->Resize(r);
    }
}

/* -------------------- Static ----------------------- */
Stk_Gui_Static::Stk_Gui_Static()
        :Stk_Gui_Widget()
{
    while(widgets.size()!=0){
        delete widgets.back();
        widgets.pop_back();
    }
}

Stk_Gui_Static::~Stk_Gui_Static()
{

}

void Stk_Gui_Static::Attach(Stk_Gui_Widget *w)
{
    Stk_Gui_Widget::Attach(w);
    widgets.push_back(w);
}

void Stk_Gui_Static::Unfasten(Stk_Gui_Widget *w)
{
    Stk_Gui_Widget::Unfasten();
    widgets.remove(w);
}

void Stk_Gui_Static::Resize(const Stk_Gui_Rect &r)
{
    Stk_Gui_Rect::Resize(r);
}

void Stk_Gui_Static::Draw()
{
    glPopMatrix();
        //glTranslatef(GetX(),GetY(),0);
    typedef list<Stk_Gui_Widget *>::iterator iter_widget;
    for(iter_widget i=widgets.begin();i!=widgets.end();i++){
        (*i)->Draw();
    }
    glPopMatrix();
}



/* ----------------------- VBox --------------------------- */
Stk_Gui_VBox::Stk_Gui_VBox()
        :Stk_Gui_Widget()
{
    mode=STK_GUI_SIZE_AUTO;
}

Stk_Gui_VBox::Stk_Gui_VBox(int md)
{
    mode=md;
}


Stk_Gui_VBox::~Stk_Gui_VBox()
{
    while(widgets.size()!=0){
        delete widgets.back();
        widgets.pop_back();
    }
}

void Stk_Gui_VBox::Draw()
{
    glPopMatrix();
        //printf("x %f y %f \n",GetX(),GetY());
        //glTranslatef(GetX(),GetY(),0);
    typedef list<Stk_Gui_Widget *>::iterator iter_widget;
    for(iter_widget i=widgets.begin();i!=widgets.end();i++){
        (*i)->Draw();
    }
    glPushMatrix();
}


void Stk_Gui_VBox::function1()
{
        // Compute the number of widgets attached
    float nb=widgets.size();
        // If no elements
    if(nb==0)
        return;
        // else
    float begin_x=GetX();
    float begin_y=GetY();
    float w_x=GetWidth();
    float w_y=GetHeight()/nb;
    typedef list<Stk_Gui_Widget *>::iterator iter_widget;
        // compute the size of wigets contained
    for(iter_widget i=widgets.begin();i!=widgets.end();i++){
        if(mode==STK_GUI_SIZE_NOAUTO){
            w_y=(*i)->GetHeight();
                //printf("%f\n",w_y);
            }
        Stk_Gui_Rect rect(begin_x,begin_y,w_x,w_y);
        (*i)->Resize(rect);
        begin_y+=w_y;
    }
}

void Stk_Gui_VBox::Attach(Stk_Gui_Widget *w)
{
    Stk_Gui_Widget::Attach(w);
    widgets.push_back(w);
    function1();
}

void Stk_Gui_VBox::Unfasten(Stk_Gui_Widget *w)
{
    Stk_Gui_Widget::Unfasten();
    widgets.remove(w);
    function1();
}


void Stk_Gui_VBox::Resize(const Stk_Gui_Rect &r)
{
    Stk_Gui_Rect::Resize(r);
    function1();
}



/* ----------------------- HBox --------------------------- */
Stk_Gui_HBox::Stk_Gui_HBox()
        :Stk_Gui_Widget()
{
    mode=STK_GUI_SIZE_AUTO;
}

Stk_Gui_HBox::Stk_Gui_HBox(int md)
        :Stk_Gui_Widget()
{
    mode=md;
}


Stk_Gui_HBox::~Stk_Gui_HBox()
{
    while(widgets.size()!=0){
        delete widgets.back();
        widgets.pop_back();
    } 
}

void Stk_Gui_HBox::Draw()
{
    glPushMatrix();
        //glTranslatef(GetX(),GetY(),0);
    typedef list<Stk_Gui_Widget *>::iterator iter_widget;
    for(iter_widget i=widgets.begin();i!=widgets.end();i++){
        (*i)->Draw();
    }
    glPopMatrix();  
}

void Stk_Gui_HBox::function1()
{
            // Compute the size of widgets attached
    float size=widgets.size();
        // If no elements
    if(size==0)
        return;
        // else
    float begin_x=GetX();
    float begin_y=GetY();
    float w_x=GetWidth()/size;
    float w_y=GetHeight();
    typedef list<Stk_Gui_Widget *>::iterator iter_widget;
        // compute the size of wigets contained
    for(iter_widget i=widgets.begin();i!=widgets.end();i++){
        if(mode==STK_GUI_SIZE_NOAUTO){
                w_x=(*i)->GetWidth();
        }
        Stk_Gui_Rect rect(begin_x,begin_y,w_x,w_y);
        (*i)->Resize(rect);
        begin_x+=w_x;
    }
}



void Stk_Gui_HBox::Attach(Stk_Gui_Widget *w)
{
    Stk_Gui_Widget::Attach(w);
    widgets.push_back(w);
    function1();
}

void Stk_Gui_HBox::Unfasten(Stk_Gui_Widget *w)
{
    Stk_Gui_Widget::Unfasten();
    widgets.remove(w);
    function1();
}


void Stk_Gui_HBox::Resize(const Stk_Gui_Rect &r)
{
    Stk_Gui_Rect::Resize(r);
    function1();
}



/* ----------------------- Window ------------------------- */


Stk_Gui_Window::Stk_Gui_Window(Stk_Handler *handler,const Stk_Gui_Rect &r)
        :Stk_Gui_Container(r)
{
    handler->AddHandler(this);
    gc=new Stk_Gui_GContext(r);
    cursor=1;
}


Stk_Gui_Window::~Stk_Gui_Window()
{
    if(father!=0){
        father->RemHandler(this);
    }
    delete gc;
}

void Stk_Gui_Window::DrawBackGround()
{}


void Stk_Gui_Window::Draw()
{
    gc->Lock();
        // draw a background
    DrawBackGround();
        // draw the container
    Stk_Gui_Container::Draw();
        // draw cursor if necessary
    if(cursor){
        DrawCursor();
    }
    gc->Free();
}

void Stk_Gui_Window::MessageReceived(Stk_Message *msg)
{
        // we have interest only in mouse event
    if(msg->what==STK_MOU_EVENT){
        int *x,*y;
        msg->FindInt("x",x);
        msg->FindInt("y",y);
            // save cursor position
        cur_x=*x;
        cur_y=*y;
    }
    
}

void Stk_Gui_Window::ShowCursor()
{
    cursor=1;
}

void Stk_Gui_Window::HideCursor()
{
    cursor=0;
}


void Stk_Gui_Window::DrawCursor()
{
    glPushAttrib(GL_DEPTH_BUFFER_BIT|GL_TEXTURE_BIT|GL_LIGHTING_BIT|GL_POINT_BIT);
    glDisable(GL_DEPTH_TEST);
    glPointSize(4.0);
    glEnable(GL_POINT_SMOOTH);
    glDisable(GL_LIGHTING);
    glDisable(GL_TEXTURE_2D);
    glBegin(GL_POINTS);
    glColor3f(1.0,1.0,1.0);
    glVertex3f(cur_x,cur_y,0);
    glEnd();
    glPointSize(1.0);
    glPopAttrib();
}
