%{
/*
 * Copyright (c) 2004 - 2013 Marc Balmer <marc@msys.ch>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <err.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>

#include "smtp-vilter.h"
#include "vilter-clamd.h"
#include "pathnames.h"

extern FILE	*clamdin;
extern char	*clamdtext;

extern int	 clamd_port;
extern int	 write_from;
extern char	*clamd_host;
extern char	*bind_addr;
extern char	*clamd_sock;
extern int	 clamd_tries;
extern int	 clamd_timeout;
extern int	 scantype;
extern int	 chroot_scanrealpath;

extern int	 clamdlineno;
static int	 clamderrcnt;
static char	*clamdcfgfile;

int	clamderror(const char *, ...);
int	clamdparse(void);
int	clamdlex(void);

%}

%union {
	long number;
	char *string;
}

%token	HOST PORT TRIES TIMEOUT SCANTYPE OPTION
%token	CHROOT_SCANREALPATH FILESYSTEM SOCKET FILDES LOCALSOCK
%token	BIND
%token	<string>	NUMBER
%token	<string>	TEXT

%%
statement	: /* empty */
		| statement '\n'
		| statement host '\n'
		| statement port '\n'
		| statement localsock '\n'
		| statement bind '\n'
		| statement tries '\n'
		| statement timeout '\n'
		| statement scantype '\n'
		| statement option '\n'
		;

host		: HOST '=' TEXT				{
			clamd_host = $3;
		}
		;

port		: PORT '=' NUMBER				{
			clamd_port = atoi($3);
			free($3);
		}
		;

localsock	: LOCALSOCK '=' TEXT			{
			clamd_sock = $3;
		}
		;

bind		: BIND '=' TEXT				{
			bind_addr = $3;
		}
		;

tries		: TRIES '=' NUMBER			{
			clamd_tries = atoi($3);
			free($3);
		}
		;
timeout		: TIMEOUT '=' NUMBER			{
			clamd_timeout = atoi($3);
			free($3);
		}
		;

scantype	: SCANTYPE '=' FILESYSTEM		{
			scantype = SCANTYPE_FILESYSTEM;
		}
		| SCANTYPE '=' SOCKET			{
			scantype = SCANTYPE_SOCKET;
		}
		| SCANTYPE '=' FILDES			{
			scantype = SCANTYPE_FILDES;
		}
		;

option		: OPTION '=' CHROOT_SCANREALPATH	{
			chroot_scanrealpath = 1;
		}
		;

%%

int
vilter_init(char *cfgfile)
{
	if (verbose)
		warnx("clamd: vilter_init()");

	/* Default values */

	clamd_port = 3310;
	if ((clamd_host = strdup("localhost")) == NULL)
		errx(1, "clamd: out of memory");
	bind_addr = NULL;
	clamd_tries = 1;
	clamd_timeout = TIMEOUT;
	scantype = SCANTYPE_FILESYSTEM;
	chroot_scanrealpath = 0;

	if (cfgfile == NULL)
		cfgfile = CLAMD_CONF;
	if (verbose)
		warnx("clamd: use config file %s", cfgfile);

	clamdlineno = 1;
	clamderrcnt = 0;
	clamdcfgfile = cfgfile;
	if ((clamdin = fopen(cfgfile, "r")) != NULL) {
		while (!feof(clamdin))
			clamdparse();
		fclose(clamdin);
	} else if (verbose)
		warnx("clamd: configuration file %s for clamd backend not "
		    "found, using default values", cfgfile);

	if (clamderrcnt)
		errx(1, "configuration file contains errors, terminating");
	return 0;
}

int
clamderror(const char *fmt, ...)
{
	va_list		 ap;
	char		*nfmt;

	++clamderrcnt;
	va_start(ap, fmt);
	if (asprintf(&nfmt, "%s, line %d: %s near '%s'",
	    clamdcfgfile, clamdlineno, fmt, clamdtext) == -1)
		errx(1, "clamd: asprintf failed");
	va_end(ap);
	fprintf(stderr, "%s\n", nfmt);
	free(nfmt);
	return 0;
}
