/*
 * Copyright (c) 1997, Matthew N. Dodd
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static const char copyright[] =
	"Copyright (c) 1997 Matthew N. Dodd\n\tAll rights reserved.\n";
#endif /* not lint */

#ifndef lint
static const char rcsid[] =
	"$Id: quotatool.c,v 1.17 1997/03/31 23:02:42 winter Exp $";
#endif /* not lint */

#ifndef __QUOTATOOL_MAIN__
#define __QUOTATOOL_MAIN__
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/param.h>
#include <sys/errno.h>
#include <ufs/ufs/quota.h>
#include <pwd.h>
#include <grp.h>
#include <fstab.h>
#include "quotatool.h"

void usage ();
void do_test ();
u_long convert_block (char*);
u_long convert_inode (char*);
time_t convert_time (char*);
int valid_fs (char*);
qmstr* mklist(int, char**);

char *m_str[] = {"M_SETQUOTA", "M_GETQUOTA", "M_TEST"};
char *m_proc[] = {"setquota", "getquota", "main"};
char *m_qtype[] = {"userquota", "groupquota"};

int f_mode;			/* flag for mode of operation	*/
int f_user	= 0;
int f_group	= 0;
int f_mtype	= 0;		/* member type 0 user 1 group	*/
char f_fsname[MAXPATHLEN];	/* filesystem to use		*/
int f_bh 	= 0;
int f_bs 	= 0;
int f_bg 	= 0;
int f_ih 	= 0;
int f_is 	= 0;
int f_ig 	= 0;
u_long v_bh	= 0;		/* block hard quota		*/
u_long v_bs	= 0;		/* block soft quota		*/
time_t v_bg	= 0;		/* block grace period		*/
u_long v_ih	= 0;		/* inode hard quota		*/
u_long v_is	= 0;		/* inode soft quota		*/
time_t v_ig	= 0;		/* inode grace period		*/

int
main(argc, argv)
	int	argc;
	char	*argv[];
{
	extern char *optarg;
	extern int optind;
	extern int optopt;
	extern int opterr;
/*	extern int optreset; */

	register char *p;
	char **av;
	int ch;
	qmstr *qm = NULL, *qm_tmp;

	av = argv;
	p = strrchr(*av, '/');
	if (p++ == NULL)
		p = *av;

	if (strncmp(p, m_proc[M_GETQUOTA], strlen(m_proc[M_GETQUOTA])) == 0)
		f_mode = M_GETQUOTA;
	else if (strncmp(p, m_proc[M_SETQUOTA], strlen(m_proc[M_SETQUOTA])) == 0)
		f_mode = M_SETQUOTA;
	else if (strncmp(p, m_proc[M_TEST], strlen(m_proc[M_TEST])) == 0)
		f_mode = M_TEST;
	else {
		fprintf(stderr, "%s : No clue what to do with myself!\n", p);
		exit(1);
	}

	if (argc < 2)
		usage();

	while ((ch = getopt(argc, argv, "ugf:b:i:?")) != -1) {
		switch(ch) {
			case 'u' :
				f_user = 1;
				break;
			case 'g' :
				f_group = 1;
				break;
			case 'f' :
				if (valid_fs(optarg)) {
					strncpy(f_fsname, optarg, MAXPATHLEN);
				}
				break;
			case 'b' :
				if (strlen(optarg) <= 1)
					usage();
				switch(optarg[0]) {
					case 'h' :
						f_bh = 1;
						v_bh = convert_block(optarg+1);
						break;
					case 's' :
						f_bs = 1;
						v_bs = convert_block(optarg+1);
						break;
					case 'g' :
						f_bg = 1;
						v_bg = convert_time(optarg+1);
						break;
					default :
						break;
				}
				break;
			case 'i' :
				if (strlen(optarg) <= 1)
					usage();
				switch(optarg[0]) {
					case 'h' :
						f_ih = 1;
						v_ih = convert_inode(optarg+1);
						break;
					case 's' :
						f_is = 1;
						v_is = convert_inode(optarg+1);
						break;
					case 'g' :
						f_ig = 1;
						v_ig = convert_time(optarg+1);
						break;
					default :
						break;
				}
				break;
			case '?' :
			default :
				usage();
				break;
		}
	}

	if (f_user && f_group) {
		f_mtype = 0;
	} else {
		f_mtype = f_group;
	}

	if (strlen(f_fsname) == 0) {
		usage();
	}

	argc -= optind;
	argv += optind;

	qm = mklist(argc, argv); 
	qm_tmp = qm;

	switch (f_mode) {
		case M_SETQUOTA :
			set_quota(qm_tmp);
			break;
		case M_GETQUOTA :
			print_quota(qm_tmp);
			break;
		case M_TEST :
		default :
			do_test();
			break;
	}

	while(qm) {
		qm_tmp = qm;
		qm = qm->next;
		free(qm_tmp);
	}

	return 0;
}

void
do_test()
{
	printf("  f_mode %s\n", m_str[f_mode]);
	printf(" f_mtype %d\n", f_mtype);
	printf("  f_user %d\n", f_user);
	printf(" f_group %d\n", f_group);
	printf("f_fsname %s\n", f_fsname);
	printf("    f_bh %d\n", f_bh);
	printf("    f_bs %d\n", f_bs);
	printf("    f_bg %d\n", f_bg);
	printf("    f_ih %d\n", f_ih);
	printf("    f_is %d\n", f_is);
	printf("    f_ig %d\n", f_ig);
	printf("    v_bh %lu\n", v_bh);
	printf("    v_bs %lu\n", v_bs);
	printf("    v_bg %lu\n", v_bg);
	printf("    v_ih %lu\n", v_ih);
	printf("    v_is %lu\n", v_is);
	printf("    v_ig %lu\n", v_ig);

	return;
}

void
usage()
{
	switch(f_mode) {
		case M_SETQUOTA :
			fprintf(stderr, "usage : %s [-u | -g] [-f fs] [-bhval] [-bsval] [-bgval] \\\n", m_proc[f_mode]);
			fprintf(stderr, "                                   [-ihval] [-isval] [-igval] user/group ...\n");
			break;
		case M_GETQUOTA :
			fprintf(stderr, "usage : %s [-u | -g] [-f fs] user/group ...\n", m_proc[f_mode]);
			break;
		case M_TEST :
			fprintf(stderr, "usage : %s [whatever]\n", m_proc[f_mode]);
			break;
	}

	exit(1);
}

u_long
convert_block(p_block)
	char	*p_block;
{
	u_long	value = 0;
	char	unit = NULL;
	u_long	r_blocks;

	if (sscanf(p_block, "%lu%c", &value, &unit) != 2) {
		if (unit == NULL)
			unit = 'k';
	}

	switch(unit) {
		case 'k' :
		case 'K' :
			r_blocks = value * 2;
			break;
		case 'm' :
		case 'M' :
			r_blocks = value * 1024 * 2;
			break;
		default :
			fprintf(stderr, "%s : '%c' is not a valid size unit.\n", 
				m_proc[f_mode], unit);
			usage();
			break;
	}

	return r_blocks;
}

time_t
convert_time(p_time)
	char	*p_time;
{
	u_long	value = 0;
	char	unit = NULL;
	time_t	r_time;

	if (sscanf(p_time, "%ld%c", &value, &unit) != 2) {
		if (unit == NULL)
			unit = 'd';
	}

	switch(unit) {
		case 'h' :
		case 'H' :
			r_time = value * 3600;
			break;
		case 'd' :
		case 'D' :
			r_time = value * 86400;
			break;
		case 'w' :
		case 'W' :
			r_time = value * 604800;
			break;
		case 'm' :
		case 'M' :
			r_time = value * 2592000;
			break;
		default :
			fprintf(stderr, "%s : '%c' is not a valid time unit.\n", 
				m_proc[f_mode], unit);
			break;
	}

	return r_time;
}

u_long
convert_inode(p_inode)
	char	*p_inode;
{
	u_long  r_inode = 0;

	sscanf(p_inode, "%lu", &r_inode);

	return r_inode;
}

int
valid_fs(fsname)
	char	*fsname;
{
	struct fstab *fs;
	
	if (fs = getfsfile(fsname)) {
		if (!strstr(fs->fs_mntops, m_qtype[f_group])) {
			fprintf(stderr, "%s : %s does not have quotas enabled.\n", m_proc[f_mode], fsname);
			exit(1);
		}
	} else {
		fprintf(stderr, "%s : %s is not a valid filesystem.\n", m_proc[f_mode], fsname);
		exit(1);
	}

	return 1;
}

qmstr* 
mklist(argc, argv)
	int	argc;
	char	*argv[];
{
	int ch, error;
	int qcmd;
	qmstr *qm, *qm_tmp;
	struct passwd *pw;
	struct group *gr;

	qm = NULL;

	qcmd = QCMD(Q_GETQUOTA, f_mtype);
	switch (f_mtype) {
		case 0 :
			for (ch = 0; ch < argc; ch++) {
 				if (!(pw = getpwnam(argv[ch]))) {
			 		fprintf(stderr, "%s : %s is not a valid user.\n", m_proc[f_mode], argv[ch]);
					exit(1);
				}

				qm_tmp = qm;
				error = 0;
				while(qm_tmp && !error) {
					if (qm_tmp->id == pw->pw_uid) {
						fprintf(stderr, "%s : %s was already specified.\n", m_proc[f_mode], pw->pw_name);
						error = 1;
					}
					qm_tmp = qm_tmp->next;
				}
				if (error)
					continue;
				
				qm_tmp = (qmstr*)malloc(sizeof(qmstr));
				strcpy(qm_tmp->name, pw->pw_name);
				qm_tmp->id = pw->pw_uid;
				qm_tmp->dq_dqb.dqb_btime = (time_t)0;
				qm_tmp->dq_dqb.dqb_itime = (time_t)0;
				if (quotactl(f_fsname, qcmd, pw->pw_uid, (char*)&qm_tmp->dq_dqb) != 0) {
					fprintf(stderr, "%s : GETQUOTA(%s) - %s\n", 
						m_proc[f_mode], pw->pw_name, strerror(errno));
					exit(1);
				}
				qm_tmp->next = NULL;
				if (qm)
					qm_tmp->next = qm;
				qm = qm_tmp;
			}
			break;
		case 1 :
			for (ch = 0; ch < argc; ch++) {
				if (!(gr = getgrnam(argv[ch]))) {
			 		fprintf(stderr, "%s : %s is not a valid group.\n",
						m_proc[f_mode], argv[ch]);
					exit(1);
				}

				qm_tmp = qm;
				error = 0;
				while(qm_tmp && !error) {
					if (qm_tmp->id == gr->gr_gid) {
						fprintf(stderr, "%s : %s was already specified.\n",
							m_proc[f_mode], gr->gr_name);
						error = 1;
					}
					qm_tmp = qm_tmp->next;
				}
				if (error)
					continue;
			
				qm_tmp = (qmstr*)malloc(sizeof(qmstr));
				strcpy(qm_tmp->name, gr->gr_name);
				qm_tmp->id = gr->gr_gid;
				if (quotactl(f_fsname, qcmd, gr->gr_gid, (char*)&qm_tmp->dq_dqb) != 0) {
					fprintf(stderr, "%s : GETQUOTA(%s) - %s\n", 
						m_proc[f_mode], gr->gr_name, strerror(errno));
					exit(1);
				}
				qm_tmp->next = NULL;
				if (qm)
					qm_tmp->next = qm;
				qm = qm_tmp;
			}
			break;
	}

	return qm;
}
