/*
 * tnmSnmpNet.c --
 *
 *	This file contains all functions that handle transport over UDP.
 *	There is also some code here to talk to the straps daemon which is
 *	used to receive and forward SNMP traps send to the privileged 162
 *	port.
 *
 * Copyright (c) 1994-1996 Technical University of Braunschweig.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#include "tnmSnmp.h"

#ifndef __WIN32__
#include <sys/un.h>
#endif

/*
 * Local variables:
 */

extern int hexdump;		/* flag that controls hexdump */
static int sock = -1;		/* socket to send/receive messages */
static int trap_sock = -1;	/* socket to receive traps */
static int trap_count = 0;	/* reference counter for trap socket */

static Tcl_File mgrSocket;
static Tcl_File trapSocket;

static char *serv_path = "/tmp/.straps-162";

/*
 * A global variable for performance measurements.
 */

#ifdef SNMP_BENCH
Tnm_SnmpMark tnmSnmpBenchMark;
#endif

/*
 * The default filename where we will find the straps binary. This
 * is normally overwritten in the Makefile.
 */

#ifndef STRAPS
#define STRAPS "/usr/local/bin/straps"
#endif

/*
 * Forward declarations for procedures defined later in this file:
 */

static int
xread			_ANSI_ARGS_((int fd, char *buf, int len));

static int
straps			_ANSI_ARGS_((Tcl_Interp *interp));

static void
DumpPacket		_ANSI_ARGS_((Tcl_Interp *interp, 
				     u_char *packet, int packetlen,
				     char *msg, struct sockaddr_in *from));
static void
ResponseProc		_ANSI_ARGS_((ClientData clientData, int mask));

static void
TrapProc		_ANSI_ARGS_((ClientData clientData, int mask));

static void
AgentProc		_ANSI_ARGS_((ClientData clientData, int mask));

static int
TrapRecv		_ANSI_ARGS_((Tcl_Interp *interp, 
				     u_char *packet, int *packetlen, 
				     struct sockaddr_in *from));
static int
AgentRecv		_ANSI_ARGS_((Tcl_Interp *interp, SNMP_Session *session,
				     u_char *packet, int *packetlen,
				     struct sockaddr_in *from));

#ifndef __WIN32__

#ifdef SIGPIPE
static void
IgnorePipe		_ANSI_ARGS_((void));
#endif


/*
 *----------------------------------------------------------------------
 *
 * IgnorePipe --
 *
 *	This procedure is a dummy signal handler to catch SIGPIPE
 *	signals. Restart signalhandler for all these bozo's outside.
 *
 * Results:
 *	None.
 * 
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

#ifdef SIGPIPE
static void
IgnorePipe()
{
    signal(SIGPIPE, IgnorePipe);
}
#endif

/*
 *----------------------------------------------------------------------
 *
 * xread --
 *
 *	This procedure reads a buffer from a file descriptor. This 
 *	wrapper is needed on broken SYS V machines to handle 
 *	interrupted system calls.
 *
 * Results:
 *	The number of bytes read.
 * 
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
xread(fd, buf, len)
     int fd;
     char *buf;
     int len;
{
    int rc;
    
    while ((rc = read(fd, buf, len)) < 0 
	   && (errno == EINTR || errno == EAGAIN)) {
	continue;
    }
    
    return rc;
}

/*
 *----------------------------------------------------------------------
 *
 * straps --
 *
 *	This procedure starts the trap forwarder daemon named
 *	straps.
 *
 * Results:
 *	A standard Tcl result.
 * 
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
straps(interp)
    Tcl_Interp *interp;
{
    int *pidArray, argc = 1;
    static char *argv[2] = { NULL, 0 };
    static Tcl_Channel channel = NULL;
    static char *straps = NULL;

    if (channel) {
	Tcl_Close((Tcl_Interp *) NULL, channel);
	channel = NULL;
    }

    if (! straps) {
        straps = getenv("TNM_STRAPS");
	if (! straps) {
	    straps = STRAPS;
	}
	straps = ckstrdup(straps);
    }
    argv[0] = straps;

#ifndef __WIN32__
    channel= Tcl_OpenCommandChannel(interp, argc, argv, 0);
#endif

    if (! channel) {
	if (straps) {
	    ckfree(straps);
	    straps = NULL;
	}
	return TCL_ERROR;
    }

    return TCL_OK;
}
#endif

/*
 *----------------------------------------------------------------------
 *
 * DumpPacket --
 *
 *	This procedure prints a hex dump of a packet. Useful for
 *	debugging this code. The message given in the third 
 *	parameter should be used to identify the received packet.
 *	The fourth parameter identifies the address and port of 
 *	the sender.
 *
 * Results:
 *	None.
 * 
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static void
DumpPacket(interp, packet, packetlen, msg, addr)
    Tcl_Interp *interp;
    u_char *packet;
    int packetlen;
    char *msg;
    struct sockaddr_in *addr;
{
    u_char *cp = packet;
    char buf[80];
    Tcl_DString dst;
    int	i = 0;

    Tcl_DStringInit(&dst);
    if (msg) {
	Tcl_DStringAppend(&dst, msg, -1);
	Tcl_DStringAppend(&dst, " ", 1);
    }
    sprintf(buf, "%3d bytes", packetlen);
    Tcl_DStringAppend(&dst, buf, -1);
    if (addr->sin_addr.s_addr) {
	sprintf(buf, " [%s:%u]", 
		inet_ntoa(addr->sin_addr), ntohs(addr->sin_port));
	Tcl_DStringAppend(&dst, buf, -1);
    }
    Tcl_DStringAppend(&dst, ":\n", 2);

    while (i++ < packetlen) {
	sprintf(buf, "%02x", *cp++);
	Tcl_DStringAppend(&dst, buf, -1);
	
	if (i++ < packetlen) {
	    sprintf(buf, "%02x ", *cp++);
	    Tcl_DStringAppend(&dst, buf, -1);
	}
	
	if ((i % 16) == 0 && i < packetlen) {
	    Tcl_DStringAppend(&dst, "\n", 1);
	}
    }
    Tcl_DStringAppend(&dst, "\n", 1);

    TnmWriteMessage(interp, Tcl_DStringValue(&dst));
    Tcl_DStringFree(&dst);
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_SnmpWait --
 *
 *	This procedure waits for a specified time for an answer. It 
 *	is used to implement synchronous operations.
 *
 * Results:
 *	1 if the socket is readable, otherwise 0.
 * 
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
Tnm_SnmpWait(ms)
    int ms;
{
    struct timeval wait;
    int width = sock + 1;
    fd_set readfds;
    
    wait.tv_sec  = ms / 1000;
    wait.tv_usec = (ms % 1000) * 1000;
    FD_ZERO (&readfds);
    FD_SET (sock, &readfds);

    return select(width, &readfds, (fd_set *) NULL, (fd_set *) NULL, &wait);
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_SnmpDelay --
 *
 *	This procedure enforces a small delay if the delay option
 *	of the session is set to value greater than 0.
 *
 * Results:
 *	None.
 * 
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

void
Tnm_SnmpDelay(session)
    SNMP_Session *session;
{
    static Tcl_Time lastTimeStamp;
    Tcl_Time currentTime;
    int delta, wtime;

    if (session->delay <= 0) return;

    TnmGetTime(&currentTime);

    if (lastTimeStamp.sec == 0 && lastTimeStamp.usec == 0) {
	lastTimeStamp = currentTime;
	return;
    }

    delta = (currentTime.sec - lastTimeStamp.sec) * 1000 
	    + (currentTime.usec - lastTimeStamp.usec) / 1000;
    wtime = session->delay - delta;

    if (wtime <= 0) {
	lastTimeStamp = currentTime;
    } else {
	struct timeval timeout;
	timeout.tv_usec = (wtime * 1000) % 1000000;
	timeout.tv_sec = (wtime * 1000) / 1000000;
	select(0, (fd_set *) NULL, (fd_set *) NULL, (fd_set *) NULL, &timeout);
	TnmGetTime(&lastTimeStamp);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_SnmpManagerOpen --
 *
 *	This procedure creates a socket used for normal management
 *	communication.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
Tnm_SnmpManagerOpen(interp)
    Tcl_Interp *interp;
{
    struct sockaddr_in name;
    int code;

    if (sock > 0) {
	return TCL_OK;
    }

    sock = TnmSocket(AF_INET, SOCK_DGRAM, 0);
    if (sock == TNM_SOCKET_ERROR) {
        Tcl_AppendResult(interp, "can not create socket: ", 
			 Tcl_PosixError(interp), (char *) NULL);
	return TCL_ERROR;
    }
    
    name.sin_family = AF_INET;
    name.sin_port = 0;
    name.sin_addr.s_addr = INADDR_ANY;

    code = TnmSocketBind(sock, (struct sockaddr *) &name, sizeof(name));
    if (code == TNM_SOCKET_ERROR) {
        Tcl_AppendResult(interp, "can not bind socket: ", 
			 Tcl_PosixError(interp), (char *) NULL);
	TnmSocketClose(sock);
	sock = -1;
	return TCL_ERROR;
    }
    
    mgrSocket = Tcl_GetFile((ClientData) sock, TNM_SOCKET_FD);
    Tcl_CreateFileHandler(mgrSocket, TCL_READABLE, 
			  ResponseProc, (ClientData *) interp);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_SnmpManagerClose --
 *
 *	This procedure closes the shared manager socket.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

void
Tnm_SnmpManagerClose()
{
    Tcl_DeleteFileHandler(mgrSocket);
    Tcl_FreeFile(mgrSocket);
    TnmSocketClose(sock);
    sock = -1;
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_SnmpTrapOpen --
 *
 *	This procedure creates a socket used to receive trap messages.
 *	Since traps are send to a privileged port, we start the straps
 *	trap multiplexer and connect to it via a UNIX domain socket.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
Tnm_SnmpTrapOpen(interp)
    Tcl_Interp *interp;
{
    int i, rc;
#ifdef __WIN32__
    struct sockaddr_in name;
    int code;
#else
    struct sockaddr_un saddr;
#endif
    int slen;

    trap_count++;

    if (trap_sock >= 0) {
	return TCL_OK;
    }

#ifdef __WIN32__
    trap_sock = TnmSocket(AF_INET, SOCK_DGRAM, 0);
    if (trap_sock == TNM_SOCKET_ERROR) {
        Tcl_AppendResult(interp, "can not create socket: ", 
			 Tcl_PosixError(interp), (char *) NULL);
	return TCL_ERROR;
    }
    
    name.sin_family = AF_INET;
    name.sin_port = 0;
    name.sin_addr.s_addr = INADDR_ANY;
    name.sin_port = htons(TNM_SNMP_TRAPPORT);

    code = TnmSocketBind(trap_sock, (struct sockaddr *) &name, sizeof(name));
    if (code == TNM_SOCKET_ERROR) {
        Tcl_AppendResult(interp, "can not bind socket: ", 
			 Tcl_PosixError(interp), (char *) NULL);
	TnmSocketClose(trap_sock);
	sock = -1;
	return TCL_ERROR;
    }
    
#else

    trap_sock = TnmSocket(AF_UNIX, SOCK_STREAM, 0);
    if (trap_sock == TNM_SOCKET_ERROR) {
	Tcl_AppendResult(interp, "can not create straps socket: ",
			 Tcl_PosixError(interp), (char *) NULL);
	return TCL_ERROR;
    }
    
    memset((char *) &saddr, 0, sizeof(saddr));
    
    saddr.sun_family = AF_UNIX;
    strcpy(saddr.sun_path, serv_path);
    slen = sizeof(saddr) - sizeof(saddr.sun_path) + strlen(saddr.sun_path);
    
    if (connect(trap_sock, (struct sockaddr *) &saddr, slen) < 0) {
	
	if (straps(interp) != TCL_OK) return TCL_ERROR;
	
	for (i = 0; i < 5; i++) {
	    sleep(1);
	    rc = connect(trap_sock, (struct sockaddr *) &saddr, slen);
	    if (rc >= 0) break;
	}
	
	if (rc < 0) {
	    Tcl_AppendResult(interp, "can not connect straps socket: ",
			     Tcl_PosixError(interp), (char *) NULL);
	    TnmSocketClose(trap_sock);
	    trap_sock = -1;    
	    return TCL_ERROR;
	}
    }

#ifdef SIGPIPE
    signal(SIGPIPE, IgnorePipe);
#endif
#endif

    trapSocket = Tcl_GetFile((ClientData) trap_sock, TNM_SOCKET_FD);
    Tcl_CreateFileHandler(trapSocket, TCL_READABLE, 
			  TrapProc, (ClientData *) interp);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_SnmpTrapClose --
 *
 *	This procedure closes the socket for incoming traps.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

void
Tnm_SnmpTrapClose()
{
    if (--trap_count == 0) {
	Tcl_DeleteFileHandler(trapSocket);
	Tcl_FreeFile(trapSocket);
	TnmSocketClose(trap_sock);
	trap_sock = -1;
	Tcl_ReapDetachedProcs();
    }
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_SnmpAgentOpen --
 *
 *	This procedure creates a socket for the agent part on a
 *	given port. If an agent socket is already created, we close
 *	the socket and open a new one.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
Tnm_SnmpAgentOpen(interp, session)
    Tcl_Interp *interp;
    SNMP_Session *session;
{
    struct sockaddr_in name;
    SNMP_Session *s;
    int code;
    
    if (session->agentSock > 0) {
	Tcl_DeleteFileHandler(session->agentSocket);
	Tcl_FreeFile(session->agentSocket);
	TnmSocketClose(session->agentSock);
    }

    name = session->maddr;
    name.sin_family = AF_INET;
    name.sin_addr.s_addr = INADDR_ANY;
    
    /*
     * Check if we can reuse a socket already opened by another
     * session. This allows to have more than one agent session
     * listening on a single transport endpoint.
     */

    for (s = sessionList; s && s->agentSock > 0; s = s->nextPtr) {
	struct sockaddr_in sname;
	int namelen = sizeof(sname);
	if (getsockname(s->agentSock, 
			(struct sockaddr *) &sname, &namelen) == -1) {
	    continue;
	}
	if (sname.sin_port == name.sin_port) {
	    session->agentSock = s->agentSock;
	    return TCL_OK;
	}
    }

    /*
     * Create a new socket for this transport endpoint and set up
     * a Tk filehandler to handle incoming messages.
     */

    session->agentSock = TnmSocket(AF_INET, SOCK_DGRAM, 0);
    if (session->agentSock == TNM_SOCKET_ERROR) {
        Tcl_AppendResult(interp, "can not create socket: ",
			 Tcl_PosixError(interp), (char *) NULL);
	session->agentSock = 0; 
        return TCL_ERROR;
    }

    code = TnmSocketBind(session->agentSock, (struct sockaddr *) &name, 
			 sizeof(name));
    if (code == TNM_SOCKET_ERROR) {
        Tcl_AppendResult(interp, "can not bind socket: ",
                         Tcl_PosixError(interp), (char *) NULL);
	TnmSocketClose(session->agentSock);
	session->agentSock = 0;
        return TCL_ERROR;
    }

    session->agentSocket = Tcl_GetFile((ClientData) session->agentSock,
				       TNM_SOCKET_FD);
    Tcl_CreateFileHandler(session->agentSocket, TCL_READABLE,
			  AgentProc, (ClientData *) session);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_SnmpAgentClose --
 *
 *	This procedure closes the socket for incoming requests.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

void
Tnm_SnmpAgentClose(session)
    SNMP_Session *session;
{
    struct sockaddr_in name;
    SNMP_Session *s;

    name = session->maddr;
    name.sin_family = AF_INET;
    name.sin_addr.s_addr = INADDR_ANY;

    /*
     * Check if the socket is still in use by some other sessions.
     */

    for (s = sessionList; s && s->agentSock > 0; s = s->nextPtr) {
	struct sockaddr_in sname;
	int namelen = sizeof(sname);
	if (getsockname(s->agentSock, 
			(struct sockaddr *) &sname, &namelen) == -1) {
	    continue;
	}
	if (s != session && sname.sin_port == name.sin_port) {
	    session->agentSock = 0;
	    return;
	}
    }

    /*
     * Still here? So we are the only one and should de-install the
     * file handler and close the socket.
     */

    Tcl_DeleteFileHandler(session->agentSocket);
    Tcl_FreeFile(session->agentSocket);
    TnmSocketClose(session->agentSock);
    session->agentSock = 0;
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_SnmpSend --
 *
 *	This procedure sends a packet to the destination address.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
Tnm_SnmpSend(interp, packet, packetlen, to)
    Tcl_Interp *interp;
    u_char *packet;
    int packetlen;
    struct sockaddr_in *to;
{
    int code;

    code = TnmSocketSendTo(sock, (char *) packet, packetlen, 0, 
			   (struct sockaddr *) to, sizeof(*to));
    if (code == TNM_SOCKET_ERROR) {
        Tcl_AppendResult(interp, "sendto failed: ", 
			 Tcl_PosixError(interp), (char *) NULL);
	return TCL_ERROR;
    }
    
    snmpStats.snmpOutPkts++;

#ifdef SNMP_BENCH
    TnmGetTime(&tnmSnmpBenchMark.sendTime);
    tnmSnmpBenchMark.sendSize = packetlen;
#endif

    if (hexdump) {
	DumpPacket(interp, packet, packetlen, "send", to);
    }
    
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_SnmpRecv --
 *
 *	This procedure reads incoming responses from the 
 *	manager socket.
 *
 * Results:
 *	A standard Tcl result. The data and the length of the
 *	packet is stored in packet and packetlen. The address of
 *	the sender is stored in from.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
Tnm_SnmpRecv(interp, packet, packetlen, from)
    Tcl_Interp *interp;
    u_char *packet;
    int	*packetlen;
    struct sockaddr_in *from;
{
    int	fromlen = sizeof(*from);

    *packetlen = TnmSocketRecvFrom(sock, (char *) packet, *packetlen, 0,
				   (struct sockaddr *) from, &fromlen);
    if (*packetlen == TNM_SOCKET_ERROR) {
	Tcl_AppendResult(interp, "recvfrom failed: ",
			 Tcl_PosixError(interp), (char *) NULL);
	return TCL_ERROR;
    }

#ifdef SNMP_BENCH
    TnmGetTime(&tnmSnmpBenchMark.recvTime);
    tnmSnmpBenchMark.recvSize = *packetlen;
#endif

    if (hexdump) {
	DumpPacket(interp, packet, *packetlen, "recv", from);
    }

    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * TrapRecv --
 *
 *	This procedure reads from the trap socket to process incoming
 *	trap messages.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
TrapRecv(interp, packet, packetlen, from)
    Tcl_Interp *interp;
    u_char *packet;
    int *packetlen;
    struct sockaddr_in *from;
{
#ifdef __WIN32__
    int	fromlen = sizeof(*from);

    *packetlen = TnmSocketRecvFrom(trap_sock, 
				   (char *) packet, *packetlen, 0,
				   (struct sockaddr *) from, &fromlen);
    if (*packetlen == TNM_SOCKET_ERROR) {
	Tcl_AppendResult(interp, "recvfrom failed: ",
			 Tcl_PosixError(interp), (char *) NULL);
	return TCL_ERROR;
    }

    if (hexdump) {
	DumpPacket(interp, packet, *packetlen, "recv", from);
    }

    return TCL_OK;
#else
    int len, rlen;
    char c;

    /*
     * Read the message from the straps daemon. We expect the 
     * following format:
     *
     * 4 byte - IP address of the remote agent.
     * 2 byte - port number of the remote agent.
     * 4 byte - length of the trap message.
     * n byte - trap message.
     */

    if (xread(trap_sock, (char *) &from->sin_addr.s_addr, 4) != 4) {
	goto errorExit;
    }
    if (xread(trap_sock, (char *) &from->sin_port, 2) != 2) {
	goto errorExit;
    }
    if (xread(trap_sock, (char *) &len, 4) != 4) {
	goto errorExit;
    }
    rlen = len < *packetlen ? len : *packetlen;
    if (xread(trap_sock, (char *) packet, rlen) <= 0) {
	goto errorExit;
    }

    /*
     * Eat up any remaining data-bytes.
     */

    while (len > *packetlen) {
        if (xread(trap_sock, &c, 1) != 1) {
	    goto errorExit;
	}
        len--;
    }

    *packetlen = rlen;

    if (hexdump) {
	DumpPacket(interp, packet, *packetlen, "recv", from);
    }

    /* 
     * Finally, make sure that the socket address belongs to the 
     * INET address family.
     */

    from->sin_family = AF_INET;
    return TCL_OK;

 errorExit:
    Tnm_SnmpTrapClose();
    Tcl_SetResult(interp, "lost connection to straps daemon", TCL_STATIC);
    return TCL_ERROR;
#endif
}

/*
 *----------------------------------------------------------------------
 *
 * AgentRecv --
 *
 *	This procedure reads from the socket to process incoming
 *	requests.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
AgentRecv(interp, session, packet, packetlen, from)
    Tcl_Interp	*interp;
    SNMP_Session *session;
    u_char *packet;
    int *packetlen;
    struct sockaddr_in *from;
{
    int	fromlen = sizeof(*from);

    *packetlen = TnmSocketRecvFrom(session->agentSock, 
				   (char *) packet, *packetlen, 0,
				   (struct sockaddr *) from, &fromlen);
    if (*packetlen == TNM_SOCKET_ERROR) {
	Tcl_AppendResult(interp, "recvfrom failed: ",
			 Tcl_PosixError(interp), (char *) NULL);
	return TCL_ERROR;
    }

    if (hexdump) {
	DumpPacket(interp, packet, *packetlen, "recv", from);
    }

    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_SnmpTimeoutProc --
 *
 *	This procedure is called from the event dispatcher whenever
 *	a timeout occurs so that we can retransmit packets.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

void
Tnm_SnmpTimeoutProc(clientData)
    ClientData clientData;
{
    Tnm_SnmpRequest *request = (Tnm_SnmpRequest *) clientData;
    SNMP_Session *session = request->session;
    Tcl_Interp *interp = request->interp;

    if (request->sends < (1 + session->retries)) {
	
	/* 
	 * Reinstall TimerHandler for this request and retransmit
	 * this request (keeping the original oid).
	 */
	
#ifdef TNM_SNMPv2U
	if (session->version == TNM_SNMPv2U && session->qos & USEC_QOS_AUTH) {
	    Tnm_SnmpUsecAuth(session, request->packet, request->packetlen);
	}
#endif
	Tnm_SnmpDelay(session);
	Tnm_SnmpSend(interp, request->packet, request->packetlen, 
		     &session->maddr);
#ifdef SNMP_BENCH
	if (request->stats.sendSize == 0) {
	    request->stats.sendSize = tnmSnmpBenchMark.sendSize;
	    request->stats.sendTime = tnmSnmpBenchMark.sendTime;
	}
#endif
        request->sends++;
	request->timer = Tcl_CreateTimerHandler(
			(session->timeout * 1000) / (session->retries + 1),
			Tnm_SnmpTimeoutProc, (ClientData *) request);

    } else {

	/*
	 * # of retransmissions reached: Evaluate the callback to
	 * notify the application and delete this request. We fake
	 * an empty pdu structure to conform to the callback 
	 * conventions.
	 */
    
	SNMP_PDU _pdu;
	SNMP_PDU *pdu = &_pdu;

	memset((char *) pdu, 0, sizeof(SNMP_PDU));
	pdu->request_id = request->id;
	pdu->error_status = TNM_SNMP_NORESPONSE;
	Tcl_DStringInit(&pdu->varbind);

	Tcl_Preserve((ClientData) request);
	Tcl_Preserve((ClientData) session);
	Tnm_SnmpDeleteRequest(request);
	if (request->proc) {
	    (request->proc) (session, pdu, request->clientData);
	}
	Tcl_Release((ClientData) session);
	Tcl_Release((ClientData) request);
	Tcl_ResetResult(interp);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * ResponseProc --
 *
 *	This procedure is called from the event dispatcher whenever
 *	a response to a management request is received.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static void
ResponseProc(clientData, mask)
    ClientData	clientData;
    int mask;
{
    Tcl_Interp *interp = (Tcl_Interp *) clientData;
    u_char packet[TNM_SNMP_MAXSIZE];
    int code, packetlen = TNM_SNMP_MAXSIZE;
    struct sockaddr_in from;

    Tcl_ResetResult(interp);
    code = Tnm_SnmpRecv(interp, packet, &packetlen, &from);
    if (code != TCL_OK) return;

    code = Tnm_SnmpDecode(interp, packet, packetlen, &from, NULL, NULL);
    if (code == TCL_ERROR) {
	Tcl_AddErrorInfo(interp, "\n    (snmp response event)");
	Tcl_BackgroundError(interp);
    }
    if (code == TCL_CONTINUE && hexdump) {
	TnmWriteMessage(interp, interp->result);
	TnmWriteMessage(interp, "\n");
    }
}

/*
 *----------------------------------------------------------------------
 *
 * TrapProc --
 *
 *	This procedure is called from the event dispatcher whenever
 *	a trap message is received.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static void
TrapProc(clientData, mask)
    ClientData clientData;
    int mask;
{
    Tcl_Interp *interp = (Tcl_Interp *) clientData;
    u_char packet[TNM_SNMP_MAXSIZE];
    int code, packetlen = TNM_SNMP_MAXSIZE;
    struct sockaddr_in from;

    Tcl_ResetResult(interp);
    code = TrapRecv(interp, packet, &packetlen, &from);
    if (code != TCL_OK) return;

    code = Tnm_SnmpDecode(interp, packet, packetlen, &from, NULL, NULL);
    if (code == TCL_ERROR) {
	Tcl_AddErrorInfo(interp, "\n    (snmp trap event)");
	Tcl_BackgroundError(interp);
    }
    if (code == TCL_CONTINUE && hexdump) {
	TnmWriteMessage(interp, interp->result);
	TnmWriteMessage(interp, "\n");
    }
}

/*
 *----------------------------------------------------------------------
 *
 * AgentProc --
 *
 *	This procedure is called from the event dispatcher whenever
 *	a request for the SNMP agent entity is received.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static void
AgentProc(clientData, mask)
    ClientData clientData;
    int mask;
{
    SNMP_Session *session = (SNMP_Session *) clientData;
    Tcl_Interp *interp = session->agentInterp;
    u_char packet[TNM_SNMP_MAXSIZE];
    int code, packetlen = TNM_SNMP_MAXSIZE;
    struct sockaddr_in from;

    if (! interp) return;

    Tcl_ResetResult(interp);
    code = AgentRecv(interp, session, packet, &packetlen, &from);
    if (code != TCL_OK) return;
    
    code = Tnm_SnmpDecode(interp, packet, packetlen, &from, NULL, NULL);
    if (code == TCL_ERROR) {
	Tcl_AddErrorInfo(interp, "\n    (snmp agent event)");
	Tcl_BackgroundError(interp);
    }
    if (code == TCL_CONTINUE && hexdump) {
	TnmWriteMessage(interp, interp->result);
	TnmWriteMessage(interp, "\n");
    }
}
