/*
 * tnmMibQuery.c --
 *
 *	Provides functions to search in the MIB tree.
 *
 * Copyright (c) 1994-1996 Technical University of Braunschweig.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#include "tnmSnmp.h"
#include "tnmMib.h"

/*
 * Global variables of the MIB extension:
 */

char *tnm_MibFileName = NULL;		/* Current MIB file name loaded.   */
char *tnm_MibModuleName = NULL;		/* Current MIB module name loaded. */
Tnm_MibNode *tnm_MibTree = NULL;	/* The root of the MIB tree.	   */
Tnm_MibTC *tnm_MibTCList = NULL;	/* List of textual conventions.	   */
Tnm_MibTC *tnm_MibTCSaveMark = NULL;	/* The first already saved	   */
					/* element of mib_TCList	   */

TnmTable tnmSnmpMibAccessTable[] = {
    { TNM_MIB_NOACCESS,   "not-accessible" },
    { TNM_MIB_READONLY,   "read-only" },
    { TNM_MIB_READCREATE, "read-create" },
    { TNM_MIB_READWRITE,  "read-write" },
    { TNM_MIB_WRITEONLY,  "write-only" },
    { TNM_MIB_FORNOTIFY,  "accessible-for-notify" },
    { 0, NULL }
};

TnmTable tnmSnmpMibMacroTable[] = {
    { TNM_MIB_OBJECTTYPE,        "OBJECT-TYPE" },
    { TNM_MIB_OBJECTIDENTITY,    "OBJECT-IDENTITY" },
    { TNM_MIB_MODULEIDENTITY,    "MODULE-IDENTITY" },
    { TNM_MIB_NOTIFICATIONTYPE,  "NOTIFICATION-TYPE" },
    { TNM_MIB_TRAPTYPE,          "TRAP-TYPE" },
    { TNM_MIB_OBJECTGROUP,       "OBJECT-GROUP" },
    { TNM_MIB_NOTIFICATIONGROUP, "NOTIFICATION-GROUP" },
    { TNM_MIB_COMPLIANCE,        "MODULE-COMPLIANCE" },
    { TNM_MIB_CAPABILITIES,      "AGENT-CAPABILITIES" },
    { 0, NULL }
};

/*
 * A private buffer that is used to assemble object identifier in 
 * dottet notation.
 */

static char oidBuffer[TNM_OIDMAXLEN * 8];

/*
 * Forward declarations for procedures defined later in this file:
 */

static char *
FormatOctetTC		_ANSI_ARGS_((char *val, char *fmt));

static char *
FormatIntTC		_ANSI_ARGS_((char *val, char *fmt));

static char *
FormatTimeTicks		_ANSI_ARGS_((char *val));

static char *
FormatOID		_ANSI_ARGS_((char *val));

static char *
ScanOctetTC		_ANSI_ARGS_((char *val, char *fmt));

static char *
ScanIntTC		_ANSI_ARGS_((char *val, char *fmt));

static char *
ScanTimeTicks		_ANSI_ARGS_((char *val));

static void
GetMibPath		_ANSI_ARGS_((Tnm_MibNode *nodePtr, char *soid));

static void
FormatUnsigned		_ANSI_ARGS_((unsigned u, char *s));


/*
 *----------------------------------------------------------------------
 *
 * FormatUnsigned --
 *
 *	This procedure formats the unsigned value in u into an 
 *	unsigned ascii string s. We avoid sprintf because this is 
 *	too slow.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static void
FormatUnsigned(u, s)
    u_int u;
    char *s;
{
    if (u < 10) {
	*s++ = '0' + u;
    } else {
	u_int t=10;
	char c = '0'+ (u % 10);
	u /= 10;
	while (u / t) t *= 10;
	while (t /= 10) *s++ = '0'+ (u / t) % 10;
	*s++ = c;
    }
    *s = '\0';
}

/*
 *----------------------------------------------------------------------
 *
 * GetMibPath --
 *
 *	This procedure writes the path to the given by nodePtr into
 *	the given character string. This is done recursively using the 
 *	pointers to the parent node.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static void
GetMibPath(nodePtr, s) 
    Tnm_MibNode *nodePtr;
    char *s;
{
    if (! nodePtr) return;
    if (nodePtr->parentPtr) {
	GetMibPath(nodePtr->parentPtr, s);
	while (*s) s++;
	*s++ = '.';
    }
    FormatUnsigned(nodePtr->subid, s);
}

/*
 *----------------------------------------------------------------------
 *
 * GetMibPath2 --
 *
 *	This procedure writes the path to the given by nodePtr into
 *	the given oid array. This is done recursively using the 
 *	pointers to the parent node.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static void
GetMibPath2(nodePtr, oid, oidLen) 
    Tnm_MibNode *nodePtr;
    Tnm_Oid *oid;
    int *oidLen;
{
    if (! nodePtr) return;
    if (nodePtr->parentPtr) {
	GetMibPath2(nodePtr->parentPtr, oid, oidLen);
    }
    oid[*oidLen] = nodePtr->subid;
    (*oidLen)++;
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_MibNodeGetOid --
 *
 *	This procedure retrieves the oid of the node given by nodePtr.
 *
 * Results:
 *	The length of the object identifier.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
Tnm_MibNodeGetOid(nodePtr, oid)
    Tnm_MibNode *nodePtr;
    Tnm_Oid *oid;
{
    int len = 0;
    GetMibPath2(nodePtr, oid, &len);
    return len;
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_MibGetOid --
 *
 *	This procedure returns the oid that belongs to label. The exact 
 *	switch turns exact matching on, which only allows object 
 *	identifier which exactly match an object type definition.
 *
 * Results:
 *	A pointer to a static string containing the object identifier
 *	in dotted notation or NULL if there is no such MIB node.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

char*
Tnm_MibGetOid(label, exact)
    char *label;
    int exact;
{
    char *expanded = Tnm_HexToOid(label);
    Tnm_MibNode *nodePtr;
    int offset = -1;

    if (expanded) label = expanded;
    nodePtr = Tnm_MibFindNode(label, &offset, exact);
    if (nodePtr) {
	if (Tnm_IsOid(label)) {
	    return label;
	}
	GetMibPath(nodePtr, oidBuffer);
	if (offset > 0) {
	    strcat(oidBuffer, label+offset);
	}
	return oidBuffer;
    }

    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_MibGetName --
 *
 *	This procedure returns the name that belongs to a descriptor.
 *	This is the reverse operation to Tnm_MibGetOid().
 *
 * Results:
 *	A pointer to a static string containing the object descriptor
 *	or NULL if there is no such MIB node.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

char*
Tnm_MibGetName(label, exact)
    char *label;
    int exact;
{
    char *expanded = Tnm_HexToOid(label);
    Tnm_MibNode *nodePtr;
    int offset = -1;
    
    if (expanded) label = expanded;
    nodePtr = Tnm_MibFindNode(label, &offset, exact);
    if (nodePtr) {
	if (offset > 0) {
	    strcpy(oidBuffer, nodePtr->label);
	    strcat(oidBuffer, label+offset);
	    return oidBuffer;
	} else {
	    return nodePtr->label;
	}
    }

    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_MibGetDescription --
 *
 *	This procedure extracts the description of a MIB object.
 *	White spaces following newline characters are removed so 
 *	that the resulting text looks like the text in the MIB 
 *	definition. Note however, that all leading white spaces 
 *	are removed which may cause problems in some rare situations.
 *
 * Results:
 *	A pointer to a static string containing the object description
 *	or NULL if there is no such MIB node.
 *
 * Side effects:
 *	The MIB file is read and which might cause unexpected results
 *	if the file is no longer available.
 *
 *----------------------------------------------------------------------
 */

char*
Tnm_MibGetDescription(name, exact)
    char *name;
    int exact;
{
    FILE *fp;
    int	ch;
    char line[81];
    Tnm_MibNode *nodePtr = Tnm_MibFindNode(name, NULL, exact);
    static Tcl_DString *result = NULL;
    
    if (result == NULL) {
	result = (Tcl_DString *) ckalloc(sizeof(Tcl_DString));
	Tcl_DStringInit(result);
    } else {
	Tcl_DStringFree(result);
    }
    
    line[0] = '\0';
    if (nodePtr) {
	int len;
	if (nodePtr->fileOffset > 0 && nodePtr->fileName != NULL) {
	    fp = fopen(nodePtr->fileName, "r");
	    if (fp == NULL) {
		perror(nodePtr->fileName);
		return "";
	    }
	    if (fseek(fp, nodePtr->fileOffset, 0) < 0) {
                perror(nodePtr->fileName);
                return "";
	    }

	    while ((ch = getc(fp)) != EOF) {
		if (ch == '"') break;
	    }

	    len = 0;
            line[0] = '\0';
	    while ((ch = getc(fp)) != EOF) {
		if (ch == '"') break;
		line[len++] = ch;
		if (ch == '\n' || len == 80) {
		    line[len] = '\0';
		    Tcl_DStringAppend(result, line, len);
		    len = 0;
		    if (ch == '\n') {
			while ((ch = getc(fp)) != EOF) {
			    if (!isspace(ch)) break;
			}
			if (ch == EOF || ch == '"') break;
			line[len++] = ch;
		    }
		}
	    }
	    if (len != 0) {
		line[len] = '\0';
		Tcl_DStringAppend(result, line, len);
	    }
	    
	    fclose(fp);
	    return Tcl_DStringValue(result);
	} else {
	    return "";
	}
    }
    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_MibGetSucc --
 *
 *	This procedure returns the labels or object identifiers of 
 *	all direct successors of a MIB node.
 *
 * Results:
 *	The pointer to a static string containing the list of successors 
 *	or NULL if there is no such MIB node.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

char*
Tnm_MibGetSucc(name)
    char *name;
{
    Tnm_MibNode *nodePtr;
    int retoid;
    static Tcl_DString *result = NULL;
    char buf[20];
    char *expanded = Tnm_HexToOid(name);

    if (expanded) name = expanded;
    nodePtr = Tnm_MibFindNode(name, NULL, 1);
    retoid = Tnm_IsOid(name);
    
    if (result == NULL) {
	result = (Tcl_DString *) ckalloc(sizeof(Tcl_DString));
	Tcl_DStringInit(result);
    } else {
	Tcl_DStringFree(result);
    }
    
    if (nodePtr) {
	if (nodePtr->childPtr) {
	    for (nodePtr = nodePtr->childPtr; 
		 nodePtr; 
		 nodePtr = nodePtr->nextPtr) {
		if (retoid) {
		    Tcl_DStringAppendElement(result, name);
		    buf[0] = '.';
		    FormatUnsigned(nodePtr->subid, buf+1);
		    Tcl_DStringAppend(result, buf, -1);
		} else {
		    Tcl_DStringAppendElement(result, nodePtr->label);
		}
	    }
	}
	return Tcl_DStringValue(result);
    }

    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_MibGetSyntax --
 *
 *	This procedure returns the syntax of an object type or an 
 *	instance as defined in the OBJECT-TYPE macro.
 *
 * Results:
 *	The pointer to a static string containing the syntax name 
 *	or NULL if there is no such MIB node.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

char*
Tnm_MibGetSyntax(name, exact)
    char *name;
    int exact;
{
    Tnm_MibNode *nodePtr = Tnm_MibFindNode(name, NULL, exact);
    char *syntax;

    if (nodePtr) {
	if (nodePtr->macro != TNM_MIB_OBJECTTYPE) {
	    return "";
	}
	if (nodePtr->tc && nodePtr->tc->name) {
	    if (*nodePtr->tc->name == '_') {
		syntax = TnmGetTableValue(tnmSnmpTypeTable, 
					  nodePtr->tc->syntax);
		return syntax ? syntax : "";
	    } else {
		return nodePtr->tc->name;
	    }
	}
	syntax = TnmGetTableValue(tnmSnmpTypeTable, nodePtr->syntax);
	return syntax ? syntax : "";
    }

    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_MibGetBaseSyntax --
 *
 *	This procedure returns the transfer syntax actually used to 
 *	encode a value. This may be different from the syntax defined
 *	in the OBJECT-TYPE macro as textual conventions are based on 
 *	a set of primitive types. Note: We have sometimes more than 
 *	one MIB syntax tag for an ASN.1 type, so we must make sure to 
 *	find the right syntax tag.
 *
 * Results:
 *	The ASN.1 transfer syntax or ASN1_OTHER if the lookup
 *	fails for some reason.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
Tnm_MibGetBaseSyntax(name, exact)
    char *name;
    int exact;
{
    int syntax = ASN1_OTHER;
    Tnm_MibNode *nodePtr = Tnm_MibFindNode(name, NULL, exact);

    if (nodePtr) {
	if (nodePtr->tc && nodePtr->tc->name) {
	    syntax = nodePtr->tc->syntax;
	} else {
	    syntax = nodePtr->syntax;
	}
#if 0
	switch (syntax) {
	  case Integer32: syntax = ASN1_INTEGER; break;
	  case Counter:	  syntax = ASN1_COUNTER32; break;
	  case Gauge:	  syntax = ASN1_GAUGE32; break;
	}
#endif
    }

    return syntax;
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_MibGetAccess --
 *
 *	This procedure returns the max access field of the OBJECT-TYPE
 *	macro used to define a MIB object.
 *
 * Results:
 *	The pointer to a static string containing the max access 
 *	string or NULL if there is no such MIB node.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

char*
Tnm_MibGetAccess(name, exact)
    char *name;
    int exact;
{
    Tnm_MibNode *nodePtr = Tnm_MibFindNode(name, NULL, exact);

    if (nodePtr) {
	char *value = TnmGetTableValue(tnmSnmpMibAccessTable, nodePtr->access);
	return value ? value : "unknown";
    }
    
    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_MibGetMacro --
 *
 *	This procedure returns the name of the ASN.1 macro that was 
 *	used to define this particular MIB node.
 *
 * Results:
 *	The pointer to a static string containing the macro name
 *	or NULL if there is no such MIB node.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

char*
Tnm_MibGetMacro(name, exact)
    char *name;
    int exact;
{
    Tnm_MibNode *nodePtr = Tnm_MibFindNode(name, NULL, exact);

    if (nodePtr) {
	char *macro = TnmGetTableValue(tnmSnmpMibMacroTable, nodePtr->macro);
	return macro ? macro : "";
    }
    
    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_MibGetModule --
 *
 *	This procedure returns the name of the module which contains
 *	the definition of a MIB node.
 *
 * Results:
 *	The pointer to a static string containing the module name
 *	or NULL if there is no such MIB node.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

char*
Tnm_MibGetModule(name, exact)
    char *name;
    int exact;
{
    Tnm_MibNode *nodePtr = Tnm_MibFindNode(name, NULL, exact);

    if (nodePtr) {
	return nodePtr->moduleName ? nodePtr->moduleName : "";
    }
    
    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * FormatOctetTC --
 *
 *	This procedure formats the octet string value according to the 
 *	textual convention stored in fmt. Parts marked XXX are untested.
 *
 * Results:
 *	The pointer to a static string containing the formatted value
 *	or NULL if there is no such MIB node.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static char *
FormatOctetTC(val, fmt)
    char *val;
    char *fmt;
{
    int pfx, have_pfx;			/* counter prefix */
    char *last_fmt;			/* save ptr to last seen fmt */
    static char *ret = NULL;
    static int ret_len = 0;
    int idx;

    /*
     * Sanity check and the simple case:
     */

    if (! fmt) {
        return val;
    }

#if 0
    if (val[0] == '0' && val[1] == 'x') {
	val += 2;
    }
#endif

    if (strcmp(fmt, "1x:") == 0) {
        return val;
    }

    if (ret == NULL) {
        ret = ckalloc(ret_len = 100);
    }
    idx = 0;			/* idx into ret buffer */

    while (*fmt && *val) {

        last_fmt = fmt;		/* save for loops */

	/* scan prefix: */
	have_pfx = pfx = 0;
	while (*fmt && isdigit(*fmt)) {
	    pfx = pfx * 10 + *fmt - '0', have_pfx = 1, fmt++;
	}
	if (! have_pfx) {
	    pfx = 1;
	}

	/* scan format: */
	if (*fmt == 'a') {

	    while (*val && pfx > 0) {
	        char c = *val++ & 0xff;
		int v = c >= 'a' ?  c - 87 : (c >= 'A' ? c - 55 : c - 48);
		if (! *val) {
		    break;
		}
		c = *val++ & 0xff;
		v = (v << 4) + (c >= 'a' ?  c - 87 :
				(c >= 'A' ? c - 55 : c - 48));
		if (idx + 100 >= ret_len) {
		    ret = ckrealloc(ret, ret_len += 100);
		}
		ret [idx++] = v;
		pfx--;
		if (*val == ':') {
		    val++;
		}
	    }
	    fmt++;

	} else if (*fmt == 'd' || *fmt == 'o' || *fmt == 'b') {

	    char format = *fmt;
	    int c, vv = 0, v;

	    fmt++;

	    /* collect octets and format: */
	    while (pfx > 0) {
	        if (! *val) {
		    break;
		}
		/* collect next byte from octal buffer and move to vv: */
		c = *val++ & 0xff;
		v = c >= 'a' ?  c - 87 : 
		  (c >= 'A' ? c - 55 : c - 48);
		if (! *val) {
		    break;
		}
		c = *val++ & 0xff;
		v = (v << 4) + (c >= 'a' ?  c - 87 :
				(c >= 'A' ? c - 55 : c - 48));
		vv = vv * 256 + v;
		
		if (*val == ':') {
		    val++;
		}

		pfx--;
	    }
		
	    if (idx + 100 >= ret_len) {
	        ret = ckrealloc(ret, ret_len += 100);
	    }
	    if (format == 'd') {
	        sprintf(ret + idx, "%d", vv);
	        idx += strlen(ret + idx);
	    } else if (format == 'o') {
	        /* XXX: completely untested */
	        sprintf(ret + idx, "%o", vv);
		idx += strlen(ret + idx);
	    } else if (format == 'b') {
	        /* XXX: completely untested */
	        int i; 
		for (i = (sizeof(int) * 8 - 1); i >= 0
		     && ! (vv & (1 << i)); i--);
		for (; i >= 0; i--)
		  ret [idx++] = vv & (1 << i) ? '1' : '0';
	    }
	} else if (*fmt == 'x') {

	    /* 
	     * A simple one: copy the following hex-digits:
	     */

	    while (pfx > 0) {		
	        if (! *val || ! val [1]) {
		    break;
		}
	        if (idx + 100 >= ret_len) {
		    ret = ckrealloc(ret, ret_len += 100);
	        }
		ret [idx++] = *val++;
		ret [idx++] = *val++;
		if (*val == ':') {
		    val++;
		}
		pfx--;
	    }
	    fmt++;
	} else {
	    fprintf(stderr, "scotty: unknown textual-format `%c'\n", *fmt);
	    fmt++;
	    break;
	}

	/*
	 * Look about a separator:
	 */

	if (*fmt && ! isdigit(*fmt) && *fmt != '*') {
	    if (have_pfx && *val) {
	        if (idx + 100 >= ret_len) {
		    ret = ckrealloc(ret, ret_len += 100);
		}
		ret [idx++] = *fmt;
	    }
	    fmt++;
	}

	/*
	 * Repeat with last format, if data is still available:
	 */

	if (! *fmt && *val) {
	    fmt = last_fmt;
	}
    }

    ret [idx] = 0;

    return ret;
}

/*
 *----------------------------------------------------------------------
 *
 * FormatIntTC --
 *
 *	This procedure formats the integer value according to the 
 *	textual convention stored in fmt. Parts marked XXX are 
 *	untested.
 *
 * Results:
 *	The pointer to a static string containing the formatted value
 *	or NULL if there is no such MIB node.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static char *
FormatIntTC(val, fmt)
    char *val;
    char *fmt;
{
    static char *ret = NULL;
    static int ret_len = 0;
    int i, idx, val_len;
    int dpt = -1;				/* decimal point */
    char format;

    /* 
     * Sanity check and the simple case:
     */

    if (! fmt) {
        return NULL;
    } else {
        format = *fmt;
    }

    if (*fmt == 'd' && ! fmt [1]) {
        return val;
    }

    /* 
     * Must be one of these: 
     */

    if (*fmt != 'd' && *fmt != 'x' && *fmt != 'o' && *fmt != 'b') {
        return NULL;
    }
    
    if (ret == NULL) {
        ret = ckalloc(ret_len = 100);
    }
    idx = 0;				/* idx into ret buffer */
	  
    /*
     * `d' format may have the form: d-[0-9]+ 
     */

    if (*fmt == 'd' && fmt [1] == '-' 
	&& fmt [2] >= '0' && fmt [2] <= '9') {
        format = *fmt;
	dpt = 0;
	for (fmt += 2; *fmt >= '0' && *fmt <= '9'; fmt++) {
	    dpt = dpt * 10 + *fmt - '0';
	}
    } else if (fmt [1]) {
        return NULL;				/* invalid */
    }

    /*
     * Check integer value;
     */

    for (val_len = 0; val [val_len]; val_len++) {
        if (! ((! val_len && val [val_len] == '-') || isdigit(val[val_len]))) {
	    return NULL;
	}
    }
    
    if (dpt >= 0) {

        /*
         * Now we have to format val_len digits with 
	 * decimal-point at dpt:
	 */

        if (dpt + val_len + 2 >= ret_len) {	/* don't care */
	     ret = ckrealloc(ret, ret_len = dpt + val_len + 2);
	}
	
	if (format == 'd') {
	    /* monadic - always first: */
	    if (*val == '-') {
	        ret [idx++] = '-', val++, val_len--;
	    }
	    if (dpt >= val_len) {
	        ret [idx++] = '0', ret [idx++] = '.';
		for (i = 0; i < dpt - val_len; i++) {
		    ret [idx++] = '0';
		}
		strcpy(ret + idx, val);
		return ret;
	    }
	    for (i = 0; i < val_len - dpt; i++) {
	        ret [idx++] = val [i];
	    }
	    ret [idx++] = '.';
	    for (; i < val_len; i++) {
	        ret [idx++] = val [i];
	    }
	} else if (format == 'o') {
	    /* XXX: completely untested */
	    sprintf(ret, "%o", atoi(val));
	    return ret;
	} else if (format == 'x') {
	    /* XXX: completely untested */
	    sprintf(ret, "%x", atoi(val));
	    return ret;
        } else if (format == 'b') {
	    int vv = atoi(val);
	    /* XXX: completely untested */
	    for (i = (sizeof(int) * 8 - 1); i > 0
		 && ! (vv & (1 << i)); i--);
	    for (; i >= 0; i--) {
	        ret [idx++] = vv & (1 << i) ? '1' : '0';
	    }
	}
    } else {
        /* error: */
	return val;
    }
    
    ret [idx] = 0;
    return ret;
}

/*
 *----------------------------------------------------------------------
 *
 * FormatTimeTicks --
 *
 *	This procedure formats an unsigned number representing a
 *	TimeTicks value into a readable string representation.
 *
 * Results:
 *	The pointer to a static string containing the formatted value
 *	or NULL if there is no such MIB node.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static char *
FormatTimeTicks(value)
    char *value;
{
    u_int d, h, m, s, f;
    static char buf[80];

    d = 0;
    while (isdigit(*value)) {
	d = 10 * d + *value - '0';
	value++;
    }
    f = d % 100; d = d /100;
    s = d % 60;  d = d / 60;
    m = d % 60;  d = d / 60;
    h = d % 24;  d = d / 24;
    sprintf(buf, "%dd %2d:%02d:%02d.%02d", d, h, m, s, f);
    return buf;
}

/*
 *----------------------------------------------------------------------
 *
 * FormatOID --
 *
 *	This procedure formats an object identifier into a readable
 *	and unique string representation.
 *
 * Results:
 *	The pointer to a static string containing the formatted value
 *	or NULL if there is no such MIB node.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static char *
FormatOID(value)
    char *value;
{
    Tnm_MibNode *nodePtr = Tnm_MibFindNode(value, NULL, 1);
    static char *buffer = NULL;
    int len;

    if (! nodePtr || nodePtr->moduleName == NULL || nodePtr->label == NULL) {
	return value;
    }

    len = strlen(nodePtr->moduleName) + strlen(nodePtr->label) + 2;
    if (buffer) {
	buffer = ckrealloc(buffer, len);
    } else {
	buffer = ckalloc(len);
    }
    strcpy(buffer, nodePtr->moduleName);
    strcat(buffer, "!");
    strcat(buffer, nodePtr->label);
    return buffer;
}

/*
 *----------------------------------------------------------------------
 *
 * ScanOctetTC --
 *
 *	This procedure scans a string unsing the textual convention and
 *	returns the octet string representation of the value. Parts 
 *	marked XXX are untested.
 *
 * Results:
 *	The pointer to a static string containing the formatted value
 *	or NULL if there is no such MIB node.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static char *
ScanOctetTC(val, fmt)
    char *val;
    char *fmt;
{
    int pfx, have_pfx;			/* counter prefix */
    char *last_fmt;			/* save ptr to last seen fmt */
    static char *ret = NULL;
    static int ret_len = 0;
    int idx;

    /*
     * Sanity check and the simple case:
     */

    if (! fmt) {
        return val;
    }

    if (! ret) {
        ret = ckalloc(ret_len = 100);
    }

    *ret = '\0';
    idx = 0;			/* idx into ret buffer */

    if (strcmp(fmt, "1x:") == 0) {
	return val;
    }
    
    /* quick ``255a'' formatting: */

    if (strcmp(fmt, "255a") == 0) {
	while (*val) {
	    if (idx + 100 >= ret_len) {
		ret = ckrealloc(ret, ret_len += 100);
	    }
	    sprintf(ret + idx, "%02x", *val & 0xff);
	    idx += 2;
	    if (*++val) {
		ret [idx++] = ':';
	    }
	}
	return ret;
    }


    /* and now to something different: */

    while (*fmt && *val) {
	last_fmt = fmt;	/* save for loops */
	/* scan prefix: */
	have_pfx = pfx = 0;
	while (*fmt && isdigit(*fmt)) {
	    pfx = pfx * 10 + *fmt - '0', have_pfx = 1, fmt++;
	}
	if (! have_pfx) {
	    pfx = 1;
	}
	
	/* scan format: */
	if (*fmt == 'a') {
	    while (*val && pfx > 0) {
		char c = *val;
		int c1 = (c & 0xf0) >> 4;
		int c2 = c & 0x0f;
		if ((c1 += '0') > '9') c1 += 7;
		if ((c2 += '0') > '9') c2 += 7;
		
		if (idx + 100 >= ret_len) {
		    ret = ckrealloc(ret, ret_len += 100);
		}
		
		ret [idx++] = c1, ret [idx++] = c2;
		if (*++val) {
		    ret [idx++] = ':';
		}
		pfx--;
	    }
	    fmt++ ;
	} else if (*fmt == 'd' || *fmt == 'o' || *fmt == 'b') {
	    int vv = 0, v, got_val = 0;
	    
	    if (*fmt == 'd' && 1 == sscanf(val, "%d", &vv)) {
		got_val = 1;
		while (isdigit(*val)) {
		    val++;
		}
	    } else if (*fmt == 'o' && 1 == sscanf(val, "%o", &vv)) {
		/* XXX: completely untested */
		got_val = 1;
		while (*val >= '0' && *val <= '7') {
		    val++;
		}
	    } else if (*fmt == 'b') {
		/* XXX: completely untested */
		while (*val == '0' || *val == '1') {
		    got_val = 1;
		    vv = (vv << 1) | (*val - '0');
		    val++;
		}
	    }
	    
	    /* XXX: tell more about unscanable strings !!! */
	    if (! got_val) {
		break;
	    }
	    
	    /* now put pfx-num bytes to the output buffer: */
	    while (pfx > 0) {
		int c1, c2;
		
		v = vv >> ((pfx - 1) * 8);
		
		if (idx + 100 >= ret_len) {
		    ret = ckrealloc(ret, ret_len += 100);
		}
		
		c1 = (v & 0xf0) >> 4;
		c2 = v & 0x0f;
		if ((c1 += '0') > '9') c1 += 7;
		if ((c2 += '0') > '9') c2 += 7;
		
		ret [idx++] = c1, ret [idx++] = c2;
		if (*val) {
		    ret [idx++] = ':';
		}
		pfx--;
	    }
	    
	    fmt++;
	} else if (*fmt == 'x') {
	    /* a simple one: copy the following hex-digits: */
	    while (pfx > 0) {		
		if (! *val || ! val [1]) {
		    break;
		}
		if (idx + 100 >= ret_len) {
		    ret = ckrealloc(ret, ret_len += 100);
		}
		ret [idx++] = *val++;
		ret [idx++] = *val++;
		if (*val == ':') {
		    val++, ret [idx++] = ':';
		}
		pfx--;
	    }
	    fmt++;
	} else {
	    fprintf(stderr, "scotty: unknown textual-format `%c'\n",
		    *fmt); 
	    fmt++;
	    break;
	}
	
	/* look about a separator: */
	if (*fmt && ! isdigit(*fmt) && *fmt != '*') {
	    if (have_pfx && *val) {
		val++;
	    }
	    fmt++;
	}
	
	/* repeat with last format, if datas avail: */
	if (! *fmt && *val) {
	    fmt = last_fmt;
	}
    }
    
    ret [idx] = 0;
    return ret;
}

/*
 *----------------------------------------------------------------------
 *
 * ScanIntTC --
 *
 *	This procedure scans a string unsing the textual convention and
 *	returns the integer representation of the value. Parts marked 
 *	XXX are untested.
 *
 * Results:
 *	The pointer to a static string containing the formatted value
 *	or NULL if there is no such MIB node.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static char *
ScanIntTC(val, fmt)
    char *val;
    char *fmt;
{
    static char ret [100];
    int dpt = -1;				/* decimal point */
    char format;

    /* 
     * Sanity check and simple case: 
     */

    if (! fmt) {
        return NULL;
    } else {
        format = *fmt;
    }

    if (*fmt == 'd' && ! fmt [1]) {
        return val;
    }

    /* 
     * Must be one of these:
     */

    if (*fmt != 'd' && *fmt != 'o' && *fmt != 'b') {
        return NULL;
    }
    
    /* `d' format may have the form: d-[0-9]+ */
    if (*fmt == 'd' && fmt [1] == '-' 
	&& fmt [2] >= '0' && fmt [2] < '9') {
        format = *fmt;
	dpt = 0;
	for (fmt += 2; *fmt >= '0' && *fmt <= '9'; fmt++) {
	    dpt = dpt * 10 + *fmt - '0';
	}
    } else if (fmt [1]) {
        return NULL;					/* wrong */
    }

    if (dpt >= 0) {

        int idx = 0, do_copy = 0;

	for (; *val; val++) {
	    if (*val == '.') {
	        continue;
	    } else if (*val != '0' || do_copy) {
	        ret [idx++] = *val;
	    } else if (*val != '0' && ! do_copy) {
	        do_copy = 1;
	    }
	}

	ret [idx] = 0;
	return ret;

    } else if (format == 'o') {

        int vv;
	if (1 == sscanf(val, "%o", &vv)) {
	    /* XXX: completely untested */
	    sprintf(ret, "%d", vv);
	    return ret;
	}
    } else if (format == 'b') {

        int vv = 0, got_val = 0;
	
	/* XXX: completely untested */
	while (*val == '0' || *val == '1') {
	    got_val = 1;
	    vv = (vv << 1) | (*val - '0');
	    val++;
	}
	  
	if (got_val) {
	    sprintf(ret, "%d", vv);
	    return ret;
	}
    }

    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * ScanTimeTicks --
 *
 *	This procedure scans a string representing a TimeTicks value 
 *	returns the unsigned number in deci-seconds.
 *
 * Results:
 *	The pointer to a static string containing the formatted value
 *	or NULL if there is no such MIB node.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static char *
ScanTimeTicks(value)
    char *value;
{
    u_int u, d, h, m, s, f, n;
    static char str[20];

    n = sscanf(value, "%dd %d:%d:%d.%d", &d, &h, &m, &s, &f);
    if (n == 5) {
	u = d * 8640000 + h * 360000 + m * 6000 + s * 100 + f;
	goto done;
    }
    n = sscanf(value, "%d:%d:%d.%d", &h, &m, &s, &f);
    if (n == 4) {
	u = h * 360000 + m * 6000 + s * 100 + f;
	goto done;
    }
    n = sscanf(value, "%d:%d:%d", &h, &m, &s);
    if (n == 3) {
	u = h * 360000 + m * 6000 + s * 100;
	goto done;
    }

    u = 0;
    while (isdigit(*value)) {
	u = 10 * u + *value - '0';
	value++;
    }

done:
    FormatUnsigned(u, str);
    return str;
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_MibFormat --
 *
 *	This procedure converts a value from its base representation
 *	into the format defined by textual conventions or enumerations.
 *
 * Results:
 *	The pointer to a static string containing the formatted value
 *	or NULL if there is no such MIB node.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

char *
Tnm_MibFormat(name, exact, value)
    char *name;
    int exact;
    char *value;
{
    Tnm_MibNode *nodePtr = Tnm_MibFindNode(name, NULL, exact);
   
    if (nodePtr) {
	if (nodePtr->tc) {

	    /*
	     * Check if we have an enumeration for this value.
	     */

	    Tnm_MibEnum *ePtr;
	    int ival = atoi(value);
	    for (ePtr = nodePtr->tc->enumList; ePtr; ePtr = ePtr->nextPtr) {
		if (ePtr->value == ival) {
		    return ePtr->label;
		}
	    }

	    /* 
	     * Check if we can apply a display hint.
	     */

	    if (nodePtr->tc->displayHint) {
		char *ret = NULL;
		if (nodePtr->syntax == ASN1_OCTET_STRING) {
		    ret = FormatOctetTC(value, nodePtr->tc->displayHint);
		} else if (nodePtr->syntax == ASN1_INTEGER) {
                    ret = FormatIntTC(value, nodePtr->tc->displayHint);
                }
		if (ret) {
		    return ret;
		}
	    }
	}
	if (nodePtr->syntax == ASN1_TIMETICKS) {
	    return FormatTimeTicks(value);
	}
	if (nodePtr->syntax == ASN1_OBJECT_IDENTIFIER) {
	    return FormatOID(value);
	}
	return value;
    }

    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_MibScan --
 *
 *	This procedure converts a value into its base representation
 *	by applying textual conventions or by converting enumerations
 *	into simple numbers.
 *
 * Results:
 *	The pointer to a static string containing the scanned value
 *	or NULL if there is no such MIB node.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

char*
Tnm_MibScan(name, exact, value)
    char *name;
    int exact;
    char *value;
{
    Tnm_MibNode *nodePtr = Tnm_MibFindNode(name, NULL, exact);
    
    if (nodePtr) {
	if (nodePtr->tc) {

	    /*
	     * Check if we have an enumeration for this value.
	     */

	    Tnm_MibEnum *ePtr;
	    for (ePtr = nodePtr->tc->enumList; ePtr; ePtr = ePtr->nextPtr) {
		if (strcmp(ePtr->label, value) == 0) {
		    static char buf[20];
		    sprintf(buf, "%d", ePtr->value);
		    return buf;
		}
	    }
	    
	    /* 
	     * Check if we can apply a display hint.
	     */

            if (nodePtr->tc->displayHint) {
		char *ret = NULL;
		if (nodePtr->syntax == ASN1_OCTET_STRING) {
		    ret = ScanOctetTC(value, nodePtr->tc->displayHint);
		} else if (nodePtr->syntax == ASN1_INTEGER) {
		    ret = ScanIntTC(value, nodePtr->tc->displayHint);
		}
                if (ret) {
		    return ret;
		}
            }
	}
	if (nodePtr->syntax == ASN1_TIMETICKS) {
	    return ScanTimeTicks(value);
	}
	if (nodePtr->syntax == ASN1_OBJECT_IDENTIFIER) {
	    return Tnm_MibGetOid(value, exact);
	}
	return value;
    }

    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_MibGetTC --
 *
 *	This procedure returns the textual convention that applies 
 *	to a MIB node. The returned value is a Tcl list containing
 *	the follwing elements: TC name, TC base syntax, TC display
 *	hint and a list containing the TC enumeration.
 *
 * Results:
 *	The pointer to a static string containing the description of
 *	the textual convention or NULL if there is no such MIB node.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

char*
Tnm_MibGetTC(name, exact)
    char *name;
    int exact;
{
    Tnm_MibNode *nodePtr = Tnm_MibFindNode(name, NULL, exact);
    static Tcl_DString *result = NULL;

    if (result == NULL) {
        result = (Tcl_DString *) ckalloc(sizeof(Tcl_DString));
        Tcl_DStringInit(result);
    } else {
	Tcl_DStringFree(result);
    }

    if (nodePtr) {
        if (nodePtr->tc) {
	    Tnm_MibEnum *ePtr;
	    char *syntax;
	    if (*nodePtr->tc->name != '_') {
		Tcl_DStringAppendElement(result, nodePtr->tc->name);
	    } else {
		Tcl_DStringAppendElement(result, "");
	    }
	    syntax = TnmGetTableValue(tnmSnmpTypeTable, nodePtr->tc->syntax);
	    Tcl_DStringAppendElement(result, syntax ? syntax : "");
	    Tcl_DStringAppendElement(result, nodePtr->tc->displayHint);
	    Tcl_DStringStartSublist(result);
	    for (ePtr = nodePtr->tc->enumList; ePtr; ePtr = ePtr->nextPtr) {
		char buf[20];
		sprintf(buf, "%d", ePtr->value);
	        Tcl_DStringStartSublist(result);
		Tcl_DStringAppendElement(result, ePtr->label);
		Tcl_DStringAppendElement(result, buf);
	        Tcl_DStringEndSublist(result);
	    }
	    Tcl_DStringEndSublist(result);
	    if (nodePtr->tc->moduleName) {
		Tcl_DStringAppendElement(result, nodePtr->tc->moduleName);
	    } else {
		Tcl_DStringAppendElement(result, "");
	    }
	    if (nodePtr->tc->fileName) {
		Tcl_DStringAppendElement(result, nodePtr->tc->fileName);
	    } else {
		Tcl_DStringAppendElement(result, "");
	    }
	}
	return Tcl_DStringValue(result);
    }
    
    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_MibGetFile --
 *
 *	This procedure returns the file name which contains the 
 *	definition of the MIB object.
 *
 * Results:
 *	The pointer to a static string containing the file name or 
 *	NULL if there is no such MIB node.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

char*
Tnm_MibGetFile(name, exact)
    char *name;
    int exact;
{
    Tnm_MibNode *nodePtr = Tnm_MibFindNode(name, NULL, exact);

    if (nodePtr) {
	return nodePtr->fileName ? nodePtr->fileName : "";
    }

    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_MibGetIndex --
 *
 *	This procedure returns the index variables that identify the
 *	instances in a conceptual table. Note, we accept the 'table'
 *	as the 'entry' MIB node.
 *
 * Results:
 *	The pointer to a static string containing the names of objects
 *	that make up the instance identifier or NULL if there is no 
 *	such MIB node.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

char*
Tnm_MibGetIndex(name, exact)
    char *name;
    int exact;
{
    Tnm_MibNode *nodePtr;
    char *expanded = Tnm_HexToOid(name);

    if (expanded) name = expanded;
    nodePtr = Tnm_MibFindNode(name, NULL, exact);
    if (nodePtr) {
	if (nodePtr->syntax == ASN1_SEQUENCE_OF && nodePtr->childPtr) {
	    nodePtr = nodePtr->childPtr;
	}
	if (nodePtr->syntax == ASN1_SEQUENCE && nodePtr->index) {
	    return nodePtr->index;
	} else {
	    return "";
	}
    }

    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_MibGetParent --
 *
 *	This procedure returns the parent of the given MIB node. Note,
 *	this version ignores any trailing instance identifier.
 *
 * Results:
 *	The pointer to a static string containing the name or object
 *	identifier of the parent node or NULL if there is no such MIB 
 *	node.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

char*
Tnm_MibGetParent(name, exact)
    char *name;
    int exact;
{
    Tnm_MibNode *nodePtr;
    char *expanded = Tnm_HexToOid(name);

    if (expanded) name = expanded;
    nodePtr = Tnm_MibFindNode(name, NULL, exact);
    if (nodePtr) {
	if (nodePtr->parentPtr && nodePtr->parentPtr->label) {
	    if (Tnm_IsOid(name)) {
		GetMibPath(nodePtr->parentPtr, oidBuffer);
		return oidBuffer;
	    } else {
		return nodePtr->parentPtr->label;
	    }
	} else {
	    return "";
	}
    }

    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_MibGetDefault --
 *
 *	This procedure returns the default value for a MIB instance.
 *
 * Results:
 *	The pointer to a static string containing the default value
 *	or NULL if there is no such MIB node.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

char*
Tnm_MibGetDefault(name, exact)
    char *name;
    int exact;
{
    Tnm_MibNode *nodePtr = Tnm_MibFindNode(name, NULL, exact);

    if (nodePtr) {
	if (nodePtr->index && nodePtr->syntax != ASN1_SEQUENCE_OF 
	    && nodePtr->syntax != ASN1_SEQUENCE ) {
	    return nodePtr->index;
	} else {
	    return "";
	}
    }

    return NULL;
}
