/*
 * tnmInit.c --
 *
 *	This is the initialization of the Tnm Tcl extension with
 *	commands to retrieve management information from TCP/IP
 *	networks.
 *
 * Copyright (c) 1993-1996 Technical University of Braunschweig.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#include "tnmInt.h"
#include "tnmPort.h"

/*
 * Forward declarations for procedures defined later in this file:
 */

static void
InitVars		_ANSI_ARGS_((Tcl_Interp *interp));

static int
InitRc			_ANSI_ARGS_((Tcl_Interp *interp));

static void
InitSafeCmds		_ANSI_ARGS_((Tcl_Interp *interp));

static int
InitUnsafeCmds		_ANSI_ARGS_((Tcl_Interp *interp));


/*
 *----------------------------------------------------------------------
 *
 * InitVars --
 *
 *	This procedure initializes all global Tcl variables that are 
 *	exported by the Tnm extension.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	Tcl variables are created.
 *
 *----------------------------------------------------------------------
 */

static void
InitVars(interp)
    Tcl_Interp *interp;
{
    char *machine, *os, *vers, *user, *tmp, *p, *path, *cacheName;
    Tcl_DString arch, cache, dst;
    
    path = getenv("TNM_LIBRARY");
    if (! path) {
	path = TNMLIB;
    }

    Tcl_SetVar2(interp, "tnm", "version", TNM_VERSION, TCL_GLOBAL_ONLY);
    Tcl_SetVar2(interp, "tnm", "url", TNM_URL, TCL_GLOBAL_ONLY);
    Tcl_SetVar2(interp, "tnm", "library", path, TCL_GLOBAL_ONLY);
    Tcl_SetVar2(interp, "tnm", "cache", path, TCL_GLOBAL_ONLY);
    Tcl_SetVar2(interp, "tnm", "host", Tcl_GetHostName(), TCL_GLOBAL_ONLY);
    user = getenv("USER");
    if (user == NULL) {
	user = getenv("USERNAME");
	if (user == NULL) {
	    user = getenv("LOGNAME");
	    if (user == NULL) {
		user = "unknown";
	    }
	}
    }
    Tcl_SetVar2(interp, "tnm", "user", user, TCL_GLOBAL_ONLY);

    /*
     * Search for a directory which allows to hold temporary files.
     * Save the directory name in the tnm(tmp) variable.
     */

    tmp = getenv("TEMP");
    if (! tmp) {
	tmp = getenv("TMP");
	if (! tmp) {
	    tmp = "/tmp";
	    if (access(tmp, W_OK) != 0) {
		tmp = ".";
	    }
	}
    }
    for (p = tmp; *p; p++) {
	if (*p == '\\') {
	    *p = '/';
	}
    }
    Tcl_SetVar2(interp, "tnm", "tmp", tmp, TCL_GLOBAL_ONLY);

    /*
     * Determine the architecture string which is used to store 
     * machine dependend files in the Tnm library area.
     */

    machine = Tcl_GetVar2(interp, "tcl_platform", "machine", TCL_GLOBAL_ONLY);
    os = Tcl_GetVar2(interp, "tcl_platform", "os", TCL_GLOBAL_ONLY);
    vers = Tcl_GetVar2(interp, "tcl_platform", "osVersion", TCL_GLOBAL_ONLY);

    Tcl_DStringInit(&arch);
    if (machine && os && vers) {
	Tcl_DStringAppend(&arch, machine, -1);
	Tcl_DStringAppend(&arch, "-", 1);
	Tcl_DStringAppend(&arch, os, -1);
	Tcl_DStringAppend(&arch, "-", 1);
	Tcl_DStringAppend(&arch, vers, -1);
    } else {
	Tcl_DStringAppend(&arch, "unknown-os", -1);
    }

#if 0
    /*
     * Initialize the tnm(cache) variable which points to a directory
     * where we can cache shared data between different instantiations
     * of the Tnm extension. We usually locate the cache in the users
     * home directory. However, if this fails (because the user does
     * not have a home), we locate the cache in the tmp file area.
     */

    Tcl_DStringInit(&dst);
    Tcl_DStringInit(&cache);
    Tcl_DStringAppend(&dst, "~/.tnm", -1);
    Tcl_DStringAppend(&dst, TNM_VERSION, -1);
    cacheName = Tcl_TranslateFileName(interp, Tcl_DStringValue(&dst), &cache);
    if (! cacheName) {
	Tcl_DStringFree(&dst);
	Tcl_DStringAppend(&dst, tmp, -1);
	Tcl_DStringAppend(&dst, "/tnm", -1);
	Tcl_DStringAppend(&dst, TNM_VERSION, -1);
	cacheName = Tcl_TranslateFileName(interp, Tcl_DStringValue(&dst),
					  &cache);
    }
    if (cacheName) {
	(void) TnmMkDir(interp, cacheName);
    }
    Tcl_SetVar2(interp, "tnm", "cache", cacheName, TCL_GLOBAL_ONLY);
    Tcl_DStringFree(&dst);
    Tcl_DStringFree(&cache);
#endif

    /*
     * Remove all white spaces and slashes from the architecture string 
     * because these characters are a potential source of problems and 
     * I really do not like white spaces in a directory name.
     */

    {
	char *d = Tcl_DStringValue(&arch);
	char *s = Tcl_DStringValue(&arch);

	while (*s) {
	    *d = *s;
	    if ((!isspace(*s)) && (*s != '/')) d++;
	    s++;
	}
	*d = '\0';
    } 

    Tcl_SetVar2(interp, "tnm", "arch", 
		Tcl_DStringValue(&arch), TCL_GLOBAL_ONLY);
    Tcl_DStringFree(&arch);
}

/*
 *----------------------------------------------------------------------
 *
 * InitRc --
 *
 *	This procedure evaluates the Tnm initialization scripts.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
InitRc(interp)
    Tcl_Interp *interp;
{
    char *fileName, *library;
    char **libArgv = NULL;
    int libArgc;

    library = Tcl_GetVar2(interp, "tnm", "library", TCL_GLOBAL_ONLY);
    if (library) {
	Tcl_SplitPath(library, &libArgc, &libArgv);
    }

    if (library) {

	int tmpArgc;
	char **tmpArgv = (char **) ckalloc((libArgc + 4) * sizeof(char *));
	Tcl_DString buffer;

	Tcl_DStringInit(&buffer);
	for (tmpArgc = 0; tmpArgc < libArgc; tmpArgc++) {
	    tmpArgv[tmpArgc] = libArgv[tmpArgc];
	}

	tmpArgv[tmpArgc++] = "library";
	tmpArgv[tmpArgc++] = "init.tcl";
	Tcl_DStringFree(&buffer);
	fileName = Tcl_JoinPath(tmpArgc, tmpArgv, &buffer);
	if (access(fileName, R_OK) != 0) {
	    tmpArgv[0] = "init.tcl";
	    tmpArgc = 1;
	    Tcl_DStringFree(&buffer);
	    fileName = Tcl_JoinPath(tmpArgc, tmpArgv, &buffer);
	    if (access(fileName, R_OK) != 0) {
		tmpArgv[0] = "..";
		tmpArgv[1] = "tnm";
		tmpArgv[2] = "library";
		tmpArgv[3] = "init.tcl";
		tmpArgc = 4;
		Tcl_DStringFree(&buffer);
		fileName = Tcl_JoinPath(tmpArgc, tmpArgv, &buffer);
		if (access(fileName, R_OK) != 0) {
		    fileName = (char *) NULL;
		}
	    }
	}

	if (fileName) {
	    if (Tcl_EvalFile(interp, fileName) != TCL_OK) {
		TnmWriteMessage(interp, interp->result);
		TnmWriteMessage(interp, "\n");
	    }
	}
	
	ckfree((char *) tmpArgv);
	ckfree((char *) libArgv);
	Tcl_DStringFree(&buffer);

	if (!fileName) {
	    Tcl_AppendResult(interp, "no initialization file: tried ",
                             library, "/library/init.tcl, init.tcl, and ",
                             "../library/init.tcl", (char *) NULL);
	    return TCL_ERROR;
	}
    }

    /*
     * Load the user specific startup file. We check to see if we
     * we have a readable startup file so that we only complain
     * about errors when we are expected to complain.
     */

    fileName = getenv("TNM_RCFILE");
    if (! fileName) {
	fileName = "~/.scottyrc";
    }

    if (fileName != NULL) {
	Tcl_DString temp;
	char *fullName;

	Tcl_DStringInit(&temp);
        fullName = Tcl_TranslateFileName(interp, fileName, &temp);
	if (fullName == NULL) {
	    TnmWriteMessage(interp, interp->result);
	    TnmWriteMessage(interp, "\n");
	} else {
	    Tcl_Channel channel;
	    channel = Tcl_OpenFileChannel(NULL, fullName, "r", 0);
	    if (channel) {
		Tcl_Close((Tcl_Interp *) NULL, channel);
		if (Tcl_EvalFile(interp, fullName) != TCL_OK) {
		    TnmWriteMessage(interp, interp->result);
                    TnmWriteMessage(interp, "\n");
		}
	    }
	}
	Tcl_DStringFree(&temp);
    }

    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * InitSafeCmds --
 *
 *	This procedure initializes all safe commands provided by the
 *	Tnm extension.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	Tcl variables are created.
 *
 *----------------------------------------------------------------------
 */

static void
InitSafeCmds(interp)
    Tcl_Interp *interp;
{
    Tcl_CmdInfo info;

    if (Tcl_GetCommandInfo(interp, "event", &info) == 0) {
	Tcl_CreateCommand(interp, "event", Tnm_EventCmd,
			  (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    }
    Tcl_CreateCommand(interp, "job", Tnm_JobCmd,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
}

/*
 *----------------------------------------------------------------------
 *
 * InitUnsafeCmds --
 *
 *	This procedure initializes all unsafe commands provided by the
 *	Tnm extension.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	Tcl variables are created.
 *
 *----------------------------------------------------------------------
 */

static int
InitUnsafeCmds(interp)
    Tcl_Interp *interp;
{
    Tcl_CreateCommand(interp, "syslog", Tnm_SyslogCmd,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "icmp", Tnm_IcmpCmd,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "dns", Tnm_DnsCmd,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "ntp", Tnm_NtpCmd,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "udp", Tnm_UdpCmd,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "sunrpc", Tnm_SunrpcCmd,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
#if 0
    Tcl_CreateCommand(interp, "rpc", Tnm_RpcCmd,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
#endif
    Tcl_CreateCommand(interp, "http", Tnm_HttpCmd,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "netdb", Tnm_NetdbCmd,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "ined", Tnm_InedCmd,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

    if (Tnm_SnmpInit(interp) != TCL_OK) {
        return TCL_ERROR;
    }

#ifdef HAVE_MSQL
    if (Msqltcl_Init(interp) == TCL_ERROR) {
        return TCL_ERROR;
    }
#endif

#ifdef HAVE_OSIMIS
    if (Tnm_CmipInit(interp) != TCL_OK) {
        return TCL_ERROR;
    }
#endif

#ifdef HAVE_GDMO
    if (Tnm_GdmoInit(interp) != TCL_OK) {
        return TCL_ERROR;
    }
#endif

    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * TnmInit --
 *
 *	This procedure is the platform independent entry point for 
 *	trusted Tcl interpreters.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	Tcl variables are created.
 *
 *----------------------------------------------------------------------
 */

int
TnmInit(interp)
    Tcl_Interp *interp;
{
    int code;

    code = Tcl_PkgProvide(interp, "Tnm", TNM_VERSION);
    if (code != TCL_OK) {
        return code;
    }

    InitVars(interp);
    InitSafeCmds(interp);
    code = InitUnsafeCmds(interp);
    if (code != TCL_OK) {
	return code;
    }
    
    return InitRc(interp);
}

/*
 *----------------------------------------------------------------------
 *
 * TnmSafeInit --
 *
 *	This procedure is the platform independet entry point for 
 *	safe Tcl interpreters.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	Tcl variables are created.
 *
 *----------------------------------------------------------------------
 */

int
TnmSafeInit(interp)
    Tcl_Interp *interp;
{
    int code;

    code = Tcl_PkgProvide(interp, "Tnm", TNM_VERSION);
    if (code != TCL_OK) {
        return code;
    }

    InitVars(interp);
    InitSafeCmds(interp);
    
    return InitRc(interp);
}
