/* saytime - talking clock for SPARCstations
**
** Copyright (C) 1990 by Jef Poskanzer <jef@acme.com>.  All rights reserved.
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
** 1. Redistributions of source code must retain the above copyright
**    notice, this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
** 
** THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
** ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
** IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
** ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
** OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
** LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
** OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
** SUCH DAMAGE.
*/

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <string.h>
#include <sys/types.h>
#include <sys/file.h>
#include <sys/fcntl.h>
#include <sys/uio.h>


#ifdef CGI
/* Define special features for use in CGI mode. */
#define WRITE_STDOUT
#define AUDIO_HEADER
#endif /* CGI */

/* Here are the defines for the thirty different phrases the program
** needs.  If you want to substitute your own voice for mine, I suggest
** you record the following lines:
**
**     The time is eight o'clock exactly.
**     The time is eight oh-eight and one second.
**     The time is eight oh-eight and eight seconds.
**     One.  Two.  Three.  Four.  Five.  Six.  Seven.  Eight.  Nine.  Ten.
**     Eleven.  Twelve.  Thirteen.  Fourteen.  Fifteen.  Sixteen.  Seventeen.
**     Eighteen.  Nineteen.  Twenty.  Thirty.  Forty.  Fifty.
**
** Then use Sun's sound demo to dissect the lines into separate files.
** It's not really that much work, it took me about an hour.
*/

#define PH_ONE		1
#define PH_TWO		2
#define PH_THREE	3
#define PH_FOUR		4
#define PH_FIVE		5
#define PH_SIX		6
#define PH_SEVEN	7
#define PH_EIGHT	8
#define PH_NINE		9
#define PH_TEN		10
#define PH_ELEVEN	11
#define PH_TWELVE	12
#define PH_THIRTEEN	13
#define PH_FOURTEEN	14
#define PH_FIFTEEN	15
#define PH_SIXTEEN	16
#define PH_SEVENTEEN	17
#define PH_EIGHTEEN	18
#define PH_NINETEEN	19
#define PH_TWENTY	20
#define PH_THIRTY	21
#define PH_FORTY	22
#define PH_FIFTY	23
#define PH_THE_TIME_IS	24
#define PH_OCLOCK	25
#define PH_OH		26
#define PH_EXACTLY	27
#define PH_AND		28
#define PH_SECOND	29
#define PH_SECONDS	30


static int audiofd = -1;


#ifdef CGI
static void
cgi_header( void )
    {
    char* content = "Content-type: audio/basic\n\n";
    if ( write( audiofd, content, strlen( content ) ) < 0 )
	{
	perror( "writing to audio device" );
	exit( 1 );
	}
    }
#endif /* CGI */


#ifdef AUDIO_HEADER

#include <sys/types.h>
#include <netinet/in.h>

/* Struct taken from audio_filehdr.h */
typedef struct {
    unsigned long magic;	/* magic number */
    unsigned long hdr_size;	/* size of this header */
    unsigned long data_size;	/* length of data (optional) */
    unsigned long encoding;	/* data encoding format */
    unsigned long sample_rate;	/* samples per second */
    unsigned long channels;	/* number of interleaved channels */
    } Audio_filehdr;
#define AUDIO_FILE_MAGIC 0x2e736e64
#define AUDIO_UNKNOWN_SIZE (~0)
#define AUDIO_FILE_ENCODING_MULAW_8 1

static void
audio_header( void )
    {
    Audio_filehdr hdr;

    hdr.magic = htonl( AUDIO_FILE_MAGIC );
    hdr.hdr_size = htonl( sizeof(hdr) );
    hdr.data_size = htonl( AUDIO_UNKNOWN_SIZE );
    hdr.encoding = htonl( AUDIO_FILE_ENCODING_MULAW_8 );
    hdr.sample_rate = htonl( 8000 );
    hdr.channels = htonl( 1 );
    if ( write( audiofd, &hdr, sizeof(hdr) ) < 0 )
	{
	perror( "writing to audio device" );
	exit( 1 );
	}
    }
#endif /* AUDIO_HEADER */


static void
sayfile( char* filename )
    {
    int filefd;
    int r, w;
    unsigned char buf[1024];
    char pathname[200];

    if ( audiofd == -1 )
	{
#ifdef WRITE_STDOUT
	audiofd = 1;
#else /* WRITE_STDOUT */
	audiofd = open( "/dev/audio", O_WRONLY | O_NDELAY );
	if ( audiofd < 0 )
	    {
	    perror( "opening /dev/audio" );
	    exit( 1 );
	    }
#endif /* WRITE_STDOUT */

#ifdef CGI
	cgi_header();
#endif /* CGI */

#ifdef AUDIO_HEADER
	audio_header();
#endif /* AUDIO_HEADER */
	}

    (void) sprintf( pathname, "%s/%s", SOUND_DIR, filename );
    filefd = open( pathname, O_RDONLY );
    if ( filefd < 0 )
	{
	perror( "opening audio file" );
	exit( 1 );
	}

    for (;;)
	{
	r = read( filefd, buf, sizeof(buf) );
	if ( r < 0 )
	    {
	    perror( "reading from audio file" );
	    exit( 1 );
	    }
	if ( r == 0 )
	    break;

	for ( ; ; )
	    {
	    w = write( audiofd, buf, r );
	    if ( w < 0 )
		{
		perror( "writing to audio device" );
		exit( 1 );
		}
	    if ( w != 0 )
		break;
	    usleep( 1000 );
	    }
	if ( w != r )
	    {
	    (void) fprintf( stderr, "read returned %d, write returned %d\n", r, w );
	    exit( 1 );
	    }
	}
    close( filefd );
    }


static void
sayclose( void )
    {
    if ( audiofd != -1 && audiofd != 1 )
	close( audiofd );
    audiofd = -1;
    }


static void
sayphrase( int phrase )
    {
    switch ( phrase )
	{
	case PH_ONE:
	sayfile( "1.ulaw" );
	break;

	case PH_TWO:
	sayfile( "2.ulaw" );
	break;

	case PH_THREE:
	sayfile( "3.ulaw" );
	break;

	case PH_FOUR:
	sayfile( "4.ulaw" );
	break;

	case PH_FIVE:
	sayfile( "5.ulaw" );
	break;

	case PH_SIX:
	sayfile( "6.ulaw" );
	break;

	case PH_SEVEN:
	sayfile( "7.ulaw" );
	break;

	case PH_EIGHT:
	sayfile( "8.ulaw" );
	break;

	case PH_NINE:
	sayfile( "9.ulaw" );
	break;

	case PH_TEN:
	sayfile( "10.ulaw" );
	break;

	case PH_ELEVEN:
	sayfile( "11.ulaw" );
	break;

	case PH_TWELVE:
	sayfile( "12.ulaw" );
	break;

	case PH_THIRTEEN:
	sayfile( "13.ulaw" );
	break;

	case PH_FOURTEEN:
	sayfile( "14.ulaw" );
	break;

	case PH_FIFTEEN:
	sayfile( "15.ulaw" );
	break;

	case PH_SIXTEEN:
	sayfile( "16.ulaw" );
	break;

	case PH_SEVENTEEN:
	sayfile( "17.ulaw" );
	break;

	case PH_EIGHTEEN:
	sayfile( "18.ulaw" );
	break;

	case PH_NINETEEN:
	sayfile( "19.ulaw" );
	break;

	case PH_TWENTY:
	sayfile( "20.ulaw" );
	break;

	case PH_THIRTY:
	sayfile( "30.ulaw" );
	break;

	case PH_FORTY:
	sayfile( "40.ulaw" );
	break;

	case PH_FIFTY:
	sayfile( "50.ulaw" );
	break;

	case PH_THE_TIME_IS:
	sayfile( "the_time_is.ulaw" );
	break;

	case PH_OCLOCK:
	sayfile( "oclock.ulaw" );
	break;

	case PH_OH:
	sayfile( "oh.ulaw" );
	break;

	case PH_EXACTLY:
	sayfile( "exactly.ulaw" );
	break;

	case PH_AND:
	sayfile( "and.ulaw" );
	break;

	case PH_SECOND:
	sayfile( "second.ulaw" );
	break;

	case PH_SECONDS:
	sayfile( "seconds.ulaw" );
	break;

	default:
	(void) fprintf( stderr, "Shouldn't happen.\n" );
	exit( 1 );
	}
    }


static void
saydigit( int n )
    {
    switch ( n )
	{
	case 1:
	sayphrase( PH_ONE );
	break;

	case 2:
	sayphrase( PH_TWO );
	break;

	case 3:
	sayphrase( PH_THREE );
	break;

	case 4:
	sayphrase( PH_FOUR );
	break;

	case 5:
	sayphrase( PH_FIVE );
	break;

	case 6:
	sayphrase( PH_SIX );
	break;

	case 7:
	sayphrase( PH_SEVEN );
	break;

	case 8:
	sayphrase( PH_EIGHT );
	break;

	case 9:
	sayphrase( PH_NINE );
	break;

	default:
	(void) fprintf( stderr, "Shouldn't happen.\n" );
	exit( 1 );
	}
    }


static void
saynumber( int n, int leadingzero )
    {
    int ones, tens;

    ones = n % 10;
    tens = n / 10;

    switch ( tens )
	{
	case 0:
	if ( leadingzero )
	    sayphrase( PH_OH );
	saydigit( ones );
	break;

	case 1:
	switch ( ones )
	    {
	    case 0:
	    sayphrase( PH_TEN );
	    break;

	    case 1:
	    sayphrase( PH_ELEVEN );
	    break;

	    case 2:
	    sayphrase( PH_TWELVE );
	    break;

	    case 3:
	    sayphrase( PH_THIRTEEN );
	    break;

	    case 4:
	    sayphrase( PH_FOURTEEN );
	    break;

	    case 5:
	    sayphrase( PH_FIFTEEN );
	    break;

	    case 6:
	    sayphrase( PH_SIXTEEN );
	    break;

	    case 7:
	    sayphrase( PH_SEVENTEEN );
	    break;

	    case 8:
	    sayphrase( PH_EIGHTEEN );
	    break;

	    case 9:
	    sayphrase( PH_NINETEEN );
	    break;

	    default:
	    (void) fprintf( stderr, "Shouldn't happen.\n" );
	    exit( 1 );
	    }
	break;

	case 2:
	sayphrase( PH_TWENTY );
	if ( ones != 0 )
	    saydigit( ones );
	break;

	case 3:
	sayphrase( PH_THIRTY );
	if ( ones != 0 )
	    saydigit( ones );
	break;

	case 4:
	sayphrase( PH_FORTY );
	if ( ones != 0 )
	    saydigit( ones );
	break;

	case 5:
	sayphrase( PH_FIFTY );
	if ( ones != 0 )
	    saydigit( ones );
	break;

	default:
	(void) fprintf( stderr, "Shouldn't happen.\n" );
	exit( 1 );
	}
    }


int
main( int argc, char** argv )
    {
    long clock;
    struct tm* t;

    clock = time( (long*) 0 );
    t = localtime( &clock );

    sayphrase( PH_THE_TIME_IS );

    if ( t->tm_hour == 0 )
	saynumber( 12, 0 );
    else if ( t->tm_hour > 12 )
	saynumber( t->tm_hour - 12, 0 );
    else
	saynumber( t->tm_hour, 0 );

    if ( t->tm_min == 0 )
	sayphrase ( PH_OCLOCK );
    else
	saynumber( t->tm_min, 1 );

    if ( t->tm_sec == 0 )
	sayphrase( PH_EXACTLY );
    else
	{
	sayphrase( PH_AND );
	saynumber( t->tm_sec, 0 );
	if ( t->tm_sec == 1 )
	    sayphrase( PH_SECOND );
	else
	    sayphrase( PH_SECONDS );
	}

    sayclose();
    exit( 0 );
    }
