/*
 * srpc_test.c
 *
 * main() and friends for SRPC test suite server.
 *
 * Author: Shelby Davis, with much taken from nasd_edrfs_main.c by Jim
 * Zelenka */
/*
 * Copyright (c) of Carnegie Mellon University, 1997,1998,1999.
 *
 * Permission to reproduce, use, and prepare derivative works of
 * this software for internal use is granted provided the copyright
 * and "No Warranty" statements are included with all reproductions
 * and derivative works. This software may also be redistributed
 * without charge provided that the copyright and "No Warranty"
 * statements are included in all redistributions.
 *
 * NO WARRANTY. THIS SOFTWARE IS FURNISHED ON AN "AS IS" BASIS.
 * CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER
 * EXPRESSED OR IMPLIED AS TO THE MATTER INCLUDING, BUT NOT LIMITED
 * TO: WARRANTY OF FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY
 * OF RESULTS OR RESULTS OBTAINED FROM USE OF THIS SOFTWARE. CARNEGIE
 * MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT
 * TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
 */


#include <nasd/nasd_options.h>
#include <nasd/nasd_types.h>
#include <nasd/nasd_freelist.h>
#include <nasd/nasd_itypes.h>
#include <nasd/nasd_mem.h>
#include <nasd/nasd_common.h>
#include <nasd/nasd_getopt.h>
#include <nasd/nasd_timer.h>
#include <nasd/nasd_sys.h>

#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

char *progname;

int svc_threads = 15;
int stack_size  = 32768;

int                           binding_type;
int                           binding_args_len;
void                         *binding_args;
int                           nondefault_binding = 0;

/* 
 * Under Linux, signals may be delivered to *all* of the threads, not
 * just one (as under DUX).  This is a violation of the POSIX pthreads
 * spec, and happens because Linux "threads" are really "processes"
 * that happen to share the same address space and file descriptor
 * set.  Signals sent with kill() or raise() will go to the thread
 * whose PID is specified (which also violates POSIX, but let's not go
 * there), but signals sent to the process group -- say, when you press
 * ^C on a tty -- go to all threads in the process group.
 * 
 * Because it is impossible to determine which thread will hit this
 * code first, it is possible for the thread that runs the shutdown
 * code to be a thread that also has cleanup work to do before it
 * exits.  If that happens, the drive will deadlock.  Therefore, we
 * create a "supervisor" thread to do the shutdown stuff, and allow
 * the thread that ran the signal handler to return.  When signals come
 * in, we simply ensure that this thread awakens.
 *
 * Under other systems, we're going to go ahead and create this extra
 * thread as well so we can avoid tricky synchronization at cleanup
 * time.
 */
nasd_thread_id_t srpc_test_master_shutdown_thread_id;
nasd_threadgroup_t srpc_test_master_shutdown_thread_group;
nasd_thread_t srpc_test_shutdown_worker_thread;
sig_atomic_t srpc_test_shutdown_signalled = 0;
int alldone = 0;

int srpc_test_stopped = 0;

pid_t srpc_test_signal_pid;

int signal_pipe[2];
#define signal_pipe_rd  signal_pipe[0]
#define signal_pipe_wr  signal_pipe[1]

#define CLOSEUP_PIPES() { \
  close(signal_pipe[0]); \
  close(signal_pipe[1]); \
}

static void handle_interrupt(nasd_threadarg_t ignored);

static void
srpc_test_shutdown_worker_proc(
  nasd_threadarg_t  ignored)
{
  fd_set mrdb, rdb;
  int ret;

  srpc_test_signal_pid = getpid();

  srpc_test_master_shutdown_thread_id = nasd_thread_self();
  NASD_THREADGROUP_RUNNING(&srpc_test_master_shutdown_thread_group);

  FD_ZERO(&mrdb);
  FD_SET(signal_pipe_rd, &mrdb);

  while(!srpc_test_shutdown_signalled) {
    rdb = mrdb;
    ret = select(signal_pipe_rd+1, &rdb, NULL, NULL, NULL);
    if (srpc_test_shutdown_signalled || FD_ISSET(signal_pipe_rd, &mrdb))
      break;
    if (ret) {
      fprintf(stderr, "EDRFS SERVER WARNING: shutdown worker thread got "
        "ret %d errno %d\n", ret, errno);
      fflush(stderr);
    }
  }

  handle_interrupt(NULL);

  /* I don't think we should ever get back here. */
}

void
handle_sigint(
  int  sig)
{
  int ret;
  char c;

  srpc_test_stopped = 1;
  c = 'a';
  ret = write(signal_pipe_wr, &c, 1);
}

static void
handle_interrupt(
  nasd_threadarg_t  ignored)
{
  nasd_thread_id_t self;
  nasd_status_t rc;

  extern int nasd_mem_use_counter;

  self = nasd_thread_self();

  fprintf(stderr, "INTERRUPTED (thread id %" NASD_THREAD_ID_FMT ")\n", self);
  fflush(stderr);

  srpc_test_stop_rpc();

#if MJU_DEBUG
  fprintf(stderr, "shutdown[%lu]: nasd_edrfs_stop_rpc() complete\n", self);
  fflush(stderr);
#endif /* MJU_DEBUG */

  rc = srpc_test_really_shutdown();
  if (rc) {
    fprintf(stderr,
      "EDRFS ERROR: got 0x%x (%s) from nasd_edrfs_really_shutdown()\n",
      rc, nasd_error_string(rc));
    fflush(stderr);
    exit(1);
  }

  nasd_mem_shutdown();

  fprintf(stderr, "Completed shutdown.\n");
  fflush(stderr);
  fflush(stdout);
#if NASD_MEM_COUNT_ALLOC > 0
  if (nasd_mem_allocated) {
    fprintf(stderr,
            "WARNING: %" NASD_MEMALLOC_FMT
            " bytes of memory still outstanding, something leaked core.\n",
            nasd_mem_allocated);
    fprintf(stderr, "memory subsystem use counter at %d\n", nasd_mem_use_counter);
    fflush(stderr);
  }
  else {
    printf("0 bytes of memory still outstanding.\n");
  }
#endif /* NASD_MEM_COUNT_ALLOC > 0 */
  fflush(stdout);
  fflush(stderr);

  CLOSEUP_PIPES();
  alldone = 1;
  exit(0);
}

nasd_status_t
srpc_test_frame_init()
{
  /* no longer any user-specific stuff to init here */
  return(NASD_SUCCESS);
}

void
usage()
{
  fprintf(stderr, "USAGE: %s [options]\n", progname);
  fprintf(stderr, "Options:\n");
  fprintf(stderr, "  -k use kernel device\n");
  fprintf(stderr, "  -l use colocated drive\n");
  fprintf(stderr, "  -s stack size [default %d]\n", stack_size);
  fprintf(stderr, "  -t service threads [default %d]\n", svc_threads);
  fprintf(stderr, "  -x ignore preexisting message queue\n");
  fflush(stderr);
  exit(1);
}

int
main(
  int     argc,
  char  **argv)
{
  nasd_uint16 ipport;
  struct timeval tv;
  nasd_status_t rc;
  char c, *env;
  int ret;

  progname = argv[0];

  /*  binding_type = NASD_BIND_DEFAULT;
  binding_args = NULL;
  binding_args_len = 0;*/

  /*  while (nasd_getopt(argc, argv, "klm:s:t:x", &c)) {
    switch(c) {
      case 'k':
        if (nondefault_binding)
          usage();
        nondefault_binding = 1;
        binding_type = NASD_BIND_KPDEV_DEFAULT;
        binding_args = &kpdev_args;
        binding_args_len = sizeof(kpdev_args);
        strcpy(kpdev_args.devname, "/dev/nasdkp0");
        break;
      case 'l':
        if (nondefault_binding)
          usage();
        nondefault_binding = 1;
        binding_type = NASD_BIND_COLOCATE;
        binding_args = &kpdev_args;
        binding_args_len = sizeof(kpdev_args);
        strcpy(kpdev_args.devname, "/dev/nasdkp0");
        break;
      case 'm':
        mountfile = nasd_optarg;
        break;
      case 's':
        if (sscanf(nasd_optarg, "%d", &stack_size) != 1)
          usage();
        break;
      case 't':
        if (sscanf(nasd_optarg, "%d", &svc_threads) != 1)
          usage();
        break;
      case 'x':
        nasd_edrfs_mq_ignore_eexist = 1;
        break;
      default:
        fprintf(stderr, "Unknown option '%c'\n", c);
        usage();
    }
    } */
  if (nasd_optind < argc)
    usage();

  ret = pipe(signal_pipe);
  if (ret) {
    fprintf(stderr, "ERROR: cannot create signal pipe\n");
    fflush(stderr);
    exit(1);
  }

  rc = nasd_threads_init();
  if (rc) {
    fprintf(stderr, "EDRFS ERROR: got 0x%x (%s) from nasd_threads_init()\n",
      rc, nasd_error_string(rc));
    fflush(stderr);
    exit(1);
  }

  rc = nasd_mem_init();
  if (rc) {
    fprintf(stderr, "EDRFS ERROR: got 0x%x (%s) from nasd_mem_init()\n",
      rc, nasd_error_string(rc));
    fflush(stderr);
    exit(1);
  }

  srpc_test_signal_pid = getpid();

  rc = nasd_init_threadgroup(&srpc_test_master_shutdown_thread_group);
  if (rc) {
    fprintf(stderr,
      "ERROR: got 0x%x (%s) initializing "
      "srpc_test_master_shutdown_thread_group\n",
      rc, nasd_error_string(rc));
    fflush(stderr);
    goto abort_srpc_test;
  }

  rc = nasd_thread_create_w_name(&srpc_test_shutdown_worker_thread,
    srpc_test_shutdown_worker_proc, NULL, "srpc_test_shutdown_worker");
  if (rc) {
    fprintf(stderr,
      "ERROR: got 0x%x (%s) creating shutdown worker thread\n",
      rc, nasd_error_string(rc));
    fflush(stderr);
    goto abort_srpc_test;
  }
  NASD_THREADGROUP_WAIT_START(&srpc_test_master_shutdown_thread_group);

  printf("EDRFS: setup RPC subsystem\n");

  if (sscanf(SRPC_TEST_SERVER_PORT, "%hu", &ipport) != 1) {
    NASD_PANIC();
  }
 
  rc = srpc_test_startup_rpc();
  if (rc) {
    fprintf(stderr, "EDRFS ERROR: got 0x%x (%s) from srpc_test_startup_rpc()\n",
      rc, nasd_error_string(rc));
    fflush(stderr);
    CLOSEUP_PIPES();
    exit(1);
  }

#ifndef LINUX
  signal(SIGUSR1, handle_sigint);
#endif /* LINUX */

#ifdef LINUX
  /* Linux's pthreads library uses SIGUSR1 and SIGUSR2, so we can't. */
  signal(SIGPWR, handle_sigint);
#endif /* LINUX */

  signal(SIGINT, handle_sigint);

  rc = srpc_test_rpc_set_stacksize(stack_size);
  if (rc == NASD_OP_NOT_SUPPORTED) {
    printf("EDRFS: setting stack size not supported\n");
  }
  else if (rc) {
    printf("EDRFS: got 0x%x (%s) setting stack size to %d\n",
           rc, nasd_error_string(rc), stack_size);
  }
  else {
    printf("EDRFS: stack size set to %d\n", stack_size);
  }

  printf("Running with pid %d\n", srpc_test_signal_pid);
  fflush(stdout);

  rc = srpc_test_mq_specific_listen(svc_threads);
  if(rc) {
    fprintf(stderr, "Couldn't start message queue listener: status=0x%x (%s)\n",
            rc, nasd_error_string(rc));
  } else {
    rc = srpc_test_rpc_listen(svc_threads, ipport);
    if (rc) {
      fprintf(stderr, "EDRFS ERROR: got 0x%x (%s) from sprc_test_rpc_listen()\n",
              rc, nasd_error_string(rc));
      fflush(stderr);
    }
  }

abort_srpc_test:
  if (!srpc_test_stopped) {
    srpc_test_stop_rpc();

    rc = srpc_test_really_shutdown();
    if (rc) {
      fprintf(stderr, "ERROR: got %s (%lx) from srpc_test_really_shutdown()\n",
        nasd_error_string(rc), (u_long)rc);
      fflush(stderr);
      CLOSEUP_PIPES();
      exit(1);
    }
  }
  else {
    /*
     * Someone else is stopping. They'll exit for us.
     */
    do {
      tv.tv_sec = 1;
      tv.tv_usec = 0;

      ret = select(1, NULL, NULL, NULL, &tv);
      if (ret) {
        break;
       }
    } while(alldone == 0);
  }

  CLOSEUP_PIPES();
  exit(2);
}

/* Local Variables:  */
/* indent-tabs-mode: nil */
/* tab-width: 2 */
/* End: */
