/*
 * Copyright (c) 2009, 2010 Nhat Minh Lê
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE
 * COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <err.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#ifndef __NetBSD__
#include <regxml/compat.h>
#endif
#include <regxml/regxml.h>
#include <regxml/xml.h>

#define DEFAULTINDENT 2
#define INBUFSIZE 8192
#define ERRBUFSIZE 128

static struct regxml *reg;
static struct regxml_elem **match;
static struct regxml_xml *xml;
static char *buf;

static char *scope, *scopeprefix;
static int noexpandentities, preservews;
static unsigned int baseindent = DEFAULTINDENT;
static int tabindent, attrline;
static int caseinsensitive, showcontext;
static enum {
	PRINT,
	RPRINT,
	COUNT,
	RLIST,
	LIST,
	QUIET
} action;

static const char *filename;
static unsigned int count;

static int found;
static regxml_xid_t last;

static void die(int);
static void printopening(struct regxml_elem *);
static void printclosing(struct regxml_elem *);
static void printcontext(struct regxml_elem *);
static void printscope(void);
static int process(void);
static void dofile(int);
static void setformat(char *);
static void usage(void);

static void
die(int code)
{
	char errbuf[ERRBUFSIZE];

	if (xml != NULL)
		(void)regxml_xml_strerror(xml, code, errbuf, ERRBUFSIZE);
	else
		(void)regxml_strerror(reg, code, errbuf, ERRBUFSIZE);
	errx(2, "%s", errbuf);
}

static void
printopening(struct regxml_elem *el)
{
	switch (el->rxe_type) {
	case REGXML_NODE:
		if (el->rxe_highid - el->rxe_lowid == 1) {
			(void)regxml_xml_printempty(xml,
			    el->rxe_name, el->rxe_attrv);
		} else {
			(void)regxml_xml_printopening(xml,
			    el->rxe_name, el->rxe_attrv);
		}
		break;
	case REGXML_TEXT:
		(void)regxml_xml_printtext(xml, el->rxe_value);
		break;
	case REGXML_COMMENT:
		(void)regxml_xml_printcomment(xml, el->rxe_value);
		break;
	case REGXML_PI:
		(void)regxml_xml_printpi(xml, el->rxe_name, el->rxe_value);
		break;
	default:
		/* NOTREACHED */
		abort();
	}
}

static void
printclosing(struct regxml_elem *el)
{
	switch (el->rxe_type) {
	case REGXML_NODE:
		if (el->rxe_highid - el->rxe_lowid != 1)
			(void)regxml_xml_printclosing(xml, el->rxe_name);
	default:
		break;
	}
}

static void
printcontext(struct regxml_elem *el)
{
	if (el == reg->rx_tree || el->rxe_lowid <= last)
		return;
	printcontext(el->rxe_parent);
	printopening(el);
	last = el->rxe_lowid;
}

static void
printscope(void)
{
	char *scbuf;

	if (asprintf(&scbuf, "%s.%u", scopeprefix,
		(unsigned int)reg->rx_maxmatch) == -1)
		err(2, "asprintf failed");
	(void)regxml_xml_printpi(xml, "regxml-scope", scbuf);
	free(scbuf);
}

static int
process(void)
{
	static struct regxml_elem *end;
	int r;

	for (;;) {
		if (end == NULL) {
			r = regxml_match(reg, match);
			switch (r) {
			case 0:
				found = 1;
				end = match[1];
				if (action == PRINT) {
					if (showcontext) {
						printcontext(reg->rx_start->
						    rxe_parent);
					}
					if (scopeprefix != NULL)
						printscope();
					printopening(reg->rx_start);
				} else if (action == RPRINT) {
					if (scopeprefix != NULL)
						printscope();
				}
				break;
			case REGXML_NOMATCH:
				if (action == RPRINT)
					printopening(reg->rx_start);
				break;
			case REGXML_CLOSING:
				if (action == PRINT && showcontext &&
				    reg->rx_start->rxe_lowid <= last) {
					printclosing(reg->rx_start);
					last = reg->rx_start->rxe_lowid;
				} else if (action == RPRINT)
					printclosing(reg->rx_start);
				break;
			default:
				return r;
			}
		} else {
			r = regxml_fetch(reg);
			switch (r) {
			case REGXML_OPENING:
				if (action == PRINT)
					printopening(reg->rx_start);
				break;
			case REGXML_CLOSING:
				if (action == PRINT)
					printclosing(reg->rx_start);
				if (reg->rx_start == end) {
					if ((action == PRINT ||
						action == RPRINT) &&
					    scopeprefix != NULL)
						printscope();
					end = NULL;
				}
				break;
			default:
				return r;
			}
		}
	}
}

static void
dofile(int flags)
{
	ssize_t n;
	int fd, r;

	if (strcmp(filename, "-") == 0)
		fd = 0;
	else {
		fd = open(filename, O_RDONLY);
		if (fd == -1)
			err(2, "open failed");
	}

	xml = NULL;		/* For die(). */
	r = regxml_xml_create(&xml, reg, flags);
	if (r != 0)
		die(r);

	regxml_xml_setfilename(xml, filename);
	regxml_xml_setindent(xml, baseindent);

	for (;;) {
		r = regxml_xml_getbuf(xml, &buf, INBUFSIZE);
		if (r != 0)
			die(r);
		n = read(fd, buf, INBUFSIZE);
		if (n <= 0)
			break;

		r = regxml_xml_pushraw(xml, NULL, (size_t)n);
		if (r != 0)
			die(r);
		r = process();
		switch (r) {
		case REGXML_EAGAIN:
			break;
		case REGXML_EOF:
			goto end;
		default:
			die(r);
		}
	}
	if (n == -1)
		err(2, "read failed");

end:
	r = regxml_xml_end(xml);
	if (r != 0)
		die(r);
	r = process();
	switch (r) {
	case REGXML_EOF:
		break;
	default:
		die(r);
	}

	regxml_xml_destroy(xml);
	regxml_reset(reg);

	if (found) {
		++count;
		if (action == PRINT || action == RPRINT)
			(void)printf("\n");
	}

	if (fd != 0) {
		if (close(fd) == -1)
			err(2, "close failed");
	}
}

static void
setformat(char *arg)
{
	char *end;

	baseindent = strtoul(arg, &end, 10);
	for (; *end != '\0'; ++end) {
		switch (*end) {
		case 'a':
			attrline = 1;
			break;
		case 't':
			tabindent = 1;
			break;
		default:
			break;
		}
	}
}

static void
usage(void)
{
	(void)fprintf(stderr,
	    "usage:\t"
	    "%s [-CcEiLlPqv] [-d sprefix] [-I indent] [-s scope] "
	    "pattern [file ...]\n",
	    getprogname());
	exit(3);
}

int
main(int argc, char *argv[])
{
	const char *pattern;
	int flags, opt, r;

	setprogname(argv[0]);

	while ((opt = getopt(argc, argv, "Ccd:EI:iLlPqs:v")) != -1) {
		switch (opt) {
		case 'C':
			action = PRINT;
			showcontext = 1;
			break;
		case 'c':
			action = COUNT;
			break;
		case 'd':
			scopeprefix = optarg;
			break;
		case 'E':
			noexpandentities = 1;
			break;
		case 'I':
			preservews = 0;
			setformat(optarg);
			break;
		case 'i':
			caseinsensitive = 1;
			break;
		case 'L':
			action = RLIST;
			break;
		case 'l':
			action = LIST;
			break;
		case 'P':
			preservews = 1;
			break;
		case 'q':
			action = QUIET;
			break;
		case 's':
			scope = optarg;
			break;
		case 'v':
			action = RPRINT;
			break;
		default:
			usage();
		}
	}

	argc -= optind;
	argv += optind;

	if (argc == 0)
		usage();
	pattern = *argv++;
	--argc;

	flags = REGXML_LONGEST;
#if 0
	if (noexpandentities)
		flags |= REGXML_ENTITIZE;
#endif
	if (showcontext || action == RPRINT)
		flags |= REGXML_MATCHSTEP;
	if (caseinsensitive)
		flags |= REGXML_ICASE;
	r = regxml_create(&reg, pattern, flags);
	if (r != 0)
		die(r);
	match = malloc(reg->rx_msize * sizeof *match);
	if (match == NULL)
		err(2, "malloc failed");

	if (scope != NULL) {
		r = regxml_setscope(reg, scope);
		if (r != 0)
			die(r);
	}

	flags = 0;
	if (!preservews)
		flags |= REGXML_XML_TRIM;
	if (!noexpandentities)
		flags |= REGXML_XML_EXPAND;
	if (tabindent)
		flags |= REGXML_XML_TABINDENT;
	if (attrline)
		flags |= REGXML_XML_ATTRLINE;

	if (argc == 0) {
		filename = "-";
		dofile(flags);
		if ((action == LIST && found) ||
		    (action == RLIST && !found))
			(void)printf("%s\n", filename);
	} else {
		while (argc-- > 0) {
			filename = *argv++;
			dofile(flags);

			if ((action == LIST && found) ||
			    (action == RLIST && !found))
				(void)printf("%s\n", filename);

			found = 0;
			last = 0;
		}
	}

	free(match);
	regxml_destroy(reg);

	if (action == COUNT)
		(void)printf("%u\n", count);
	return count > 0 ? 0 : 1;
}
