/*File - SWAP_EM.CPP                                                         */
/*****************************************************************************/
/*                                                                           */
/*Description:                                                               */
/*                                                                           */
/*     This files provides two functions to swap the bytes of two and        */
/*four byte variables.  These functions can be used to convert between       */
/*Motorola and Intel byte ordering.                                          */
/*                                                                           */
/*                                                                           */
/*Copyright Distributed Processing Technology, Corp.                         */
/*       140 Candace Dr.                                                     */
/*       Maitland, Fl.     32751   USA                                       */
/*       Phone: (407) 830-5522  Fax: (407) 260-5366                          */
/*       All Rights Reserved                                                 */
/*                                                                           */
/*Author:     Doug Anderson                                                  */
/*Date:          9/16/93                                                     */
/*                                                                           */
/*Editors:                                                                   */
/*                                                                           */
/*Remarks:                                                                   */
/*                                                                           */
/*                                                                           */
/*****************************************************************************/


/*Include Files -------------------------------------------------------------*/

#include     <osd_util.h>


/*Function - swap2 - start                                                   */
/*===========================================================================*/
/*                                                                           */
/*Description:                                                               */
/*                                                                           */
/*    This function swaps the MSB & LSB of a two byte variable.              */
/*                                                                           */
/*Parameters:                                                                */
/*                                                                           */
/*Return Value:                                                              */
/*                                                                           */
/*Global Variables Affected:                                                 */
/*                                                                           */
/*Remarks: (Side effects, Assumptions, Warnings...)                          */
/*                                                                           */
/*                                                                           */
/*---------------------------------------------------------------------------*/

uSHORT     osdSwap2(uSHORT *inShort)
{
#if defined ( _DPT_BIG_ENDIAN )

#   if (defined ( _DPT_STRICT_ALIGN ) )
	uSHORT i;

	i = (unsigned short)*((unsigned char *)inShort);
	i <<= 8;
	i |= (unsigned short)*(((unsigned char *)inShort)+1);
	return (i);
#   else
        return(*inShort);
#   endif


#else
  uSHORT i;

  i = *inShort & 0x0ff;
  *inShort >>= 8;
  *inShort |= (i << 8);
  return(*inShort);

#endif
}
/*swap2() - end                                                              */


/*Function - swap4 - start                                                   */
/*===========================================================================*/
/*                                                                           */
/*Description:                                                               */
/*                                                                           */
/*    This function reverses the byte ordering of a four byte variable.      */
/*                                                                           */
/*Parameters:                                                                */
/*                                                                           */
/*Return Value:                                                              */
/*                                                                           */
/*Global Variables Affected:                                                 */
/*                                                                           */
/*Remarks: (Side effects, Assumptions, Warnings...)                          */
/*                                                                           */
/*                                                                           */
/*---------------------------------------------------------------------------*/

uLONG     osdSwap4(uLONG *inLong)
{

#if defined ( _DPT_BIG_ENDIAN )

#   if (defined ( _DPT_STRICT_ALIGN ))
	uLONG i;

	i = (unsigned long)*((unsigned char *)inLong) << 24;
	i |= (unsigned long)*(((unsigned char *)inLong)+1) << 16;
	i |= (unsigned long)*(((unsigned char *)inLong)+2) << 8;
	i |= (unsigned long)*(((unsigned char *)inLong)+3);
	return (i);
#   else
        return(*inLong);
#   endif

#else
  char src[4], *dst;
  int i;

  *(uLONG *)src =  *inLong;
  dst = (char *)inLong;
  for(i = 0; i < 4; ++i)
   {
     dst[i] = src[3 - i];
   }
  return(*inLong);

#endif
}


/*swap4() - end                                                              */

uLONG	osdSwap3(uLONG *inLong)
{

#if defined ( _DPT_BIG_ENDIAN )

#   if (defined ( _DPT_STRICT_ALIGN ))
	uLONG i;

	i = (unsigned long)*(((unsigned char *)inLong)+0) << 16;
	i |= (unsigned long)*(((unsigned char *)inLong)+1) << 8;
	i |= (unsigned long)*(((unsigned char *)inLong)+2);
	return (i);
#   else
        return(*inLong);
#   endif

#else

  return((osdSwap4(inLong) >> 8));

#endif
}


/*Function - trueSwap2 - start                                               */
/*===========================================================================*/
/*                                                                           */
/*Description:                                                               */
/*                                                                           */
/*    This function ALWAYS swaps the MSB & LSB of a two byte variable.       */
/*                                                                           */
/*Parameters:                                                                */
/*                                                                           */
/*Return Value:                                                              */
/*                                                                           */
/*Global Variables Affected:                                                 */
/*                                                                           */
/*Remarks: (Side effects, Assumptions, Warnings...)                          */
/*                                                                           */
/*                                                                           */
/*---------------------------------------------------------------------------*/

uSHORT     trueSwap2(uSHORT *inShort)
{

  uSHORT outShort;
  char *src, *dst;
  int i;

  src = (char *)inShort;
  dst = (char *)&outShort;

  dst[0] = src[1];
  dst[1] = src[0];

  src[0] = dst[0];
  src[1] = dst[1];

  return outShort;

}
/* trueSwap2() - end                                                         */


/*Function - trueSwap4 - start                                               */
/*===========================================================================*/
/*                                                                           */
/*Description:                                                               */
/*                                                                           */
/* This function ALWAYS reverses the byte ordering of a four byte variable.  */
/*                                                                           */
/*Parameters:                                                                */
/*                                                                           */
/*Return Value:                                                              */
/*                                                                           */
/*Global Variables Affected:                                                 */
/*                                                                           */
/*Remarks: (Side effects, Assumptions, Warnings...)                          */
/*                                                                           */
/*                                                                           */
/*---------------------------------------------------------------------------*/

uLONG     trueSwap4(uLONG *inLong)
{

  uLONG outLong;
  char *src, *dst;
  int i;

  src = (char *)inLong;
  dst = (char *)&outLong;
  for(i = 0; i < 4; ++i)
   {
     dst[i] = src[3 - i];
   }

  for(i = 0; i < 4; ++i)
   {
     src[i] = dst[i];
   }

  return outLong;

}
/* trueSwap4() - end                                                        */

//Function - netSwap4() - start
//===========================================================================
//
//Description:
//
//    This function ensures that the input four byte variable is returned
//    in DPT network order (little-endian).  On big-endian machines this
//    function will swap the four byte variable.  On little-endian machines
//    this function will return the input value.
//
//Parameters:
//
//   val = 4 byte variable to be byte reversed.
//
//Return Value:
//
//   The variable in network order.
//
//Global Variables Affected:
//
//Remarks: (Side effects, Assumptions, Warnings...)
//
//---------------------------------------------------------------------------
uLONG netSwap4(uLONG val)
{
#if defined (_DPT_BIG_ENDIAN)

   // we need to swap the value and return it
   uSHORT  wd0, wd1;

   wd0 = (uSHORT) val;
   wd1 = (uSHORT) (val >> 16);
   wd0 = (wd0 >> 8) | (wd0 << 8);
   wd1 = (wd1 >> 8) | (wd1 << 8);

   return (((uLONG)wd0 << 16) | wd1);

#else

   // just return the value for little-endian machines
   return val;

#endif  // big_endian
}
//netSwap4() - end

