//! Qore namespace
namespace Qore {
//! UUID namespace
namespace UUID {
//! The UUID class provides methods that allow UUIDs to be created, manipulated, and parsed
/** */
class UUID {

public:
//! Clears the UUID and sets the null value (\c "00000000-0000-0000-0000-000000000000")
/** */
nothing clear();

public:
//! Compares the current UUID with another UUID and returns -1, 0, or 1 if the other UUID is greater than, equal to, or less than the current UUID respectively
/** @par Code Flags:
    @ref CONSTANT

    @param other the other UUID to compare
    
    @return -1, 0, or 1 if the other UUID is greater than, equal to, or less than the current UUID respectively

    @par Example:
    @code
my int $rc = $uuid.compare($other_uuid);
    @endcode
 */
int compare(UUID other);

public:
//! Generates a UUID
/** @param flags @ref uuid_generation_constants combined with binary or to affect how the UUID is generated

    @par Examples:
    - to generate a UUID using the default algorithm:
    @code
my UUID $uuid();
    @endcode
    - to make a UUID object from an explicit value:
    @code
my UUID $uuid("93c40733-9b99-46c6-91a5-f3989067dc1e");
    @endcode
    - to create a UUID with the null value:
    @code
my UUID $uuid(UUID::Empty);
    @endcode
 */
 constructor(int flags = Qore::UUID::None);

public:
//! parses a string to construct a UUID
/** @param uuid_str a UUID string in a format as in the following example: \c "93c40733-9b99-46c6-91a5-f3989067dc1e"

    @par Example:
    - to make a UUID object from an explicit value:
    @code
my UUID $uuid("93c40733-9b99-46c6-91a5-f3989067dc1e");
    @endcode

    @throw UUID-STRING-ERROR invalid UUID string (invalid format, invalid characters, etc)
 */
 constructor(string uuid_str);

public:
//! Creates an exact copy of the UUID object
/** @par Example:
    @code
my UUID $new_uuid = $uuid.copy();
    @endcode
 */
 copy();

public:
//! Generates a new UUID value for the object; the current value of the UUID object is overwritten with the newly-generated value
/** @param flags @ref uuid_generation_constants combined with binary or to affect how the UUID is generated

    @par Examples:
    - to generate a UUID using the default algorithm:
    @code
$uuid.generate();
    @endcode
    - to create a UUID with the null value:
    @code
$uuid.generate(UUID::Empty);
    @endcode
 */
nothing generate(int flags = Qore::UUID::None);

public:
//! Returns True if the current %UUID is the null %UUID, or False if not
/** @return True if the current %UUID is the null %UUID, or False if not

   @par Code Flags:
    @ref CONSTANT

     @par Example:
    @code
my bool $b = $uuid.isNull();
    @endcode
 */
bool isNull();

public:
//! sets the UUID from the given string
/** @param uuid_str a UUID string in a format as in the following example: \c "93c40733-9b99-46c6-91a5-f3989067dc1e"

    @par Example:
    @code
$uuid.set("93c40733-9b99-46c6-91a5-f3989067dc1e");
    @endcode

    @throw UUID-STRING-ERROR invalid UUID string (invalid format, invalid characters, etc)
 */
nothing set(string uuid_str);

public:
//! Returns the string representation of the UUID
/** @par Code Flags:
    @ref CONSTANT

    @param flag one of the @ref uuid_formatting_constants to affect the formatting of the %UUID string output

    @return the string representation of the UUID, ex: \c "f5c5fd74-5b3e-4f02-8772-f8a80da9b028"

    @par Example:
    @code
my string $str = $uuid.toString();
    @endcode
 */
string toString(int flag = Qore::UUID::LowerCase);

public:
//! Returns a UUID string without having to create a UUID object
/** @par Code Flags:
    @ref CONSTANT

    @param str_flag one of the @ref uuid_formatting_constants to affect the formatting of the %UUID string output
    @param gen_flags @ref uuid_generation_constants combined with binary or to affect how the UUID is generated
    
    @return the string representation of the UUID according to the arguments, ex: \c "f5c5fd74-5b3e-4f02-8772-f8a80da9b028"
 */
static string get(int str_flag = Qore::UUID::LowerCase, int gen_flags = Qore::UUID::None);
};
/** @defgroup uuid_generation_constants UUID Generation Constants
//@{
 */

//! The UUID will be assigned the null value on creation
    const Empty = QUF_EMPTY;
//! If case is supported in the UUID::toString() method; this is always True
    const HAVE_UNPARSE_CASE = True;
//! No value; this is the default
    const None = QUF_NONE;
//! Force the use of the all-random UUID format, even if a high-quality random number generator is not available for the library
    const Random = QUF_RANDOM;
//! Force the use of the alternative algorithm that uses the current time and local ethernet MAC address (if available); note that this algorithm can leak information about when and where  the UUID was generated
    const Time = QUF_TIME;
/** @} */
/** @defgroup uuid_formatting_constants UUID Formatting Constants
//@{
 */
namespace Qore::UUID;

//! All alphebetic hexadecimal characters will be in lower case; this is the default
    const LowerCase = QUF_LOWER_CASE;
//! All alphebetic hexadecimal characters will be in upper case
    const UpperCase = QUF_UPPER_CASE;
/** @} */
}
}
